/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2003 The T-Struts Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE T-STRUTS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE T-STRUTS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the T-Struts Project.
 */
package jp.ossc.tstruts.common;

import java.io.Serializable;
import java.lang.reflect.Array;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.beanutils.ConversionException;
import org.apache.commons.beanutils.DynaBean;
import org.apache.commons.beanutils.DynaClass;
import org.apache.commons.beanutils.DynaProperty;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import jp.ossc.tstruts.util.MyConvertUtils;

/**
 * [Ui[DynaBeanB<p>
 * 
 * @version $Name: release-1_1-1_0 $
 * @author N.Kurosawa
 * @since 1.0
 */
public class DynaUserProfile implements DynaBean, Serializable, Cloneable {
	private static Log log = LogFactory.getLog(DynaUserProfile.class);

	class ValueHolder implements Serializable, Cloneable {
		String name;
		Object value;
		public ValueHolder() {
		}
		
		public ValueHolder(String name, Object value) {
			this.name = name;
			this.value = value;
		}
		
		public Object clone() {
			ValueHolder n = new ValueHolder(name, value);
			return n;
		}
	}

	private DynaClass dynaClass;
	private ValueHolder[] values;
	private transient Map valuesMap = null;
	
	public DynaUserProfile() {
		dynaClass = null;
		values = null;
	}

	public DynaUserProfile(DynaClass dynaClass)
	throws InstantiationException, IllegalAccessException {
		setDynaClass(dynaClass);
	}

	public Object clone() {
		try {
			DynaUserProfile n = new DynaUserProfile(dynaClass);
			n.values = new ValueHolder[values.length];
			for (int i = 0; i < values.length; ++i) {
				n.values[i] = (ValueHolder)values[i].clone();
			}
			return n;
		} catch (Exception e) {
			log.error("Failed to clone DynaUserProfile.", e);
			return null;
		}
		
	}
	public void setDynaClass(DynaClass dynaClass)
	throws InstantiationException, IllegalAccessException {
		this.dynaClass = dynaClass;
		DynaProperty properties[] = dynaClass.getDynaProperties();
		values = new ValueHolder[properties.length];
		for (int i = 0; i < values.length; ++i)
			values[i] = new ValueHolder(properties[i].getName(), null);
	}

	/**
	 *
	 */
	public DynaClass getDynaClass() {
		return dynaClass;
	}

	protected Map getValuesMap() {
		if (valuesMap != null)
			return valuesMap;
		valuesMap = new HashMap();
		for (int i = 0; i < values.length; ++i) {
			valuesMap.put(values[i].name, values[i]);
		}
		return valuesMap;
	}

	protected ValueHolder getValueHolder(String name) {
		return (ValueHolder)getValuesMap().get(name);
	}
	
	protected ValueHolder[] getValueHolders() {
		return values;
	}

	/**
	 *
	 */
	public boolean contains(String name, String key) {
		ValueHolder holder = getValueHolder(name);
		if (holder == null || holder.value == null) {
			throw new NullPointerException
				("No mapped value for '" + name + "(" + key + ")'");
		} else if (holder.value instanceof Map) {
			return (((Map) holder.value).containsKey(key));
		} else {
			throw new IllegalArgumentException
				("Non-mapped property for '" + name + "(" + key + ")'");
		}
	}

	/**
	 *
	 */

	public Object get(String name, int index) {
		ValueHolder holder = getValueHolder(name);
		if (holder == null || holder.value == null) {
			throw new NullPointerException
				("No indexed value for '" + name + "[" + index + "]'");
		} else if (holder.value.getClass().isArray()) {
			return (Array.get(holder.value, index));
		} else if (holder.value instanceof List) {
			return ((List) holder.value).get(index);
		} else {
			throw new IllegalArgumentException
				("Non-indexed property for '" + name + "[" + index + "]'");
		}
	}

	/**
	 *
	 */

	public Object get(String name, String key) {
		ValueHolder holder = getValueHolder(name);
		if (holder == null) {
			throw new NullPointerException
				("No mapped value for '" + name + "(" + key + ")'");
		} else if (holder.value instanceof Map) {
			return (((Map) holder.value).get(key));
		} else {
			throw new IllegalArgumentException
				("Non-mapped property for '" + name + "(" + key + ")'");
		}
	}

	/**
	 *
	 */
	public Object get(String name) {

		// Return any non-null value for the specified property
		ValueHolder holder = getValueHolder(name);
		if (holder == null) {
			throw new NullPointerException
					("The type for property " + name + " is invalid");
		} 
		return holder.value;
	}

	/**
	 * boolean^̃f[^擾郆[eBeB\bh
	 */
	public boolean getBoolean(String name) {
		Object value = get(name);
		if (value == null)
			throw new NullPointerException("Property[" + name + "] is not found in UserProfile.");
		return MyConvertUtils.convertBoolean(value);
	}

	public byte getByte(String name) {
		//TODO TBD
		return (byte)0;
	}

	/**
	 *
	 */
	public void remove(String name, String key) {
		ValueHolder holder = getValueHolder(name);
		if (holder == null || holder.value == null) {
			throw new NullPointerException
				("No mapped value for '" + name + "(" + key + ")'");
		} else if (holder.value instanceof Map) {
			((Map) holder.value).remove(key);
		} else {
			throw new IllegalArgumentException
				("Non-mapped property for '" + name + "(" + key + ")'");
		}
	}

	/**
	 *
	 */

	public void set(String name, int index, Object value) {
		ValueHolder holder = getValueHolder(name);
		if (holder == null || holder.value == null) {
			throw new NullPointerException
				("No indexed value for '" + name + "[" + index + "]'");
		} else if (holder.value.getClass().isArray()) {
			Array.set(holder.value, index, value);
		} else if (holder.value instanceof List) {
			try {
				((List) holder.value).set(index, value);
			} catch (ClassCastException e) {
				throw new ConversionException(e.getMessage());
			}
		} else {
			throw new IllegalArgumentException
				("Non-indexed property for '" + name + "[" + index + "]'");
		}
	}

	/**
	 *
	 */

	public void set(String name, Object value) {
		DynaProperty descriptor = getDynaClass().getDynaProperty(name);
		if (descriptor.getType() == null) {
			throw new NullPointerException
				("The type for property " + name + " is invalid");
		}
		if (value == null) {
			if (descriptor.getType().isPrimitive()) {
				throw new NullPointerException
					("Primitive value for '" + name + "'");
			}
		} else if (!isDynaAssignable(descriptor.getType(), value.getClass())) {
			throw new ConversionException
				("Cannot assign value of type '" +
				 value.getClass().getName() +
				 "' to property '" + name + "' of type '" +
				 descriptor.getType().getName() + "'");
		}
		ValueHolder holder = getValueHolder(name);
		holder.value = value;
	}

	/**
	 *
	 */

	public void set(String name, String key, Object value) {
		ValueHolder holder = getValueHolder(name);
		if (holder == null || holder.value == null) {
			throw new NullPointerException
				("No mapped value for '" + name + "(" + key + ")'");
		} else if (holder.value instanceof Map) {
			((Map) holder.value).put(key, value);
		} else {
			throw new IllegalArgumentException
				("Non-mapped property for '" + name + "(" + key + ")'");
		}
	}

	/**
	 * Indicates if an object of the source class is assignable to the destination class.
	 *
	 * @param dest Destination class
	 * @param source Source class
	 */
	protected boolean isDynaAssignable(Class dest, Class source) {

		if (dest.isAssignableFrom(source) ||
			((dest == Boolean.TYPE) && (source == Boolean.class)) ||
			((dest == Byte.TYPE) && (source == Byte.class)) ||
			((dest == Character.TYPE) && (source == Character.class)) ||
			((dest == Double.TYPE) && (source == Double.class)) ||
			((dest == Float.TYPE) && (source == Float.class)) ||
			((dest == Integer.TYPE) && (source == Integer.class)) ||
			((dest == Long.TYPE) && (source == Long.class)) ||
			((dest == Short.TYPE) && (source == Short.class))) {
			return (true);
		} else {
			return (false);
		}

	}
}
