/********************************************************************/
/* Copyright (c) 2019 System fugen G.K. and Yuzi Mizuno          */
/* All rights reserved.                                             */
/* ***************************************************** */

// GLUserPreference.h
#pragma once
#include "mgGL/Color.h"
#include "Misc/UserPrefName.h"
#include "Misc/UserPrefValue.h"

class MGContext;
class MGCursorRestriction;

/// @class UserPreference Misc/GLUserPreference.h
/// Access to user preference data.
class UserPreference
{
	typedef std::map<UserPrefName, UserPrefBoolValue> UserPrefBoolMap;
	typedef std::map<UserPrefName, UserPrefIntValue> UserPrefIntMap;
	typedef std::map<UserPrefName, UserPrefDoubleValue> UserPrefDoubleMap;
	typedef std::map<UserPrefName, UserPrefStringValue> UserPrefStringMap;

public:

	/// Access type
	/// @sa GetBoolValue, GetIntValue, GetDoubleValue, GetStringValue
	enum Access	{
		CurrentValue,
		LowerValue,
		UpperValue,
		DefaultValue,
	};

	/// The default constructor.
	UserPreference();

	/// The destructor.
	~UserPreference();

	/// Initialize the user preference.
	void Init();

	///Export this preference to the context ctx.
	void exportToContext(MGContext& ctx);

	///Export this preference to the MGCursorRestriction cr.
	void exportToCRestrict(MGCursorRestriction& cr)const;
	
	///Import this preference from the MGCursorRestriction cr.
	//(the angle mode and angle value from cr)
	void importFromCRestrict(const MGCursorRestriction& cr);

	/// Set registry data to all of the user preference data.
	/// @pre Init() has been called.
	void LoadRegistry(CWinApp& app);

	/// Save user preference to registry.
	void SaveRegistry(CWinApp& app);

	///@name Access to user preference value of type bool.
	//@{
	bool GetBoolValue(UserPrefName key, Access acc = CurrentValue) const;
	void SetBoolValue(UserPrefName key, bool bValue);
	//@}

	///@name Access to user preference value of type int.
	//@{
	int GetIntValue(UserPrefName key, Access acc = CurrentValue) const;
	void GetIntValue(UserPrefName key, int& nValue, int& nLower, int& nUpper) const;
	void SetIntValue(UserPrefName key, int nValue);
	//@}

	///@name Access to user preference value of type double.
	//@{
	double GetDoubleValue(UserPrefName key, Access acc = CurrentValue) const;
	void GetDoubleValue(UserPrefName key, double& dVal, double& dLower, double& dUpper) const;
	void SetDoubleValue(UserPrefName key, double dValue);
	//@}

	///@name Access to user preference value of type CString.
	//@{
	CString GetStringValue(UserPrefName key, Access acc = CurrentValue) const;
	void SetStringValue(UserPrefName key, const CString& strValue);
	//@}

	///@name Access to user preference value of MGColor.
	//@{
	const MGColor& GetBackgroundColor()const{return m_backgroundColor;};
	const MGColor& GetHilightColor()const{return m_hilightColor;};
	const MGColor& GetObjectColor()const {return m_objectColor;};

	const MGColor* GetGridColors()const{return m_gridColor;};
		//Let const MGColor* color=GetGridColors(), 
		//then color[0]=grid color, [1]=X-axis color, [2]=Y-axis color, [3]=Z-axis color.
	//@}


private:
	void InitBool(UserPrefName key, LPCTSTR section, LPCTSTR entry, bool value);
	void InitInt(UserPrefName key, LPCTSTR section, LPCTSTR entry, int value, int lower, int upper);
	void InitDouble(UserPrefName key, LPCTSTR section, LPCTSTR entry, double value, double lower, double upper);
	void InitString(UserPrefName key, LPCTSTR section, LPCTSTR entry, const CString& value);

	// Copy constructor and assignment operator are not allowed to use.

	UserPreference(const UserPreference& other);
	UserPreference& operator=(const UserPreference& other);

	// ȈSingletonΉ
public:
	static UserPreference& getInstance();
	static void setInstance(UserPreference* pref);
private:
	static UserPreference* m_pPreference;
	
	UserPrefBoolMap m_pref_bool; ///< user-preferred values of type bool
	UserPrefIntMap m_pref_int; ///< user-preferred values of type int
	UserPrefDoubleMap m_pref_double; ///< user-preferred values of type double
	UserPrefStringMap m_pref_str; ///< user-preferred values of type CString

	MGColor m_backgroundColor;
	MGColor m_hilightColor;
	MGColor m_objectColor;

	MGColor m_gridColor[4];
		//[0]=grid color, [1]=X-axis color, [2]=Y-axis color, [3]=Z-axis color.
};

const MGColor& getUserPrefBackgroundColor();
const MGColor& getUserPrefSelectionColor();
const MGColor& getUserPrefObjectColor();
const MGColor* getUserPrefGridColor();
	//Let const MGColor* color=getUserPrefGridColor(), 
	//then color[0]=grid color, [1]=X-axis color, [2]=Y-axis color, [3]=Z-axis color.
