package jp.sourceforge.sxdbutils.extras.dao;

import java.sql.SQLException;
import java.util.List;

import jp.sourceforge.sxdbutils.query.AttributeQueryFactoryBuilder;
import jp.sourceforge.sxdbutils.query.QueryFactory;
import jp.sourceforge.sxdbutils.template.EntityPersistenceHelper;

public abstract class AbstractAttributeCLUDTemplate<E, B extends AttributeQueryFactoryBuilder<E>> extends
		AbstractAttributeSelectTemplate<E> implements CRUDOperation<E>{

	protected EntityPersistenceHelper<E> getEntityPersistenceHelper() {
		return new EntityPersistenceHelper<E>();
	}

	protected abstract B createQueryFactoryBuilder();

	protected abstract void updateConfig(B config);

	protected abstract void insertConfig(B config);

	protected abstract void deleteConfig(B config);

	private final Object updateFoctoryLock = new Object();
	private QueryFactory<E> updateFactory;

	private void ensureUpdateFactory() {
		synchronized (updateFoctoryLock) {
			if (this.updateFactory == null) {
				B builder = createQueryFactoryBuilder();
				updateConfig(builder);
				this.updateFactory = builder.buildUpdate();
			}
		}
	}

	private final Object insertFoctoryLock = new Object();
	private QueryFactory<E> insertFactory;

	private void ensureInsertFactory() {
		synchronized (insertFoctoryLock) {
			if (this.insertFactory == null) {
				B builder = createQueryFactoryBuilder();
				insertConfig(builder);
				this.insertFactory = builder.buildInsert();
			}
		}
	}

	private final Object deleteFoctoryLock = new Object();
	private QueryFactory<E> deleteFactory;

	private void ensureDeleteFactory() {
		synchronized (deleteFoctoryLock) {
			if (this.deleteFactory == null) {
				B builder = createQueryFactoryBuilder();
				deleteConfig(builder);
				this.deleteFactory = builder.buildDelete();
			}
		}
	}

	//----------------update
	public int update(E entity) throws SQLException {
		ensureUpdateFactory();
		return getEntityPersistenceHelper().execute(getConnection(), this.updateFactory, entity);
	}

	public int[] update(List<E> entites) throws SQLException {
		ensureUpdateFactory();
		return getEntityPersistenceHelper().executeBatch(getConnection(), this.updateFactory, entites);
	}

	//----------------insert
	public int insert(E entity) throws SQLException {
		ensureInsertFactory();
		return getEntityPersistenceHelper().execute(getConnection(), this.insertFactory, entity);
	}

	public int[] insert(List<E> entites) throws SQLException {
		ensureInsertFactory();
		return getEntityPersistenceHelper().executeBatch(getConnection(), this.insertFactory, entites);
	}

	//----------------delete
	public int delete(E entity) throws SQLException {
		ensureDeleteFactory();
		return getEntityPersistenceHelper().execute(getConnection(), this.deleteFactory, entity);
	}

	public int[] delete(List<E> entites) throws SQLException {
		ensureDeleteFactory();
		return getEntityPersistenceHelper().executeBatch(getConnection(), this.deleteFactory, entites);
	}

}
