package jp.sourceforge.sxdbutils.util;

import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;

import org.apache.commons.dbutils.DbUtils;

public class DatabaseMetaDataUtil {
	private DatabaseMetaDataUtil() {
	}

	/**
	 * 
	 * @param dbMetaData
	 * @param tableName
	 * @return Collection<String>
	 * @throws SQLException
	 */
	public static Collection<String> getPrimaryKeyNames(DatabaseMetaData dbMetaData, String tableName) throws SQLException {
		String schema = null;
		int index = tableName.indexOf('.');
		if (index >= 0) {
			schema = tableName.substring(0, index);
			tableName = tableName.substring(index + 1);
		}
		String convertedTableName = convertIdentifier(dbMetaData, tableName);
		Collection<String> result = getPrimaryKeyNames(dbMetaData, convertIdentifier(dbMetaData, schema), convertedTableName);
		if (!result.isEmpty()) {
			return result;
		}
		result = getPrimaryKeyNames(dbMetaData, schema, tableName);
		if (!result.isEmpty()) {
			return result;
		}
		// だめならスキーマ指定をなくす
		if (schema != null) {
			result = getPrimaryKeyNames(dbMetaData, null, convertedTableName);
			if (!result.isEmpty()) {
				return result;
			}
			result = getPrimaryKeyNames(dbMetaData, null, tableName);
		}
		return result;
	}

	/**
	 * 
	 * @param dbMetaData
	 * @param schema
	 * @param tableName
	 * @return Collection<String>
	 */
	public static Collection<String> getPrimaryKeyNames(DatabaseMetaData dbMetaData, String schema, String tableName) {
		Collection<String> result = new ArrayList<String>();
		ResultSet rs = null;
		try {
			rs = dbMetaData.getPrimaryKeys(null, schema, tableName);
			while (rs.next()) {
				result.add(rs.getString(4));
			}
		} catch (SQLException ex) {
			throw new RuntimeException(ex);
		} finally {
			DbUtils.closeQuietly(rs);
		}
		return result;
	}

	/**
	 * 
	 * @param dbMetaData
	 * @param tableName
	 * @return Collection<DatabaseColumnInfo>
	 * @throws SQLException
	 */
	public static Collection<DatabaseColumnInfo> getColumnInfos(DatabaseMetaData dbMetaData, String tableName) throws SQLException {
		String schema = null;
		int index = tableName.indexOf('.');
		if (index >= 0) {
			schema = tableName.substring(0, index);
			tableName = tableName.substring(index + 1);
		}
		String convertedTableName = convertIdentifier(dbMetaData, tableName);

		Collection<DatabaseColumnInfo> result = getColumnInfos(dbMetaData, convertIdentifier(dbMetaData, schema), convertedTableName);

		if (!result.isEmpty()) {
			return result;
		}

		result = getColumnInfos(dbMetaData, schema, tableName);
		if (!result.isEmpty()) {
			return result;
		}
		// だめならスキーマ指定をなくす
		if (schema != null) {
			result = getColumnInfos(dbMetaData, null, convertedTableName);
			if (!result.isEmpty()) {
				return result;
			}
			result = getColumnInfos(dbMetaData, null, tableName);
		}
		return result;
	}

	/**
	 * 
	 * @param dbMetaData
	 * @param schema
	 * @param tableName
	 * @return Collection<DatabaseColumnInfo>
	 * @throws SQLException
	 */
	private static Collection<DatabaseColumnInfo> getColumnInfos(DatabaseMetaData dbMetaData, String schema, String tableName)
			throws SQLException {
		Collection<DatabaseColumnInfo> result = new ArrayList<DatabaseColumnInfo>();
		ResultSet rs = null;
		try {
			rs = dbMetaData.getColumns(null, schema, tableName, null);
			while (rs.next()) {
				DatabaseColumnInfo info = new DatabaseColumnInfo();
				info.setColumnName(rs.getString(DatabaseColumnInfo.COLUMN_NAME_INDEX));
				info.setColumnSize(rs.getInt(DatabaseColumnInfo.COLUMN_SIZE_INDEX));
				info.setDataType(rs.getInt(DatabaseColumnInfo.DATA_TYPE_INDEX));
				info.setDecimalDigits(rs.getInt(DatabaseColumnInfo.DECIMAL_DIGITS_INDEX));
				info.setTypeName(rs.getString(DatabaseColumnInfo.TYPE_NAME_INDEX));
				result.add(info);
			}
		} finally {
			DbUtils.closeQuietly(rs);
		}
		return result;
	}

	public static String convertIdentifier(DatabaseMetaData dbMetaData, String identifier) throws SQLException {

		if (identifier == null) {
			return null;
		}
		if (!dbMetaData.supportsMixedCaseIdentifiers()) {
			if (dbMetaData.storesUpperCaseIdentifiers()) {
				return identifier.toUpperCase();
			}
			return identifier.toLowerCase();
		}
		return identifier;
	}

}
