package jp.sourceforge.sxdbutils.query;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import jp.sourceforge.sxdbutils.TypeMappings;
import jp.sourceforge.sxdbutils.ValueType;
import jp.sourceforge.sxdbutils.mapping.ColumnNameMapping;
import jp.sourceforge.sxdbutils.mapping.NameMapping;
import jp.sourceforge.sxdbutils.meta.AttributeDescpriotr;
import jp.sourceforge.sxdbutils.meta.PersistenceEntry;
import jp.sourceforge.sxdbutils.meta.VersionColumnPersistenceEntry;
import jp.sourceforge.sxdbutils.util.OthersUtils;

/**
 * AttributeQueryFactoryのビルダーです。
 * 
 * @author chinpei
 * 
 */
public abstract class AttributeQueryFactoryBuilder<E> {

	private static final NameMapping DEFAULT_NAME_MAPPING = new ColumnNameMapping();

	protected String tableName;
	protected final Class<E> baseClass;
	protected final NameMapping nameMapping;
	protected Set<String> whereKeyColumnSet;
	protected String versionColumnName;

	/**
	 * KEY：カラム名、VALUE：SQLタイプ
	 */
	protected Map<String, Object> columnTypeMap = new HashMap<String, Object>();

	// List<String>
	protected Set<String> excludeColumns = new HashSet<String>();

	public AttributeQueryFactoryBuilder(Class<E> baseClass) {
		this(baseClass, DEFAULT_NAME_MAPPING);
	}

	public AttributeQueryFactoryBuilder(Class<E> baseClass, NameMapping nameMapping) {
		this.baseClass = baseClass;
		this.nameMapping = nameMapping;
		this.tableName = OthersUtils.getShortClassName(baseClass);
	}

	private boolean isWhereKey(String columnName) {
		if (this.whereKeyColumnSet == null)
			return false;
		return this.whereKeyColumnSet.contains(columnName.toLowerCase());
	}

	public QueryFactory<E> buildUpdate() {
		PersistenceEntry[] beanColumns = this.toPersistenceEntries();

		StringBuffer columnBuffer = new StringBuffer(beanColumns.length * 7);

		List<PersistenceEntry> updateColumnProperties = new ArrayList<PersistenceEntry>();
		List<PersistenceEntry> whereColumnProperties = new ArrayList<PersistenceEntry>();

		columnBuffer.append("update ").append(this.tableName).append(" set ");
		boolean flg = false;
		for (int i = 0; i < beanColumns.length; i++) {

			PersistenceEntry beanColumn = beanColumns[i];
			if (isWhereKey(beanColumn.getColumnName())) {
				whereColumnProperties.add(beanColumn);
				continue;
			}
			if (flg) {
				columnBuffer.append(',');
			} else {
				flg = true;
			}
			columnBuffer.append(beanColumn.getColumnName()).append("=?");
			if (versionColumnName != null && versionColumnName.equalsIgnoreCase(beanColumn.getColumnName())) {
				whereColumnProperties.add(beanColumn);
				updateColumnProperties.add(new VersionColumnPersistenceEntry(beanColumn));
			} else {
				updateColumnProperties.add(beanColumn);
			}
		}
		StringBuffer whereBuffer = null;
		if (!whereColumnProperties.isEmpty()) {
			for (Iterator<PersistenceEntry> iterator = whereColumnProperties.iterator(); iterator.hasNext();) {
				if (whereBuffer == null)
					whereBuffer = new StringBuffer(" where ");
				else
					whereBuffer.append(" and ");
				PersistenceEntry columnProperty = iterator.next();
				whereBuffer.append(columnProperty.getColumnName()).append("=?");
			}
		}

		List<PersistenceEntry> list = new ArrayList<PersistenceEntry>(updateColumnProperties);
		list.addAll(whereColumnProperties);
		if (whereBuffer != null)
			columnBuffer.append(whereBuffer);
		String sql = columnBuffer.toString();
		return new AttributeQueryFactory<E>(sql, (PersistenceEntry[]) list.toArray(new PersistenceEntry[list.size()]));
	}

	public QueryFactory<E> buildInsert() {
		PersistenceEntry[] entries = this.toPersistenceEntries();
		StringBuffer columnBuffer = new StringBuffer(entries.length * 7);
		StringBuffer valueBuffer = new StringBuffer(entries.length * 2);

		columnBuffer.append("insert into ").append(this.tableName).append("(");
		valueBuffer.append(" values ( ");
		boolean flg = false;
		for (int i = 0; i < entries.length; i++) {

			if (flg) {
				columnBuffer.append(',');
				valueBuffer.append(',');
			} else {
				flg = true;
			}
			PersistenceEntry columnProperty = entries[i];
			columnBuffer.append(columnProperty.getColumnName());
			valueBuffer.append('?');
		}
		columnBuffer.append(" ) ");
		valueBuffer.append(" ) ");
		String sql = columnBuffer.append(valueBuffer).toString();

		return new AttributeQueryFactory<E>(sql, entries);

	}

	public QueryFactory<E> buildDelete() {
		PersistenceEntry[] entries = this.toPersistenceEntries();

		StringBuffer deleteBuffer = new StringBuffer(entries.length * 7);

		List<PersistenceEntry> whereEntries = new ArrayList<PersistenceEntry>();

		deleteBuffer.append("delete from ").append(this.tableName);
		for (int i = 0; i < entries.length; i++) {

			PersistenceEntry entry = entries[i];
			if (isWhereKey(entry.getColumnName())) {
				whereEntries.add(entry);
				continue;
			}
			if (versionColumnName != null && versionColumnName.equalsIgnoreCase(entry.getColumnName())) {
				whereEntries.add(entry);
			}
		}
		StringBuffer whereBuffer = null;
		if (!whereEntries.isEmpty()) {
			for (Iterator<PersistenceEntry> iterator = whereEntries.iterator(); iterator.hasNext();) {
				if (whereBuffer == null)
					whereBuffer = new StringBuffer(" where ");
				else
					whereBuffer.append(" and ");
				PersistenceEntry entry = (PersistenceEntry) iterator.next();
				whereBuffer.append(entry.getColumnName()).append("=?");
			}
		}

		if (whereBuffer != null)
			deleteBuffer.append(whereBuffer);
		String sql = deleteBuffer.toString();
		return new AttributeQueryFactory<E>(sql, (PersistenceEntry[]) whereEntries
				.toArray(new PersistenceEntry[whereEntries.size()]));
	}

	protected abstract PersistenceEntry[] toPersistenceEntries();

	protected ValueType getValueType(AttributeDescpriotr attributeDescpriotr, Integer sqlType) {
		return sqlType == null ? TypeMappings.getValueType(attributeDescpriotr.getType()) : TypeMappings.getValueType(
				attributeDescpriotr.getType(), sqlType.intValue());
	}

}
