package jp.sourceforge.sxdbutils.mapping;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.sourceforge.sxdbutils.util.CaseInsensitiveHashMap;
import jp.sourceforge.sxdbutils.util.ReflectionUtil;

/**
 * 定数アノテーションにより、カラム名が指定できる {@link NameMapping}です。
 * 
 * @author chinpei
 * 
 */
public class ConstantAnnoNameMapping<E> implements NameMapping {
	private static final NameMapping DEFAULT_COMVERSION = new ColumnNameMapping();

	private static final String ANNOTATION_SUFFIC = "_COLUMN";
	protected final OverwriteNameMapping delegateNameMapping;

	public ConstantAnnoNameMapping(Class<E> beanClass) {
		this(beanClass, DEFAULT_COMVERSION);
	}

	public ConstantAnnoNameMapping(Class<E> beanClass, NameMapping defaultNameMapping) {
		Map<String, String> columnToPropertyMap = new HashMap<String, String>();
		Field[] fields = getAnnotationFields(beanClass);
		for (int i = 0; i < fields.length; i++) {
			Field field = fields[i];
			String propertyName = ReflectionUtil.removalSuffixFieldName(field, ANNOTATION_SUFFIC);
			String columnName = getColumnNameFromAnnotation(field);
			columnToPropertyMap.put(propertyName, columnName);
		}
		this.delegateNameMapping = new OverwriteNameMapping(columnToPropertyMap, defaultNameMapping);
	}

	protected final String getColumnNameFromAnnotation(Field field) {
		return ReflectionUtil.getString(field);
	}

	/**
	 * アノテーションを保持したフィールドを返します。
	 * 
	 * @param beanClass
	 * @return
	 */
	protected final Field[] getAnnotationFields(Class<E> beanClass) {
		List<Field> list = new ArrayList<Field>();
		readAnnotationFieldsToList(beanClass, list);
		return (Field[]) list.toArray(new Field[list.size()]);
	}

	protected void readAnnotationFieldsToList(Class<? super E> beanClass, List<Field> list) {
		if ("java.lang.Object".equals(beanClass.getName()))
			return;
		readAnnotationFieldsToList(beanClass.getSuperclass(), list);
		Field[] fields = beanClass.getDeclaredFields();
		for (int i = 0; i < fields.length; i++) {
			Field field = fields[i];
			if (isColumnAnnotation(field))
				list.add(field);
		}
	}

	public String toIntermediateNameFromAttrName(String bindObjectName) {
		return this.delegateNameMapping.toIntermediateNameFromAttrName(bindObjectName);
	}

	public final Map<String, String> createIntermediateMap() {
		return new CaseInsensitiveHashMap<String>();
	}

	public String toIntermediateNameFromColumnName(String columnName) {
		return this.delegateNameMapping.toIntermediateNameFromColumnName(columnName);
	}

	public String toColumnNameFromAttrName(String bindObjectName) {
		return this.delegateNameMapping.toColumnNameFromAttrName(bindObjectName);
	}

	/**
	 * カラムアノテーションの判定
	 * 
	 * @param field
	 * @return
	 */
	private boolean isColumnAnnotation(Field field) {
		if (!ReflectionUtil.isPublicStaticFinalString(field)) {
			return false;
		}
		if (field.getName().endsWith(ANNOTATION_SUFFIC)) {
			return true;
		}
		return false;
	}
}
