/*
  This program is a conversion from original Swiss Ephemeris Software.
*/
/* Copyright (C) 1997, 1998 Astrodienst AG, Switzerland.  All rights reserved.
  
  This file is part of Swiss Ephemeris Free Edition.
  
  Swiss Ephemeris is distributed with NO WARRANTY OF ANY KIND.  No author
  or distributor accepts any responsibility for the consequences of using it,
  or for whether it serves any particular purpose or works at all, unless he
  or she says so in writing.  Refer to the Swiss Ephemeris Public License
  ("SEPL" or the "License") for full details.
  
  Every copy of Swiss Ephemeris must include a copy of the License,
  normally in a plain ASCII text file named LICENSE.  The License grants you
  the right to copy, modify and redistribute Swiss Ephemeris, but only
  under certain conditions described in the License.  Among other things, the
  License requires that the copyright notices and this notice be preserved on
  all copies.

  For uses of the Swiss Ephemeris which do not fall under the definitions
  laid down in the Public License, the Swiss Ephemeris Professional Edition
  must be purchased by the developer before he/she distributes any of his
  software or makes available any product or service built upon the use of
  the Swiss Ephemeris.

  Authors of the Swiss Ephemeris: Dieter Koch and Alois Treindl

  The authors of Swiss Ephemeris have no control or influence over any of
  the derived works, i.e. over software or services created by other
  programmers which use Swiss Ephemeris functions.

  The names of the authors or of the copyright holder (Astrodienst) must not
  be used for promoting any software, product or service which uses or contains
  the Swiss Ephemeris. This copyright notice is the ONLY place where the
  names of the authors can legally appear, except in cases where they have
  given special permission in writing.

  The trademarks 'Swiss Ephemeris' and 'Swiss Ephemeris inside' may be used
  for promoting such software, products or services.
*/

package jp.co.isic.SwissEphemeris;

//import java.lang.*;
//import java.io.*;
//import jp.co.isic.SwissEphemeris.*;

public class SwissMosher extends SwissEphemeris {

  /* default ephemeris used when no ephemeris flagbit is set */
  public static final int SEFLG_DEFAULTEPH = SEFLG_SWIEPH;

  static final int MPC_CERES = 1;
  static final int MPC_PALLAS = 2;
  static final int MPC_JUNO = 3;
  static final int MPC_VESTA = 4;
  static final int MPC_CHIRON = 2060;
  static final int MPC_PHOLUS = 5145;

  public static final int SEI_EPSILON = -2;
  public static final int SEI_NUTATION = -1;
  public static final int SEI_EMB = 0;
  public static final int SEI_EARTH = 0;
  public static final int SEI_SUN = 0;
  public static final int SEI_MOON = 1;
  public static final int SEI_MERCURY = 2;
  public static final int SEI_VENUS = 3;
  public static final int SEI_MARS = 4;
  public static final int SEI_JUPITER = 5;
  public static final int SEI_SATURN = 6;
  public static final int SEI_URANUS = 7;
  public static final int SEI_NEPTUNE = 8;
  public static final int SEI_PLUTO = 9;
  public static final int SEI_SUNBARY = 10;	/* barycentric sun */
  public static final int SEI_ANYBODY = 11;	/* any asteroid */
  public static final int SEI_CHIRON = 12;
  public static final int SEI_PHOLUS = 13;
  public static final int SEI_CERES = 14;
  public static final int SEI_PALLAS = 15;
  public static final int SEI_JUNO = 16;
  public static final int SEI_VESTA = 17;

  public static final int SEI_MEAN_NODE = 0;
  public static final int SEI_TRUE_NODE = 1;
  public static final int SEI_MEAN_APOG = 2;
  public static final int SEI_OSCU_APOG = 3;

  public static final int SEI_FLG_HELIO = 1;
  public static final int SEI_FLG_ROTATE = 2;
  public static final int SEI_FLG_ELLIPSE = 4;
  /* TRUE, if heliocentric earth is given instead of barycentric sun
   * i.e. bary sun is computed from barycentric and heliocentric earth */
  public static final int SEI_FLG_EMBHEL = 8;

  public static final int SEI_FILE_PLANET = 0;
  public static final int SEI_FILE_MOON = 1;
  public static final int SEI_FILE_MAIN_AST = 2;
  public static final int SEI_FILE_ANY_AST = 3;
  public static final int SEI_FILE_FIXSTAR = 4;

  public static final int SEI_FILE_TEST_ENDIAN = (0x616263);	/* abc*/
  public static final int SEI_FILE_BIGENDIAN = 0;
  public static final int SEI_FILE_NOREORD = 0;
  public static final int SEI_FILE_LITENDIAN = 1;
  public static final int SEI_FILE_REORD = 2;

  public static final int SEI_FILE_NMAXPLAN = 50;
  public static final int SEI_FILE_EFPOSBEGIN = 500;

  public static final String SE_FILE_SUFFIX = "se1"; //$NON-NLS-1$

  public static final int SEI_NEPHFILES = 7;
  public static final int SEI_CURR_FPOS = -1;

  /* Chiron's orbit becomes chaotic 
   * before 720 AD and after 4606 AD, because of close encounters
   * with Saturn. Accepting a maximum error of 5 degrees, 
   * the ephemeris is good between the following dates:
   */
  static final double CHIRON_START = 1958470.5;  	/* 1.1.650 */
  static final double CHIRON_END = 3419437.5;  	/* 1.1.4650 */

  /* Pholus's orbit is unstable as well, because he sometimes
   * approaches Saturn.
   * Accepting a maximum error of 5 degrees,
   * the ephemeris is good after the following date:
   */
  static final double PHOLUS_START = 314845.5;  	/* 1.1.-3850 */

  static final double MOSHPLEPH_START = 625000.5;
  static final double MOSHPLEPH_END = 2818000.5;
  static final double MOSHLUEPH_START = 625000.5;
  static final double MOSHLUEPH_END = 2818000.5;
  /* 14 Feb -5410 00:00 ET jul.cal.*/  
  static final double MOSHNDEPH_START = 254900.5;
  /* 11 Dec 5409 00:00 ET, greg. cal */
  static final double MOSHNDEPH_END = 3697000.5;

  static final int MAXORD = 40;
  static final double NCTIES = 6.0;	/* number of centuries per eph. file */

  /* we always use Astronomical Almanac constants, if available */
  static final double MOON_MEAN_DIST = 384400000.0;	/* in m, AA 1996, F2 */
  static final double MOON_MEAN_INCL = 5.1453964;	/* AA 1996, D2 */
  static final double MOON_MEAN_ECC = 0.054900489;	/* AA 1996, F2 */
  /* #define SUN_EARTH_MRAT  328900.5             /* Su/(Ea+Mo) AA 1996, K6 */
  static final double SUN_EARTH_MRAT = 332946.0;	/* Su / (Ea only) */   
  static final double EARTH_MOON_MRAT = (1 / 0.012300034);/* AA 1996, K6 */
  static final double AUNIT = 1.4959787066e+11;	/* au in meters, AA 1996 K6 */
  static final double CLIGHT = 2.99792458e+8;	/* m/s, AA 1996 K6 */
  /* G * M(sun), m^3/sec^2, AA 1996 K6 */
  static final double HELGRAVCONST = 1.32712438e+20;
  /* G * M(earth) m^3/sec^2, AA 1996 K6 */
  static final double GEOGCONST = 3.98600448e+14;
  /* Gaussian gravitational constant K6 */
  static final double KGAUSS = 0.01720209895;
  static final double PRECESSCONST = 50.290966;       	/* AA 1996 K6 */
  /*  Meeus germ. p 391 */
  static final double SUN_RADIUS = (959.63 / 3600 * DEGTORAD);
  static final double EARTH_RADIUS = 6378137.0;		/* AA 1998 K13 */
  /* AA 1998 K13 */
  static final double EARTH_OBLATENESS = (1.0/ 298.257223563);
  /* in rad/day, expl. suppl., p 162 */
  static final double EARTH_ROT_SPEED = (7.2921151467e-5 * 86400);

  /* 8.3167 minutes (days), AA K6 */
  static final double LIGHTTIME_AUNIT = (499.004782/3600/24);

  /* node of ecliptic measured on ecliptic 2000 */
  static final double SSY_PLANE_NODE_E2000 = (107.582569 * DEGTORAD);
  
  /* node of ecliptic measured on solar system rotation plane */
  static final double SSY_PLANE_NODE = (107.58883388 * DEGTORAD);
  
  /* inclination of ecliptic against solar system rotation plane */
  static final double SSY_PLANE_INCL = (1.578701 * DEGTORAD);

  static final double KM_S_TO_AU_CTY = 21.095;		/* km/s to AU/year */
  static final double MOON_SPEED_INTV = 0.00005;/* 4.32 seconds (in days) */
  static final double PLAN_SPEED_INTV = 0.0001;	/* 8.64 seconds (in days) */
  static final double MEAN_NODE_SPEED_INTV = 0.001;		
  static final double NODE_CALC_INTV = 0.0001;		
  static final double NODE_CALC_INTV_MOSH = 0.1;		
  static final double NUT_SPEED_INTV = 0.0001;
  static final double DEFL_SPEED_INTV = 0.0000005;

  static int IS_PLANET = 0;
  static int IS_MOON = 1;
  static int IS_ANY_BODY = 2;
  static int IS_MAIN_ASTEROID = 3;

  static boolean	DO_SAVE = true;
  static boolean	NO_SAVE = false;

  private long epheflag_sv = 0;
  private save_positions sd;

  /*************
   * constants *
   *************/
  static final int pnoext2int[] =
  {SEI_SUN, SEI_MOON, SEI_MERCURY, SEI_VENUS, SEI_MARS, SEI_JUPITER,
   SEI_SATURN, SEI_URANUS, SEI_NEPTUNE, SEI_PLUTO, 0, 0, 0, 0, SEI_EARTH,
   SEI_CHIRON, SEI_PHOLUS, SEI_CERES, SEI_PALLAS, SEI_JUNO, SEI_VESTA, };

  /**************************** 
   * exports from sweph.c 
   ****************************/

  /* planets, moon, nodes etc. */
  /**
   * <code>swe_calc</code> method.
   * The routine called by the user.
   * It checks whether a position for the same planet, the same t, and the
   * same flag bits has already been computed. 
   * If yes, this position is returned. Otherwise it is computed.
   * -> If the SEFLG_SPEED flag has been specified, the speed will be returned
   * at offset 3 of position array x[]. Its precision is probably better 
   * than 0.002"/day.
   * -> If the SEFLG_SPEED3 flag has been specified, the speed will be computed
   * from three positions. This speed is less accurate than SEFLG_SPEED,
   * i.e. better than 0.1"/day. And it is much slower. It is used for 
   * program tests only.
   * -> If no speed flag has been specified, no speed will be returned.
   *
   * @param tjd a <code>double</code> value
   * @param ipl an <code>int</code> value
   * @param iflag an <code>int</code> value
   * @param xx[] a <code>double</code> value
   * @param serr a <code>String</code> value
   * @return an <code>int</code> value
   */
  public int
  swe_calc(double tjd, int ipl, int iflag, double xx[], String serr) {
    int i, j;
    long iflgcoor;
    long iflgsave = iflag;
    long epheflag;
    /*double x[6], *xs, x0[24], x2[24];*/
    double
      x[] = new double[6], x0[] = new double[24], x2[] = new double[24];
    int	xsIndex = 0;
    double dt;
    if ((iflag & SEFLG_JPLEPH) != 0) {
      iflag = iflag & ~SEFLG_EPHMASK;
      iflag |= SEFLG_MOSEPH;
    }
    /* if ephemeris flag != ephemeris flag of last call,
     * we clear the save area, to prevent swecalc() using
     * previously computed data for current calculation.
     * except with ipl = SE_ECL_NUT which is not dependent 
     * on ephemeris, and except if change is from 
     * ephemeris = 0 to ephemeris = SEFLG_DEFAULTEPH
     * or vice-versa.
     */
    epheflag = iflag & SEFLG_EPHMASK;
    if ((epheflag & SEFLG_DEFAULTEPH) != 0)
      epheflag = 0;
    if (epheflag_sv != epheflag && ipl != SE_ECL_NUT) {
      swe_close();
      epheflag_sv = epheflag;
    }
    /* high precision speed prevails fast speed */
    if ((iflag & SEFLG_SPEED3) != 0 && (iflag & SEFLG_SPEED) != 0)
      iflag = iflag & ~SEFLG_SPEED3;
    /* cartesian flag excludes radians flag */
    if ((iflag & SEFLG_XYZ) != 0 && (iflag & SEFLG_RADIANS) != 0)
      iflag = iflag & ~SEFLG_RADIANS;
    /* pointer to save area */
    if (ipl < SE_NPLANETS && ipl >= SE_SUN)
      sd = swed.savedat[ipl];
    else
      /* other bodies, e.g. asteroids called with ipl = SE_AST_OFFSET + MPC# */
      sd = swed.savedat[SE_NPLANETS];
    /* 
     * if position is available in save area, it is returned.
     * this is the case, if tjd = tsave and iflag = iflgsave.
     * coordinate flags can be neglected, because save area 
     * provides all coordinate types.
     * if ipl > SE_AST(EROID)_OFFSET, ipl must be checked, 
     * because all asteroids called by MPC number share the same
     * save area.
     */ 
    iflgcoor = SEFLG_EQUATORIAL | SEFLG_XYZ | SEFLG_RADIANS;
    if (sd.tsave == tjd && tjd != 0 && ipl == sd.ipl) {
      if ((sd.iflgsave & ~iflgcoor) == (iflag & ~iflgcoor)) 
	/* goto end_swe_calc;*/
	if ((iflag & SEFLG_EQUATORIAL) != 0)
	  /*xs = sd.xsaves[12];	/* equatorial coordinates */
	  xsIndex = 12;
	else
	  /*xs = sd.xsaves[0];	/* ecliptic coordinates */
	  xsIndex = 0;
      if ((iflag & SEFLG_XYZ) != 0)
	/*xs = xs[6];		/* cartesian coordinates */
	xsIndex = 6;
      if (ipl == SE_ECL_NUT)
	i = 4;
      else
	i = 3;
      for (j = 0; j < i; j++)
	/*x[j] = xs[j];*/
	x[j] = sd.xsaves[xsIndex+j];
      for (j = i; j < 6; j++)
	x[j] = 0;
      if ((iflag & (SEFLG_SPEED3 | SEFLG_SPEED)) != 0) {
	for (j = 3; j < 6; j++)
	  /*x[j] = xs[j];*/
	  x[j] = sd.xsaves[xsIndex+j];
      }
      if ((iflag & SEFLG_RADIANS) != 0) {
	if (ipl == SE_ECL_NUT) {
	  for (j = 0; j < 4; j++)
	    x[j] *= DEGTORAD;
	} else {
	  for (j = 0; j < 2; j++)
	    x[j] *= DEGTORAD;
	  if ((iflag & (SEFLG_SPEED3 | SEFLG_SPEED)) != 0) {
	    for (j = 3; j < 5; j++) 
	      x[j] *= DEGTORAD;
	  }
	}
      }
      for (i = 0; i <= 5; i++)
	xx[i] = x[i];
      iflag = (int)sd.iflgsave;
      /* if no ephemeris has been specified, do not return chosen ephemeris */
      if ((iflgsave & SEFLG_EPHMASK) == 0)
	iflag = iflag & ~SEFLG_DEFAULTEPH;
      return iflag;
    }
    /* 
     * otherwise, new position must be computed 
     */
    if ((iflag & SEFLG_SPEED3) == 0) {
      /* 
       * with high precision speed from one call of swecalc() 
       * (FAST speed)
       */
      sd.tsave = tjd;
      sd.ipl = ipl;
      if ((sd.iflgsave = swecalc(tjd, ipl, iflag, sd.xsaves, serr)) == ERR) {
	/*goto return_error;*/
	/*return_error:*/
	for (i = 0; i <= 5; i++)
	  xx[i] = 0;
	return ERR;
      }
    } else {
      /* 
       * with speed from three calls of swecalc(), slower and less accurate.
       * (SLOW speed, for test only)
       */
      sd.tsave = tjd;
      sd.ipl = ipl;
      switch(ipl) {
      case SE_MOON:
	dt = MOON_SPEED_INTV;
	break;
      case SE_OSCU_APOG:
      case SE_TRUE_NODE:
	/* this is the optimum dt with Moshier ephemeris, but not with
	 * JPL ephemeris or SWISSEPH. To avoid completely false speed
	 * in case that JPL is wanted but the program returns Moshier,
	 * we use Moshier optimum.
	 * For precise speed, use JPL and FAST speed computation,
	 */
	dt = NODE_CALC_INTV_MOSH;
	break;
      default:
	dt = PLAN_SPEED_INTV;
	break;
      } 
      if ((sd.iflgsave = swecalc(tjd-dt, ipl, iflag, x0, serr)) == ERR) {
	/*goto return_error;*/
	/*return_error:*/
	for (i = 0; i <= 5; i++)
	  xx[i] = 0;
	return ERR;
      }
      if ((sd.iflgsave = swecalc(tjd+dt, ipl, iflag, x2, serr)) == ERR) {
	/*goto return_error;*/
	/*return_error:*/
	for (i = 0; i <= 5; i++)
	  xx[i] = 0;
	return ERR;
      }
      if ((sd.iflgsave = swecalc(tjd, ipl, iflag, sd.xsaves, serr)) == ERR) {
	/*goto return_error;*/
	/*return_error:*/
	for (i = 0; i <= 5; i++)
	  xx[i] = 0;
	return ERR;
      }
      denormalize_positions(x0, sd.xsaves, x2);
      calc_speed(x0, sd.xsaves, x2, dt);
    }
    /*end_swe_calc:*/
    if ((iflag & SEFLG_EQUATORIAL) != 0)
      /*xs = sd.xsaves[12];	/* equatorial coordinates */
      xsIndex = 12;
    else
      /*xs = sd.xsaves[0];	/* ecliptic coordinates */
      xsIndex = 0;
    if ((iflag & SEFLG_XYZ) != 0)
      /*xs = xs[6];		/* cartesian coordinates */
      xsIndex = 6;
    if (ipl == SE_ECL_NUT)
      i = 4;
    else
      i = 3;
    for (j = 0; j < i; j++)
      /*x[j] = xs[j];*/
      x[j] = sd.xsaves[xsIndex+j];
    for (j = i; j < 6; j++)
      x[j] = 0;
    if ((iflag & (SEFLG_SPEED3 | SEFLG_SPEED)) != 0) {
      for (j = 3; j < 6; j++)
	/*x[j] = xs[j];*/
	x[j] = sd.xsaves[xsIndex+j];
    }
    if ((iflag & SEFLG_RADIANS) != 0) {
      if (ipl == SE_ECL_NUT) {
	for (j = 0; j < 4; j++)
	  x[j] *= DEGTORAD;
      } else {
	for (j = 0; j < 2; j++)
	  x[j] *= DEGTORAD;
	if ((iflag & (SEFLG_SPEED3 | SEFLG_SPEED)) != 0) {
	  for (j = 3; j < 5; j++) 
	    x[j] *= DEGTORAD;
	}
      }
    }
    for (i = 0; i <= 5; i++)
      xx[i] = x[i];
    iflag = (int)sd.iflgsave;
    /* if no ephemeris has been specified, do not return chosen ephemeris */
    if ((iflgsave & SEFLG_EPHMASK) == 0)
      iflag = iflag & ~SEFLG_DEFAULTEPH;
    return iflag;
  }

  public int
  swe_calc_ut(double tjd_ut, int ipl, int iflag, double xx[], String serr) {
    return swe_calc(tjd_ut
		    + SwephLib.swe_deltat(tjd_ut), ipl, iflag, xx, serr);
  }

  /**
   * Describe <code>swecalc</code> method here.
   *
   * @param tjd a <code>double</code> value
   * @param ipl an <code>int</code> value
   * @param iflag a <code>long</code> value
   * @param x[] a <code>double</code> value
   * @param serr a <code>String</code> value
   * @return a <code>long</code> value
   */
  long swecalc(double tjd, int ipl, long iflag, double x[], String serr) {
    int i;
    int ipli, ipli_ast, ifno;
    int retc;
    long epheflag = SEFLG_DEFAULTEPH;
    plan_data pdp;
    plan_data pedp = swed.pldat[SEI_EARTH];
    plan_data psdp = swed.pldat[SEI_SUNBARY];
    plan_data ndp;
    /*double xp[], xp2[];*/
    double	xp[];
    String	serr2;
    serr = ""; //$NON-NLS-1$
    serr2 = ""; //$NON-NLS-1$
    /****************************************** 
     * iflag plausible?                       * 
     ******************************************/
    iflag = plaus_iflag(iflag);
    /****************************************** 
     * which ephemeris is wanted, which is used?
     * Three ephemerides are possible: MOSEPH, SWIEPH, JPLEPH.
     * JPLEPH is best, SWIEPH is nearly as good, MOSEPH is least precise.
     * The availability of the various ephemerides depends on the installed
     * ephemeris files in the users ephemeris directory. This can change at
     * any time.
     * Swisseph should try to fulfil the wish of the user for a specific
     * ephemeris, but use a less precise one if the desired ephemeris is not
     * available for the given date and body. 
     * If internal ephemeris errors are detected
     * (data error, file length error) an error is returned.
     * If the time range is bad but another ephemeris can deliver this range,
     * the other ephemeris is used.
     * If no ephemeris is specified, DEFAULTEPH is assumed as desired.
     * DEFAULTEPH is defined at compile time, usually as JPLEPH.
     * The caller learns from the return flag which ephemeris was used.
     * ephe_flag is extracted from iflag, but can change later if the
     * desired ephe is not available.
     ******************************************/
    if ((iflag & SEFLG_MOSEPH) != 0)
      epheflag = SEFLG_MOSEPH;
    /*
    if (iflag & SEFLG_SWIEPH)
      epheflag = SEFLG_SWIEPH;
    if (iflag & SEFLG_JPLEPH)
      epheflag = SEFLG_JPLEPH;
    */
    /* no barycentric calculations with Moshier ephemeris */
    if ((iflag & SEFLG_BARYCTR) != 0 && (iflag & SEFLG_MOSEPH) != 0) {
      serr = "barycentric Moshier positions are not supported."; //$NON-NLS-1$
      return ERR;
    }
    if ((iflag & SEFLG_SIDEREAL) != 0 && !swed.ayana_is_set)
      swe_set_sid_mode(SE_SIDM_FAGAN_BRADLEY, 0, 0);
    /****************************************** 
     * obliquity of ecliptic 2000 and of date * 
     ******************************************/
    swi_check_ecliptic(tjd);
    /******************************************
     * nutation                               * 
     ******************************************/
    swi_check_nutation(tjd, iflag);
    /****************************************** 
     * select planet and ephemeris            * 
     *                                        * 
     * ecliptic and nutation                  * 
     ******************************************/
    if (ipl == SE_ECL_NUT) {
      x[0] = swed.oec.eps + swed.nut.nutlo[1];	/* true ecliptic */
      x[1] = swed.oec.eps;			/* mean ecliptic */
      x[2] = swed.nut.nutlo[0];		/* nutation in longitude */
      x[3] = swed.nut.nutlo[1];		/* nutation in obliquity */
      /*if ((iflag & SEFLG_RADIANS) == 0)*/
      for (i = 0; i <= 3; i++)
	x[i] *= RADTODEG;
      return(iflag);
    }
    /****************************************** 
     * moon                                   * 
     ******************************************/
    else if (ipl == SE_MOON) {
      /* internal planet number */
      ipli = SEI_MOON;
      pdp = swed.pldat[ipli];
      xp = pdp.xreturn;
      switch((int)epheflag) {
      case SEFLG_MOSEPH:
	retc = swi_moshmoon(tjd, DO_SAVE, null, serr);/**/
	if (retc == ERR) {
	  /*goto return_error;*/
	  /*return_error:;*/
	  for (i = 0; i < 24; i++)
	    x[i] = 0;
	  return ERR;
	}
	/* for hel. position, we need earth as well */
	retc = swi_moshplan(tjd, SEI_EARTH, DO_SAVE, null, null, serr);/**/
	if (retc == ERR) {
	  /*goto return_error;*/
	  /*return_error:;*/
	  for (i = 0; i < 24; i++)
	    x[i] = 0;
	  return ERR;
	}
	break;
      default:
	break;
      } 
      /* heliocentric, lighttime etc. */
      if ((retc = app_pos_etc_moon(iflag, serr)) != OK) {
	/*goto return_error; /* retc may be wrong with sidereal calculation */
	/*return_error:;*/
	for (i = 0; i < 24; i++)
	  x[i] = 0;
	return ERR;
      }
    }
    else if (ipl == SE_SUN 	/* main planet */
	     || ipl == SE_MERCURY
	     || ipl == SE_VENUS
	     || ipl == SE_MARS
	     || ipl == SE_JUPITER
	     || ipl == SE_SATURN
	     || ipl == SE_URANUS
	     || ipl == SE_NEPTUNE
	     || ipl == SE_PLUTO
	     || ipl == SE_EARTH) {
      if ((iflag & SEFLG_HELCTR) != 0) {
	if (ipl == SE_SUN) {
	  /* heliocentric position of Sun does not exist */
	  for (i = 0; i < 24; i++)
	    x[i] = 0;
	  return iflag;
	} 
      } else if ((iflag & SEFLG_BARYCTR) != 0) {
	;
      } else {		/* geocentric */
	if (ipl == SE_EARTH) {
	  /* geocentric position of Earth does not exist */
	  for (i = 0; i < 24; i++)
	    x[i] = 0;
	  return iflag;
	}
      }
      /* internal planet number */
      ipli = pnoext2int[ipl];
      pdp = swed.pldat[ipli];
      xp = pdp.xreturn;
      retc = main_planet(tjd, ipli, epheflag, iflag, serr);
      if (retc == ERR) {
	/*goto return_error;*/
	/*return_error:;*/
	for (i = 0; i < 24; i++)
	  x[i] = 0;
	return ERR;
      }
      /* iflag has possibly changed in main_planet() */
      iflag = pdp.xflgs;
    }
    /*********************i************************ 
     * mean lunar node                            * 
     * for comment s. moshmoon.c, swi_mean_node() *
     **********************************************/
    else if (ipl == SE_MEAN_NODE) {
      if ((iflag & SEFLG_HELCTR) != 0 || (iflag & SEFLG_BARYCTR) != 0) {
	/* heliocentric/barycentric lunar node not allowed */
	for (i = 0; i < 24; i++)
	  x[i] = 0;
	return iflag;
      }
      ndp = swed.nddat[SEI_MEAN_NODE];
      xp = ndp.xreturn;
      /*xp2 = ndp.x;*/
      retc = swi_mean_node(tjd, ndp.x, 0, serr);
      if (retc == ERR) {
	/*goto return_error;*/
	/*return_error:;*/
	for (i = 0; i < 24; i++)
	  x[i] = 0;
	return ERR;
      }
      /* speed (is almost constant; variation < 0.001 arcsec) */
      /*retc = swi_mean_node(tjd - MEAN_NODE_SPEED_INTV, xp2+3, serr);*/
      retc = swi_mean_node(tjd - MEAN_NODE_SPEED_INTV,
			   ndp.x, 3, serr);
      if (retc == ERR) {
	/*goto return_error;*/
	for (i = 0; i < 24; i++)
	  x[i] = 0;
	return ERR;
      }
      /*
      xp2[3] = swe_difrad2n(xp2[0], xp2[3]) / MEAN_NODE_SPEED_INTV;
      xp2[4] = xp2[5] = 0;
      */
      ndp.x[3] =
	SwephLib.swe_difrad2n(ndp.x[0], ndp.x[3]) / MEAN_NODE_SPEED_INTV;
      ndp.x[4] = ndp.x[5] = 0;
      ndp.teval = tjd;
      ndp.xflgs = -1; 	
      /* lighttime etc. */
      if ((retc = app_pos_etc_mean(SEI_MEAN_NODE, iflag, serr)) != OK) {
	/*goto return_error;*/
	for (i = 0; i < 24; i++)
	  x[i] = 0;
	return ERR;
      }
      /* to avoid infinitesimal deviations from latitude = 0 
       * that result from conversions */
      if ((iflag & SEFLG_SIDEREAL) == 0 && (iflag & SEFLG_J2000) == 0) {
	ndp.xreturn[1] = 0.0;	/* ecl. latitude       */
	ndp.xreturn[4] = 0.0;	/*               speed */
	ndp.xreturn[5] = 0.0;	/*      radial   speed */
	ndp.xreturn[8] = 0.0;	/* z coordinate        */
	ndp.xreturn[11] = 0.0;	/*               speed */
      }
      if (retc == ERR) {
	/*goto return_error;*/
	for (i = 0; i < 24; i++)
	  x[i] = 0;
	return ERR;
      }
    }
    /********************************************** 
     * mean lunar apogee ('dark moon', 'lilith')  *
     * for comment s. moshmoon.c, swi_mean_apog() *
     **********************************************/
    else if (ipl == SE_MEAN_APOG) {
      if ((iflag & SEFLG_HELCTR) != 0 || (iflag & SEFLG_BARYCTR) != 0) {
	/* heliocentric/barycentric lunar apogee not allowed */
	for (i = 0; i < 24; i++)
	  x[i] = 0;
	return iflag;
      }
      ndp = swed.nddat[SEI_MEAN_APOG];
      xp = ndp.xreturn;
      /*xp2 = ndp.x;*/
      retc = swi_mean_apog(tjd, ndp.x, 0, serr);
      if (retc == ERR) {
	/*goto return_error;*/
	for (i = 0; i < 24; i++)
	  x[i] = 0;
	return ERR;
      }
      /* speed (is not constant! variation ~= several arcsec) */
      /*retc = swi_mean_apog(tjd - MEAN_NODE_SPEED_INTV, xp2+3, serr);*/
      retc = swi_mean_apog(tjd - MEAN_NODE_SPEED_INTV, ndp.x, 3, serr);
      if (retc == ERR) {
	/*goto return_error;*/
	for (i = 0; i < 24; i++)
	  x[i] = 0;
	return ERR;
      }
      for(i = 0; i <= 1; i++)
	/*xp2[3+i] = swe_difrad2n(xp2[i], xp2[3+i]) / MEAN_NODE_SPEED_INTV;*/
	ndp.x[3+i] =
	  SwephLib.swe_difrad2n(ndp.x[i], ndp.x[3+i]) / MEAN_NODE_SPEED_INTV;
      /*xp2[5] = 0;*/
      ndp.x[5] = 0;
      ndp.teval = tjd;
      ndp.xflgs = -1; 	
      /* lighttime etc. */
      if ((retc = app_pos_etc_mean(SEI_MEAN_APOG, iflag, serr)) != OK) {
	/*goto return_error;*/
	for (i = 0; i < 24; i++)
	  x[i] = 0;
	return ERR;
      }
      /* to avoid infinitesimal deviations from r-speed = 0 
       * that result from conversions */
      ndp.xreturn[5] = 0.0;	/*               speed */
      if (retc == ERR) {
	/*goto return_error;*/
	for (i = 0; i < 24; i++)
	  x[i] = 0;
	return ERR;
      }
    }
    /*********************************************** 
     * osculating lunar node ('true node')         *    
     ***********************************************/
    else if (ipl == SE_TRUE_NODE) {
      if ((iflag & SEFLG_HELCTR) != 0 || (iflag & SEFLG_BARYCTR) != 0) {
	/* heliocentric/barycentric lunar node not allowed */
	for (i = 0; i < 24; i++)
	  x[i] = 0;
	return iflag;
      }
      ndp = swed.nddat[SEI_TRUE_NODE];
      xp = ndp.xreturn;
      retc = lunar_osc_elem(tjd, SEI_TRUE_NODE, iflag, serr); 
      iflag = ndp.xflgs;
      /* to avoid infinitesimal deviations from latitude = 0 
       * that result from conversions */
      if ((iflag & SEFLG_SIDEREAL) == 0 && (iflag & SEFLG_J2000) == 0) {
	ndp.xreturn[1] = 0.0;	/* ecl. latitude       */
	ndp.xreturn[4] = 0.0;	/*               speed */
	ndp.xreturn[8] = 0.0;	/* z coordinate        */
	ndp.xreturn[11] = 0.0;	/*               speed */
      }
      if (retc == ERR) {
	/*goto return_error;*/
	for (i = 0; i < 24; i++)
	  x[i] = 0;
	return ERR;
      }
    }
    /*********************************************** 
     * osculating lunar apogee                     *    
     ***********************************************/
    else if (ipl == SE_OSCU_APOG) {
      if ((iflag & SEFLG_HELCTR) != 0 || (iflag & SEFLG_BARYCTR) != 0) {
	/* heliocentric/barycentric lunar apogee not allowed */
	for (i = 0; i < 24; i++)
	  x[i] = 0;
	return iflag;
      }
      ndp = swed.nddat[SEI_OSCU_APOG];
      xp = ndp.xreturn;
      retc = lunar_osc_elem(tjd, SEI_OSCU_APOG, iflag, serr); 
      iflag = ndp.xflgs;
      if (retc == ERR) {
	/*goto return_error;*/
	for (i = 0; i < 24; i++)
	  x[i] = 0;
	return ERR;
      }
    }
    /*********************************************** 
     * minor planets                               *    
     ***********************************************/
    else if (ipl == SE_CHIRON 
	     || ipl == SE_PHOLUS
	     || ipl == SE_CERES		/* Ceres - Vesta */
	     || ipl == SE_PALLAS		
	     || ipl == SE_JUNO	
	     || ipl == SE_VESTA
	     || ipl > SE_AST_OFFSET) {			
      /* internal planet number */
      if (ipl < SE_NPLANETS) 
	ipli = pnoext2int[ipl];
      else if (ipl <= SE_AST_OFFSET + MPC_VESTA) {
	ipli = SEI_CERES + ipl - SE_AST_OFFSET - 1;
	ipl = SE_CERES + ipl - SE_AST_OFFSET - 1;
      } else 			/* any asteroid except*/
	ipli = SEI_ANYBODY;
      if (ipli == SEI_ANYBODY)
	ipli_ast = ipl;
      else
	ipli_ast = ipli;
      pdp = swed.pldat[ipli];
      xp = pdp.xreturn;
      if (ipli_ast > SE_AST_OFFSET)
	ifno = SEI_FILE_ANY_AST;
      else
	ifno = SEI_FILE_MAIN_AST;
      if (ipli == SEI_CHIRON && (tjd < CHIRON_START || tjd > CHIRON_END)) {
	  serr = "Chiron's ephemeris is restricted to JD "; //$NON-NLS-1$
	  serr += CHIRON_START;
	  serr += " - JD "; //$NON-NLS-1$
	  serr += CHIRON_END;
	/*
	if (serr != NULL)
	  sprintf(serr,
		  "Chiron's ephemeris is restricted to JD %8.1f - JD %8.1f",
		  CHIRON_START, CHIRON_END);
	*/
	return ERR;
      }
      if (ipli == SEI_PHOLUS && tjd < PHOLUS_START) {
	serr = "Pholus's ephemeris is restricted to the time after JD "; //$NON-NLS-1$
	serr += PHOLUS_START;
	/*
	if (serr != NULL)
	  sprintf(serr, 
		  "Pholus's ephemeris is restricted to the time after JD %8.1f",
		  PHOLUS_START);
	*/
	return ERR;
      }
      /*do_asteroid:*/
      do {
	/* earth and sun are also needed */
	retc = main_planet(tjd, SEI_EARTH, epheflag, iflag, serr);
	if (retc == ERR) {
	  /*goto return_error;*/
	  for (i = 0; i < 24; i++)
	    x[i] = 0;
	  return ERR;
	}
	/* iflag (ephemeris bit) has possibly changed in main_planet() */
	iflag = swed.pldat[SEI_EARTH].xflgs;
	/* asteroid */
	if (serr != null && serr != "") { //$NON-NLS-1$
	  serr2 = serr;
	  serr = ""; //$NON-NLS-1$
	}
	/* asteroid */
	retc = sweph(tjd, ipli_ast, ifno, iflag, psdp.x, DO_SAVE, null, serr);
	if (retc == ERR || retc == NOT_AVAILABLE) {
	  /*goto return_error;*/
	  for (i = 0; i < 24; i++)
	    x[i] = 0;
	  return ERR;
	}
	retc = app_pos_etc_plan(ipli_ast, iflag, serr);
	if (retc == ERR) {
	  /*goto return_error;*/
	  for (i = 0; i < 24; i++)
	    x[i] = 0;
	  return ERR;
	}
	/* app_pos_etc_plan() might have failed, if t(light-time)
	 * is beyond ephemeris range. in this case redo with Moshier 
	 */
	if (retc != NOT_AVAILABLE && retc != BEYOND_EPH_LIMITS) {
	  break;
	}
	if (epheflag == SEFLG_MOSEPH) { 
	  /*goto return_error;*/
	  for (i = 0; i < 24; i++)
	    x[i] = 0;
	  return ERR;
	}
	iflag = (iflag & ~SEFLG_EPHMASK) | SEFLG_MOSEPH;
	epheflag = SEFLG_MOSEPH;
	/*strcat(serr, "\nusing Moshier eph.; ");*/
	serr += "\nusing Moshier eph.; "; //$NON-NLS-1$
      }
      while (true);
      /*goto do_asteroid;*/

      /* add warnings from earth/sun computation */
      serr = "sun: "; //$NON-NLS-1$
      if (serr2 != "") { //$NON-NLS-1$
	serr2 = serr2.substring(0, AS_MAXCH-5);
	serr += serr2;
      }
    }
    /*********************************************** 
     * fictitious planets                          *    
     * (Isis-Transpluto and Uranian planets)       *
     ***********************************************/
    else if (ipl >= SE_FICT_OFFSET && ipl <= SE_FICT_MAX) {
      /* internal planet number */
      ipli = SEI_ANYBODY;
      pdp = swed.pldat[ipli];
      xp = pdp.xreturn;

      /*do_fict_plan:*/
      do {
	/* the earth for geocentric position */
	retc = main_planet(tjd, SEI_EARTH, epheflag, iflag, serr);
	/* iflag (ephemeris bit) has possibly changed in main_planet() */
	iflag = swed.pldat[SEI_EARTH].xflgs;
	/* planet from osculating elements */
	if (swi_osc_el_plan(tjd, pdp.x, ipl-SE_FICT_OFFSET,
			    ipli, pedp.x, psdp.x, serr) != OK) {
	  /*goto return_error;*/
	  for (i = 0; i < 24; i++)
	    x[i] = 0;
	  return ERR;
	}
	if (retc == ERR) {
	  /*goto return_error;*/
	  for (i = 0; i < 24; i++)
	    x[i] = 0;
	  return ERR;
	}
	retc = app_pos_etc_plan_osc(ipl, ipli, iflag, serr);
	if (retc == ERR) {
	  /*goto return_error;*/
	  for (i = 0; i < 24; i++)
	    x[i] = 0;
	  return ERR;
	}
	/* app_pos_etc_plan_osc() might have failed, if t(light-time)
	 * is beyond ephemeris range. in this case redo with Moshier 
	 */
	if (retc != NOT_AVAILABLE && retc != BEYOND_EPH_LIMITS) {
	  break;
	}
	if (epheflag == SEFLG_MOSEPH) { 
	  /*goto return_error;*/
	  for (i = 0; i < 24; i++)
	    x[i] = 0;
	  return ERR;
	}
	iflag = (iflag & ~SEFLG_EPHMASK) | SEFLG_MOSEPH;
	epheflag = SEFLG_MOSEPH;
	if (serr != null && serr.length() + 30 < AS_MAXCH)
	  /*strcat(serr, "\nusing Moshier eph.; ");*/
	  serr += "\nusing Moshier eph.; "; //$NON-NLS-1$
	/*goto do_fict_plan;*/
      }
      while (true);
    }
    /*********************************************** 
     * invalid body number                         *    
     ***********************************************/
    else {
      serr = "illegal planet number "; //$NON-NLS-1$
      serr += ipl;
      serr = "."; //$NON-NLS-1$
      /*
      if (serr != null) {
	sprintf(serr, "illegal planet number %d.", ipl);
      }
      */
      /*goto return_error;*/
      for (i = 0; i < 24; i++)
	x[i] = 0;
      return ERR;
    }
    for (i = 0; i < 24; i++)
      x[i] = xp[i];
    return(iflag);
  }

  /* fixed stars */
  /**
   * <code>swe_fixstar</code> method
   * NOP stub
   *
   * @param star a <code>String</code> value
   * @param tjd a <code>double</code> value
   * @param iflag an <code>int</code> value
   * @param xx[] a <code>double</code> value
   * @param serr a <code>String</code> value
   * @return an <code>int</code> value
   */
  public int
  swe_fixstar(String star, double tjd, int iflag, double xx[], String serr) {
    return 0;
  }

  /**
   * Describe <code>swe_fixstar_ut</code> method here.
   *
   * @param star a <code>String</code> value
   * @param tjd a <code>double</code> value
   * @param iflag an <code>int</code> value
   * @param xx[] a <code>double</code> value
   * @param serr a <code>String</code> value
   * @return an <code>int</code> value
   */
  public int
    swe_fixstar_ut(String star, double tjd_ut, int iflag,
		   double xx[], String serr) {
    return swe_fixstar(star, tjd_ut + SwephLib.swe_deltat(tjd_ut),
		       iflag, xx, serr);
  }

  /**
   * <code>swe_close</code> method close Swiss Ephemeris.
   *
   */
  public void swe_close() {
    int i;
    /* free planets data space */
    for (i = 0; i < SEI_NPLANETS; i++) {
      /*
      if (swed.pldat[i].segp != NULL) {
	free((void *) swed.pldat[i].segp);
      }
      if (swed.pldat[i].refep != NULL) {
	free((void *) swed.pldat[i].refep);
      }
      memset((void *) &swed.pldat[i], 0, sizeof(struct plan_data));
      */
      swed.pldat[i].clear();
    }
    /* free planets data space */
    for (i = 0; i <= SE_NPLANETS; i++) /* "<=" is correct! see decl. */
      /*memset((void *) &swed.savedat[i], 0, sizeof(struct save_positions));*/
      swed.savedat[i].clear();
    /* clear node data space */
    for (i = 0; i < SEI_NNODE_ETC; i++) {
      /*memset((void *) &swed.nddat[i], 0, sizeof(struct plan_data));*/
      swed.nddat[i].clear();
    }
    /*
    memset((void *) &swed.oec, 0, sizeof(struct epsilon));
    memset((void *) &swed.oec2000, 0, sizeof(struct epsilon));
    memset((void *) &swed.nut, 0, sizeof(struct nut));
    memset((void *) &swed.nut2000, 0, sizeof(struct nut));
    memset((void *) &swed.nutv, 0, sizeof(struct nut));
    */
    swed.oec.clear();
    swed.oec2000.clear();
    swed.nut.clear();
    swed.nut2000.clear();
    swed.nutv.clear();
  }

  /* calculates obliquity of ecliptic and stores it together
   * with its date, sine, and cosine
   */
  static void calc_epsilon(double tjd, epsilon e)
  {
    e.teps = tjd;
    e.eps = SwephLib.swi_epsiln(tjd);
    e.seps = Math.sin(e.eps);
    e.ceps = Math.cos(e.eps);
  }

  /**
   * <code>main_planet</code> method
   * computes a main planet from any ephemeris, if it 
   * has not yet been computed for this date.
   * since a geocentric position requires the earth, the
   * earth's position will be computed as well. With SWISSEPH
   * files the barycentric sun will be done as well.
   * With Moshier, the moon will be done as well.
   *
   * tjd 		= julian day
   * ipli		= body number
   * epheflag	= which ephemeris? JPL, SWISSEPH, Moshier?
   * iflag	= other flags
   *
   * the geocentric apparent position of ipli (or whatever has
   * been specified in iflag) will be saved in
   * &swed.pldat[ipli].xreturn[];
   *
   * the barycentric (heliocentric with Moshier) position J2000
   * will be kept in 
   * &swed.pldat[ipli].x[];
   *
   * @param tjd a <code>double</code> value
   * @param ipli an <code>int</code> value
   * @param epheflag a <code>long</code> value
   * @param iflag a <code>long</code> value
   * @param serr a <code>String</code> value
   * @return an <code>int</code> value
   */
  int main_planet(double tjd, int ipli, long epheflag, long iflag,
			 String serr)
  {
    int retc;
    switch((int)epheflag) {
    case SEFLG_MOSEPH:
      retc = swi_moshplan(tjd, ipli, DO_SAVE, null, null, serr);/**/
      if (retc == ERR)
	return ERR;
      /* geocentric, lighttime etc. */
      if (ipli == SEI_SUN)
	retc = app_pos_etc_sun(iflag, serr)/**/;
      else
	retc = app_pos_etc_plan(ipli, iflag, serr);
      if (retc == ERR)
	return ERR;
      break;
    default:
      break;
    } 
    return OK;
  }

  /**
   * <code>main_planet_bary</code> method
   * computes a main planet from any ephemeris or returns
   * it again, if it has been computed before.
   * In barycentric equatorial position of the J2000 equinox.
   * The earth's position is computed as well. With SWISSEPH
   * and JPL ephemeris the barycentric sun is computed, too.
   * With Moshier, the moon is returned, as well.
   *
   * tjd 		= julian day
   * ipli		= body number
   * epheflag	= which ephemeris? JPL, SWISSEPH, Moshier?
   * iflag	= other flags
   * xp, xe, xs, and xm are the pointers, where the program
   * either finds or stores (if not found) the barycentric 
   * (heliocentric with Moshier) positions of the following 
   * bodies:
   * xp		planet
   * xe		earth
   * xs		sun
   * xm		moon
   * 
   * xm is used with Moshier only 
   *
   * @param tjd a <code>double</code> value
   * @param ipli an <code>int</code> value
   * @param epheflag a <code>long</code> value
   * @param iflag a <code>long</code> value
   * @param do_save a <code>boolean</code> value
   * @param xp[] a <code>double</code> value
   * @param xe[] a <code>double</code> value
   * @param xs[] a <code>double</code> value
   * @param xm[] a <code>double</code> value
   * @param serr a <code>String</code> value
   * @return an <code>int</code> value
   */
  int main_planet_bary(double tjd, int ipli,
			      long epheflag, long iflag, boolean do_save,
			      double xp[], double xe[],
			      double xs[], double xm[],
			      String serr)
  {
    int i, retc;
    switch((int)epheflag) {
    case SEFLG_MOSEPH:
      retc = swi_moshplan(tjd, ipli, do_save, xp, xe, serr);/**/
      if (retc == ERR)
	return ERR;
      for (i = 0; i <= 5; i++)
	xs[i] = 0;
      break;
    default:
      break;
    } 
    return OK;
  }

  /**
   * <code>sweph</code> method
   * NOP stub
   *
   * this function looks for an ephemeris file, 
   * opens it, if not yet open,
   * reads constants, if not yet read,
   * computes a planet, if not yet computed 
   * attention: asteroids are heliocentric
   *            other planets barycentric
   * 
   * tjd 		julian date
   * ipli		SEI_ planet number
   * ifno		ephemeris file number
   * xsunb	INPUT (!) array of 6 doubles containing barycentric sun
   *              (must be given with asteroids)
   * do_save	boolean: save result in save area
   * xp		return array of 6 doubles for planet's position
   * serr		error string
   *
   * @param tjd a <code>double</code> value
   * @param ipli an <code>int</code> value
   * @param ifno an <code>int</code> value
   * @param iflag a <code>long</code> value
   * @param xsunb[] a <code>double</code> value
   * @param do_save a <code>boolean</code> value
   * @param xpret[] a <code>double</code> value
   * @param serr a <code>String</code> value
   * @return an <code>int</code> value
   */
  int sweph(double tjd, int ipli, int ifno, long iflag,
	    double xsunb[], boolean do_save, double xpret[], String serr)
  {
    return NOT_AVAILABLE;
  }
  
  /**
   * <code>app_pos_etc_plan</code> method
   * converts planets from barycentric to geocentric,
   * apparent positions
   * precession and nutation
   * according to flags
   * ipli		planet number
   * iflag	flags
   * serr         error string
   *
   * @param ipli an <code>int</code> value
   * @param iflag a <code>long</code> value
   * @param serr a <code>String</code> value
   * @return an <code>int</code> value
   */
  int app_pos_etc_plan(int ipli, long iflag, String serr)
  {
    int i, j, niter, retc = OK;
    int  ifno, ibody;//,ipl;
    long flg1, flg2;
    double xx[], dx[], dt, t, dtsave_for_defl;
    xx = new double[6];
    dx = new double[3];
    double xobs[], xobs2[];
    xobs = new double[6];
    xobs2 = new double[6];
    double xearth[];//, xsun[];
    xearth = new double[6];
    //xsun = new double[6];
    double xxsp[], xxsv[];
    xxsp = new double[6];
    xxsv = new double[6];
    plan_data pedp = swed.pldat[SEI_EARTH];
    plan_data pdp;
    epsilon oe = swed.oec2000;
    long epheflag = iflag & SEFLG_EPHMASK;
    t = dtsave_for_defl = 0;	/* dummy assignment to silence gcc */
    /* ephemeris file */
    if (ipli > SE_AST_OFFSET) {
      ifno = SEI_FILE_ANY_AST;	
      ibody = IS_ANY_BODY;
      pdp = swed.pldat[SEI_ANYBODY];
    } else if (ipli == SEI_CHIRON
	       || ipli == SEI_PHOLUS
	       || ipli == SEI_CERES
	       || ipli == SEI_PALLAS
	       || ipli == SEI_JUNO
	       || ipli == SEI_VESTA) {
      ifno = SEI_FILE_MAIN_AST;	
      ibody = IS_MAIN_ASTEROID;
      pdp = swed.pldat[ipli];
    } else {
      ifno = SEI_FILE_PLANET;
      ibody = IS_PLANET;
      pdp = swed.pldat[ipli];
    }
    /* if the same conversions have already been done for the same 
     * date, then return */
    flg1 = iflag & ~SEFLG_EQUATORIAL & ~SEFLG_XYZ;
    flg2 = pdp.xflgs & ~SEFLG_EQUATORIAL & ~SEFLG_XYZ;
    if (flg1 == flg2) {
      pdp.xflgs = iflag;
      pdp.iephe = iflag & SEFLG_EPHMASK;
      return OK;
    }
    /* the conversions will be done with xx[]. */
    for (i = 0; i <= 5; i++) 
      xx[i] = pdp.x[i];
    /* if heliocentric position is wanted */
    if ((iflag & SEFLG_HELCTR) != 0) {
      if (pdp.iephe == SEFLG_JPLEPH || pdp.iephe == SEFLG_SWIEPH)
	for (i = 0; i <= 5; i++) 
	  xx[i] -= swed.pldat[SEI_SUNBARY].x[i];
    }
    /************************************
     * observer: geocenter or topocenter
     ************************************/
    /* if topocentric position is wanted  */
    if ((iflag & SEFLG_TOPOCTR) != 0) { 
      if (swed.topd.teval != pedp.teval
	  || swed.topd.teval == 0) {
	if (swi_get_observer(pedp.teval, iflag, DO_SAVE, xobs, serr) != OK)
	  return ERR;
      } else {
	for (i = 0; i <= 5; i++)
	  xobs[i] = swed.topd.xobs[i];
      }
      /* barycentric position of observer */
      for (i = 0; i <= 5; i++)
	xobs[i] = xobs[i] + pedp.x[i];	
    } else {
      /* barycentric position of geocenter */
      for (i = 0; i <= 5; i++)
	xobs[i] = pedp.x[i];
    }
    /*******************************
     * light-time geocentric       * 
     *******************************/
    if ((iflag & SEFLG_TRUEPOS) == 0) {
      /* number of iterations - 1 */
      if (pdp.iephe == SEFLG_JPLEPH || pdp.iephe == SEFLG_SWIEPH) 
	niter = 1;
      else 	/* SEFLG_MOSEPH or planet from osculating elements */
	niter = 0;
      if ((iflag & SEFLG_SPEED) != 0) {
	/* 
	 * Apparent speed is influenced by the fact that dt changes with
	 * motion. This makes a difference of several hundredths of an
	 * arc second. To take this into account, we compute 
	 * 1. true position - apparent position at time t - 1.
	 * 2. true position - apparent position at time t.
	 * 3. the difference between the two is the part of the daily motion 
	 * that results from the change of dt.
	 */
	for (i = 0; i <= 2; i++)
	  xxsv[i] = xxsp[i] = xx[i] - xx[i+3];
	for (j = 0; j <= niter; j++) {
	  for (i = 0; i <= 2; i++) {
	    dx[i] = xxsp[i];
	    if ((iflag & SEFLG_HELCTR) == 0 && (iflag & SEFLG_BARYCTR) == 0)
	      dx[i] -= (xobs[i] - xobs[i+3]);
	  }
	  /* new dt */
	  dt = Math.sqrt(square_sum(dx, 0)) * AUNIT / CLIGHT / 86400.0;     
	  for (i = 0; i <= 2; i++) 	/* rough apparent position at t-1 */
	    xxsp[i] = xxsv[i] - dt * pdp.x[i+3];
	}
	/* true position - apparent position at time t-1 */
	for (i = 0; i <= 2; i++) 
	  xxsp[i] = xxsv[i] - xxsp[i];
      }
      /* dt and t(apparent) */
      for (j = 0; j <= niter; j++) {
	for (i = 0; i <= 2; i++) {
	  dx[i] = xx[i];
	  if ((iflag & SEFLG_HELCTR) == 0 && (iflag & SEFLG_BARYCTR) == 0)
	    dx[i] -= xobs[i];
	}
	dt = Math.sqrt(square_sum(dx, 0)) * AUNIT / CLIGHT / 86400.0;    
	/* new t */
	t = pdp.teval - dt;
	dtsave_for_defl = dt;
	for (i = 0; i <= 2; i++) 	/* rough apparent position at t*/
	  xx[i] = pdp.x[i] - dt * pdp.x[i+3];
      }
      /* part of daily motion resulting from change of dt */
      if ((iflag & SEFLG_SPEED) != 0) 
	for (i = 0; i <= 2; i++) 
	  xxsp[i] = pdp.x[i] - xx[i] - xxsp[i];
      /* new position, accounting for light-time (accurate) */
      switch((int)epheflag) {
      case SEFLG_MOSEPH:
      default:
	/* 
	 * with moshier or other ephemerides, subtraction of dt * speed 
	 * is sufficient (has been done in light-time iteration above)
	 */
        /* if speed flag is true, we call swi_moshplan() for new t.
	 * this does not increase position precision,
	 * but speed precision, which becomes better than 0.01"/day.
	 * for precise speed, we need earth as well.
	 */
	if ((iflag & SEFLG_SPEED) != 0
	    && (iflag & (SEFLG_HELCTR | SEFLG_BARYCTR)) == 0) { 	
	  if (ibody == IS_PLANET)
	    retc = swi_moshplan(t, ipli, NO_SAVE, xxsv, xearth, serr);
          else {		/* if asteroid */
	    retc = sweph(t, ipli, ifno, iflag, null, NO_SAVE, xxsv, serr);
	    if (retc == OK)
	      retc = swi_moshplan(t, SEI_EARTH, NO_SAVE, xearth, xearth, serr);
          }
	  if (retc != OK)
	    return(retc);
	  /* only speed is taken from this computation, otherwise position
	   * calculations with and without speed would not agree.
	   * The difference would be about 0.01", which is far below
	   * the intrinsic error of the moshier ephemeris.
	   */
	  for (i = 3; i <= 5; i++)
	    xx[i] = xxsv[i];
        }
	break;
      }
      if ((iflag & SEFLG_HELCTR) != 0) {
	if (pdp.iephe == SEFLG_JPLEPH || pdp.iephe == SEFLG_SWIEPH) 
	  for (i = 0; i <= 5; i++) 
	    xx[i] -= swed.pldat[SEI_SUNBARY].x[i];
      }
      if ((iflag & SEFLG_SPEED) != 0) {
	/* observer position for t(light-time) */
	if ((iflag & SEFLG_TOPOCTR) != 0) {
	  if (swi_get_observer(t, iflag, NO_SAVE, xobs2, serr) != OK)
	    return ERR;
	  for (i = 0; i <= 5; i++)
	    xobs2[i] += xearth[i];
	} else {
	  for (i = 0; i <= 5; i++)
	    xobs2[i] = xearth[i];
	}
      }
    }
    /*******************************
     * conversion to geocenter     * 
     *******************************/
    if ((iflag & SEFLG_HELCTR) == 0 && (iflag & SEFLG_BARYCTR) == 0) {
      /* subtract earth */
      for (i = 0; i <= 5; i++) 
	xx[i] -= xobs[i]; 
      if ((iflag & SEFLG_TRUEPOS) == 0 ) {
	/* 
	 * Apparent speed is also influenced by
	 * the change of dt during motion.
	 * Neglect of this would result in an error of several 0.01"
	 */
	if ((iflag & SEFLG_SPEED) != 0)
	  for (i = 3; i <= 5; i++) 
	    xx[i] -= xxsp[i-3]; 	
      }
    }
    if ((iflag & SEFLG_SPEED) == 0)
      for (i = 3; i <= 5; i++)
	xx[i] = 0;
    /************************************
     * relativistic deflection of light *
     ************************************/
    if ((iflag & SEFLG_TRUEPOS) == 0 && (iflag & SEFLG_NOGDEFL) == 0)
      /* SEFLG_NOGDEFL is on, if SEFLG_HELCTR or SEFLG_BARYCTR */
      swi_deflect_light(xx, dtsave_for_defl, iflag);
    /**********************************
     * 'annual' aberration of light   *
     **********************************/
    if ((iflag & SEFLG_TRUEPOS) == 0 && (iflag & SEFLG_NOABERR) == 0) {
      /* SEFLG_NOABERR is on, if SEFLG_HELCTR or SEFLG_BARYCTR */
      swi_aberr_light(xx, xobs, iflag);
      /* 
       * Apparent speed is also influenced by
       * the difference of speed of the earth between t and t-dt. 
       * Neglecting this would involve an error of several 0.1"
       */
      if ((iflag & SEFLG_SPEED) != 0)
	for (i = 3; i <= 5; i++) 
	  xx[i] += xobs[i] - xobs2[i];
    }
    /* save J2000 coordinates; required for sidereal positions */
    for (i = 0; i <= 5; i++)
      xxsv[i] = xx[i];
    /************************************************
     * precession, equator 2000 -> equator of date *
     ************************************************/
    if ((iflag & SEFLG_J2000) == 0) {
      SwephLib.swi_precess(xx, 0, pdp.teval, J2000_TO_J);
      if ((iflag & SEFLG_SPEED) != 0)
	swi_precess_speed(xx, 0, pdp.teval, J2000_TO_J);
      oe = swed.oec;
    } else
      oe = swed.oec2000;
    return app_pos_rest(pdp, iflag, xx, xxsv, oe, serr);
  }

  int app_pos_rest(plan_data pdp, long iflag, 
		   double xx[], double x2000[],
		   epsilon oe, String serr) 
  {
    int i;
    /************************************************
     * nutation                                     *
     ************************************************/
    if ((iflag & SEFLG_NONUT) == 0)
      swi_nutate(xx, iflag, false);
    /* now we have equatorial cartesian coordinates; save them */
    for (i = 0; i <= 5; i++)
      pdp.xreturn[18+i] = xx[i];
    /************************************************
     * transformation to ecliptic.                  *
     * with sidereal calc. this will be overwritten *
     * afterwards.                                  *
     ************************************************/
    SwephLib.swi_coortrf2(xx, 0, xx, 0, oe.seps, oe.ceps);
    if ((iflag & SEFLG_SPEED) != 0)
      SwephLib.swi_coortrf2(xx, 3, xx, 3, oe.seps, oe.ceps);
    if ((iflag & SEFLG_NONUT) == 0) {
      SwephLib.swi_coortrf2(xx, 0, xx, 0, swed.nut.snut, swed.nut.cnut);
      if ((iflag & SEFLG_SPEED) != 0)
	SwephLib.swi_coortrf2(xx, 3, xx, 3, swed.nut.snut, swed.nut.cnut);
    }
    /* now we have ecliptic cartesian coordinates */
    for (i = 0; i <= 5; i++)
      pdp.xreturn[6+i] = xx[i];
    /************************************
     * sidereal positions               *
     ************************************/
    if ((iflag & SEFLG_SIDEREAL) != 0) {
      /* project onto ecliptic t0 */
      if ((swed.sidd.sid_mode & SE_SIDBIT_ECL_T0) != 0) {
	if (swi_trop_ra2sid_lon(x2000, pdp.xreturn, 6, pdp.xreturn, 18,
				iflag, serr) != OK)
	  return ERR;
	/* project onto solar system equator */
      } else if ((swed.sidd.sid_mode & SE_SIDBIT_SSY_PLANE) != 0) {
	if (swi_trop_ra2sid_lon_sosy(x2000, pdp.xreturn, 6, pdp.xreturn, 18,
				     iflag, serr) != OK)
	  return ERR;
      } else {
	/* traditional algorithm */
	SwephLib.swi_cartpol_sp(pdp.xreturn, 6, pdp.xreturn, 0); 
	pdp.xreturn[0] -= swe_get_ayanamsa(pdp.teval) * DEGTORAD;
	SwephLib.swi_polcart_sp(pdp.xreturn, 0, pdp.xreturn, 6); 
      }
    } 
    /************************************************
     * transformation to polar coordinates          *
     ************************************************/
    SwephLib.swi_cartpol_sp(pdp.xreturn, 18, pdp.xreturn, 12); 
    SwephLib.swi_cartpol_sp(pdp.xreturn, 6, pdp.xreturn, 0); 
    /********************** 
     * radians to degrees *
     **********************/
    /*if ((iflag & SEFLG_RADIANS) == 0) {*/
    for (i = 0; i < 2; i++) {
      pdp.xreturn[i] *= RADTODEG;		/* ecliptic */
      pdp.xreturn[i+3] *= RADTODEG;
      pdp.xreturn[i+12] *= RADTODEG;	/* equator */
      pdp.xreturn[i+15] *= RADTODEG;
    }
    /*}*/
    /* save, what has been done */
    pdp.xflgs = iflag;
    pdp.iephe = iflag & SEFLG_EPHMASK;
    return OK;
  }

  /**
   * <code>swi_trop_ra2sid_lon</code> method
   * input coordinates are J2000, cartesian.
   * xout 	ecliptical sidereal position
   * xoutr 	equatorial sidereal position
   *
   * @param xin[] a <code>double</code> value
   * @param xout[] a <code>double</code> value
   * @param xoutIndex an <code>int</code> value
   * @param xoutr[] a <code>double</code> value
   * @param xoutrIndex an <code>int</code> value
   * @param iflag a <code>long</code> value
   * @param serr a <code>String</code> value
   * @return an <code>int</code> value
   */
  int swi_trop_ra2sid_lon(double xin[],
			  double xout[], int xoutIndex,
			  double xoutr[], int xoutrIndex,
			  long iflag, String serr)
  {
    double x[];
    x = new double[6];
    int i;
    sid_data sip = swed.sidd;
    epsilon oectmp = new epsilon();
    for (i = 0; i <= 5; i++)
      x[i] = xin[i];
    if (sip.t0 != J2000) {
      SwephLib.swi_precess(x, 0, sip.t0, J2000_TO_J);
      SwephLib.swi_precess(x, 3, sip.t0, J2000_TO_J);	/* speed */
    }
    for (i = 0; i <= 5; i++)
      xoutr[i] = x[i];
    calc_epsilon(swed.sidd.t0, oectmp);
    SwephLib.swi_coortrf2(x, 0, x, 0, oectmp.seps, oectmp.ceps);
    if ((iflag & SEFLG_SPEED) != 0)
      SwephLib.swi_coortrf2(x, 3, x, 3, oectmp.seps, oectmp.ceps);
    /* to polar coordinates */
    SwephLib.swi_cartpol_sp(x, 0, x, 0); 
    /* subtract ayan_t0 */
    x[0] -= sip.ayan_t0 * DEGTORAD;
    /* back to cartesian */
    SwephLib.swi_polcart_sp(x, 0, xout, 0); 
    return OK;
  }

  /**
   * <code>swi_trop_ra2sid_lon_sosy</code> method
   * input coordinates are J2000, cartesian.
   * xout 	ecliptical sidereal position
   * xoutr 	equatorial sidereal position
   *
   * @param xin[] a <code>double</code> value
   * @param xout[] a <code>double</code> value
   * @param xoutIndex an <code>int</code> value
   * @param xoutr[] a <code>double</code> value
   * @param xoutrIndex an <code>int</code> value
   * @param iflag a <code>long</code> value
   * @param serr a <code>String</code> value
   * @return an <code>int</code> value
   */
  int swi_trop_ra2sid_lon_sosy(double xin[],
			       double xout[], int xoutIndex,
			       double xoutr[], int xoutrIndex,
			       long iflag, String serr)
  {
    double x[], x0[];
    x = new double[6];
    x0 = new double[6];
    int i;
    sid_data sip = swed.sidd;
    epsilon oe = swed.oec2000;
    double plane_node = SSY_PLANE_NODE_E2000;
    double plane_incl = SSY_PLANE_INCL;
    for (i = 0; i <= 5; i++)
      x[i] = xin[i];
    /* planet to ecliptic 2000 */
    SwephLib.swi_coortrf2(x, 0, x, 0, oe.seps, oe.ceps);
    if ((iflag & SEFLG_SPEED) != 0)
      SwephLib.swi_coortrf2(x, 3, x, 3, oe.seps, oe.ceps);
    /* to polar coordinates */
    SwephLib.swi_cartpol_sp(x, 0, x, 0); 
    /* to solar system equator */
    x[0] -= plane_node;
    SwephLib.swi_polcart_sp(x, 0, x, 0);
    SwephLib.swi_coortrf(x, 0, x, 0, plane_incl);
    SwephLib.swi_coortrf(x, 3, x, 3, plane_incl);
    SwephLib.swi_cartpol_sp(x, 0, x, 0); 
    /* zero point of t0 in J2000 system */
    x0[0] = 1; 
    x0[1] = x0[2] = 0;
    if (sip.t0 != J2000)
      SwephLib.swi_precess(x0, 0, sip.t0, J_TO_J2000);
    /* zero point to ecliptic 2000 */
    SwephLib.swi_coortrf2(x0, 0, x0, 0, oe.seps, oe.ceps);
    /* to polar coordinates */
    SwephLib.swi_cartpol(x0, 0, x0, 0); 
    /* to solar system equator */
    x0[0] -= plane_node;
    SwephLib.swi_polcart(x0, 0, x0, 0);
    SwephLib.swi_coortrf(x0, 0, x0, 0, plane_incl);
    SwephLib.swi_cartpol(x0, 0, x0, 0); 
    /* measure planet from zero point */
    x[0] -= x0[0];
    x[0] *= RADTODEG;
    /* subtract ayan_t0 */
    x[0] -= sip.ayan_t0;
    x[0] = SwephLib.swe_degnorm(x[0]) * DEGTORAD;
    /* back to cartesian */
    SwephLib.swi_polcart_sp(x, 0, xout, 0); 
    return OK;
  }

  /* converts planets from barycentric to geocentric,
   * apparent positions
   * precession and nutation
   * according to flags
   * ipli		planet number
   * iflag	flags
   */
  int app_pos_etc_plan_osc(int ipl, int ipli, long iflag,
				  String serr)
  {
    int i, j, niter, retc;
    double xx[], dx[], dt, dtsave_for_defl;
    xx = new double[6];
    dx = new double[3];
    double xearth[], xsun[], xmoon[];
    xearth = new double[6];
    xsun = new double[6];
    xmoon = new double[6];
    double xxsv[], xxsp[], xobs[], xobs2[];
    xxsv = new double[6];
    xxsp = new double[6];
    xobs = new double[6];
    xobs2 = new double[6];
    double t;
    plan_data pdp = swed.pldat[ipli];
    plan_data pedp = swed.pldat[SEI_EARTH];
    plan_data psdp = swed.pldat[SEI_SUNBARY];
    epsilon oe = swed.oec2000;
    long epheflag = SEFLG_DEFAULTEPH;
    dt = dtsave_for_defl = 0;	/* dummy assign to silence gcc */
    if ((iflag & SEFLG_MOSEPH) != 0)
      epheflag = SEFLG_MOSEPH;
    /*
    else if (iflag & SEFLG_SWIEPH)
      epheflag = SEFLG_SWIEPH;
    else if (iflag & SEFLG_JPLEPH)
      epheflag = SEFLG_JPLEPH;
    */
    /* the conversions will be done with xx[]. */
    for (i = 0; i <= 5; i++) 
      xx[i] = pdp.x[i];
    /************************************
     * barycentric position is required *
     ************************************/
    /* = heliocentric position with Moshier ephemeris */
    /************************************
     * observer: geocenter or topocenter
     ************************************/
    /* if topocentric position is wanted  */
    if ((iflag & SEFLG_TOPOCTR) != 0) { 
      if (swed.topd.teval != pedp.teval
	  || swed.topd.teval == 0) {
	if (swi_get_observer(pedp.teval, iflag, DO_SAVE, xobs, serr) != OK)
	  return ERR;
      } else {
	for (i = 0; i <= 5; i++)
	  xobs[i] = swed.topd.xobs[i];
      }
      /* barycentric position of observer */
      for (i = 0; i <= 5; i++)
	xobs[i] = xobs[i] + pedp.x[i];	
    } else if ((iflag & SEFLG_BARYCTR) != 0) {
      for (i = 0; i <= 5; i++)
	xobs[i] = 0;	
    } else if ((iflag & SEFLG_HELCTR) != 0) {
      if ((iflag & SEFLG_MOSEPH) != 0) {
	for (i = 0; i <= 5; i++)
	  xobs[i] = 0;	
      } else {
	for (i = 0; i <= 5; i++)
	  xobs[i] = psdp.x[i];	
      }
    } else {
      for (i = 0; i <= 5; i++)
	xobs[i] = pedp.x[i];	
    }
    /*******************************
     * light-time                  * 
     *******************************/
    if ((iflag & SEFLG_TRUEPOS) == 0) {
      niter = 1;
      if ((iflag & SEFLG_SPEED) != 0) {
	/* 
	 * Apparent speed is influenced by the fact that dt changes with
	 * motion. This makes a difference of several hundredths of an
	 * arc second. To take this into account, we compute 
	 * 1. true position - apparent position at time t - 1.
	 * 2. true position - apparent position at time t.
	 * 3. the difference between the two is the daily motion resulting from
	 * the change of dt.
	 */
	for (i = 0; i <= 2; i++)
	  xxsv[i] = xxsp[i] = xx[i] - xx[i+3];
	for (j = 0; j <= niter; j++) {
	  for (i = 0; i <= 2; i++) {
	    dx[i] = xxsp[i];
	    if ((iflag & SEFLG_HELCTR) == 0 && (iflag & SEFLG_BARYCTR) == 0)
	      dx[i] -= (xobs[i] - xobs[i+3]);
	  }
	  /* new dt */
	  dt = Math.sqrt(square_sum(dx, 0)) * AUNIT / CLIGHT / 86400.0;     
	  for (i = 0; i <= 2; i++) 
	    xxsp[i] = xxsv[i] - dt * pdp.x[i+3];/* rough apparent position */
	}
	/* true position - apparent position at time t-1 */
	for (i = 0; i <= 2; i++) 
	  xxsp[i] = xxsv[i] - xxsp[i];
      }
      /* dt and t(apparent) */
      for (j = 0; j <= niter; j++) {
	for (i = 0; i <= 2; i++) {
	  dx[i] = xx[i];
	  if ((iflag & SEFLG_HELCTR) == 0 && (iflag & SEFLG_BARYCTR) == 0)
	    dx[i] -= xobs[i];
	}
	/* new dt */
	dt = Math.sqrt(square_sum(dx, 0)) * AUNIT / CLIGHT / 86400.0;     
	dtsave_for_defl = dt;
	/* new position: subtract t * speed 
	 */
	for (i = 0; i <= 2; i++) {
	  xx[i] = pdp.x[i] - dt * pdp.x[i+3];/**/
	  xx[i+3] = pdp.x[i+3];
	}
      }
      if ((iflag & SEFLG_SPEED) != 0) {
	/* part of daily motion resulting from change of dt */
	for (i = 0; i <= 2; i++) 
	  xxsp[i] = pdp.x[i] - xx[i] - xxsp[i];
	t = pdp.teval - dt;
	/* for accuracy in speed, we will need earth as well */
	retc = main_planet_bary(t, SEI_EARTH, epheflag, iflag, NO_SAVE,
				xearth, xearth, xsun, xmoon, serr);
	if (swi_osc_el_plan(t, xx, ipl-SE_FICT_OFFSET, ipli,
			    xearth, xsun, serr) != OK)
	  return ERR;
	if (retc != OK)
	  return(retc);
	if ((iflag & SEFLG_TOPOCTR) != 0) {
	  if (swi_get_observer(t, iflag, NO_SAVE, xobs2, serr) != OK)
	    return ERR;
	  for (i = 0; i <= 5; i++)
	    xobs2[i] += xearth[i];
	} else {
	  for (i = 0; i <= 5; i++)
	    xobs2[i] = xearth[i];
	}
      }
    }
    /*******************************
     * conversion to geocenter     * 
     *******************************/
    for (i = 0; i <= 5; i++) 
      xx[i] -= xobs[i]; 
    if ((iflag & SEFLG_TRUEPOS) == 0) {
      /* 
       * Apparent speed is also influenced by
       * the change of dt during motion.
       * Neglect of this would result in an error of several 0.01"
       */
      if ((iflag & SEFLG_SPEED) != 0)
	for (i = 3; i <= 5; i++) 
	  xx[i] -= xxsp[i-3]; 
    }
    if ((iflag & SEFLG_SPEED) == 0)
      for (i = 3; i <= 5; i++)
	xx[i] = 0;
    /************************************
     * relativistic deflection of light *
     ************************************/
    if ((iflag & SEFLG_TRUEPOS) == 0 && (iflag & SEFLG_NOGDEFL) == 0) 
      /* SEFLG_NOGDEFL is on, if SEFLG_HELCTR or SEFLG_BARYCTR */
      swi_deflect_light(xx, dtsave_for_defl, iflag);
    /**********************************
     * 'annual' aberration of light   *
     **********************************/
    if ((iflag & SEFLG_TRUEPOS) == 0 && (iflag & SEFLG_NOABERR) == 0) {
      /* SEFLG_NOABERR is on, if SEFLG_HELCTR or SEFLG_BARYCTR */
      swi_aberr_light(xx, xobs, iflag);
      /* 
       * Apparent speed is also influenced by
       * the difference of speed of the earth between t and t-dt. 
       * Neglecting this would involve an error of several 0.1"
       */
      if ((iflag & SEFLG_SPEED) != 0)
	for (i = 3; i <= 5; i++) 
	  xx[i] += xobs[i] - xobs2[i];
    }
    /* save J2000 coordinates; required for sidereal positions */
    for (i = 0; i <= 5; i++)
      xxsv[i] = xx[i];
    /************************************************
     * precession, equator 2000 -> equator of date *
     ************************************************/
    if ((iflag & SEFLG_J2000) == 0) {
      SwephLib.swi_precess(xx, 0, pdp.teval, J2000_TO_J);
      if ((iflag & SEFLG_SPEED) != 0)
	swi_precess_speed(xx, 0, pdp.teval, J2000_TO_J);
      oe = swed.oec;
    } else
      oe = swed.oec2000;
    return app_pos_rest(pdp, iflag, xx, xxsv, oe, serr);
  }

  /**
   * <code>swi_precess_speed</code> method
   * influence of precession on speed 
   * xx		position and speed of planet in equatorial cartesian
   *		coordinates
   *
   * @param xx[] a <code>double</code> value
   * @param xxIndex an <code>int</code> value
   * @param t a <code>double</code> value
   * @param direction an <code>int</code> value
   */
  void swi_precess_speed(double xx[], int xxIndex, double t, int direction) 
  {
    epsilon oe;
    double fac;
    double tprec = (t - J2000) / 36525.0;
    if (direction == J2000_TO_J) {
      fac = 1;
      oe = swed.oec;
    } else {
      fac = -1;
      oe = swed.oec2000;
    }
    /* first correct rotation.
     * this costs some sines and cosines, but neglect might
     * involve an error > 1"/day */
    SwephLib.swi_precess(xx, xxIndex+3, t, direction); 
    /* then add 0.137"/day */
    SwephLib.swi_coortrf2(xx, xxIndex, xx, xxIndex, oe.seps, oe.ceps);
    SwephLib.swi_coortrf2(xx, xxIndex+3, xx, xxIndex+3, oe.seps, oe.ceps);
    SwephLib.swi_cartpol_sp(xx, xxIndex, xx, xxIndex);
    xx[xxIndex+3]
      += (50.290966 + 0.0222226 * tprec) / 3600 / 365.25 * DEGTORAD * fac;
    /* formula from Montenbruck, German 1994, p. 18 */
    SwephLib.swi_polcart_sp(xx, xxIndex, xx, xxIndex);
    SwephLib.swi_coortrf2(xx, xxIndex, xx, xxIndex, -oe.seps, oe.ceps);
    SwephLib.swi_coortrf2(xx, xxIndex+3, xx, xxIndex+3, -oe.seps, oe.ceps);
  }

  /**
   * <code>swi_nutate</code> method
   * multiplies cartesian equatorial coordinates with previously
   * calculated nutation matrix. also corrects speed. 
   *
   * @param xx[] a <code>double</code> value
   * @param iflag a <code>long</code> value
   * @param backward a <code>boolean</code> value
   */
  void swi_nutate(double xx[], long iflag, boolean backward)
  {
    int i;
    double x[], xv[];
    x = new double[6];
    xv = new double[6];
    for (i = 0; i <= 2; i++) {
      if (backward) 
	x[i] = xx[0] * swed.nut.matrix[i][0] + 
	  xx[1] * swed.nut.matrix[i][1] + 
	  xx[2] * swed.nut.matrix[i][2];
      else 
	x[i] = xx[0] * swed.nut.matrix[0][i] + 
	  xx[1] * swed.nut.matrix[1][i] + 
	  xx[2] * swed.nut.matrix[2][i];
    }
    if ((iflag & SEFLG_SPEED) != 0) {
      /* correct speed:
       * first correct rotation */
      for (i = 0; i <= 2; i++) {
	if (backward) 
	  x[i+3] = xx[3] * swed.nut.matrix[i][0] + 
	    xx[4] * swed.nut.matrix[i][1] + 
	    xx[5] * swed.nut.matrix[i][2];
	else
	  x[i+3] = xx[3] * swed.nut.matrix[0][i] + 
	    xx[4] * swed.nut.matrix[1][i] + 
	    xx[5] * swed.nut.matrix[2][i];
      }
      /* then apparent motion due to change of nutation during day.
       * this makes a difference of 0.01" */
      for (i = 0; i <= 2; i++) {
	if (backward) 
	  xv[i] = xx[0] * swed.nutv.matrix[i][0] + 
	    xx[1] * swed.nutv.matrix[i][1] + 
	    xx[2] * swed.nutv.matrix[i][2];
	else
	  xv[i] = xx[0] * swed.nutv.matrix[0][i] + 
	    xx[1] * swed.nutv.matrix[1][i] + 
	    xx[2] * swed.nutv.matrix[2][i];
	/* new speed */
	xx[3+i] = x[3+i] + (x[i] - xv[i]) / NUT_SPEED_INTV;
      }
    }
    /* new position */
    for (i = 0; i <= 2; i++) 
      xx[i] = x[i];
  }

  /**
   * <code>swi_aberr_light</code> method
   * computes 'annual' aberration
   * xx		planet's position accounted for light-time 
   *              and gravitational light deflection
   * xe    	earth's position and speed
   *
   * @param xx[] a <code>double</code> value
   * @param xe[] a <code>double</code> value
   * @param iflag a <code>long</code> value
   */
  void swi_aberr_light(double xx[], double xe[], long iflag) {
    int i;
    double xxs[], v[], u[], ru;
    xxs = new double[6];
    v = new double[6];
    u = new double[6];
    double xx2[], dx1, dx2;
    xx2 = new double[6];
    double b_1, f1, f2;
    double v2;
    double intv = PLAN_SPEED_INTV;
    for (i = 0; i <= 5; i++)
      u[i] = xxs[i] = xx[i];
    ru = Math.sqrt(square_sum(u, 0));
    for (i = 0; i <= 2; i++) 
      v[i] = xe[i+3] / 24.0 / 3600.0 / CLIGHT * AUNIT;
    v2 = square_sum(v, 0);
    b_1 = Math.sqrt(1 - v2);
    f1 = dot_prod(u, 0, v, 0) / ru;
    f2 = 1.0 + f1 / (1.0 + b_1);
    for (i = 0; i <= 2; i++) 
      xx[i] = (b_1*xx[i] + f2*ru*v[i]) / (1.0 + f1);
    if ((iflag & SEFLG_SPEED) != 0) {
      /* correction of speed
       * the influence of aberration on apparent velocity can
       * reach 0.4"/day
       */
      for (i = 0; i <= 2; i++) 
	u[i] = xxs[i] - intv * xxs[i+3];
      ru = Math.sqrt(square_sum(u, 0));
      f1 = dot_prod(u, 0, v, 0) / ru;
      f2 = 1.0 + f1 / (1.0 + b_1);
      for (i = 0; i <= 2; i++) 
	xx2[i] = (b_1*u[i] + f2*ru*v[i]) / (1.0 + f1);
      for (i = 0; i <= 2; i++) {
	dx1 = xx[i] - xxs[i];
	dx2 = xx2[i] - u[i];
	dx1 -= dx2;
	xx[i+3] += dx1 / intv;
      }
    }
  }

  /**
   * <code>swi_deflect_light</code> method
   * computes relativistic light deflection by the sun
   * ipli 	sweph internal planet number 
   * xx		planet's position accounted for light-time
   * dt		dt of light-time
   *
   * @param xx[] a <code>double</code> value
   * @param dt a <code>double</code> value
   * @param iflag a <code>long</code> value
   */
  void swi_deflect_light(double xx[], double dt, long iflag) 
  {
    int i;
    double xx2[];
    xx2 = new double[6];
    double u[], e[], q[], ru, re, rq, uq, ue, qe, g1, g2;
    u = new double[6];
    e = new double[6];
    q = new double[6];
    double xx3[], dx1, dx2, dtsp;
    xx3 = new double[6];
    double xsun[], xearth[];
    xsun = new double[6];
    xearth = new double[6];
    double sina, sin_sunr, meff_fact;
    plan_data pedp = swed.pldat[SEI_EARTH];
    plan_data psdp = swed.pldat[SEI_SUNBARY];
    long iephe = pedp.iephe;
    for (i = 0; i <= 5; i++)
      xearth[i] = pedp.x[i];
    if ((iflag & SEFLG_TOPOCTR) != 0)
      for (i = 0; i <= 5; i++)
	xearth[i] += swed.topd.xobs[i];
    /* U = planetbary(t-tau) - earthbary(t) = planetgeo */
    for (i = 0; i <= 2; i++) 
      u[i] = xx[i]; 
    /* Eh = earthbary(t) - sunbary(t) = earthhel */
    if (iephe == SEFLG_JPLEPH || iephe == SEFLG_SWIEPH)
      for (i = 0; i <= 2; i++) 
	e[i] = xearth[i] - psdp.x[i]; 
    else
      for (i = 0; i <= 2; i++) 
	e[i] = xearth[i];
    /* Q = planetbary(t-tau) - sunbary(t-tau) = 'planethel' */
    /* first compute sunbary(t-tau) for */
    if (iephe == SEFLG_JPLEPH || iephe == SEFLG_SWIEPH) {
      for (i = 0; i <= 2; i++)
	/* this is sufficient precision */
	xsun[i] = psdp.x[i] - dt * psdp.x[i+3];
      for (i = 3; i <= 5; i++)
	xsun[i] = psdp.x[i];
    } else {
      for (i = 0; i <= 5; i++)
	xsun[i] = psdp.x[i];
    }
    for (i = 0; i <= 2; i++)
      q[i] = xx[i] + xearth[i] - xsun[i];
    ru = Math.sqrt(square_sum(u, 0));
    rq = Math.sqrt(square_sum(q, 0));
    re = Math.sqrt(square_sum(e, 0));
    for (i = 0; i <= 2; i++) {
      u[i] /= ru;
      q[i] /= rq;
      e[i] /= re;
    }
    uq = dot_prod(u, 0, q, 0);
    ue = dot_prod(u, 0, e, 0);
    qe = dot_prod(q, 0, e, 0);
    /* When a planet approaches the center of the sun in superior
     * conjunction, the formula for the deflection angle as given
     * in Expl. Suppl. p. 136 cannot be used. The deflection seems
     * to increase rapidly towards infinity. The reason is that the 
     * formula considers the sun as a point mass. AA recommends to 
     * set deflection = 0 in such a case. 
     * However, to get a continous motion, we modify the formula
     * for a non-point-mass, taking into account the mass distribution
     * within the sun. For more info, s. meff().
     */
    sina = Math.sqrt(1 - ue * ue);	/* sin(angle) between sun and planet */
    sin_sunr = SUN_RADIUS / re; 	/* sine of sun radius (= sun radius) */
    if (sina < sin_sunr) 	
      meff_fact = meff(sina / sin_sunr);
    else
      meff_fact = 1;
    g1 = 2.0 * HELGRAVCONST * meff_fact / CLIGHT / CLIGHT / AUNIT / re; 
    g2 = 1.0 + qe;
    /* compute deflected position */
    for (i = 0; i <= 2; i++) 
      xx2[i] = ru * (u[i] + g1/g2 * (uq * e[i] - ue * q[i]));
    if ((iflag & SEFLG_SPEED) != 0) {
      /* correction of speed
       * influence of light deflection on a planet's apparent speed:
       * for an outer planet at the solar limb with 
       * |v(planet) - v(sun)| = 1 degree, this makes a difference of 7"/day. 
       * if the planet is within the solar disc, the difference may increase
       * to 30" or more.
       * e.g. mercury at j2434871.45: 
       *	distance from sun 		45"
       *	1. speed without deflection     2$B%&(B10'10".4034
       *    2. speed with deflection        2$B%&(B10'42".8460 (-speed flag)
       *    3. speed with deflection        2$B%&(B10'43".4824 (< 3 positions/
       *						   -speed3 flag)
       * 3. is not very precise. Smaller dt would give result closer to 2.,
       * but will probably never be as good as 2, unless long doubles are 
       * used. (try also j2434871.46!!)
       * however, in such a case speed changes rapidly. before being
       * passed by the sun, the planet accelerates, and after the sun
       * has passed it slows down. some time later it regains 'normal'
       * speed.
       * to compute speed, we do the same calculation as above with
       * slightly different u, e, q, and find out the difference in
       * deflection.
       */
      dtsp = -DEFL_SPEED_INTV;
      /* U = planetbary(t-tau) - earthbary(t) = planetgeo */
      for (i = 0; i <= 2; i++) 
	u[i] = xx[i] - dtsp * xx[i+3]; 
      /* Eh = earthbary(t) - sunbary(t) = earthhel */
      if (iephe == SEFLG_JPLEPH || iephe == SEFLG_SWIEPH) {
	for (i = 0; i <= 2; i++) 
	  e[i] = xearth[i] - psdp.x[i] -
	    dtsp * (xearth[i+3] - psdp.x[i+3]); 
      } else
	for (i = 0; i <= 2; i++) 
	  e[i] = xearth[i] - dtsp * xearth[i+3];
      /* Q = planetbary(t-tau) - sunbary(t-tau) = 'planethel' */
      for (i = 0; i <= 2; i++)
	q[i] = u[i] + xearth[i] - xsun[i] -
	  dtsp * (xearth[i+3] - xsun[i+3]); 
      ru = Math.sqrt(square_sum(u, 0));
      rq = Math.sqrt(square_sum(q, 0));
      re = Math.sqrt(square_sum(e, 0));
      for (i = 0; i <= 2; i++) {
	u[i] /= ru;
	q[i] /= rq;
	e[i] /= re;
      }
      uq = dot_prod(u, 0, q, 0);
      ue = dot_prod(u, 0, e, 0);
      qe = dot_prod(q, 0, e, 0);
      sina = Math.sqrt(1 - ue * ue);	/* sin(angle) between sun and planet */
      sin_sunr = SUN_RADIUS / re; 	/* sine of sun radius (= sun radius) */
      if (sina < sin_sunr) 	
	meff_fact = meff(sina / sin_sunr);
      else
	meff_fact = 1;
      g1 = 2.0 * HELGRAVCONST * meff_fact / CLIGHT / CLIGHT / AUNIT / re; 
      g2 = 1.0 + qe;
      for (i = 0; i <= 2; i++) 
	xx3[i] = ru * (u[i] + g1/g2 * (uq * e[i] - ue * q[i]));
      for (i = 0; i <= 2; i++) {
	dx1 = xx2[i] - xx[i];
	dx2 = xx3[i] - u[i] * ru;
	dx1 -= dx2;
	xx[i+3] += dx1 / dtsp;
      }
    } /* endif speed */
    /* deflected position */
    for (i = 0; i <= 2; i++) 
      xx[i] = xx2[i];
  }

  /**
   * <code>app_pos_etc_sun</code> method
   * converts the sun from barycentric to geocentric,
   *          the earth from barycentric to heliocentric
   * computes
   * apparent position,
   * precession, and nutation
   * according to flags
   * iflag	flags
   * serr         error string
   *
   * @param iflag a <code>long</code> value
   * @param serr a <code>String</code> value
   * @return an <code>int</code> value
   */
  int app_pos_etc_sun(long iflag, String serr)
  {
    int i, j, niter, retc = OK;
    long flg1, flg2;
    double xx[], xxsv[], dx[], dt, t;
    xx = new double[6];
    xxsv = new double[6];
    dx = new double[3];
    double xearth[], xsun[], xobs[];
    xearth = new double[6];
    xsun = new double[6];
    xobs = new double[6];
    plan_data pedp = swed.pldat[SEI_EARTH];
    plan_data psdp = swed.pldat[SEI_SUNBARY];
    epsilon oe = swed.oec2000;
    /* if the same conversions have already been done for the same 
     * date, then return */
    flg1 = iflag & ~SEFLG_EQUATORIAL & ~SEFLG_XYZ;
    flg2 = pedp.xflgs & ~SEFLG_EQUATORIAL & ~SEFLG_XYZ;
    if (flg1 == flg2) {
      pedp.xflgs = iflag;
      pedp.iephe = iflag & SEFLG_EPHMASK;
      return OK;
    }
    /************************************
     * observer: geocenter or topocenter
     ************************************/
    /* if topocentric position is wanted  */
    if ((iflag & SEFLG_TOPOCTR) != 0) { 
      if (swed.topd.teval != pedp.teval
	  || swed.topd.teval == 0) {
	if (swi_get_observer(pedp.teval, iflag, DO_SAVE, xobs, serr) != OK)
	  return ERR;
      } else {
	for (i = 0; i <= 5; i++)
	  xobs[i] = swed.topd.xobs[i];
      }
      /* barycentric position of observer */
      for (i = 0; i <= 5; i++)
	xobs[i] = xobs[i] + pedp.x[i];	
    } else {
      /* barycentric position of geocenter */
      for (i = 0; i <= 5; i++)
	xobs[i] = pedp.x[i];
    }
    /***************************************
     * true heliocentric position of earth *
     ***************************************/
    if (pedp.iephe == SEFLG_MOSEPH || (iflag & SEFLG_BARYCTR) != 0)
      for (i = 0; i <= 5; i++) 
	xx[i] = xobs[i];
    else 
      for (i = 0; i <= 5; i++) 
	xx[i] = xobs[i] - psdp.x[i];
    /*******************************
     * light-time                  * 
     *******************************/
    if ((iflag & SEFLG_TRUEPOS) == 0) {
      /* number of iterations - 1 
       * the following if() does the following:
       * with jpl and swiss ephemeris:
       *   with geocentric computation of sun:
       *     light-time correction of barycentric sun position.
       *   with heliocentric or barycentric computation of earth:
       *     light-time correction of barycentric earth position.
       * with moshier ephemeris (heliocentric!!!): 
       *   with geocentric computation of sun:
       *     nothing! (aberration will be done later)
       *   with heliocentric or barycentric computation of earth:
       *     light-time correction of heliocentric earth position.
       */
      if (pedp.iephe == SEFLG_JPLEPH || pedp.iephe == SEFLG_SWIEPH
	  || (iflag & SEFLG_HELCTR) != 0 || (iflag & SEFLG_BARYCTR) != 0) {
	for (i = 0; i <= 5; i++) {
	  xearth[i] = xobs[i];
	  if (pedp.iephe == SEFLG_MOSEPH)
	    xsun[i] = 0;
	  else 
	    xsun[i] = psdp.x[i];
	}
	niter = 1;	/* # of iterations */
	for (j = 0; j <= niter; j++) {
	  /* distance earth-sun */
	  for (i = 0; i <= 2; i++) {
	    dx[i] = xearth[i];
	    if ((iflag & SEFLG_BARYCTR) == 0)
	      dx[i] -= xsun[i];
	  }
	  /* new t */
	  dt = Math.sqrt(square_sum(dx, 0)) * AUNIT / CLIGHT / 86400.0;     
	  t = pedp.teval - dt;
	  /* new position */
	  switch((int)pedp.iephe) {
	    /* if geocentric sun, new sun at t' 
	     * if heliocentric or barycentric earth, new earth at t' */
	  case SEFLG_MOSEPH:
	    if ((iflag & SEFLG_HELCTR) != 0 || (iflag & SEFLG_BARYCTR) != 0)
	      retc = swi_moshplan(t, SEI_EARTH, NO_SAVE, xearth, xearth, serr);
	    /* with moshier there is no barycentric sun */
	    break;
          default:
	    retc = ERR;
	    break;
	  } 
	  if (retc != OK)
	    return(retc);
	} 
	/* apparent heliocentric earth */
	for (i = 0; i <= 5; i++) {
	  xx[i] = xearth[i];
	  if ((iflag & SEFLG_BARYCTR) == 0)
	    xx[i] -= xsun[i];
	}
      } 
    } 
    if ((iflag & SEFLG_SPEED) == 0)
      for (i = 3; i <= 5; i++)
	xx[i] = 0;
    /*******************************
     * conversion to geocenter     * 
     *******************************/
    if ((iflag & SEFLG_HELCTR) == 0 && (iflag & SEFLG_BARYCTR) == 0)
      for (i = 0; i <= 5; i++) 
	xx[i] = -xx[i]; 
    /**********************************
     * 'annual' aberration of light   *
     **********************************/
    if ((iflag & SEFLG_TRUEPOS) == 0 && (iflag & SEFLG_NOABERR) == 0) {
      /* SEFLG_NOABERR is on, if SEFLG_HELCTR or SEFLG_BARYCTR */
      swi_aberr_light(xx, xobs, iflag);
    }
    /* save J2000 coordinates; required for sidereal positions */
    for (i = 0; i <= 5; i++)
      xxsv[i] = xx[i];
    /************************************************
     * precession, equator 2000 -> equator of date *
     ************************************************/
    if ((iflag & SEFLG_J2000) == 0) {
      SwephLib.swi_precess(xx, 0, pedp.teval, J2000_TO_J);/**/
      if ((iflag & SEFLG_SPEED) != 0)
	swi_precess_speed(xx, 0, pedp.teval, J2000_TO_J);/**/ 
      oe = swed.oec;
    } else
      oe = swed.oec2000;
    return app_pos_rest(pedp, iflag, xx, xxsv, oe, serr);
  }

  /**
   * <code>app_pos_etc_moon</code> method
   * transforms the position of the moon:
   * heliocentric position
   * barycentric position
   * astrometric position
   * apparent position
   * precession and nutation
   * 
   * note: 
   * for apparent positions, we consider the earth-moon
   * system as independant.
   * for astrometric positions (SEFLG_NOABERR), we 
   * consider the motions of the earth and the moon 
   * related to the solar system barycenter.
   *
   * @param iflag a <code>long</code> value
   * @param serr a <code>String</code> value
   * @return an <code>int</code> value
   */
  int app_pos_etc_moon(long iflag, String serr)
  {
    int i;
    long flg1, flg2;
    double xx[], xxsv[], xobs[], xxm[], xs[], xe[], xobs2[], dt;
    xx = new double[6];
    xxsv = new double[6];
    xobs = new double[6];
    xxm = new double[6];
    xs = new double[6];
    xe = new double[6];
    xobs2 = new double[6];
    plan_data pedp = swed.pldat[SEI_EARTH];
    plan_data psdp = swed.pldat[SEI_SUNBARY];
    plan_data pdp = swed.pldat[SEI_MOON];
    epsilon oe = swed.oec;
    double t; 
  //  long retc; 
    /* if the same conversions have already been done for the same 
     * date, then return */
    flg1 = iflag & ~SEFLG_EQUATORIAL & ~SEFLG_XYZ;
    flg2 = pdp.xflgs & ~SEFLG_EQUATORIAL & ~SEFLG_XYZ;
    if (flg1 == flg2) {
      pdp.xflgs = iflag;
      pdp.iephe = iflag & SEFLG_EPHMASK;
      return OK;
    }
    /* the conversions will be done with xx[]. */
    for (i = 0; i <= 5; i++) {
      xx[i] = pdp.x[i];
      xxm[i] = xx[i];
    }
    /***********************************
     * to solar system barycentric
     ***********************************/
    for (i = 0; i <= 5; i++)
      xx[i] += pedp.x[i]; 
    /*******************************
     * observer
     *******************************/
    if ((iflag & SEFLG_TOPOCTR) != 0) {
      if (swed.topd.teval != pdp.teval
	  || swed.topd.teval == 0) {
	if (swi_get_observer(pdp.teval, iflag, DO_SAVE, xobs, null) != OK)
	  return ERR;
      } else {
	for (i = 0; i <= 5; i++)
	  xobs[i] = swed.topd.xobs[i];
      }
      for (i = 0; i <= 5; i++)
	xxm[i] -= xobs[i];
      for (i = 0; i <= 5; i++)
	xobs[i] += pedp.x[i];
    } else if ((iflag & SEFLG_BARYCTR) != 0) { 
      for (i = 0; i <= 5; i++)
	xobs[i] = 0;
      for (i = 0; i <= 5; i++)
	xxm[i] += pedp.x[i];
    } else if ((iflag & SEFLG_HELCTR) != 0) {
      for (i = 0; i <= 5; i++)
	xobs[i] = psdp.x[i];
      for (i = 0; i <= 5; i++)
	xxm[i] += pedp.x[i] - psdp.x[i];
    } else {
      for (i = 0; i <= 5; i++)
	xobs[i] = pedp.x[i];
    }
    /*******************************
     * light-time                  * 
     *******************************/
    if ((iflag & SEFLG_TRUEPOS) == 0) {
      dt = Math.sqrt(square_sum(xxm, 0)) * AUNIT / CLIGHT / 86400.0;     
      t = pdp.teval - dt;
      switch((int)pdp.iephe) {
      case SEFLG_MOSEPH:
        /* this method results in an error of a milliarcsec in speed */
        for (i = 0; i <= 2; i++) {
          xx[i] -= dt * xx[i+3];
          xe[i] = pedp.x[i] - dt * pedp.x[i+3];
	  xe[i+3] = pedp.x[i+3];
	  xs[i] = 0;
	  xs[i+3] = 0;
        }
        break;
      } 
      if ((iflag & SEFLG_TOPOCTR) != 0) {
	if (swi_get_observer(t, iflag, NO_SAVE, xobs2, null) != OK)
	  return ERR;
	for (i = 0; i <= 5; i++)
	  xobs2[i] += xe[i];
      } else if ((iflag & SEFLG_BARYCTR) != 0) {
	for (i = 0; i <= 5; i++)
	  xobs2[i] = 0;
      } else if ((iflag & SEFLG_HELCTR) != 0) {
	for (i = 0; i <= 5; i++)
	  xobs2[i] = xs[i];
      } else {
	for (i = 0; i <= 5; i++)
	  xobs2[i] = xe[i];
      }
    }
    /*************************
     * to correct center 
     *************************/
    for (i = 0; i <= 5; i++)
      xx[i] -= xobs[i];
    /**********************************
     * 'annual' aberration of light   *
     **********************************/
    if ((iflag & SEFLG_TRUEPOS) == 0 && (iflag & SEFLG_NOABERR) == 0) {
      /* SEFLG_NOABERR is on, if SEFLG_HELCTR or SEFLG_BARYCTR */
      swi_aberr_light(xx, xobs, iflag);
      /* 
       * Apparent speed is also influenced by
       * the difference of speed of the earth between t and t-dt. 
       * Neglecting this would lead to an error of several 0.1"
       */
      if ((iflag & SEFLG_SPEED) != 0)
	for (i = 3; i <= 5; i++) 
	  xx[i] += xobs[i] - xobs2[i];
    }
    /* if !speedflag, speed = 0 */
    if ((iflag & SEFLG_SPEED) == 0)
      for (i = 3; i <= 5; i++)
	xx[i] = 0;
    /* save J2000 coordinates; required for sidereal positions */
    for (i = 0; i <= 5; i++)
      xxsv[i] = xx[i];
    /************************************************
     * precession, equator 2000 -> equator of date *
     ************************************************/
    if ((iflag & SEFLG_J2000) == 0) {
      SwephLib.swi_precess(xx, 0, pdp.teval, J2000_TO_J);
      if ((iflag & SEFLG_SPEED) != 0)
	swi_precess_speed(xx, 0, pdp.teval, J2000_TO_J); 
      oe = swed.oec;
    } else
      oe = swed.oec2000;
    return app_pos_rest(pdp, iflag, xx, xxsv, oe, serr);
  }

  /**
   * <code>app_pos_etc_mean</code> method
   * transforms position of mean lunar node or apogee:
   * input is polar coordinates in mean ecliptic of date.
   * output is, according to iflag:
   * position accounted for light-time
   * position referred to J2000 (i.e. precession subtracted)
   * position with nutation 
   * equatorial coordinates
   * cartesian coordinates
   * heliocentric position is not allowed ??????????????
   *         DAS WAERE ZIEMLICH AUFWENDIG. SONNE UND ERDE MUESSTEN
   *         SCHON VORHANDEN SEIN!
   * ipl		bodynumber (SE_MEAN_NODE or SE_MEAN_APOG)
   * iflag	flags
   * serr         error string
   *
   * @param ipl an <code>int</code> value
   * @param iflag a <code>long</code> value
   * @param serr a <code>String</code> value
   * @return an <code>int</code> value
   */
  int app_pos_etc_mean(int ipl, long iflag, String serr) 
  {
    int i;
    long flg1, flg2;
    double xx[], xxsv[];
    xx = new double[6];
    xxsv = new double[6];
    plan_data pdp = swed.nddat[ipl];
    epsilon oe;
    /* if the same conversions have already been done for the same 
     * date, then return */
    flg1 = iflag & ~SEFLG_EQUATORIAL & ~SEFLG_XYZ;
    flg2 = pdp.xflgs & ~SEFLG_EQUATORIAL & ~SEFLG_XYZ;
    if (flg1 == flg2) {
      pdp.xflgs = iflag;
      pdp.iephe = iflag & SEFLG_EPHMASK;
      return OK;
    }
    for (i = 0; i <= 5; i++)
      xx[i] = pdp.x[i];
    /* cartesian equatorial coordinates */
    SwephLib.swi_polcart_sp(xx, 0, xx, 0);
    SwephLib.swi_coortrf2(xx, 0, xx, 0, -swed.oec.seps, swed.oec.ceps);
    SwephLib.swi_coortrf2(xx, 3, xx, 3, -swed.oec.seps, swed.oec.ceps);
    if ((iflag & SEFLG_SPEED) == 0)
      for (i = 3; i <= 5; i++)
	xx[i] = 0;
    /* J2000 coordinates; required for sidereal positions */
    if (((iflag & SEFLG_SIDEREAL) != 0 
	 && (swed.sidd.sid_mode & SE_SIDBIT_ECL_T0) != 0)
	|| (swed.sidd.sid_mode & SE_SIDBIT_SSY_PLANE) != 0) {
      for (i = 0; i <= 5; i++)
	xxsv[i] = xx[i];
      /* xxsv is not J2000 yet! */
      if (pdp.teval != J2000) {
	SwephLib.swi_precess(xxsv, 0, pdp.teval, J_TO_J2000);
	if ((iflag & SEFLG_SPEED) != 0)
	  swi_precess_speed(xxsv, 0, pdp.teval, J_TO_J2000); 
      }
    }
    /*****************************************************
     * if no precession, equator of date -> equator 2000 *
     *****************************************************/
    if ((iflag & SEFLG_J2000) != 0) {
      SwephLib.swi_precess(xx, 0, pdp.teval, J_TO_J2000); 
      if ((iflag & SEFLG_SPEED) != 0)
	swi_precess_speed(xx, 0, pdp.teval, J_TO_J2000); 
      oe = swed.oec2000;
    } else
      oe = swed.oec;
    return app_pos_rest(pdp, iflag, xx, xxsv, oe, serr);
  }

  /* calculates the nutation matrix
   * nu		pointer to nutation data structure
   * oe		pointer to epsilon data structure
   */
  static void nut_matrix(nut nu, epsilon oe) 
  {
    double psi, eps;
    double sinpsi, cospsi, sineps, coseps, sineps0, coseps0;
    psi = nu.nutlo[0];
    eps = oe.eps + nu.nutlo[1];
    sinpsi = Math.sin(psi);
    cospsi = Math.cos(psi);
    sineps0 = oe.seps;
    coseps0 = oe.ceps;
    sineps = Math.sin(eps);
    coseps = Math.cos(eps);
    nu.matrix[0][0] = cospsi;
    nu.matrix[0][1] = sinpsi * coseps;
    nu.matrix[0][2] = sinpsi * sineps;
    nu.matrix[1][0] = -sinpsi * coseps0;
    nu.matrix[1][1] = cospsi * coseps * coseps0 + sineps * sineps0;
    nu.matrix[1][2] = cospsi * sineps * coseps0 - coseps * sineps0;
    nu.matrix[2][0] = -sinpsi * sineps0;
    nu.matrix[2][1] = cospsi * coseps * sineps0 - sineps * coseps0;
    nu.matrix[2][2] = cospsi * sineps * sineps0 + coseps * coseps0;
  }

  /**
   * <code>lunar_osc_elem</code> method
   * lunar osculating elements, i.e.
   * osculating node ('true' node) and
   * osculating apogee ('black moon', 'lilith').
   * tjd		julian day
   * ipl		body number, i.e. SEI_TRUE_NODE or SEI_OSCU_APOG
   * iflag	flags (which ephemeris, nutation, etc.)
   * serr		error string
   *
   * definitions and remarks:
   * the osculating node and the osculating apogee are defined
   * as the orbital elements of the momentary lunar orbit.
   * their advantage is that when the moon crosses the ecliptic,
   * it is really at the osculating node, and when it passes
   * its greatest distance from earth it is really at the
   * osculating apogee. with the mean elements this is not
   * the case. (some define the apogee as the second focus of 
   * the lunar ellipse. but, as seen from the geocenter, both 
   * points are in the same direction.)
   * problems:
   * the osculating apogee is given in the 'New International
   * Ephemerides' (Editions St. Michel) as the 'True Lilith'.
   * however, this name is misleading. this point is based on
   * the idea that the lunar orbit can be approximated by an
   * ellipse. 
   * arguments against this: 
   * 1. this procedure considers celestial motions as two body
   *    problems. this is quite good for planets, but not for
   *    the moon. the strong gravitational attraction of the sun 
   *    destroys the idea of an ellipse.
   * 2. the NIE 'True Lilith' has strong oscillations around the
   *    mean one with an amplitude of about 30 degrees. however,
   *    when the moon is in apogee, its distance from the mean
   *    apogee never exceeds 5 degrees.
   * besides, the computation of NIE is INACCURATE. the mistake 
   * reaches 20 arc minutes.
   * According to Santoni, the point was calculated using 'les 58
   * premiers termes correctifs au Perigee moyen' published by
   * Chapront and Chapront-Touze. And he adds: "Nous constatons
   * que meme en utilisant ces 58 termes CORRECTIFS, l'erreur peut
   * atteindre 0,5d!" (p. 13) We avoid this error, computing the
   * orbital elements directly from the position and the speed vector.
   *
   * how about the node? it is less problematic, because we
   * we needn't derive it from an orbital ellipse. we can say:
   * the axis of the osculating nodes is the intersection line of
   * the actual orbital plane of the moon and the plane of the 
   * ecliptic. or: the osculating nodes are the intersections of
   * the two great circles representing the momentary apparent 
   * orbit of the moon and the ecliptic. in this way they make
   * some sense. then, the nodes are really an axis, and they
   * have no geocentric distance. however, in this routine
   * we give a distance derived from the osculating ellipse.
   * the node could also be defined as the intersection axis
   * of the lunar orbital plane and the solar orbital plane,
   * which is not precisely identical to the ecliptic. this 
   * would make a difference of several arcseconds.
   *
   * is it possible to keep the idea of a continuously moving
   * apogee that is exact at the moment when the moon passes
   * its greatest distance from earth?
   * to achieve this, we would probably have to interpolate between 
   * the actual apogees. 
   * the nodes could also be computed by interpolation. the resulting
   * nodes would deviate from the so-called 'true node' by less than
   * 30 arc minutes.
   *
   * sidereal and j2000 true node are first computed for the ecliptic
   * of epoch and then precessed to ecliptic of t0(ayanamsa) or J2000.
   * there is another procedure that computes the node for the ecliptic
   * of t0(ayanamsa) or J2000. it is excluded by
   * #ifdef SID_TNODE_FROM_ECL_T0
   *
   * @param tjd a <code>double</code> value
   * @param ipl an <code>int</code> value
   * @param iflag a <code>long</code> value
   * @param serr a <code>String</code> value
   * @return an <code>int</code> value
   */
  int lunar_osc_elem(double tjd, int ipl, long iflag, String serr) 
  {
    int i, j, istart;
  //  int ipli = SEI_MOON;
    long epheflag = SEFLG_DEFAULTEPH; 
    int retc = ERR; 
    long flg1, flg2;
    plan_data ndp, ndnp, ndap;
    epsilon oe;
    double speed_intv = NODE_CALC_INTV;	/* to silence gcc warning */
    double a, b;
    double xpos[][], xx[][], xxa[][], xnorm[], r[];
    xpos = new double[3][6];
    for (i = 0; i < 3; i++) {
      xpos[i] = new double[6];
    }
    xx = new double[3][6];
    for (i = 0; i < 3; i++) {
      xx[i] = new double[6];
    }
    xxa = new double[3][6];
    for (i = 0; i < 3; i++) {
      xxa[i] = new double[6];
    }
    xnorm = new double[6];
    r = new double[6];
    /*double *xp;*/
    double rxy, rxyz, t, fac, sgn;//, dt;
    double sinnode, cosnode, sinincl, cosincl, sinu, cosu, sinE, cosE;
    double uu, ny, sema, ecce, Gmsm, c2, v2, pp;
    long speedf1, speedf2;
    sid_data sip = swed.sidd;
    epsilon oectmp = new epsilon();
    if ((iflag & SEFLG_SIDEREAL) != 0) {
      calc_epsilon(sip.t0, oectmp);
      oe = oectmp;
    } else if ((iflag & SEFLG_J2000) != 0)
      oe = swed.oec2000;
    else 
      oe = swed.oec;
    ndp = swed.nddat[ipl];
    /* if elements have already been computed for this date, return 
     * if speed flag has been turned on, recompute */
    flg1 = iflag & ~SEFLG_EQUATORIAL & ~SEFLG_XYZ;
    flg2 = ndp.xflgs & ~SEFLG_EQUATORIAL & ~SEFLG_XYZ;
    speedf1 = ndp.xflgs & SEFLG_SPEED;
    speedf2 = iflag & SEFLG_SPEED;
    if (tjd == ndp.teval 
	&& tjd != 0 
	&& flg1 == flg2
	&& (speedf2 == 0 || speedf1 != 0)) {
      ndp.xflgs = iflag;
      ndp.iephe = iflag & SEFLG_EPHMASK;
      return OK;
    }
    /* the geocentric position vector and the speed vector of the
     * moon make up the lunar orbital plane. the position vector 
     * of the node is along the intersection line of the orbital 
     * plane and the plane of the ecliptic.
     * to calculate the osculating node, we need one lunar position
     * with speed.
     * to calculate the speed of the osculating node, we need 
     * three lunar positions and the speed of each of them.
     * this is relatively cheap, if the jpl-moon or the swisseph
     * moon is used. with the moshier moon this is much more 
     * expensive, because then we need 9 lunar positions for 
     * three speeds. but one position and speed can normally
     * be taken from swed.pldat[moon], which corresponds to
     * three moshier moon calculations.
     * the same is also true for the osculating apogee: we need 
     * three lunar positions and speeds.
     */
    /*********************************************
     * now three lunar positions with speeds     * 
     *********************************************/
    if ((iflag & SEFLG_MOSEPH) != 0)
      epheflag = SEFLG_MOSEPH;
    /*
    else if (iflag & SEFLG_SWIEPH)
      epheflag = SEFLG_SWIEPH;
    else if (iflag & SEFLG_JPLEPH)
      epheflag = SEFLG_JPLEPH;
    */
    /* there may be a moon of wrong ephemeris in save area
     * force new computation: */
    swed.pldat[SEI_MOON].teval = 0;
    if ((iflag & SEFLG_SPEED) != 0) 
      istart = 0;
    else 
      istart = 2;
    serr = ""; //$NON-NLS-1$
    switch((int)epheflag) {
    case SEFLG_MOSEPH:
      /* with moshier moon, we need a greater speed_intv, because here the
       * node and apogee oscillate wildly within small intervals */
      speed_intv = NODE_CALC_INTV_MOSH;
      for (i = istart; i <= 2; i++) {
	if (i == 0)
	  t = tjd - speed_intv;
        else if (i == 1)
	  t = tjd + speed_intv;
        else 
	  t = tjd;
	retc = swi_moshmoon(t, NO_SAVE, xpos[i], serr);/**/
	if (retc == ERR)
	  return(retc);
	/* precession and nutation etc. */
	/* retc is always ok */
	retc = swi_plan_for_osc_elem(iflag, t, xpos[i]);
      }
      break;
    default:
      break;
    }

    /*
    if (retc == NOT_AVAILABLE || retc == BEYOND_EPH_LIMITS)
      goto three_positions;
    */
    /*********************************************
     * node with speed                           * 
     *********************************************/
    /* node is always needed, even if apogee is wanted */
    ndnp = swed.nddat[SEI_TRUE_NODE];
    /* three nodes */
    for (i = istart; i <= 2; i++) {
      if (Math.abs(xpos[i][5]) < 1e-15)
	xpos[i][5] = 1e-15;
      fac = xpos[i][2] / xpos[i][5];
      sgn = xpos[i][5] / Math.abs(xpos[i][5]);
      for (j = 0; j <= 2; j++)
	xx[i][j] = (xpos[i][j] - fac * xpos[i][j+3]) * sgn;
    }
    /* now we have the correct direction of the node, the
     * intersection of the lunar plane and the ecliptic plane.
     * the distance is the distance of the point where the tangent
     * of the lunar motion penetrates the ecliptic plane.
     * this can be very large, e.g. j2415080.37372.
     * below, a new distance will be derived from the osculating 
     * ellipse. 
     */
    /* save position and speed */
    for (i = 0; i <= 2; i++) {
      ndnp.x[i] = xx[2][i];
      if ((iflag & SEFLG_SPEED) != 0) {
	b = (xx[1][i] - xx[0][i]) / 2;
	a = (xx[1][i] + xx[0][i]) / 2 - xx[2][i];
	ndnp.x[i+3] = (2 * a + b) / speed_intv;
      } else
	ndnp.x[i+3] = 0;
      ndnp.teval = tjd;
      ndnp.iephe = epheflag;
    }
    /************************************************************
     * apogee with speed                                        * 
     * must be computed anyway to get the node's distance       *
     ************************************************************/
    ndap = swed.nddat[SEI_OSCU_APOG];
    Gmsm =
      GEOGCONST * (1 + 1 / EARTH_MOON_MRAT) /AUNIT/AUNIT/AUNIT*86400.0*86400.0;
    /* three apogees */
    for (i = istart; i <= 2; i++) {
      /* node */
      rxy =  Math.sqrt(xx[i][0] * xx[i][0] + xx[i][1] * xx[i][1]);
      cosnode = xx[i][0] / rxy;	
      sinnode = xx[i][1] / rxy;
      /* inclination */
      SwephLib.swi_cross_prod(xpos[i], 0, xpos[i], 3, xnorm);
      rxy =  xnorm[0] * xnorm[0] + xnorm[1] * xnorm[1];
      c2 = (rxy + xnorm[2] * xnorm[2]);
      rxyz = Math.sqrt(c2);
      rxy = Math.sqrt(rxy);
      sinincl = rxy / rxyz;
      cosincl = Math.sqrt(1 - sinincl * sinincl);
      /* argument of latitude */
      cosu = xpos[i][0] * cosnode + xpos[i][1] * sinnode;
      sinu = xpos[i][2] / sinincl;	
      uu = Math.atan2(sinu, cosu);	
      /* semi-axis */
      rxyz = Math.sqrt(square_sum(xpos[i], 0));
      v2 = square_sum(xpos[i], 3);
      sema = 1 / (2 / rxyz - v2 / Gmsm);	
      /* eccentricity */
      pp = c2 / Gmsm;
      ecce = Math.sqrt(1 - pp / sema);	
      /* eccentric anomaly */
      cosE = 1 / ecce * (1 - rxyz / sema);	
      sinE = 1 / ecce / Math.sqrt(sema * Gmsm) * dot_prod(xpos[i], 0,
							  xpos[i], 3);
      /* true anomaly */
      ny = 2 * Math.atan(Math.sqrt((1+ecce)/(1-ecce)) * sinE / (1 + cosE));
      /* distance of apogee from ascending node */
      xxa[i][0] = SwephLib.swi_mod2PI(uu - ny + Math.PI);
      xxa[i][1] = 0;			/* latitude */
      xxa[i][2] = sema * (1 + ecce);	/* distance */
      /* transformation to ecliptic coordinates */
      SwephLib.swi_polcart(xxa[i], 0, xxa[i], 0);
      SwephLib.swi_coortrf2(xxa[i], 0, xxa[i], 0, -sinincl, cosincl);
      SwephLib.swi_cartpol(xxa[i], 0, xxa[i], 0);
      /* adding node, we get apogee in ecl. coord. */
      xxa[i][0] += Math.atan2(sinnode, cosnode);
      SwephLib.swi_polcart(xxa[i], 0, xxa[i], 0);
      /* new distance of node from orbital ellipse:
       * true anomaly of node: */
      ny = SwephLib.swi_mod2PI(ny - uu);
      /* eccentric anomaly */
      cosE = Math.cos(2 * Math.atan(Math.tan(ny / 2)
				    / Math.sqrt((1+ecce) / (1-ecce))));
      /* new distance */
      r[0] = sema * (1 - ecce * cosE);
      /* old node distance */
      r[1] = Math.sqrt(square_sum(xx[i], 0));
      /* correct length of position vector */
      for (j = 0; j <= 2; j++)
	xx[i][j] *= r[0] / r[1];
    }
    /* save position and speed */
    for (i = 0; i <= 2; i++) {
      /* apogee */
      ndap.x[i] = xxa[2][i];
      if ((iflag & SEFLG_SPEED) != 0)
	ndap.x[i+3] = (xxa[1][i] - xxa[0][i]) / speed_intv / 2;
      else
	ndap.x[i+3] = 0;
      ndap.teval = tjd;
      ndap.iephe = epheflag;
      /* node */
      ndnp.x[i] = xx[2][i];
      if ((iflag & SEFLG_SPEED) != 0)
	ndnp.x[i+3] = (xx[1][i] - xx[0][i]) / speed_intv / 2;/**/    
      else
	ndnp.x[i+3] = 0;
    }
    /**********************************************************************
     * precession and nutation have already been taken into account
     * because the computation is on the basis of lunar positions
     * that have gone through swi_plan_for_osc_elem. 
     * light-time is already contained in lunar positions.
     * now compute polar and equatorial coordinates:
     **********************************************************************/
    for (j = 0; j <= 1; j++) {
      double x[];
      x = new double[6];
      if (j == 0) 
	ndp = swed.nddat[SEI_TRUE_NODE];
      else
	ndp = swed.nddat[SEI_OSCU_APOG];
      for (i = 0; i < 24; i++) {
	ndp.xreturn[i] = 0;
      }
      /* cartesian ecliptic */
      for (i = 0; i <= 5; i++) 
	ndp.xreturn[6+i] = ndp.x[i];
      /* polar ecliptic */
      SwephLib.swi_cartpol_sp(ndp.xreturn, 6, ndp.xreturn, 0);
      /* cartesian equatorial */
      SwephLib.swi_coortrf2(ndp.xreturn, 6, ndp.xreturn, 18,
			    -oe.seps, oe.ceps);
      if ((iflag & SEFLG_SPEED) != 0)
	SwephLib.swi_coortrf2(ndp.xreturn, 9, ndp.xreturn, 21,
			      -oe.seps, oe.ceps);
      /* sideral: we return NORMAL equatorial coordinates, there are no
       * sidereal ones */
      if ((iflag & SEFLG_SIDEREAL) != 0) {
	/* to J2000 */
	SwephLib.swi_precess(ndp.xreturn, 18, sip.t0, J_TO_J2000);
	if ((iflag & SEFLG_SPEED) != 0)
	  swi_precess_speed(ndp.xreturn, 21, sip.t0, J_TO_J2000);
	if ((iflag & SEFLG_J2000) == 0) {
	  /* to tjd */
	  SwephLib.swi_precess(ndp.xreturn, 18, tjd, J2000_TO_J);
	  if ((iflag & SEFLG_SPEED) != 0)
	    swi_precess_speed(ndp.xreturn, 21, tjd, J2000_TO_J);
	}
      }
      if ((iflag & SEFLG_NONUT) == 0) {
	SwephLib.swi_coortrf2(ndp.xreturn, 18, ndp.xreturn, 18,
			      -swed.nut.snut, swed.nut.cnut);
	if ((iflag & SEFLG_SPEED) != 0)
	  SwephLib.swi_coortrf2(ndp.xreturn, 21, ndp.xreturn, 21,
				swed.nut.snut, swed.nut.cnut);
      }
      /* polar equatorial */
      SwephLib.swi_cartpol_sp(ndp.xreturn, 18, ndp.xreturn, 12);
      ndp.xflgs = iflag;
      ndp.iephe = iflag & SEFLG_EPHMASK;
      /* node and apogee are already referred to t0; 
       * nothing has to be done */
      if ((iflag & SEFLG_SIDEREAL) != 0) {
	/* node and apogee are referred to t; 
	 * the ecliptic position must be transformed to t0 */
	/* rigorous algorithm */
	if ((swed.sidd.sid_mode & SE_SIDBIT_ECL_T0) != 0 
	    || (swed.sidd.sid_mode & SE_SIDBIT_SSY_PLANE) != 0) {
	  for (i = 0; i <= 5; i++)
	    x[i] = ndp.xreturn[18+i];
	  /* remove nutation */
	  if ((iflag & SEFLG_NONUT) == 0)
	    swi_nutate(x, iflag, true);
	  /* precess to J2000 */
	  SwephLib.swi_precess(x, 0, tjd, J_TO_J2000);
	  if ((iflag & SEFLG_SPEED) != 0)
	    swi_precess_speed(x, 0, tjd, J_TO_J2000);
	  if ((swed.sidd.sid_mode & SE_SIDBIT_ECL_T0) != 0) 
	    swi_trop_ra2sid_lon(x, ndp.xreturn, 6, ndp.xreturn, 18,
				iflag, null);
	  /* project onto solar system equator */
	  else if ((swed.sidd.sid_mode & SE_SIDBIT_SSY_PLANE) != 0)
	    swi_trop_ra2sid_lon_sosy(x, ndp.xreturn, 6, ndp.xreturn, 18,
				     iflag, null);
	  /* to polar */
	  SwephLib.swi_cartpol_sp(ndp.xreturn, 6, ndp.xreturn, 0);
	  SwephLib.swi_cartpol_sp(ndp.xreturn, 18, ndp.xreturn, 12);
	  /* traditional algorithm;
	   * this is a bit clumsy, but allows us to keep the
	   * sidereal code together */
	} else {
	  SwephLib.swi_cartpol_sp(ndp.xreturn, 6, ndp.xreturn, 0); 
	  ndp.xreturn[0] -= swe_get_ayanamsa(ndp.teval) * DEGTORAD;
	  SwephLib.swi_polcart_sp(ndp.xreturn, 0, ndp.xreturn, 6); 
	}
      } else if ((iflag & SEFLG_J2000) != 0) {
	/* node and apogee are referred to t; 
	 * the ecliptic position must be transformed to J2000 */
	for (i = 0; i <= 5; i++)
	  x[i] = ndp.xreturn[18+i];
	/* precess to J2000 */
	SwephLib.swi_precess(x, 0, tjd, J_TO_J2000);
	if ((iflag & SEFLG_SPEED) != 0)
	  swi_precess_speed(x, 0, tjd, J_TO_J2000);
	for (i = 0; i <= 5; i++)
	  ndp.xreturn[18+i] = x[i];
	SwephLib.swi_cartpol_sp(ndp.xreturn, 18, ndp.xreturn, 12);
	SwephLib.swi_coortrf2(ndp.xreturn, 18, ndp.xreturn, 6,
			      swed.oec2000.seps, swed.oec2000.ceps);
	if ((iflag & SEFLG_SPEED) != 0)
	  SwephLib.swi_coortrf2(ndp.xreturn, 21, ndp.xreturn, 9,
				swed.oec2000.seps, swed.oec2000.ceps);
	SwephLib.swi_cartpol_sp(ndp.xreturn, 6, ndp.xreturn, 0);
      }
      /********************** 
       * radians to degrees *
       **********************/
      /*if (!(iflag & SEFLG_RADIANS)) {*/
      for (i = 0; i < 2; i++) {
        ndp.xreturn[i] *= RADTODEG;		/* ecliptic */
        ndp.xreturn[i+3] *= RADTODEG;
        ndp.xreturn[i+12] *= RADTODEG;	/* equator */
        ndp.xreturn[i+15] *= RADTODEG;
      }
      ndp.xreturn[0] = SwephLib.swe_degnorm(ndp.xreturn[0]);
      ndp.xreturn[12] = SwephLib.swe_degnorm(ndp.xreturn[12]);
      /*}*/
    }
    return OK;
  }

  /**
   * <code>swi_plan_for_osc_elem</code> method
   * transforms the position of the moon in a way we can use it
   * for calculation of osculating node and apogee:
   * precession and nutation (attention to speed vector!)
   * according to flags
   * iflag	flags
   * tjd          time for which the element is computed
   *              i.e. date of ecliptic
   * xx           array equatorial cartesian position and speed
   * serr         error string
   *
   * @param iflag a <code>long</code> value
   * @param tjd a <code>double</code> value
   * @param xx[] a <code>double</code> value
   * @return an <code>int</code> value
   */
  int swi_plan_for_osc_elem(long iflag, double tjd, double xx[])
  {
    int i;
    double x[];
    x = new double[6];
    nut nuttmp = new nut();
    nut nutp = nuttmp;	/* dummy assign, to silence gcc warning */
    epsilon oe = swed.oec;
    epsilon oectmp = new epsilon();
    /************************************************
     * precession, equator 2000 -> equator of date  *
     * attention: speed vector has to be rotated,   *
     * but daily precession 0.137" may not be added!*/
    sid_data sip = swed.sidd;
    /* For sidereal calculation we need node refered*
     * to ecliptic of t0 of ayanamsa                *
     ************************************************/
    if ((iflag & SEFLG_SIDEREAL) != 0) {
      tjd = sip.t0;
      SwephLib.swi_precess(xx, 0, tjd, J2000_TO_J);
      SwephLib.swi_precess(xx, 3, tjd, J2000_TO_J); 
      calc_epsilon(tjd, oectmp);
      oe = oectmp;
    } else if ((iflag & SEFLG_J2000) == 0) {
      SwephLib.swi_precess(xx, 0, tjd, J2000_TO_J);
      SwephLib.swi_precess(xx, 3, tjd, J2000_TO_J); 
      /* epsilon */
      if (tjd == swed.oec.teps)
	oe = swed.oec;
      else if (tjd == J2000)
	oe = swed.oec2000;
      else  {
	calc_epsilon(tjd, oectmp);
	oe = oectmp;
      }
    } else	/* if SEFLG_J2000 */
      oe = swed.oec2000;
    /************************************************
     * nutation                                     *
     * again: speed vector must be rotated, but not *
     * added 'speed' of nutation                    *
     ************************************************/
    if ((iflag & SEFLG_NONUT) == 0) {
      if (tjd == swed.nut.tnut)
	nutp = swed.nut;
      else if (tjd == J2000)
	nutp = swed.nut2000;
      else if (tjd == swed.nutv.tnut)
	nutp = swed.nutv;
      else {
	nutp = nuttmp;
	SwephLib.swi_nutation(tjd, nutp.nutlo);
	nutp.tnut = tjd;
	nutp.snut = Math.sin(nutp.nutlo[1]);
	nutp.cnut = Math.cos(nutp.nutlo[1]);
	nut_matrix(nutp, oe);
      }
      for (i = 0; i <= 2; i++) 
	x[i] = xx[0] * nutp.matrix[0][i] + 
	  xx[1] * nutp.matrix[1][i] + 
	  xx[2] * nutp.matrix[2][i];
      /* speed:
       * rotation only */
      for (i = 0; i <= 2; i++) 
	x[i+3] = xx[3] * nutp.matrix[0][i] + 
	  xx[4] * nutp.matrix[1][i] + 
	  xx[5] * nutp.matrix[2][i];
      for (i = 0; i <= 5; i++) 
	xx[i] = x[i];
    }
    /************************************************
     * transformation to ecliptic                   *
     ************************************************/
    SwephLib.swi_coortrf2(xx, 0, xx, 0, oe.seps, oe.ceps);
    SwephLib.swi_coortrf2(xx, 3, xx, 3, oe.seps, oe.ceps);
    if ((iflag & SEFLG_SIDEREAL) != 0) {
      /* subtract ayan_t0 */
      SwephLib.swi_cartpol_sp(xx, 0, xx, 0);
      xx[0] -= sip.ayan_t0;
      SwephLib.swi_polcart_sp(xx, 0, xx, 0);
    } else 
      if ((iflag & SEFLG_NONUT) == 0) {
	SwephLib.swi_coortrf2(xx, 0, xx, 0, nutp.snut, nutp.cnut);
	SwephLib.swi_coortrf2(xx, 3, xx, 3, nutp.snut, nutp.cnut);
      }
    return(OK);
  }

  /**
   * Describe <code>meff</code> method here.
   *
   * @param r a <code>double</code> value
   * @return a <code>double</code> value
   */
  double meff(double r)
  {
    double eff_arr_r[] = {
      /*
       * r ,
       * m_eff for photon passing the sun at min distance r (fraction of Rsun)
       * the values where computed with sun_model.c, which is a classic
       * treatment of a photon passing a gravity field, multiplied by 2.
       * The sun mass distribution m(r) is from Michael Stix, The Sun, p. 47.
       */
      1.000,
      0.990,
      0.980,
      0.970,
      0.960,
      0.950,
      0.940,
      0.930,
      0.920,
      0.910,
      0.900,
      0.890,
      0.880,
      0.870,
      0.860,
      0.850,
      0.840,
      0.830,
      0.820,
      0.810,
      0.800,
      0.790,
      0.780,
      0.770,
      0.760,
      0.750,
      0.740,
      0.730,
      0.720,
      0.710,
      0.700,
      0.690,
      0.680,
      0.670,
      0.660,
      0.650,
      0.640,
      0.630,
      0.620,
      0.610,
      0.600,
      0.590,
      0.580,
      0.570,
      0.560,
      0.550,
      0.540,
      0.530,
      0.520,
      0.510,
      0.500,
      0.490,
      0.480,
      0.470,
      0.460,
      0.450,
      0.440,
      0.430,
      0.420,
      0.410,
      0.400,
      0.390,
      0.380,
      0.370,
      0.360,
      0.350,
      0.340,
      0.330,
      0.320,
      0.310,
      0.300,
      0.290,
      0.280,
      0.270,
      0.260,
      0.250,
      0.240,
      0.230,
      0.220,
      0.210,
      0.200,
      0.190,
      0.180,
      0.170,
      0.160,
      0.150,
      0.140,
      0.130,
      0.120,
      0.110,
      0.100,
      0.090,
      0.080,
      0.070,
      0.060,
      0.050,
      0.040,
      0.030,
      0.020,
      0.010,
      0.000
    };
    double eff_arr_m[] = {
      /*
       * r ,
       * m_eff for photon passing the sun at min distance r (fraction of Rsun)
       * the values where computed with sun_model.c, which is a classic
       * treatment of a photon passing a gravity field, multiplied by 2.
       * The sun mass distribution m(r) is from Michael Stix, The Sun, p. 47.
       */
      1.000000,
      0.999979,
      0.999940,
      0.999881,
      0.999811,
      0.999724,
      0.999622,
      0.999497,
      0.999354,
      0.999192,
      0.999000,
      0.998786,
      0.998535,
      0.998242,
      0.997919,
      0.997571,
      0.997198,
      0.996792,
      0.996316,
      0.995791,
      0.995226,
      0.994625,
      0.993991,
      0.993326,
      0.992598,
      0.991770,
      0.990873,
      0.989919,
      0.988912,
      0.987856,
      0.986755,
      0.985610,
      0.984398,
      0.982986,
      0.981437,
      0.979779,
      0.978024,
      0.976182,
      0.974256,
      0.972253,
      0.970174,
      0.968024,
      0.965594,
      0.962797,
      0.959758,
      0.956515,
      0.953088,
      0.949495,
      0.945741,
      0.941838,
      0.937790,
      0.933563,
      0.928668,
      0.923288,
      0.917527,
      0.911432,
      0.905035,
      0.898353,
      0.891022,
      0.882940,
      0.874312,
      0.865206,
      0.855423,
      0.844619,
      0.833074,
      0.820876,
      0.808031,
      0.793962,
      0.778931,
      0.763021,
      0.745815,
      0.727557,
      0.708234,
      0.687583,
      0.665741,
      0.642597,
      0.618252,
      0.592586,
      0.565747,
      0.537697,
      0.508554,
      0.478420,
      0.447322,
      0.415454,
      0.382892,
      0.349955,
      0.316691,
      0.283565,
      0.250431,
      0.218327,
      0.186794,
      0.156287,
      0.128421,
      0.102237,
      0.077393,
      0.054833,
      0.036361,
      0.020953,
      0.009645,
      0.002767,
      0.000000
    };
    double f, m;
    int i;
    if (r <= 0)
      return 0.0;
    else if (r >= 1)
      return 1.0;
    for (i = 0; eff_arr_r[i] > r; i++)
      ;	/* empty body */
    f = (r - eff_arr_r[i-1]) / (eff_arr_r[i] - eff_arr_r[i-1]);
    m = eff_arr_m[i-1] + f * (eff_arr_m[i] - eff_arr_m[i-1]);
    return m;
  }

  /**
   * Describe <code>denormalize_positions</code> method here.
   *
   * @param x0[] a <code>double</code> value
   * @param x1[] a <code>double</code> value
   * @param x2[] a <code>double</code> value
   */
  void denormalize_positions(double x0[], double x1[], double x2[]) 
  {
    int i;
    /* x*[0] = ecliptic longitude, x*[12] = rectascension */
    for (i = 0; i <= 12; i += 12) {
      if (x1[i] - x0[i] < -180)
	x0[i] -= 360;
      if (x1[i] - x0[i] > 180)
	x0[i] += 360;
      if (x1[i] - x2[i] < -180)
	x2[i] -= 360;
      if (x1[i] - x2[i] > 180)
	x2[i] += 360;
    }
  }

  void calc_speed(double x0[], double x1[], double x2[], double dt)
  {
    int i, j, k;
    double a, b;
    for (j = 0; j <= 18; j += 6) {
      for (i = 0; i < 3; i++) {
	k = j + i;
	b = (x2[k] - x0[k]) / 2;
	a = (x2[k] + x0[k]) / 2 - x1[k];
	x1[k+3] = (2 * a + b) / dt;
      }
    }
  }

  /**
   * Describe <code>swi_check_ecliptic</code> method here.
   *
   * @param tjd a <code>double</code> value
   */
  void swi_check_ecliptic(double tjd)
  {
    if (swed.oec2000.teps != J2000) {
      calc_epsilon(J2000, swed.oec2000);
    }
    if (tjd == J2000) {
      swed.oec.teps = swed.oec2000.teps;
      swed.oec.eps = swed.oec2000.eps;
      swed.oec.seps = swed.oec2000.seps;
      swed.oec.ceps = swed.oec2000.ceps;
      return;
    }
    if (swed.oec.teps != tjd || tjd == 0) {
      calc_epsilon(tjd, swed.oec);
    }
  }

  /**
   * <code>swi_check_nutation</code> method
   * computes nutation, if it is wanted and has not yet been computed.
   * if speed flag has been turned on since last computation, 
   * nutation is recomputed
  *
   * @param tjd a <code>double</code> value
   * @param iflag a <code>long</code> value
   */
  long nutflag = 0;
  void swi_check_nutation(double tjd, long iflag)
  {
    long speedf1, speedf2;
    double t;
    speedf1 = nutflag & SEFLG_SPEED;
    speedf2 = iflag & SEFLG_SPEED;
    if ((iflag & SEFLG_NONUT) == 0
	&& (tjd != swed.nut.tnut || tjd == 0
	    || (speedf1 == 0 && speedf2 != 0))) {
      SwephLib.swi_nutation(tjd, swed.nut.nutlo);
      swed.nut.tnut = tjd;
      swed.nut.snut = Math.sin(swed.nut.nutlo[1]);
      swed.nut.cnut = Math.cos(swed.nut.nutlo[1]);
      nutflag = iflag;
      nut_matrix(swed.nut, swed.oec);
      if ((iflag & SEFLG_SPEED) != 0) {
	/* once more for 'speed' of nutation, which is needed for 
	 * planetary speeds */
	t = tjd - NUT_SPEED_INTV;
	SwephLib.swi_nutation(t, swed.nutv.nutlo);
	swed.nutv.tnut = t;
	swed.nutv.snut = Math.sin(swed.nutv.nutlo[1]);
	swed.nutv.cnut = Math.cos(swed.nutv.nutlo[1]);
	nut_matrix(swed.nutv, swed.oec);
      }
    }
  }

  /**
   * Describe <code>plaus_iflag</code> method here.
   *
   * @param iflag a <code>long</code> value
   * @return a <code>long</code> value
   */
  long plaus_iflag(long iflag)
  {
    long epheflag = 0;
    /* if topocentric bit, turn helio- and barycentric bits off */
    if ((iflag & SEFLG_TOPOCTR) != 0)
      iflag = iflag & ~(SEFLG_HELCTR | SEFLG_BARYCTR); 
    /* if heliocentric bit, turn aberration and deflection off */
    if ((iflag & SEFLG_HELCTR) != 0) 
      iflag |= SEFLG_NOABERR | SEFLG_NOGDEFL; /*iflag |= SEFLG_TRUEPOS;*/
    /* same, if barycentric bit */
    if ((iflag & SEFLG_BARYCTR) != 0) 
      iflag |= SEFLG_NOABERR | SEFLG_NOGDEFL; /*iflag |= SEFLG_TRUEPOS;*/
    /* if no_precession bit is set, set also no_nutation bit */
    if ((iflag & SEFLG_J2000) != 0)
      iflag |= SEFLG_NONUT;
    /* if sidereal bit is set, set also no_nutation bit */
    if ((iflag & SEFLG_SIDEREAL) != 0)
      iflag |= SEFLG_NONUT;
    /* if truepos is set, turn off grav. defl. and aberration */
    if ((iflag & SEFLG_TRUEPOS) != 0)
      iflag |= (SEFLG_NOGDEFL | SEFLG_NOABERR);
    if ((iflag & SEFLG_MOSEPH) != 0)
      epheflag = SEFLG_MOSEPH;
    if ((iflag & SEFLG_SWIEPH) != 0)
      epheflag = SEFLG_SWIEPH;
    if ((iflag & SEFLG_JPLEPH) != 0)
      epheflag = SEFLG_JPLEPH;
    if (epheflag == 0)
      epheflag = SEFLG_DEFAULTEPH;
    epheflag = SEFLG_MOSEPH;
    /* delete wrong ephe bits from flag */
    iflag = (iflag & ~SEFLG_EPHMASK) | epheflag;

    return iflag;
  }

  /* set directory path of ephemeris files */
  public void swe_set_ephe_path(String path) {
  }

  /* set file name of JPL file */
  public void swe_set_jpl_file(String fname) {
  }

  /* get planet name */
  public String swe_get_planet_name(int ipl, String spname) {
    int i;
  //  long retc;
  //  double xp[];
  //  xp = new double[6];
    if (ipl != 0 && ipl == swed.i_saved_planet_name) {
      spname = swed.saved_planet_name;
      return spname;
    }
    switch(ipl) {
    case SE_SUN: 
      spname = SE_NAME_SUN;
      break;
    case SE_MOON: 
      spname = SE_NAME_MOON;
      break;
    case SE_MERCURY: 
      spname = SE_NAME_MERCURY;
      break;
    case SE_VENUS: 
      spname = SE_NAME_VENUS;
      break;
    case SE_MARS: 
      spname = SE_NAME_MARS;
      break;
    case SE_JUPITER: 
      spname = SE_NAME_JUPITER;
      break;
    case SE_SATURN: 
      spname = SE_NAME_SATURN;
      break;
    case SE_URANUS: 
      spname = SE_NAME_URANUS;
      break;
    case SE_NEPTUNE: 
      spname = SE_NAME_NEPTUNE;
      break;
    case SE_PLUTO: 
      spname = SE_NAME_PLUTO;
      break;
    case SE_MEAN_NODE: 
      spname = SE_NAME_MEAN_NODE;
      break;
    case SE_TRUE_NODE: 
      spname = SE_NAME_TRUE_NODE;
      break;
    case SE_MEAN_APOG: 
      spname = SE_NAME_MEAN_APOG;
      break;
    case SE_OSCU_APOG: 
      spname = SE_NAME_OSCU_APOG;
      break;  
    case SE_EARTH: 
      spname = SE_NAME_EARTH;
      break;
    case SE_CHIRON: 
    case SE_AST_OFFSET + MPC_CHIRON: 
      spname = SE_NAME_CHIRON;
      break;
    case SE_PHOLUS: 
    case SE_AST_OFFSET + MPC_PHOLUS: 
      spname = SE_NAME_PHOLUS;
      break;
    case SE_CERES: 
    case SE_AST_OFFSET + MPC_CERES: 
      spname = SE_NAME_CERES;
      break;
    case SE_PALLAS: 
    case SE_AST_OFFSET + MPC_PALLAS: 
      spname = SE_NAME_PALLAS;
      break;
    case SE_JUNO: 
    case SE_AST_OFFSET + MPC_JUNO: 
      spname = SE_NAME_JUNO;
      break;
    case SE_VESTA: 
    case SE_AST_OFFSET + MPC_VESTA: 
      spname = SE_NAME_VESTA;
      break;
    default: 
      /* fictitious planets */
      if (ipl >= SE_FICT_OFFSET && ipl <= SE_FICT_MAX) {
        spname = swi_get_fict_name(ipl - SE_FICT_OFFSET);
        break;
      }
      /* asteroids */
      if (ipl > SE_AST_OFFSET) {
	/* if name is already available */
        /* If there is a provisional designation only in ephemeris file,
         * we look for a name in seasnam.txt, which can be updated by
         * the user.
         * Some old ephemeris files return a '?' in the first position.
         * There are still a couple of unnamed bodies that got their
         * provisional designation before 1925, when the current method
         * of provisional designations was introduced. They have an 'A'
         * as the first character, e.g. A924 RC. 
         * The file seasnam.txt may contain comments starting with '#'.
         * There must be at least two columns: 
         * 1. asteroid catalog number
         * 2. asteroid name
         * The asteroid number may or may not be in brackets
         */
      } else  {
	i = ipl;
	spname = ""; //$NON-NLS-1$
	spname += i;
      }
      break;
    }
    if (spname.length() < 80) {
      swed.i_saved_planet_name = ipl;
      swed.saved_planet_name = spname;
    }
    return spname;
  }

  /**
   * <code>swe_set_topo</code> method
   * set geographic position of observer
   *
   * @param geolon a <code>double</code> value
   * @param geolat a <code>double</code> value
   * @param geoalt a <code>double</code> value
   */
  public void
    swe_set_topo(double geolon, double geolat, double geoalt) {
    swed.topd.geolon = geolon;
    swed.topd.geolat = geolat;
    swed.topd.geoalt = geoalt;
    swed.geopos_is_set = true;
    /* to force new calculation of observer position vector */
    swed.topd.teval = 0;
    /* to force new calculation of light-time etc. 
     */
    swi_force_app_pos_etc();
  }

  void swi_force_app_pos_etc()
  {
    int i;
    for (i = 0; i < SEI_NPLANETS; i++)
      swed.pldat[i].xflgs = -1;
    for (i = 0; i < SEI_NNODE_ETC; i++)
      swed.nddat[i].xflgs = -1;
    for (i = 0; i < SE_NPLANETS; i++) {
      swed.savedat[i].tsave = 0;
      swed.savedat[i].iflgsave = -1;
    }
  }

  int swi_get_observer(double tjd, long iflag, 
		       boolean do_save, double xobs[], String serr)
  {
    int i;
    double sidt, delt, tjd_ut, eps, nut, nutlo[];
    nutlo = new double[2];
    double f = EARTH_OBLATENESS;
    double re = EARTH_RADIUS; 
    double cosfi, sinfi, cc, ss, cosl, sinl, h;
    if (!swed.geopos_is_set) {
      serr = "geographic position has not been set"; //$NON-NLS-1$
      return ERR;
    }
    /* geocentric position of observer depends on sidereal time,
     * which depends on UT. 
     * compute UT from ET. this UT will be slightly different
     * from the user's UT, but this difference is extremely small.
     */
    delt = SwephLib.swe_deltat(tjd);
    tjd_ut = tjd - delt;
    if (swed.oec.teps == tjd && swed.nut.tnut == tjd) {
      eps = swed.oec.eps;
      nutlo[1] = swed.nut.nutlo[1];
      nutlo[0] = swed.nut.nutlo[0];
    } else {
      eps = SwephLib.swi_epsiln(tjd);
      if ((iflag & SEFLG_NONUT) == 0) 
	SwephLib.swi_nutation(tjd, nutlo);
    }
    if ((iflag & SEFLG_NONUT) != 0) {
      nut = 0;
    } else {
      eps += nutlo[1];
      nut = nutlo[0];
    }
    /* mean or apparent sidereal time, depending on whether or
     * not SEFLG_NONUT is set */
    sidt = SwephLib.swe_sidtime0(tjd_ut, eps, nut);
    sidt *= 15;	/* in degrees */
    /* length of position and speed vectors;
     * the height above sea level must be taken into account.
     * with the moon, an altitude of 3000 m makes a difference 
     * of about 2 arc seconds.
     * height is referred to the average sea level. however, 
     * the spheroid (geoid), which is defined by the average 
     * sea level (or rather by all points of same gravitational
     * potential), is of irregular shape and cannot easily
     * be taken into account. therefore, we refer height to 
     * the surface of the ellipsoid. the resulting error 
     * is below 500 m, i.e. 0.2 - 0.3 arc seconds with the moon.
     */
    cosfi = Math.cos(swed.topd.geolat * DEGTORAD);
    sinfi = Math.sin(swed.topd.geolat * DEGTORAD);
    cc= 1 / Math.sqrt(cosfi * cosfi + (1-f) * (1-f) * sinfi * sinfi); 
    ss= (1-f) * (1-f) * cc; 
    cosl = Math.cos((swed.topd.geolon + sidt) * DEGTORAD);
    sinl = Math.sin((swed.topd.geolon + sidt) * DEGTORAD);
    h = swed.topd.geoalt;
    xobs[0] = (re * cc + h) * cosfi * cosl;
    xobs[1] = (re * cc + h) * cosfi * sinl;
    xobs[2] = (re * ss + h) * sinfi;
    SwephLib.swi_cartpol(xobs, 0, xobs, 0);
    xobs[3] = EARTH_ROT_SPEED;		/* speed */
    xobs[4] = xobs[5] = 0;
    SwephLib.swi_polcart_sp(xobs, 0, xobs, 0);
    /* to AUNIT */
    for (i = 0; i <= 5; i++)
      xobs[i] /= AUNIT;
    /* subtract nutation, set backward flag */
    if ((iflag & SEFLG_NONUT) == 0) {
      SwephLib.swi_coortrf2(xobs, 0, xobs, 0, -swed.nut.snut, swed.nut.cnut);
      if ((iflag & SEFLG_SPEED) != 0)
	SwephLib.swi_coortrf2(xobs, 3, xobs, 3, -swed.nut.snut, swed.nut.cnut);
      swi_nutate(xobs, iflag, true);
    }
    /* precess to J2000 */
    SwephLib.swi_precess(xobs, 0, tjd, J_TO_J2000);
    if ((iflag & SEFLG_SPEED) != 0)
      swi_precess_speed(xobs, 0, tjd, J_TO_J2000);
    /* save */
    if (do_save) {
      for (i = 0; i <= 5; i++)
	swed.topd.xobs[i] = xobs[i];
      swed.topd.teval = tjd;
      swed.topd.tjd_ut = tjd_ut;	/* -> save area */
    }
    return OK;
  }

  /* set sidereal mode */
  public void
    swe_set_sid_mode(int sid_mode, double t0, double ayan_t0) {
    /* Ayanamsas */
    double ayanamsa_t0[] = {
      2433282.5,	/* 0: Fagan/Bradley (Default) */
      J1900,   /* 1: Lahiri (Robert Hand) */
      J1900,   /* 2: De Luce (Robert Hand) */
      J1900,   /* 3: Raman (Robert Hand) */
      J1900,   /* 4: Ushashashi (Robert Hand) */
      J1900,  /* 5: Krishnamurti (Robert Hand) */
      J1900, /* 6: Djwhal Khool; (Graham Dawson)  
	      *    Aquarius entered on 1 July 2117 */
      J1900,  /* 7: Yukteshwar; (David Cochrane) */
      J1900,  /* 8: JN Bhasin; (David Cochrane) */
      1684532.5,      /* 9: Babylonian, Kugler 1 */
      1684532.5,      /*10: Babylonian, Kugler 2 */
      1684532.5,      /*11: Babylonian, Kugler 3 */
      1684532.5,      /*12: Babylonian, Huber */
      1673941,       /*13: Babylonian, Mercier;
		       eta Piscium culminates with zero point */
      1684532.5,   /*14: t0 is defined by Aldebaran at 15 Taurus */
      1674484,        /*15: Hipparchos */
      1927135.8747793,       /*16: Sassanian */
      1746443.513,           /*17: Galactic Center at 0 Sagittarius */
      J2000,	                /*18: J2000 */
      J1900,	                /*19: J1900 */
      B1950	                /*20: B1950 */
    };
    double ayanamsa_ayant0[] = {
      24.042044444,	/* 0: Fagan/Bradley (Default) */
      360 - 337.53953,   /* 1: Lahiri (Robert Hand) */
      360 - 333.58695,   /* 2: De Luce (Robert Hand) */
      360 - 338.98556,   /* 3: Raman (Robert Hand) */
      360 - 341.33904,   /* 4: Ushashashi (Robert Hand) */
      360 - 337.636111,  /* 5: Krishnamurti (Robert Hand) */
      360 - 333.0369024, /* 6: Djwhal Khool; (Graham Dawson)  
			  *    Aquarius entered on 1 July 2117 */
      360 - 338.917778,  /* 7: Yukteshwar; (David Cochrane) */
      360 - 338.634444,  /* 8: JN Bhasin; (David Cochrane) */
      -3.36667,      /* 9: Babylonian, Kugler 1 */
      -4.76667,      /*10: Babylonian, Kugler 2 */
      -5.61667,      /*11: Babylonian, Kugler 3 */
      -4.56667,      /*12: Babylonian, Huber */
      -5.079167,       /*13: Babylonian, Mercier;
			*    eta Piscium culminates with zero point */
      -4.44088389,   /*14: t0 is defined by Aldebaran
		      *	at 15 Taurus */
      -9.33333,        /*15: Hipparchos */
      0,       /*16: Sassanian */
      0,           /*17: Galactic Center at 0 Sagittarius */
      0,	                /*18: J2000 */
      0,	                /*19: J1900 */
      0	                /*20: B1950 */
    };

    sid_data sip = swed.sidd;
    sip.sid_mode = sid_mode;
    if (sid_mode >= SE_SIDBITS)
      sid_mode %= SE_SIDBITS;
    /* standard equinoxes: positions always referred to ecliptic of t0 */
    if (sid_mode == SE_SIDM_J2000 
	|| sid_mode == SE_SIDM_J1900 
	|| sid_mode == SE_SIDM_B1950) {
      sip.sid_mode |= SE_SIDBIT_ECL_T0;
    }
    if (sid_mode >= SE_NSIDM_PREDEF && sid_mode != SE_SIDM_USER) {
      sip.sid_mode = sid_mode = SE_SIDM_FAGAN_BRADLEY;
    }
    swed.ayana_is_set = true;
    if (sid_mode == SE_SIDM_USER) {
      sip.t0 = t0;
      sip.ayan_t0 = ayan_t0;
    } else {
      sip.t0 = ayanamsa_t0[sid_mode];
      sip.ayan_t0 = ayanamsa_ayant0[sid_mode];
    }
    swi_force_app_pos_etc();
  }

  /* get ayanamsa */
  /* the ayanamsa (precession in longitude) 
   * according to Newcomb's definition: 360 -
   * longitude of the vernal point of t referred to the
   * ecliptic of t0.
   */
  public double swe_get_ayanamsa(double tjd_et) {
    double x[], eps;
    x = new double[6];
    sid_data sip = swed.sidd;
    if (!swed.ayana_is_set)
      swe_set_sid_mode(SE_SIDM_FAGAN_BRADLEY, 0, 0);
    /* vernal point (tjd), cartesian */
    x[0] = 1; 
    x[1] = x[2] = 0;
    /* to J2000 */
    if (tjd_et != J2000)
      SwephLib.swi_precess(x, 0, tjd_et, J_TO_J2000);
    /* to t0 */
    SwephLib.swi_precess(x, 0, sip.t0, J2000_TO_J);
    /* to ecliptic */
    eps = SwephLib.swi_epsiln(sip.t0);
    SwephLib.swi_coortrf(x, 0, x, 0, eps);
    /* to polar */
    SwephLib.swi_cartpol(x, 0, x, 0);
    /* subtract initial value of ayanamsa */
    x[0] = x[0] * RADTODEG - sip.ayan_t0;
    /* get ayanamsa */
    return SwephLib.swe_degnorm(-x[0]);
  }
  
  public double swe_get_ayanamsa_ut(double tjd_ut) {
    return swe_get_ayanamsa(tjd_ut + SwephLib.swe_deltat(tjd_ut));
  }
  
  public String swe_get_ayanamsa_name(int isidmode) {
    String ayanamsa_name[] = {
      "Fagan/Bradley", //$NON-NLS-1$
      "Lahiri", //$NON-NLS-1$
      "De Luce", //$NON-NLS-1$
      "Raman", //$NON-NLS-1$
      "Ushashashi", //$NON-NLS-1$
      "Krishnamurti", //$NON-NLS-1$
      "Djwhal Khul", //$NON-NLS-1$
      "Yukteshwar", //$NON-NLS-1$
      "J.N. Bhasin", //$NON-NLS-1$
      "Babylonian/Kugler 1", //$NON-NLS-1$
      "Babylonian/Kugler 2", //$NON-NLS-1$
      "Babylonian/Kugler 3", //$NON-NLS-1$
      "Babylonian/Huber", //$NON-NLS-1$
      "Babylonian/Eta Piscium", //$NON-NLS-1$
      "Babylonian/Aldebaran = 15 Tau", //$NON-NLS-1$
      "Hipparchos", //$NON-NLS-1$
      "Galact. Center = 0 Sag", //$NON-NLS-1$
      "J2000", //$NON-NLS-1$
      "J1900", //$NON-NLS-1$
      "B1950", //$NON-NLS-1$
    };

    if (isidmode < SE_NSIDM_PREDEF)
      return ayanamsa_name[isidmode];
    return null;
  }

  /*
   * stuff exported from swemplan.c and swemmoon.c 
   * and constants used inside these functions.
   ************************************************************/

  static double STR = 4.8481368110953599359e-6;	/* radians per arc second */

  double mods3600(double x) {
    return (x - 1.296e6 * Math.floor ((x)/1.296e6));
  }

  double ss[][] = new double[9][24];
  double cc[][] = new double[9][24];

  /* Moshier ephemeris.
   * computes heliocentric cartesian equatorial coordinates of
   * equinox 2000
   * for earth and a planet
   * tjd		julian day
   * ipli		internal SWEPH planet number
   * xp		array of 6 doubles for planet's position and speed
   * xe		                       earth's
   * serr		error string
   */
  int swi_moshplan(double tjd, int ipli, boolean do_save,
		   double xpret[], double xeret[], String serr) 
  {
    int pnoint2msh[]   = {2, 2, 0, 1, 3, 4, 5, 6, 7, 8, };

    int i;
    boolean do_earth = false;
    double dx[], x2[], xxe[], xxp[];
    dx = new double[3];
    x2 = new double[3];
    xxe = new double[6];
    xxp = new double[6];
    double xp[], xe[];
    double dt; 
    String s;
    int iplm = pnoint2msh[ipli];
    plan_data pdp = swed.pldat[ipli];
    plan_data pedp = swed.pldat[SEI_EARTH];
    double seps2000 = swed.oec2000.seps;
    double ceps2000 = swed.oec2000.ceps;
    if (do_save) {
      xp = pdp.x;
      xe = pedp.x;
    } else {
      xp = xxp;
      xe = xxe;
    }
    if (do_save || ipli == SEI_EARTH || xeret != null) 
      do_earth = true;
    /* tjd beyond ephemeris limits, give some margin for spped at edge */
    if (tjd < MOSHPLEPH_START - 0.3 || tjd > MOSHPLEPH_END + 0.3) {
      if (serr != null && serr != "") { //$NON-NLS-1$
      }
      else {
	serr = ""; //$NON-NLS-1$
      }
      s = "jd " + tjd + " outside Moshier planet range " //$NON-NLS-1$ //$NON-NLS-2$
	+ MOSHPLEPH_START + " .. " + MOSHPLEPH_END + " "; //$NON-NLS-1$ //$NON-NLS-2$
      /*
	sprintf(s, "jd %f outside Moshier planet range %.2f .. %.2f ",
	tjd, MOSHPLEPH_START, MOSHPLEPH_END);
      */
      if (serr.length() + s.length() < AS_MAXCH)
	serr += s;
      return(ERR);
    }  
    /* earth, for geocentric position */
    if (do_earth) {
      if (tjd == pedp.teval
	  && pedp.iephe == SEFLG_MOSEPH) {
	xe = pedp.x;
      } else {
	/* emb */
	swi_moshplan2(tjd, pnoint2msh[SEI_EMB], xe); /* emb hel. ecl. 2000 polar */ 
	SwephLib.swi_polcart(xe, 0, xe, 0);	/* to cartesian */
	/* and equator 2000 */
	SwephLib.swi_coortrf2(xe, 0, xe, 0, -seps2000, ceps2000);
	embofs_mosh(tjd, xe);		  /* emb -> earth */
	if (do_save) {
	  pedp.teval = tjd;		  
	  pedp.xflgs = -1;
	  pedp.iephe = SEFLG_MOSEPH;
	}
	/* one more position for speed. */
	swi_moshplan2(tjd - PLAN_SPEED_INTV, pnoint2msh[SEI_EMB], x2); 
	SwephLib.swi_polcart(x2, 0, x2, 0);
	SwephLib.swi_coortrf2(x2, 0, x2, 0, -seps2000, ceps2000);
	embofs_mosh(tjd - PLAN_SPEED_INTV, x2);/**/
	for (i = 0; i <= 2; i++) 
	  dx[i] = (xe[i] - x2[i]) / PLAN_SPEED_INTV;
	/* store speed */
	for (i = 0; i <= 2; i++) {
	  xe[i+3] = dx[i];
	}
      }
      if (xeret != null)
	for (i = 0; i <= 5; i++) 
	  xeret[i] = xe[i];
    }
    /* earth is the planet wanted */
    if (ipli == SEI_EARTH) {
      xp = xe;
    } else {
      /* other planet */
      /* if planet has already been computed, return */
      if (tjd == pdp.teval && pdp.iephe == SEFLG_MOSEPH) {
	xp = pdp.x;
      } else { 
	swi_moshplan2(tjd, iplm, xp); 
	SwephLib.swi_polcart(xp, 0, xp, 0);
	SwephLib.swi_coortrf2(xp, 0, xp, 0, -seps2000, ceps2000);
	if (do_save) {
	  pdp.teval = tjd;/**/
	  pdp.xflgs = -1;
	  pdp.iephe = SEFLG_MOSEPH;
	}
	/* one more position for speed. 
	 * the following dt gives good speed for light-time correction
	 */
	dt = PLAN_SPEED_INTV;
	swi_moshplan2(tjd - dt, iplm, x2); 
	SwephLib.swi_polcart(x2, 0, x2, 0);
	SwephLib.swi_coortrf2(x2, 0, x2, 0, -seps2000, ceps2000);
	for (i = 0; i <= 2; i++) 
	  dx[i] = (xp[i] - x2[i]) / dt;
	/* store speed */
	for (i = 0; i <= 2; i++) {
	  xp[i+3] = dx[i];
	}
      }
      if (xpret != null)
	for (i = 0; i <= 5; i++)
	  xpret[i] = xp[i];
    }
    return(OK);
  }

  int swi_moshplan2 (double J, int iplm, double pobj[])
  {
    double TIMESCALE = 3652500.0;

    /* From Simon et al (1994)  */
    double freqs[] =
    {
      /* Arc sec per 10000 Julian years.  */
      53810162868.8982,
      21066413643.3548,
      12959774228.3429,
      6890507749.3988,
      1092566037.7991,
      439960985.5372,
      154248119.3933,
      78655032.0744,
      52272245.1795
    };

    double phases[] =
    {
      /* Arc sec.  */
      252.25090552 * 3600.,
      181.97980085 * 3600.,
      100.46645683 * 3600.,
      355.43299958 * 3600.,
      34.35151874 * 3600.,
      50.07744430 * 3600.,
      314.05500511 * 3600.,
      304.34866548 * 3600.,
      860492.1546,
    };

    int i, j, k, m, k1, ip, np, nt;
    byte p[];
    int pIndex;
    double pl[], pb[], pr[];
    int plIndex, pbIndex, prIndex;
    double su, cu, sv, cv, T;
    double t, sl, sb, sr;
	plantbl plan = Swemptab.getPlanTbl(iplm);
    byte max_harmonic[] = plan.max_harmonic;
    double	distance = plan.distance;

    T = (J - J2000) / TIMESCALE;
    /* Calculate sin( i*MM ), etc. for needed multiple angles.  */
    for (i = 0; i < 9; i++)
      {
	if ((j = max_harmonic[i]) > 0)
	  {
	    sr = (mods3600 (freqs[i] * T) + phases[i]) * STR;
	    sscc (i, sr, j);
	  }
      }

    /* Point to start of table of arguments. */
    p = plan.arg_tbl;
    pIndex = 0;
    /* Point to tabulated cosine and sine amplitudes.  */
    pl = plan.lon_tbl;
    plIndex = 0;
    pb = plan.lat_tbl;
    pbIndex = 0;
    pr = plan.rad_tbl;
    prIndex = 0;
    sl = 0.0;
    sb = 0.0;
    sr = 0.0;

    for (;;)
      {
	/* argument of sine and cosine */
	/* Number of periodic arguments. */
	/*np = *p++;*/
	np = p[pIndex++];
	if (np < 0)
	  break;
	if (np == 0)
	  {			/* It is a polynomial term.  */
	    /*nt = *p++;*/
	    nt = p[pIndex++];
	    /* Longitude polynomial. */
	    /*cu = *pl++;*/
	    cu = pl[plIndex++];
	    for (ip = 0; ip < nt; ip++)
	      {
		/*cu = cu * T + *pl++;*/
		cu = cu * T + pl[plIndex++];
	      }
	    sl +=  mods3600 (cu);
	    /* Latitude polynomial. */
	    /*cu = *pb++;*/
	    cu = pb[pbIndex++];
	    for (ip = 0; ip < nt; ip++)
	      {
		/*cu = cu * T + *pb++;*/
		cu = cu * T + pb[pbIndex++];
	      }
	    sb += cu;
	    /* Radius polynomial. */
	    /*cu = *pr++;*/
	    cu = pr[prIndex++];
	    for (ip = 0; ip < nt; ip++)
	      {
		/*cu = cu * T + *pr++;*/
		cu = cu * T + pr[prIndex++];
	      }
	    sr += cu;
	    continue;
	  }
	k1 = 0;
	cv = 0.0;
	sv = 0.0;
	for (ip = 0; ip < np; ip++)
	  {
	    /* What harmonic.  */
	    /*j = *p++;*/
	    j = p[pIndex++];
	    /* Which planet.  */
	    /*m = *p++ - 1;*/
	    m = p[pIndex++] - 1;
	    if (j != 0)
	      {
		k = j;
		if (j < 0)
		  k = -k;
		k -= 1;
		su = ss[m][k];	/* sin(k*angle) */
		if (j < 0)
		  su = -su;
		cu = cc[m][k];
		if (k1 == 0)
		  {		/* set first angle */
		    sv = su;
		    cv = cu;
		    k1 = 1;
		  }
		else
		  {		/* combine angles */
		    t = su * cv + cu * sv;
		    cv = cu * cv - su * sv;
		    sv = t;
		  }
	      }
	  }
	/* Highest power of T.  */
	/*nt = *p++;*/
	nt = p[pIndex++];
	/* Longitude. */
	/*
	cu = *pl++;
	su = *pl++;
	*/
	cu = pl[plIndex++];
	su = pl[plIndex++];
	for (ip = 0; ip < nt; ip++)
	  {
	    /*
	    cu = cu * T + *pl++;
	    su = su * T + *pl++;
	    */
	    cu = cu * T + pl[plIndex++];
	    su = su * T + pl[plIndex++];
	  }
	sl += cu * cv + su * sv;
	/* Latitiude. */
	/*
	cu = *pb++;
	su = *pb++;
	*/
	cu = pb[pbIndex++];
	su = pb[pbIndex++];
	for (ip = 0; ip < nt; ip++)
	  {
	    /*
	    cu = cu * T + *pb++;
	    su = su * T + *pb++;
	    */
	    cu = cu * T + pb[pbIndex++];
	    su = su * T + pb[pbIndex++];
	  }
	sb += cu * cv + su * sv;
	/* Radius. */
	/*
	cu = *pr++;
	su = *pr++;
	*/
	cu = pr[prIndex++];
	su = pr[prIndex++];
	for (ip = 0; ip < nt; ip++)
	  {
	    /*
	    cu = cu * T + *pr++;
	    su = su * T + *pr++;
	    */
	    cu = cu * T + pr[prIndex++];
	    su = su * T + pr[prIndex++];
	  }
	sr += cu * cv + su * sv;
      }
    pobj[0] = STR * sl;
    pobj[1] = STR * sb;
    pobj[2] = STR * distance * sr + distance;
    return OK;
  }

  /* Prepare lookup table of sin and cos ( i*Lj )
   * for required multiple angles
   */
  void sscc (int k, double arg, int n)
  {
    double cu, su, cv, sv, s;
    int i;

    su = Math.sin (arg);
    cu = Math.cos (arg);
    ss[k][0] = su;		/* sin(L) */
    cc[k][0] = cu;		/* cos(L) */
    sv = 2.0 * su * cu;
    cv = cu * cu - su * su;
    ss[k][1] = sv;		/* sin(2L) */
    cc[k][1] = cv;
    for (i = 2; i < n; i++)
      {
	s = su * cv + cu * sv;
	cv = cu * cv - su * sv;
	sv = s;
	ss[k][i] = sv;		/* sin( i+1 L ) */
	cc[k][i] = cv;
      }
  }

  /* Adjust position from Earth-Moon barycenter to Earth
 *
 * J = Julian day number
 * xemb = rectangular equatorial coordinates of Earth
 */
  void embofs_mosh(double tjd, double xemb[]) 
  {
    double T, M, a, L, B, p;
    double smp, cmp, s2mp, c2mp, s2d, c2d, sf, cf;
    double s2f, sx, cx, xyz[];
    xyz = new double[6];
    double seps = swed.oec.seps;
    double ceps = swed.oec.ceps;
    int i;
    /* Short series for position of the Moon
     */
    T = (tjd-J1900)/36525.0;
    /* Mean anomaly of moon (MP) */
    a = SwephLib.swe_degnorm(((1.44e-5*T + 0.009192)*T
			      + 477198.8491)*T + 296.104608);
    a *= DEGTORAD;
    smp = Math.sin(a);
    cmp = Math.cos(a);
    s2mp = 2.0*smp*cmp;		/* sin(2MP) */
    c2mp = cmp*cmp - smp*smp;	/* cos(2MP) */
    /* Mean elongation of moon (D) */
    a = SwephLib.swe_degnorm(((1.9e-6*T - 0.001436)*T
			      + 445267.1142)*T + 350.737486);
    a  = 2.0 * DEGTORAD * a;
    s2d = Math.sin(a);
    c2d = Math.cos(a);
    /* Mean distance of moon from its ascending node (F) */
    a = SwephLib.swe_degnorm((( -3.e-7*T - 0.003211)*T
			      + 483202.0251)*T + 11.250889);
    a  *= DEGTORAD;
    sf = Math.sin(a);
    cf = Math.cos(a);
    s2f = 2.0*sf*cf;	/* sin(2F) */
    sx = s2d*cmp - c2d*smp;	/* sin(2D - MP) */
    cx = c2d*cmp + s2d*smp;	/* cos(2D - MP) */
    /* Mean longitude of moon (LP) */
    L = ((1.9e-6*T - 0.001133)*T + 481267.8831)*T + 270.434164;
    /* Mean anomaly of sun (M) */
    M = SwephLib.swe_degnorm((( -3.3e-6*T - 1.50e-4)*T
			      + 35999.0498)*T + 358.475833);
    /* Ecliptic longitude of the moon */
    L =	L
      + 6.288750*smp
      + 1.274018*sx
      + 0.658309*s2d
      + 0.213616*s2mp
      - 0.185596*Math.sin( DEGTORAD * M )
      - 0.114336*s2f;
    /* Ecliptic latitude of the moon */
    a = smp*cf;
    sx = cmp*sf;
    B =	  5.128189*sf
      + 0.280606*(a+sx)		/* sin(MP+F) */
      + 0.277693*(a-sx)		/* sin(MP-F) */
      + 0.173238*(s2d*cf - c2d*sf);	/* sin(2D-F) */
    B *= DEGTORAD;
    /* Parallax of the moon */
    p =	 0.950724
      +0.051818*cmp
      +0.009531*cx
      +0.007843*c2d
      +0.002824*c2mp;
    p *= DEGTORAD;
    /* Elongation of Moon from Sun
     */
    L = SwephLib.swe_degnorm(L);
    L *= DEGTORAD;
    /* Distance in au */
    a = 4.263523e-5/Math.sin(p);
    /* Convert to rectangular ecliptic coordinates */
    xyz[0] = L;
    xyz[1] = B;
    xyz[2] = a;
    SwephLib.swi_polcart(xyz, 0, xyz, 0);
    /* Convert to equatorial */
    SwephLib.swi_coortrf2(xyz, 0, xyz, 0, -seps, ceps);
    /* Precess to equinox of J2000.0 */
    SwephLib.swi_precess(xyz, 0, tjd, J_TO_J2000);/**/
    /* now emb -> earth */
    for (i = 0; i <= 2; i++)
      xemb[i] -= xyz[i] / (EARTH_MOON_MRAT + 1.0);
  }

  /* NOP stub */
  /* computes a planet from osculating elements *
   * tjd		julian day
   * ipl		body number
   * ipli 	body number in planetary data structure
   * iflag	flags
   */
  int swi_osc_el_plan(double tjd, double xp[], int ipl, int ipli,
		      double xearth[], double xsun[], String serr)
  {
    return ERR;
  }

  String swi_get_fict_name(int ipl)
  {
    String plan_fict_nam[] =
    {"Cupido", "Hades", "Zeus", "Kronos",  //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
     "Apollon", "Admetos", "Vulkanus", "Poseidon", //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
     "Isis-Transpluto", "Nibiru", "Harrington", //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
     "Leverrier", "Adams", //$NON-NLS-1$ //$NON-NLS-2$
     "Lowell", "Pickering",}; //$NON-NLS-1$ //$NON-NLS-2$
    String pname = ""; //$NON-NLS-1$
    if (ipl >= plan_fict_nam.length) {
      pname = "name not found"; //$NON-NLS-1$
    }
    else {
      pname = plan_fict_nam[ipl];
    }
    return pname;
  }

  /*
   * Expansions for the geocentric ecliptic longitude,
   * latitude, and distance of the Moon referred to the mean equinox
   * and ecliptic of date.
   *
   * This version of cmoon.c adjusts the ELP2000-85 analytical Lunar
   * theory of Chapront-Touze and Chapront to fit the Jet Propulsion
   * Laboratory's DE404 long ephemeris on the interval from 3000 B.C.
   * to 3000 A.D.
   *
   * The fit is much better in the remote past and future if
   * secular terms are included in the arguments of the oscillatory
   * perturbations.  Such adjustments cannot easily be incorporated
   * into the 1991 lunar tables.  In this program the traditional
   * literal arguments are used instead, with mean elements adjusted
   * for a best fit to the reference ephemeris.
   *
   * This program omits many oscillatory terms from the analytical
   * theory which, if they were included, would yield a much higher
   * accuracy for modern dates.  Detailed statistics of the precision
   * are given in the table below.  Comparing at 64-day intervals
   * over the period -3000 to +3000, the maximum discrepancies noted
   * were 7" longitude, 5" latitude, and 5 x 10^-8 au radius.
   * The expressions used for precession in this comparision were
   * those of Simon et al (1994).
   * 
   * The adjusted coefficients were found by an unweighted least squares
   * fit to the numerical ephemeris in the mentioned test interval.
   * The approximation error increases rapidly outside this interval.
   * J. Chapront (1994) has described the basic fitting procedure.
   *
   * A major change from DE200 to DE404 is in the coefficient
   * of tidal acceleration of the Moon, which causes the Moon's
   * longitude to depart by about -0.9" per century squared
   * from DE200.  Uncertainty in this quantity continues to
   * be the limiting factor in long term projections of the Moon's
   * ephemeris.
   *
   * Since the Lunar theory is cast in the ecliptic of date, it makes
   * some difference what formula you use for precession.  The adjustment
   * to DE404 was carried out relative to the mean equinox and ecliptic
   * of date as defined in Williams (1994).  An earlier version of this
   * program used the precession given by Simon et al (1994).  The difference
   * between these two precession formulas amounts to about 12" in Lunar
   * longitude at 3000 B.C.
   *
   *    Maximum deviations between DE404 and this program
   *    in a set of 34274 samples spaced 64 days apart
   *
   *   Interval     Longitude  Latitude  Radius
   *   Julian Year   arc sec   arc sec   10^-8 au             
   * -3000 to -2500    5.66      4.66     4.93
   * -2500 to -2000    5.49      3.98     4.56
   * -2000 to -1500    6.98      4.17     4.81
   * -1500 to -1000    5.74      3.53     4.87
   * -1000 to -500     5.95      3.42     4.67
   * -500 to     0     4.94      3.07     4.04
   *    0 to   500     4.42      2.65     4.55
   *  500 to  1000     5.68      3.30     3.99
   * 1000 to  1500     4.32      3.21     3.83
   * 1500 to  2000     2.70      2.69     3.71
   * 2000 to  2500     3.35      2.32     3.85
   * 2500 to  3000     4.62      2.39     4.11
   *
   *
   *
   * References:
   *
   *   James G. Williams, "Contributions to the Earth's obliquity rate,
   *   precession, and nutation,"  Astron. J. 108, 711-724 (1994)
   *
   *   DE403 and DE404 ephemerides by E. M. Standish, X. X. Newhall, and
   *   J. G. Williams are at the JPL computer site navigator.jpl.nasa.gov.
   *
   *   J. L. Simon, P. Bretagnon, J. Chapront, M. Chapront-Touze', G. Francou,
   *   and J. Laskar, "Numerical Expressions for precession formulae and
   *   mean elements for the Moon and the planets," Astronomy and Astrophysics
   *   282, 663-683 (1994)
   *
   *   P. Bretagnon and Francou, G., "Planetary theories in rectangular
   *   and spherical variables. VSOP87 solutions," Astronomy and
   *   Astrophysics 202, 309-315 (1988)
   *
   *   M. Chapront-Touze' and J. Chapront, "ELP2000-85: a semi-analytical
   *   lunar ephemeris adequate for historical times," Astronomy and
   *   Astrophysics 190, 342-352 (1988).
   *
   *   M. Chapront-Touze' and J. Chapront, _Lunar Tables and
   *   Programs from 4000 B.C. to A.D. 8000_, Willmann-Bell (1991)
   *
   *   J. Laskar, "Secular terms of classical planetary theories
   *   using the results of general theory," Astronomy and Astrophysics
   *   157, 59070 (1986)
   *
   *   S. L. Moshier, "Comparison of a 7000-year lunar ephemeris
   *   with analytical theory," Astronomy and Astrophysics 262,
   *   613-616 (1992)
   *
   *   J. Chapront, "Representation of planetary ephemerides by frequency
   *   analysis.  Application to the five outer planets,"  Astronomy and
   *   Astrophysics Suppl. Ser. 109, 181-192 (1994)
   *
   *
   * Entry swi_moshmoon2() returns the geometric position of the Moon
   * relative to the Earth.  Its calling procedure is as follows:
   *
   * double JD;       input Julian Ephemeris Date
   * double pol[3];   output ecliptic polar coordinatees in radians and au
   *                  pol[0] longitude, pol[1] latitude, pol[2] radius
   * swi_moshmoon2( JD, pol );
   *
   * - S. L. Moshier, August, 1991
   * DE200 fit: July, 1992
   * DE404 fit: October, 1995
   *
   * Dieter Koch: adaptation to SWISSEPH, April 1996
   */

  /* Moshier's moom
   * tjd	julian day
   * xpm	array of 6 doubles for moon's position and speed vectors
   * serr	pointer to error string
   */
  int swi_moshmoon(double tjd, boolean do_save, double xpmret[], String serr) 
  {
    int i;
    double a, b, x1[], x2[], t;
    x1 = new double[6];
    x2 = new double[6];
    double xx[], xpm[];
    xx = new double[6];
    plan_data pdp = swed.pldat[SEI_MOON];
    String s;
    if (do_save)
      xpm = pdp.x;
    else
      xpm = xx;
    /* allow 0.2 day tolerance so that true node interval fits in */
    if (tjd < MOSHLUEPH_START - 0.2 || tjd > MOSHLUEPH_END + 0.2) {
      if (serr != null && serr != "") { //$NON-NLS-1$
      }
      else {
	serr = ""; //$NON-NLS-1$
      }
      s = "jd "; //$NON-NLS-1$
      s += tjd;
      s += " outside Moshier's Moon range "; //$NON-NLS-1$
      s+= MOSHLUEPH_START;
      s+= " .. "; //$NON-NLS-1$
      s+= MOSHLUEPH_END;
      if (serr.length() + s.length() < AS_MAXCH)
	serr += s;
      return(ERR);
    }  
    /* if moon has already been computed */
    if (tjd == pdp.teval && pdp.iephe == SEFLG_MOSEPH) {
      if (xpmret != null)
	for (i = 0; i <= 5; i++)
	  xpmret[i] = pdp.x[i];
      return(OK);
    }
    /* else compute moon */
    swi_moshmoon2(tjd, xpm);
    if (do_save) {
      pdp.teval = tjd;
      pdp.xflgs = -1;
      pdp.iephe = SEFLG_MOSEPH;
    }
    /* Moshier moon is referred to ecliptic of date. But we need 
     * equatorial positions for several reasons.
     * e.g. computation of earth from emb and moon
     *                  of heliocentric moon
     * Besides, this helps to keep the program structure simpler 
     */
    ecldat_equ2000(tjd, xpm);
    /* speed */
    /* from 2 other positions. */
    /* one would be good enough for computation of osculating node, 
     * but not for osculating apogee */
    t = tjd + MOON_SPEED_INTV;
    swi_moshmoon2(t, x1);
    ecldat_equ2000(t, x1);
    t = tjd - MOON_SPEED_INTV;
    swi_moshmoon2(t, x2);
    ecldat_equ2000(t, x2);
    for (i = 0; i <= 2; i++) {
      b = (x1[i] - x2[i]) / 2;
      a = (x1[i] + x2[i]) / 2 - xpm[i];
      xpm[i+3] = (2 * a + b) / MOON_SPEED_INTV;
    }
    if (xpmret != null)
      for (i = 0; i <= 5; i++)
	xpmret[i] = xpm[i];
    return(OK);
  }

  /* The following coefficients were calculated by a simultaneous least
   * squares fit between the analytical theory and DE404 on the finite
   * interval from -3000 to +3000.
   * The coefficients were estimated from 34,247 Lunar positions.
   */
  static double z[] = {
    /* The following are scaled in arc seconds, time in Julian centuries.
       They replace the corresponding terms in the mean elements.  */
    -1.312045233711e+01, /* F, t^2 */
    -1.138215912580e-03, /* F, t^3 */
    -9.646018347184e-06, /* F, t^4 */
    3.146734198839e+01, /* l, t^2 */
    4.768357585780e-02, /* l, t^3 */
    -3.421689790404e-04, /* l, t^4 */
    -6.847070905410e+00, /* D, t^2 */
    -5.834100476561e-03, /* D, t^3 */
    -2.905334122698e-04, /* D, t^4 */
    -5.663161722088e+00, /* L, t^2 */
    5.722859298199e-03, /* L, t^3 */
    -8.466472828815e-05, /* L, t^4 */
    /* The following longitude terms are in arc seconds times 10^5.  */
    -8.429817796435e+01, /* t^2 cos(18V - 16E - l) */
    -2.072552484689e+02, /* t^2 sin(18V - 16E - l) */
    7.876842214863e+00, /* t^2 cos(10V - 3E - l) */
    1.836463749022e+00, /* t^2 sin(10V - 3E - l) */
    -1.557471855361e+01, /* t^2 cos(8V - 13E) */
    -2.006969124724e+01, /* t^2 sin(8V - 13E) */
    2.152670284757e+01, /* t^2 cos(4E - 8M + 3J) */
    -6.179946916139e+00, /* t^2 sin(4E - 8M + 3J) */
    -9.070028191196e-01, /* t^2 cos(18V - 16E) */
    -1.270848233038e+01, /* t^2 sin(18V - 16E) */
    -2.145589319058e+00, /* t^2 cos(2J - 5S) */
    1.381936399935e+01, /* t^2 sin(2J - 5S) */
    -1.999840061168e+00, /* t^3 sin(l') */
  };

  double l;		/* Moon's ecliptic longitude */
  double B;		/* Ecliptic latitude */

  double moonpol[];

  double LP;
  double M;
  double MP;
  double D;
  double NF;
  double T;
  double T2;

  double T3;
  double T4;
  double f;
  double g;
  double Ve;
  double Ea;
  double Ma;
  double Ju;
  double Sa;
  double cg;
  double sg;
  double l1;
  double l2;
  double l3;
  double l4;

  /* Calculate geometric coordinates of Moon
   * without light time or nutation correction.
   */
  int swi_moshmoon2(double J, double pol[])
  {
    int i;
    T = (J-J2000)/36525.0;
    T2 = T*T;
    mean_elements();
    moon1();
    moon2();
    moon3();
    moon4();
    for( i=0; i<3; i++ ) 
      pol[i] = moonpol[i];
    return(0);
  }

  void moon1()
  {
    double a;
    /* Mean longitudes of planets (Laskar, Bretagnon) */
    Ve = mods3600( 210664136.4335482 * T + 655127.283046 );
    Ve += ((((((((
		  -9.36e-023 * T
		  - 1.95e-20 ) * T
		 + 6.097e-18 ) * T
		+ 4.43201e-15 ) * T
	       + 2.509418e-13 ) * T
	      - 3.0622898e-10 ) * T
	     - 2.26602516e-9 ) * T
	    - 1.4244812531e-5 ) * T
	   + 0.005871373088 ) * T2;
    Ea = mods3600( 129597742.26669231  * T +  361679.214649 );
    Ea += (((((((( -1.16e-22 * T
		   + 2.976e-19 ) * T
		 + 2.8460e-17 ) * T
		- 1.08402e-14 ) * T
	       - 1.226182e-12 ) * T
	      + 1.7228268e-10 ) * T
	     + 1.515912254e-7 ) * T
	    + 8.863982531e-6 ) * T
	   - 2.0199859001e-2 ) * T2;
    Ma = mods3600(  68905077.59284 * T + 1279559.78866 );
    Ma += (-1.043e-5*T + 9.38012e-3)*T2;
    Ju = mods3600( 10925660.428608 * T +  123665.342120 );
    Ju += (1.543273e-5*T - 3.06037836351e-1)*T2;
    Sa = mods3600( 4399609.65932 * T + 180278.89694 );
    Sa += (( 4.475946e-8*T - 6.874806E-5 ) * T + 7.56161437443E-1)*T2;
    sscc( 0, STR*D, 6 );
    sscc( 1, STR*M,  4 );
    sscc( 2, STR*MP, 4 );
    sscc( 3, STR*NF, 4 );
    moonpol = new double[3];
    moonpol[0] = 0.0;
    moonpol[1] = 0.0;
    moonpol[2] = 0.0;

    final int NLRT2 = 25;
    short LRT2[] = {
      /*
	Multiply by T^2
	Longitude    Radius
	D  l' l  F  .00001" .00001km   */

      0, 1, 0, 0,  487,   -36,
      2,-1,-1, 0, -150,   111,
      2,-1, 0, 0, -120,   149,
      0, 1,-1, 0,  108,    95,
      0, 1, 1, 0,   80,   -77,
      2, 1,-1, 0,   21,   -18,
      2, 1, 0, 0,   20,   -23,
      1, 1, 0, 0,  -13,    12,
      2,-2, 0, 0,  -12,    14,
      2,-1, 1, 0,  -11,     9,
      2,-2,-1, 0,  -11,     7,
      0, 2, 0, 0,   11,     0,
      2,-1,-2, 0,   -6,    -7,
      0, 1,-2, 0,    7,     5,
      0, 1, 2, 0,    6,    -4,
      2, 2,-1, 0,    5,    -3,
      0, 2,-1, 0,    5,     3,
      4,-1,-1, 0,   -3,     3,
      2, 0, 0, 0,    3,    -4,
      4,-1,-2, 0,   -2,     0,
      2, 1,-2, 0,   -2,     0,
      2,-1, 0,-2,   -2,     0,
      2, 1, 1, 0,    2,    -2,
      2, 0,-1, 0,    2,     0,
      0, 2, 1, 0,    2,     0,
    };

    final int NLRT = 38;
    short LRT[] = {
      /*
	Multiply by T
	Longitude    Radius
	D  l' l  F   .1"  .00001" .1km  .00001km */

      0, 1, 0, 0,    16, 7680,    -1,-2302,
      2,-1,-1, 0,    -5,-1642,     3, 8245,
      2,-1, 0, 0,    -4,-1383,     5, 1395,
      0, 1,-1, 0,     3, 7115,     3, 2654,
      0, 1, 1, 0,     2, 7560,    -2,-6396,
      2, 1,-1, 0,     0, 7118,     0,-6068,
      2, 1, 0, 0,     0, 6128,     0,-7754,
      1, 1, 0, 0,     0,-4516,     0, 4194,
      2,-2, 0, 0,     0,-4048,     0, 4970,
      0, 2, 0, 0,     0, 3747,     0, -540,
      2,-2,-1, 0,     0,-3707,     0, 2490,
      2,-1, 1, 0,     0,-3649,     0, 3222,
      0, 1,-2, 0,     0, 2438,     0, 1760,
      2,-1,-2, 0,     0,-2165,     0,-2530,
      0, 1, 2, 0,     0, 1923,     0,-1450,
      0, 2,-1, 0,     0, 1292,     0, 1070,
      2, 2,-1, 0,     0, 1271,     0,-6070,
      4,-1,-1, 0,     0,-1098,     0,  990,
      2, 0, 0, 0,     0, 1073,     0,-1360,
      2, 0,-1, 0,     0,  839,     0, -630,
      2, 1, 1, 0,     0,  734,     0, -660,
      4,-1,-2, 0,     0, -688,     0,  480,
      2, 1,-2, 0,     0, -630,     0,    0,
      0, 2, 1, 0,     0,  587,     0, -590,
      2,-1, 0,-2,     0, -540,     0, -170,
      4,-1, 0, 0,     0, -468,     0,  390,
      2,-2, 1, 0,     0, -378,     0,  330,
      2, 1, 0,-2,     0,  364,     0,    0,
      1, 1, 1, 0,     0, -317,     0,  240,
      2,-1, 2, 0,     0, -295,     0,  210,
      1, 1,-1, 0,     0, -270,     0, -210,
      2,-3, 0, 0,     0, -256,     0,  310,
      2,-3,-1, 0,     0, -187,     0,  110,
      0, 1,-3, 0,     0,  169,     0,  110,
      4, 1,-1, 0,     0,  158,     0, -150,
      4,-2,-1, 0,     0, -155,     0,  140,
      0, 0, 1, 0,     0,  155,     0, -250,
      2,-2,-2, 0,     0, -148,     0, -170,
    };

    final int NBT2 = 12;
    short BT2[] = {
      /*
	Multiply by T^2
	Latitiude
	D  l' l  F  .00001" */

      2,-1, 0,-1,  -22,
      2, 1, 0,-1,    9,
      2,-1, 0, 1,   -6,
      2,-1,-1, 1,   -6,
      2,-1,-1,-1,   -5,
      0, 1, 0, 1,    5,
      0, 1,-1,-1,    5,
      0, 1, 1, 1,    4,
      0, 1, 1,-1,    4,
      0, 1, 0,-1,    4,
      0, 1,-1, 1,    4,
      2,-2, 0,-1,   -2,
    };

    /* terms in T^2, scale 1.0 = 10^-5" */
    chewm( LRT2, NLRT2, 4, 2, moonpol );
    chewm( BT2, NBT2, 4, 4, moonpol );
    f = 18 * Ve - 16 * Ea;
    g = STR*(f - MP );  /* 18V - 16E - l */
    cg = Math.cos(g);
    sg = Math.sin(g);
    l = 6.367278 * cg + 12.747036 * sg;  /* t^0 */
    l1 = 23123.70 * cg - 10570.02 * sg;  /* t^1 */
    l2 = z[12] * cg + z[13] * sg;        /* t^2 */
    moonpol[2] += 5.01 * cg + 2.72 * sg;
    g = STR * (10.*Ve - 3.*Ea - MP);
    cg = Math.cos(g);
    sg = Math.sin(g);
    l += -0.253102 * cg + 0.503359 * sg;
    l1 += 1258.46 * cg + 707.29 * sg;
    l2 += z[14] * cg + z[15] * sg;
    g = STR*(8.*Ve - 13.*Ea);
    cg = Math.cos(g);
    sg = Math.sin(g);
    l += -0.187231 * cg - 0.127481 * sg;
    l1 += -319.87 * cg - 18.34 * sg;
    l2 += z[16] * cg + z[17] * sg;
    a = 4.0*Ea - 8.0*Ma + 3.0*Ju;
    g = STR * a;
    cg = Math.cos(g);
    sg = Math.sin(g);
    l += -0.866287 * cg + 0.248192 * sg;
    l1 += 41.87 * cg + 1053.97 * sg;
    l2 += z[18] * cg + z[19] * sg;
    g = STR*(a - MP);
    cg = Math.cos(g);
    sg = Math.sin(g);
    l += -0.165009 * cg + 0.044176 * sg;
    l1 += 4.67 * cg + 201.55 * sg;
    g = STR*f;  /* 18V - 16E */
    cg = Math.cos(g);
    sg = Math.sin(g);
    l += 0.330401 * cg + 0.661362 * sg;
    l1 += 1202.67 * cg - 555.59 * sg;
    l2 += z[20] * cg + z[21] * sg;
    g = STR*(f - 2.0*MP );  /* 18V - 16E - 2l */
    cg = Math.cos(g);
    sg = Math.sin(g);
    l += 0.352185 * cg + 0.705041 * sg;
    l1 += 1283.59 * cg - 586.43 * sg;
    g = STR * (2.0*Ju - 5.0*Sa);
    cg = Math.cos(g);
    sg = Math.sin(g);
    l += -0.034700 * cg + 0.160041 * sg;
    l2 += z[22] * cg + z[23] * sg;
    g = STR * (LP - NF);
    cg = Math.cos(g);
    sg = Math.sin(g);
    l += 0.000116 * cg + 7.063040 * sg;
    l1 +=  298.8 * sg;
    /* T^3 terms */
    sg = Math.sin( STR * M );
    /* l3 +=  z[24] * sg;			moshier! l3 not initialized! */
    l3 =  z[24] * sg;			
    l4 = 0;					
    g = STR * (2.0*D - M);
    sg = Math.sin(g);
    cg = Math.cos(g);
    moonpol[2] +=  -0.2655 * cg * T;
    g = STR * (M - MP);
    moonpol[2] +=  -0.1568 * Math.cos( g ) * T;
    g = STR * (M + MP);
    moonpol[2] +=  0.1309 * Math.cos( g ) * T;
    g = STR * (2.0*(D + M) - MP);
    sg = Math.sin(g);
    cg = Math.cos(g);
    moonpol[2] +=   0.5568 * cg * T;
    l2 += moonpol[0];
    g = STR*(2.0*D - M - MP);
    moonpol[2] +=  -0.1910 * Math.cos( g ) * T;
    moonpol[1] *= T;
    moonpol[2] *= T;

    final int NBT = 16;
    short BT[] = {
      /*
	Multiply by T
	Latitude
	D  l' l  F  .00001"  */

      2,-1, 0,-1, -7430,
      2, 1, 0,-1,  3043,
      2,-1,-1, 1, -2229,
      2,-1, 0, 1, -1999,
      2,-1,-1,-1, -1869,
      0, 1,-1,-1,  1696,
      0, 1, 0, 1,  1623,
      0, 1,-1, 1,  1418,
      0, 1, 1, 1,  1339,
      0, 1, 1,-1,  1278,
      0, 1, 0,-1,  1217,
      2,-2, 0,-1,  -547,
      2,-1, 1,-1,  -443,
      2, 1,-1, 1,   331,
      2, 1, 0, 1,   317,
      2, 0, 0,-1,   295,
    };

    /* terms in T */
    moonpol[0] = 0.0;
    chewm( BT, NBT, 4, 4, moonpol );
    chewm( LRT, NLRT, 4, 1, moonpol );
    g = STR*(f - MP - NF - 2355767.6); /* 18V - 16E - l - F */
    moonpol[1] +=  -1127. * Math.sin(g);
    g = STR*(f - MP + NF - 235353.6); /* 18V - 16E - l + F */
    moonpol[1] +=  -1123. * Math.sin(g);
    g = STR*(Ea + D + 51987.6);
    moonpol[1] +=  1303. * Math.sin(g);
    g = STR*LP;
    moonpol[1] +=  342. * Math.sin(g);
    g = STR*(2.*Ve - 3.*Ea);
    cg = Math.cos(g);
    sg = Math.sin(g);
    l +=  -0.343550 * cg - 0.000276 * sg;
    l1 +=  105.90 * cg + 336.53 * sg;
    g = STR*(f - 2.*D); /* 18V - 16E - 2D */
    cg = Math.cos(g);
    sg = Math.sin(g);
    l += 0.074668 * cg + 0.149501 * sg;
    l1 += 271.77 * cg - 124.20 * sg;
    g = STR*(f - 2.*D - MP);
    cg = Math.cos(g);
    sg = Math.sin(g);
    l += 0.073444 * cg + 0.147094 * sg;
    l1 += 265.24 * cg - 121.16 * sg;
    g = STR*(f + 2.*D - MP);
    cg = Math.cos(g);
    sg = Math.sin(g);
    l += 0.072844 * cg + 0.145829 * sg;
    l1 += 265.18 * cg - 121.29 * sg;
    g = STR*(f + 2.*(D - MP));
    cg = Math.cos(g);
    sg = Math.sin(g);
    l += 0.070201 * cg + 0.140542 * sg;
    l1 += 255.36 * cg - 116.79 * sg;
    g = STR*(Ea + D - NF);
    cg = Math.cos(g);
    sg = Math.sin(g);
    l += 0.288209 * cg - 0.025901 * sg;
    l1 += -63.51 * cg - 240.14 * sg;
    g = STR*(2.*Ea - 3.*Ju + 2.*D - MP);
    cg = Math.cos(g);
    sg = Math.sin(g);
    l += 0.077865 * cg + 0.438460 * sg;
    l1 += 210.57 * cg + 124.84 * sg;
    g = STR*(Ea - 2.*Ma);
    cg = Math.cos(g);
    sg = Math.sin(g);
    l += -0.216579 * cg + 0.241702 * sg;
    l1 += 197.67 * cg + 125.23 * sg;
    g = STR*(a + MP);
    cg = Math.cos(g);
    sg = Math.sin(g);
    l += -0.165009 * cg + 0.044176 * sg;
    l1 += 4.67 * cg + 201.55 * sg;
    g = STR*(a + 2.*D - MP);
    cg = Math.cos(g);
    sg = Math.sin(g);
    l += -0.133533 * cg + 0.041116 * sg;
    l1 +=  6.95 * cg + 187.07 * sg;
    g = STR*(a - 2.*D + MP);
    cg = Math.cos(g);
    sg = Math.sin(g);
    l += -0.133430 * cg + 0.041079 * sg;
    l1 +=  6.28 * cg + 169.08 * sg;
    g = STR*(3.*Ve - 4.*Ea);
    cg = Math.cos(g);
    sg = Math.sin(g);
    l += -0.175074 * cg + 0.003035 * sg;
    l1 +=  49.17 * cg + 150.57 * sg;
    g = STR*(2.*(Ea + D - MP) - 3.*Ju + 213534.);
    l1 +=  158.4 * Math.sin(g);
    l1 += moonpol[0];
    a = 0.1 * T; /* set amplitude scale of 1.0 = 10^-4 arcsec */
    moonpol[1] *= a;
    moonpol[2] *= a;
  }

  void moon2()
  {
    /* terms in T^0 */
    g = STR*(2*(Ea-Ju+D)-MP+648431.172);
    l += 1.14307 * Math.sin(g);
    g = STR*(Ve-Ea+648035.568);
    l += 0.82155 * Math.sin(g);
    g = STR*(3*(Ve-Ea)+2*D-MP+647933.184);
    l += 0.64371 * Math.sin(g);
    g = STR*(Ea-Ju+4424.04);
    l += 0.63880 * Math.sin(g);
    g = STR*(LP + MP - NF + 4.68);
    l += 0.49331 * Math.sin(g);
    g = STR*(LP - MP - NF + 4.68);
    l += 0.4914 * Math.sin(g);
    g = STR*(LP+NF+2.52);
    l += 0.36061 * Math.sin(g);
    g = STR*(2.*Ve - 2.*Ea + 736.2);
    l += 0.30154 * Math.sin(g);
    g = STR*(2.*Ea - 3.*Ju + 2.*D - 2.*MP + 36138.2);
    l += 0.28282 * Math.sin(g);
    g = STR*(2.*Ea - 2.*Ju + 2.*D - 2.*MP + 311.0);
    l += 0.24516 * Math.sin(g);
    g = STR*(Ea - Ju - 2.*D + MP + 6275.88);
    l += 0.21117 * Math.sin(g);
    g = STR*(2.*(Ea - Ma) - 846.36);
    l += 0.19444 * Math.sin(g);
    g = STR*(2.*(Ea - Ju) + 1569.96);
    l -= 0.18457 * Math.sin(g);
    g = STR*(2.*(Ea - Ju) - MP - 55.8);
    l += 0.18256 * Math.sin(g);
    g = STR*(Ea - Ju - 2.*D + 6490.08);
    l += 0.16499 * Math.sin(g);
    g = STR*(Ea - 2.*Ju - 212378.4);
    l += 0.16427 * Math.sin(g);
    g = STR*(2.*(Ve - Ea - D) + MP + 1122.48);
    l += 0.16088 * Math.sin(g);
    g = STR*(Ve - Ea - MP + 32.04);
    l -= 0.15350 * Math.sin(g);
    g = STR*(Ea - Ju - MP + 4488.88);
    l += 0.14346 * Math.sin(g);
    g = STR*(2.*(Ve - Ea + D) - MP - 8.64);
    l += 0.13594 * Math.sin(g);
    g = STR*(2.*(Ve - Ea - D) + 1319.76);
    l += 0.13432 * Math.sin(g);
    g = STR*(Ve - Ea - 2.*D + MP - 56.16);
    l -= 0.13122 * Math.sin(g);
    g = STR*(Ve - Ea + MP + 54.36);
    l -= 0.12722 * Math.sin(g);
    g = STR*(3.*(Ve - Ea) - MP + 433.8);
    l += 0.12539 * Math.sin(g);
    g = STR*(Ea - Ju + MP + 4002.12);
    l += 0.10994 * Math.sin(g);
    g = STR*(20.*Ve - 21.*Ea - 2.*D + MP - 317511.72);
    l += 0.10652 * Math.sin(g);
    g = STR*(26.*Ve - 29.*Ea - MP + 270002.52);
    l += 0.10490 * Math.sin(g);
    g = STR*(3.*Ve - 4.*Ea + D - MP - 322765.56);
    l += 0.10386 * Math.sin(g);
    g = STR*(LP+648002.556);
    B =  8.04508 * Math.sin(g);
    g = STR*(Ea+D+996048.252);
    B += 1.51021 * Math.sin(g);
    g = STR*(f - MP + NF + 95554.332);
    B += 0.63037 * Math.sin(g);
    g = STR*(f - MP - NF + 95553.792);
    B += 0.63014 * Math.sin(g);
    g = STR*(LP - MP + 2.9);
    B +=  0.45587 * Math.sin(g);
    g = STR*(LP + MP + 2.5);
    B +=  -0.41573 * Math.sin(g);
    g = STR*(LP - 2.0*NF + 3.2);
    B +=  0.32623 * Math.sin(g);
    g = STR*(LP - 2.0*D + 2.5);
    B +=  0.29855 * Math.sin(g);
  }

  void moon3()
  {
    final int NLR = 118;
    short LR[] = {
      /*
	Longitude    Radius
	D  l' l  F    1"  .0001"  1km  .0001km */

      0, 0, 1, 0, 22639, 5858,-20905,-3550,
      2, 0,-1, 0,  4586, 4383, -3699,-1109,
      2, 0, 0, 0,  2369, 9139, -2955,-9676,
      0, 0, 2, 0,   769,  257,  -569,-9251,
      0, 1, 0, 0,  -666,-4171,    48, 8883,
      0, 0, 0, 2,  -411,-5957,    -3,-1483,
      2, 0,-2, 0,   211, 6556,   246, 1585,
      2,-1,-1, 0,   205, 4358,  -152,-1377,
      2, 0, 1, 0,   191, 9562,  -170,-7331,
      2,-1, 0, 0,   164, 7285,  -204,-5860,
      0, 1,-1, 0,  -147,-3213,  -129,-6201,
      1, 0, 0, 0,  -124,-9881,   108, 7427,
      0, 1, 1, 0,  -109,-3803,   104, 7552,
      2, 0, 0,-2,    55, 1771,    10, 3211,
      0, 0, 1, 2,   -45, -996,     0,    0,
      0, 0, 1,-2,    39, 5333,    79, 6606,
      4, 0,-1, 0,    38, 4298,   -34,-7825,
      0, 0, 3, 0,    36, 1238,   -23,-2104,
      4, 0,-2, 0,    30, 7726,   -21,-6363,
      2, 1,-1, 0,   -28,-3971,    24, 2085,
      2, 1, 0, 0,   -24,-3582,    30, 8238,
      1, 0,-1, 0,   -18,-5847,    -8,-3791,
      1, 1, 0, 0,    17, 9545,   -16,-6747,
      2,-1, 1, 0,    14, 5303,   -12,-8314,
      2, 0, 2, 0,    14, 3797,   -10,-4448,
      4, 0, 0, 0,    13, 8991,   -11,-6500,
      2, 0,-3, 0,    13, 1941,    14, 4027,
      0, 1,-2, 0,    -9,-6791,    -7,  -27,
      2, 0,-1, 2,    -9,-3659,     0, 7740,
      2,-1,-2, 0,     8, 6055,    10,  562,
      1, 0, 1, 0,    -8,-4531,     6, 3220,
      2,-2, 0, 0,     8,  502,    -9,-8845,
      0, 1, 2, 0,    -7,-6302,     5, 7509,
      0, 2, 0, 0,    -7,-4475,     1,  657,
      2,-2,-1, 0,     7, 3712,    -4,-9501,
      2, 0, 1,-2,    -6,-3832,     4, 1311,
      2, 0, 0, 2,    -5,-7416,     0,    0,
      4,-1,-1, 0,     4, 3740,    -3,-9580,
      0, 0, 2, 2,    -3,-9976,     0,    0,
      3, 0,-1, 0,    -3,-2097,     3, 2582,
      2, 1, 1, 0,    -2,-9145,     2, 6164,
      4,-1,-2, 0,     2, 7319,    -1,-8970,
      0, 2,-1, 0,    -2,-5679,    -2,-1171,
      2, 2,-1, 0,    -2,-5212,     2, 3536,
      2, 1,-2, 0,     2, 4889,     0, 1437,
      2,-1, 0,-2,     2, 1461,     0, 6571,
      4, 0, 1, 0,     1, 9777,    -1,-4226,
      0, 0, 4, 0,     1, 9337,    -1,-1169,
      4,-1, 0, 0,     1, 8708,    -1,-5714,
      1, 0,-2, 0,    -1,-7530,    -1,-7385,
      2, 1, 0,-2,    -1,-4372,     0,-1357,
      0, 0, 2,-2,    -1,-3726,    -4,-4212,
      1, 1, 1, 0,     1, 2618,     0,-9333,
      3, 0,-2, 0,    -1,-2241,     0, 8624,
      4, 0,-3, 0,     1, 1868,     0,-5142,
      2,-1, 2, 0,     1, 1770,     0,-8488,
      0, 2, 1, 0,    -1,-1617,     1, 1655,
      1, 1,-1, 0,     1,  777,     0, 8512,
      2, 0, 3, 0,     1,  595,     0,-6697,
      2, 0, 1, 2,     0,-9902,     0,    0,
      2, 0,-4, 0,     0, 9483,     0, 7785,
      2,-2, 1, 0,     0, 7517,     0,-6575,
      0, 1,-3, 0,     0,-6694,     0,-4224,
      4, 1,-1, 0,     0,-6352,     0, 5788,
      1, 0, 2, 0,     0,-5840,     0, 3785,
      1, 0, 0,-2,     0,-5833,     0,-7956,
      6, 0,-2, 0,     0, 5716,     0,-4225,
      2, 0,-2,-2,     0,-5606,     0, 4726,
      1,-1, 0, 0,     0,-5569,     0, 4976,
      0, 1, 3, 0,     0,-5459,     0, 3551,
      2, 0,-2, 2,     0,-5357,     0, 7740,
      2, 0,-1,-2,     0, 1790,     8, 7516,
      3, 0, 0, 0,     0, 4042,    -1,-4189,
      2,-1,-3, 0,     0, 4784,     0, 4950,
      2,-1, 3, 0,     0,  932,     0, -585,
      2, 0, 2,-2,     0,-4538,     0, 2840,
      2,-1,-1, 2,     0,-4262,     0,  373,
      0, 0, 0, 4,     0, 4203,     0,    0,
      0, 1, 0, 2,     0, 4134,     0,-1580,
      6, 0,-1, 0,     0, 3945,     0,-2866,
      2,-1, 0, 2,     0,-3821,     0,    0,
      2,-1, 1,-2,     0,-3745,     0, 2094,
      4, 1,-2, 0,     0,-3576,     0, 2370,
      1, 1,-2, 0,     0, 3497,     0, 3323,
      2,-3, 0, 0,     0, 3398,     0,-4107,
      0, 0, 3, 2,     0,-3286,     0,    0,
      4,-2,-1, 0,     0,-3087,     0,-2790,
      0, 1,-1,-2,     0, 3015,     0,    0,
      4, 0,-1,-2,     0, 3009,     0,-3218,
      2,-2,-2, 0,     0, 2942,     0, 3430,
      6, 0,-3, 0,     0, 2925,     0,-1832,
      2, 1, 2, 0,     0,-2902,     0, 2125,
      4, 1, 0, 0,     0,-2891,     0, 2445,
      4,-1, 1, 0,     0, 2825,     0,-2029,
      3, 1,-1, 0,     0, 2737,     0,-2126,
      0, 1, 1, 2,     0, 2634,     0,    0,
      1, 0, 0, 2,     0, 2543,     0,    0,
      3, 0, 0,-2,     0,-2530,     0, 2010,
      2, 2,-2, 0,     0,-2499,     0,-1089,
      2,-3,-1, 0,     0, 2469,     0,-1481,
      3,-1,-1, 0,     0,-2314,     0, 2556,
      4, 0, 2, 0,     0, 2185,     0,-1392,
      4, 0,-1, 2,     0,-2013,     0, 0,
      0, 2,-2, 0,     0,-1931,     0, 0,
      2, 2, 0, 0,     0,-1858,     0, 0,
      2, 1,-3, 0,     0, 1762,     0, 0,
      4, 0,-2, 2,     0,-1698,     0, 0,
      4,-2,-2, 0,     0, 1578,     0,-1083,
      4,-2, 0, 0,     0, 1522,     0,-1281,
      3, 1, 0, 0,     0, 1499,     0,-1077,
      1,-1,-1, 0,     0,-1364,     0, 1141,
      1,-3, 0, 0,     0,-1281,     0, 0,
      6, 0, 0, 0,     0, 1261,     0, -859,
      2, 0, 2, 2,     0,-1239,     0, 0,
      1,-1, 1, 0,     0,-1207,     0, 1100,
      0, 0, 5, 0,     0, 1110,     0, -589,
      0, 3, 0, 0,     0,-1013,     0,  213,
      4,-1,-3, 0,     0,  998,     0, 0,
    };

    final int NMB = 77;
    short MB[] = {
      /*
	Latitude
	D  l' l  F    1"  .0001" */

      0, 0, 0, 1,18461, 2387,
      0, 0, 1, 1, 1010, 1671,
      0, 0, 1,-1,  999, 6936,
      2, 0, 0,-1,  623, 6524,
      2, 0,-1, 1,  199, 4837,
      2, 0,-1,-1,  166, 5741,
      2, 0, 0, 1,  117, 2607,
      0, 0, 2, 1,   61, 9120,
      2, 0, 1,-1,   33, 3572,
      0, 0, 2,-1,   31, 7597,
      2,-1, 0,-1,   29, 5766,
      2, 0,-2,-1,   15, 5663,
      2, 0, 1, 1,   15, 1216,
      2, 1, 0,-1,  -12, -941,
      2,-1,-1, 1,    8, 8681,
      2,-1, 0, 1,    7, 9586,
      2,-1,-1,-1,    7, 4346,
      0, 1,-1,-1,   -6,-7314,
      4, 0,-1,-1,    6, 5796,
      0, 1, 0, 1,   -6,-4601,
      0, 0, 0, 3,   -6,-2965,
      0, 1,-1, 1,   -5,-6324,
      1, 0, 0, 1,   -5,-3684,
      0, 1, 1, 1,   -5,-3113,
      0, 1, 1,-1,   -5, -759,
      0, 1, 0,-1,   -4,-8396,
      1, 0, 0,-1,   -4,-8057,
      0, 0, 3, 1,    3, 9841,
      4, 0, 0,-1,    3, 6745,
      4, 0,-1, 1,    2, 9985,
      0, 0, 1,-3,    2, 7986,
      4, 0,-2, 1,    2, 4139,
      2, 0, 0,-3,    2, 1863,
      2, 0, 2,-1,    2, 1462,
      2,-1, 1,-1,    1, 7660,
      2, 0,-2, 1,   -1,-6244,
      0, 0, 3,-1,    1, 5813,
      2, 0, 2, 1,    1, 5198,
      2, 0,-3,-1,    1, 5156,
      2, 1,-1, 1,   -1,-3178,
      2, 1, 0, 1,   -1,-2643,
      4, 0, 0, 1,    1, 1919,
      2,-1, 1, 1,    1, 1346,
      2,-2, 0,-1,    1,  859,
      0, 0, 1, 3,   -1, -194,
      2, 1, 1,-1,    0,-8227,
      1, 1, 0,-1,    0, 8042,
      1, 1, 0, 1,    0, 8026,
      0, 1,-2,-1,    0,-7932,
      2, 1,-1,-1,    0,-7910,
      1, 0, 1, 1,    0,-6674,
      2,-1,-2,-1,    0, 6502,
      0, 1, 2, 1,    0,-6388,
      4, 0,-2,-1,    0, 6337,
      4,-1,-1,-1,    0, 5958,
      1, 0, 1,-1,    0,-5889,
      4, 0, 1,-1,    0, 4734,
      1, 0,-1,-1,    0,-4299,
      4,-1, 0,-1,    0, 4149,
      2,-2, 0, 1,    0, 3835,
      3, 0, 0,-1,    0,-3518,
      4,-1,-1, 1,    0, 3388,
      2, 0,-1,-3,    0, 3291,
      2,-2,-1, 1,    0, 3147,
      0, 1, 2,-1,    0,-3129,
      3, 0,-1,-1,    0,-3052,
      0, 1,-2, 1,    0,-3013,
      2, 0, 1,-3,    0,-2912,
      2,-2,-1,-1,    0, 2686,
      0, 0, 4, 1,    0, 2633,
      2, 0,-3, 1,    0, 2541,
      2, 0,-1, 3,    0,-2448,
      2, 1, 1, 1,    0,-2370,
      4,-1,-2, 1,    0, 2138,
      4, 0, 1, 1,    0, 2126,
      3, 0,-1, 1,    0,-2059,
      4, 1,-1,-1,    0,-1719,
    };

    /* terms in T^0 */
    moonpol[0] = 0.0;
    chewm( LR, NLR, 4, 1, moonpol );
    chewm( MB, NMB, 4, 3, moonpol );
    l += (((l4 * T + l3) * T + l2) * T + l1) * T * 1.0e-5;
    moonpol[0] = LP + l + 1.0e-4 * moonpol[0];
    moonpol[1] = 1.0e-4 * moonpol[1] + B;
    moonpol[2] = 1.0e-4 * moonpol[2] + 385000.52899; /* kilometers */
  }

  /* Compute final ecliptic polar coordinates
 */
  void moon4()
  {
    moonpol[2] /= AUNIT / 1000;
    moonpol[0] = STR * mods3600( moonpol[0] );
    moonpol[1] = STR * moonpol[1];
  }

  /* mean lunar node
   * J		julian day
   * pol		return array for position and velocity
   *              (polar coordinates of ecliptic of date)
   */
  int swi_mean_node(double J, double pol[], int polIndex, String serr)
  {
    String s;
    T = (J-J2000)/36525.0;
    T2 = T*T;
    T3 = T*T2;
    T4 = T2*T2;
    /* with elements from swi_moshmoon2(), which are fitted to jpl-ephemeris */
    if (J < MOSHNDEPH_START || J > MOSHNDEPH_END) {
      if (serr != null && serr != "") { //$NON-NLS-1$
      }
      else {
	serr = ""; //$NON-NLS-1$
      }
      s = "jd "; //$NON-NLS-1$
      s += J;
      s += " outside mean node range "; //$NON-NLS-1$
      s+= MOSHNDEPH_START;
      s+= " .. "; //$NON-NLS-1$
      s+= MOSHNDEPH_END;
      if (serr.length() + s.length() < AS_MAXCH)
	serr += s;
      return ERR;
    }
    mean_elements();
    /* longitude */
    pol[0+polIndex] = SwephLib.swi_mod2PI((LP - NF) * STR);
    /* latitude */
    pol[1+polIndex] = 0.0;
    /* distance */
    /* or should it be derived from mean * orbital ellipse? */
    pol[2+polIndex] = MOON_MEAN_DIST / AUNIT;
    return OK;
  }

  /* mean lunar apogee ('dark moon', 'lilith')
   * J		julian day
   * pol		return array for position
   *              (polar coordinates of ecliptic of date)
   * serr         error return string
   */
  int swi_mean_apog(double J, double pol[], int polIndex, String serr)
  {
    double node;
    String s;
    T = (J-J2000)/36525.0;
    T2 = T*T;
    T3 = T*T2;
    T4 = T2*T2;
    /* with elements from swi_moshmoon2(), which are fitted to jpl-ephemeris */
    if (J < MOSHNDEPH_START || J > MOSHNDEPH_END) {
      if (serr != null && serr != "") { //$NON-NLS-1$
      }
      else {
	serr = ""; //$NON-NLS-1$
      }
      s = "jd "; //$NON-NLS-1$
      s += J;
      s += " outside mean node apogee range "; //$NON-NLS-1$
      s+= MOSHNDEPH_START;
      s+= " .. "; //$NON-NLS-1$
      s+= MOSHNDEPH_END;
      if (serr.length() + s.length() < AS_MAXCH)
	serr += s;
      return ERR;
    }
    mean_elements();
    pol[0+polIndex] = SwephLib.swi_mod2PI((LP - MP) * STR + Math.PI);
    pol[1+polIndex] = 0;
    /* apogee */
    pol[2+polIndex] = MOON_MEAN_DIST * (1 + MOON_MEAN_ECC) / AUNIT;
    /* Lilith or Dark Moon is either the empty focal point of the mean 
     * lunar ellipse or, for some people, its apogee ("aphelion"). 
     * This is 180 degrees from the perigee.
     *
     * Since the lunar orbit is not in the ecliptic, the apogee must be
     * projected onto the ecliptic.
     * Joelle de Gravelaine has in her book "Lilith der schwarze Mond"
     * (Astrodata, 1990) an ephemeris which gives noon (12.00) positions
     * but does not project them onto the ecliptic.
     * This results in a mistake of several arc minutes.
     *
     * There is also another problem. The other focal point doesn't
     * coincide with the geocenter but with the barycenter of the
     * earth-moon-system. The difference is about 4700 km. If one
     * took this into account, it would result in an oscillation
     * of the Black Moon. If defined as the apogee, this oscillation 
     * would be about +/- 40 arcmin.
     * If defined as the second focus, the effect is very large:
     * +/- 6 deg!
     * We neglect this influence.
     */
    /* apogee is now projected onto ecliptic */
    node = (LP - NF) * STR;
    pol[0+polIndex] = SwephLib.swi_mod2PI(pol[0+polIndex] - node);
    SwephLib.swi_polcart(pol, polIndex, pol, polIndex);
    SwephLib.swi_coortrf(pol, polIndex, pol, polIndex,
			 -MOON_MEAN_INCL * DEGTORAD);
    SwephLib.swi_cartpol(pol, polIndex, pol, polIndex);
    pol[0+polIndex] = SwephLib.swi_mod2PI(pol[0+polIndex] + node);
    return OK;
  }

  /* Program to step through the perturbation table
   */
  void chewm(short pt[], int nlines, int nangles, int typflg, double ans[] )
  {
    int ptIndex = 0;
    int i, j, k, k1, m;
    double cu, su, cv, sv, ff;
    for( i=0; i<nlines; i++ ) {
      k1 = 0;
      sv = 0.0;
      cv = 0.0;
      for( m=0; m<nangles; m++ ) {
	/*j = *pt++; /* multiple angle factor */
	j = pt[ptIndex++]; /* multiple angle factor */
	if( j != 0 ) {
	  k = j;
	  if( j < 0 ) k = -k; /* make angle factor > 0 */
	  /* sin, cos (k*angle) from lookup table */
	  su = ss[m][k-1];
	  cu = cc[m][k-1];
	  if( j < 0 ) su = -su; /* negative angle factor */
	  if( k1 == 0 ) {
	    /* Set sin, cos of first angle. */
	    sv = su;
	    cv = cu;
	    k1 = 1;
	  }
	  else {
	    /* Combine angles by trigonometry. */
	    ff =  su*cv + cu*sv;
	    cv = cu*cv - su*sv;
	    sv = ff;
	  }
	}
      }
      /* Accumulate
       */
      switch( typflg ) {
	/* large longitude and radius */
      case 1:
	/*j = *pt++;*/
	j = pt[ptIndex++];
	/*k = *pt++;*/
	k = pt[ptIndex++];
	ans[0] += (10000.0 * j  + k) * sv;
	/*j = *pt++;*/
	j = pt[ptIndex++];
	/*k = *pt++;*/
	k = pt[ptIndex++];
	if( k != 0 ) ans[2] += (10000.0 * j  + k) * cv;
	break;
	/* longitude and radius */
      case 2:
	/*j = *pt++;*/
	j = pt[ptIndex++];
	/*k = *pt++;*/
	k = pt[ptIndex++];
	ans[0] += j * sv;
	ans[2] += k * cv;
	break;
	/* large latitude */
      case 3:
	/*j = *pt++;*/
	j = pt[ptIndex++];
	/*k = *pt++;*/
	k = pt[ptIndex++];
	ans[1] += ( 10000.0*j + k)*sv;
	break;
	/* latitude */
      case 4:
	/*j = *pt++;*/
	j = pt[ptIndex++];
	ans[1] += j * sv;
	break;
      }
    }
  }

  /* converts from polar coordinates of ecliptic of date
   *          to   cartesian coordinates of equator 2000
   * tjd		date
   * x 		array of position
   */
  void ecldat_equ2000(double tjd, double xpm[]) {
    /* cartesian */
    SwephLib.swi_polcart(xpm, 0, xpm, 0);
    /* equatorial */
    SwephLib.swi_coortrf2(xpm, 0, xpm, 0, -swed.oec.seps, swed.oec.ceps);
    /* j2000 */
    SwephLib.swi_precess(xpm, 0, tjd, J_TO_J2000);/**/
  }

  void mean_elements()
  {
    double fracT = Math.IEEEremainder(T, 1);
    /* Mean anomaly of sun = l' (J. Laskar) */
    /*M =  mods3600(129596581.038354 * T +  1287104.76154);*/
    M =  mods3600(129600000.0 * fracT - 3418.961646 * T +  1287104.76154);
    M += ((((((((
		 1.62e-20 * T
		 - 1.0390e-17 ) * T
		- 3.83508e-15 ) * T
	       + 4.237343e-13 ) * T
	      + 8.8555011e-11 ) * T
	     - 4.77258489e-8 ) * T
	    - 1.1297037031e-5 ) * T
	   + 1.4732069041e-4 ) * T
	  - 0.552891801772 ) * T2;
    /* Mean distance of moon from its ascending node = F */
    NF = mods3600( 1739527263.0983 * T + 335779.55755 );
    /* Mean anomaly of moon = l */
    MP = mods3600( 1717915923.4728 * T +  485868.28096 );
    /* Mean elongation of moon = D */
    D = mods3600( 1602961601.4603 * T + 1072260.73512 );
    /* Mean longitude of moon */
    LP = mods3600( 1732564372.83264 * T +  785939.95571 );                      
    /* Higher degree secular terms found by least squares fit */
    NF += (((((z[5] *T+z[4] )*T + z[3] )*T + z[2] )*T + z[1] )*T + z[0] )*T2;
    MP += (((((z[11]*T+z[10])*T + z[9] )*T + z[8] )*T + z[7] )*T + z[6] )*T2;
    D  += (((((z[17]*T+z[16])*T + z[15])*T + z[14])*T + z[13])*T + z[12])*T2;
    LP += (((((z[23]*T+z[22])*T + z[21])*T + z[20])*T + z[19])*T + z[18])*T2;    
    /* sensitivity of mean elements
     *    delta argument = scale factor times delta amplitude (arcsec)
     * cos l  9.0019 = mean eccentricity
     * cos 2D 43.6
     * cos F  11.2 (latitude term)
     */
  }
  
}

