////////////////////////////////////////////////////////////////////////////////
// sudokuki - C++ graphical sudoku game                                       //
// Copyright (C) 2007-2009 Sylvain Vedrenne                                   //
//                                                                            //
// This program is free software; you can redistribute it and/or              //
// modify it under the terms of the GNU General Public License                //
// as published by the Free Software Foundation; either version 2             //
// of the License, or (at your option) any later version.                     //
//                                                                            //
// This program is distributed in the hope that it will be useful,            //
// but WITHOUT ANY WARRANTY; without even the implied warranty of             //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the              //
// GNU General Public License for more details.                               //
//                                                                            //
// You should have received a copy of the GNU General Public License along    //
// with this program; if not, write to the Free Software Foundation, Inc.,    //
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.              //
////////////////////////////////////////////////////////////////////////////////
#if DISABLE_NLS
#define _(STR) STR
#else
#include <glibmm/i18n.h>
#endif/*DISABLE_NLS*/

#include "sudokukimm_license.h"
#include "sudokukimm_html.h"
#include "sudokukimm.h"
#include "sudokuki_icon.h"
#include "ui_string.h"
#include <iostream>
#include <vector>
#include <stdio.h>

#include <fstream>

#include <gtkmm.h> //TODO: Replace it by several includes when ready

#include "../generator.h"

/* use this to configure local traces */
#define TRACES_SUDOKUKIMM_CC 0

/* use this to deactivate local traces when ALL_TRACES is defined */
#define NO_TRACES_SUDOKUKIMM_CC 1

#if (ALL_TRACES/*defined in sudokukimm.h*/ ||TRACES_SUDOKUKIMM_CC)&&!NO_TRACES_SUDOKUKIMM_CC
#define DBGM_TRACES 1
#else
#define DBGM_TRACES 0
#endif
#include "../dbgm_traces.h"

std::string Sudokukimm::sudokuki_version = std::string(VERSION);
std::string Sudokukimm::symbol = std::string("0123456789");
std::map<char, int> Sudokukimm::values;

void
Sudokukimm::set_window_position( int x, int y )
{
	m_window_x_pos = x;
	m_window_y_pos = y;
}

int
Sudokukimm::get_window_x_pos()
{
	return m_window_x_pos;
}

int
Sudokukimm::get_window_y_pos()
{
	return m_window_y_pos;
}

void
Sudokukimm::reinit_window()
{
	if ( !m_running ) {
		DBGM("not running : DO NOTHING");
		return;
	}

	DBGF("***** ***** ***** m_locale=%s", m_locale.c_str());

	Glib::setenv("LANGUAGE", m_locale, true);
	Glib::setenv("LANG", m_locale, true);
	Glib::setenv("LC_ALL", m_locale, true);
	Glib::setenv("LC_MESSAGES", m_locale, true);
	{
#if !__SUDOKUKI_WIN32__
		extern int  _nl_msg_cat_cntr;
		// GNU/Linux: seems necessary to refresh the locale for gettext
		++_nl_msg_cat_cntr;
#endif/*__SUDOKUKI_WIN32__*/
	}

	DBGF("Glib::getenv(OUTPUT_CHARSET):%s", (Glib::getenv ("OUTPUT_CHARSET")).c_str() );
	DBGF("Glib::getenv(LOCALE_SYSTEM_DEFAULT):%s", (Glib::getenv("LOCALE_SYSTEM_DEFAULT")).c_str() );
	DBGF("Glib::getenv(LANGUAGE):%s", (Glib::getenv("LANGUAGE")).c_str() );
	DBGF("Glib::getenv(LC_ALL):%s", (Glib::getenv("LC_ALL")).c_str() );
	DBGF("Glib::getenv(LC_MESSAGES):%s", (Glib::getenv("LC_MESSAGES")).c_str() );
	DBGF("Glib::getenv(LANG):%s", (Glib::getenv("LANG")).c_str() );

	Gtk::Main::quit();

	DBGF("%s %s (%s)", _("Sudokuki, version "), sudokuki_version.c_str(), new_locale.c_str());
	delete m_psudokukimm_window;
	m_psudokukimm_window = new SudokukimmWindow( this );

	run();
}

Sudokukimm::Sudokukimm(std::string locale) : m_locale(locale)
{
	m_window_x_pos = m_window_y_pos = -1;

	m_running = false;
	m_level = 1;
	m_kanji_mode = 0;
	m_dead_end_state = false;
	m_rating_limit_lower = 0;
	m_rating_limit_higher = 5700;
	m_grid_resolved = false;

	DBGF("##### Sudokukimm ##### locale=%s", m_locale.c_str());

	DBGF("Glib::getenv(OUTPUT_CHARSET):%s", (Glib::getenv("OUTPUT_CHARSET")).c_str() );
	DBGF("Glib::getenv(LOCALE_SYSTEM_DEFAULT):%s", (Glib::getenv("LOCALE_SYSTEM_DEFAULT")).c_str());
	DBGF("Glib::getenv(LANGUAGE):%s", (Glib::getenv("LANGUAGE")).c_str() );
	DBGF("Glib::getenv(LC_ALL):%s", (Glib::getenv("LC_ALL")).c_str() );
	DBGF("Glib::getenv(LC_MESSAGES):%s", (Glib::getenv("LC_MESSAGES")).c_str() );
	DBGF("Glib::getenv(LANG):%s", (Glib::getenv("LANG")).c_str() );

	values['1'] = 1; values['2'] = 2; values['3'] = 3;
	values['4'] = 4; values['5'] = 5; values['6'] = 6;
	values['7'] = 7; values['8'] = 8; values['9'] = 9;
	std::fill( m_generated_grid, m_generated_grid + 82, '\0' );

	cout << _("Sudokuki, version ")	<< sudokuki_version << " (" << m_locale << ")\n";

	generate_grid( m_grid_string );
	DBGF("Generated grid string: %s", m_grid_string.c_str());
	std::copy(m_grid_string.c_str(), m_grid_string.c_str() + 82, m_generated_grid );

	m_psolver = new solver;
	m_psolver->init( m_generated_grid );
	m_pplayer = new player;
	m_pplayer->init( m_generated_grid );

	//	m_psolver->resolve();
	m_psudokukimm_window = new SudokukimmWindow(this);
}

Sudokukimm::~Sudokukimm()
{
	delete m_psolver;
	delete m_pplayer;
}

void
Sudokukimm::remove_spaces( std::string* string_table )
{
	int idx = 0;
	while( (idx=string_table->find_first_of(' ', idx)) >= 0 ) {
		string_table->replace(idx, 1, "");
	}
	idx = 0;
	while( (idx=string_table->find_first_of('\n', idx)) >= 0 ) {
		string_table->replace(idx, 1, "");
	}
}

void
Sudokukimm::generate_grid(std::string &string_table)
{
	// generate a sudoku grid
	static Glib::Rand random( (unsigned int)time(0) );

	char generated_table[128];

	std::vector<int> ratings;

	int rating = -1;
	while( true ) {
		int seed = random.get_int();
		char* p_table = generated_table;
		DBGF("seed: %u\n", seed);
		grid_generate( seed, &p_table, &rating );

		if ( m_rating_limit_lower<rating && rating<=m_rating_limit_higher ) {
			DBGF("rating:%d m_rating_limit_lower:%d m_rating_limit_higher:%d", rating, m_rating_limit_lower, m_rating_limit_higher);
			break;
		}
		DBGF("Generated grid:\n%s\n", generated_table);
	}

	DBGF("Generated grid:\n%s\n", generated_table);
	string_table.assign(generated_table, 128);
	remove_spaces( &string_table );

	return;
}

void
Sudokukimm::run()
{
	DBGF("DETECTED_LANGUAGE:%s", _("DETECTED_LANGUAGE"));

	m_locale = _("DETECTED_LANGUAGE");
	m_running = true;

	Gtk::Main::run(*m_psudokukimm_window);
}

std::string
Sudokukimm::get_locale()
{
	return m_locale;
}

bool
Sudokukimm::get_kanji_mode()
{
	return m_kanji_mode;
}

bool
Sudokukimm::get_dead_end_state()
{
	return m_dead_end_state;
}

int
Sudokukimm::get_level()
{
	return m_level;
}

int
Sudokukimm::solver_get_cell_value(int li, int co)
{
	return m_psolver->get_cell_value(li, co);
}

bool
Sudokukimm::player_is_value_possible(int li, int co, int val)
{
	return m_pplayer->is_value_possible(li, co, val);
}

int
Sudokukimm::player_get_cell_value(int li, int co)
{
	return m_pplayer->get_cell_value(li, co);
}

int
Sudokukimm::player_set_cell_value(int li, int co, char ch)
{
	int val = values[ch];
	int res = m_pplayer->set_cell_value(li, co, val);
	DBGF("(%d,%d)=>res:%d", li, co, res);
	return res;
}

void
Sudokukimm::player_reset()
{
	DBGF("Sudokukimm::player_reset() generated_grid:%s", m_generated_grid);

	m_pplayer->init( m_generated_grid );
}

void
Sudokukimm::player_reset( std::string game )
{
	DBGM("Sudokukimm::reset_player( std::string game )");

	m_pplayer->init( game );
}

void
Sudokukimm::player_fill_additional_moves( std::string game_with_moves )
{
	DBGM("Sudokukimm::reset_player( std::string game )");

	m_pplayer->fill_additional_moves( game_with_moves );
}

void
Sudokukimm::solver_init()
{
	DBGM("Sudokukimm::init_solver()");

	//	m_psolver = new solver();
	m_psolver->init( m_generated_grid );
	m_psolver->resolve();
}

unsigned int
Sudokukimm::solver_get_how_many_numbers()
{
	return m_psolver->get_number_of_filled_cells();
}

unsigned int
Sudokukimm::player_get_how_many_numbers()
{
	return m_pplayer->get_number_of_filled_cells();
}

void
Sudokukimm::reset_generated_grid()
{
	generate_grid( m_grid_string );
	DBGF("Generated grid string: %s", m_grid_string.c_str());
	std::copy(m_grid_string.c_str(), m_grid_string.c_str() + 82, m_generated_grid );
}

void
Sudokukimm::reset_generated_grid( std::string new_grid )
{
	DBGF("Sudokukimm::reset_generated_grid(%s", new_grid.c_str());
	std::copy(new_grid.c_str(), new_grid.c_str() + 82, m_generated_grid );
	DBGM("Sudokukimm::reset_generated_grid( std::string )");
}

void
Sudokukimm::go_back_one_grid()
{
	m_pplayer->go_back_one_grid();
	DBGM("Sudokukimm::go_back_one_grid()");
}

void
Sudokukimm::set_locale( std::string next_locale )
{
	m_locale = next_locale;
}

void
Sudokukimm::set_kanji_mode( int next_kanji_mode )
{
	DBGF("next_kanji_mode:%d", next_kanji_mode);
	m_kanji_mode = next_kanji_mode;
}

void
Sudokukimm::set_dead_end_state( bool dead_end_state )
{
	DBGF("dead end state becomes:%d", dead_end_state);
	m_dead_end_state = dead_end_state;
}

void
Sudokukimm::set_level( int next_level )
{
	DBGF("next_level:%d", next_level);

	m_level = next_level;
	set_rating_limits();
}

void
Sudokukimm::set_rating_limits()
{
	switch (m_level) {
	case 1:
		m_rating_limit_lower = 0;
		m_rating_limit_higher = 5700;
		break;
	case 2:
		m_rating_limit_lower = 5700;
		m_rating_limit_higher = 6700;
		break;
	case 3:
		m_rating_limit_lower = 6700;
		m_rating_limit_higher = 11000;
		break;
	case 4:
		m_rating_limit_lower = 11000;
		m_rating_limit_higher = 15000;
		break;
	case 5:
		m_rating_limit_lower = 15000;
		m_rating_limit_higher = 999999999;
		break;
	default:
		assert(false);
	}
}

void
Sudokukimm::set_grid_resolved( bool next_is_resolved )
{
	m_grid_resolved = next_is_resolved;
}

bool
Sudokukimm::get_grid_resolved()
{
	return m_grid_resolved;
}
