////////////////////////////////////////////////////////////////////////////////
// sudokuki - C++ graphical sudoku game                                       //
// Copyright (C) 2007-2009 Sylvain Vedrenne                                   //
//                                                                            //
// This program is free software; you can redistribute it and/or              //
// modify it under the terms of the GNU General Public License                //
// as published by the Free Software Foundation; either version 2             //
// of the License, or (at your option) any later version.                     //
//                                                                            //
// This program is distributed in the hope that it will be useful,            //
// but WITHOUT ANY WARRANTY; without even the implied warranty of             //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the              //
// GNU General Public License for more details.                               //
//                                                                            //
// You should have received a copy of the GNU General Public License along    //
// with this program; if not, write to the Free Software Foundation, Inc.,    //
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.              //
////////////////////////////////////////////////////////////////////////////////
#include <iostream>
#include "grid.hh"
#define DBGM_TRACES 0
#include "../dbgm_traces.h"
using namespace std;

int
grid::pop_min_poss_cell( short* li, short* co, short* nb )
{
	// Goal: Return a cell (li, co) with a minimum number (nb) of possible values

	short min_nb = 10;
	short min_nb_li = -1;
	short min_nb_co = -1;
	bool found_min = false;

	bool grid_complete = true;

	for ( short l=0; l<nb_lines; l++ ) {
		for ( short c=0; c<nb_cols; c++ ) {

			if ( ! is_cell_filled(l,c) ) {
				grid_complete = false;
			}

			if ( ! is_cell_tried(l,c) ) { // cell not yet tried (during resolve)
				DBGF(":%d,%d", l, c);
				if ( (0 < cell_poss_nb(l,c)) && (cell_poss_nb(l,c) < min_nb) ) {
					min_nb = cell_poss_nb(l,c);
					min_nb_li = l;
					min_nb_co = c;
				}

				if ( 0 == cell_poss_nb(l,c) && is_cell_free(l,c) ) {
					DBGM("found free cell with no more values...DEAD-END!...");
					return 3;
				}

				if ( 1 == cell_poss_nb(l,c) && is_cell_free(l,c) ) {
					found_min = true;
					DBGM("found min ( internal loop) break...");
					break;
				}
			} else {
				// cell already tried, do not pop it
			}
		}
		if ( found_min ) {
			DBGM("found min ( outside loop) break...");
			break;
		}
		DBGM("");
	}
	assert ( 0 < min_nb );
	if ( min_nb < 10  && is_cell_free(min_nb_li, min_nb_co) ) {
		found_min = true;
	}
	if ( grid_complete ) {
		return 2; // grid complete: all cells have been filled
	}
	if ( found_min ) {
		*(nb) = min_nb;
		*(li) = min_nb_li;
		*(co) = min_nb_co;
		assert( *(cell(*li,*co)+11) < 0 );
		return 1; // Ok: cell (li, co) is empty and has the lowest nb of possible values
	}

	return -1; // error
}

int
grid::pop_value( short li, short co, short* val )
{
	DBGM("value_pop()");
	assert( 0 <= li && li <= 8 );
	assert( 0 <= co && co <= 8 );

	short* p_tmp_cell = cell(li,co);

	for ( short v=1; v<=9; v++ ) {
		if ( VAL::SCREENED != *(p_tmp_cell+v) ) { // value not yet screened
				assert( *(p_tmp_cell) < 0 );  // cell not yet filled!
				*(val) = v;
				return 1;
		}
	}

	return -1; // error
}

int
grid::is_cell_free( const short li, const short co )
{
	// Returns: - NO(0) if the cell is not free (either filled, or screened...)
	//          - YES(1) if the cell is free (can be filled with an appropriate value)
	if ( short(VAL::UNUSED) == get_value_in_cell(li, co)
		 || short(VAL::SCREENED) == get_value_in_cell(li, co)
		 || is_cell_filled(li,co) ) {
		return 0; // the cell is not free (cannot be filled)
	} else {
		return 1; // cell free
	}
}

bool
grid::is_cell_filled( const short li, const short co )
{
	// returns the cell's "emptyness" : YES(1) if the cell is filled, NO(0) if the cell is empty

	if ( 1 <= get_value_in_cell(li, co) ) {
		return true; // cell filled
	} else {
		return false; // cell not filled
	}
}

int
grid::set_cell_value( const short li, const short co, const short val )
{
	DBGF("cell_value_set(): li:%d co:%d val:%d", li, co, val);
	assert ( 0!= cell(li,co) );



	DBGF("was: %d -> value_set: %d", get_value_in_cell(li, co), val);
	assert( (0<val && val <10) || short(VAL::EMPTY)==val );


//	assert( is_cell_empty( li, co ) );

	short* p_cell = cell(li,co);

	if ( ! is_cell_free(li,co) ) {
		assert( false );
		return 0; // cell not free, refuse to set a new value!
	}

	DBGF("cell_value_set(%d,%d):%d", li, co, val);

	*( p_cell ) = val;

	// update other cells' fields: poss_values, number_of_poss_values, "emptyness"
	if ( short(VAL::EMPTY) != val ) {
		// all cells from the same column cannot have the same value val
	 	for ( int l=0; l<nb_lines; l++ ) {
			if ( li != l ) {
				screen_cell_value( l, co, val );

				if ( ! is_cell_filled( l, co )
					&& 0 == cell_poss_nb( l, co ) ) {
					// IMPASSE !
					DBGM("");
					return 3; // IMPASSE
				}
			}
		}
		// all cells from the same line cannot have the same value val
		for ( int c=0; c<nb_cols; c++ ) {
			if ( co != c ) {
				screen_cell_value( li, c, val );

				if ( ! is_cell_filled( li, c )
					&& 0 == cell_poss_nb( li, c ) ) {
					// IMPASSE !
					DBGM("");
					return 3; // IMPASSE
				}
			}
		}
		// all cells from the same quarter cannot have the same value val
		short lmin, lmax, cmin, cmax;
		lmin = ( li/3 ) * 3; // (li/3:integer division!)*3 = > 0,3,6
		lmax = lmin + 3;
		cmin = ( co/3 ) * 3; // (co/3:integer division!)*3 = > 0,3,6
		cmax = cmin + 3;
		for ( int l=lmin; l<lmax; l++ ) {
			for ( int c=cmin; c<cmax; c++ ) {
				screen_cell_value( l, c, val );

				if ( ! is_cell_filled( l, c )
					&& 0 == cell_poss_nb( l, c ) ) {
					// IMPASSE !
					DBGM("");
					return 3; // IMPASSE
				}
			}
		}

		*( p_cell + 10 ) = 0; // now, possible values for this cell=0
		*( p_cell + 11 ) = 0; // now, this cell will never be popped
		*( p_cell + val ) = 0; // the value in the cell is no more "possible"

		DBGM("");
		DBGF("cell_value_set(%d,%d):%d", li, co, val);
		DBGM("");
	}

	return 1; // Ok
}

int
grid::is_value_available( const short li, const short co, const short val )
{
	if ( *( cell(li,co) + val ) < 0 ) {
		return 1; // value not screened in this cell = value possible in this cell
	} else {
		return 0;
	}
}

int
grid::is_value_screened( const short li, const short co, const short val )
{
	if ( *( cell(li,co) + val ) < 0 ) {
		return 0; // value not screened in this cell = value possible in this cell
	} else {
		return 1; // value screened = value no more possible in this cell
	}
}

short int
grid::get_value_in_cell(const short li, const short co)
{
	short int val = *( cell(li,co) );
	DBGF("(%d, %d)=>%d", li, co, val);
	assert ((1 <= val && val <= 9) || (VAL::EMPTY==val) || (VAL::UNUSED==val) || (VAL::SCREENED==val));
	return val;
}

int
grid::screen_cell_value( const short li, const short co, const short val )
{
	if ( ! ( 0 < val && val < 10 ) ) {
		return -1; // error
	}
	DBGF("(%d,%d:%d", li, co, val);
	short* p_cell = cell(li, co);

	if ( *( p_cell + val ) < 0 ) { // value possible in this cell
		*( p_cell + val ) = 0; // this value is now screened
		if ( 0 < cell_poss_nb(li, co) ) { // some possible values are left
			*( p_cell + 10 ) = cell_poss_nb(li, co) - 1; // less 1 possible value
		}
	}
	assert ( 0 <= cell_poss_nb(li, co) );
	return 1;
}

int
grid::is_cell_tried( const short li, const short co )
{
	// returns if the cell has been tried during a "resolve" action, yes(1) or no(0)
	if ( 0 < *( cell(li,co) + 11 ) ) {
		return 1; // this cell was tried during a "resolve" operation
	} else
	if ( -1 == *( cell(li,co) + 11 ) ) {
		return 0; // cell not tried
	} else {
		return -1; // error
	}
}

short
grid::cell_poss_nb( const short li, const short co )
{
	// returns the number of possible values for the cell
	return *(cell(li,co) + 10);
}

short*
grid::cell( const short li, const short co )
{
	// returns a pointer pointing to a cell memory (line li, column co)
	return (m_pstart + nb_flags*(nb_cols*li + co) );
}
