	page	,132

;-----------------------------------------------------------------------------
;
;  This file is part of doskey.com.
; 
;  Copyright (C) 2001-2011 Paul Houle (http://paulhoule.com)
; 
;  This program is free software; you can redistribute it and/or modify
;  it under the terms of the GNU General Public License as published by
;  the Free Software Foundation; either version 2 of the License, or
;  (at your option) any later version.
; 
;  This program is distributed in the hope that it will be useful,
;  but WITHOUT ANY WARRANTY; without even the implied warranty of
;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;  GNU General Public License for more details.
; 
;  You should have received a copy of the GNU General Public License
;  along with this program; if not, write to the Free Software
;  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
;
;  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;
;	DOSKEY resident string list support functions.
;
; A string list is a list of strings, formatted exactly as the DOS
; environment (string, 0, string, 0, 0).
;
; One of the strings in the list may be "active."  This is simply a
; pointer to somewhere in list memory, maintained inside the string list
; control structure (STRINGLIST).  It either points to within one of the
; table strings, or to the string list's terminating 0 entry.
;
;-----------------------------------------------------------------------------

	.nolist
	include	dkdefs.inc	;include constant definitions, externdef's
	.list
	option	noljmp		;disallow automatic jmp-lengthening

;-----------------------------------------------------------------------------
;
; StringListGetFreespace:
;
; Attempts to acquire cx contiguous bytes of String List Pool freespace.
; Returns a pointer to the start of the freespace that was obtained, and a
; count of the actual number of freespace bytes made available.
;
; The returned pointer is 1 beyond the string list terminator, so cx bytes
; at the returned address can be used w/o damaging the string list.
;
; This function may cause the oldest list strings to be deleted to acquire
; the freespace.  The entire list will be emptied, if necessary.
;
; Note the number of bytes acquired may not be as many as requested - if the
; entire pool is smaller than the requested size, acquiring the full amount
; is not possible.  Also, the amount acquired may exceed what is requested.
;
; We attempt to keep the Active pointer correct; if we can't, (the active
; entry needs to be deleted), Active is pointed to the string list
; terminator.
;
; GetHistoryFreespace:
;
; Same as above, only gets the freespace from the history pool.  This is
; done to acquire scratch memory for various tasks on occasion, at the cost
; of the loss of some of the oldest history entries.
;
; IN:
;   es:bx= (bx unnecessary for GetHistoryFreespace) pointer to initialized
;	STRINGLIST structure.
;   cx= count of bytes requested.
; OUT:
;   es:di= address of first byte of acquired freespace.
;   dx= actual number of bytes acquired.
;   si,bx,cx= unchanged (bx set to offset slHistory for GetHistoryFreespace).
;   none

GetHistoryFreespace label near

	mov	bx,offset slHistory

StringListGetFreespace proc near
	assume	bx:near ptr STRINGLIST

	push	ds			;prepare to work in control structure
	push	es			;  segment
	pop	ds
	push	si

	mov	si,[bx].Head		;si= first (oldest) string in pool
	mov	di,[bx].Tail		;di= address of terminating 0
	mov	dx,[bx].Next		;dx= current freespace
	sub	dx,di
	dec	dx
	.while	1			;loop to see if deletion is required,
	  .break .if cx <= dx		;break= enough free, stop deletion
	  .break .if si == di		;break= nothing more to delete
	  .repeat			;delete next oldest string
	    inc	dx			;(update freespace count as we go)
	    lodsb
	  .until al == 0
	.endw
	.if	si != [bx].Head		;if deletion(s) to perform,
	  push	cx
	  mov	cx,di			;cx= bytes to move to perform deletion
	  sub	cx,si
	  .if	[bx].Active < si	;if active entry is being deleted,
	    mov [bx].Active,di		;set active ptr to list terminator
	  .endif
	  mov	di,[bx].Head		;pack saved data back over deletions
	  rep movsb
	  sub	si,di			;si= amount of bytes deleted
	  sub	[bx].Active,si		;adjust active pointer
	  mov	[bx].Tail,di		;save list terminator address
	  pop	cx
	.endif

	mov	al,0			;terminate, point di at freespace
	stosb

	pop	si
	pop	ds
	ret
	assume	bx:nothing
StringListGetFreespace endp

;-----------------------------------------------------------------------------
;
; Appends a string to a string list.  If there isn't enough free memory to
; append the string, existing strings are deleted (oldest first).  If all
; old strings need to be deleted, and there still isn't enough memory to
; store the new string, nothing is stored (the string list will be empty).
;
; We attempt to keep the Active pointer correct; if we can't, (the active
; entry needs to be deleted), Active is pointed to the string list
; terminator.
;
; IN:
;   es:bx= pointer to initialized STRINGLIST structure
;   ds:si= pointer to string to append
; OUT:
;   bx= unchanged
;   carry set if insufficient pool space to append string

StringListAppend proc near
	assume	bx:near ptr STRINGLIST

	push	si			;save start of string to add
	xor	cx,cx			;cx= length of string to add
	.repeat
	  lodsb
	  inc	cx
	.until	al == 0
	call	StringListGetFreespace	;attempt to get required freespace
	pop	si			;restore string start address

	cmp	dx,cx			;carry= insufficient space to append
	.if	!carry?			;if room to append new string,
	  dec	di			;point to string list terminator
	  cmp	di,es:[bx].Active	;zero if active ptr is to terminator
	  rep movsb			;append new string to list
	  .if	zero?			;if need to update active ptr,

ExecuteClear label near

	    mov es:[bx].Active,di	;do so
	  .endif
	  mov	es:[bx].Tail,di		;update terminator address
	  xor	ax,ax			;ax= 0/clear carry (append successful)
	  stosb				;re-terminate
	.endif

	ret
	assume	bx:nothing
StringListAppend endp

;-----------------------------------------------------------------------------
;
; Clears (empties) a string list.
;
; IN:
;   es:bx= pointer to initialized STRINGLIST structure (only .Head and .Next
;	need to be valid).
; OUT:
;   bx-dx, si= preserved
;   di= destroyed

StringListClear label near
	assume	bx:near ptr STRINGLIST
	mov	di,es:[bx].Head
	jmp short ExecuteClear
	assume	bx:nothing

;-----------------------------------------------------------------------------
;
; StringListPoolSearch:
;
; Searches string list for string at ds:si.  Input is NOT via a string list
; control structure (STRINGLIST); instead, a pointer to the start of the
; list's string pool (STRINGLIST.Head) is passed in es:di.
;
; Comparison is case-insensitive.  The source (si) string is only matched at
; the start of each list string.  If the list entry is longer, a match is
; still found (zero return status).  If the list string is shorter, a match
; is not found (non-zero return status).
;
; If a match is made, es:di is left at string list character immediately
; following last compared byte (within or at end of matched string).
;
; IN:
;   ds:si= source string (0 terminated)
;   es:di= Start of string list pool.  Note the string list pool is
;	formatted exactly as the DOS environment, so this function can be
;	used to search for environment strings.
; OUT:
;   bx-dx= preserved
;   es:si= if match found, points to start of matched string list entry.
;	If match not found, points to string list's terminating zero.
;   es:di= if match found, points to string list character following last
;	matching byte.
;	If match not found, points to string list's terminating zero.
;
;   zero status if match found (source length <= matched table string length)

StringListPoolSearchProc proc near

	.repeat				;loop to process all table strings,

	  push	si
	  dec	di
	  .repeat			;loop to compare vs. target string,
	    inc	di
	    lodsb			;get source character as lowercase
	    .break .if al == 0		;break= match found
	    call ToLower
	    mov	ah,al
	    mov	al,es:[di]		;get table character as lowercase
	    call ToLower
;110911	  .until ah != al		;loop while strings still match
	    db 38h,0c4h ;cmp ah,al	;JWasm /Zg bug kludge		110911
	  .until !zero?			;done to keep JWasm .com	110911
					;  IDENTICAL to ML .com		110911
					;(remove when JWasm fixed)	110911
	  pop	si
	  .break .if zero?		;break= found matching string, exit

	  pop	ax			;discard saved entry-start address
	  push	cx
	  xor	ax,ax			;scan to next string in table
	  mov	ch,255
	  repne scasb
	  pop	cx
StringListPoolSearch label near
	  push	di			;save start of string list entry
	  mov	al,es:[di]
	  sub	al,1
	.until	carry?
	pop	si			;si= start of last processed entry

	ret
StringListPoolSearchProc endp

;-----------------------------------------------------------------------------
;
; End Module
;
;-----------------------------------------------------------------------------

	ENDSEG
	end
