	page	,132

;-----------------------------------------------------------------------------
;
;  This file is part of doskey.com.
; 
;  Copyright (C) 2001-2011 Paul Houle (http://paulhoule.com)
; 
;  This program is free software; you can redistribute it and/or modify
;  it under the terms of the GNU General Public License as published by
;  the Free Software Foundation; either version 2 of the License, or
;  (at your option) any later version.
; 
;  This program is distributed in the hope that it will be useful,
;  but WITHOUT ANY WARRANTY; without even the implied warranty of
;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;  GNU General Public License for more details.
; 
;  You should have received a copy of the GNU General Public License
;  along with this program; if not, write to the Free Software
;  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
;
;  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;
; DOSKEY display routines.
;
;-----------------------------------------------------------------------------

	.nolist
	include	dkdefs.inc	;include constant definitions, externdef's
	.list
	option	noljmp		;disallow automatic jmp-lengthening

;-----------------------------------------------------------------------------
;
; Master line editing display update routines.
;
; OverwriteFormattedMsg:
;
; An "update" string ([si]) is displayed over an "onscreen" string ([di]).
; If the onscreen string is visually longer (as opposed to physically
; longer, since single bytes can cause multiple chars to be displayed), the
; remainder of the onscreen string is erased.
;
; The cursor is assumed to point to the onscreen string.  After the display,
; the cursor is restored to its starting location.
;
; IN:
;   si= address of update string
;   di= address of onscreen string
;   scInfo= incoming screen information (cursor locations, etc).
; OUT:
;   scInfo= updated screen information
;   ax= destroyed
;   bx/cx/dx/si/di= preserved
;
; OverwriteFormattedMsgAl:
;
; Same as OverwriteFormattedMsg, except that the onscreen string is passed as
; al,[di+1] (al may be the onscreen string 0 terminator).  This facilitates
; overstrike updates - in this case the onscreen string and the update string
; are the same (stored in the same location), except for the 1st byte.

OverwriteFormattedMsg proc near
	mov	al,[di]			;set up as OverwriteFormattedMsgAl

OverwriteFormattedMsgAl label near
	push	cx
	push	dx
	push	si
	push	di

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Save starting and entry cursor location.  Starting, so we can reset
	; the cursor after the display; entry, so we can determine if
	; scrolling has altered the starting location we have saved.

	mov	dx,word ptr scInfo.CurrentCol
	push	dx			 ;save starting cursor location
	push	word ptr scInfo.EntryCol ;save entry cursor location

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; dx holds the screen position of the onscreen string, as it is
	; virtually displayed.  The physical position (reflecting the update
	; string) is maintained in scInfo.  cx will hold the difference in
	; the total display columns (onscreen - update).

	xor	cx,cx

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Display onscreen string virtually, to get it's width.
	; Display update string physically, getting it's width as well, and
	; to put it on the screen.
	;
	; Note: since the onscreen string is already "onscreen," we know
	; it's virtual display will not affect the entry col,row.  The
	; update string may; if it does, that's what is needed.

	jmp short @F			;jmp= start onscreen display
	.while	1
	  .repeat			;start dummy block (to support breaks)
	    .break .if di == 0		;break= onscreen string completed
	    mov	al,[di]			;get next onscreen byte
	    @@:
	    inc	di
	    .if	al == 0			;if hit end of onscreen string,
	      xor di,di			;stop it's display for good
	      .break
	    .endif
	    xchg word ptr scInfo.CurrentCol,dx ;virtual display onscreen byte
	    call DisplayFormattedAlVirtual
	    xchg dx,word ptr scInfo.CurrentCol
	    add  cx,ax			;update display width difference
	  .until 1			;end dummy block
	  lodsb				;get next update string byte
	  .if	al == 0			;if hit end of update string,
	    dec	si			;don't go beyond it's terminator
	    .continue .if di != 0	;continue if more of onscreen string
	    .break			;else stop (both strings completed)
	  .endif
	  call	DisplayFormattedAl	;physically display update byte
	  sub	cx,ax			;update display width difference
	  .if	zero?			;if widths are equal,
	    .break .if si == di		;break= rest of strings same, exit
	  .endif
	.endw

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Display completed of both strings.  If cx > 0, the update string was
	; shorter than the onscreen string, and the remainder of the onscreen
	; string needs to be erased.

	test	cx,cx			;update string same or longer?
	jle	@F			;jmp= yes, nothing to erase
	  .repeat			;erase remainder of onscreen string
	    mov	 al,' '
	    call DisplayFormattedAl
	  .untilcxz
	@@:

	;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	; Display completed.  Restore cursor to starting location and exit.
	; If scrolling occurred, adjust the starting location accordingly.

	pop	cx			;restore entry cursor location
	pop	ax			;restore starting cursor location
	sub	ch,scInfo.EntryRow	;ch= number of rows scrolled (0-n)
	sub	ah,ch			;ax= updated starting cursor location
	mov	word ptr scInfo.CurrentCol,ax ;set as new current location

	pop	di
	pop	si
	pop	dx
	pop	cx

OverwriteFormattedMsg endp

; NOTE: fall through to SyncCursor...

;-----------------------------------------------------------------------------
;
; "Synchronize" onscreen and scInfo current cursor location, by positioning
; the onscreen cursor appropriately.

SyncCursor proc near
	push	bx
	push	dx
	mov	dx,word ptr scInfo.CurrentCol
	mov	ah,0fh			;al= video mode, bh= video page
	int	10h
	mov	ah,2			;set cursor location, dh= row, dl= col
	int	10h
	pop	dx
	pop	bx
	ret
SyncCursor endp

;-----------------------------------------------------------------------------
;
; Lowest-level display routines.  Routines beginning with 'Display' support
; cursor location update, routines beginning with 'Output' do not.
;
;
; DisplayFormattedMsg: displays zero-terminated string at si.
; DisplayFormatted:    displays cx bytes at si (or until a zero encountered).
;
; The specified bytes are formatted and displayed; also, the current scInfo
; cursor location values are updated to reflect movement caused by the
; display.  If the cursor location is not needed, its incoming value can be
; undefined, and the updated output value can be ignored (no ill effects
; will arise from illegal incoming cursor locations).
;
; Note: for the cursor locations to be updated correctly, the scInfo
; structure must be initialized.  This is also true if TAB characters are
; to be expanded to the correct number of spaces.
;
; The source data can be a zero terminated string, or a string specified by
; length.  In either case, if a zero is encountered, display stops (the
; zero is not formatted and displayed).  Therefore, a zero will never be
; considered as part of any source data.
;
; Formatting involves the following:
;
;   CR is converted to CR/LF.
;   TAB is converted to the equivalent number of spaces, using a fixed tab
;	stop of 8.  This requires a valid current cursor location for the
;	display to be accurate (unless a CR occurs prior to the first TAB).
;   Control characters are mapped to ^X, except that LF (Ctl<J>)
;	and CMDSEP (Ctl<T>) are displayed as-is.
;
; There are equivalent functions with a "Virtual" suffix that operate
; identically to the above, except that all display is inhibited.  This
; allows examination of what a display would do to the cursor locations
; before an actual display is performed.
;
; IN:
;   si= address of source (unformatted) bytes
;;;;   cx= length of source (DisplayFormatted/DisplayFormattedVirtual only)
;   scInfo= incoming screen information (cursor locations, etc)
; OUT:
;   si= advanced beyond terminating zero, or beyond length limit if used
;   scInfo= updated screen information (cursor locations, etc).
;   ax= physical columns displayed (only valid if no CR's or LF's in msg)
;   bx/cx/dx/di= preserved
;
;
; DisplayFormattedAl:	Display al formatted, with cursor postion update.
; DisplayCRLF:		Output CRLF with cursor postion update.
; DisplayCRLFVirtual:	Output CRLF virtual with cursor postion update.
; OutputCRLF:		Output CRLF, without cursor position update.
;
; IN:
;   scInfo= incoming screen information (cursor locations, etc).
; OUT:
;   scInfo= updated screen information (cursor locations, etc).
;   ax= physical columns displayed (only valid if no CR's or LF's in msg)
;   other registers preserved
;
;
; DisplayUnformattedAl: Display al - no formatting, no screen info update
; DisplayFormattedAl: Display al formatted, with screen info update
;
; IN:
;   al= character to display unformatted (must not be zero)
; OUT:
;   ax= physical columns displayed (only valid if no CR's or LF's in msg)
;   other registers preserved

DisplayFormattedAlVirtual proc near
	stc
	skip
DisplayFormattedAl label near
	  clc
	endskip
	push	cx
	push	dx
	push	di
	mov	cx,0			;init count of physical output cols
	mov	di,offset WriteAl	;assume no virtualization
	.if	carry?			;if assumption wrong,
	  mov	di,offset WriteAlVirtual ;else use virtual display
	.endif
	call	DisplayFormattedByte	;display the formatted byte
	xchg	ax,cx			;ax= count of formatted columns
	pop	di
	pop	dx
	pop	cx
	ret
DisplayFormattedAlVirtual endp

DisplayFormattedMsg proc near		;use zero terminated source
	mov	al,0feh

	skip
;;;DisplayFormatted label near		;use string length in cx
	  mov	al,0
	endskip

	skip
DisplayFormattedMsgVirtual label near
	  mov	al,0ffh			;use zero terminated source
	endskip

	skip
;;;DisplayFormattedVirtual label near	;use string length in cx
	  mov	al,1
	endskip

	push	cx
	push	dx
	push	di

	xor	cx,cx			;init count of physical output cols

	mov	di,offset WriteAlVirtual ;assume virtual output
	sar	al,1			;test assumption
	.if	!carry?			;if assumption wrong,
	  mov	di,offset WriteAl	;cancel virtualization
	.endif

	jmp short @F			;jmp= start display
	.repeat
	  call	DisplayFormattedByte	;display formatted byte
	  @@:
	  lodsb
	.until	al == 0

	pop	di
PopDxCxAndReturn label near
	xchg	ax,cx			;ax= count of physical output cols
	pop	dx
	pop	cx
	ret
DisplayFormattedMsg endp



DisplayFormattedByte proc near
	.if	al < 20h		;if special character,
	  .repeat			;start dummy block (to support breaks)
	    .break .if al == CMDSEP	;don't map separator (Ctl<T>)
	    .break .if al == CR		;break= handle CR in di routine
	    .break .if al == LF		;break= handle LF in di routine
	    .if al == TAB		;if tab (special case),
	      .repeat			;loop to display spaces,
		mov  al,' '		;display a space
		inc  cx			;bump count of physical output cols
		call di
	      .until !(scInfo.CurrentCol & 7) ;loop until a tab stop hit
	      ret			;done with tab display
	    .endif
	    push ax			;display ^X
	    mov  al,'^'
	    inc  cx			;bump count of physical output cols
	    call di
	    pop  ax
	    add  al,'@'
	  .until 1			;end dummy block
	.endif
	inc	cx			;bump count of physical output cols
	jmp	di			;execute display function
DisplayFormattedByte endp



DisplayCRLF label near
	call	OutputCRLF		;output CRLF
DisplayCRLFVirtual label near
	mov	ax,word ptr scInfo.CurrentCol ;al= current col, ah= row
	jmp short CursorWrap		;reset to column 0, bump row

OutputCRLF label near
	mov	al,CR
	call	OutputAl
	mov	al,LF

OutputAl label near
	push	cx
	push	dx
	xor	cx,cx			;flag unformatted display

WriteAl proc	near
	mov	dl,al
	mov	ah,2
	int	21h
	jcxz	PopDxCxAndReturn	;jmp= exit if Output call

	; al can't be any control char other than CR or LF at this point

WriteAlVirtual label near
	cmp	al,CR			;check if char CR or LF
	mov	ax,word ptr scInfo.CurrentCol ;al= current col, ah= row
	jbe	WriteCRLF		;jmp= CR/LF
	  inc	ax			;advance column
	  .if	al == scInfo.TotalCols	;if hit end-of-row,
CursorWrap  label near
	    mov	al,0			;cause wrap to start of next row
	    @@:
	    inc	ah			;advance row count
	    .if ah == scInfo.TotalRows	;if screen scrolled up,
	      dec ah			;peg current to bottom of screen
	      .if scInfo.EntryRow != 0	;if entry not pegged at top,
		dec scInfo.EntryRow	;reduce entry cursor location
	      .endif
	    .endif
	  .endif
	  mov	word ptr scInfo.CurrentCol,ax ;store new cursor address
	  ret
	WriteCRLF:
	jne	@B			;jmp= LF, just advance row number
	mov	scInfo.CurrentCol,0	;else CR, reset column
	mov	al,LF			;and follow with LF
	jmp	di
WriteAl endp

;-----------------------------------------------------------------------------
;
; End Module
;
;-----------------------------------------------------------------------------

	ENDSEG
	end
