/*
 * Copyright (c) 2005, 2013, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */


package java.net;


/**
 * A CookieStore object represents a storage for cookie. Can store and retrieve
 * cookies.
 *
 * <p>{@link java.net.CookieManager CookieManager} will call {@code CookieStore.add} to save cookies
 * for every incoming HTTP response, and call {@code CookieStore.get} to
 * retrieve cookie for every outgoing HTTP request. A CookieStore
 * is responsible for removing HttpCookie instances which have expired.
 *
 * @author Edward Wang
 * @since 1.6
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public interface CookieStore {

/**
 * Adds one HTTP cookie to the store. This is called for every
 * incoming HTTP response.
 *
 * <p>A cookie to store may or may not be associated with an URI. If it
 * is not associated with an URI, the cookie's domain and path attribute
 * will indicate where it comes from. If it is associated with an URI and
 * its domain and path attribute are not specified, given URI will indicate
 * where this cookie comes from.
 *
 * <p>If a cookie corresponding to the given URI already exists,
 * then it is replaced with the new one.
 *
 * @param uri       the uri this cookie associated with.
 *                  if {@code null}, this cookie will not be associated
 *                  with an URI
 * @param cookie    the cookie to store
 *
 * @throws java.lang.NullPointerException if {@code cookie} is {@code null}
 *
 * @see #get
 *
 */

public void add(java.net.URI uri, java.net.HttpCookie cookie);

/**
 * Retrieve cookies associated with given URI, or whose domain matches the
 * given URI. Only cookies that have not expired are returned.
 * This is called for every outgoing HTTP request.
 *
 * @return          an immutable list of HttpCookie,
 *                  return empty list if no cookies match the given URI
 *
 * @param uri       the uri associated with the cookies to be returned
 *
 * @throws java.lang.NullPointerException if {@code uri} is {@code null}
 *
 * @see #add
 *
 */

public java.util.List<java.net.HttpCookie> get(java.net.URI uri);

/**
 * Get all not-expired cookies in cookie store.
 *
 * @return          an immutable list of http cookies;
 *                  return empty list if there's no http cookie in store
 */

public java.util.List<java.net.HttpCookie> getCookies();

/**
 * Get all URIs which identify the cookies in this cookie store.
 *
 * @return          an immutable list of URIs;
 *                  return empty list if no cookie in this cookie store
 *                  is associated with an URI
 */

public java.util.List<java.net.URI> getURIs();

/**
 * Remove a cookie from store.
 *
 * @param uri       the uri this cookie associated with.
 *                  if {@code null}, the cookie to be removed is not associated
 *                  with an URI when added; if not {@code null}, the cookie
 *                  to be removed is associated with the given URI when added.
 * @param cookie    the cookie to remove
 *
 * @return          {@code true} if this store contained the specified cookie
 *
 * @throws java.lang.NullPointerException if {@code cookie} is {@code null}
 */

public boolean remove(java.net.URI uri, java.net.HttpCookie cookie);

/**
 * Remove all cookies in this cookie store.
 *
 * @return          {@code true} if this store changed as a result of the call
 */

public boolean removeAll();
}

