/*
 * Copyright (C) 2019 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.net;

import java.util.concurrent.Executor;
import android.os.PersistableBundle;
import android.os.Parcelable;

/**
 * Class that provides utilities for collecting network connectivity diagnostics information.
 * Connectivity information is made available through triggerable diagnostics tools and by listening
 * to System validations. Some diagnostics information may be permissions-restricted.
 *
 * <p>ConnectivityDiagnosticsManager is intended for use by applications offering network
 * connectivity on a user device. These tools will provide several mechanisms for these applications
 * to be alerted to network conditions as well as diagnose potential network issues themselves.
 *
 * <p>The primary responsibilities of this class are to:
 *
 * <ul>
 *   <li>Allow permissioned applications to register and unregister callbacks for network event
 *       notifications
 *   <li>Invoke callbacks for network event notifications, including:
 *       <ul>
 *         <li>Network validations
 *         <li>Data stalls
 *         <li>Connectivity reports from applications
 *       </ul>
 * </ul>
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public class ConnectivityDiagnosticsManager {

ConnectivityDiagnosticsManager() { throw new RuntimeException("Stub!"); }

/**
 * Registers a ConnectivityDiagnosticsCallback with the System.
 *
 * <p>Only apps that offer network connectivity to the user should be registering callbacks.
 * These are the only apps whose callbacks will be invoked by the system. Apps considered to
 * meet these conditions include:
 *
 * <ul>
 *   <li>Carrier apps with active subscriptions
 *   <li>Active VPNs
 *   <li>WiFi Suggesters
 * </ul>
 *
 * <p>Callbacks registered by apps not meeting the above criteria will not be invoked.
 *
 * <p>If a registering app loses its relevant permissions, any callbacks it registered will
 * silently stop receiving callbacks. Note that registering apps must also have location
 * permissions to receive callbacks as some Networks may be location-bound (such as WiFi
 * networks).
 *
 * <p>Each register() call <b>MUST</b> use a ConnectivityDiagnosticsCallback instance that is
 * not currently registered. If a ConnectivityDiagnosticsCallback instance is registered with
 * multiple NetworkRequests, an IllegalArgumentException will be thrown.
 *
 * <p>To avoid performance issues due to apps leaking callbacks, the system will limit the
 * number of outstanding requests to 100 per app (identified by their UID), shared with
 * callbacks in {@link android.net.ConnectivityManager ConnectivityManager}. Registering a callback with this method will count
 * toward this limit. If this limit is exceeded, an exception will be thrown. To avoid hitting
 * this issue and to conserve resources, make sure to unregister the callbacks with
 * {@link #unregisterConnectivityDiagnosticsCallback}.
 *
 * @param request The NetworkRequest that will be used to match with Networks for which
 *     callbacks will be fired
 * This value cannot be {@code null}.
 * @param e The Executor to be used for running the callback method invocations
 * This value cannot be {@code null}.
 * @param callback The ConnectivityDiagnosticsCallback that the caller wants registered with the
 *     System
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if the same callback instance is registered with multiple
 *     NetworkRequests
 * @throws java.lang.RuntimeException if the app already has too many callbacks registered.
 */

public void registerConnectivityDiagnosticsCallback(@android.annotation.NonNull android.net.NetworkRequest request, @android.annotation.NonNull java.util.concurrent.Executor e, @android.annotation.NonNull android.net.ConnectivityDiagnosticsManager.ConnectivityDiagnosticsCallback callback) { throw new RuntimeException("Stub!"); }

/**
 * Unregisters a ConnectivityDiagnosticsCallback with the System.
 *
 * <p>If the given callback is not currently registered with the System, this operation will be
 * a no-op.
 *
 * @param callback The ConnectivityDiagnosticsCallback to be unregistered from the System.

 * This value cannot be {@code null}.
 */

public void unregisterConnectivityDiagnosticsCallback(@android.annotation.NonNull android.net.ConnectivityDiagnosticsManager.ConnectivityDiagnosticsCallback callback) { throw new RuntimeException("Stub!"); }
/**
 * Abstract base class for Connectivity Diagnostics callbacks. Used for notifications about
 * network connectivity events. Must be extended by applications wanting notifications.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public abstract static class ConnectivityDiagnosticsCallback {

public ConnectivityDiagnosticsCallback() { throw new RuntimeException("Stub!"); }

/**
 * Called when the platform completes a data connectivity check. This will also be invoked
 * immediately upon registration for each network matching the request with the latest
 * report, if a report has already been generated for that network.
 *
 * <p>The Network specified in the ConnectivityReport may not be active any more when this
 * method is invoked.
 *
 * @param report The ConnectivityReport containing information about a connectivity check

 * This value cannot be {@code null}.
 */

public void onConnectivityReportAvailable(@android.annotation.NonNull android.net.ConnectivityDiagnosticsManager.ConnectivityReport report) { throw new RuntimeException("Stub!"); }

/**
 * Called when the platform suspects a data stall on some Network.
 *
 * <p>The Network specified in the DataStallReport may not be active any more when this
 * method is invoked.
 *
 * @param report The DataStallReport containing information about the suspected data stall

 * This value cannot be {@code null}.
 */

public void onDataStallSuspected(@android.annotation.NonNull android.net.ConnectivityDiagnosticsManager.DataStallReport report) { throw new RuntimeException("Stub!"); }

/**
 * Called when any app reports connectivity to the System.
 *
 * @param network The Network for which connectivity has been reported
 * This value cannot be {@code null}.
 * @param hasConnectivity The connectivity reported to the System
 */

public void onNetworkConnectivityReported(@android.annotation.NonNull android.net.Network network, boolean hasConnectivity) { throw new RuntimeException("Stub!"); }
}

/** Class that includes connectivity information for a specific Network at a specific time. */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static final class ConnectivityReport implements android.os.Parcelable {

/**
 * Constructor for ConnectivityReport.
 *
 * <p>Apps should obtain instances through {@link android.net.ConnectivityDiagnosticsManager.ConnectivityDiagnosticsCallback#onConnectivityReportAvailable  } instead of instantiating
 * their own instances (unless for testing purposes).
 *
 * @param network The Network for which this ConnectivityReport applies
 * This value cannot be {@code null}.
 * @param reportTimestamp The timestamp for the report
 * @param linkProperties The LinkProperties available on network at reportTimestamp
 * This value cannot be {@code null}.
 * @param networkCapabilities The NetworkCapabilities available on network at
 *     reportTimestamp
 * This value cannot be {@code null}.
 * @param additionalInfo A PersistableBundle that may contain additional info about the
 *     report

 * This value cannot be {@code null}.
 */

public ConnectivityReport(@android.annotation.NonNull android.net.Network network, long reportTimestamp, @android.annotation.NonNull android.net.LinkProperties linkProperties, @android.annotation.NonNull android.net.NetworkCapabilities networkCapabilities, @android.annotation.NonNull android.os.PersistableBundle additionalInfo) { throw new RuntimeException("Stub!"); }

/**
 * Returns the Network for this ConnectivityReport.
 *
 * @return The Network for which this ConnectivityReport applied

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.Network getNetwork() { throw new RuntimeException("Stub!"); }

/**
 * Returns the epoch timestamp (milliseconds) for when this report was taken.
 *
 * @return The timestamp for the report. Taken from {@link java.lang.System#currentTimeMillis System#currentTimeMillis}.
 */

public long getReportTimestamp() { throw new RuntimeException("Stub!"); }

/**
 * Returns the LinkProperties available when this report was taken.
 *
 * @return LinkProperties available on the Network at the reported timestamp

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.LinkProperties getLinkProperties() { throw new RuntimeException("Stub!"); }

/**
 * Returns the NetworkCapabilities when this report was taken.
 *
 * @return NetworkCapabilities available on the Network at the reported timestamp

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.NetworkCapabilities getNetworkCapabilities() { throw new RuntimeException("Stub!"); }

/**
 * Returns a PersistableBundle with additional info for this report.
 *
 * @return PersistableBundle that may contain additional info about the report

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.os.PersistableBundle getAdditionalInfo() { throw new RuntimeException("Stub!"); }

/**
 * {@inheritDoc}

 * @param o This value may be {@code null}.
 */

public boolean equals(@android.annotation.Nullable java.lang.Object o) { throw new RuntimeException("Stub!"); }

public int hashCode() { throw new RuntimeException("Stub!"); }

/** {@inheritDoc} */

public int describeContents() { throw new RuntimeException("Stub!"); }

/**
 * {@inheritDoc}
 * @param dest This value cannot be {@code null}.
 */

public void writeToParcel(@android.annotation.NonNull android.os.Parcel dest, int flags) { throw new RuntimeException("Stub!"); }

/** Implement the Parcelable interface */

@android.annotation.NonNull public static final android.os.Parcelable.Creator<android.net.ConnectivityDiagnosticsManager.ConnectivityReport> CREATOR;
static { CREATOR = null; }

/**
 * A bitmask of network validation probes that were attempted.
 *
 * <p>These probes may have failed or may be incomplete at the time of this report.
 *
 * <p>The possible bits values reported by this key are:
 * {@link #NETWORK_PROBE_DNS},
 * {@link #NETWORK_PROBE_HTTP},
 * {@link #NETWORK_PROBE_HTTPS},
 * {@link #NETWORK_PROBE_FALLBACK},
 * {@link #NETWORK_PROBE_PRIVATE_DNS}.

 * <br>
 * Value is {@link android.net.ConnectivityDiagnosticsManager.ConnectivityReport#NETWORK_PROBE_DNS}, {@link android.net.ConnectivityDiagnosticsManager.ConnectivityReport#NETWORK_PROBE_HTTP}, {@link android.net.ConnectivityDiagnosticsManager.ConnectivityReport#NETWORK_PROBE_HTTPS}, {@link android.net.ConnectivityDiagnosticsManager.ConnectivityReport#NETWORK_PROBE_FALLBACK}, or {@link android.net.ConnectivityDiagnosticsManager.ConnectivityReport#NETWORK_PROBE_PRIVATE_DNS}
 */

public static final java.lang.String KEY_NETWORK_PROBES_ATTEMPTED_BITMASK = "networkProbesAttempted";

/**
 * A bitmask of network validation probes that succeeded.
 *
 * <p>The possible bits values reported by this key are:
 * {@link #NETWORK_PROBE_DNS},
 * {@link #NETWORK_PROBE_HTTP},
 * {@link #NETWORK_PROBE_HTTPS},
 * {@link #NETWORK_PROBE_FALLBACK},
 * {@link #NETWORK_PROBE_PRIVATE_DNS}.

 * <br>
 * Value is {@link android.net.ConnectivityDiagnosticsManager.ConnectivityReport#NETWORK_PROBE_DNS}, {@link android.net.ConnectivityDiagnosticsManager.ConnectivityReport#NETWORK_PROBE_HTTP}, {@link android.net.ConnectivityDiagnosticsManager.ConnectivityReport#NETWORK_PROBE_HTTPS}, {@link android.net.ConnectivityDiagnosticsManager.ConnectivityReport#NETWORK_PROBE_FALLBACK}, or {@link android.net.ConnectivityDiagnosticsManager.ConnectivityReport#NETWORK_PROBE_PRIVATE_DNS}
 */

public static final java.lang.String KEY_NETWORK_PROBES_SUCCEEDED_BITMASK = "networkProbesSucceeded";

/**
 * The overall validation result for the Network being reported on.
 *
 * <p>The possible values for this key are:
 * {@link #NETWORK_VALIDATION_RESULT_INVALID},
 * {@link #NETWORK_VALIDATION_RESULT_VALID},
 * {@link #NETWORK_VALIDATION_RESULT_PARTIALLY_VALID},
 * {@link #NETWORK_VALIDATION_RESULT_SKIPPED}.
 *
 * <br>
 * Value is {@link android.net.ConnectivityDiagnosticsManager.ConnectivityReport#NETWORK_VALIDATION_RESULT_INVALID}, {@link android.net.ConnectivityDiagnosticsManager.ConnectivityReport#NETWORK_VALIDATION_RESULT_VALID}, {@link android.net.ConnectivityDiagnosticsManager.ConnectivityReport#NETWORK_VALIDATION_RESULT_PARTIALLY_VALID}, or {@link android.net.ConnectivityDiagnosticsManager.ConnectivityReport#NETWORK_VALIDATION_RESULT_SKIPPED}
 * @see android.net.NetworkCapabilities#NET_CAPABILITY_VALIDATED
 */

public static final java.lang.String KEY_NETWORK_VALIDATION_RESULT = "networkValidationResult";

/** DNS probe. */

public static final int NETWORK_PROBE_DNS = 4; // 0x4

/** Captive portal fallback probe. */

public static final int NETWORK_PROBE_FALLBACK = 32; // 0x20

/** HTTP probe. */

public static final int NETWORK_PROBE_HTTP = 8; // 0x8

/** HTTPS probe. */

public static final int NETWORK_PROBE_HTTPS = 16; // 0x10

/** Private DNS (DNS over TLS) probd. */

public static final int NETWORK_PROBE_PRIVATE_DNS = 64; // 0x40

/**
 * The overall status of the network is that it is invalid; it neither provides
 * connectivity nor has been exempted from validation.
 */

public static final int NETWORK_VALIDATION_RESULT_INVALID = 0; // 0x0

/**
 * The overall status of the network is that it provides partial connectivity; some
 * probed services succeeded but others failed.
 */

public static final int NETWORK_VALIDATION_RESULT_PARTIALLY_VALID = 2; // 0x2

/**
 * Due to the properties of the network, validation was not performed.
 */

public static final int NETWORK_VALIDATION_RESULT_SKIPPED = 3; // 0x3

/**
 * The overall status of the network is that it is valid, this may be because it provides
 * full Internet access (all probes succeeded), or because other properties of the network
 * caused probes not to be run.
 */

public static final int NETWORK_VALIDATION_RESULT_VALID = 1; // 0x1
}

/** Class that includes information for a suspected data stall on a specific Network */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static final class DataStallReport implements android.os.Parcelable {

/**
 * Constructor for DataStallReport.
 *
 * <p>Apps should obtain instances through {@link android.net.ConnectivityDiagnosticsManager.ConnectivityDiagnosticsCallback#onDataStallSuspected  } instead of instantiating their own
 * instances (unless for testing purposes).
 *
 * @param network The Network for which this DataStallReport applies
 * This value cannot be {@code null}.
 * @param reportTimestamp The timestamp for the report
 * @param detectionMethod The detection method used to identify this data stall
 * Value is {@link android.net.ConnectivityDiagnosticsManager.DataStallReport#DETECTION_METHOD_DNS_EVENTS}, or {@link android.net.ConnectivityDiagnosticsManager.DataStallReport#DETECTION_METHOD_TCP_METRICS}
 * @param linkProperties The LinkProperties available on network at reportTimestamp
 * This value cannot be {@code null}.
 * @param networkCapabilities The NetworkCapabilities available on network at
 *     reportTimestamp
 * This value cannot be {@code null}.
 * @param stallDetails A PersistableBundle that may contain additional info about the report

 * This value cannot be {@code null}.
 */

public DataStallReport(@android.annotation.NonNull android.net.Network network, long reportTimestamp, int detectionMethod, @android.annotation.NonNull android.net.LinkProperties linkProperties, @android.annotation.NonNull android.net.NetworkCapabilities networkCapabilities, @android.annotation.NonNull android.os.PersistableBundle stallDetails) { throw new RuntimeException("Stub!"); }

/**
 * Returns the Network for this DataStallReport.
 *
 * @return The Network for which this DataStallReport applied

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.Network getNetwork() { throw new RuntimeException("Stub!"); }

/**
 * Returns the epoch timestamp (milliseconds) for when this report was taken.
 *
 * @return The timestamp for the report. Taken from {@link java.lang.System#currentTimeMillis System#currentTimeMillis}.
 */

public long getReportTimestamp() { throw new RuntimeException("Stub!"); }

/**
 * Returns the bitmask of detection methods used to identify this suspected data stall.
 *
 * @return The bitmask of detection methods used to identify the suspected data stall
 */

public int getDetectionMethod() { throw new RuntimeException("Stub!"); }

/**
 * Returns the LinkProperties available when this report was taken.
 *
 * @return LinkProperties available on the Network at the reported timestamp

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.LinkProperties getLinkProperties() { throw new RuntimeException("Stub!"); }

/**
 * Returns the NetworkCapabilities when this report was taken.
 *
 * @return NetworkCapabilities available on the Network at the reported timestamp

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.NetworkCapabilities getNetworkCapabilities() { throw new RuntimeException("Stub!"); }

/**
 * Returns a PersistableBundle with additional info for this report.
 *
 * <p>Gets a bundle with details about the suspected data stall including information
 * specific to the monitoring method that detected the data stall.
 *
 * @return PersistableBundle that may contain additional information on the suspected data
 *     stall

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.os.PersistableBundle getStallDetails() { throw new RuntimeException("Stub!"); }

/**
 * {@inheritDoc}

 * @param o This value may be {@code null}.
 */

public boolean equals(@android.annotation.Nullable java.lang.Object o) { throw new RuntimeException("Stub!"); }

public int hashCode() { throw new RuntimeException("Stub!"); }

/** {@inheritDoc} */

public int describeContents() { throw new RuntimeException("Stub!"); }

/**
 * {@inheritDoc}
 * @param dest This value cannot be {@code null}.
 */

public void writeToParcel(@android.annotation.NonNull android.os.Parcel dest, int flags) { throw new RuntimeException("Stub!"); }

/** Implement the Parcelable interface */

@android.annotation.NonNull public static final android.os.Parcelable.Creator<android.net.ConnectivityDiagnosticsManager.DataStallReport> CREATOR;
static { CREATOR = null; }

/**
 * Indicates that the Data Stall was detected using DNS events.
 */

public static final int DETECTION_METHOD_DNS_EVENTS = 1; // 0x1

/**
 * Indicates that the Data Stall was detected using TCP metrics.
 */

public static final int DETECTION_METHOD_TCP_METRICS = 2; // 0x2

/**
 * This key represents the consecutive number of DNS timeouts that have occurred.
 *
 * <p>The consecutive count will be reset any time a DNS response is received.
 *
 * <p>This key will be included if the data stall detection method is
 * {@link #DETECTION_METHOD_DNS_EVENTS}.
 *
 * <p>This value is an int.
 */

public static final java.lang.String KEY_DNS_CONSECUTIVE_TIMEOUTS = "dnsConsecutiveTimeouts";

/**
 * This key represents the period in milliseconds over which other included TCP metrics
 * were measured.
 *
 * <p>This key will be included if the data stall detection method is
 * {@link #DETECTION_METHOD_TCP_METRICS}.
 *
 * <p>This value is an int.
 */

public static final java.lang.String KEY_TCP_METRICS_COLLECTION_PERIOD_MILLIS = "tcpMetricsCollectionPeriodMillis";

/**
 * This key represents the fail rate of TCP packets when the suspected data stall was
 * detected.
 *
 * <p>This key will be included if the data stall detection method is
 * {@link #DETECTION_METHOD_TCP_METRICS}.
 *
 * <p>This value is an int percentage between 0 and 100.
 */

public static final java.lang.String KEY_TCP_PACKET_FAIL_RATE = "tcpPacketFailRate";
}

}

