/*
 * Copyright (C) 2019 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.media;

import android.media.MediaCodec.CryptoInfo;
import java.io.IOException;

/**
 * Parses media container formats and extracts contained media samples and metadata.
 *
 * <p>This class provides access to a battery of low-level media container parsers. Each instance of
 * this class is associated to a specific media parser implementation which is suitable for
 * extraction from a specific media container format. The media parser implementation assignment
 * depends on the factory method (see {@link #create} and {@link #createByName}) used to create the
 * instance.
 *
 * <p>Users must implement the following to use this class.
 *
 * <ul>
 *   <li>{@link android.media.MediaParser.InputReader InputReader}: Provides the media container's bytes to parse.
 *   <li>{@link android.media.MediaParser.OutputConsumer OutputConsumer}: Provides a sink for all extracted data and metadata.
 * </ul>
 *
 * <p>The following code snippet includes a usage example:
 *
 * <pre>
 * MyOutputConsumer myOutputConsumer = new MyOutputConsumer();
 * MyInputReader myInputReader = new MyInputReader("www.example.com");
 * MediaParser mediaParser = MediaParser.create(myOutputConsumer);
 *
 * while (mediaParser.advance(myInputReader)) {}
 *
 * mediaParser.release();
 * mediaParser = null;
 * </pre>
 *
 * <p>The following code snippet provides a rudimentary {@link android.media.MediaParser.OutputConsumer OutputConsumer} sample implementation
 * which extracts and publishes all video samples:
 *
 * <pre>
 * class VideoOutputConsumer implements MediaParser.OutputConsumer {
 *
 *     private byte[] sampleDataBuffer = new byte[4096];
 *     private byte[] discardedDataBuffer = new byte[4096];
 *     private int videoTrackIndex = -1;
 *     private int bytesWrittenCount = 0;
 *
 *     &#64;Override
 *     public void onSeekMapFound(int i, &#64;NonNull MediaFormat mediaFormat) {
 *       // Do nothing.
 *     }
 *
 *     &#64;Override
 *     public void onTrackDataFound(int i, &#64;NonNull TrackData trackData) {
 *       MediaFormat mediaFormat = trackData.mediaFormat;
 *       if (videoTrackIndex == -1 &amp;&amp;
 *           mediaFormat
 *               .getString(MediaFormat.KEY_MIME, &#47;* defaultValue= *&#47; "")
 *               .startsWith("video/")) {
 *         videoTrackIndex = i;
 *       }
 *     }
 *
 *     &#64;Override
 *     public void onSampleDataFound(int trackIndex, &#64;NonNull InputReader inputReader)
 *         throws IOException {
 *       int numberOfBytesToRead = (int) inputReader.getLength();
 *       if (videoTrackIndex != trackIndex) {
 *         // Discard contents.
 *         inputReader.read(
 *             discardedDataBuffer,
 *             &#47;* offset= *&#47; 0,
 *             Math.min(discardDataBuffer.length, numberOfBytesToRead));
 *       } else {
 *         ensureSpaceInBuffer(numberOfBytesToRead);
 *         int bytesRead = inputReader.read(
 *             sampleDataBuffer, bytesWrittenCount, numberOfBytesToRead);
 *         bytesWrittenCount += bytesRead;
 *       }
 *     }
 *
 *     &#64;Override
 *     public void onSampleCompleted(
 *         int trackIndex,
 *         long timeMicros,
 *         int flags,
 *         int size,
 *         int offset,
 *         &#64;Nullable CryptoInfo cryptoData) {
 *       if (videoTrackIndex != trackIndex) {
 *         return; // It's not the video track. Ignore.
 *       }
 *       byte[] sampleData = new byte[size];
 *       int sampleStartOffset = bytesWrittenCount - size - offset;
 *       System.arraycopy(
 *           sampleDataBuffer,
 *           sampleStartOffset,
 *           sampleData,
 *           &#47;* destPos= *&#47; 0,
 *           size);
 *       // Place trailing bytes at the start of the buffer.
 *       System.arraycopy(
 *           sampleDataBuffer,
 *           bytesWrittenCount - offset,
 *           sampleDataBuffer,
 *           &#47;* destPos= *&#47; 0,
 *           &#47;* size= *&#47; offset);
 *       bytesWrittenCount = bytesWrittenCount - offset;
 *       publishSample(sampleData, timeMicros, flags);
 *     }
 *
 *    private void ensureSpaceInBuffer(int numberOfBytesToRead) {
 *      int requiredLength = bytesWrittenCount + numberOfBytesToRead;
 *      if (requiredLength &gt; sampleDataBuffer.length) {
 *        sampleDataBuffer = Arrays.copyOf(sampleDataBuffer, requiredLength);
 *      }
 *    }
 *
 *   }
 *
 * </pre>
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public final class MediaParser {

private MediaParser() { throw new RuntimeException("Stub!"); }

/**
 * Creates an instance backed by the parser with the given {@code name}. The returned instance
 * will attempt parsing without sniffing the content.
 *
 * @param name The name of the parser that will be associated with the created instance.
 * This value cannot be {@code null}.
 * Value is {@link android.media.MediaParser#PARSER_NAME_UNKNOWN}, {@link android.media.MediaParser#PARSER_NAME_MATROSKA}, {@link android.media.MediaParser#PARSER_NAME_FMP4}, {@link android.media.MediaParser#PARSER_NAME_MP4}, {@link android.media.MediaParser#PARSER_NAME_MP3}, {@link android.media.MediaParser#PARSER_NAME_ADTS}, {@link android.media.MediaParser#PARSER_NAME_AC3}, {@link android.media.MediaParser#PARSER_NAME_TS}, {@link android.media.MediaParser#PARSER_NAME_FLV}, {@link android.media.MediaParser#PARSER_NAME_OGG}, {@link android.media.MediaParser#PARSER_NAME_PS}, {@link android.media.MediaParser#PARSER_NAME_WAV}, {@link android.media.MediaParser#PARSER_NAME_AMR}, {@link android.media.MediaParser#PARSER_NAME_AC4}, or {@link android.media.MediaParser#PARSER_NAME_FLAC}
 * @param outputConsumer The {@link android.media.MediaParser.OutputConsumer OutputConsumer} to which track data and samples are pushed.
 * This value cannot be {@code null}.
 * @return A new instance.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException If an invalid name is provided.
 */

@android.annotation.NonNull
public static android.media.MediaParser createByName(@android.annotation.NonNull java.lang.String name, @android.annotation.NonNull android.media.MediaParser.OutputConsumer outputConsumer) { throw new RuntimeException("Stub!"); }

/**
 * Creates an instance whose backing parser will be selected by sniffing the content during the
 * first {@link #advance} call. Parser implementations will sniff the content in order of
 * appearance in {@code parserNames}.
 *
 * @param outputConsumer The {@link android.media.MediaParser.OutputConsumer OutputConsumer} to which extracted data is output.
 * This value cannot be {@code null}.
 * @param parserNames The names of the parsers to sniff the content with. If empty, a default
 *     array of names is used.
 * This value cannot be {@code null}.
 * Value is {@link android.media.MediaParser#PARSER_NAME_UNKNOWN}, {@link android.media.MediaParser#PARSER_NAME_MATROSKA}, {@link android.media.MediaParser#PARSER_NAME_FMP4}, {@link android.media.MediaParser#PARSER_NAME_MP4}, {@link android.media.MediaParser#PARSER_NAME_MP3}, {@link android.media.MediaParser#PARSER_NAME_ADTS}, {@link android.media.MediaParser#PARSER_NAME_AC3}, {@link android.media.MediaParser#PARSER_NAME_TS}, {@link android.media.MediaParser#PARSER_NAME_FLV}, {@link android.media.MediaParser#PARSER_NAME_OGG}, {@link android.media.MediaParser#PARSER_NAME_PS}, {@link android.media.MediaParser#PARSER_NAME_WAV}, {@link android.media.MediaParser#PARSER_NAME_AMR}, {@link android.media.MediaParser#PARSER_NAME_AC4}, or {@link android.media.MediaParser#PARSER_NAME_FLAC}
 * @return A new instance.
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public static android.media.MediaParser create(@android.annotation.NonNull android.media.MediaParser.OutputConsumer outputConsumer, @android.annotation.NonNull java.lang.String... parserNames) { throw new RuntimeException("Stub!"); }

/**
 * Returns an immutable list with the names of the parsers that are suitable for container
 * formats with the given {@link android.media.MediaFormat MediaFormat}.
 *
 * <p>A parser supports a {@link android.media.MediaFormat MediaFormat} if the mime type associated with {@link android.media.MediaFormat#KEY_MIME  } corresponds to the supported container format.
 *
 * @param mediaFormat The {@link android.media.MediaFormat MediaFormat} to check support for.
 * This value cannot be {@code null}.
 * @return The parser names that support the given {@code mediaFormat}, or the list of all
 *     parsers available if no container specific format information is provided.
 
 * This value cannot be {@code null}.
 
 * Value is {@link android.media.MediaParser#PARSER_NAME_UNKNOWN}, {@link android.media.MediaParser#PARSER_NAME_MATROSKA}, {@link android.media.MediaParser#PARSER_NAME_FMP4}, {@link android.media.MediaParser#PARSER_NAME_MP4}, {@link android.media.MediaParser#PARSER_NAME_MP3}, {@link android.media.MediaParser#PARSER_NAME_ADTS}, {@link android.media.MediaParser#PARSER_NAME_AC3}, {@link android.media.MediaParser#PARSER_NAME_TS}, {@link android.media.MediaParser#PARSER_NAME_FLV}, {@link android.media.MediaParser#PARSER_NAME_OGG}, {@link android.media.MediaParser#PARSER_NAME_PS}, {@link android.media.MediaParser#PARSER_NAME_WAV}, {@link android.media.MediaParser#PARSER_NAME_AMR}, {@link android.media.MediaParser#PARSER_NAME_AC4}, or {@link android.media.MediaParser#PARSER_NAME_FLAC}
 */

@android.annotation.NonNull
public static java.util.List<java.lang.String> getParserNames(@android.annotation.NonNull android.media.MediaFormat mediaFormat) { throw new RuntimeException("Stub!"); }

/**
 * Sets parser-specific parameters which allow customizing behavior.
 *
 * <p>Must be called before the first call to {@link #advance}.
 *
 * @param parameterName The name of the parameter to set. See {@code PARAMETER_*} constants for
 *     documentation on possible values.
 * This value cannot be {@code null}.
 * Value is {@link android.media.MediaParser#PARAMETER_ADTS_ENABLE_CBR_SEEKING}, {@link android.media.MediaParser#PARAMETER_AMR_ENABLE_CBR_SEEKING}, {@link android.media.MediaParser#PARAMETER_FLAC_DISABLE_ID3}, {@link android.media.MediaParser#PARAMETER_MP4_IGNORE_EDIT_LISTS}, {@link android.media.MediaParser#PARAMETER_MP4_IGNORE_TFDT_BOX}, {@link android.media.MediaParser#PARAMETER_MP4_TREAT_VIDEO_FRAMES_AS_KEYFRAMES}, {@link android.media.MediaParser#PARAMETER_MATROSKA_DISABLE_CUES_SEEKING}, {@link android.media.MediaParser#PARAMETER_MP3_DISABLE_ID3}, {@link android.media.MediaParser#PARAMETER_MP3_ENABLE_CBR_SEEKING}, {@link android.media.MediaParser#PARAMETER_MP3_ENABLE_INDEX_SEEKING}, {@link android.media.MediaParser#PARAMETER_TS_MODE}, {@link android.media.MediaParser#PARAMETER_TS_ALLOW_NON_IDR_AVC_KEYFRAMES}, {@link android.media.MediaParser#PARAMETER_TS_IGNORE_AAC_STREAM}, {@link android.media.MediaParser#PARAMETER_TS_IGNORE_AVC_STREAM}, {@link android.media.MediaParser#PARAMETER_TS_IGNORE_SPLICE_INFO_STREAM}, {@link android.media.MediaParser#PARAMETER_TS_DETECT_ACCESS_UNITS}, {@link android.media.MediaParser#PARAMETER_TS_ENABLE_HDMV_DTS_AUDIO_STREAMS}, android.media.MediaParser.PARAMETER_IN_BAND_CRYPTO_INFO, or android.media.MediaParser.PARAMETER_INCLUDE_SUPPLEMENTAL_DATA
 * @param value The value to set for the given {@code parameterName}. See {@code PARAMETER_*}
 *     constants for documentation on the expected types.
 * This value cannot be {@code null}.
 * @return This instance, for convenience.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalStateException If called after calling {@link #advance} on the same instance.
 */

@android.annotation.NonNull
public android.media.MediaParser setParameter(@android.annotation.NonNull java.lang.String parameterName, @android.annotation.NonNull java.lang.Object value) { throw new RuntimeException("Stub!"); }

/**
 * Returns whether the given {@code parameterName} is supported by this parser.
 *
 * @param parameterName The parameter name to check support for. One of the {@code PARAMETER_*}
 *     constants.
 * This value cannot be {@code null}.
 * Value is {@link android.media.MediaParser#PARAMETER_ADTS_ENABLE_CBR_SEEKING}, {@link android.media.MediaParser#PARAMETER_AMR_ENABLE_CBR_SEEKING}, {@link android.media.MediaParser#PARAMETER_FLAC_DISABLE_ID3}, {@link android.media.MediaParser#PARAMETER_MP4_IGNORE_EDIT_LISTS}, {@link android.media.MediaParser#PARAMETER_MP4_IGNORE_TFDT_BOX}, {@link android.media.MediaParser#PARAMETER_MP4_TREAT_VIDEO_FRAMES_AS_KEYFRAMES}, {@link android.media.MediaParser#PARAMETER_MATROSKA_DISABLE_CUES_SEEKING}, {@link android.media.MediaParser#PARAMETER_MP3_DISABLE_ID3}, {@link android.media.MediaParser#PARAMETER_MP3_ENABLE_CBR_SEEKING}, {@link android.media.MediaParser#PARAMETER_MP3_ENABLE_INDEX_SEEKING}, {@link android.media.MediaParser#PARAMETER_TS_MODE}, {@link android.media.MediaParser#PARAMETER_TS_ALLOW_NON_IDR_AVC_KEYFRAMES}, {@link android.media.MediaParser#PARAMETER_TS_IGNORE_AAC_STREAM}, {@link android.media.MediaParser#PARAMETER_TS_IGNORE_AVC_STREAM}, {@link android.media.MediaParser#PARAMETER_TS_IGNORE_SPLICE_INFO_STREAM}, {@link android.media.MediaParser#PARAMETER_TS_DETECT_ACCESS_UNITS}, {@link android.media.MediaParser#PARAMETER_TS_ENABLE_HDMV_DTS_AUDIO_STREAMS}, android.media.MediaParser.PARAMETER_IN_BAND_CRYPTO_INFO, or android.media.MediaParser.PARAMETER_INCLUDE_SUPPLEMENTAL_DATA
 * @return Whether the given {@code parameterName} is supported.
 */

public boolean supportsParameter(@android.annotation.NonNull java.lang.String parameterName) { throw new RuntimeException("Stub!"); }

/**
 * Returns the name of the backing parser implementation.
 *
 * <p>If this instance was creating using {@link #createByName}, the provided name is returned.
 * If this instance was created using {@link #create}, this method will return {@link
 * #PARSER_NAME_UNKNOWN} until the first call to {@link #advance}, after which the name of the
 * backing parser implementation is returned.
 *
 * @return The name of the backing parser implementation, or null if the backing parser
 *     implementation has not yet been selected.

 * Value is {@link android.media.MediaParser#PARSER_NAME_UNKNOWN}, {@link android.media.MediaParser#PARSER_NAME_MATROSKA}, {@link android.media.MediaParser#PARSER_NAME_FMP4}, {@link android.media.MediaParser#PARSER_NAME_MP4}, {@link android.media.MediaParser#PARSER_NAME_MP3}, {@link android.media.MediaParser#PARSER_NAME_ADTS}, {@link android.media.MediaParser#PARSER_NAME_AC3}, {@link android.media.MediaParser#PARSER_NAME_TS}, {@link android.media.MediaParser#PARSER_NAME_FLV}, {@link android.media.MediaParser#PARSER_NAME_OGG}, {@link android.media.MediaParser#PARSER_NAME_PS}, {@link android.media.MediaParser#PARSER_NAME_WAV}, {@link android.media.MediaParser#PARSER_NAME_AMR}, {@link android.media.MediaParser#PARSER_NAME_AC4}, or {@link android.media.MediaParser#PARSER_NAME_FLAC}
 */

@android.annotation.NonNull
public java.lang.String getParserName() { throw new RuntimeException("Stub!"); }

/**
 * Makes progress in the extraction of the input media stream, unless the end of the input has
 * been reached.
 *
 * <p>This method will block until some progress has been made.
 *
 * <p>If this instance was created using {@link #create}, the first call to this method will
 * sniff the content using the selected parser implementations.
 *
 * @param seekableInputReader The {@link android.media.MediaParser.SeekableInputReader SeekableInputReader} from which to obtain the media
 *     container data.
 * This value cannot be {@code null}.
 * @return Whether there is any data left to extract. Returns false if the end of input has been
 *     reached.
 * @throws java.io.IOException If an error occurs while reading from the {@link android.media.MediaParser.SeekableInputReader SeekableInputReader}.
 * @throws android.media.MediaParser.UnrecognizedInputFormatException If the format cannot be recognized by any of the
 *     underlying parser implementations.
 */

public boolean advance(@android.annotation.NonNull android.media.MediaParser.SeekableInputReader seekableInputReader) throws java.io.IOException { throw new RuntimeException("Stub!"); }

/**
 * Seeks within the media container being extracted.
 *
 * <p>{@link android.media.MediaParser.SeekPoint SeekPoints} can be obtained from the {@link android.media.MediaParser.SeekMap SeekMap} passed to {@link android.media.MediaParser.OutputConsumer#onSeekMapFound(android.media.MediaParser.SeekMap)  }.
 *
 * <p>Following a call to this method, the {@link android.media.MediaParser.InputReader InputReader} passed to the next invocation of
 * {@link #advance} must provide data starting from {@link android.media.MediaParser.SeekPoint#position SeekPoint#position} in the stream.
 *
 * @param seekPoint The {@link android.media.MediaParser.SeekPoint SeekPoint} to seek to.

 * This value cannot be {@code null}.
 */

public void seek(@android.annotation.NonNull android.media.MediaParser.SeekPoint seekPoint) { throw new RuntimeException("Stub!"); }

/**
 * Releases any acquired resources.
 *
 * <p>After calling this method, this instance becomes unusable and no other methods should be
 * invoked.
 */

public void release() { throw new RuntimeException("Stub!"); }

/**
 * @param logSessionId This value cannot be {@code null}.
 */

public void setLogSessionId(@android.annotation.NonNull android.media.metrics.LogSessionId logSessionId) { throw new RuntimeException("Stub!"); }

/**
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.media.metrics.LogSessionId getLogSessionId() { throw new RuntimeException("Stub!"); }

/**
 * Sets whether constant bitrate seeking should be enabled for ADTS parsing. {@code boolean}
 * expected. Default value is {@code false}.
 */

public static final java.lang.String PARAMETER_ADTS_ENABLE_CBR_SEEKING = "android.media.mediaparser.adts.enableCbrSeeking";

/**
 * Sets whether constant bitrate seeking should be enabled for AMR. {@code boolean} expected.
 * Default value is {@code false}.
 */

public static final java.lang.String PARAMETER_AMR_ENABLE_CBR_SEEKING = "android.media.mediaparser.amr.enableCbrSeeking";

/**
 * Sets whether the ID3 track should be disabled for FLAC. {@code boolean} expected. Default
 * value is {@code false}.
 */

public static final java.lang.String PARAMETER_FLAC_DISABLE_ID3 = "android.media.mediaparser.flac.disableId3";

/**
 * Sets whether Matroska parsing should avoid seeking to the cues element. {@code boolean}
 * expected. Default value is {@code false}.
 *
 * <p>If this flag is enabled and the cues element occurs after the first cluster, then the
 * media is treated as unseekable.
 */

public static final java.lang.String PARAMETER_MATROSKA_DISABLE_CUES_SEEKING = "android.media.mediaparser.matroska.disableCuesSeeking";

/**
 * Sets whether the ID3 track should be disabled for MP3. {@code boolean} expected. Default
 * value is {@code false}.
 */

public static final java.lang.String PARAMETER_MP3_DISABLE_ID3 = "android.media.mediaparser.mp3.disableId3";

/**
 * Sets whether constant bitrate seeking should be enabled for MP3. {@code boolean} expected.
 * Default value is {@code false}.
 */

public static final java.lang.String PARAMETER_MP3_ENABLE_CBR_SEEKING = "android.media.mediaparser.mp3.enableCbrSeeking";

/**
 * Sets whether MP3 parsing should generate a time-to-byte mapping. {@code boolean} expected.
 * Default value is {@code false}.
 *
 * <p>Enabling this flag may require to scan a significant portion of the file to compute a seek
 * point. Therefore, it should only be used if:
 *
 * <ul>
 *   <li>the file is small, or
 *   <li>the bitrate is variable (or the type of bitrate is unknown) and the seeking metadata
 *       provided in the file is not precise enough (or is not present).
 * </ul>
 */

public static final java.lang.String PARAMETER_MP3_ENABLE_INDEX_SEEKING = "android.media.mediaparser.mp3.enableIndexSeeking";

/**
 * Sets whether MP4 parsing should ignore edit lists. {@code boolean} expected. Default value is
 * {@code false}.
 */

public static final java.lang.String PARAMETER_MP4_IGNORE_EDIT_LISTS = "android.media.mediaparser.mp4.ignoreEditLists";

/**
 * Sets whether MP4 parsing should ignore the tfdt box. {@code boolean} expected. Default value
 * is {@code false}.
 */

public static final java.lang.String PARAMETER_MP4_IGNORE_TFDT_BOX = "android.media.mediaparser.mp4.ignoreTfdtBox";

/**
 * Sets whether MP4 parsing should treat all video frames as key frames. {@code boolean}
 * expected. Default value is {@code false}.
 */

public static final java.lang.String PARAMETER_MP4_TREAT_VIDEO_FRAMES_AS_KEYFRAMES = "android.media.mediaparser.mp4.treatVideoFramesAsKeyframes";

/**
 * Sets whether TS should treat samples consisting of non-IDR I slices as synchronization
 * samples (key-frames). {@code boolean} expected. Default value is {@code false}.
 */

public static final java.lang.String PARAMETER_TS_ALLOW_NON_IDR_AVC_KEYFRAMES = "android.media.mediaparser.ts.allowNonIdrAvcKeyframes";

/**
 * Sets whether TS parsing should split AVC stream into access units based on slice headers.
 * {@code boolean} expected. Default value is {@code false}.
 *
 * <p>This flag should be left disabled if the stream contains access units delimiters in order
 * to avoid unnecessary computational costs.
 */

public static final java.lang.String PARAMETER_TS_DETECT_ACCESS_UNITS = "android.media.mediaparser.ts.ignoreDetectAccessUnits";

/**
 * Sets whether TS parsing should handle HDMV DTS audio streams. {@code boolean} expected.
 * Default value is {@code false}.
 *
 * <p>Enabling this flag will disable the detection of SCTE subtitles.
 */

public static final java.lang.String PARAMETER_TS_ENABLE_HDMV_DTS_AUDIO_STREAMS = "android.media.mediaparser.ts.enableHdmvDtsAudioStreams";

/**
 * Sets whether TS parsing should ignore AAC elementary streams. {@code boolean} expected.
 * Default value is {@code false}.
 */

public static final java.lang.String PARAMETER_TS_IGNORE_AAC_STREAM = "android.media.mediaparser.ts.ignoreAacStream";

/**
 * Sets whether TS parsing should ignore AVC elementary streams. {@code boolean} expected.
 * Default value is {@code false}.
 */

public static final java.lang.String PARAMETER_TS_IGNORE_AVC_STREAM = "android.media.mediaparser.ts.ignoreAvcStream";

/**
 * Sets whether TS parsing should ignore splice information streams. {@code boolean} expected.
 * Default value is {@code false}.
 */

public static final java.lang.String PARAMETER_TS_IGNORE_SPLICE_INFO_STREAM = "android.media.mediaparser.ts.ignoreSpliceInfoStream";

/**
 * Sets the operation mode for TS parsing. {@code String} expected. Valid values are {@code
 * "single_pmt"}, {@code "multi_pmt"}, and {@code "hls"}. Default value is {@code "single_pmt"}.
 *
 * <p>The operation modes alter the way TS behaves so that it can handle certain kinds of
 * commonly-occurring malformed media.
 *
 * <ul>
 *   <li>{@code "single_pmt"}: Only the first found PMT is parsed. Others are ignored, even if
 *       more PMTs are declared in the PAT.
 *   <li>{@code "multi_pmt"}: Behave as described in ISO/IEC 13818-1.
 *   <li>{@code "hls"}: Enable {@code "single_pmt"} mode, and ignore continuity counters.
 * </ul>
 */

public static final java.lang.String PARAMETER_TS_MODE = "android.media.mediaparser.ts.mode";

/**
 * Parser for the AC-3 container format, as defined in Digital Audio Compression Standard
 * (AC-3).
 */

public static final java.lang.String PARSER_NAME_AC3 = "android.media.mediaparser.Ac3Parser";

/**
 * Parser for the AC-4 container format, as defined by Dolby AC-4: Audio delivery for
 * Next-Generation Entertainment Services.
 */

public static final java.lang.String PARSER_NAME_AC4 = "android.media.mediaparser.Ac4Parser";

/** Parser for the ADTS container format, as defined in ISO/IEC 13818-7. */

public static final java.lang.String PARSER_NAME_ADTS = "android.media.mediaparser.AdtsParser";

/** Parser for the AMR container format, as defined in RFC 4867. */

public static final java.lang.String PARSER_NAME_AMR = "android.media.mediaparser.AmrParser";

/**
 * Parser for the FLAC container format, as defined in the <a
 * href="https://xiph.org/flac/">spec</a>.
 */

public static final java.lang.String PARSER_NAME_FLAC = "android.media.mediaparser.FlacParser";

/**
 * Parser for the FLV container format, as defined in Adobe Flash Video File Format
 * Specification.
 */

public static final java.lang.String PARSER_NAME_FLV = "android.media.mediaparser.FlvParser";

/**
 * Parser for fragmented files using the MP4 container format, as defined in ISO/IEC 14496-12.
 */

public static final java.lang.String PARSER_NAME_FMP4 = "android.media.mediaparser.FragmentedMp4Parser";

/**
 * Parser for the Matroska container format, as defined in the <a
 * href="https://matroska.org/technical/specs/">spec</a>.
 */

public static final java.lang.String PARSER_NAME_MATROSKA = "android.media.mediaparser.MatroskaParser";

/** Parser for the MP3 container format, as defined in ISO/IEC 11172-3. */

public static final java.lang.String PARSER_NAME_MP3 = "android.media.mediaparser.Mp3Parser";

/**
 * Parser for non-fragmented files using the MP4 container format, as defined in ISO/IEC
 * 14496-12.
 */

public static final java.lang.String PARSER_NAME_MP4 = "android.media.mediaparser.Mp4Parser";

/** Parser for the OGG container format, as defined in RFC 3533. */

public static final java.lang.String PARSER_NAME_OGG = "android.media.mediaparser.OggParser";

/** Parser for the PS container format, as defined in ISO/IEC 11172-1. */

public static final java.lang.String PARSER_NAME_PS = "android.media.mediaparser.PsParser";

/** Parser for the TS container format, as defined in ISO/IEC 13818-1. */

public static final java.lang.String PARSER_NAME_TS = "android.media.mediaparser.TsParser";

/** Parser name returned by {@link #getParserName()} when no parser has been selected yet. */

public static final java.lang.String PARSER_NAME_UNKNOWN = "android.media.mediaparser.UNKNOWN";

/**
 * Parser for the WAV container format, as defined in Multimedia Programming Interface and Data
 * Specifications.
 */

public static final java.lang.String PARSER_NAME_WAV = "android.media.mediaparser.WavParser";

/** Indicates that the sample should be decoded but not rendered. */

public static final int SAMPLE_FLAG_DECODE_ONLY = -2147483648; // 0x80000000

/** Indicates that the sample is (at least partially) encrypted. */

public static final int SAMPLE_FLAG_ENCRYPTED = 1073741824; // 0x40000000

/**
 * Indicates that the sample has supplemental data.
 *
 * <p>Samples will not have this flag set unless the {@code
 * "android.media.mediaparser.includeSupplementalData"} parameter is set to {@code true} via
 * {@link #setParameter}.
 *
 * <p>Samples with supplemental data have the following sample data format:
 *
 * <ul>
 *   <li>If the {@code "android.media.mediaparser.inBandCryptoInfo"} parameter is set, all
 *       encryption information.
 *   <li>(4 bytes) {@code sample_data_size}: The size of the actual sample data, not including
 *       supplemental data or encryption information.
 *   <li>({@code sample_data_size} bytes): The media sample data.
 *   <li>(remaining bytes) The supplemental data.
 * </ul>
 */

public static final int SAMPLE_FLAG_HAS_SUPPLEMENTAL_DATA = 268435456; // 0x10000000

/** Indicates that the sample holds a synchronization sample. */

public static final int SAMPLE_FLAG_KEY_FRAME = 1; // 0x1

/** Indicates that the sample is known to contain the last media sample of the stream. */

public static final int SAMPLE_FLAG_LAST_SAMPLE = 536870912; // 0x20000000
/** Provides input data to {@link android.media.MediaParser MediaParser}. */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static interface InputReader {

/**
 * Reads up to {@code readLength} bytes of data and stores them into {@code buffer},
 * starting at index {@code offset}.
 *
 * <p>This method blocks until at least one byte is read, the end of input is detected, or
 * an exception is thrown. The read position advances to the first unread byte.
 *
 * @param buffer The buffer into which the read data should be stored.
 * This value cannot be {@code null}.
 * @param offset The start offset into {@code buffer} at which data should be written.
 * @param readLength The maximum number of bytes to read.
 * @return The non-zero number of bytes read, or -1 if no data is available because the end
 *     of the input has been reached.
 * @throws java.io.IOException If an error occurs reading from the source.
 */

public int read(@android.annotation.NonNull byte[] buffer, int offset, int readLength) throws java.io.IOException;

/** Returns the current read position (byte offset) in the stream. */

public long getPosition();

/** Returns the length of the input in bytes, or -1 if the length is unknown. */

public long getLength();
}

/** Receives extracted media sample data and metadata from {@link android.media.MediaParser MediaParser}. */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static interface OutputConsumer {

/**
 * Called when a {@link android.media.MediaParser.SeekMap SeekMap} has been extracted from the stream.
 *
 * <p>This method is called at least once before any samples are {@link #onSampleCompleted
 * complete}. May be called multiple times after that in order to add {@link android.media.MediaParser.SeekPoint  SeekPoints}.
 *
 * @param seekMap The extracted {@link android.media.MediaParser.SeekMap SeekMap}.

 * This value cannot be {@code null}.
 */

public void onSeekMapFound(@android.annotation.NonNull android.media.MediaParser.SeekMap seekMap);

/**
 * Called when the number of tracks is found.
 *
 * @param numberOfTracks The number of tracks in the stream.
 */

public void onTrackCountFound(int numberOfTracks);

/**
 * Called when new {@link android.media.MediaParser.TrackData TrackData} is found in the stream.
 *
 * @param trackIndex The index of the track for which the {@link android.media.MediaParser.TrackData TrackData} was extracted.
 * @param trackData The extracted {@link android.media.MediaParser.TrackData TrackData}.

 * This value cannot be {@code null}.
 */

public void onTrackDataFound(int trackIndex, @android.annotation.NonNull android.media.MediaParser.TrackData trackData);

/**
 * Called when sample data is found in the stream.
 *
 * <p>If the invocation of this method returns before the entire {@code inputReader} {@link android.media.MediaParser.InputReader#getLength() length} is consumed, the method will be called again for the
 * implementer to read the remaining data. Implementers should surface any thrown {@link java.io.IOException IOException} caused by reading from {@code input}.
 *
 * @param trackIndex The index of the track to which the sample data corresponds.
 * @param inputReader The {@link android.media.MediaParser.InputReader InputReader} from which to read the data.
 * This value cannot be {@code null}.
 * @throws java.io.IOException If an exception occurs while reading from {@code inputReader}.
 */

public void onSampleDataFound(int trackIndex, @android.annotation.NonNull android.media.MediaParser.InputReader inputReader) throws java.io.IOException;

/**
 * Called once all the data of a sample has been passed to {@link #onSampleDataFound}.
 *
 * <p>Includes sample metadata, like presentation timestamp and flags.
 *
 * @param trackIndex The index of the track to which the sample corresponds.
 * @param timeMicros The media timestamp associated with the sample, in microseconds.
 * @param flags Flags associated with the sample. See the {@code SAMPLE_FLAG_*} constants.
 * Value is either <code>0</code> or a combination of {@link android.media.MediaParser#SAMPLE_FLAG_KEY_FRAME}, {@link android.media.MediaParser#SAMPLE_FLAG_HAS_SUPPLEMENTAL_DATA}, {@link android.media.MediaParser#SAMPLE_FLAG_LAST_SAMPLE}, {@link android.media.MediaParser#SAMPLE_FLAG_ENCRYPTED}, and {@link android.media.MediaParser#SAMPLE_FLAG_DECODE_ONLY}
 * @param size The size of the sample data, in bytes.
 * @param offset The number of bytes that have been consumed by {@code
 *     onSampleDataFound(int, MediaParser.InputReader)} for the specified track, since the
 *     last byte belonging to the sample whose metadata is being passed.
 * @param cryptoInfo Encryption data required to decrypt the sample. May be null for
 *     unencrypted samples. Implementors should treat any output {@link android.media.MediaCodec.CryptoInfo CryptoInfo}
 *     instances as immutable. MediaParser will not modify any output {@code cryptoInfos}
 *     and implementors should not modify them either.
 */

public void onSampleCompleted(int trackIndex, long timeMicros, int flags, int size, int offset, @android.annotation.Nullable android.media.MediaCodec.CryptoInfo cryptoInfo);
}

/** Thrown when an error occurs while parsing a media stream. */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static final class ParsingException extends java.io.IOException {

private ParsingException() { throw new RuntimeException("Stub!"); }
}

/**
 * Maps seek positions to {@link android.media.MediaParser.SeekPoint SeekPoints} in the stream.
 *
 * <p>A {@link android.media.MediaParser.SeekPoint SeekPoint} is a position in the stream from which a player may successfully start
 * playing media samples.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static final class SeekMap {

private SeekMap() { throw new RuntimeException("Stub!"); }

/** Returns whether seeking is supported. */

public boolean isSeekable() { throw new RuntimeException("Stub!"); }

/**
 * Returns the duration of the stream in microseconds or {@link #UNKNOWN_DURATION} if the
 * duration is unknown.
 */

public long getDurationMicros() { throw new RuntimeException("Stub!"); }

/**
 * Obtains {@link android.media.MediaParser.SeekPoint SeekPoints} for the specified seek time in microseconds.
 *
 * <p>{@code getSeekPoints(timeMicros).first} contains the latest seek point for samples
 * with timestamp equal to or smaller than {@code timeMicros}.
 *
 * <p>{@code getSeekPoints(timeMicros).second} contains the earliest seek point for samples
 * with timestamp equal to or greater than {@code timeMicros}. If a seek point exists for
 * {@code timeMicros}, the returned pair will contain the same {@link android.media.MediaParser.SeekPoint SeekPoint} twice.
 *
 * @param timeMicros A seek time in microseconds.
 * @return The corresponding {@link android.media.MediaParser.SeekPoint SeekPoints}.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.util.Pair<android.media.MediaParser.SeekPoint,android.media.MediaParser.SeekPoint> getSeekPoints(long timeMicros) { throw new RuntimeException("Stub!"); }

/** Returned by {@link #getDurationMicros()} when the duration is unknown. */

public static final int UNKNOWN_DURATION = -2147483648; // 0x80000000
}

/** Defines a seek point in a media stream. */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static final class SeekPoint {

private SeekPoint() { throw new RuntimeException("Stub!"); }

/**
 * {@inheritDoc}

 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public java.lang.String toString() { throw new RuntimeException("Stub!"); }

/**
 * {@inheritDoc}

 * @param obj This value may be {@code null}.
 */

public boolean equals(@android.annotation.Nullable java.lang.Object obj) { throw new RuntimeException("Stub!"); }

public int hashCode() { throw new RuntimeException("Stub!"); }

/** A {@link android.media.MediaParser.SeekPoint SeekPoint} whose time and byte offset are both set to 0. */

@android.annotation.NonNull public static final android.media.MediaParser.SeekPoint START;
static { START = null; }

/** The byte offset of the seek point. */

public final long position;
{ position = 0; }

/** The time of the seek point, in microseconds. */

public final long timeMicros;
{ timeMicros = 0; }
}

/** {@link android.media.MediaParser.InputReader InputReader} that allows setting the read position. */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static interface SeekableInputReader extends android.media.MediaParser.InputReader {

/**
 * Sets the read position at the given {@code position}.
 *
 * <p>{@link #advance} will immediately return after calling this method.
 *
 * @param position The position to seek to, in bytes.
 */

public void seekToPosition(long position);
}

/** Holds information associated with a track. */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static final class TrackData {

private TrackData() { throw new RuntimeException("Stub!"); }

/**
 * Holds {@link android.media.DrmInitData DrmInitData} necessary to acquire keys associated with the track, or null if
 * the track has no encryption data.
 */

@android.annotation.Nullable public final android.media.DrmInitData drmInitData;
{ drmInitData = null; }

/** Holds {@link android.media.MediaFormat MediaFormat} information for the track. */

@android.annotation.NonNull public final android.media.MediaFormat mediaFormat;
{ mediaFormat = null; }
}

/**
 * Thrown if all parser implementations provided to {@link #create} failed to sniff the input
 * content.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static final class UnrecognizedInputFormatException extends java.io.IOException {

private UnrecognizedInputFormatException() { throw new RuntimeException("Stub!"); }
}

}

