/* Copyright (C) 2016 J.F.Dockes
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser General Public License as published by
 *     the Free Software Foundation; either version 2.1 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Lesser General Public License for more details.
 *
 *     You should have received a copy of the GNU Lesser General Public License
 *     along with this program; if not, write to the
 *     Free Software Foundation, Inc.,
 *     59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#ifndef _CONTENTDIRECTORY_H_INCLUDED_
#define _CONTENTDIRECTORY_H_INCLUDED_

#include <string>
#include <vector>

#include "libupnpp/device/device.hxx"
#include "libupnpp/soaphelp.hxx"

#include "cdplugins/cdplugin.hxx"
#include "mediaserver.hxx"

using namespace UPnPP;

// Content Directory service
//
// The actual work is done by separate plugins (CDPlugin interface),
// depending of where the data comes from (streaming service, e.g,
// Tidal, Qobuz, or local file store - uprcl). There is currently a
// single implementation of CDPlugin (PlgWithSlave) for which most of
// the differentiated work is done by a Python subprocess specific to
// the backend.
//
// The CDPluginServices interface provides some service callbacks to
// the implementations.
//
// The class is responsible for generating the root directory,
// according to what services are enabled. For deeper directories, the
// dispatch to the right service is performed according to the first
// element in the object ID (e.g. $qobuz$).
//
// Accordingly, the media access also depends on the type of backend,
// and some media URIs are also formed in order to be dispatchable
// (e.g. http://xxx/qobuz/...). Some also point directly to a server
// implemented by the plugin. Depending on the service, either
// redirection or proxying is used to supply the data stream.
//
// We also have a special operation mode where the browsing is done by
// the CP and we are only used for accessing the streams (see
// constructor comment).
class ContentDirectory : public UPnPProvider::UpnpService,
                         public CDPluginServices {
public:

    // Constructor.
    // 
    // Enabled==false: means that no content directory service is
    // actually needed. No service will be enabled, but we perform
    // some initializations. This is used in conjunction with
    // OHCredentials. The Control Point (typ. Kazoo) does the
    // browsing, and sends tidal://xx or qobuz://xx URIs to
    // OHPlaylist. The URI contain the service trackid, and are
    // translated by OHPlaylist/OHCredentials into the form which is
    // generated when this class does the browsing. The data access is
    // then performed as if the URIs had been generated by us.
    ContentDirectory(MediaServer *dev, bool enabled);

    ~ContentDirectory();

    /// Check if the configuration indicates that the media server
    /// needs to be started.
    static bool mediaServerNeeded();

    /////////// CDPluginServices methods

    /// Return plugin based on path prefix
    CDPlugin *getpluginforpath(const std::string& path);
    /// Friendly name
    virtual std::string getfname();

    /// Retrieve the IP address and port for the libupnp server. URLs
    /// intended to be served this way (by adding a vdir) should use
    /// these as host/port. Not used at present.
    virtual std::string getupnpaddr(CDPlugin *);
    virtual int getupnpport(CDPlugin *);
    /// Host to use in proxy urls. This is normally the same as the
    /// UPnP host, but can be forced in the configuration. The
    /// microhttp server listens on all addresses.
    virtual std::string microhttphost();
    
private:
    int actGetSearchCapabilities(const SoapIncoming& sc, SoapOutgoing& data);
    int actGetSortCapabilities(const SoapIncoming& sc, SoapOutgoing& data);
    int actGetSystemUpdateID(const SoapIncoming& sc, SoapOutgoing& data);
    int actBrowse(const SoapIncoming& sc, SoapOutgoing& data);
    int actSearch(const SoapIncoming& sc, SoapOutgoing& data);

    class Internal;
    Internal *m;
};
#endif

