# -*- coding: UTF-8 -*-
#/**
# * Software Name : pycrate
# * Version : 0.4
# *
# * Copyright 2019. Benoit Michau. P1Sec.
# *
# * This library is free software; you can redistribute it and/or
# * modify it under the terms of the GNU Lesser General Public
# * License as published by the Free Software Foundation; either
# * version 2.1 of the License, or (at your option) any later version.
# *
# * This library is distributed in the hope that it will be useful,
# * but WITHOUT ANY WARRANTY; without even the implied warranty of
# * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# * Lesser General Public License for more details.
# *
# * You should have received a copy of the GNU Lesser General Public
# * License along with this library; if not, write to the Free Software
# * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
# * MA 02110-1301  USA
# *
# *--------------------------------------------------------
# * File Name : pycrate_diameter/Diameter3GPP.py
# * Created : 2019-08-01
# * Authors : Benoit Michau 
# *--------------------------------------------------------
#*/

__all__ = [
    'TGPPAVPInfos_dict',
    'FMT_LUT_TS29230',
    'AVP3GPPCodes_dict',
    'Grouped',
    'AVP3GPP',
    'Diameter3GPP'
    ]


#------------------------------------------------------------------------------#
# 3GPP TS 29.230, Rel.15.6
# https://portal.3gpp.org/desktopmodules/Specifications/SpecificationDetails.aspx?specificationId=1683
# 3GPP specific implementation (with AVP 3GPP specific format)
#------------------------------------------------------------------------------#

from pycrate_core.elt               import Sequence
from pycrate_diameter.Diameter      import *
from pycrate_diameter.DiameterIETF  import *


#------------------------------------------------------------------------------#
# 3GPP-augmented AVP
#------------------------------------------------------------------------------#

class AVP3GPP(AVPGeneric):
    pass


class Grouped(Sequence):
    _GEN = AVP3GPP()


#------------------------------------------------------------------------------#
# 7.1 3GPP specific AVP codes
#------------------------------------------------------------------------------#

TGPPAVPInfos_dict = {
    100 : ('3GPP-WLAN-APN-Id', OctetString),
    101 : ('3GPP-WLAN-QoS-Filter-Rule', UTF8String),
    102 : ('3GPP-WLAN-QoS-Filter-Support', OctetString),
    300 : ('Authentication-Method', Enumerated),
    301 : ('Authentication-Information-SIM', OctetString),
    302 : ('Authorization-Information-SIM', OctetString),
    303 : ('WLAN-User-Data', Grouped),
    304 : ('Charging-Data', Grouped),
    305 : ('WLAN-Access', Enumerated),
    306 : ('WLAN-3GPP-IP-Access', Enumerated),
    307 : ('APN-Authorized', Grouped),
    309 : ('APN-Barring-Type', Enumerated),
    310 : ('WLAN-Direct-IP-Access', Enumerated),
    311 : ('Session-Request-Type', Enumerated),
    312 : ('Routing-Policy', IPFilterRule),
    313 : ('Max-Requested-Bandwidth', OctetString),
    314 : ('Charging-Characteristics', Integer32),
    315 : ('Charging-Nodes', Grouped),
    316 : ('Primary-OCS-Charging-Function-Name', DiameterIdentity),
    317 : ('Secondary-OCS-Charging-Function-Name', DiameterIdentity),
    318 : ('3GPP-AAA-Server-Name', DiameterIdentity),
    319 : ('Maximum-Number-Accesses', Unsigned32),
    400 : ('GBA-UserSecSettings', OctetString),
    401 : ('Transaction-Identifier', OctetString),
    402 : ('NAF-Hostname', OctetString),
    403 : ('GAA-Service-Identifier', OctetString),
    404 : ('Key-ExpiryTime', Time),
    405 : ('ME-Key-Material', OctetString),
    406 : ('UICC-Key-Material', OctetString),
    407 : ('GBA_U-Awareness-Indicator', Enumerated),
    408 : ('BootstrapInfoCreationTime', Time),
    409 : ('GUSS-Timestamp', Time),
    410 : ('GBA-Type', Enumerated),
    411 : ('UE-Id', OctetString),
    412 : ('UE-Id-Type', Enumerated),
    413 : ('UICC-App-Label', OctetString),
    414 : ('UICC-ME', Enumerated),
    415 : ('Requested-Key-Lifetime', Time),
    416 : ('Private-Identity-Request', Enumerated),
    417 : ('GBA-Push-Info', OctetString),
    418 : ('NAF-SA-Identifier', OctetString),
    419 : ('Security-Feature-Request', OctetString),
    420 : ('Security-Feature-Response', OctetString),
    500 : ('Abort-Cause', Enumerated),
    501 : ('Access-Network-Charging-Address', Address),
    502 : ('Access-Network-Charging-Identifier', Grouped),
    503 : ('Access-Network-Charging-Identifier-Value', OctetString),
    504 : ('AF-Application-Identifier', OctetString),
    505 : ('AF-Charging-Identifier', OctetString),
    506 : ('Authorization-Token', OctetString),
    507 : ('Flow-Description', IPFilterRule),
    508 : ('Flow-Grouping', Grouped),
    509 : ('Flow-Number', Unsigned32),
    510 : ('Flows', Grouped),
    511 : ('Flow-Status', Enumerated),
    512 : ('Flow-Usage', Enumerated),
    513 : ('Specific-Action', Enumerated),
    514 : ('Max-Requested-Bandwidth', Unsigned32),
    515 : ('Max-Requested-Bandwidth-DL', Unsigned32),
    516 : ('Max-Requested-Bandwidth-UL', Unsigned32),
    517 : ('Media-Component-Description', Grouped),
    518 : ('Media-Component-Number', Unsigned32),
    519 : ('Media-Sub-ComponentAVP', Grouped),
    520 : ('Media-Type', Enumerated),
    521 : ('RR-Bandwidth', Unsigned32),
    522 : ('RS-Bandwidth', Unsigned32),
    523 : ('SIP-Forking-Indication', Enumerated),
    524 : ('Codec-Data', OctetString),
    525 : ('Service-URN', OctetString),
    526 : ('Acceptable-Service-Info', Grouped),
    527 : ('Service-Info-Status', Enumerated),
    528 : ('MPS-Identifier', OctetString),
    529 : ('AF-Signalling-Protocol', Enumerated),
    530 : ('Sponsored-Connectivity-Data', Grouped),
    531 : ('Sponsor-Identity', OctetString),
    532 : ('Application-Service-Provider-Identity', OctetString),
    533 : ('Rx-Request-Type', Enumerated),
    534 : ('Min-Requested-Bandwidth-DL', Unsigned32),
    535 : ('Min-Requested-Bandwidth-UL', Unsigned32),
    536 : ('Required-Access-Info', Enumerated),
    537 : ('IP-Domain-Id', OctetString),
    538 : ('GCS-Identifier', OctetString),
    539 : ('Sharing-Key-DL', Unsigned32),
    540 : ('Sharing-Key-UL', Unsigned32),
    541 : ('Retry-Interval', Unsigned32),
    542 : ('Sponsoring-Action', Enumerated),
    543 : ('Max-Supported-Bandwidth-DL', Unsigned32),
    544 : ('Max-Supported-Bandwidth-UL', Unsigned32),
    545 : ('Min-Desired-Bandwidth-DL', Unsigned32),
    546 : ('Min-Desired-Bandwidth-UL', Unsigned32),
    547 : ('MCPTT-Identifier', OctetString),
    548 : ('Service-Authorization-Info', Unsigned32),
    549 : ('Media-Component-Status', Unsigned32),
    550 : ('Priority-Sharing-Indicator', Enumerated),
    551 : ('AF-Requested-Data', Unsigned32),
    552 : ('Content-Version', Unsigned64),
    553 : ('Pre-emption-Control-Info', Unsigned32),
    554 : ('Extended-Max-Requested-BW-DL', Unsigned32),
    555 : ('Extended-Max-Requested-BW-UL', Unsigned32),
    556 : ('Extended-Max-Supported-BW-DL', Unsigned32),
    557 : ('Extended-Max-Supported-BW-UL', Unsigned32),
    558 : ('Extended-Min-Desired-BW-DL', Unsigned32),
    559 : ('Extended-Min-Desired-BW-UL', Unsigned32),
    560 : ('Extended-Min-Requested-BW-DL', Unsigned32),
    561 : ('Extended-Min-Requested-BW-UL', Unsigned32),
    562 : ('MCVideo-Identifier', OctetString),
    600 : ('Visited-Network-Identifier', OctetString),
    601 : ('Public-Identity', UTF8String),
    602 : ('Server-Name', UTF8String),
    603 : ('Server-Capabilities', Grouped),
    604 : ('Mandatory-Capability', Unsigned32),
    605 : ('Optional-Capability', Unsigned32),
    606 : ('User-Data', OctetString),
    607 : ('SIP-Number-Auth-Items', Unsigned32),
    608 : ('SIP-Authentication-Scheme', UTF8String),
    609 : ('SIP-Authenticate', OctetString),
    610 : ('SIP-Authorization', OctetString),
    611 : ('SIP-Authentication-Context', OctetString),
    612 : ('SIP-Auth-Data-Item', Grouped),
    613 : ('SIP-Item-Number', Unsigned32),
    614 : ('Server-Assignment-Type', Enumerated),
    615 : ('Deregistration-Reason', Grouped),
    616 : ('Reason-Code', Enumerated),
    617 : ('Reason-Info', UTF8String),
    618 : ('Charging-Information', Grouped),
    619 : ('Primary-Event-Charging-Function-Name', DiameterURI),
    620 : ('Secondary-Event-Charging-Function-Name', DiameterURI),
    621 : ('Primary-Charging-Collection-Function-Name', DiameterURI),
    622 : ('Secondary-Charging-Collection-Function-Name', DiameterURI),
    623 : ('User-Authorization-Type', Enumerated),
    624 : ('User-Data-Already-Available', Enumerated),
    625 : ('Confidentiality-Key', OctetString),
    626 : ('Integrity-Key', OctetString),
    627 : ('User-Data-Request-Type', Enumerated),
    628 : ('Supported-Features', Grouped),
    629 : ('Feature-List-ID', Unsigned32),
    630 : ('Feature-List', Unsigned32),
    631 : ('Supported-Applications', Grouped),
    632 : ('Associated-Identities', Grouped),
    633 : ('Originating-Request', Enumerated),
    634 : ('Wildcarded-Public-Identity', UTF8String),
    635 : ('SIP-Digest-Authenticate', Grouped),
    636 : ('Wildcarded-IMPU', UTF8String),
    637 : ('UAR-Flags', Unsigned32),
    638 : ('Loose-Route-Indication', Enumerated),
    639 : ('SCSCF-Restoration-Info', Grouped),
    640 : ('Path', OctetString),
    641 : ('Contact', OctetString),
    642 : ('Subscription-Info', Grouped),
    643 : ('Call-ID-SIP-Header', OctetString),
    644 : ('From-SIP-Header', OctetString),
    645 : ('To-SIP-Header', OctetString),
    646 : ('Record-Route', OctetString),
    647 : ('Associated-Registered-Identities', Grouped),
    648 : ('Multiple-Registration-Indication', Enumerated),
    649 : ('Restoration-Info', Grouped),
    650 : ('Session-Priority', Enumerated),
    651 : ('Identity-with-Emergency-Registration', Grouped),
    652 : ('Priviledged-Sender-Indication', Enumerated),
    653 : ('LIA-Flags', Unsigned32),
    654 : ('Initial-CSeq-Sequence-Number', Unsigned32),
    655 : ('SAR-Flags', Unsigned32),
    656 : ('Allowed-WAF-WWSF-Identities', Grouped),
    657 : ('WebRTC-Authentication-Function-Name', UTF8String),
    658 : ('WebRTC-Web-Server-Function-Name', UTF8String),
    659 : ('RTR-Flags', Unsigned32),
    700 : ('User-Identity', Grouped),
    701 : ('MSISDN', OctetString),
    702 : ('User-Data', OctetString),
    703 : ('Data-Reference', Enumerated),
    704 : ('Service-Indication', OctetString),
    705 : ('Subs-Req-Type', Enumerated),
    706 : ('Requested-Domain', Enumerated),
    707 : ('Current-Location', Enumerated),
    708 : ('Identity-Set', Enumerated),
    709 : ('Expiry-Time', Time),
    710 : ('Send-Data-Indication', Enumerated),
    711 : ('DSAI-Tag', OctetString),
    712 : ('One-Time-Notification', Enumerated),
    713 : ('Requested-Nodes', Unsigned32),
    714 : ('Serving-Node-Indication', Enumerated),
    715 : ('Repository-Data-ID', Grouped),
    716 : ('Sequence-Number', Unsigned32),
    717 : ('Pre-paging-Supported', Enumerated),
    718 : ('Local-Time-Zone-Indication', Enumerated),
    719 : ('UDR-Flags', Unsigned32),
    720 : ('Call-Reference-Info', Grouped),
    721 : ('Call-Reference-Number', OctetString),
    722 : ('AS-Number', OctetString),
    823 : ('Event-Type', Grouped),
    824 : ('SIP-Method', UTF8String),
    825 : ('Event', UTF8String),
    826 : ('Content-Type', UTF8String),
    827 : ('Content-Length', Unsigned32),
    828 : ('Content-Disposition', UTF8String),
    829 : ('Role-of-Node', Enumerated),
    830 : ('User-Session-Id', UTF8String),
    831 : ('Calling-Party-Address', UTF8String),
    832 : ('Called-Party-Address', UTF8String),
    833 : ('Time-Stamps', Grouped),
    834 : ('SIP-Request-Timestamp', Time),
    835 : ('SIP-Response-Timestamp', Time),
    836 : ('Application-Server', UTF8String),
    837 : ('Application-provided-called-party-address', UTF8String),
    838 : ('Inter-Operator-Identifier', Grouped),
    839 : ('Originating-IOI', UTF8String),
    840 : ('Terminating-IOI', UTF8String),
    841 : ('IMS-Charging-Identifier', UTF8String),
    842 : ('SDP-Session-Description', UTF8String),
    843 : ('SDP-Media-Component', Grouped),
    844 : ('SDP-Media-Name', UTF8String),
    845 : ('SDP-Media-Description', UTF8String),
    846 : ('CG-Address', Address),
    847 : ('GGSN-Address', Address),
    848 : ('Served-Party-IP-Address', Address),
    849 : ('Authorized-QoS', UTF8String),
    850 : ('Application-Server-Information', Grouped),
    851 : ('Trunk-Group-Id', Grouped),
    852 : ('Incoming-Trunk-Group-Id', UTF8String),
    853 : ('Outgoing-Trunk-Group-Id', UTF8String),
    854 : ('Bearer-Service', OctetString),
    855 : ('Service-Id', UTF8String),
    856 : ('Associated-URI', UTF8String),
    857 : ('Charged-Party', UTF8String),
    858 : ('PoC-Controlling-Address', UTF8String),
    859 : ('PoC-Group-Name', UTF8String),
    860 : ('Cause', Grouped),
    861 : ('Cause-Code', Integer32),
    862 : ('Node-Functionality', Enumerated),
    863 : ('Service-Specific-Data', UTF8String),
    864 : ('Originator', Enumerated),
    865 : ('PS-Furnish-Charging-Information', Grouped),
    866 : ('PS-Free-Format-Data', OctetString),
    867 : ('PS-Append-Free-Format-Data', Enumerated),
    868 : ('Time-Quota-Threshold', Unsigned32),
    869 : ('Volume-Quota-Threshold', Unsigned32),
    870 : ('Trigger-Type', Enumerated),
    871 : ('Quota-Holding-Time', Unsigned32),
    872 : ('Reporting-Reason', Enumerated),
    873 : ('Service-Information', Grouped),
    874 : ('PS-Information', Grouped),
    876 : ('IMS-Information', Grouped),
    877 : ('MMS-Information', Grouped),
    878 : ('LCS-Information', Grouped),
    879 : ('PoC-Information', Grouped),
    880 : ('MBMS-Information', Grouped),
    881 : ('Quota-Consumption-Time', Unsigned32),
    882 : ('Media-Initiator-Flag', Enumerated),
    883 : ('PoC-Server-Role', Enumerated),
    884 : ('PoC-Session-Type', Enumerated),
    885 : ('Number-Of-Participants', Unsigned32),
    886 : ('Originator-Address', Grouped),
    887 : ('Participants-Involved', UTF8String),
    888 : ('Expires', Unsigned32),
    889 : ('Message-Body', Grouped),
    897 : ('Address-Data', UTF8String),
    898 : ('Address-Domain', Grouped),
    899 : ('Address-Type', Enumerated),
    900 : ('TMGI', OctetString),
    901 : ('Required-MBMS-Bearer-Capabilities', UTF8String),
    902 : ('MBMS-StartStop-Indication', Enumerated),
    903 : ('MBMS-Service-Area', OctetString),
    904 : ('MBMS-Session-Duration', Unsigned32),
    905 : ('Alternative-APN', UTF8String),
    906 : ('MBMS-Service-Type', Enumerated),
    907 : ('MBMS-2G-3G-Indicator', Enumerated),
    908 : ('MBMS-Session-Identity', OctetString),
    909 : ('RAI', UTF8String),
    910 : ('Additional-MBMS-Trace-Info', OctetString),
    911 : ('MBMS-Time-To-Data-Transfer', Unsigned32),
    912 : ('MBMS-Session-Identity-Repetition-Number', Unsigned32),
    913 : ('MBMS-Required-QoS', UTF8String),
    914 : ('MBMS-Counting-Information', Enumerated),
    915 : ('MBMS-User-Data-Mode-Indication', Enumerated),
    916 : ('MBMS-GGSN-Address', UTF8String),
    917 : ('MBMS-GGSN-IPv6-Address', UTF8String),
    918 : ('MBMS-BMSC-SSM-IP-Address', UTF8String),
    919 : ('MBMS-BMSC-SSM-IPv6-Address', UTF8String),
    920 : ('MBMS-Flow-Identifier', OctetString),
    921 : ('CN-IP-Multicast-Distribution', Enumerated),
    922 : ('MBMS-HC-Indicator', Enumerated),
    923 : ('MBMS-Access-Indicator', Enumerated),
    924 : ('MBMS-GW-SSM-IP-Address', OctetString),
    925 : ('MBMS-GW-SSM-IPv6-Address', OctetString),
    926 : ('MBMS-BMSC-SSM-UDP-Port', OctetString),
    927 : ('MBMS-GW-UDP-Port', OctetString),
    928 : ('MBMS-GW-UDP-Port-Indicator', Enumerated),
    929 : ('MBMS-Data-Transfer-Start', Unsigned64),
    930 : ('MBMS-Data-Transfer-Stop', Unsigned64),
    931 : ('MBMS-Flags', Unsigned32),
    932 : ('Restart-Counter', Unsigned32),
    933 : ('Diagnostic-Info', Unsigned32),
    934 : ('MBMS-Cell-List', OctetString),
    1000 : ('Bearer-Usage', Enumerated),
    1001 : ('Charging-Rule-Install', Grouped),
    1002 : ('Charging-Rule-Remove', Grouped),
    1003 : ('Charging-Rule-Definition', Grouped),
    1004 : ('Charging-Rule-Base-Name', UTF8String),
    1005 : ('Charging-Rule-Name', OctetString),
    1006 : ('Event-Trigger', Enumerated),
    1007 : ('Metering-Method', Enumerated),
    1008 : ('Offline', Enumerated),
    1009 : ('Online', Enumerated),
    1010 : ('Precedence', Unsigned32),
    1011 : ('Reporting-Level', Enumerated),
    1012 : ('TFT-Filter', IPFilterRule),
    1013 : ('TFT-Packet-Filter-Information', Grouped),
    1014 : ('ToS-Traffic-Class', OctetString),
    1016 : ('QoS-Information', Grouped),
    1018 : ('Charging-Rule-Report', Grouped),
    1019 : ('PCC-Rule-Status', Enumerated),
    1020 : ('Bearer-Identifier', OctetString),
    1021 : ('Bearer-Operation', Enumerated),
    1022 : ('Access-Network-Charging-Identifier-Gx', Grouped),
    1023 : ('Bearer-Control-Mode', Enumerated),
    1024 : ('Network-Request-Support', Enumerated),
    1025 : ('Guaranteed-Bitrate-DL', Unsigned32),
    1026 : ('Guaranteed-Bitrate-UL', Unsigned32),
    1027 : ('IP-CAN-Type', Enumerated),
    1028 : ('QoS-Class-Identifier', Enumerated),
    1029 : ('QoS-Negotiation', Enumerated),
    1030 : ('QoS-Upgrade', Enumerated),
    1031 : ('Rule-Failure-Code', Enumerated),
    1032 : ('RAT-Type', Enumerated),
    1033 : ('Event-Report-Indication', Grouped),
    1034 : ('Allocation-Retention-Priority', Grouped),
    1035 : ('CoA-IP-Address', Address),
    1036 : ('Tunnel-Header-Filter', IPFilterRule),
    1037 : ('Tunnel-Header-Length', Unsigned32),
    1038 : ('Tunnel-Information', Grouped),
    1039 : ('CoA-Information', Grouped),
    1040 : ('APN-Aggregate-Max-Bitrate-DL', Unsigned32),
    1041 : ('APN-Aggregate-Max-Bitrate-UL', Unsigned32),
    1042 : ('Revalidation-Time', Time),
    1043 : ('Rule-Activation-Time', Time),
    1044 : ('Rule-DeActivation-Time', Time),
    1045 : ('Session-Release-Cause', Enumerated),
    1046 : ('Priority-Level', Unsigned32),
    1047 : ('Pre-emption-Capability', Enumerated),
    1048 : ('Pre-emption-Vulnerability', Enumerated),
    1049 : ('Default-EPS-Bearer-QoS', Grouped),
    1050 : ('AN-GW-Address', Address),
    1051 : ('QoS-Rule-Install', Grouped),
    1052 : ('QoS-Rule-Remove', Grouped),
    1053 : ('QoS-Rule-Definition', Grouped),
    1054 : ('QoS-Rule-Name', OctetString),
    1055 : ('QoS-Rule-Report', Grouped),
    1056 : ('Security-Parameter-Index', OctetString),
    1057 : ('Flow-Label', OctetString),
    1058 : ('Flow-Information', Grouped),
    1059 : ('Packet-Filter-Content', IPFilterRule),
    1060 : ('Packet-Filter-Identifier', OctetString),
    1061 : ('Packet-Filter-Information', Grouped),
    1062 : ('Packet-Filter-Operation', Enumerated),
    1063 : ('Resource-Allocation-Notification', Enumerated),
    1064 : ('Session-Linking-Indicator', Enumerated),
    1065 : ('PDN-Connection-ID', OctetString),
    1066 : ('Monitoring-Key', OctetString),
    1067 : ('Usage-Monitoring-Information', Grouped),
    1068 : ('Usage-Monitoring-Level', Enumerated),
    1069 : ('Usage-Monitoring-Report', Enumerated),
    1070 : ('Usage-Monitoring-Support', Enumerated),
    1071 : ('CSG-Information-Reporting', Enumerated),
    1072 : ('Packet-Filter-Usage', Enumerated),
    1073 : ('Charging-Correlation-Indicator', Enumerated),
    1074 : ('QoS-Rule-Base-Name', UTF8String),
    1075 : ('Routing-Rule-Remove', Grouped),
    1076 : ('Routing-Rule-Definition', Grouped),
    1077 : ('Routing-Rule-Identifier', OctetString),
    1078 : ('Routing-Filter', Grouped),
    1079 : ('Routing-IP-Address', Address),
    1080 : ('Flow-Direction', Enumerated),
    1081 : ('Routing-Rule-Install', Grouped),
    1082 : ('Credit-Management-Status', Unsigned32),
    1085 : ('Redirect-Information', Grouped),
    1086 : ('Redirect-Support', Enumerated),
    1087 : ('TDF-Information', Grouped),
    1088 : ('TDF-Application-Identifier', OctetString),
    1089 : ('TDF-Destination-Host', DiameterIdentity),
    1090 : ('TDF-Destination-Realm', DiameterIdentity),
    1091 : ('TDF-IP-address', Address),
    1092 : ('ADC-Rule-Install', Grouped),
    1093 : ('ADC-Rule-Remove', Grouped),
    1094 : ('ADC-Rule-Definition', Grouped),
    1095 : ('ADC-Rule-Base-Name', UTF8String),
    1096 : ('ADC-Rule-Name', OctetString),
    1097 : ('ADC-Rule-Report', Grouped),
    1098 : ('Application-Detection-Information', Grouped),
    1099 : ('PS-to-CS-Session-Continuity', Enumerated),
    1100 : ('Served-User-Identity', Grouped),
    1101 : ('VASP-ID', UTF8String),
    1102 : ('VAS-ID', UTF8String),
    1103 : ('Trigger-Event', Enumerated),
    1104 : ('Sender-Address', UTF8String),
    1105 : ('Initial-Recipient-Address', Grouped),
    1106 : ('Result-Recipient-Address', Grouped),
    1107 : ('Sequence-Number', Unsigned32),
    1108 : ('Recipient-Address', UTF8String),
    1109 : ('Routeing-Address', UTF8String),
    1110 : ('Originating-Interface', Enumerated),
    1111 : ('Delivery-Report', Enumerated),
    1112 : ('Read-Reply', Enumerated),
    1113 : ('Sender-Visibility', Enumerated),
    1114 : ('Service-Key', UTF8String),
    1115 : ('Billing-Information', UTF8String),
    1116 : ('Status', Grouped),
    1117 : ('Status-Code', UTF8String),
    1118 : ('Status-Text', UTF8String),
    1200 : ('Domain-Name', UTF8String),
    1201 : ('Recipient-Address', Grouped),
    1202 : ('Submission-Time', Time),
    1203 : ('MM-Content-Type', Grouped),
    1204 : ('Type-Number', Enumerated),
    1205 : ('Additional-Type-Information', UTF8String),
    1206 : ('Content-Size', Unsigned32),
    1207 : ('Additional-Content-Information', Grouped),
    1208 : ('Addressee-Type', Enumerated),
    1209 : ('Priority', Enumerated),
    1210 : ('Message-ID', UTF8String),
    1211 : ('Message-Type', Enumerated),
    1212 : ('Message-Size', Unsigned32),
    1213 : ('Message-Class', Grouped),
    1214 : ('Class-Identifier', Enumerated),
    1215 : ('Token-Text', UTF8String),
    1216 : ('Delivery-Report-Requested', Enumerated),
    1217 : ('Adaptations', Enumerated),
    1218 : ('Applic-ID', UTF8String),
    1219 : ('Aux-Applic-Info', UTF8String),
    1220 : ('Content-Class', Enumerated),
    1221 : ('DRM-Content', Enumerated),
    1222 : ('Read-Reply-Report-Requested', Enumerated),
    1223 : ('Reply-Applic-ID', UTF8String),
    1224 : ('File-Repair-Supported', Enumerated),
    1225 : ('MBMS-User-Service-Type', Enumerated),
    1226 : ('Unit-Quota-Threshold', Unsigned32),
    1227 : ('PDP-Address', Address),
    1228 : ('SGSN-Address', Address),
    1229 : ('PoC-Session-Id', UTF8String),
    1230 : ('Deferred-Location-Even-Type', UTF8String),
    1231 : ('LCS-APN', UTF8String),
    1232 : ('LCS-Client-Id', Grouped),
    1233 : ('LCS-Client-Dialed-By-MS', UTF8String),
    1234 : ('LCS-Client-External-ID', UTF8String),
    1235 : ('LCS-Client-Name', Grouped),
    1236 : ('LCS-Data-Coding-Scheme', UTF8String),
    1237 : ('LCS-Format-Indicator', Enumerated),
    1238 : ('LCS-Name-String', UTF8String),
    1239 : ('LCS-Requestor-Id', Grouped),
    1240 : ('LCS-Requestor-Id-String', UTF8String),
    1241 : ('LCS-Client-Type', Enumerated),
    1242 : ('Location-Estimate', OctetString),
    1243 : ('Location-Estimate-Type', Enumerated),
    1244 : ('Location-Type', Grouped),
    1245 : ('Positioning-Data', UTF8String),
    1247 : ('PDP-Context-Type', Enumerated),
    1248 : ('MMBox-Storage-Requested', Enumerated),
    1249 : ('Service-Specific-Info', Grouped),
    1250 : ('Called-Asserted-Identity', UTF8String),
    1251 : ('Requested-Party-Address', UTF8String),
    1252 : ('PoC-User-Role', Grouped),
    1253 : ('PoC-User-Role-IDs', UTF8String),
    1254 : ('PoC-User-Role-info-Units', Enumerated),
    1255 : ('Talk-Burst-Exchange', Grouped),
    1256 : ('Service-Generic-Information', Grouped),
    1257 : ('Service-Specific-Type', Unsigned32),
    1258 : ('Event-Charging-TimeStamp', Time),
    1259 : ('Participant-Access-Priority', Enumerated),
    1260 : ('Participant-Group', Grouped),
    1261 : ('PoC-Change-Conditions', Enumerated),
    1262 : ('PoC-Change-Time', Time),
    1263 : ('Access-Network-Information', OctetString),
    1264 : ('Trigger', Grouped),
    1265 : ('Base-Time-Interval', Unsigned32),
    1266 : ('Envelope', Grouped),
    1267 : ('Envelope-End-Time', Time),
    1268 : ('Envelope-Reporting', Enumerated),
    1269 : ('Envelope-Start-Time', Time),
    1270 : ('Time-Quota-Mechanism', Grouped),
    1271 : ('Time-Quota-Type', Enumerated),
    1272 : ('Early-Media-Description', Grouped),
    1273 : ('SDP-TimeStamps', Grouped),
    1274 : ('SDP-Offer-Timestamp', Time),
    1275 : ('SDP-Answer-Timestamp', Time),
    1276 : ('AF-Correlation-Information', Grouped),
    1277 : ('PoC-Session-Initiation-type', Enumerated),
    1278 : ('Offline-Charging', Grouped),
    1279 : ('User-Participating-Type', Enumerated),
    1280 : ('Alternate-Charged-Party-Address', UTF8String),
    1281 : ('IMS-Communication-Service-Identifier', UTF8String),
    1282 : ('Number-Of-Received-Talk-Bursts', Unsigned32),
    1283 : ('Number-Of-Talk-Bursts', Unsigned32),
    1284 : ('Received-Talk-Burst-Time', Unsigned32),
    1285 : ('Received-Talk-Burst-Volume', Unsigned32),
    1286 : ('Talk-Burst-Time', Unsigned32),
    1287 : ('Talk-Burst-Volume', Unsigned32),
    1288 : ('Media-Initiator-Party', UTF8String),
    1300 : ('PC5-Radio-Technology', Enumerated),
    1301 : ('RAN-End-Time', Time),
    1302 : ('RAN-Secondary-RAT-Usage-Report', Grouped),
    1303 : ('RAN-Start-Time', Time),
    1304 : ('Secondary-RAT-Type', OctetString),
    1305 : ('Civic-Address-Information', UTF8String),
    1306 : ('WLAN-Operator-Id', Grouped),
    1307 : ('WLAN-Operator-Name', UTF8String),
    1308 : ('WLAN-PLMN-Id', UTF8String),
    1309 : ('API-Content', UTF8String),
    1310 : ('API-Direction', Enumerated),
    1311 : ('API-Identifier', OctetString),
    1312 : ('API-Invocation-Timestamp', Time),
    1313 : ('API-Result-Code', Unsigned32),
    1314 : ('API-Size', Unsigned64),
    1315 : ('API-Network-Service-Node', Enumerated),
    1316 : ('Exposure-Function-API-Information', Grouped),
    1317 : ('SCEF-Address', Address),
    1318 : ('TLTRI', Unsigned32),
    1319 : ('TTRI', Unsigned32),
    1400 : ('Subscription-Data', Grouped),
    1401 : ('Terminal-Information', Grouped),
    1402 : ('IMEI', UTF8String),
    1403 : ('Software-Version', UTF8String),
    1404 : ('QoS-Subscribed', UTF8String),
    1405 : ('ULR-Flags', Unsigned32),
    1406 : ('ULA-Flags', Unsigned32),
    1407 : ('Visited-PLMN-Id', OctetString),
    1408 : ('Requested-EUTRAN-Authentication-Info', Grouped),
    1409 : ('Requested-UTRAN-GERAN-Authentication-Info', Grouped),
    1410 : ('Number-Of-Requested-Vectors', Unsigned32),
    1411 : ('Re-Synchronization-Info', OctetString),
    1412 : ('Immediate-Response-Preferred', Unsigned32),
    1413 : ('Authentication-Info', Grouped),
    1414 : ('E-UTRAN-Vector', Grouped),
    1415 : ('UTRAN-Vector', Grouped),
    1416 : ('GERAN-Vector', Grouped),
    1417 : ('Network-Access-Mode', Enumerated),
    1418 : ('HPLMN-ODB', Enumerated),
    1419 : ('Item-Number', Unsigned32),
    1420 : ('Cancellation-Type', Enumerated),
    1421 : ('DSR-Flags', Unsigned32),
    1422 : ('DSA-Flags', Unsigned32),
    1423 : ('Context-Identifier', Unsigned32),
    1424 : ('Subscriber-Status', Enumerated),
    1425 : ('Operator-Determined-Barring', Unsigned32),
    1426 : ('Access-Restriction-Data', Unsigned32),
    1427 : ('APN-OI-Replacement', UTF8String),
    1428 : ('All-APN-Configurations-Included-Indicator', Enumerated),
    1429 : ('APN-Configuration-Profile', Grouped),
    1430 : ('APN-Configuration', Grouped),
    1431 : ('EPS-Subscribed-QoS-Profile', Grouped),
    1432 : ('VPLMN-Dynamic-Address-Allowed', Enumerated),
    1433 : ('STN-SR', OctetString),
    1434 : ('Alert-Reason', Enumerated),
    1435 : ('AMBR', Grouped),
    1436 : ('CSG-Subscription-Data', Grouped),
    1437 : ('CSG-Id', Unsigned32),
    1438 : ('PDN-GW-Allocation-Type', Enumerated),
    1439 : ('Expiration-Date', Time),
    1440 : ('RAT-Frequency-Selection-Priority-ID', Unsigned32),
    1441 : ('IDA-Flags', Unsigned32),
    1442 : ('PUA-Flags', Unsigned32),
    1443 : ('NOR-Flags', Unsigned32),
    1444 : ('User-Id', UTF8String),
    1445 : ('Equipment-Status', Enumerated),
    1446 : ('Regional-Subscription-Zone-Code', OctetString),
    1447 : ('RAND', OctetString),
    1448 : ('XRES', OctetString),
    1449 : ('AUTN', OctetString),
    1450 : ('KASME', OctetString),
    1452 : ('Trace-Collection-Entity', Address),
    1453 : ('Kc', OctetString),
    1454 : ('SRES', OctetString),
    1456 : ('PDN-Type', Enumerated),
    1457 : ('Roaming-Restricted-Due-To-Unsupported-Feature', Enumerated),
    1458 : ('Trace-Data', Grouped),
    1459 : ('Trace-Reference', OctetString),
    1462 : ('Trace-Depth', Enumerated),
    1463 : ('Trace-NE-Type-List', OctetString),
    1464 : ('Trace-Interface-List', OctetString),
    1465 : ('Trace-Event-List', OctetString),
    1466 : ('OMC-Id', OctetString),
    1467 : ('GPRS-Subscription-Data', Grouped),
    1468 : ('Complete-Data-List-Included-Indicator', Enumerated),
    1469 : ('PDP-Context', Grouped),
    1470 : ('PDP-Type', OctetString),
    1471 : ('3GPP2-MEID', OctetString),
    1472 : ('Specific-APN-Info', Grouped),
    1473 : ('LCS-Info', Grouped),
    1474 : ('GMLC-Number', OctetString),
    1475 : ('LCS-PrivacyException', Grouped),
    1476 : ('SS-Code', OctetString),
    1477 : ('SS-Status', OctetString),
    1478 : ('Notification-To-UE-User', Enumerated),
    1479 : ('External-Client', Grouped),
    1480 : ('Client-Identity', OctetString),
    1481 : ('GMLC-Restriction', Enumerated),
    1482 : ('PLMN-Client', Enumerated),
    1483 : ('Service-Type', Grouped),
    1484 : ('ServiceTypeIdentity', Unsigned32),
    1485 : ('MO-LR', Grouped),
    1486 : ('Teleservice-List', Grouped),
    1487 : ('TS-Code', Enumerated),
    1488 : ('Call-Barring-Info', Grouped),
    1489 : ('SGSN-Number', OctetString),
    1490 : ('IDR-Flags', Unsigned32),
    1491 : ('ICS-Indicator', Enumerated),
    1492 : ('IMS-Voice-Over-PS-Sessions-Supported', Enumerated),
    1493 : ('Homogeneous-Support-of-IMS-Voice-Over-PS-Sessions', Enumerated),
    1494 : ('Last-UE-Activity-Time', Time),
    1495 : ('EPS-User-State', Grouped),
    1496 : ('EPS-Location-Information', Grouped),
    1497 : ('MME-User-State', Grouped),
    1498 : ('SGSN-User-State', Grouped),
    1499 : ('User-State', Enumerated),
    1500 : ('Non-3GPP-User-Data', Grouped),
    1501 : ('Non-3GPP-IP-Access', Enumerated),
    1502 : ('Non-3GPP-IP-Access-APN', Enumerated),
    1503 : ('AN-Trusted', Enumerated),
    1504 : ('ANID', UTF8String),
    1505 : ('Trace-Info', Grouped),
    1506 : ('MIP-FA-RK', OctetString),
    1507 : ('MIP-FA-RK-SPI', Unsigned32),
    1508 : ('PPR-Flags', Unsigned32),
    1509 : ('WLAN-Identifier', Grouped),
    1510 : ('TWAN-Access-Info', Grouped),
    1511 : ('Access-Authorization-Flags', Unsigned32),
    1512 : ('TWAN-Default-APN-Context-Id', Unsigned32),
    1516 : ('Full-Network-Name', OctetString),
    1517 : ('Short-Network-Name', OctetString),
    1518 : ('AAA-Failure-Indication', Unsigned32),
    1519 : ('Transport-Access-Type', Enumerated),
    1520 : ('DER-Flags', Unsigned32),
    1521 : ('DEA-Flags', Unsigned32),
    1522 : ('RAR-Flags', Unsigned32),
    1523 : ('DER-S6b-Flags', Unsigned32),
    1524 : ('SSID', UTF8String),
    1525 : ('HESSID', UTF8String),
    1526 : ('Access-Network-Info', Grouped),
    1527 : ('TWAN-Connection-Mode', Unsigned32),
    1528 : ('TWAN-Connectivity-Parameters', Grouped),
    1529 : ('Connectivity-Flags', Unsigned32),
    1530 : ('TWAN-PCO', OctetString),
    1531 : ('TWAG-CP-Address', Address),
    1532 : ('TWAG-UP-Address', UTF8String),
    1533 : ('TWAN-S2a-Failure-Cause', Unsigned32),
    1534 : ('SM-Back-Off-Timer', Unsigned32),
    1535 : ('WLCP-Key', OctetString),
    1536 : ('Origination-Time-Stamp', Unsigned64),
    1537 : ('Maximum-Wait-Time', Unsigned32),
    1538 : ('Emergency-Services', Unsigned32),
    1539 : ('AAR-Flags', Unsigned32),
    1540 : ('IMEI-Check-In-VPLMN-Result', Unsigned32),
    1541 : ('ERP-Authorization', Unsigned32),
    1600 : ('MME-Location-Information', Grouped),
    1601 : ('SGSN-Location-Information', Grouped),
    1602 : ('E-UTRAN-Cell-Global-Identity', OctetString),
    1603 : ('Tracking-Area-Identity', OctetString),
    1604 : ('Cell-Global-Identity', OctetString),
    1605 : ('Routing-Area-Identity', OctetString),
    1606 : ('Location-Area-Identity', OctetString),
    1607 : ('Service-Area-Identity', OctetString),
    1608 : ('Geographical-Information', OctetString),
    1609 : ('Geodetic-Information', OctetString),
    1610 : ('Current-Location-Retrieved', Enumerated),
    1611 : ('Age-Of-Location-Information', Unsigned32),
    1612 : ('Active-APN', Grouped),
    1613 : ('SIPTO-Permission', Enumerated),
    1614 : ('Error-Diagnostic', Enumerated),
    1615 : ('UE-SRVCC-Capability', Enumerated),
    1616 : ('MPS-Priority', Unsigned32),
    1617 : ('VPLMN-LIPA-Allowed', Enumerated),
    1618 : ('LIPA-Permission', Enumerated),
    1619 : ('Subscribed-Periodic-RAU-TAU-Timer', Unsigned32),
    1620 : ('Ext-PDP-Type', OctetString),
    1621 : ('Ext-PDP-Address', Address),
    1622 : ('MDT-Configuration', Grouped),
    1623 : ('Job-Type', Enumerated),
    1624 : ('Area-Scope', Grouped),
    1625 : ('List-Of-Measurements', Unsigned32),
    1626 : ('Reporting-Trigger', Unsigned32),
    1627 : ('Report-Interval', Enumerated),
    1628 : ('Report-Amount', Enumerated),
    1629 : ('Event-ThresholdRSRP', Unsigned32),
    1630 : ('Event-ThresholdRSRQ', Unsigned32),
    1631 : ('Logging-Interval', Enumerated),
    1632 : ('Logging-Duration', Enumerated),
    1633 : ('Relay-Node-Indicator', Enumerated),
    1634 : ('MDT-User-Consent', Enumerated),
    1635 : ('PUR-Flags', Unsigned32),
    1636 : ('Subscribed-VSRVCC', Enumerated),
    1637 : ('Equivalent-PLMN-List', Grouped),
    1638 : ('CLR-Flags', Unsigned32),
    1639 : ('UVR-Flags', Unsigned32),
    1640 : ('UVA-Flags', Unsigned32),
    1641 : ('VPLMN-CSG-Subscription-Data', Grouped),
    1642 : ('Time-Zone', UTF8String),
    1643 : ('A-MSISDN', OctetString),
    1645 : ('MME-Number-for-MT-SMS', OctetString),
    1648 : ('SMS-Register-Request', Enumerated),
    1649 : ('Local-Time-Zone', Grouped),
    1650 : ('Daylight-Saving-Time', Enumerated),
    1651 : ('GMLC-Info', Grouped),
    1652 : ('GMLC-Name', DiameterIdentity),
    1653 : ('GMLC-Realm', DiameterIdentity),
    1654 : ('Subscription-Data-Flags', Unsigned32),
    1655 : ('Measurement-Period-LTE', Enumerated),
    1656 : ('Measurement-Period-UMTS', Enumerated),
    1657 : ('Collection-Period-RRM-LTE', Enumerated),
    1658 : ('Collection-Period-RRM-UMTS', Enumerated),
    1659 : ('Positioning-Method', OctetString),
    1660 : ('Measurement-Quantity', OctetString),
    1661 : ('Event-Threshold-Event-1F', Integer32),
    1662 : ('Event-Threshold-Event-1I', Integer32),
    1663 : ('Restoration-Priority', Unsigned32),
    1664 : ('SGs-MME-Identity', UTF8String),
    1665 : ('SIPTO-Local-Network-Permission', Unsigned32),
    1666 : ('Coupled-Node-Diameter-ID', DiameterIdentity),
    1667 : ('WLAN-offloadability', Grouped),
    1668 : ('WLAN-offloadability-EUTRAN', Unsigned32),
    1669 : ('WLAN-offloadability-UTRAN', Unsigned32),
    1670 : ('Reset-ID', OctetString),
    1671 : ('MDT-Allowed-PLMN-Id', OctetString),
    1672 : ('Adjacent-PLMNs', Grouped),
    1673 : ('Adjacent-Access-Restriction-Data', Grouped),
    1674 : ('DL-Buffering-Suggested-Packet-Count', Integer32),
    1675 : ('IMSI-Group-Id', Grouped),
    1676 : ('Group-Service-Id', Unsigned32),
    1677 : ('Group-PLMN-Id', OctetString),
    1678 : ('Local-Group-Id', OctetString),
    1679 : ('AIR-Flags', Unsigned32),
    1680 : ('UE-Usage-Type', Unsigned32),
    1681 : ('Non-IP-PDN-Type-Indicator', Enumerated),
    1682 : ('Non-IP-Data-Delivery-Mechanism', Unsigned32),
    1683 : ('Additional-Context-Identifier', Unsigned32),
    1684 : ('SCEF-Realm', DiameterIdentity),
    1685 : ('Subscription-Data-Deletion', Grouped),
    1686 : ('Preferred-Data-Mode', Unsigned32),
    1687 : ('Emergency-Info', Grouped),
    1688 : ('V2X-Subscription-Data', Grouped),
    1689 : ('V2X-Permission', Unsigned32),
    1690 : ('PDN-Connection-Continuity', Unsigned32),
    1691 : ('eDRX-Cycle-Length', Grouped),
    1692 : ('eDRX-Cycle-Length-Value', OctetString),
    1693 : ('UE-PC5-AMBR', Unsigned32),
    1694 : ('MBSFN-Area', Grouped),
    1695 : ('MBSFN-Area-ID', Unsigned32),
    1696 : ('Carrier-Frequency', Unsigned32),
    1697 : ('RDS-Indicator', Enumerated),
    1698 : ('Service-Gap-Time', Unsigned32),
    1699 : ('Aerial-UE-Subscription-Information', Unsigned32),
    1700 : ('Broadcast-Location-Assistance-Data-Types', Unsigned64),
    1701 : ('Paging-Time-Window', Grouped),
    1702 : ('Operation-Mode', Unsigned32),
    1703 : ('Paging-Time-Window-Length', OctetString),
    1704 : ('Core-Network-Restrictions', Unsigned32),
    1705 : ('eDRX-Related-RAT', Grouped),
    1706 : ('Interworking-5GS-Indicator', Enumerated),
    2000 : ('SMS-Information', Grouped),
    2001 : ('Data-Coding-Scheme', Integer32),
    2002 : ('Destination-Interface', Grouped),
    2003 : ('Interface-Id', UTF8String),
    2004 : ('Interface-Port', UTF8String),
    2005 : ('Interface-Text', UTF8String),
    2006 : ('Interface-Type', Enumerated),
    2007 : ('SM-Message-Type', Enumerated),
    2008 : ('Originating-SCCP-Address', Address),
    2009 : ('Originator-Interface', Grouped),
    2010 : ('Recipient-SCCP-Address', Address),
    2011 : ('Reply-Path-Requested', Enumerated),
    2012 : ('SM-Discharge-Time', Time),
    2013 : ('SM-Protocol-ID', OctetString),
    2014 : ('SM-Status', OctetString),
    2015 : ('SM-User-Data-Header', OctetString),
    2016 : ('SMS-Node', Enumerated),
    2017 : ('SMSC-Address', Address),
    2018 : ('Client-Address', Address),
    2019 : ('Number-of-Messages-Sent', Unsigned32),
    2020 : ('Low-Balance-Indication', Enumerated),
    2021 : ('Remaining-Balance', Grouped),
    2022 : ('Refund-Information', OctetString),
    2023 : ('Carrier-Select-Routing-Information', UTF8String),
    2024 : ('Number-Portability-Routing-Information', UTF8String),
    2025 : ('PoC-Event-Type', Enumerated),
    2026 : ('Recipient-Info', Grouped),
    2027 : ('Originator-Received-Address', Grouped),
    2028 : ('Recipient-Received-Address', Grouped),
    2029 : ('SM-Service-Type', Enumerated),
    2030 : ('MMTel-Information', Grouped),
    2031 : ('MMTel-Sservice-Type', Unsigned32),
    2032 : ('Service-Mode', Unsigned32),
    2033 : ('Subscriber-Role', Enumerated),
    2034 : ('Number-Of-Diversions', Unsigned32),
    2035 : ('Associated-Party-Address', UTF8String),
    2036 : ('SDP-Type', Enumerated),
    2037 : ('Change-Condition', Integer32),
    2038 : ('Change-Time', Time),
    2039 : ('Diagnostics', Integer32),
    2040 : ('Service-Data-Container', Grouped),
    2041 : ('Start-Time', Time),
    2042 : ('Stop-Time', Time),
    2043 : ('Time-First-Usage', Time),
    2044 : ('Time-Last-Usage', Time),
    2045 : ('Time-Usage', Unsigned32),
    2046 : ('Traffic-Data-Volumes', Grouped),
    2047 : ('Serving-Node-Type', Enumerated),
    2048 : ('Supplementary-Service', Grouped),
    2049 : ('Participant-Action-Type', Enumerated),
    2050 : ('PDN-Connection-Charging-ID', Unsigned32),
    2051 : ('Dynamic-Address-Flag', Enumerated),
    2052 : ('Accumulated-Cost', Grouped),
    2053 : ('AoC-Cost-Information', Grouped),
    2054 : ('AoC-Information', Grouped),
    2055 : ('AoC-Request-Type', Enumerated),
    2056 : ('Current-Tariff', Grouped),
    2057 : ('Next-Tariff', Grouped),
    2058 : ('Rate-Element', Grouped),
    2059 : ('Scale-Factor', Grouped),
    2060 : ('Tariff-Information', Grouped),
    2061 : ('Unit-Cost', Grouped),
    2062 : ('Incremental-Cost', Grouped),
    2063 : ('Local-Sequence-Number', Unsigned32),
    2064 : ('Node-Id', UTF8String),
    2065 : ('SGW-Change', Enumerated),
    2066 : ('Charging-Characteristic-Selection-Mode', Enumerated),
    2067 : ('SGW-Address', Address),
    2068 : ('Dynamic-Address-Flag-Extension', Enumerated),
    2101 : ('Application-Server-ID', UTF8String),
    2102 : ('Application-Service-Type', Enumerated),
    2103 : ('Application-Session-ID', Unsigned32),
    2104 : ('Delivery-Status', UTF8String),
    2110 : ('IM-Information', Grouped),
    2111 : ('Number-Of-Messages-Successfully-Exploded', Unsigned32),
    2112 : ('Number-Of-Messages-Successfully-Sent', Unsigned32),
    2113 : ('Total-Number-Of-Messages-Exploded', Unsigned32),
    2114 : ('Total-Number-Of-Messages-Sent', Unsigned32),
    2115 : ('DCD-Information', Grouped),
    2116 : ('Content-ID', UTF8String),
    2117 : ('Content-provider-ID', UTF8String),
    2118 : ('Charge-Reason-Code', Enumerated),
    2200 : ('Subsession-Decision-Info', Grouped),
    2201 : ('Subsession-Enforcement-Info', Grouped),
    2202 : ('Subsession-Id', Unsigned32),
    2203 : ('Subsession-Operation', Enumerated),
    2204 : ('Multiple-BBERF-Action', Enumerated),
    2205 : ('UE-Local-IPv6-Prefix', OctetString),
    2206 : ('DRA-Deployment', Enumerated),
    2207 : ('PCRF-Address', DiameterIdentity),
    2208 : ('DRA-Binding', Enumerated),
    2301 : ('SIP-Request-Timestamp-Fraction', Unsigned32),
    2302 : ('SIP-Response-Timestamp-Fraction', Unsigned32),
    2303 : ('Online-Charging-Flag', Enumerated),
    2304 : ('CUG-Information', OctetString),
    2305 : ('Real-Time-Tariff-Information', Grouped),
    2306 : ('Tariff-XML', UTF8String),
    2307 : ('MBMSGW-Address', Address),
    2308 : ('IMSI-Unauthenticated-Flag', Enumerated),
    2309 : ('Account-Expiration', Time),
    2310 : ('AoC-Format', Enumerated),
    2311 : ('AoC-Service', Enumerated),
    2312 : ('AoC-Service-Obligatory-Type', Grouped),
    2313 : ('AoC-Service-Type', Enumerated),
    2314 : ('AoC-Subscription-Information', Grouped),
    2315 : ('Preferred-AoC-Currency', Unsigned32),
    2317 : ('CSG-Access-Mode', Enumerated),
    2318 : ('CSG-Membership-Indication', Enumerated),
    2319 : ('User-CSG-Information', Grouped),
    2320 : ('Outgoing-Session-Id', UTF8String),
    2321 : ('Initial-IMS-Charging-Identifier', UTF8String),
    2322 : ('IMS-Emergency-Indicator', Enumerated),
    2323 : ('MBMS-Charged-Party', Enumerated),
    2400 : ('LMSI', OctetString),
    2401 : ('Serving-Node', Grouped),
    2402 : ('MME-Name', DiameterIdentity),
    2403 : ('MSC-Number', OctetString),
    2404 : ('LCS-Capabilities-Sets', Unsigned32),
    2405 : ('GMLC-Address', Address),
    2406 : ('Additional-Serving-Node', Grouped),
    2407 : ('PPR-Address', Address),
    2408 : ('MME-Realm', DiameterIdentity),
    2409 : ('SGSN-Name', DiameterIdentity),
    2410 : ('SGSN-Realm', DiameterIdentity),
    2411 : ('RIA-Flags', Unsigned32),
    2500 : ('SLg-Location-Type', Enumerated),
    2501 : ('LCS-EPS-Client-Name', Grouped),
    2502 : ('LCS-Requestor-Name', Grouped),
    2503 : ('LCS-Priority', Unsigned32),
    2504 : ('LCS-QoS', Grouped),
    2505 : ('Horizontal-Accuracy', Unsigned32),
    2506 : ('Vertical-Accuracy', Unsigned32),
    2507 : ('Vertical-Requested', Enumerated),
    2508 : ('Velocity-Requested', Enumerated),
    2509 : ('Response-Time', Enumerated),
    2510 : ('Supported-GAD-Shapes', Unsigned32),
    2511 : ('LCS-Codeword', UTF8String),
    2512 : ('LCS-Privacy-Check', Enumerated),
    2513 : ('Accuracy-Fulfilment-Indicator', Enumerated),
    2514 : ('Age-Of-Location-Estimate', Unsigned32),
    2515 : ('Velocity-Estimate', OctetString),
    2516 : ('EUTRAN-Positioning-Data', OctetString),
    2517 : ('ECGI', OctetString),
    2518 : ('Location-Event', Enumerated),
    2519 : ('Pseudonym-Indicator', Enumerated),
    2520 : ('LCS-Service-Type-ID', Unsigned32),
    2521 : ('LCS-Privacy-Check-Non-Session', Grouped),
    2522 : ('LCS-Privacy-Check-Session', Grouped),
    2523 : ('LCS-QoS-Class', Enumerated),
    2524 : ('GERAN-Positioning-Info', Grouped),
    2525 : ('GERAN-Positioning-Data', OctetString),
    2526 : ('GERAN-GANSS-Positioning-Data', OctetString),
    2527 : ('UTRAN-Positioning-Info', Grouped),
    2528 : ('UTRAN-Positioning-Data', OctetString),
    2529 : ('UTRAN-GANSS-Positioning-Data', OctetString),
    2530 : ('LRR-Flags', Unsigned32),
    2531 : ('LCS-Reference-Number', OctetString),
    2532 : ('Deferred-Location-Type', Unsigned32),
    2533 : ('Area-Event-Info', Grouped),
    2534 : ('Area-Definition', Grouped),
    2535 : ('Area', Grouped),
    2536 : ('Area-Type', Unsigned32),
    2537 : ('Area-Identification', Grouped),
    2538 : ('Occurrence-Info', Enumerated),
    2539 : ('Interval-Time', Unsigned32),
    2540 : ('Periodic-LDR-Information', Grouped),
    2541 : ('Reporting-Amount', Unsigned32),
    2542 : ('Reporting-Interval', Unsigned32),
    2543 : ('Reporting-PLMN-List', Grouped),
    2544 : ('PLMN-ID-List', Grouped),
    2545 : ('PLR-Flags', Unsigned32),
    2546 : ('PLA-Flags', Unsigned32),
    2547 : ('Deferred-MT-LR-Data', Grouped),
    2548 : ('Termination-Cause', Unsigned32),
    2549 : ('LRA-Flags', Unsigned32),
    2550 : ('Periodic-Location-Support-Indicator', Enumerated),
    2551 : ('Prioritized-List-Indicator', Enumerated),
    2552 : ('ESMLC-Cell-Info', Grouped),
    2553 : ('Cell-Portion-ID', Unsigned32),
    2554 : ('1xRTT-RCID', OctetString),
    2555 : ('Delayed-Location-Reporting-Data', Grouped),
    2556 : ('Civic-Address', UTF8String),
    2557 : ('Barometric-Pressure', Unsigned32),
    2558 : ('UTRAN-Additional-Positioning-Data', OctetString),
    2559 : ('Motion-Event-Info', Grouped),
    2560 : ('Linear-Distance', Unsigned32),
    2561 : ('Maximum-Interval', Unsigned32),
    2562 : ('Sampling-Interval', Unsigned32),
    2563 : ('Reporting-Duration', Unsigned32),
    2564 : ('Reporting-Location-Requirements', Unsigned32),
    2565 : ('Additional-Area', Grouped),
    2601 : ('IMS-Application-Reference-Identifier', UTF8String),
    2602 : ('Low-Priority-Indicator', Enumerated),
    2603 : ('IP-Realm-Default-Indicator', Enumerated),
    2604 : ('Local-GW-Inserted-Indicator', Enumerated),
    2605 : ('Transcoder-Inserted-Indicator', Enumerated),
    2606 : ('PDP-Address-Prefix-Length', Unsigned32),
    2701 : ('Transit-IOI-List', UTF8String),
    2702 : ('Status-AS-Code', Enumerated),
    2703 : ('NNI-Information', Grouped),
    2704 : ('NNI-Type', Enumerated),
    2705 : ('Neighbour-Node-Address', Address),
    2706 : ('Relationship-Mode', Enumerated),
    2707 : ('Session-Direction', Enumerated),
    2708 : ('From-Address', UTF8String),
    2709 : ('Access-Transfer-Information', Grouped),
    2710 : ('Access-Transfer-Type', Enumerated),
    2711 : ('Related-IMS-Charging-Identifier', UTF8String),
    2712 : ('Related-IMS-Charging-Identifier-Node', Address),
    2713 : ('IMS-Visited-Network-Identifier', UTF8String),
    2714 : ('TWAN-User-Location-Info', Grouped),
    2716 : ('BSSID', UTF8String),
    2717 : ('TAD-Identifier', Enumerated),
    2801 : ('ADC-Revalidation-Time', Time),
    2802 : ('TDF-Application-Instance-Identifier', OctetString),
    2803 : ('HeNB-BBF-FQDN', UTF8String),
    2804 : ('HeNB-Local-IP-Address', Address),
    2805 : ('UE-Local-IP-Address', Address),
    2806 : ('UDP-Source-Port', Unsigned32),
    2807 : ('CS-Service-Qos-Request-Identifier', OctetString),
    2808 : ('CS-Service-QoS-Request-Operation', Enumerated),
    2809 : ('Mute-Notification', Enumerated),
    2810 : ('Monitoring-Time', Time),
    2811 : ('AN-GW-Status', Enumerated),
    2812 : ('User-Location-Info-Time', Grouped),
    2813 : ('CS-Service-Resource-Report', Time),
    2814 : ('CS-Service-Resource-Failure-Cause', Enumerated),
    2815 : ('CS-Service-Resource-Result-Operation', Enumerated),
    2816 : ('Default-QoS-Information', Grouped),
    2817 : ('Default-QoS-Name', UTF8String),
    2818 : ('Conditional-APN-Aggregate-Max-Bitrate', Grouped),
    2819 : ('RAN-NAS-Release-Cause', OctetString),
    2820 : ('Presence-Reporting-Area-Elements-List', OctetString),
    2821 : ('Presence-Reporting-Area-Identifier', OctetString),
    2822 : ('Presence-Reporting-Area-Information', Grouped),
    2823 : ('Presence-Reporting-Area-Status', Unsigned32),
    2824 : ('NetLoc-Access-Support', Unsigned32),
    2825 : ('Fixed-User-Location-Info', Grouped),
    2826 : ('PCSCF-Restoration-Indication', Unsigned32),
    2827 : ('IP-CAN-Session-Charging-Scope', Enumerated),
    2828 : ('Monitoring-Flags', Unsigned32),
    2829 : ('Default-Access', Enumerated),
    2830 : ('NBIFOM-Mode', Enumerated),
    2831 : ('NBIFOM-Support', Enumerated),
    2832 : ('RAN-Rule-Support', Unsigned32),
    2833 : ('Access-Availability-Change-Reason', Unsigned32),
    2834 : ('Routing-Rule-Failure-Code', Unsigned32),
    2835 : ('Routing-Rule-Report', Grouped),
    2836 : ('Traffic-Steering-Policy-Identifier-DL', OctetString),
    2837 : ('Traffic-Steering-Policy-Identifier-UL', OctetString),
    2838 : ('Request-Type', Unsigned32),
    2839 : ('Execution-Time', Time),
    2840 : ('Conditional-Policy-Information', Grouped),
    2841 : ('Resource-Release-Notification', Enumerated),
    2842 : ('Removal-Of-Access', Enumerated),
    2843 : ('TCP-Source-Port', Unsigned32),
    2844 : ('Default-Bearer-Indication', Enumerated),
    2845 : ('PRA-Install', Grouped),
    2846 : ('PRA-Remove', Grouped),
    2847 : ('3GPP-PS-Data-Off-Status', Enumerated),
    2848 : ('Extended-APN-AMBR-DL', Unsigned32),
    2849 : ('Extended-APN-AMBR-UL', Unsigned32),
    2850 : ('Extended-GBR-DL', Unsigned32),
    2851 : ('Extended-GBR-UL', Unsigned32),
    2852 : ('Max-PLR-DL', Unsigned32),
    2853 : ('Max-PLR-UL', Unsigned32),
    2854 : ('UE-Status', Unsigned32),
    2901 : ('Policy-Counter-Identifier', Grouped),
    2902 : ('Policy-Counter-Status', UTF8String),
    2903 : ('Policy-Counter-Status-Report', Grouped),
    2904 : ('SL-Request-Type', Enumerated),
    2905 : ('Pending-Policy-Counter-Information', Grouped),
    2906 : ('Pending-Policy-Counter-Change-Time', Time),
    2907 : ('SN-Request-Type', Unsigned32),
    3001 : ('Device-Action', Grouped),
    3002 : ('Device-Notification', Grouped),
    3003 : ('Trigger-Data', Grouped),
    3004 : ('Payload', OctetString),
    3005 : ('Action-Type', Enumerated),
    3006 : ('Priority-Indication', Enumerated),
    3007 : ('Reference-Number', Unsigned32),
    3008 : ('Request-Status', Enumerated),
    3009 : ('Delivery-Outcome', Enumerated),
    3010 : ('Application-Port-Identifier', Unsigned32),
    3011 : ('Old-Reference-Number', Unsigned32),
    3012 : ('Feature-Supported-In-Final-Target', Unsigned32),
    3100 : ('IP-SM-GW-Number', OctetString),
    3101 : ('IP-SM-GW-Name', DiameterIdentity),
    3102 : ('User-Identifier', Grouped),
    3103 : ('Service-ID', Enumerated),
    3104 : ('SCS-Identity', OctetString),
    3105 : ('Service-Parameters', Grouped),
    3106 : ('T4-Parameters', Grouped),
    3107 : ('Service-Data', Grouped),
    3108 : ('T4-Data', Grouped),
    3109 : ('HSS-Cause', Unsigned32),
    3110 : ('SIR-Flags', Unsigned32),
    3111 : ('External-Identifier', UTF8String),
    3112 : ('IP-SM-GW-Realm', DiameterIdentity),
    3113 : ('AESE-Communication-Pattern', Grouped),
    3114 : ('Communication-Pattern-Set', Grouped),
    3115 : ('Periodic-Communication-Indicator', Unsigned32),
    3116 : ('Communication-Duration-Time', Unsigned32),
    3117 : ('Periodic-Time', Unsigned32),
    3118 : ('Scheduled-Communication-Time', Grouped),
    3119 : ('Stationary-Indication', Unsigned32),
    3120 : ('AESE-Communication-Pattern-Config-Status', Grouped),
    3121 : ('AESE-Error-Report', Grouped),
    3122 : ('Monitoring-Event-Configuration', Grouped),
    3123 : ('Monitoring-Event-Report', Grouped),
    3124 : ('SCEF-Reference-ID', Unsigned32),
    3125 : ('SCEF-ID', DiameterIdentity),
    3126 : ('SCEF-Reference-ID-for-Deletion', Unsigned32),
    3127 : ('Monitoring-Type', Unsigned32),
    3128 : ('Maximum-Number-of-Reports', Unsigned32),
    3129 : ('UE-Reachability-Configuration', Grouped),
    3130 : ('Monitoring-Duration', Time),
    3131 : ('Maximum-Detection-Time', Unsigned32),
    3132 : ('Reachability-Type', Unsigned32),
    3133 : ('Maximum-Latency', Unsigned32),
    3134 : ('Maximum-Response-Time', Unsigned32),
    3135 : ('Location-Information-Configuration', Grouped),
    3136 : ('MONTE-Location-Type', Unsigned32),
    3137 : ('Accuracy', Unsigned32),
    3138 : ('Association-Type', Unsigned32),
    3139 : ('Roaming-Information', Unsigned32),
    3140 : ('Reachability-Information', Unsigned32),
    3141 : ('IMEI-Change', Unsigned32),
    3142 : ('Monitoring-Event-Config-Status', Grouped),
    3143 : ('Supported-Services', Grouped),
    3144 : ('Supported-Monitoring-Event', Unsigned64),
    3145 : ('CIR-Flags', Unsigned32),
    3146 : ('Service-Result', Grouped),
    3147 : ('Service-Result-Code', Unsigned32),
    3148 : ('Reference-ID-Validity-Time', Time),
    3149 : ('Event-handling', Unsigned32),
    3150 : ('NIDD-Authorization-Request', Grouped),
    3151 : ('NIDD-Authorization-Response', Grouped),
    3152 : ('Service-Report', Grouped),
    3153 : ('Node-Type', Unsigned32),
    3154 : ('S6t-HSS-Cause', Unsigned32),
    3155 : ('Enhanced-Coverage-Restriction', Grouped),
    3156 : ('Enhanced-Coverage-Restriction-Data', Grouped),
    3157 : ('Restricted-PLMN-List', Grouped),
    3158 : ('Allowed-PLMN-List', Grouped),
    3159 : ('Requested-Validity-Time', Time),
    3160 : ('Granted-Validity-Time', Time),
    3161 : ('NIDD-Authorization-Update', Grouped),
    3162 : ('Loss-Of-Connectivity-Reason', Unsigned32),
    3163 : ('Group-Reporting-Guard-Timer', Unsigned32),
    3164 : ('CIA-Flags', Unsigned32),
    3165 : ('Group-Monitoring-Event-Report', Grouped),
    3166 : ('Group-Monitoring-Event-Report-Item', Grouped),
    3167 : ('RIR-Flags', Unsigned32),
    3168 : ('Type-Of-External-Identifier', Unsigned32),
    3169 : ('APN-Validity-Time', Grouped),
    3170 : ('Suggested-Network-Configuration', Grouped),
    3171 : ('Monitoring-Event-Report-Status', Grouped),
    3172 : ('PLMN-ID-Requested', Enumerated),
    3173 : ('AdditionalIdentifiers', Grouped),
    3174 : ('NIR-Flags', Unsigned32),
    3175 : ('Reporting-Time-Stamp', Time),
    3176 : ('Group-User-Identifier', Grouped),
    3177 : ('NIA-Flags', Unsigned32),
    3178 : ('MTC-Provider-Info', Grouped),
    3179 : ('MTC-Provider-ID', UTF8String),
    3200 : ('SM-Delivery-Outcome-T4', Enumerated),
    3201 : ('Absent-Subscriber-Diagnostic-T4', Enumerated),
    3202 : ('Trigger-Action', Unsigned32),
    3203 : ('MTC-Error-Diagnostic', Unsigned32),
    3300 : ('SC-Address', OctetString),
    3301 : ('SM-RP-UI', OctetString),
    3302 : ('TFR-Flags', Unsigned32),
    3303 : ('SM-Delivery-Failure-Cause', Grouped),
    3304 : ('SM-Enumerated-Delivery-Failure-Cause', Enumerated),
    3305 : ('SM-Diagnostic-Info', OctetString),
    3306 : ('SM-Delivery-Timer', Unsigned32),
    3307 : ('SM-Delivery-Start-Time', Time),
    3308 : ('SM-RP-MTI', Enumerated),
    3309 : ('SM-RP-SMEA', OctetString),
    3310 : ('SRR-Flags', Unsigned32),
    3311 : ('SM-Delivery-Not-Intended', Enumerated),
    3312 : ('MWD-Status', Unsigned32),
    3313 : ('MME-Absent-User-Subscriber-Diagnostic-SM', Unsigned32),
    3314 : ('MSC-Absent-User-Subscriber-Diagnostic-SM', Unsigned32),
    3315 : ('SGSN-Absent-User-Subscriber-DiagnosticSM', Unsigned32),
    3316 : ('SM-Delivery-Outcome', Grouped),
    3317 : ('MME-SM-Delivery-Outcome', Grouped),
    3318 : ('MSC-SM-Delivery-Outcome', Grouped),
    3319 : ('SGSN-SM-Delivery-Outcome', Grouped),
    3320 : ('IP-SM-GW-SM-Delivery-Outcome', Grouped),
    3321 : ('SM-Delivery-Cause', Enumerated),
    3322 : ('Absent-User-Diagnostic-SM', Unsigned32),
    3323 : ('RDR-Flags', Unsigned32),
    3324 : ('SMSMI-Correlation-ID', Unsigned32),
    3325 : ('HSS-ID', UTF8String),
    3326 : ('Originating-SIP-URI', UTF8String),
    3327 : ('Destination-SIP-URI', UTF8String),
    3328 : ('OFR-Flags', Unsigned32),
    3329 : ('Maximum-UE-Availability-Time', Time),
    3330 : ('Maximum-Retransmission-Time', Time),
    3331 : ('Requested-Retransmission-Time', Time),
    3332 : ('SMS-GMSC-Address', OctetString),
    3333 : ('SMS-GMSC-Alert-Event', Unsigned32),
    3401 : ('Reason-Header', UTF8String),
    3402 : ('Instance-Id', UTF8String),
    3403 : ('Route-Header-Received', UTF8String),
    3404 : ('Route-Header-Transmitted', UTF8String),
    3405 : ('SM-Device-Trigger-Information', Grouped),
    3406 : ('MTC-IWF-Address', Address),
    3407 : ('SM-Device-Trigger-Indicator', Enumerated),
    3408 : ('SM-Sequence-Number', Unsigned32),
    3409 : ('SMS-Result', Unsigned32),
    3410 : ('VCS-Information', Grouped),
    3411 : ('Basic-Service-Code', Grouped),
    3412 : ('Bearer-Capability', OctetString),
    3413 : ('Teleservice', OctetString),
    3414 : ('ISUP-Location-Number', OctetString),
    3415 : ('Forwarding-Pending', Enumerated),
    3416 : ('ISUP-Release-Cause', Unsigned32),
    3417 : ('MSC-Address', OctetString),
    3418 : ('Network-Call-Reference-Number', OctetString),
    3419 : ('Start-of-Charging', Time),
    3420 : ('VLR-Number', OctetString),
    3421 : ('CN-Operator-Selection-Entity', Enumerated),
    3422 : ('ISUP-Cause-Diagnostics', OctetString),
    3423 : ('ISUP-Cause-Location', Unsigned32),
    3424 : ('ISUP-Cause-Value', Unsigned32),
    3425 : ('EPDG-Address', Address),
    3426 : ('Announcing-UE-HPLMN-Identifier', UTF8String),
    3427 : ('Announcing-UE-VPLMN-Identifier', UTF8String),
    3428 : ('Coverage-Status', Enumerated),
    3429 : ('Layer-2-Group-ID', OctetString),
    3430 : ('Monitored-PLMN-Identifier', UTF8String),
    3431 : ('Monitoring-UE-HPLMN-Identifier', UTF8String),
    3432 : ('Monitoring-UE-Identifier', UTF8String),
    3433 : ('Monitoring-UE-VPLMN-Identifier', UTF8String),
    3434 : ('ProSe-PC3-Control-Protocol-Cause', Integer32),
    3435 : ('ProSe-PC3-EPC-Control-Protocol-Cause', Integer32),
    3436 : ('Requested-PLMN-Identifier', UTF8String),
    3437 : ('Requestor-PLMN-Identifier', UTF8String),
    3438 : ('Role-Of-ProSe-Function', Enumerated),
    3439 : ('Usage-Information-Report-Sequence-Number', Integer32),
    3440 : ('ProSe-3rd-Party-Application-ID', UTF8String),
    3441 : ('ProSe-Direct-Communication-Transmission-Data-Container', Grouped),
    3442 : ('ProSe-Direct-Discovery-Model', Enumerated),
    3443 : ('ProSe-Event-Type', Enumerated),
    3444 : ('ProSe-Function-IP-Address', Address),
    3445 : ('ProSe-Functionality', Enumerated),
    3446 : ('ProSe-Group-IP-Multicast-Addresses', Address),
    3447 : ('ProSe-Information', Grouped),
    3448 : ('ProSe-Range-Class', Enumerated),
    3449 : ('ProSe-Reason-For-Cancellation', Enumerated),
    3450 : ('ProSe-Request-Timestamp', Time),
    3451 : ('ProSe-Role-Of-UE', Enumerated),
    3452 : ('ProSe-Source-IP-Address', Address),
    3453 : ('ProSe-UE-ID', OctetString),
    3454 : ('Proximity-Alert-Indication', Enumerated),
    3455 : ('Proximity-Alert-Timestamp', Time),
    3456 : ('Proximity-Cancellation-Timestamp', Time),
    3457 : ('ProSe-Function-PLMN-Identifier', UTF8String),
    3458 : ('Application-Specific-Data', OctetString),
    3459 : ('Coverage-Info', Grouped),
    3460 : ('Location-Info', Grouped),
    3461 : ('ProSe-Direct-Communication-Reception-Data-Container', Grouped),
    3462 : ('Radio-Frequency', OctetString),
    3463 : ('Radio-Parameter-Set-Info', Grouped),
    3464 : ('Radio-Parameter-Set-Values', OctetString),
    3465 : ('Radio-Resources-Indicator', Integer32),
    3466 : ('Time-First-Reception', Time),
    3467 : ('Time-First-Transmission', Time),
    3468 : ('Transmitter-Info', Grouped),
    3500 : ('BMSC‑Address', Address),
    3501 : ('BMSC‑Port', Unsigned32),
    3502 : ('MBMS‑Bearer‑Event', Unsigned32),
    3503 : ('MBMS‑Bearer‑Event‑Notification', Grouped),
    3504 : ('MBMS‑Bearer‑Request', Grouped),
    3505 : ('MBMS‑Bearer‑Response', Grouped),
    3506 : ('MBMS‑Bearer‑Result', Unsigned32),
    3507 : ('MBMS‑Start‑Time', Time),
    3508 : ('Radio‑Frequency', Unsigned32),
    3509 : ('TMGI‑Allocation‑Request', Grouped),
    3510 : ('TMGI‑Allocation‑Response', Grouped),
    3511 : ('TMGI‑Allocation‑Result', Unsigned32),
    3512 : ('TMGI‑Deallocation‑Request', Grouped),
    3513 : ('TMGI‑Deallocation‑Response', Grouped),
    3514 : ('TMGI‑Deallocation‑Result', Unsigned32),
    3515 : ('TMGI‑Expiry', Grouped),
    3516 : ('TMGI‑Number', Unsigned32),
    3517 : ('MB2U‑Security', Unsigned32),
    3518 : ('Local-M1-Information', Grouped),
    3519 : ('Local-MB2-U-Information', Grouped),
    3520 : ('MBMS-eNB-IP-Multicast-Address', Address),
    3521 : ('MBMS-eNB-IPv6-Multicast-Address', Address),
    3522 : ('MBMS-GW-SSM-IP-Address', Address),
    3523 : ('MBMS-GW-SSM-IPv6-Address', Address),
    3524 : ('Common-Tunnel-Endpoint-Identifier', OctetString),
    3525 : ('FEC-Request', OctetString),
    3526 : ('ROHC‑Request', Grouped),
    3527 : ('ROHC-Full-Header-Periodicity', Float32),
    3528 : ('ROHC‑Profile', Unsigned32),
    3529 : ('Userplane-Protocol‑Result', Grouped),
    3530 : ('ROHC‑Result', Unsigned32),
    3531 : ('FEC‑Result', Unsigned32),
    3532 : ('ROHC-Max-CID', Unsigned32),
    3600 : ('Origin-App-Layer-User-Id', UTF8String),
    3601 : ('Target-App-Layer-User-Id', UTF8String),
    3602 : ('ProSe-Function-ID', OctetString),
    3603 : ('ProSe-Request-Type', Unsigned32),
    3604 : ('PDUID', OctetString),
    3605 : ('Application-Data', UTF8String),
    3606 : ('Allowed-Suffixes-Number', Unsigned32),
    3607 : ('Monitor-Target', Grouped),
    3608 : ('ProSe-Code-Suffix-Mask', Grouped),
    3609 : ('Suffix-Code', OctetString),
    3610 : ('Suffix-Mask', OctetString),
    3611 : ('Banned-User-Target', Grouped),
    3612 : ('Metadata-Indicator', Unsigned32),
    3701 : ('ProSe-Subscription-Data', Grouped),
    3702 : ('ProSe-Permission', Unsigned32),
    3703 : ('ProSe-Allowed-PLMN', Grouped),
    3704 : ('ProSe-Direct-Allowed', Unsigned32),
    3705 : ('UPR-Flags', Unsigned32),
    3706 : ('PNR-Flags', Unsigned32),
    3707 : ('ProSe-Initial-Location-Information', Grouped),
    3708 : ('Authorized-Discovery-Range', Unsigned32),
    3801 : ('App-Layer-User-Id', UTF8String),
    3802 : ('Assistance-info', Grouped),
    3803 : ('Assistance-Info-Validity-Timer', Unsigned32),
    3804 : ('Discovery-Type', Unsigned32),
    3805 : ('Filter-Id', OctetString),
    3806 : ('MAC-Address', UTF8String),
    3807 : ('Match-Report', Grouped),
    3808 : ('Operating-Channel', Unsigned32),
    3809 : ('P2P-Features', Unsigned32),
    3810 : ('ProSe-App-Code', UTF8String),
    3811 : ('ProSe-App-Id', UTF8String),
    3812 : ('ProSe-App-Mask', UTF8String),
    3813 : ('ProSe-Discovery-Filter', Grouped),
    3814 : ('PRR-Flags', Unsigned32),
    3815 : ('ProSe-Validity-Timer', Unsigned32),
    3816 : ('Requesting-EPUID', UTF8String),
    3817 : ('Targeted-EPUID', UTF8String),
    3818 : ('Time-Window', Unsigned32),
    3819 : ('WiFi-P2P-Assistance-Info', Grouped),
    3820 : ('WLAN-Assistance-Info', Grouped),
    3821 : ('WLAN-Link-Layer-Id', OctetString),
    3822 : ('WLAN-Link-Layer-Id-List', Grouped),
    3823 : ('Location-Update-Trigger', Grouped),
    3824 : ('Location-Update-Event-Type', Unsigned32),
    3825 : ('Change-Of-Area-Type', Grouped),
    3826 : ('Location-Update-Event-Trigger', Unsigned32),
    3827 : ('Report-Cardinality', Enumerated),
    3828 : ('Minimum-Interval-Time', Unsigned32),
    3829 : ('Periodic-Location-Type', Grouped),
    3830 : ('Location-Report-Interval-Time', Unsigned32),
    3831 : ('Total-Number-Of-Reports', Unsigned32),
    3832 : ('Validity-Time-Announce', Unsigned32),
    3833 : ('Validity-Time-Monitor', Unsigned32),
    3834 : ('Validity-Time-Communication', Unsigned32),
    3835 : ('ProSe-App-Code-Info', Grouped),
    3836 : ('MIC', OctetString),
    3837 : ('UTC-based-Counter', Unsigned32),
    3838 : ('ProSe-Match-Refresh-Timer', Unsigned32),
    3839 : ('ProSe-Metadata-Index-Mask', OctetString),
    3840 : ('App-Identifier', Grouped),
    3841 : ('OS-ID', OctetString),
    3842 : ('OS-App-ID', UTF8String),
    3843 : ('Requesting-RPAUID', UTF8String),
    3844 : ('Target-RPAUID', UTF8String),
    3845 : ('Target-PDUID', OctetString),
    3846 : ('ProSe-Restricted-Code', OctetString),
    3847 : ('ProSe-Restricted-Code-Suffix-Range', OctetString),
    3848 : ('Beginning-Suffix', OctetString),
    3849 : ('Ending-Suffix', OctetString),
    3850 : ('Discovery-Entry-ID', Unsigned32),
    3851 : ('Match-Timestamp', Time),
    3852 : ('PMR-Flags', Unsigned32),
    3853 : ('ProSe-Application-Metadata', UTF8String),
    3854 : ('Discovery-Auth-Request', Grouped),
    3855 : ('Discovery-Auth-Response', Grouped),
    3856 : ('Match-Request', Grouped),
    3857 : ('Match-Report-Info', Grouped),
    3858 : ('Banned-RPAUID', UTF8String),
    3859 : ('Banned-PDUID', OctetString),
    3860 : ('Code-Receiving-Security-Material', Grouped),
    3861 : ('Code-Sending-Security-Material', Grouped),
    3862 : ('DUSK', OctetString),
    3863 : ('DUIK', OctetString),
    3864 : ('DUCK', OctetString),
    3865 : ('MIC-Check-indicator', Unsigned32),
    3866 : ('Encrypted-Bitmask', OctetString),
    3867 : ('ProSe-App-Code-Suffix-Range', OctetString),
    3868 : ('PC5-tech', OctetString),
    3901 : ('Enhanced-Diagnostics', Grouped),
    3902 : ('Inter-UE-Transfer', Enumerated),
    3903 : ('TWAG-Address', Address),
    3904 : ('Announcement-Information', Grouped),
    3905 : ('Announcement-Identifier', Unsigned32),
    3906 : ('Announcement-Order', Unsigned32),
    3907 : ('Variable-Part', Grouped),
    3908 : ('Variable-Part-Order', Unsigned32),
    3909 : ('Variable-Part-Type', Unsigned32),
    3910 : ('Variable-Part-Value', UTF8String),
    3911 : ('Time-Indicator', Unsigned32),
    3912 : ('Quota-Indicator', Enumerated),
    3913 : ('Play-Alternative', Enumerated),
    3914 : ('Language', UTF8String),
    3915 : ('Privacy-Indicator', Enumerated),
    3916 : ('Called-Identity', UTF8String),
    3917 : ('Called-Identity-Change', Grouped),
    3918 : ('UWAN-User-Location-Info', Grouped),
    3919 : ('Monitoring-Event-Configuration-Activity', Integer32),
    3920 : ('Monitoring-Event-Report-Data', Grouped),
    3921 : ('Monitoring-Event-Information', Grouped),
    3922 : ('Monitoring-Event-Functionality', Integer32),
    3923 : ('Monitoring-Event-Report-Number', Unsigned32),
    3924 : ('Cellular-Network-Information', OctetString),
    3925 : ('Related-Change-Condition-Information', Grouped),
    3926 : ('Related-Trigger', Grouped),
    3927 : ('CPDT-Information', Grouped),
    3928 : ('NIDD-Submission', Grouped),
    3929 : ('Serving-Node-Identity', DiameterIdentity),
    3930 : ('CP-CIoT-EPS-Optimisation-Indicator', Enumerated),
    3931 : ('SGi-PtP-Tunnelling-Method', Enumerated),
    3932 : ('UNI-PDU-CP-Only-Flag', Enumerated),
    3933 : ('APN-Rate-Control', Grouped),
    3934 : ('APN-Rate-Control-Downlink', Grouped),
    3935 : ('APN-Rate-Control-Uplink', Grouped),
    3936 : ('Additional-Exception-Reports', Enumerated),
    3937 : ('Rate-Control-Max-Message-Size', Unsigned32),
    3938 : ('Rate-Control-Max-Rate', Unsigned32),
    3939 : ('Rate-Control-Time-Unit', Unsigned32),
    3940 : ('SCS-AS-Address', Grouped),
    3941 : ('SCS-Address', Address),
    3942 : ('SCS-Realm', DiameterIdentity),
    4000 : ('Aggregated-Congestion-Info', Grouped),
    4001 : ('Aggregated-RUCI-Report', Grouped),
    4002 : ('Congestion-Level-Definition', Grouped),
    4003 : ('Congestion-Level-Range', Unsigned32),
    4004 : ('Congestion-Level-Set-Id', Unsigned32),
    4005 : ('Congestion-Level-Value', Unsigned32),
    4006 : ('Congestion-Location-Id', Grouped),
    4007 : ('Conditional-Restriction', Unsigned32),
    4008 : ('eNodeB-Id', OctetString),
    4009 : ('IMSI-List', OctetString),
    4010 : ('RCAF-Id', DiameterIdentity),
    4011 : ('Reporting-Restriction', Unsigned32),
    4012 : ('RUCI-Action', Unsigned32),
    4013 : ('Extended-eNodeB-ID', OctetString),
    4101 : ('Network-Congestion-Area-Report', Grouped),
    4102 : ('Ns-Request-Type', Unsigned32),
    4201 : ('Network-Area-Info-List', OctetString),
    4202 : ('Reference-Id', OctetString),
    4203 : ('Transfer-Request-Type', Unsigned32),
    4204 : ('Time-Window', Grouped),
    4205 : ('Transfer-End-Time', Time),
    4206 : ('Transfer-Start-Time', Time),
    4207 : ('Transfer-Policy', Grouped),
    4208 : ('Transfer-Policy-Id', Unsigned32),
    4209 : ('Number-of-UEs', Unsigned32),
    4210 : ('Access-Network-Reports', Grouped),
    4211 : ('Event-Configuration', Grouped),
    4212 : ('Event-Configuration-State', Unsigned32),
    4213 : ('Event-Configuration-Status', Grouped),
    4214 : ('Event-Reporting-Results', Unsigned32),
    4215 : ('Event-Reports', Grouped),
    4216 : ('Extended-SCEF-Reference-ID', Unsigned64),
    4300 : ('Communication-Failure-Information', Grouped),
    4301 : ('Cause-Type', Unsigned32),
    4302 : ('S1AP-Cause', Unsigned32),
    4303 : ('RANAP-Cause', Unsigned32),
    4304 : ('GMM-Cause', Unsigned32),
    4305 : ('SM-Cause', Unsigned32),
    4306 : ('Number-of-UE-Per-Location-Configuration', Grouped),
    4307 : ('Number-of-UE-Per-Location-Report', Grouped),
    4308 : ('UE-Count', Unsigned32),
    4309 : ('BSSGP-Cause', Unsigned32),
    4310 : ('Serving-PLMN-Rate-Control', Grouped),
    4311 : ('Uplink-Rate-Limit', Unsigned32),
    4312 : ('Downlink-Rate-Limit', Unsigned32),
    4313 : ('Extended-PCO', OctetString),
    4314 : ('Connection-Action', Unsigned32),
    4315 : ('Non-IP-Data', OctetString),
    4316 : ('SCEF-Wait-Time', Time),
    4317 : ('CMR-Flags', Unsigned32),
    4318 : ('RRC-Cause-Counter', Grouped),
    4319 : ('Counter-Value', Unsigned32),
    4320 : ('RRC-Counter-Timestamp', Time),
    4321 : ('TDA-Flags', Unsigned32),
    4322 : ('Idle-Status-Indication', Grouped),
    4323 : ('Idle-Status-Timestamp', Time),
    4324 : ('Active-Time', Unsigned32),
    4400 : ('Charging-Per-IP-CAN-Session-Indicator', Enumerated),
    4401 : ('Access-Network-Info-Change', Enumerated),
    4402 : ('Discoveree-UE-HPLMN-Identifier', UTF8String),
    4403 : ('Discoveree-UE-VPLMN-Identifier', UTF8String),
    4404 : ('Discoverer-UE-HPLMN-Identifier', UTF8String),
    4405 : ('Discoverer-UE-VPLMN-Identifier', UTF8String),
    4406 : ('3GPP-PS-Data-Off-Status', Enumerated),
    4407 : ('Unused-Quota-Timer', Unsigned32),
    4408 : ('Announcing-PLMN-ID', UTF8String),
    4409 : ('ProSe-UE-to-Network-Relay-UE-ID', OctetString),
    4410 : ('ProSe-Target-Layer-2-ID', OctetString),
    4411 : ('Relay-IP-address', Address),
    4412 : ('Target-IP-Address', Address),
    4413 : ('FE-Identifier-List', UTF8String),
    4500 : ('MCPTT-ID', UTF8String),
    4501 : ('Data-Identification', Grouped),
    4502 : ('Data-Identification-Prefix', Unsigned32),
    4503 : ('Data-Identification-Flags', Unsigned64),
    4504 : ('DPR-Flags', Unsigned32),
    4505 : ('DPA-Flags', Unsigned32),
    4506 : ('DUR-Flags', Unsigned32),
    4507 : ('DUA-Flags', Unsigned32),
    4508 : ('NDR-Flags', Unsigned32),
    4509 : ('NDA-Flags', Unsigned32),
    4510 : ('User-Data-Id', Unsigned32),
    4511 : ('MC-Service-User-Profile-Data', Grouped),
    4512 : ('Sequence-Number', Unsigned32),
    4513 : ('MCVideo-ID', UTF8String),
    4514 : ('MC-Data-ID', UTF8String),
    4600 : ('V2X-PC5-Allowed-PLMN', Grouped),
    4601 : ('V2X-Update-Flags', Unsigned32),
    4602 : ('V2X-Notify-Flags', Unsigned32),
    4700 : ('V2X-Authorization-Data', Grouped),
    4701 : ('V2X-Permission-in-VPLMN', Unsigned32),
    4702 : ('V2X-Application-Server', Grouped),
    }


# extend existing dictionnary and LUT

FMT_LUT_TS29230     = dict(FMT_LUT_IETF)
AVP3GPPCodes_dict   = dict(AVPCodes_dict)

for code, (name, fmt) in TGPPAVPInfos_dict.items():
    FMT_LUT_TS29230[code]   = fmt
    AVP3GPPCodes_dict[code] = name


# this is a crappy hack to update the dict for all AVPHdr with the 3GPP dict one
AVPHdr._GEN[0]._dic     = AVP3GPPCodes_dict
AVP3GPP._GEN[0][0]._dic = AVP3GPPCodes_dict
Grouped._GEN[0][0]._dic = AVP3GPPCodes_dict


#------------------------------------------------------------------------------#
# 3GPP-augmented Diameter
#------------------------------------------------------------------------------#

AVP3GPP.FMT_LUT = FMT_LUT_TS29230


class Diameter3GPP(DiameterGeneric):
    _GEN = (
        DiameterHdr(),
        Sequence('AVPs', GEN=AVP3GPP(), hier=1)
        )


