package main

import (
	"fmt"
	"strings"
	"time"
)

func (d *Descriptor) clientMakeAPIBase(sb *strings.Builder) {
	fmt.Fprintf(sb, "&%s{\n", d.APIStructName())
	for _, field := range apiFields {
		if field.ifLogin || field.ifTemplate {
			continue
		}
		fmt.Fprintf(sb, "\t%s: c.%s,\n", field.name, field.name)
	}
	fmt.Fprint(sb, "}")
}

func (d *Descriptor) clientMakeAPI(sb *strings.Builder) {
	if d.RequiresLogin && d.CachePolicy != CacheNone {
		panic("we don't support requiresLogin with caching")
	}
	if d.RequiresLogin {
		fmt.Fprintf(sb, "&%s{\n", d.WithLoginAPIStructName())
		fmt.Fprint(sb, "\tAPI:")
		d.clientMakeAPIBase(sb)
		fmt.Fprint(sb, ",\n")
		fmt.Fprint(sb, "\tJSONCodec: c.JSONCodec,\n")
		fmt.Fprint(sb, "\tKVStore: c.KVStore,\n")
		fmt.Fprint(sb, "\tRegisterAPI: &simpleRegisterAPI{\n")
		for _, field := range apiFields {
			if field.ifLogin || field.ifTemplate {
				continue
			}
			fmt.Fprintf(sb, "\t%s: c.%s,\n", field.name, field.name)
		}
		fmt.Fprint(sb, "\t},\n")
		fmt.Fprint(sb, "\tLoginAPI: &simpleLoginAPI{\n")
		for _, field := range apiFields {
			if field.ifLogin || field.ifTemplate {
				continue
			}
			fmt.Fprintf(sb, "\t%s: c.%s,\n", field.name, field.name)
		}
		fmt.Fprint(sb, "\t},\n")
		fmt.Fprint(sb, "}\n")
		return
	}
	if d.CachePolicy != CacheNone {
		fmt.Fprintf(sb, "&%s{\n", d.WithCacheAPIStructName())
		fmt.Fprint(sb, "\tAPI:")
		d.clientMakeAPIBase(sb)
		fmt.Fprint(sb, ",\n")
		fmt.Fprint(sb, "\tGobCodec: c.GobCodec,\n")
		fmt.Fprint(sb, "\tKVStore: c.KVStore,\n")
		fmt.Fprint(sb, "}\n")
		return
	}
	d.clientMakeAPIBase(sb)
	fmt.Fprint(sb, "\n")
}

func (d *Descriptor) genClientNewCaller(sb *strings.Builder) {
	fmt.Fprintf(sb, "func (c *Client) new%sCaller() ", d.Name)
	fmt.Fprintf(sb, "%s {\n", d.CallerInterfaceName())
	fmt.Fprint(sb, "\treturn ")
	d.clientMakeAPI(sb)
	fmt.Fprint(sb, "}\n\n")
}

func (d *Descriptor) genClientCall(sb *strings.Builder) {
	fmt.Fprintf(sb, "// %s calls the %s API.\n", d.Name, d.Name)
	fmt.Fprintf(sb, "func (c *Client) %s(\n", d.Name)
	fmt.Fprintf(sb, "ctx context.Context, req %s,\n) ", d.RequestTypeName())
	fmt.Fprintf(sb, "(%s, error) {\n", d.ResponseTypeName())
	fmt.Fprintf(sb, "\tapi := c.new%sCaller()\n", d.Name)
	fmt.Fprint(sb, "\treturn api.Call(ctx, req)\n")
	fmt.Fprint(sb, "}\n\n")
}

// GenClientCallGo generates clientcall.go.
func GenClientCallGo(file string) {
	var sb strings.Builder
	fmt.Fprint(&sb, "// Code generated by go generate; DO NOT EDIT.\n")
	fmt.Fprintf(&sb, "// %s\n\n", time.Now())
	fmt.Fprint(&sb, "package ooapi\n\n")
	fmt.Fprintf(&sb, "//go:generate go run ./internal/generator -file %s\n\n", file)
	fmt.Fprint(&sb, "import (\n")
	fmt.Fprint(&sb, "\t\"context\"\n")
	fmt.Fprint(&sb, "\n")
	fmt.Fprint(&sb, "\t\"github.com/ooni/probe-cli/v3/internal/ooapi/apimodel\"\n")
	fmt.Fprint(&sb, ")\n")
	for _, desc := range Descriptors {
		switch desc.Name {
		case "Register", "Login":
			// We don't want to generate these APIs as toplevel.
			continue
		}
		desc.genClientNewCaller(&sb)
		desc.genClientCall(&sb)
	}
	writefile(file, &sb)
}
