"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const HostType_1 = __importDefault(require("../types/HostType"));
const logger_1 = require("@nrchkb/logger");
const Storage_1 = require("../Storage");
const NRCHKBError_1 = __importDefault(require("../NRCHKBError"));
module.exports = function (node) {
    const log = (0, logger_1.logger)('NRCHKB', 'ServiceUtils2', node.config.name, node);
    const HapNodeJS = require('hap-nodejs');
    const Service = HapNodeJS.Service;
    const Characteristic = HapNodeJS.Characteristic;
    const CameraSource = require('../cameraSource').Camera;
    const NO_RESPONSE_MSG = 'NO_RESPONSE';
    const output = function (allCharacteristics, event, { oldValue, newValue }, connection) {
        const eventObject = typeof event === 'object' ? event : { name: event };
        log.debug(`${eventObject.name} event, oldValue: ${oldValue}, newValue: ${newValue}, connection ${connection === null || connection === void 0 ? void 0 : connection.sessionID}`);
        const msg = {
            name: node.name,
            topic: node.config.topic ? node.config.topic : node.topic_in,
        };
        msg.payload = {};
        msg.hap = {
            newValue,
            event: eventObject,
            allChars: allCharacteristics.reduce((allChars, singleChar) => {
                allChars[singleChar.displayName] = singleChar.value;
                return allChars;
            }, {}),
        };
        if (oldValue !== undefined) {
            msg.hap.oldValue = oldValue;
        }
        const key = this.constructor.name;
        msg.payload[key] = newValue;
        if (connection) {
            msg.hap.session = {
                sessionID: connection.sessionID,
                username: connection.username,
                remoteAddress: connection.remoteAddress,
                localAddress: connection.localAddress,
                httpPort: connection.remotePort,
            };
        }
        node.setStatus({
            fill: 'yellow',
            shape: 'dot',
            text: `[${eventObject.name}] ${key}${newValue != undefined ? `: ${newValue}` : ''}`,
        }, 3000);
        log.debug(`${node.name} received ${eventObject.name} ${key}: ${newValue}`);
        if (connection || node.hostNode.config.allowMessagePassthrough) {
            node.send(msg);
        }
    };
    const onCharacteristicGet = (allCharacteristics) => function (callback, _context, connection) {
        const characteristic = this;
        const oldValue = characteristic.value;
        const delayedCallback = (value) => {
            const newValue = value !== null && value !== void 0 ? value : characteristic.value;
            if (callback) {
                try {
                    callback(node.accessory.reachable
                        ? characteristic.statusCode
                        : new Error(NO_RESPONSE_MSG), newValue);
                }
                catch (_) { }
            }
            output.call(characteristic, allCharacteristics, { name: "get" }, { oldValue, newValue }, connection);
        };
        if (node.config.useEventCallback) {
            const callbackID = Storage_1.Storage.saveCallback({
                event: "get",
                callback: delayedCallback,
            });
            log.debug(`Registered callback ${callbackID} for Characteristic ${characteristic.displayName}`);
            output.call(this, allCharacteristics, {
                name: "get",
                context: { callbackID },
            }, { oldValue }, connection);
        }
        else {
            delayedCallback();
        }
    };
    const onCharacteristicSet = (allCharacteristics) => function (newValue, callback, _context, connection) {
        try {
            if (callback) {
                callback(node.accessory.reachable
                    ? null
                    : new Error(NO_RESPONSE_MSG));
            }
        }
        catch (_) { }
        output.call(this, allCharacteristics, "set", { newValue }, connection);
    };
    const onCharacteristicChange = (allCharacteristics) => function (change) {
        const { oldValue, newValue, context, originator, reason } = change;
        if (oldValue != newValue) {
            output.call(this, allCharacteristics, {
                name: "change",
                context: { reason },
            }, { oldValue, newValue, context }, originator);
        }
    };
    const onInput = function (msg) {
        var _a;
        if (msg.payload) {
            const type = typeof msg.payload;
            if (type !== 'object') {
                log.error(`Invalid payload type: ${type}`);
                return;
            }
        }
        else {
            log.error('Invalid message (payload missing)');
            return;
        }
        const topic = node.config.topic ? node.config.topic : node.name;
        if (node.config.filter && msg.topic !== topic) {
            log.debug("msg.topic doesn't match configured value and filter is enabled. Dropping message.");
            return;
        }
        let context = null;
        if (msg.payload.Context) {
            context = msg.payload.Context;
            delete msg.payload.Context;
        }
        node.topic_in = (_a = msg.topic) !== null && _a !== void 0 ? _a : '';
        Object.keys(msg.payload).map((key) => {
            var _a, _b;
            if (node.supported.indexOf(key) < 0) {
                if (node.config.useEventCallback &&
                    Storage_1.Storage.uuid4Validate(key)) {
                    const callbackID = key;
                    const callbackValue = (_a = msg.payload) === null || _a === void 0 ? void 0 : _a[key];
                    const eventCallback = Storage_1.Storage.loadCallback(callbackID);
                    if (eventCallback) {
                        log.debug(`Calling ${eventCallback.event} callback ${callbackID}`);
                        eventCallback.callback(callbackValue);
                    }
                    else {
                        log.error(`Callback ${callbackID} timeout`);
                    }
                }
                else {
                    log.error(`Instead of ${key} try one of these characteristics: ${node.supported.join(', ')}`);
                }
            }
            else {
                const value = (_b = msg.payload) === null || _b === void 0 ? void 0 : _b[key];
                if ((node.config.isParent &&
                    node.config.hostType == HostType_1.default.BRIDGE) ||
                    (!node.config.isParent &&
                        node.hostNode.hostType == HostType_1.default.BRIDGE)) {
                    node.accessory.updateReachability(value !== NO_RESPONSE_MSG);
                }
                const characteristic = node.service.getCharacteristic(Characteristic[key]);
                if (context !== null) {
                    characteristic.setValue(value, undefined, context);
                }
                else {
                    characteristic.setValue(value);
                }
            }
        });
    };
    const onClose = function (removed, done) {
        const characteristics = node.service.characteristics.concat(node.service.optionalCharacteristics);
        characteristics.forEach(function (characteristic) {
            characteristic.removeListener('get', node.onCharacteristicGet);
            characteristic.removeListener('set', node.onCharacteristicSet);
            characteristic.removeListener('change', node.onCharacteristicChange);
        });
        if (node.config.isParent) {
            node.accessory.removeListener('identify', node.onIdentify);
        }
        if (removed) {
            if (node.config.isParent) {
                node.hostNode.host.removeBridgedAccessories([node.accessory]);
                node.accessory.destroy();
            }
            else {
                node.accessory.removeService(node.service);
                node.parentService.removeLinkedService(node.service);
            }
        }
        done();
    };
    const getOrCreate = function (accessory, serviceInformation, parentService) {
        const newService = new Service[serviceInformation.serviceName](serviceInformation.name, serviceInformation.UUID);
        log.debug(`Looking for service with UUID ${serviceInformation.UUID} ...`);
        let service = accessory.services.find((service) => {
            return newService.subtype === service.subtype;
        });
        if (service && newService.UUID !== service.UUID) {
            log.debug('... service type changed! Removing the old service.');
            accessory.removeService(service);
            service = undefined;
        }
        if (!service) {
            log.debug(`... didn't find it. Adding new ${serviceInformation.serviceName} service.`);
            if (serviceInformation.serviceName === 'CameraControl') {
                configureCameraSource(accessory, newService, serviceInformation.config);
                service = newService;
            }
            else {
                service = accessory.addService(newService);
            }
        }
        else {
            log.debug('... found it! Updating it.');
            service
                .getCharacteristic(Characteristic.Name)
                .setValue(serviceInformation.name);
        }
        if (parentService) {
            if (serviceInformation.serviceName === 'CameraControl') {
                log.debug('... and adding service to accessory.');
            }
            else if (service) {
                log.debug('... and linking service to parent.');
                parentService.addLinkedService(service);
            }
        }
        return service;
    };
    const configureCameraSource = function (accessory, service, config) {
        if (config.cameraConfigSource) {
            log.debug('Configuring Camera Source');
            if (!config.cameraConfigVideoProcessor) {
                log.error('Missing configuration for CameraControl: videoProcessor cannot be empty!');
            }
            else {
                accessory.configureCameraSource(new CameraSource(service, config, node));
            }
        }
        else {
            log.error('Missing configuration for CameraControl.');
        }
    };
    const waitForParent = () => {
        log.debug('Waiting for Parent Service');
        return new Promise((resolve) => {
            node.setStatus({
                fill: 'blue',
                shape: 'dot',
                text: 'Waiting for Parent Service',
            });
            const checkAndWait = () => {
                const parentNode = node.RED.nodes.getNode(node.config.parentService);
                if (parentNode && parentNode.configured) {
                    resolve(parentNode);
                }
                else {
                    setTimeout(checkAndWait, 1000);
                }
            };
            checkAndWait();
        }).catch((error) => {
            log.error(`Waiting for Parent Service failed due to: ${error}`);
            throw new NRCHKBError_1.default(error);
        });
    };
    const handleWaitForSetup = (config, msg, resolve) => {
        if (node.setupDone) {
            return;
        }
        if (msg.hasOwnProperty('payload') &&
            msg.payload.hasOwnProperty('nrchkb') &&
            msg.payload.nrchkb.hasOwnProperty('setup')) {
            node.setupDone = true;
            const newConfig = Object.assign(Object.assign({}, config), msg.payload.nrchkb.setup);
            node.removeListener('input', node.handleWaitForSetup);
            resolve(newConfig);
        }
        else {
            log.error('Invalid message (required {"payload":{"nrchkb":{"setup":{}}}})');
        }
    };
    return {
        getOrCreate,
        onCharacteristicGet,
        onCharacteristicSet,
        onCharacteristicChange,
        onInput,
        onClose,
        waitForParent,
        handleWaitForSetup,
    };
};
