"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const HostType_1 = __importDefault(require("./types/HostType"));
const preload_1 = __importDefault(require("semver/preload"));
const hap_nodejs_1 = require("hap-nodejs");
const HapCategories_1 = __importDefault(require("./types/HapCategories"));
const semver_1 = require("semver");
const logger_1 = require("@nrchkb/logger");
module.exports = (RED, hostType) => {
    const MdnsUtils = require('./utils/MdnsUtils')();
    const init = function (config) {
        var _a, _b, _c, _d, _e;
        const self = this;
        const log = logger_1.logger('NRCHKB', 'HAPHostNode', config.bridgeName, self);
        self.hostType = hostType;
        RED.nodes.createNode(self, config);
        self.config = config;
        self.name = config.bridgeName;
        if (!hostNameValidator(config.bridgeName)) {
            log.error('Host name is incorrect', false);
            throw Error('Host name is incorrect');
        }
        if (preload_1.default.parse(config.firmwareRev) == null) {
            config.firmwareRev = new semver_1.SemVer('0.0.0');
        }
        if (config.customMdnsConfig) {
            self.mdnsConfig = {};
            if (MdnsUtils.checkMulticast(config.mdnsMulticast)) {
                self.mdnsConfig.multicast = config.mdnsMulticast;
            }
            if (MdnsUtils.checkInterface(config.mdnsInterface)) {
                self.mdnsConfig.interface = config.mdnsInterface;
            }
            if (MdnsUtils.checkPort(config.mdnsPort)) {
                self.mdnsConfig.port = parseInt((_a = config.mdnsPort) === null || _a === void 0 ? void 0 : _a.toString());
            }
            if (MdnsUtils.checkIp(config.mdnsIp)) {
                self.mdnsConfig.ip = config.mdnsIp;
            }
            if (MdnsUtils.checkTtl(config.mdnsTtl)) {
                self.mdnsConfig.ttl = parseInt((_b = config.mdnsTtl) === null || _b === void 0 ? void 0 : _b.toString());
            }
            if (MdnsUtils.checkLoopback(config.mdnsLoopback)) {
                self.mdnsConfig.loopback = config.mdnsLoopback;
            }
            if (MdnsUtils.checkReuseAddr(config.mdnsReuseAddr)) {
                self.mdnsConfig.reuseAddr = config.mdnsReuseAddr;
            }
        }
        self.accessoryCategory = (self.hostType == HostType_1.default.BRIDGE
            ? HapCategories_1.default.BRIDGE
            : self.config.accessoryCategory);
        self.published = false;
        self.bridgeUsername = macify(self.id);
        const hostUUID = hap_nodejs_1.uuid.generate(self.id);
        const hostTypeName = self.hostType == HostType_1.default.BRIDGE ? 'Bridge' : 'Standalone Accessory';
        log.debug(`Creating ${hostTypeName} with UUID ${hostUUID}`);
        if (self.hostType == HostType_1.default.BRIDGE) {
            self.host = new hap_nodejs_1.Bridge(self.name, hostUUID);
        }
        else {
            self.host = new hap_nodejs_1.Accessory(self.name, hostUUID);
        }
        self.publish = function () {
            var _a, _b, _c;
            if (self.hostType == HostType_1.default.BRIDGE) {
                log.debug(`Publishing ${hostTypeName} with pin code ${self.config.pinCode} and ${self.host.bridgedAccessories.length} accessories`);
            }
            else {
                log.debug(`Publishing ${hostTypeName} with pin code ${self.config.pinCode}`);
            }
            if ((self.config.port && self.config.port == 1880) ||
                (((_a = self.mdnsConfig) === null || _a === void 0 ? void 0 : _a.port) && ((_b = self.mdnsConfig) === null || _b === void 0 ? void 0 : _b.port) == 1880)) {
                log.error(`Cannot publish on ${hostTypeName} port 1880 as it is reserved for node-red`);
                self.published = false;
                return false;
            }
            self.host.publish({
                username: self.bridgeUsername,
                port: self.config.port && !isNaN(self.config.port)
                    ? self.config.port
                    : 0,
                pincode: self.config.pinCode,
                category: self.accessoryCategory,
                mdns: self.mdnsConfig,
                advertiser: (_c = self.config.advertiser) !== null && _c !== void 0 ? _c : "bonjour-hap",
            }, self.config.allowInsecureRequest);
            self.published = true;
            return true;
        };
        self.on('close', function (removed, done) {
            if (removed) {
                self.host.destroy();
            }
            else {
                self.host.unpublish();
                self.published = false;
            }
            done();
        });
        self.host.on('identify', function (paired, callback) {
            if (paired) {
                log.debug(`Identify called on paired ${hostTypeName}`);
            }
            else {
                log.debug(`Identify called on unpaired ${hostTypeName}`);
            }
            callback();
        });
        const accessoryInformationService = self.host.getService(hap_nodejs_1.Service.AccessoryInformation) ||
            self.host.addService(hap_nodejs_1.Service.AccessoryInformation);
        accessoryInformationService
            .setCharacteristic(hap_nodejs_1.Characteristic.Manufacturer, self.config.manufacturer)
            .setCharacteristic(hap_nodejs_1.Characteristic.SerialNumber, self.config.serialNo)
            .setCharacteristic(hap_nodejs_1.Characteristic.Model, self.config.model)
            .setCharacteristic(hap_nodejs_1.Characteristic.FirmwareRevision, (_c = self.config.firmwareRev) === null || _c === void 0 ? void 0 : _c.toString())
            .setCharacteristic(hap_nodejs_1.Characteristic.HardwareRevision, (_d = self.config.hardwareRev) === null || _d === void 0 ? void 0 : _d.toString())
            .setCharacteristic(hap_nodejs_1.Characteristic.SoftwareRevision, (_e = self.config.softwareRev) === null || _e === void 0 ? void 0 : _e.toString());
    };
    const macify = (nodeId) => {
        if (nodeId) {
            const noDecimalStr = nodeId.replace('.', '');
            const paddedStr = noDecimalStr.padEnd(12, '0');
            const match = paddedStr.match(/.{1,2}/g);
            if (match) {
                return match.join(':').substr(0, 17).toUpperCase();
            }
            else {
                throw Error(`match failed in macify process for padded string ${paddedStr}`);
            }
        }
        else {
            throw Error('nodeId cannot be empty in macify process');
        }
    };
    const hostNameValidator = function (hostName) {
        return hostName ? /^[^.]{1,64}$/.test(hostName) : false;
    };
    return {
        init,
        macify,
    };
};
