#!/usr/bin/env vpython3
# Copyright 2022 The Chromium Authors
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.
"""Implements commands for flashing a Fuchsia device."""

import argparse
import json
import os
import sys
import time

from typing import Optional, Tuple

from common import register_device_args, run_ffx_command
from compatible_utils import get_sdk_hash
from ffx_integration import ScopedFfxConfig


def _get_system_info(target: Optional[str]) -> Tuple[str, str]:
    """Retrieves installed OS version from device.

    Returns:
        Tuple of strings, containing (product, version number).
    """

    info_json = json.loads(
        run_ffx_command(('target', 'show', '--json'),
                        target_id=target,
                        capture_output=True).stdout.strip())
    for info in info_json:
        if info['title'] == 'Build':
            return (info['child'][1]['value'], info['child'][0]['value'])

    # If the information was not retrieved, return empty strings to indicate
    # unknown system info.
    return ('', '')


def flash(system_image_dir: str,
          os_check: str,
          target: Optional[str],
          serial_num: Optional[str] = None) -> None:
    """Flash the device."""

    if os_check == 'ignore':
        return
    if not system_image_dir:
        raise ValueError('System image directory must be specified.')
    if (os_check == 'check'
            and get_sdk_hash(system_image_dir) == _get_system_info(target)):
        return
    manifest = os.path.join(system_image_dir, 'flash-manifest.manifest')
    with ScopedFfxConfig('fastboot.reboot.reconnect_timeout', '120'):
        if serial_num:
            with ScopedFfxConfig('discovery.zedboot.enabled', 'true'):
                run_ffx_command(('target', 'reboot', '-b'),
                                target,
                                check=False)
            for _ in range(10):
                time.sleep(10)
                if run_ffx_command(('target', 'list', serial_num),
                                   check=False).returncode == 0:
                    break
            run_ffx_command(('target', 'flash', manifest), serial_num)
        else:
            run_ffx_command(('target', 'flash', manifest), target)
    run_ffx_command(('target', 'wait'), target)


def register_flash_args(arg_parser: argparse.ArgumentParser,
                        default_os_check: Optional[str] = 'check') -> None:
    """Register common arguments for device flashing."""

    serve_args = arg_parser.add_argument_group('flash',
                                               'device flashing arguments')
    serve_args.add_argument('--system-image-dir',
                            help='Specify the directory that contains the '
                            'Fuchsia image used to pave the device. Only '
                            'needs to be specified if "os_check" is not '
                            '"ignore".')
    serve_args.add_argument('--serial-num',
                            default=os.environ.get('FUCHSIA_FASTBOOT_SERNUM'),
                            help='Serial number of the device. Should be '
                            'specified for devices that do not have an image '
                            'flashed.')
    serve_args.add_argument('--os-check',
                            choices=['check', 'update', 'ignore'],
                            default=default_os_check,
                            help='Sets the OS version enforcement policy. If '
                            '"check", then the deployment process will halt '
                            'if the target\'s version does not match. If '
                            '"update", then the target device will '
                            'be reflashed. If "ignore", then the OS version '
                            'will not be checked.')


def main():
    """Stand-alone function for flashing a device."""
    parser = argparse.ArgumentParser()
    register_device_args(parser)
    register_flash_args(parser)
    args = parser.parse_args()
    flash(args.system_image_dir, args.os_check, args.target_id,
          args.serial_num)


if __name__ == '__main__':
    sys.exit(main())
