--[[
 * MIT License
 *
 * Copyright (c) 2022 Jianhui Zhao <zhaojh329@gmail.com>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
--]]

local file = require 'eco.core.file'
local sys = require 'eco.core.sys'
local bufio = require 'eco.bufio'

local concat = table.concat
local type = type

local M = {}

local exec_methods = {}

function exec_methods:release()
    local mt = getmetatable(self)

    if not mt.stdout_fd then
        return
    end

    file.close(mt.stdout_fd)
    file.close(mt.stderr_fd)

    self.stdout_fd = nil
    self.stderr_fd = nil
end

function exec_methods:wait(timeout)
    local mt = getmetatable(self)
    return mt.child_w:wait(timeout or 30.0)
end

function exec_methods:pid()
    local mt = getmetatable(self)
    return mt.pid
end

local function exec_read(b, pattern, timeout)
    if type(pattern) == 'number' then
        if pattern <= 0 then return '' end
        return b:read(pattern, timeout)
    end

    if pattern == '*a' then
        local data = {}
        local chunk, err
        while true do
            chunk, err = b:read(4096)
            if not chunk then break end
            data[#data + 1] = chunk
        end

        if #data == 0 then
            return nil, err
        end

        if err == 'closed' then
            return concat(data)
        end

        return nil, err, concat(data)
    end

    if not pattern or pattern == '*l' then
        return b:readline(timeout)
    end

    error('invalid pattern:' .. tostring(pattern))
end

function exec_methods:read_stdout(pattern, timeout)
    local mt = getmetatable(self)

    if not mt.stdout_fd then
        return nil, 'closeed'
    end

    return exec_read(mt.stdout_b, pattern, timeout)
end

function exec_methods:read_stderr(pattern, timeout)
    local mt = getmetatable(self)

    if not mt.stderr_fd then
        return nil, 'closeed'
    end

    return exec_read(mt.stderr_b, pattern, timeout)
end

local function create_bufio(fd)
    local reader = {
        w = eco.watcher(eco.IO, fd),
        fd = fd
    }

    function reader:read(n, timeout)
        if not self.w:wait(timeout) then
            return nil, 'timeout'
        end

        local data, err = file.read(self.fd, n, timeout)
        if not data then
            return nil, err
        end

        if #data == 0 then
            return nil, 'closed'
        end

        return data
    end

    function reader:read2b(b, timeout)
        if b:room() == 0 then
            return nil, 'buffer is full'
        end

        if not self.w:wait(timeout) then
            return nil, 'timeout'
        end

        local r, err = file.read_to_buffer(self.fd, b, timeout)
        if not r then
            return nil, err
        end

        if r == 0 then
            return nil, 'closed'
        end

        return r, err
    end

    return bufio.new(reader)
end

function M.exec(...)
    local pid, stdout_fd, stderr_fd = sys.exec(...)
    if not pid then
        return nil, stdout_fd
    end

    local p = {}

    if tonumber(_VERSION:match('%d%.%d')) < 5.2 then
        local __prox = newproxy(true)
        getmetatable(__prox).__gc = function() p:release() end
        p[__prox] = true
    end

    return setmetatable(p, {
        pid = pid,
        stdout_fd = stdout_fd,
        stderr_fd = stderr_fd,
        child_w = eco.watcher(eco.CHILD, pid),
        stdout_b = create_bufio(stdout_fd),
        stderr_b = create_bufio(stderr_fd),
        __index = exec_methods,
        __gc = exec_methods.release
    })
end

function M.signal(sig, cb, ...)
    eco.run(function(...)
        local w = eco.watcher(eco.SIGNAL, sig)
        while true do
            w:wait()
            cb(...)
        end
    end, ...)
end

return setmetatable(M, { __index = sys })
