/*   FILE: state-play.vala -- A state machine implementing lcdgrilo's menu
 * AUTHOR: W. Michael Petullo <mike@flyn.org>
 *   DATE: 01 December 2013 
 *
 * Copyright (c) 2013 W. Michael Petullo <new@flyn.org>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * stop:        stop currently playing, clear playlist, and pop state.
 * pauseplay:   pause if playing else transition from paused to playing.
 * seekback:    move one track back on the playlist.
 * seekforward: move one track forward on the playlist.
 * next:        scan forward in currently playing song.
 * previous:    scan back in currently playing song.
 * transition:  no operation.
 */

using Gst;

const int64 SCAN_JUMP_SECONDS = 5;
const bool  FORWARD           = true;
const bool  BACKWARD          = false;

class StatePlay : State {
	public StatePlay (LCDPlayer player) {
		this.player = player;

		GLib.Idle.add (player.play);
	}

	// Print selected item.
	public override void print_selected () {
		player.io.output (player.current_title());
	}

	// Seek to previous track.
	public override void seekback () {
		player.backward();
		GLib.Idle.add (player.play);
		player.io.output ("Playing " + player.current_title());
	}

	// Seek to next track.
	public override void seekforward () {
		player.forward();
		GLib.Idle.add (player.play);
		player.io.output ("Playing " + player.current_title());
	}

	// Select next menu item.
	public override void next () {
		var builder = new StringBuilder ();
		int64 position = seek (FORWARD);
		builder.printf ("%" + int64.FORMAT + ":%.2" + int64.FORMAT, position / (Gst.SECOND * 60), (position / Gst.SECOND) % 60);
		
		player.io.output (builder.str);
	}

	// Select previous menu item.
	public override void previous () {
		var builder = new StringBuilder ();
		int64 position = seek (BACKWARD);
		builder.printf ("%" + int64.FORMAT + ":%.2" + int64.FORMAT, position / (Gst.SECOND * 60), (position / Gst.SECOND) % 60);
		
		player.io.output (builder.str);
	}

	// Play currently selected song; do not transition to new state.
	public override State transition () {
		GLib.Idle.add (player.play);
		return this;
	}

	// Seek and return new absolute position in song.
	private int64 seek (bool forward) {
		return player.seek(forward);
	}
}
