/*   FILE: lcdgrilo.vala -- A simple DAAP music player
 * AUTHOR: W. Michael Petullo <mike@flyn.org>
 *   DATE: 27 November 2013
 *
 * Copyright (c) 2013 W. Michael Petullo <new@flyn.org>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using Grl;
using Gst;
using Pifacecad;
using Posix;

bool no_pifacecad = false;
string tag        = null;
bool debug        = false;

const OptionEntry[] options = {
	{ "no-pifacecad", 'n', 0, OptionArg.NONE, ref no_pifacecad,
	  "Do not try to use PiFace CAD for I/O", null },
	{ "tag", 't', 0, OptionArg.STRING, ref tag,
	  "Only show sources with the following tag", null },
	{ "debug", 'd', 0, OptionArg.NONE, ref debug,
	  "Print debugging messages", null },
	{ null }
};

private class LCDPlayer {
	public Stack<State> stateStack { get; set; default = null; }
	public dynamic Gst.Element element;
	public dynamic Gee.ArrayList<Grl.Media> playlist = new Gee.ArrayList<Grl.Media> ();
	public InputOutput io;
	private int current = 0;
	private Transitions transitions;
	private MainLoop loop;
	private Gee.ArrayList<Grl.Source> sources = new Gee.ArrayList<unowned Grl.Source> ();
	private uint watch_id = 0;

	public LCDPlayer () {
		element = Gst.ElementFactory.make ("playbin", "play");
		// Audio and soft-volume, don't decode video.
		element.flags = 0x00000002 | 0x00000010;

		io          = new InputOutput(!no_pifacecad);
		stateStack  = new Stack<State> ();
		transitions = new Transitions(io, stateStack);

		var registry = Grl.Registry.get_default ();

		registry.source_added.connect((source) => {
			if ((source.get_supported_operations() & Grl.SupportedOps.BROWSE) == 0)
				return;

			if (tag != null) {
				if (! (tag in source.get_tags()))
					return;
			}

			sources.add (source);

			// Possibly replace "No media source found" message.
			stateStack.peek ().print_selected ();
		});

		registry.source_removed.connect ((source) => {
			sources.remove (source);
		});

		try {
			registry.load_all_plugins (false);
		} catch (GLib.Error e) {
			GLib.error ("%s", e.message);
		}

		if (tag == null) {
			try {
				registry.activate_plugin_by_id ("grl-daap");
				// registry.activate_plugin_by_id ("grl-jamendo");
				// registry.activate_plugin_by_id ("grl-magnatune");
			} catch (GLib.Error e) {
				GLib.error ("%s", e.message);
			}
		} else {
			registry.activate_all_plugins ();
		}

	}

	private bool bus_callback (Gst.Bus bus, Gst.Message message) {
		bool fnval = true;

                switch (message.type) {
                case MessageType.ERROR:
			// Terminate; lcdgrilo will hopefully restart.
                        GLib.Error err;
                        string debug;
                        message.parse_error (out err, out debug);
			var builder = new StringBuilder ();
			builder.printf ("Error: %s\n", err.message);
			io.output (builder.str);
			GLib.Process.exit (1);
                case MessageType.EOS:
			// Play the next queued song.
			forward();
			GLib.Idle.add (play);
			fnval = false;
                        break;
                default:
                        break;
                }

                return fnval;
        }

	public void stop() {
		element.set_state (Gst.State.NULL);
		GLib.Source.remove(watch_id);
		playlist.clear();
	}

	public void pauseplay() {
		Gst.State state; 
		element.get_state (out state, null, 0);
		if (Gst.State.PLAYING == state) {
			element.set_state (Gst.State.PAUSED);
			io.output("Paused");
		} else if (Gst.State.PAUSED == state) {
			element.set_state (Gst.State.PLAYING);
			stateStack.peek ().print_selected ();
		}
	}

	public bool play() {
		Gst.State state;

		element.get_state(out state, null, 0);
		if (state == Gst.State.PLAYING) {
			// Nothing to do; might have just added another song to queue.
			return false;
		}

		var media = playlist[current];

		element.uri = media.get_url ();
		element.current_audio = media.get_int (Grl.MetadataKey.AUDIO_TRACK);

		Gst.Bus bus = element.get_bus ();
		watch_id = bus.add_watch (GLib.Priority.DEFAULT, bus_callback);

		element.set_state (Gst.State.PLAYING);

		return false;
	}

	public void forward() {
		element.set_state (Gst.State.NULL);
		GLib.Source.remove(watch_id);
		current = (current + 1) % playlist.size;
	}

	public void backward() {
		element.set_state (Gst.State.NULL);
		GLib.Source.remove(watch_id);
		current = (current + (playlist.size - 1)) % playlist.size;
	}

	public int64 seek(bool forward) {
		int64 current_position, next_position;

		element.query_position (Gst.Format.TIME, out current_position);

		next_position = current_position + (SCAN_JUMP_SECONDS * Gst.SECOND * (forward ? 1 : -1));
		next_position = next_position < 0 ? 0 : next_position;

		element.seek_simple (Gst.Format.TIME, Gst.SeekFlags.FLUSH | Gst.SeekFlags.KEY_UNIT, next_position);

		return next_position;
	}

	public string current_title() {
		if (playlist.size == 0) {
			return "Play queue empty";
		} else {
			return playlist[current].get_title ();
		}
	}

	public void run() {
		loop = new MainLoop ();

		stateStack.push (new StateChooseSource (loop, this, sources));

		loop.run ();

		if (! no_pifacecad) {
			Pifacecad.close ();
		}
	}
}

int main (string[] args) {     
	var opt_context = new OptionContext ("- Media player");

	opt_context.add_group (Grl.init_get_option_group ());
	opt_context.add_group (Gst.init_get_option_group ());
	opt_context.add_main_entries (options, null);

	try {
		opt_context.parse (ref args);
	} catch (OptionError e) {
		GLib.stderr.printf("%s: %s\n", args[0], e.message);
		GLib.stderr.printf("Try '%s --help' for more information.\n", args[0]);
		return 1;
	}

	if (debug) {
		GLib.Environment.set_variable("G_MESSAGES_DEBUG", "all", true);
	}

	// Do not echo terminal input.
	Posix.termios old_tio, new_tio;
	Posix.tcgetattr(Posix.STDIN_FILENO, out old_tio);
	new_tio = old_tio;
	new_tio.c_lflag &=(~ICANON & ~ECHO);
	Posix.tcsetattr(STDIN_FILENO, TCSANOW, new_tio);

	var player = new LCDPlayer ();
	player.run();

	// Restore terminal input echo.
	Posix.tcsetattr(STDIN_FILENO, TCSANOW, old_tio);

	return 0;
}
