/* $Id$
 * $URL$
 *
 * This plugin adds some basic linked list functions to lcd4linux.
 *
 * https://lcd4linux.bulix.org/wiki/plugin_list
 * 
 * Copyright (C) 2003 Michael Reinelt <michael@reinelt.co.at>
 * Copyright (C) 2004, 2005, 2006, 2007, 2008 The LCD4Linux Team <lcd4linux-devel@users.sourceforge.net>
 * Copyright (C) 2015 Marcus Menzel <codingmax@gmx-topmail.de>
 *
 * This file is part of LCD4Linux.
 *
 * LCD4Linux is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * LCD4Linux is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */



#include "config.h"
#include "cfg.h"
#include "debug.h"
#include "plugin.h"
#include "evaluator.h"

#include <stdlib.h>
#include <string.h>

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif

#define SECTION "Plugin:List"
#define PLUGIN "List"
#define LIST_LIMIT_MAX 1000
#define LIST_LIMIT_DEFAULT 10

/* copied from plugin_layout.c */
#define my_debug(args...) message (2, __FILE__ ": " args)
#define my_info(args...)  message (1, "Info  Plugin '" PLUGIN "': " args)
#define my_error(args...) message (0, "ERROR Plugin '" PLUGIN "': " args)
#define my_noMemory() message (0, "ERROR Couldn't allocate new memory (%s:%d).",PLUGIN, __LINE__)

typedef struct Element Element;

typedef struct Element {

    RESULT *value;
    Element *next;

} Element;

typedef struct List {

    char *name;
    int size;
    int limit;
    Element *firstElement;

} List;


typedef struct Plugin {

    List **lists;
    int listNb;

} Plugin;

static Plugin *plugin = NULL;



static void my_set_result(RESULT * result, RESULT * rVal)
{
    if (rVal->type & R_STRING)
	SetResult(&result, R_STRING, rVal->string);
    else
	SetResult(&result, R_NUMBER, &rVal->number);
}


static void my_free_element(Element * element)
{

    if (element == NULL)
	return;

    if (element->value != NULL) {
	free(element->value->string);
	free(element->value);
    }
    free(element);
}


static void my_free_Element(Element * e)
{

    if (e == NULL)
	return;

    if (e->value != NULL) {
	DelResult(e->value);
	free(e->value);
    }
    free(e);
}


static void my_free_list(List * list)
{

    if (list == NULL)
	return;

    Element *e;
    Element *next;
    for (e = list->firstElement; e != NULL; e = next) {
	next = e->next;
	my_free_Element(e);
    }
    free(list->name);
    free(list);
}

static void my_free_plugin()
{

    if (plugin == NULL)
	return;

    int i;
    if (plugin->listNb > 0)
	for (i = 0; i < plugin->listNb; i++)
	    my_free_list(plugin->lists[i]);

    free(plugin->lists);
    free(plugin);
    plugin = NULL;
}


static RESULT *my_cfg_get_result(const char *section, const char *key)
{

    char *expression = cfg_get_raw(section, key, NULL);

    if (expression == NULL)
	return NULL;

    void *tree = NULL;
    RESULT *result = malloc(sizeof(RESULT));
    result->string = NULL;

    if (*expression == '\0' || strcmp(expression, "''") == 0) {
	result->type = R_STRING;
	result->size = 1;
	result->string = strdup("");
	return result;
    }

    if (Compile(expression, &tree) == 0 && Eval(tree, result) == 0) {
	DelTree(tree);
	return result;
    }
    DelTree(tree);
    return NULL;
}





static int my_init_list(int idxL)
{

    List *list = malloc(sizeof(List));
    if (list == NULL) {
	my_noMemory();
	return -1;
    }
    plugin->lists[idxL] = list;

    list->name = NULL;
    list->limit = LIST_LIMIT_DEFAULT;
    list->size = 0;
    list->firstElement = NULL;

    int i;

    char *key = cfg_make_str("List%d.name", idxL + 1);
    char *defVal = cfg_make_str("List%d", idxL + 1);
    char *name = cfg_get(SECTION, key, defVal);
    free(key);
    free(defVal);

    if (name == NULL || *name == '\0') {
	my_error("List %d has no valide name.", idxL + 1);
	free(name);
	return -1;
    }
    for (i = 0; i < idxL; i++) {
	if (strcmp(plugin->lists[i]->name, name) == 0) {
	    my_error("List name '%s' is not unique.", name);
	    free(name);
	    return -1;
	}
    }
    list->name = name;
    my_info("List%d.name = '%s'.", idxL + 1, name);


    key = cfg_make_str("List%d.limit", idxL + 1);
    int limit = 0;
    int r = cfg_number(SECTION, key, LIST_LIMIT_DEFAULT, 0,
		       LIST_LIMIT_MAX, &limit);
    free(key);


    if (r < 0) {
	my_error("List %d has no valide limit.", idxL + 1);
	return -1;
    }
    list->limit = limit;



    Element *prev = NULL;

    for (i = 0; i < LIST_LIMIT_MAX; i++) {

	char *key = cfg_make_str("List%d.value%d", idxL + 1, i + 1);

	RESULT *res = my_cfg_get_result(SECTION, key);
	free(key);

	if (res == NULL)
	    break;

	Element *element = malloc(sizeof(Element));
	if (element == NULL) {
	    my_noMemory();
	    return -1;
	}

	element->value = res;
	element->next = NULL;

	if (prev == NULL)
	    list->firstElement = element;
	else
	    prev->next = element;

	list->size++;
	prev = element;
    }

    if (list->limit < list->size) {
	list->limit = list->size;
	my_error("Given limit of list '%s' is lower than its initial size.", list->name);
	my_error("Limit has been increased to %d.", list->limit);
    }

    return 0;
}

static int my_init_plugin()
{

    plugin = malloc(sizeof(Plugin));
    if (plugin == NULL) {
	my_noMemory();
	return -1;
    }

    plugin->lists = NULL;
    plugin->listNb = 0;


    int active = 0;
    if (cfg_number(SECTION, "active", 0, 0, 1, &active) != 1) {
	my_error("No valide 'active' value. Set to inactive.");
	return -1;
    }

    if (!active) {
	my_info("Set inactive.");
	return 0;
    } else
	my_info("Try to init.");


    char *prefix = cfg_make_str("%s.List", SECTION);
    int listNb = cfg_count(prefix, LIST_LIMIT_MAX);
    free(prefix);

    if (listNb <= 0) {
	my_error("No lists found.");
	return -1;
    }

    plugin->listNb = listNb;
    my_info("%d lists found.", plugin->listNb);

    plugin->lists = malloc(plugin->listNb * sizeof(List *));

    if (plugin->lists == NULL) {
	my_noMemory();
	return -1;
    }

    int i;
    for (i = 0; i < plugin->listNb; i++)
	plugin->lists[i] = NULL;

    int status = 1;
    for (i = 0; status == 1 && i < plugin->listNb; i++)
	status = my_init_list(i) < 0 ? -1 : 1;

    return status;
}


static void my_init()
{
    int status = my_init_plugin();

    if (status != 1) {
	if (status < 0) {
	    my_error("ERROR while initializing plugin.");
	    my_error("Plugin will NOT work. Every call will return -1 (error).");
	}
	my_free_plugin();
	plugin = NULL;
    }
}



static Element *my_get_element(int idxL, int idxE)
{

    if (idxL < 0 || idxL >= plugin->listNb)
	return NULL;

    List *list = plugin->lists[idxL];

    if (idxE < 0 || idxE >= list->size)
	return NULL;

    Element *elem;
    int idx = 0;
    for (elem = list->firstElement; elem != NULL; elem = elem->next, idx++) {
	if (idx == idxE)
	    return elem;
    }
    return NULL;
}

static int my_get_list_idx(RESULT * result, RESULT * rList, char *prefix)
{

    double error = -1;

    static int errorShown = 0;
    if (plugin == NULL) {

	if (!errorShown) {
	    errorShown = 1;
	    my_error("Plugin not active but attempt of using it was detected.");
	}
	SetResult(&result, R_NUMBER, &error);
	return -1;
    }

    int idxL = -1;

    if (rList->type & R_STRING) {
	int i;
	for (i = 0; i < plugin->listNb; i++)
	    if (strcmp(rList->string, plugin->lists[i]->name) == 0)
		return i;
    }

    if (rList->type & R_NUMBER) {
	idxL = (int) rList->number - 1;
	if (idxL + 1 != rList->number || idxL < 0 || idxL >= plugin->listNb)
	    idxL = -1;
    }

    if (idxL < 0) {
	SetResult(&result, R_NUMBER, &error);
	my_error("Couldn't find list %s to apply %s.", R2S(rList), prefix);
    }
    return idxL;
}


static int my_get_element_idx(RESULT * result, int idxL, RESULT * rElem, int oneMore, char *prefix)
{

    if (idxL < 0)
	return -1;

    double error = -1;
    int idxE = -1;

    if (rElem->type & R_NUMBER) {

	idxE = (int) rElem->number - 1;
	if (idxE + 1 != rElem->number || idxE < 0 || idxE - (oneMore ? 1 : 0) >= plugin->lists[idxL]->size) {

	    idxE = -1;
	}
    }

    if (idxE < 0)
	my_error("Invalide index %s of list '%s' to apply %s.", R2S(rElem), plugin->lists[idxL]->name, prefix);
    SetResult(&result, R_NUMBER, &error);

    return idxE;

}


static void plugin_list_size(RESULT * result, RESULT * rList)
{

    int idxL = my_get_list_idx(result, rList, "size()");
    if (idxL < 0)
	return;

    double d = plugin->lists[idxL]->size;
    SetResult(&result, R_NUMBER, &d);
}

static void plugin_list_limit(RESULT * result, RESULT * rList)
{

    int idxL = my_get_list_idx(result, rList, "limit()");
    if (idxL < 0)
	return;

    double d = plugin->lists[idxL]->limit;
    SetResult(&result, R_NUMBER, &d);
}


static void plugin_list_get(RESULT * result, RESULT * rList, RESULT * rElem)
{

    int idxL = my_get_list_idx(result, rList, "get()");
    int idxE = my_get_element_idx(result, idxL, rElem, 0, "get()");
    if (idxE < 0)
	return;

    Element *elem = my_get_element(idxL, idxE);

    if (elem != NULL) {
	my_set_result(result, elem->value);
    } else {
	double d = -1;
	SetResult(&result, R_NUMBER, &d);
    }
}


/* not 100% safe because of mixed RESULT types and atof in R2N() */
static void plugin_list_find(RESULT * result, RESULT * rList, RESULT * rVal)
{

    int idxL = my_get_list_idx(result, rList, "find()");
    if (idxL < 0)
	return;

    List *list = plugin->lists[idxL];


    double idxE = -1;
    int i = 0;
    Element *e;
    for (e = list->firstElement; e != NULL; e = e->next, i++) {

	int type = (e->value->type & rVal->type);
	if (type == 0)
	    continue;
	if (type & R_STRING) {
	    if (strcmp(R2S(e->value), R2S(rVal)) == 0) {
		idxE = i;
		break;
	    } else
		continue;
	}
	if (type & R_NUMBER) {
	    if (R2N(e->value) == R2N(rVal)) {
		idxE = i;
		break;
	    } else
		continue;
	}
    }

    if (idxE >= 0)
	idxE++;
    SetResult(&result, R_NUMBER, &idxE);
}



static void plugin_list_set(RESULT * result, RESULT * rList, RESULT * rElem, RESULT * rValue)
{

    int idxL = my_get_list_idx(result, rList, "set()");
    int idxE = my_get_element_idx(result, idxL, rElem, 0, "set()");
    if (idxE < 0)
	return;

    Element *elem = my_get_element(idxL, idxE);
    CopyResult(&elem->value, rValue);

    double d = 0;
    SetResult(&result, R_NUMBER, &d);
}

static void plugin_list_add(RESULT * result, RESULT * rList, RESULT * rElem, RESULT * rValue)
{

    int idxL = my_get_list_idx(result, rList, "add()");
    int idxE = my_get_element_idx(result, idxL, rElem, 1, "add()");
    if (idxE < 0)
	return;

    double d = -1;
    List *list = plugin->lists[idxL];

    if (list->size >= list->limit) {

	my_error("Couldn't add element to list '%s'. Limit reached.", list->name);
	SetResult(&result, R_NUMBER, &d);
	return;
    }

    RESULT *value = malloc(sizeof(RESULT));
    CopyResult(&value, rValue);

    Element *elem = malloc(sizeof(Element));

    elem->value = value;

    Element **prevNextPtr = NULL;

    if (idxE == 0)
	prevNextPtr = &plugin->lists[idxL]->firstElement;
    else
	prevNextPtr = &my_get_element(idxL, idxE - 1)->next;

    elem->next = *prevNextPtr;
    *prevNextPtr = elem;

    list->size++;

    d = 0;
    SetResult(&result, R_NUMBER, &d);
}


static void plugin_list_delete(RESULT * result, RESULT * rList, RESULT * rElem)
{

    int idxL = my_get_list_idx(result, rList, "delete()");
    int idxE = my_get_element_idx(result, idxL, rElem, 0, "delete()");
    if (idxE < 0)
	return;

    List *list = plugin->lists[idxL];
    Element *elem = my_get_element(idxL, idxE);
    Element **prevNextPtr = NULL;

    if (idxE == 0)
	prevNextPtr = &plugin->lists[idxL]->firstElement;
    else
	prevNextPtr = &my_get_element(idxL, idxE - 1)->next;

    *prevNextPtr = elem->next;

    list->size--;

    my_free_element(elem);

    double d = 0;
    SetResult(&result, R_NUMBER, &d);
}


int plugin_init_list(void)
{
    my_init();

    AddFunction("list::size", 1, plugin_list_size);
    AddFunction("list::limit", 1, plugin_list_limit);
    AddFunction("list::get", 2, plugin_list_get);
    AddFunction("list::find", 2, plugin_list_find);
    AddFunction("list::set", 3, plugin_list_set);
    AddFunction("list::add", 3, plugin_list_add);
    AddFunction("list::delete", 2, plugin_list_delete);

    return 0;
}

void plugin_exit_list(void)
{
    my_free_plugin();
}
