"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BatteryCreator = void 0;
const z2mModels_1 = require("../z2mModels");
const hap_1 = require("../hap");
const helpers_1 = require("../helpers");
const monitor_1 = require("./monitor");
class BatteryCreator {
    createServicesFromExposes(accessory, exposes) {
        const endpointMap = (0, helpers_1.groupByEndpoint)(exposes.filter(e => (0, z2mModels_1.exposesHasProperty)(e) && (0, z2mModels_1.exposesIsPublished)(e) && !accessory.isPropertyExcluded(e.property) && ((e.name === 'battery' && (0, z2mModels_1.exposesHasNumericRangeProperty)(e))
            || (e.name === 'battery_low' && (0, z2mModels_1.exposesHasBinaryProperty)(e)))).map(e => e));
        endpointMap.forEach((value, key) => {
            if (!accessory.isServiceHandlerIdKnown(BatteryHandler.generateIdentifier(key))) {
                this.createService(key, value, accessory);
            }
        });
    }
    createService(endpoint, exposes, accessory) {
        try {
            const handler = new BatteryHandler(endpoint, exposes, accessory);
            accessory.registerServiceHandler(handler);
        }
        catch (error) {
            accessory.log.warn('Failed to setup battery service ' +
                `for accessory ${accessory.displayName} for endpoint ${endpoint}: ${error}`);
        }
    }
}
exports.BatteryCreator = BatteryCreator;
class BatteryHandler {
    constructor(endpoint, exposes, accessory) {
        this.accessory = accessory;
        this.monitors = [];
        this.identifier = BatteryHandler.generateIdentifier(endpoint);
        const serviceName = accessory.getDefaultServiceDisplayName(endpoint);
        accessory.log.debug(`Configuring Battery Service for ${serviceName}`);
        const service = accessory.getOrAddService(new hap_1.hap.Service.BatteryService(serviceName, endpoint));
        (0, helpers_1.getOrAddCharacteristic)(service, hap_1.hap.Characteristic.BatteryLevel);
        (0, helpers_1.getOrAddCharacteristic)(service, hap_1.hap.Characteristic.StatusLowBattery);
        (0, helpers_1.getOrAddCharacteristic)(service, hap_1.hap.Characteristic.ChargingState);
        // Note: no defined exposes name for the charge state, so assuming batteries are non-chargeable.
        service.updateCharacteristic(hap_1.hap.Characteristic.ChargingState, hap_1.hap.Characteristic.ChargingState.NOT_CHARGEABLE);
        this.batteryLevelExpose = exposes.find(e => e.name === 'battery');
        this.batteryLowExpose = exposes.find(e => e.name === 'battery_low');
        if (this.batteryLevelExpose !== undefined) {
            this.monitors.push(new monitor_1.NumericCharacteristicMonitor(this.batteryLevelExpose.property, service, hap_1.hap.Characteristic.BatteryLevel, this.batteryLevelExpose.value_min, this.batteryLevelExpose.value_max));
        }
        else {
            if (this.batteryLowExpose === undefined) {
                throw new Error(`Can NOT create Battery Service (${serviceName}), if both 'battery' and 'battery_low' are missing.`);
            }
            // Mimic the battery level based on battery low indication.
            const fakeLevels = new Map();
            fakeLevels.set(this.batteryLowExpose.value_on, 0);
            fakeLevels.set(this.batteryLowExpose.value_off, 100);
            this.monitors.push(new monitor_1.MappingCharacteristicMonitor(this.batteryLowExpose.property, service, hap_1.hap.Characteristic.BatteryLevel, fakeLevels));
        }
        if (this.batteryLowExpose !== undefined) {
            const batteryLowMapping = new Map();
            batteryLowMapping.set(this.batteryLowExpose.value_on, hap_1.hap.Characteristic.StatusLowBattery.BATTERY_LEVEL_LOW);
            batteryLowMapping.set(this.batteryLowExpose.value_off, hap_1.hap.Characteristic.StatusLowBattery.BATTERY_LEVEL_NORMAL);
            this.monitors.push(new monitor_1.MappingCharacteristicMonitor(this.batteryLowExpose.property, service, hap_1.hap.Characteristic.StatusLowBattery, batteryLowMapping));
        }
        else {
            if (this.batteryLevelExpose === undefined) {
                throw new Error(`Can NOT create Battery Service (${serviceName}), if both 'battery' and 'battery_low' are missing.`);
            }
            this.monitors.push(new monitor_1.BinaryConditionCharacteristicMonitor(this.batteryLevelExpose.property, service, hap_1.hap.Characteristic.StatusLowBattery, (v) => (v < 30), hap_1.hap.Characteristic.StatusLowBattery.BATTERY_LEVEL_LOW, hap_1.hap.Characteristic.StatusLowBattery.BATTERY_LEVEL_NORMAL));
        }
    }
    get getableKeys() {
        const keys = [];
        if (this.batteryLevelExpose !== undefined && (0, z2mModels_1.exposesCanBeGet)(this.batteryLevelExpose)) {
            keys.push(this.batteryLevelExpose.property);
        }
        if (this.batteryLowExpose !== undefined && (0, z2mModels_1.exposesCanBeGet)(this.batteryLowExpose)) {
            keys.push(this.batteryLowExpose.property);
        }
        return keys;
    }
    updateState(state) {
        this.monitors.forEach(m => m.callback(state));
    }
    static generateIdentifier(endpoint) {
        let identifier = hap_1.hap.Service.BatteryService.UUID;
        if (endpoint !== undefined) {
            identifier += '_' + endpoint.trim();
        }
        return identifier;
    }
}
//# sourceMappingURL=battery.js.map