// Copyright (c) Microsoft Corporation.
// Licensed under the MIT license.
import { AbortController, AbortError } from "@azure/abort-controller";
import { Constants, RetryOperationType, StandardAbortMessage, retry, translate, } from "@azure/core-amqp";
import { types, } from "rhea-promise";
import { fromRheaMessage } from "./eventData";
import { getEventPositionFilter } from "./eventPosition";
import { logErrorStackTrace, logger } from "./log";
import { LinkEntity } from "./linkEntity";
import { getRetryAttemptTimeoutInMs } from "./util/retries";
import { createAbortablePromise } from "@azure/core-util";
/**
 * Describes the EventHubReceiver that will receive event data from EventHub.
 * @internal
 */
export class EventHubReceiver extends LinkEntity {
    /**
     * Instantiates a receiver that can be used to receive events over an AMQP receiver link in
     * either batching or streaming mode.
     * @param context -        The connection context corresponding to the EventHubClient instance
     * @param consumerGroup -  The consumer group from which the receiver should receive events from.
     * @param partitionId -    The Partition ID from which to receive.
     * @param eventPosition -  The position in the stream from where to start receiving events.
     * @param options -      Receiver options.
     */
    constructor(context, consumerGroup, partitionId, eventPosition, options = {}) {
        super(context, {
            partitionId: partitionId,
            name: context.config.getReceiverAddress(partitionId, consumerGroup),
        });
        /**
         * The sequence number of the most recently received AMQP message.
         */
        this._checkpoint = -1;
        /**
         * Indicates if messages are being received from this receiver.
         */
        this._isReceivingMessages = false;
        /**
         * Indicated if messages are being received in streaming mode.
         */
        this._isStreaming = false;
        /**
         * Denotes if close() was called on this receiver
         */
        this._isClosed = false;
        this.queue = [];
        this.consumerGroup = consumerGroup;
        this.address = context.config.getReceiverAddress(partitionId, this.consumerGroup);
        this.audience = context.config.getReceiverAudience(partitionId, this.consumerGroup);
        this.ownerLevel = options.ownerLevel;
        this.eventPosition = eventPosition;
        this.options = options;
        this.runtimeInfo = {};
    }
    /**
     * Returns sequenceNumber of the last event received from the service. This will not match the
     * last event received by `EventHubConsumer` when the `queue` is not empty
     * @readonly
     */
    get checkpoint() {
        return this._checkpoint;
    }
    /**
     * Indicates if messages are being received from this receiver.
     * @readonly
     */
    get isReceivingMessages() {
        return this._isReceivingMessages;
    }
    /**
     * Indicates if the receiver has been closed.
     */
    get isClosed() {
        return this._isClosed;
    }
    /**
     * The last enqueued event information. This property will only
     * be enabled when `trackLastEnqueuedEventProperties` option is set to true
     * @readonly
     */
    get lastEnqueuedEventProperties() {
        return this.runtimeInfo;
    }
    _onAmqpMessage(context) {
        if (!context.message) {
            return;
        }
        const data = fromRheaMessage(context.message, !!this.options.skipParsingBodyAsJson);
        const rawMessage = data.getRawAmqpMessage();
        const receivedEventData = {
            body: data.body,
            properties: data.properties,
            offset: data.offset,
            sequenceNumber: data.sequenceNumber,
            enqueuedTimeUtc: data.enqueuedTimeUtc,
            partitionKey: data.partitionKey,
            systemProperties: data.systemProperties,
            getRawAmqpMessage() {
                return rawMessage;
            },
        };
        if (data.correlationId != null) {
            receivedEventData.correlationId = data.correlationId;
        }
        if (data.contentType != null) {
            receivedEventData.contentType = data.contentType;
        }
        if (data.messageId != null) {
            receivedEventData.messageId = data.messageId;
        }
        this._checkpoint = receivedEventData.sequenceNumber;
        if (this.options.trackLastEnqueuedEventProperties && data) {
            this.runtimeInfo.sequenceNumber = data.lastSequenceNumber;
            this.runtimeInfo.enqueuedOn = data.lastEnqueuedTime;
            this.runtimeInfo.offset = data.lastEnqueuedOffset;
            this.runtimeInfo.retrievedOn = data.retrievalTime;
        }
        if (this._isStreaming) {
            this._addCredit(1);
        }
        this.queue.push(receivedEventData);
    }
    _onAmqpError(context) {
        const rheaReceiver = this._receiver || context.receiver;
        const amqpError = rheaReceiver && rheaReceiver.error;
        logger.verbose("[%s] 'receiver_error' event occurred on the receiver '%s' with address '%s'. " +
            "The associated error is: %O", this._context.connectionId, this.name, this.address, amqpError);
        if (this._onError && amqpError) {
            const error = translate(amqpError);
            logErrorStackTrace(error);
            this._onError(error);
        }
    }
    _onAmqpSessionError(context) {
        const sessionError = context.session && context.session.error;
        logger.verbose("[%s] 'session_error' event occurred on the session of receiver '%s' with address '%s'. " +
            "The associated error is: %O", this._context.connectionId, this.name, this.address, sessionError);
        if (this._onError && sessionError) {
            const error = translate(sessionError);
            logErrorStackTrace(error);
            this._onError(error);
        }
    }
    async _onAmqpClose(context) {
        const rheaReceiver = this._receiver || context.receiver;
        logger.verbose("[%s] 'receiver_close' event occurred on the receiver '%s' with address '%s'. " +
            "Value for isItselfClosed on the receiver is: '%s' " +
            "Value for isConnecting on the session is: '%s'.", this._context.connectionId, this.name, this.address, rheaReceiver ? rheaReceiver.isItselfClosed().toString() : undefined, this.isConnecting);
        if (rheaReceiver && !this.isConnecting) {
            // Call close to clean up timers & other resources
            await rheaReceiver.close().catch((err) => {
                logger.verbose("[%s] Error when closing receiver [%s] after 'receiver_close' event: %O", this._context.connectionId, this.name, err);
            });
        }
    }
    async _onAmqpSessionClose(context) {
        const rheaReceiver = this._receiver || context.receiver;
        logger.verbose("[%s] 'session_close' event occurred on the session of receiver '%s' with address '%s'. " +
            "Value for isSessionItselfClosed on the session is: '%s' " +
            "Value for isConnecting on the session is: '%s'.", this._context.connectionId, this.name, this.address, rheaReceiver ? rheaReceiver.isSessionItselfClosed().toString() : undefined, this.isConnecting);
        if (rheaReceiver && !this.isConnecting) {
            // Call close to clean up timers & other resources
            await rheaReceiver.close().catch((err) => {
                logger.verbose("[%s] Error when closing receiver [%s] after 'session_close' event: %O", this._context.connectionId, this.name, err);
            });
        }
    }
    abort() {
        var _a;
        // Cancellation is user-intended, so log to info instead of warning.
        logger.info(`[${this._context.connectionId}] The receive operation on the Receiver "${this.name}" with address "${this.address}" has been cancelled by the user.`);
        (_a = this._onError) === null || _a === void 0 ? void 0 : _a.call(this, new AbortError(StandardAbortMessage));
        return this.close();
    }
    /**
     * Clears the user-provided handlers and updates the receiving messages flag.
     */
    clearHandlers() {
        if (!this)
            return;
        this._onError = undefined;
        this._isReceivingMessages = false;
        this._isStreaming = false;
    }
    /**
     * Closes the underlying AMQP receiver.
     */
    async close() {
        this.clearHandlers();
        if (!this._receiver) {
            return;
        }
        const receiverLink = this._receiver;
        this._deleteFromCache();
        return this._closeLink(receiverLink)
            .catch((err) => {
            logger.warning(`[${this._context.connectionId}] An error occurred while closing receiver ${this.name}: ${err === null || err === void 0 ? void 0 : err.name}: ${err === null || err === void 0 ? void 0 : err.message}`);
            logErrorStackTrace(err);
            throw err;
        })
            .finally(() => {
            this._isClosed = true;
        });
    }
    /**
     * Determines whether the AMQP receiver link is open. If open then returns true else returns false.
     * @returns boolean
     */
    isOpen() {
        const result = Boolean(this._receiver && this._receiver.isOpen());
        logger.verbose("[%s] Receiver '%s' with address '%s' is open? -> %s", this._context.connectionId, this.name, this.address, result);
        return result;
    }
    _addCredit(credit) {
        var _a;
        (_a = this._receiver) === null || _a === void 0 ? void 0 : _a.addCredit(credit);
    }
    _deleteFromCache() {
        this._receiver = undefined;
        delete this._context.receivers[this.name];
        logger.verbose("[%s] Deleted the receiver '%s' from the client cache.", this._context.connectionId, this.name);
    }
    /**
     * Creates a new AMQP receiver under a new AMQP session.
     */
    async initialize({ abortSignal, timeoutInMs, }) {
        try {
            const isOpen = this.isOpen();
            if (this.isConnecting || isOpen) {
                logger.verbose("[%s] The receiver '%s' with address '%s' is open -> %s and is connecting -> %s. Hence not reconnecting.", this._context.connectionId, this.name, this.address, isOpen, this.isConnecting);
                return;
            }
            this.isConnecting = true;
            logger.verbose("[%s] The receiver '%s' with address '%s' is trying to connect", this._context.connectionId, this.name, this.address);
            // Wait for the connectionContext to be ready to open the link.
            await this._context.readyToOpenLink({ abortSignal });
            await this._negotiateClaim({ setTokenRenewal: false, abortSignal, timeoutInMs });
            const receiverOptions = {
                onClose: (context) => this._onAmqpClose(context),
                onError: (context) => this._onAmqpError(context),
                onMessage: (context) => this._onAmqpMessage(context),
                onSessionClose: (context) => this._onAmqpSessionClose(context),
                onSessionError: (context) => this._onAmqpSessionError(context),
            };
            if (this.checkpoint > -1) {
                receiverOptions.eventPosition = { sequenceNumber: this.checkpoint };
            }
            const options = this._createReceiverOptions(receiverOptions);
            logger.verbose("[%s] Trying to create receiver '%s' with options %O", this._context.connectionId, this.name, options);
            this._receiver = await this._context.connection.createReceiver(Object.assign(Object.assign({}, options), { abortSignal }));
            this.isConnecting = false;
            logger.verbose("[%s] Receiver '%s' created with receiver options: %O", this._context.connectionId, this.name, options);
            // store the underlying link in a cache
            this._context.receivers[this.name] = this;
            this._ensureTokenRenewal();
        }
        catch (err) {
            this.isConnecting = false;
            const error = translate(err);
            logger.warning("[%s] An error occurred while creating the receiver '%s': %s", this._context.connectionId, this.name, `${error === null || error === void 0 ? void 0 : error.name}: ${error === null || error === void 0 ? void 0 : error.message}`);
            logErrorStackTrace(err);
            throw error;
        }
    }
    /**
     * Creates the options that need to be specified while creating an AMQP receiver link.
     */
    _createReceiverOptions(options) {
        const receiverOptions = {
            name: this.name,
            autoaccept: true,
            source: {
                address: this.address,
            },
            credit_window: 0,
            onMessage: options.onMessage,
            onError: options.onError,
            onClose: options.onClose,
            onSessionError: options.onSessionError,
            onSessionClose: options.onSessionClose,
        };
        if (typeof this.ownerLevel === "number") {
            receiverOptions.properties = {
                [Constants.attachEpoch]: types.wrap_long(this.ownerLevel),
            };
        }
        if (this.options.trackLastEnqueuedEventProperties) {
            receiverOptions.desired_capabilities = Constants.enableReceiverRuntimeMetricName;
        }
        const eventPosition = options.eventPosition || this.eventPosition;
        if (eventPosition) {
            // Set filter on the receiver if event position is specified.
            const filterClause = getEventPositionFilter(eventPosition);
            if (filterClause) {
                receiverOptions.source.filter = {
                    "apache.org:selector-filter:string": types.wrap_described(filterClause, 0x468c00000004),
                };
            }
        }
        return receiverOptions;
    }
    /**
     * Returns a promise that resolves to an array of events received from the service.
     *
     * @param maxMessageCount - The maximum number of messages to receive.
     * @param maxWaitTimeInSeconds - The maximum amount of time to wait to build up the requested message count;
     * If not provided, it defaults to 60 seconds.
     * @param abortSignal - An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     *
     * @throws AbortError if the operation is cancelled via the abortSignal.
     * @throws MessagingError if an error is encountered while receiving a message.
     * @throws Error if the underlying connection or receiver has been closed.
     * Create a new EventHubConsumer using the EventHubClient createConsumer method.
     * @throws Error if the receiver is already receiving messages.
     */
    async receiveBatch(maxMessageCount, maxWaitTimeInSeconds = 60, abortSignal) {
        var _a;
        this._isReceivingMessages = true;
        this._isStreaming = false;
        const cleanupBeforeAbort = () => {
            logger.info(`[${this._context.connectionId}] The request operation on the Receiver "${this.name}" with address "${this.address}" has been cancelled by the user.`);
            return this.close();
        };
        const tryOpenLink = () => createAbortablePromise((resolve, reject) => this.initialize({
            abortSignal,
            timeoutInMs: getRetryAttemptTimeoutInMs(this.options.retryOptions),
        })
            .then(resolve)
            .catch(reject), {
            abortSignal,
            abortErrorMsg: StandardAbortMessage,
        });
        /** The time to wait in ms before attempting to read from the queue */
        const readIntervalWaitTimeInMs = 20;
        const retrieveEvents = () => {
            const eventsToRetrieveCount = Math.max(maxMessageCount - this.queue.length, 0);
            logger.verbose("[%s] Receiver '%s' already has %d events and wants to receive %d more events.", this._context.connectionId, this.name, this.queue.length, eventsToRetrieveCount);
            if (abortSignal === null || abortSignal === void 0 ? void 0 : abortSignal.aborted) {
                cleanupBeforeAbort();
                return Promise.reject(new AbortError(StandardAbortMessage));
            }
            return this._isClosed || this._context.wasConnectionCloseCalled
                ? Promise.resolve(this.queue.splice(0))
                : eventsToRetrieveCount === 0
                    ? Promise.resolve(this.queue.splice(0, maxMessageCount))
                    : new Promise((resolve, reject) => {
                        this._onError = (err) => {
                            if (err.name === "AbortError") {
                                cleanupBeforeAbort();
                            }
                            this.clearHandlers();
                            reject(err);
                        };
                        // eslint-disable-next-line promise/catch-or-return
                        tryOpenLink()
                            .then(() => {
                            var _a, _b;
                            // add credits
                            const existingCredits = (_b = (_a = this._receiver) === null || _a === void 0 ? void 0 : _a.credit) !== null && _b !== void 0 ? _b : 0;
                            const creditsToAdd = Math.max(eventsToRetrieveCount - existingCredits, 0);
                            this._addCredit(creditsToAdd);
                            logger.verbose("[%s] Setting the wait timer for %d seconds for receiver '%s'.", this._context.connectionId, maxWaitTimeInSeconds, this.name);
                            return; // to make eslint happy
                        })
                            .then(() => waitForEvents(maxMessageCount, maxWaitTimeInSeconds * 1000, readIntervalWaitTimeInMs, this.queue, {
                            abortSignal,
                            cleanupBeforeAbort,
                            receivedAfterWait: () => logger.info("[%s] Batching Receiver '%s', %d messages received within %d seconds.", this._context.connectionId, this.name, Math.min(maxMessageCount, this.queue.length), maxWaitTimeInSeconds),
                            receivedAlready: () => logger.info("[%s] Batching Receiver '%s', %d messages already received.", this._context.connectionId, this.name, maxMessageCount, maxWaitTimeInSeconds),
                            receivedNone: () => logger.info("[%s] Batching Receiver '%s', no messages received when max wait time in seconds %d is over.", this._context.connectionId, this.name, maxWaitTimeInSeconds),
                        }))
                            .catch(reject)
                            .then(resolve);
                    })
                        .then(() => this.queue.splice(0, maxMessageCount))
                        .finally(this.clearHandlers);
        };
        return retry(Object.defineProperties({
            operation: retrieveEvents,
            operationType: RetryOperationType.receiveMessage,
            abortSignal: abortSignal,
            retryOptions: (_a = this.options.retryOptions) !== null && _a !== void 0 ? _a : {},
        }, {
            connectionId: {
                enumerable: true,
                get: () => this._context.connectionId,
            },
            connectionHost: {
                enumerable: true,
                get: () => this._context.config.host,
            },
        }));
    }
}
function delay(waitTimeInMs, options) {
    let token;
    return createAbortablePromise((resolve) => {
        token = setTimeout(resolve, waitTimeInMs);
    }, options).finally(() => clearTimeout(token));
}
export function checkOnInterval(waitTimeInMs, check, options) {
    let token;
    return createAbortablePromise((resolve) => {
        token = setInterval(() => {
            if (check()) {
                resolve();
            }
        }, waitTimeInMs);
    }, options).finally(() => clearInterval(token));
}
/**
 * Returns a promise that will resolve when it is time to read from the queue
 * @param maxEventCount - The maximum number of events to receive.
 * @param maxWaitTimeInMs - The maximum time to wait in ms for the queue to contain any events.
 * @param readIntervalWaitTimeInMs - The time interval to wait in ms before checking the queue.
 * @param queue - The queue to read from.
 * @param options - The options bag.
 * @returns a promise that will resolve when it is time to read from the queue
 */
export function waitForEvents(maxEventCount, maxWaitTimeInMs, readIntervalWaitTimeInMs, queue, options) {
    const { abortSignal: clientAbortSignal, cleanupBeforeAbort, receivedNone, receivedAfterWait, receivedAlready, } = options !== null && options !== void 0 ? options : {};
    const aborter = new AbortController();
    const { signal: abortSignal } = new AbortController([
        aborter.signal,
        ...(clientAbortSignal ? [clientAbortSignal] : []),
    ]);
    const updatedOptions = {
        abortSignal,
        abortErrorMsg: StandardAbortMessage,
        cleanupBeforeAbort: () => {
            if (clientAbortSignal === null || clientAbortSignal === void 0 ? void 0 : clientAbortSignal.aborted) {
                cleanupBeforeAbort === null || cleanupBeforeAbort === void 0 ? void 0 : cleanupBeforeAbort();
            }
        },
    };
    return queue.length >= maxEventCount
        ? Promise.resolve().then(receivedAlready)
        : Promise.race([
            checkOnInterval(readIntervalWaitTimeInMs, () => queue.length > 0, updatedOptions)
                .then(() => delay(readIntervalWaitTimeInMs, updatedOptions))
                .then(receivedAfterWait),
            delay(maxWaitTimeInMs, updatedOptions).then(receivedNone),
        ]).finally(() => aborter.abort());
}
//# sourceMappingURL=eventHubReceiver.js.map