/******************************************************************************

                              Copyright (c) 2009
                            Lantiq Deutschland GmbH
                     Am Campeon 3; 85579 Neubiberg, Germany

  For licensing information, see the file 'LICENSE' in the root folder of
  this software module.

******************************************************************************/

/**
   \file drv_vmmc_sig_cptd.c
   Definitions of the functions for Call Progress Tone Detection module.
*/

#include "drv_api.h"
#include "drv_vmmc_sig_priv.h"
#include "drv_vmmc_api.h"

/* ============================= */
/* Local Macros & Definitions    */
/* ============================= */
/* default detection level tolerance, -20dB */
#define CPTD_DETECTION_LEVEL_TOLERANCE  200
/* allow -42dB total power in a pause step */
#define CPTCOEFF_POW_PAUSE       0x0002
/* allow -10dB maximum frequency power to total power ratio for valid tone
   detection */
#define CPTCOEFF_FP_TP_R_DEFAULT 0x075D

/*lint --esym( 750, CONST_CPT_HAMMING, CONST_CPT_BLACKMAN)
   One of both remains unused. */
/** hamming constant for cpt level calculation */
#define CONST_CPT_HAMMING        (  610)
/** blackman constant for cpt level calculation */
#define CONST_CPT_BLACKMAN       (-1580)
/** define which constant is taken for cpt level calculation */
#define CONST_CPT                CONST_CPT_BLACKMAN
/* return the maximum of two scalar values */
#define MAX(x,y) ((x) > (y) ? (x) : (y))

/* ============================= */
/* Local function declaration    */
/* ============================= */

static IFX_int32_t vmmc_sig_CPTD_SetCoeff (IFX_TAPI_TONE_SIMPLE_t const *pTone,
                                           RES_CPTD_COEF_t *pCmd,
                                           DSP_CPTD_FL_t *pFrameLength);
static IFX_int32_t vmmc_sig_CPTD_Conf (VMMC_CHANNEL *pCh,
                                       IFX_TAPI_TONE_SIMPLE_t const *pTone,
                                       IFX_int32_t signal,
                                       SIG_CPTD_CTRL_t *pCmd,
                                       DSP_CPTD_FL_t frameLength);

static IFX_int16_t vmmc_sig_CPTD_CalcFreq (IFX_uint32_t f);
static IFX_int16_t vmmc_sig_CPTD_CalcLevel (IFX_int32_t level);

/* ============================= */
/* Local function definitions    */
/* ============================= */

/**
   function to calculate the cpt frequency coefficients

   \param f - frequency in [Hz]
   \return
      cpt frequency coefficient
   \remarks
      none
*/
static IFX_int16_t vmmc_sig_CPTD_CalcFreq (IFX_uint32_t f)
{
   IFX_int16_t coef = 0;

   /*
      The formula to calculate the cpt frequency is f_coef = 2 * cos (
      2*pi*f[Hz] / 8000) * 2^14 = 2^15 * cos (pi*f[Hz] / 4000)

      Symmetrie: cos (phi) = sin (pi/2 - phi) --> we will use the taylor
      approximation for sin in the interval [pi/4 .. 3/4pi]

      with X = pi*f/4000

      Approximation with Taylor for cos in [-pi/4 .. +pi/4] = 2^15 * ( 1 - (X^2
      /2) + (X^4 /24) ) = 2^15 - (2^15 * X^2 / 2) + (2^15 * X^4 /24) = 2^15 -
      (2^15 * X^2 / 2) + A

      to ensure that f^4 does not overflow, use f/10 A = 2^15 * X^4 /24 = 2^15
      * (pi*f /4000)^4 /24 = 2^15 * (pi/4000)^4 * f^4 /24 = 2^15 * (pi/400)^4
      *(f/10)^4 /24

      Approximation with Taylor for sin in [-pi/4 .. +pi/4] = 2^15 * ( X - (X^3
      /6) + (X^5 /120) ) = (2^15 * X) - (2^15 * X^3 /6) + (2^15 * X^5 /120) =
      (2^15 * X) - (2^15 * X^3 /6) + B

      to ensure that f^5 does not overflow, use f/20 B = 2^15 * X^5 /120 = 2^15
      * (pi*f /4000)^5 / 120 = 2^15 * (pi/4000)^5 * f^5 / 120 = 2^15 *
      (pi*20/4000)^5 * (f/20)^5 /120 = 2^15 * (pi/200)^5 * (f/20)^5 /120 */

   if (f <= 1000)
   {
      /* cos approximation using the taylor algorithm */
      /* this formula covers the cos from 0 to pi/4 */
      coef =
         (IFX_int16_t) ((C2_15 - ((f * f * 1011) / 100000)) +
                        ((((f / 10) * (f / 10) * (f / 10) * (f / 10)) /
                          192487) - 1));
   }
   else if (f <= 2000)
   {
      /* sin approximation using the taylor algorithm */
      /* this formula covers the cos from pi/4 to pi/2 */
      f = 2000 - f;
      coef =
         (IFX_int16_t) (((25736 * f) / 1000 - (f * f * f / 377948)) +
                        ((f / 20) * (f / 20) * (f / 20) * (f / 20) * (f / 20) /
                         3829411));
   }
   else if (f <= 3000)
   {
      /* sin approximation using the taylor algorithm */
      /* this formula covers the cos from pi/2 to 3/4pi */
      f = f - 2000;
      coef =
         -(IFX_int16_t) (((((25736 * f) / 1000) -
                           (f * f * f / 377948)) +
                          ((f / 20) * (f / 20) * (f / 20) * (f / 20) *
                           (f / 20) / 3829411)));
   }
   else if (f <= 4000)
   {
      /* cos approximation using the taylor algorithm */
      /* this formula covers the cos from 3/4 pi to pi */
      f = 4000 - f;
      coef =
         -(IFX_int16_t) (((C2_15 - ((f * f * 1011) / 100000)) +
                          ((f / 10) * (f / 10) * (f / 10) * (f / 10) /
                           192487)));
   }
   else
   {
      VMMC_ASSERT (IFX_FALSE);
   }

   return coef;
}

/**
   function to calculate the cpt level coefficient

   \param level - level in [0.1 dB]
   \return
      cpt level coefficient
   \remarks
      CONST_CPT has to be set by define to
      CONST_CPT_HAMMING or CONST_CPT_BLACKMAN
*/
static IFX_int16_t vmmc_sig_CPTD_CalcLevel (IFX_int32_t level)
{
   long lvl;
   IFX_uint32_t tenExp;
   IFX_uint32_t shift;
   IFX_int32_t exp, i;

   /* calculate the desired level in mdB and the exponent */
   lvl = level * 100;
   exp = (IFX_int32_t) (-lvl - CONST_CPT) / 10;

   /* set the initial shift factor according to the level */
   if (lvl <= -20000)
      shift = 1000;
   else if (lvl <= -10000)
      shift = 10000;
   else
      shift = 10000;

   /* the initial value of the result (tenExp) is the shift factor which will
      be compensated in the last step (calculating the coefficient) return
      ((xxxxx * shift) / tenExp); */
   tenExp = shift;

   /* go over all elements in the tens array, starting with the largest entry
      and ... */
   for (i = 27; i >= 0; i--)
   {
      /* ... loop while the current tens[i][0] is part of the remaining exp */
      while (exp >= ((IFX_int32_t) tens[i][0]))
      {
         /* calculate part of the result for tens[i][0], check which accuracy
            of the tens[i][1] can be used to multiply tenExp without overflow */
         if ((C2_31 / tenExp) > tens[i][1])
         {
            /* use the unscaled tens[i][1] value to calculate the tenExp share */
            tenExp *= tens[i][1];
            tenExp /= 100000;
         }
         else if ((C2_31 / tenExp) > ROUND_DIV10 (tens[i][1]))
         {
            /* scale the tens[i][1] value by 10 to calculate the tenExp share */
            tenExp *= ROUND_DIV10 (tens[i][1]);
            tenExp /= 10000;
         }
         else if ((C2_31 / tenExp) > ROUND_DIV100 (tens[i][1]))
         {
            /* scale the tens[i][1] value by 100 to calculate the tenExp share */
            tenExp *= ROUND_DIV100 (tens[i][1]);
            tenExp /= 1000;
         }
         else
         {
            /* scale the tens[i][1] value by 1000 to calculate the tenExp share
             */
            tenExp *= ROUND_DIV1000 (tens[i][1]);
            tenExp /= 100;
         }

         /* calculate the remaining exp, i.e. subtract that part of exponent
            that has been calculated in this step */
         exp -= ((IFX_int32_t) tens[i][0]);
      }
   }

   /* calculate the coefficient according to the specification... */
   return (IFX_int16_t) ROUND (((C2_14 * shift) / (tenExp)));
}

/**
   Start the call progress tone detector (CPTD)

   \param pLLChannel Handle to TAPI low level channel structure
   \param pTone    Handle to the simple tone structure
   \param signal   the type of signal and direction to detect.
         - 1: Receive path (downstream)
         - 2: Transmit path (upstream)

   \return
      - VMMC_statusCmdWr Writing the command failed
      - VMMC_statusFuncParm Wrong parameters passed. This code is returned
        when the signal does not contain rx or tx or is set to both.
      - VMMC_statusNotSupported Requested action is not supported. This code
        is returned when a WLEC is requested but not supported by the VoFW.
      - VMMC_statusNoRes No free resources to playout the CID is available.
   - VMMC_statusOk if successful

   \remarks
   If the CPTD is already running it is automatically stopped, configured and
   then started again.
*/
IFX_return_t VMMC_TAPI_LL_SIG_CPTD_Start (IFX_TAPI_LL_CH_t *pLLChannel,
                                         IFX_TAPI_TONE_SIMPLE_t const *pTone,
                                         IFX_int32_t signal)
{
#if (VMMC_CFG_FEATURES & VMMC_FEAT_VOICE)
   VMMC_CHANNEL *pCh = (VMMC_CHANNEL *) pLLChannel;
   VMMC_DEVICE *pDev = pCh->pParent;
   SIG_CPTD_CTRL_t *p_fw_sig_cptd = &pCh->pSIG->fw_sig_cptd;
   IFX_int32_t ret = IFX_ERROR;
   IFX_uint8_t ch = pCh->nChannel - 1;
   RES_CPTD_COEF_t CoefCmd;
   DSP_CPTD_FL_t frameLength;

   /* if CPTD detector is running stop it first before configuring it */
   if (p_fw_sig_cptd->EN != 0)
   {
      /* disable CPTD */
      p_fw_sig_cptd->EN = 0;
      ret = CmdWrite (pDev, (IFX_uint32_t *)p_fw_sig_cptd, SIG_CPTD_CTRL_LEN);
   }

   /* set CPTD coefficients ************************************************* */
   ret = VMMC_SIG_AutoChStop (pCh, IFX_TRUE);
   if (ret == IFX_SUCCESS)
   {
      memset (&CoefCmd, 0, sizeof (RES_CPTD_COEF_t));
      CoefCmd.CMD = CMD_EOP;
      CoefCmd.CHAN = ch;
      CoefCmd.MOD = MOD_RESOURCE;
      CoefCmd.ECMD = RES_CPTD_COEF_ECMD;

      /* nFrameSize is depending on the tone cadence and will be decided
         inside vmmc_sig_CPTD_SetCoeff */
      vmmc_sig_CPTD_SetCoeff (pTone, &CoefCmd, &frameLength);
      /* write CPTD coefficients */
      ret = CmdWrite (pDev, (IFX_uint32_t *)&CoefCmd, RES_CPTD_COEF_LEN);
   }

   /* activate CPTD ********************************************************* */
   if (ret == IFX_SUCCESS)
   {
      /* on activation of the CPTD also nFrameSize is configured as
         determined above... */
      /*lint -e{644} to get here also the code above must have been executed
                     that sets the frameLength variable. */
      ret = vmmc_sig_CPTD_Conf (pCh, pTone, signal, p_fw_sig_cptd, frameLength);
   }
   if (ret == IFX_SUCCESS)
   {
      /* enable CPTD */
      ret = CmdWrite (pDev, (IFX_uint32_t *)p_fw_sig_cptd, SIG_CPTD_CTRL_LEN);
   }
   return ret;
#else
   return IFX_ERROR;
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_VOICE) */
}

/**
   Stop the call progress tone detector (CPTD)

   \param pLLChannel Handle to TAPI low level channel structure

   \return
      - VMMC_statusCmdWr Writing the command failed
      - VMMC_statusOk if successful
*/
IFX_return_t VMMC_TAPI_LL_SIG_CPTD_Stop (IFX_TAPI_LL_CH_t *pLLChannel)
{
   VMMC_CHANNEL *pCh = (VMMC_CHANNEL *) pLLChannel;
   VMMC_DEVICE *pDev = pCh->pParent;
   SIG_CPTD_CTRL_t *p_fw_sig_cptd = &pCh->pSIG->fw_sig_cptd;
   IFX_int32_t ret = VMMC_statusOk;

   /* Only stop detector if it is currently running */
   if (p_fw_sig_cptd->EN != 0)
   {
      /* disable CPTD */
      p_fw_sig_cptd->EN = 0;

      ret = CmdWrite (pDev, (IFX_uint32_t *)p_fw_sig_cptd, SIG_CPTD_CTRL_LEN);
      if (ret == IFX_SUCCESS)
      {
         ret = VMMC_SIG_AutoChStop (pCh, IFX_FALSE);
      }
   }

   return ret;
}


/**
   Configures the call progress tone detector

   \param  pCh          Pointer to VMMC channel structure.
   \param  pTone        pointer to the simple tone to detect.
   \param  signal       Signal as specified in IFX_TAPI_TONE_CPTD_DIRECTION_t.
   \param  pCmd         Pointer to the command structure to fill.
   \param  frameLength  Selects the frame length.

   \return
   IFX_SUCCESS or IFX_ERROR
*/
static IFX_int32_t vmmc_sig_CPTD_Conf (VMMC_CHANNEL *pCh,
                                       IFX_TAPI_TONE_SIMPLE_t const *pTone,
                                       IFX_int32_t signal,
                                       SIG_CPTD_CTRL_t *pCmd,
                                       DSP_CPTD_FL_t frameLength)
{
   /* set frame length */
   pCmd->FL = frameLength;
   /* set total power */
   pCmd->TP = DSP_CPTD_TP_250_3400_HZ;
   /* select window type */
   pCmd->WS = DSP_CPTD_WS_HAMMING;
   /* set resource number */
   pCmd->CPTNR = (pCh->nChannel - 1);
   /* enable CPTD */
   pCmd->EN = 1;

   /* check for continuos tone */
   if (pTone->cadence[1] == 0)
   {
      pCmd->CNT = 1;
   }
   switch (signal)
   {
   case IFX_TAPI_TONE_CPTD_DIRECTION_RX:
      /* for receive path input I2 is used */
      pCmd->IS = SIG_CPTD_CTRL_IS_I2;
      break;
   case IFX_TAPI_TONE_CPTD_DIRECTION_TX:
      pCmd->IS = SIG_CPTD_CTRL_IS_I1;
      break;
   default:
      SET_ERROR (VMMC_ERR_FUNC_PARM);
      return IFX_ERROR;
   }

   return IFX_SUCCESS;
}


/**
   Configures call progress tone detection coefficients.

   \param  pTone        Pointer to internal simple tone table entry.
   \param  pCmd         Coefficient command to fill.
   \param  pFrameLength Returns which frame length was selected.

   \return
   IFX_SUCCESS
*/
static IFX_int32_t vmmc_sig_CPTD_SetCoeff (IFX_TAPI_TONE_SIMPLE_t const *pTone,
                                           RES_CPTD_COEF_t *pCmd,
                                           DSP_CPTD_FL_t *pFrameLength)
{
   IFX_uint8_t nr = 0;
   IFX_uint8_t i;
   IFX_uint16_t val;
   unsigned int nTenPercentTimeTolerance,
                nMaxTenPercentTimeTolerance = 0,
                nAbsoluteTimeTolerance;

   /* set frame length, in case no step is <= 200ms we use a frame lenth of
      32ms, otherwise 16ms */
   *pFrameLength = DSP_CPTD_FL_32;
   for (i = 0; i < IFX_TAPI_TONE_STEPS_MAX; ++i)
   {
      /* check for last cadence step */
      if (pTone->cadence[i] == 0)
         break;
      /* if one of the steps is shorter or eaqual to 200ms we reduce the frame
         length to 16 ms */
      if (pTone->cadence[i] <= 200)
         *pFrameLength = DSP_CPTD_FL_16;
   }

   /* set nAbsoluteTimeTolerance to roughly 1.2 times the frame length */
   switch (*pFrameLength)
   {
      case DSP_CPTD_FL_16:
         nAbsoluteTimeTolerance =  40;
         break;
      case DSP_CPTD_FL_64:
         nAbsoluteTimeTolerance = 160;
         break;
      case DSP_CPTD_FL_32:
      default:
         nAbsoluteTimeTolerance =  80;
         break;
   }

   /* default settings */
   /* program allowed twist to 6 dB */
   pCmd->TWIST_12 = 0x20;
   pCmd->TWIST_34 = 0x20;
   pCmd->POW_PAUSE = CPTCOEFF_POW_PAUSE;
   pCmd->FP_TP_R = CPTCOEFF_FP_TP_R_DEFAULT;

   /* set frequency and level A for F_1. Freq A is always set. Tone API assures
      it, the detection level is always below the defined tone. */
   pCmd->GOE_1 = (IFX_uint16_t) vmmc_sig_CPTD_CalcFreq (pTone->freqA);
   pCmd->LEV_1 =
      (IFX_uint16_t) vmmc_sig_CPTD_CalcLevel (pTone->levelA -
                                              CPTD_DETECTION_LEVEL_TOLERANCE);
   /* set frequency and level B for F_2 */
   if (pTone->freqB)
   {
      pCmd->GOE_2 = (IFX_uint16_t) vmmc_sig_CPTD_CalcFreq (pTone->freqB);
      pCmd->LEV_2 =
         (IFX_uint16_t) vmmc_sig_CPTD_CalcLevel (pTone->levelB -
                                                 CPTD_DETECTION_LEVEL_TOLERANCE);
   }
   else
   {
      pCmd->GOE_2 = RES_CPTD_COEF_FX_0HZ;
      pCmd->LEV_2 = RES_CPTD_COEF_LEVEL_0DB;
   }
   /* set frequency and level C for F_3 */
   if (pTone->freqC)
   {
      pCmd->GOE_3 = (IFX_uint16_t) vmmc_sig_CPTD_CalcFreq (pTone->freqC);
      pCmd->LEV_3 =
         (IFX_uint16_t) vmmc_sig_CPTD_CalcLevel (pTone->levelC -
                                                 CPTD_DETECTION_LEVEL_TOLERANCE);
   }
   else
   {
      pCmd->GOE_3 = RES_CPTD_COEF_FX_0HZ;
      pCmd->LEV_3 = RES_CPTD_COEF_LEVEL_0DB;
   }
   /* set frequency and level D for F_4 */
   if (pTone->freqD)
   {
      pCmd->GOE_4 = (IFX_uint16_t) vmmc_sig_CPTD_CalcFreq (pTone->freqD);
      pCmd->LEV_4 =
         (IFX_uint16_t) vmmc_sig_CPTD_CalcLevel (pTone->levelD -
                                                 CPTD_DETECTION_LEVEL_TOLERANCE);
   }
   else
   {
      pCmd->GOE_4 = RES_CPTD_COEF_FX_0HZ;
      pCmd->LEV_4 = RES_CPTD_COEF_LEVEL_0DB;
   }

   /* set step times: T_x */
   for (i = 0; i < 4; ++i)
   {
      /* check for last cadence step */
      if (pTone->cadence[i] == 0)
         break;

      /* to allow +/- 10% deviation in the time, we'll reduce each time by 10%
         and program +TIM_TOL to 2 * MAX ( cadence [i] / 10 ). In addition we
         check if the tolerance for each step as well as +TIM_TOL is smaller
         than nAbsoluteTimeTolerance, if so we use the latter one. */
      nTenPercentTimeTolerance = pTone->cadence[i] / 10;
      /* limit the time tolerance to 8 bit / 2 */
      if (nTenPercentTimeTolerance > 127)
         nTenPercentTimeTolerance = 127;
      if (nTenPercentTimeTolerance > nMaxTenPercentTimeTolerance)
         nMaxTenPercentTimeTolerance = nTenPercentTimeTolerance;

      val = (IFX_uint16_t) (pTone->cadence[i] - MAX(nAbsoluteTimeTolerance,
                                                    nTenPercentTimeTolerance));
      switch (i)
      {
         case 0:
            pCmd->T_1 = val;
            break;
         case 1:
            pCmd->T_2 = val;
            break;
         case 2:
            pCmd->T_3 = val;
            break;
         case 3:
            pCmd->T_4 = val;
            break;
         default:
            /* to make the compiler happy -- do nothing */
            break;
      }

      /* if no freq is selected in this step, activate pause detection */
      if ((pTone->frequencies[i] & IFX_TAPI_TONE_FREQALL) == 0)
      {
         val = RES_CPTD_COEF_P;
      }
      else
      {
         /* set mask for MSK_i - use frequency A. Initialize the field */
         if (pTone->frequencies[i] & IFX_TAPI_TONE_FREQA)
            val = RES_CPTD_COEF_F1;
         else
            val = 0;
         /* set mask for MSK_i - use frequency B */
         if (pTone->frequencies[i] & IFX_TAPI_TONE_FREQB)
            val |= RES_CPTD_COEF_F2;
         /* set mask for MSK_i - use frequency C */
         if (pTone->frequencies[i] & IFX_TAPI_TONE_FREQC)
            val |= RES_CPTD_COEF_F3;
         /* set mask for MSK_i - use frequency D */
         if (pTone->frequencies[i] & IFX_TAPI_TONE_FREQD)
            val |= RES_CPTD_COEF_F4;
      }
      switch (i)
      {
         case 0:
            pCmd->MSK_1 = val;
            break;
         case 1:
            pCmd->MSK_2 = val;
            break;
         case 2:
            pCmd->MSK_3 = val;
            break;
         case 3:
            pCmd->MSK_4 = val;
            break;
         default:
            /* to make the compiler happy -- do nothing */
            break;
      }
      nr++;
   }
   if (pTone->cadence[1] == 0 && pTone->pause == 0)
   {
      /* set tolerance +TIM_TOL */
      pCmd->TIM_TOL =
                (2 * MAX (nAbsoluteTimeTolerance, nMaxTenPercentTimeTolerance));
      /* return because nothing more to do */
      return IFX_SUCCESS;
   }
   /* set end of steps */
   switch (i)
   {
      case 1:
         pCmd->MSK_1 |= RES_CPTD_COEF_E;
         break;
      case 2:
         pCmd->MSK_2 |= RES_CPTD_COEF_E;
         break;
      case 3:
         pCmd->MSK_3 |= RES_CPTD_COEF_E;
         break;
      case 4:
         pCmd->MSK_4 |= RES_CPTD_COEF_E;
         break;
      default:
         /* to make the compiler happy -- do nothing */
         break;
   }

   /* set tolerance +TIM_TOL and NR of successfully fulfilled timing
      requirement steps required to pass */
   pCmd->TIM_TOL =
                (2 * MAX (nAbsoluteTimeTolerance, nMaxTenPercentTimeTolerance));
   pCmd->NR = nr;

   return IFX_SUCCESS;
}
