/******************************************************************************

                              Copyright (c) 2009
                            Lantiq Deutschland GmbH
                     Am Campeon 3; 85579 Neubiberg, Germany

  For licensing information, see the file 'LICENSE' in the root folder of
  this software module.

******************************************************************************
   Module      : drv_vmmc_bbd.c
   Desription  :
*******************************************************************************/

/* ============================= */
/* Includes                      */
/* ============================= */
#include "drv_vmmc_api.h"
#include "drv_vmmc_bbd.h"
#include "drv_vmmc_alm.h"
#include "drv_vmmc_alm_priv.h"      /* required to access bIdleExtNeedRestore */

#if (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO)
#include "drv_vmmc_audio.h"
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO) */

#include "ifxos_time.h"

/* ============================= */
/* Local defines                 */
/* ============================= */

#define VMMC_CMD_MOD_ECMD_MASK      0x1F00FF00
#define VMMC_WL_SDD_BASIC_CFG       0x04000400
#define VMMC_WL_SDD_RING_CFG        0x04000500
#define VMMC_WL_SDD_DCDC_CFG        0x04000C00

#define IDLE_EXT_TOGGLE_SLEEP_MS    5

/* maximum of downloads per BBD block,
   can be set also at compile time  */
#ifndef BBD_VMMC_BLOCK_MAXDWNLD
#define BBD_VMMC_BLOCK_MAXDWNLD              10
#endif /* BBD_VIN_BLOCK_MAXDWNLD */

/* ============================= */
/* BBD tag definitions           */
/* ============================= */
#if defined SYSTEM_DANUBE
#define BBD_VMMC_MAGIC                       0x766D6D63 /* "vmmc" */
#elif defined(SYSTEM_AR9)
#define BBD_VMMC_MAGIC                       0x41523921 /* "AR9"  */
#elif defined(SYSTEM_VR9)
#define BBD_VMMC_MAGIC                       0x56523921 /* "VR9"  */
#else
#error system undefined
#endif

/** BBD blocks tags */
/* 0x1XXX tags : DC/AC Coefficients */
#define BBD_COMPATIBILITY_BLOCK_TAG          0x000C
#define BBD_VMMC_CRAM_BLOCK                  0x1001
#define BBD_VMMC_SLIC_BLOCK                  0x1002
#define BBD_VMMC_RING_CFG_BLOCK              0x1003
#define BBD_VMMC_DC_THRESHOLDS_BLOCK         0x1004
#define BBD_VMMC_VOFW_COEFF                  0x1005
#define BBD_VMMC_CRAM_NB_DATA                0x1006
#define BBD_VMMC_CRAM_WB_DATA                0x1007

/** SLIC TYPE Defines */
#define BBD_VMMC_SLIC_TYPE_DC                0x0101
#define BBD_VMMC_SLIC_TYPE_E                 0x0201

/** Transparent Download block */
#define BBD_VMMC_TRANSPARENT_BLOCK           0x1200

/** FXO CRAM block */
#define BBD_VMMC_FXO_CRAM_BLOCK              0x1010


/* ============================= */
/* Local variable definition     */
/* ============================= */

/* ============================= */
/* Global variable definition    */
/* ============================= */

/* registration of supported VMMC bbd blocks
   downloadable channelwise */
static const IFX_uint16_t VMMC_CH_BBD_Blocks[] =
{
   BBD_VMMC_CRAM_BLOCK,
   BBD_VMMC_SLIC_BLOCK,
   BBD_VMMC_RING_CFG_BLOCK,
   BBD_VMMC_DC_THRESHOLDS_BLOCK,
   BBD_VMMC_VOFW_COEFF,
   BBD_VMMC_CRAM_NB_DATA,
   BBD_VMMC_CRAM_WB_DATA,
   BBD_VMMC_TRANSPARENT_BLOCK,
   BBD_VMMC_FXO_CRAM_BLOCK

#if (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO)
,
   BBD_VMMC_IPPHONE_IDLE,
   BBD_VMMC_IPPHONE_HANDS_VOL1,
   BBD_VMMC_IPPHONE_HANDS_VOL2,
   BBD_VMMC_IPPHONE_HANDS_VOL3,
   BBD_VMMC_IPPHONE_HANDS_VOL4,
   BBD_VMMC_IPPHONE_HANDS_VOL5,
   BBD_VMMC_IPPHONE_HANDS_VOL6,
   BBD_VMMC_IPPHONE_HANDS_VOL7,
   BBD_VMMC_IPPHONE_HANDS_VOL8,

   BBD_VMMC_IPPHONE_HEADS_VOL1,
   BBD_VMMC_IPPHONE_HEADS_VOL2,
   BBD_VMMC_IPPHONE_HEADS_VOL3,
   BBD_VMMC_IPPHONE_HEADS_VOL4,
   BBD_VMMC_IPPHONE_HEADS_VOL5,
   BBD_VMMC_IPPHONE_HEADS_VOL6,
   BBD_VMMC_IPPHONE_HEADS_VOL7,
   BBD_VMMC_IPPHONE_HEADS_VOL8,

   BBD_VMMC_IPPHONE_HANDSF_MUFFLED_VOL1,
   BBD_VMMC_IPPHONE_HANDSF_MUFFLED_VOL2,
   BBD_VMMC_IPPHONE_HANDSF_MUFFLED_VOL3,
   BBD_VMMC_IPPHONE_HANDSF_MUFFLED_VOL4,
   BBD_VMMC_IPPHONE_HANDSF_MUFFLED_VOL5,
   BBD_VMMC_IPPHONE_HANDSF_MUFFLED_VOL6,
   BBD_VMMC_IPPHONE_HANDSF_MUFFLED_VOL7,
   BBD_VMMC_IPPHONE_HANDSF_MUFFLED_VOL8,

   BBD_VMMC_IPPHONE_HANDSF_NORMAL_VOL1,
   BBD_VMMC_IPPHONE_HANDSF_NORMAL_VOL2,
   BBD_VMMC_IPPHONE_HANDSF_NORMAL_VOL3,
   BBD_VMMC_IPPHONE_HANDSF_NORMAL_VOL4,
   BBD_VMMC_IPPHONE_HANDSF_NORMAL_VOL5,
   BBD_VMMC_IPPHONE_HANDSF_NORMAL_VOL6,
   BBD_VMMC_IPPHONE_HANDSF_NORMAL_VOL7,
   BBD_VMMC_IPPHONE_HANDSF_NORMAL_VOL8,

   BBD_VMMC_IPPHONE_HANDSF_ECHOIC_VOL1,
   BBD_VMMC_IPPHONE_HANDSF_ECHOIC_VOL2,
   BBD_VMMC_IPPHONE_HANDSF_ECHOIC_VOL3,
   BBD_VMMC_IPPHONE_HANDSF_ECHOIC_VOL4,
   BBD_VMMC_IPPHONE_HANDSF_ECHOIC_VOL5,
   BBD_VMMC_IPPHONE_HANDSF_ECHOIC_VOL6,
   BBD_VMMC_IPPHONE_HANDSF_ECHOIC_VOL7,
   BBD_VMMC_IPPHONE_HANDSF_ECHOIC_VOL8,

   BBD_VMMC_IPPHONE_HANDS_OL_VOL1,
   BBD_VMMC_IPPHONE_HANDS_OL_VOL2,
   BBD_VMMC_IPPHONE_HANDS_OL_VOL3,
   BBD_VMMC_IPPHONE_HANDS_OL_VOL4,
   BBD_VMMC_IPPHONE_HANDS_OL_VOL5,
   BBD_VMMC_IPPHONE_HANDS_OL_VOL6,
   BBD_VMMC_IPPHONE_HANDS_OL_VOL7,
   BBD_VMMC_IPPHONE_HANDS_OL_VOL8,

   BBD_VMMC_IPPHONE_HEADS_OL_VOL1,
   BBD_VMMC_IPPHONE_HEADS_OL_VOL2,
   BBD_VMMC_IPPHONE_HEADS_OL_VOL3,
   BBD_VMMC_IPPHONE_HEADS_OL_VOL4,
   BBD_VMMC_IPPHONE_HEADS_OL_VOL5,
   BBD_VMMC_IPPHONE_HEADS_OL_VOL6,
   BBD_VMMC_IPPHONE_HEADS_OL_VOL7,
   BBD_VMMC_IPPHONE_HEADS_OL_VOL8,

   BBD_VMMC_IPPHONE_HANDS_WB_VOL1,
   BBD_VMMC_IPPHONE_HANDS_WB_VOL2,
   BBD_VMMC_IPPHONE_HANDS_WB_VOL3,
   BBD_VMMC_IPPHONE_HANDS_WB_VOL4,
   BBD_VMMC_IPPHONE_HANDS_WB_VOL5,
   BBD_VMMC_IPPHONE_HANDS_WB_VOL6,
   BBD_VMMC_IPPHONE_HANDS_WB_VOL7,
   BBD_VMMC_IPPHONE_HANDS_WB_VOL8,

   BBD_VMMC_IPPHONE_HEADS_WB_VOL1,
   BBD_VMMC_IPPHONE_HEADS_WB_VOL2,
   BBD_VMMC_IPPHONE_HEADS_WB_VOL3,
   BBD_VMMC_IPPHONE_HEADS_WB_VOL4,
   BBD_VMMC_IPPHONE_HEADS_WB_VOL5,
   BBD_VMMC_IPPHONE_HEADS_WB_VOL6,
   BBD_VMMC_IPPHONE_HEADS_WB_VOL7,
   BBD_VMMC_IPPHONE_HEADS_WB_VOL8,

   BBD_VMMC_IPPHONE_HANDSF_WB_MUFFLED_VOL1,
   BBD_VMMC_IPPHONE_HANDSF_WB_MUFFLED_VOL2,
   BBD_VMMC_IPPHONE_HANDSF_WB_MUFFLED_VOL3,
   BBD_VMMC_IPPHONE_HANDSF_WB_MUFFLED_VOL4,
   BBD_VMMC_IPPHONE_HANDSF_WB_MUFFLED_VOL5,
   BBD_VMMC_IPPHONE_HANDSF_WB_MUFFLED_VOL6,
   BBD_VMMC_IPPHONE_HANDSF_WB_MUFFLED_VOL7,
   BBD_VMMC_IPPHONE_HANDSF_WB_MUFFLED_VOL8,

   BBD_VMMC_IPPHONE_HANDSF_WB_NORMAL_VOL1,
   BBD_VMMC_IPPHONE_HANDSF_WB_NORMAL_VOL2,
   BBD_VMMC_IPPHONE_HANDSF_WB_NORMAL_VOL3,
   BBD_VMMC_IPPHONE_HANDSF_WB_NORMAL_VOL4,
   BBD_VMMC_IPPHONE_HANDSF_WB_NORMAL_VOL5,
   BBD_VMMC_IPPHONE_HANDSF_WB_NORMAL_VOL6,
   BBD_VMMC_IPPHONE_HANDSF_WB_NORMAL_VOL7,
   BBD_VMMC_IPPHONE_HANDSF_WB_NORMAL_VOL8,

   BBD_VMMC_IPPHONE_HANDSF_WB_ECHOIC_VOL1,
   BBD_VMMC_IPPHONE_HANDSF_WB_ECHOIC_VOL2,
   BBD_VMMC_IPPHONE_HANDSF_WB_ECHOIC_VOL3,
   BBD_VMMC_IPPHONE_HANDSF_WB_ECHOIC_VOL4,
   BBD_VMMC_IPPHONE_HANDSF_WB_ECHOIC_VOL5,
   BBD_VMMC_IPPHONE_HANDSF_WB_ECHOIC_VOL6,
   BBD_VMMC_IPPHONE_HANDSF_WB_ECHOIC_VOL7,
   BBD_VMMC_IPPHONE_HANDSF_WB_ECHOIC_VOL8,

   BBD_VMMC_IPPHONE_HANDS_OL_WB_VOL1,
   BBD_VMMC_IPPHONE_HANDS_OL_WB_VOL2,
   BBD_VMMC_IPPHONE_HANDS_OL_WB_VOL3,
   BBD_VMMC_IPPHONE_HANDS_OL_WB_VOL4,
   BBD_VMMC_IPPHONE_HANDS_OL_WB_VOL5,
   BBD_VMMC_IPPHONE_HANDS_OL_WB_VOL6,
   BBD_VMMC_IPPHONE_HANDS_OL_WB_VOL7,
   BBD_VMMC_IPPHONE_HANDS_OL_WB_VOL8,

   BBD_VMMC_IPPHONE_HEADS_OL_WB_VOL1,
   BBD_VMMC_IPPHONE_HEADS_OL_WB_VOL2,
   BBD_VMMC_IPPHONE_HEADS_OL_WB_VOL3,
   BBD_VMMC_IPPHONE_HEADS_OL_WB_VOL4,
   BBD_VMMC_IPPHONE_HEADS_OL_WB_VOL5,
   BBD_VMMC_IPPHONE_HEADS_OL_WB_VOL6,
   BBD_VMMC_IPPHONE_HEADS_OL_WB_VOL7,
   BBD_VMMC_IPPHONE_HEADS_OL_WB_VOL8,

   BBD_VMMC_IPPHONE_RINGTONE_VOL1,
   BBD_VMMC_IPPHONE_RINGTONE_VOL2,
   BBD_VMMC_IPPHONE_RINGTONE_VOL3,
   BBD_VMMC_IPPHONE_RINGTONE_VOL4,
   BBD_VMMC_IPPHONE_RINGTONE_VOL5,
   BBD_VMMC_IPPHONE_RINGTONE_VOL6,
   BBD_VMMC_IPPHONE_RINGTONE_VOL7,
   BBD_VMMC_IPPHONE_RINGTONE_VOL8
#endif  /*  VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO */
};

#if 0
/* AC DSP start addresses for ch 0 and ch 1 */
enum
{
   VMMC_ACDSP_STARTADDR_CH0 =  0x1000,
   VMMC_ACDSP_STARTADDR_CH1 =  0x2000
};

/* download flags */
enum
{
   /* user has set BBD ptr in init structure */
   VMMC_IO_BBD  = 0x01,
   /* user has set PRAM fw ptr in init structure */
   VMMC_IO_PRAM = 0x02,
   /* user has set AC ptr in init structure */
   VMMC_IO_AC   = 0x08,
   /* user has set CRAM ptr in init structure */
   VMMC_IO_CRAM = 0x10
};
#endif


/* ============================= */
/* Local function definition     */
/* ============================= */

static IFX_int32_t vmmc_BBD_VoFWCoeff        (VMMC_CHANNEL *pCh, bbd_block_t *pBBDblock);
static IFX_int32_t vmmc_BBD_WhiteListedCmdWr (VMMC_CHANNEL *pCh, bbd_block_t *pBBDblock);

static IFX_int32_t vmmc_BBD_RingCfg  (VMMC_CHANNEL   *pCh,
                                      IFX_boolean_t   bBroadcast,
                                      VMMC_RingCfg_t *p_ringCfg);
static IFX_int32_t vmmc_BBD_SetDcThr (VMMC_CHANNEL *pCh,
                                      IFX_boolean_t bBroadcast,
                                      VMMC_DcThr_t *p_dcThr);
static IFX_int32_t vmmc_BBD_SetSlic  (VMMC_CHANNEL  *pCh,
                                      IFX_boolean_t  bBroadcast,
                                      IFX_uint16_t   slic_val);
static IFX_int32_t vmmc_BBD_DownloadChCram  (VMMC_CHANNEL *pCh,
                                             bbd_block_t  *bbd_cram);
static IFX_int32_t vmmc_BBD_UpdateBasicConfigCache (VMMC_CHANNEL *pCh);
static IFX_int32_t vmmc_BBD_PrepareIdleExtBit (VMMC_CHANNEL *pCh);
static IFX_int32_t vmmc_BBD_RestoreIdleExtBit (VMMC_CHANNEL *pCh);
static IFX_int32_t vmmc_BBD_UpdateIdleExtBit (VMMC_CHANNEL *pCh,
                                              IFX_uint8_t val,
                                              IFX_boolean_t *pbUpdated);

#ifdef VMMC_BBD_DEBUG
static IFX_void_t print_bbd_format(bbd_format_t *pBBD)
{
   IFX_uint32_t i;

   TRACE(VMMC, DBG_LEVEL_HIGH, ("pBBD->buf  = 0x%08X\n", pBBD->buf));
   TRACE(VMMC, DBG_LEVEL_HIGH, ("pBBD->size = %d byte\n", pBBD->size));

   for(i=0; i<pBBD->size; i++)
   {
      if( i!=0 && (i%8 == 0))
         TRACE(VMMC, DBG_LEVEL_HIGH, ("\n"));
      TRACE(VMMC, DBG_LEVEL_HIGH, ("%02X", pBBD->buf[i]));
   }
   return;
}

static IFX_void_t print_bbd_block(bbd_block_t *pBBD)
{
   IFX_uint32_t i;

   TRACE(VMMC, DBG_LEVEL_HIGH, ("pBBD->identifier = 0x%08X\n", pBBD->identifier));
   TRACE(VMMC, DBG_LEVEL_HIGH, ("pBBD->tag        = 0x%04X\n", pBBD->tag));
   TRACE(VMMC, DBG_LEVEL_HIGH, ("pBBD->version    = 0x%04X\n", pBBD->version));
   TRACE(VMMC, DBG_LEVEL_HIGH, ("pBBD->index      = 0x%04X\n", pBBD->index));
   TRACE(VMMC, DBG_LEVEL_HIGH, ("pBBD->size       = 0x%08X\n", pBBD->size));
   TRACE(VMMC, DBG_LEVEL_HIGH, ("pBBD->pData      = 0x%08X\n", pBBD->pData));
   for(i=0; i<pBBD->size; i++)
   {
      if( i!=0 && (i%8 == 0))
         TRACE(VMMC, DBG_LEVEL_HIGH, ("\n"));
      TRACE(VMMC, DBG_LEVEL_HIGH, ("%02X", pBBD->pData[i]));
   }
   TRACE(VMMC, DBG_LEVEL_HIGH, ("\n"));
   return;
}
#endif /* VMMC_BBD_DEBUG */

/* ============================= */
/* Global function definition    */
/* ============================= */

/**
   Download a BBD file.

   \param  pLLDev       Pointer to the device structure.
   \param  pProc        Pointer to low-level device initialization structure.

   \return
   - VMMC_statusOk if successful
*/
IFX_int32_t VMMC_TAPI_LL_BBD_Dnld(IFX_TAPI_LL_DEV_t *pLLDev,
                                  IFX_void_t const *pProc)
{
   VMMC_DEVICE *pDev = (VMMC_DEVICE*)pLLDev;
   VMMC_IO_INIT IoInit;
   VMMC_DOWNLOAD dwld;
   IFX_int32_t ret = VMMC_statusOk;

   if (pProc == IFX_NULL)
   {
      /* reset all init pointers and init flag */
      memset (&IoInit, 0, sizeof (IoInit));
      /* set additional default flags */
      IoInit.nFlags = FW_AUTODWLD;
   }
   else
   {
      /* The init struct is specific for the LL driver and only known here.
         Because HL does not know the struct it cannot copy it and so it is
         copied here in the LL driver. */
      VMMC_OS_CpyUsr2Kern (&IoInit, pProc, sizeof (VMMC_IO_INIT));
   }

   /* BBD Download */
   if(IoInit.pBBDbuf != IFX_NULL)
   {
      dwld.bbdDwld.buf  = IoInit.pBBDbuf;
      dwld.bbdDwld.size = IoInit.bbd_size;
      ret = VMMC_BBD_Download((VMMC_CHANNEL *) pDev, &dwld.bbdDwld);
      if (ret != IFX_SUCCESS)
      {
         TRACE(VMMC,DBG_LEVEL_HIGH, ("INFO: BBD Buffer Download failed.\n"));
      }
      else
      {
         TRACE(VMMC,DBG_LEVEL_LOW, ("INFO: BBD Buffer provided successfully.\n"));
      }
   }
   else
   {
      TRACE(VMMC,DBG_LEVEL_HIGH, ("INFO: No BBD Buffer provided.\n"));
   }

   RETURN_DEVSTATUS (ret);
}


#if 0
/**
   Sets Others pointers (!= EDSP) as read from BBD buffer, if any.
\param
   pBBD     - handle to user bbd buffer
\param
   pBlock   - handle to bbd block which stores download data.
\param
   nBBDTag  - BBD tag of download type.
\return
   none
\remark
   -  Pointer pBlock will be filled accordingly, if there is any data of given
      tag in BBD  buffer.
   -  Caller must make sure that ptrs are valid and that bbd integrity is ok.
*/
IFX_void_t VMMC_BBD_SetPtr (bbd_format_t *pBBD, bbd_block_t *pBlock,
                            IFX_uint16_t nBBDTag)
{
   if (pBBD->buf == NULL)
   {
      pBlock->pData = NULL;
      TRACE (VMMC, DBG_LEVEL_LOW,
             ("WARN: No BBD Buffer => No Block of Tag %04X!\n", nBBDTag));
      return;
   }
   /* set bbd block characteristics */
   memset (pBlock, 0, sizeof (bbd_block_t));
   pBlock->identifier = BBD_VMMC_MAGIC;
   pBlock->tag        = nBBDTag;
   /* go through bbd buffer and check if there is any block of given tag */
   bbd_get_block (pBBD, pBlock);
   if ((pBlock->pData == NULL) || (pBlock->size == 0))
   {
      /* sorry ... no block of given tag available in bbd data buffer */
      pBlock->pData    = NULL;
      TRACE (VMMC, DBG_LEVEL_HIGH,
             ("no block of tag 0x%4X in given BBD buffer\n", nBBDTag));
   }
}
#endif


#if (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO)
/**
   Check passed bdd block tag if it is an audio ring tag (BBD_AUDIO_RING).
\param
   tag   - 16 bit bbd tag
\return
   IFX_TRUE in case of audio opmode tag or IFX_FALSE otherwise
*/
static IFX_boolean_t VMMC_BBD_Audio_Ring_Tag( IFX_uint16_t tag )
{
   IFX_boolean_t     retval=IFX_FALSE;

   if( BBD_VMMC_IPPHONE_RINGTONE_VOL1 <= tag &&
       tag <= BBD_VMMC_IPPHONE_RINGTONE_VOL8 )
      retval = IFX_TRUE;

   return retval;
}


/**
   Check passed bdd block tag if it is an audio opmode tag (BBD_AUDIO_OPMODE).
\param
   tag   - 16 bit bbd tag
\return
   IFX_TRUE in case of audio opmode tag or IFX_FALSE otherwise
*/
static IFX_boolean_t VMMC_BBD_Audio_Block_Tag( IFX_uint16_t tag )
{

   IFX_uint16_t      tag_tmp = tag;
   IFX_boolean_t     retval=IFX_FALSE;

   /* clear mute bit for tag comparison */
   if(tag_tmp & BBD_VMMC_IPPHONE_MUTE)
   {
      tag_tmp = tag_tmp & ~BBD_VMMC_IPPHONE_MUTE;
   }

   if( (BBD_VMMC_IPPHONE_IDLE == tag_tmp) ||
       (BBD_VMMC_IPPHONE_HANDS_VOL1 <= tag_tmp &&
       tag_tmp <= BBD_VMMC_IPPHONE_HANDS_VOL8) ||
       (BBD_VMMC_IPPHONE_HEADS_VOL1 <= tag_tmp &&
       tag_tmp <= BBD_VMMC_IPPHONE_HEADS_VOL8) ||
       (BBD_VMMC_IPPHONE_HANDSF_MUFFLED_VOL1 <= tag_tmp &&
       tag_tmp <= BBD_VMMC_IPPHONE_HANDSF_MUFFLED_VOL8) ||
       (BBD_VMMC_IPPHONE_HANDSF_NORMAL_VOL1 <= tag_tmp &&
       tag_tmp <= BBD_VMMC_IPPHONE_HANDSF_NORMAL_VOL8) ||
       (BBD_VMMC_IPPHONE_HANDSF_ECHOIC_VOL1 <= tag_tmp &&
       tag_tmp <= BBD_VMMC_IPPHONE_HANDSF_ECHOIC_VOL8) ||
       (BBD_VMMC_IPPHONE_HANDS_OL_VOL1 <= tag_tmp &&
       tag_tmp <= BBD_VMMC_IPPHONE_HANDS_OL_VOL8) ||
       (BBD_VMMC_IPPHONE_HEADS_OL_VOL1 <= tag_tmp &&
       tag_tmp <= BBD_VMMC_IPPHONE_HEADS_OL_VOL8) ||
       (BBD_VMMC_IPPHONE_HANDS_WB_VOL1 <= tag_tmp &&
       tag_tmp <= BBD_VMMC_IPPHONE_HANDS_WB_VOL8) ||
       (BBD_VMMC_IPPHONE_HEADS_WB_VOL1 <= tag_tmp &&
       tag_tmp <= BBD_VMMC_IPPHONE_HEADS_WB_VOL8) ||
       (BBD_VMMC_IPPHONE_HANDSF_WB_MUFFLED_VOL1 <= tag_tmp &&
       tag_tmp <= BBD_VMMC_IPPHONE_HANDSF_WB_MUFFLED_VOL8) ||
       (BBD_VMMC_IPPHONE_HANDSF_WB_NORMAL_VOL1 <= tag_tmp &&
       tag_tmp <= BBD_VMMC_IPPHONE_HANDSF_WB_NORMAL_VOL8) ||
       (BBD_VMMC_IPPHONE_HANDSF_WB_ECHOIC_VOL1 <= tag_tmp &&
       tag_tmp <= BBD_VMMC_IPPHONE_HANDSF_WB_ECHOIC_VOL8) ||
       (BBD_VMMC_IPPHONE_HANDS_OL_WB_VOL1 <= tag_tmp &&
       tag_tmp <= BBD_VMMC_IPPHONE_HANDS_OL_WB_VOL8) ||
       (BBD_VMMC_IPPHONE_HEADS_OL_WB_VOL1 <= tag_tmp &&
       tag_tmp <= BBD_VMMC_IPPHONE_HEADS_OL_WB_VOL8) )
   {
      retval = IFX_TRUE;
   }
   return retval;

}
#endif  /* VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO */

/**
   Handle a bdd block, internally we'll dispatch depending on block type.
   This function is used within the context of a single channel.
\param
   pCh   - handle to channel
\param
   pBBDblock  - handle to a kernel level copy of the bbd block
\return
   IFX_SUCCESS or IFX_ERROR
*/
static IFX_int32_t VMMC_BBD_BlockHandler(VMMC_CHANNEL *pCh, bbd_block_t *pBBDblock)
{
#ifdef SYSTEM_DANUBE
   VMMC_ALM_SAMPLING_COEFS_t  alm_smpl;
#endif /* SYSTEM_DANUBE */
   VMMC_RingCfg_t             ringCfg;
   VMMC_DcThr_t               dcThr;
   IFX_uint16_t               slic_val;
   IFX_int32_t                ret = IFX_SUCCESS;

   TRACE(VMMC, DBG_LEVEL_LOW,
         ("bbd block with tag 0x%04X passed\n", pBBDblock->tag));

   /* for FXO line allowed blocks are FXO_CRAM and TRANSPARENT */
   if (pCh->pALM->line_type_fxs != IFX_TRUE)
   {
      switch (pBBDblock->tag)
      {
         case  BBD_VMMC_FXO_CRAM_BLOCK:
            ret = vmmc_BBD_DownloadChCram (pCh, pBBDblock);
            break;
         case BBD_VMMC_TRANSPARENT_BLOCK:
            ret = vmmc_BBD_WhiteListedCmdWr(pCh, pBBDblock);
            break;
         default:
            TRACE (VMMC, DBG_LEVEL_LOW, ("VMMC device driver WARNING:"
                " unsupported block tag 0x%04X for FXO line%d. Skipped.\n",
                pBBDblock->tag, pCh->nChannel-1));
            break;
      }
   }
   else
   {
      /* FXS line */

#if (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO)
      /* check for BBD_AUDIO_OPMODE tag and process block in case of valid tag */
      if( VMMC_BBD_Audio_Block_Tag(pBBDblock->tag) == IFX_TRUE)
         ret = VMMC_AUDIO_BBD_AudioOpmode(pCh, pBBDblock);
      else

      /* check for BBD_AUDIO_RING tag and process block in case of valid tag */
      if( VMMC_BBD_Audio_Ring_Tag(pBBDblock->tag) == IFX_TRUE)
      {
         ret = VMMC_AUDIO_BBD_AudioRing(pCh, pBBDblock);
      }
      else
#endif  /* VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO */
      switch (pBBDblock->tag)
      {
         case  BBD_VMMC_CRAM_BLOCK:
            ret = vmmc_BBD_DownloadChCram (pCh, pBBDblock);
            break;

         case BBD_VMMC_SLIC_BLOCK:
            cpb2w (&slic_val, pBBDblock->pData, sizeof (IFX_uint16_t));
            ret = vmmc_BBD_SetSlic (pCh, IFX_FALSE, slic_val);
            break;

         case BBD_VMMC_RING_CFG_BLOCK:
            memset(&ringCfg, 0, sizeof (ringCfg));
            cpb2w (&ringCfg.ring_freq, &pBBDblock->pData[0],
                sizeof (IFX_uint16_t));
            cpb2w (&ringCfg.ring_amp, &pBBDblock->pData[2],
                sizeof (IFX_uint16_t));
            cpb2w (&ringCfg.ring_hook_level, &pBBDblock->pData[4],
                sizeof (IFX_uint16_t));
            /* version 2 or greater of the bbd block also has a field for
               the fast ring trip type */
            if (pBBDblock->version >= 2)
            {
               IFX_uint16_t nVal;
               cpb2w (&nVal, &pBBDblock->pData[6], sizeof (IFX_uint16_t));
               switch (nVal)
               {
                  case VMMC_RING_TRIP_TYPE_NORMAL:
                  case VMMC_RING_TRIP_TYPE_FAST:
                     ringCfg.ring_trip_type = (VMMC_RING_TRIP_TYPE_t)nVal;
                     break;
                  default:
                     ret = IFX_ERROR;
                     break;
               }
            }
            else
            {
               /* for all other versions of this block, use the default
                  fast ring trip setting (normal) */
               ringCfg.ring_trip_type = VMMC_RING_TRIP_TYPE_NORMAL;
            }
            /* version 3 or greater of bbd block also has the field for
               ring trip dup time */
            if (pBBDblock->version >= 3)
            {
               cpb2w (&ringCfg.ring_trip_dup_time, &pBBDblock->pData[8],
                   sizeof (IFX_uint16_t));
            }
            else
            {
               /* for all other versions of this block, use defaults */
               ringCfg.ring_trip_dup_time = VMMC_DEFAULT_RING_TRIP_DUP_TIME;
            }
            /* version 4 or greater of bbd block also has the field for
               ring dc offset */
            if (pBBDblock->version >= 4)
            {
               cpb2w (&ringCfg.ring_dco, &pBBDblock->pData[10],
                   sizeof (IFX_uint16_t));
            }
            else
            {
               /* otherwise use the default value */
               ringCfg.ring_dco           = VMMC_DEFAULT_RING_DC_OFFSET;
            }
            if (ret == IFX_SUCCESS)
               ret = vmmc_BBD_RingCfg (pCh, IFX_FALSE, &ringCfg);
            break;

         case BBD_VMMC_DC_THRESHOLDS_BLOCK:
            memset (&dcThr, 0, sizeof (dcThr));
            cpb2w (&dcThr.hook_dup_time, &pBBDblock->pData[0],
                sizeof (IFX_uint16_t));
            cpb2w (&dcThr.onhook_time, &pBBDblock->pData[2],
                sizeof (IFX_uint16_t));
            /* version 2 or greater of bbd block also has the field for
               overtemperature dup time */
            if (pBBDblock->version >= 2)
            {
               cpb2w (&dcThr.ovt_dup_time, &pBBDblock->pData[4],
                   sizeof (IFX_uint16_t));
            }
            else
            {
               /* for all other versions of this block, use the default
                  overtemperature dup time */
               dcThr.ovt_dup_time = VMMC_DEFAULT_OVT_DUP_TIME;
            }
            ret = vmmc_BBD_SetDcThr (pCh, IFX_FALSE, &dcThr);
            break;

         case BBD_VMMC_VOFW_COEFF:
            vmmc_BBD_VoFWCoeff (pCh, pBBDblock);
            break;

#ifdef SYSTEM_DANUBE
         case BBD_VMMC_CRAM_NB_DATA:
         case BBD_VMMC_CRAM_WB_DATA:
            /* Sampling Rate coefficients (req. for NB/WB switch of the ALM) */
            /* endianess handling */
            cpb2w (&alm_smpl.blk0_offset, &pBBDblock->pData[0],
                sizeof (VMMC_ALM_SAMPLING_COEFS_t));

            /* narrow-/wideband coefficient storage */
            ret = VMMC_ALM_SamplingModeCoefficientStore (pCh,
               (pBBDblock->tag == BBD_VMMC_CRAM_NB_DATA) ? NB_8_KHZ : WB_16_KHZ,
               &alm_smpl);
            break;
#endif /* SYSTEM_DANUBE */

         case BBD_VMMC_TRANSPARENT_BLOCK:
            ret = vmmc_BBD_WhiteListedCmdWr(pCh, pBBDblock);
            break;

         default:
            TRACE (VMMC, DBG_LEVEL_LOW, ("VMMC device driver WARNING:"
                " unsupported block tag 0x%04X for FXS line%d. Skipped.\n",
                pBBDblock->tag, pCh->nChannel-1));
            /* parsing continues */
            break;
      }
   } /* if */
   return ret;
}


/**
   Does a BBD download on given channel
   - ioctl FIO_VMMC_BBD_DOWNLOAD
   This function is also called during initialisation, in this case a pointer
   to the device strucutre is passed instead of a channel context. In this
   case the downloads are executed on all channels.
\param
   pCh   - handle to channel or device structure
\param
   pBBD  - handle to user BBD buffer
\return
   IFX_SUCCESS or IFX_ERROR
\remark
   ioctl call to download a BBD buffer on given channel
*/
IFX_int32_t VMMC_BBD_Download (VMMC_CHANNEL *pCh, bbd_format_t *pBBD)
{
   VMMC_DEVICE      *pDev = IFX_NULL;
   IFX_boolean_t     bBroadcast = IFX_FALSE;
   IFX_int32_t       ret = IFX_SUCCESS;
   IFX_uint32_t      i, j, block_num;
   bbd_error_t       bbd_err;
   bbd_block_t       bbd_vmmc_block;
   bbd_format_t      bbd;
#if !(VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO)
   IFX_uint32_t      ch;
#endif /* !(VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO) */

   /* detect channel or device context */
   if (pCh->nChannel == 0)
   {
      pDev = /*lint --e(826)*/ (VMMC_DEVICE *) pCh;
      bBroadcast = IFX_TRUE;
      pCh  = &pDev->pChannel[0];
   }
#if !(VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO)
   else if (pCh->nChannel > pCh->pParent->caps.nALI)
   {  /* currently the download only covery ALM channels - if the
         application tries to download on a channel which doesn't
         have an AlM we return with error.
         If we should extend the download to coder channels as well
         we need to move this check down to the individual block
         handling... */
      TRACE(VMMC, DBG_LEVEL_HIGH,
           ("VMMC: BBD cannot be downloaded on ch=%d (max ali=%d)\n",
            pCh->nChannel, pCh->pParent->caps.nALI));
      return IFX_ERROR;
   }
#endif /* !(VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO) */
   else
   {
      pDev = pCh->pParent;
   }

   /* initializations */
   memset (&bbd, 0, sizeof(bbd));
   memset (&bbd_vmmc_block, 0, sizeof (bbd_vmmc_block));

   if (pBBD->buf != IFX_NULL)
   {
      /* get local memory for bbd buffer */
      bbd.buf  = VMMC_OS_MapBuffer(pBBD->buf, pBBD->size);
      if (bbd.buf == NULL)
         return IFX_ERROR;
      /* set size */
      bbd.size = pBBD->size;
      /* check BBD Buffer integrity */
      bbd_err = bbd_check_integrity (&bbd, BBD_VMMC_MAGIC);
      if (bbd_err != BBD_INTG_OK)
      {
         VMMC_OS_UnmapBuffer(bbd.buf);
         TRACE (VMMC, DBG_LEVEL_HIGH, ("VMMC: BBD download fail (0x%X)\n", bbd_err));
         return IFX_ERROR;
      }
   }
   else
   {
      /* FIXME: improve, point to a default bbd buffer here */
   }

   if (IFX_TRUE == pDev->bSlicSupportsIdleMode)
   {
      IFX_uint8_t i;

      if (!bBroadcast)
      {
         /* update SDD_BasicConfig cache*/
         ret = vmmc_BBD_UpdateBasicConfigCache (pCh);
         if (VMMC_SUCCESS(ret))
         {
            ret = vmmc_BBD_PrepareIdleExtBit (pCh);
         }
      }
      else
      {
         for (i=0; i < pDev->caps.nALI; i++)
         {
            pCh = &pDev->pChannel[i];
            /* update SDD_BasicConfig cache*/
            ret = vmmc_BBD_UpdateBasicConfigCache (pCh);
            if (VMMC_SUCCESS(ret))
            {
               ret = vmmc_BBD_PrepareIdleExtBit (pCh);
            }
            if (ret != IFX_SUCCESS)
               break;
         }
      }
      if (ret != IFX_SUCCESS)
      {
         TRACE (VMMC, DBG_LEVEL_HIGH,
               ("VMMC: BBD download: IdleExt bit reset fail\n"));
         return IFX_ERROR;
      }
      /* sleep 5ms */
      IFXOS_MSecSleep(IDLE_EXT_TOGGLE_SLEEP_MS);
   }

   /* let go through bbd buffer and download
      any channel block of relevance found, VMMC_CH_BBD_Blocks is a constant
      array of 16 bit entries defining the different supported
      block types */
   block_num  = (sizeof (VMMC_CH_BBD_Blocks) / sizeof (IFX_uint16_t));
   bbd_vmmc_block.identifier = BBD_VMMC_MAGIC;
   for (i = 0; i < block_num; i++)
   {
      /* the order of block search and handling is defined in
         const VMMC_CH_BBD_Blocks array. */
      bbd_vmmc_block.tag = VMMC_CH_BBD_Blocks [i];

      /* sniff blocks of this tag upto maximum allowed */
      for (j = 0; j < BBD_VMMC_BLOCK_MAXDWNLD ; j++)
      {
         /* look at block of this index and download it if available */
         bbd_vmmc_block.index = j;
         bbd_get_block (&bbd, &bbd_vmmc_block);
         if ((bbd_vmmc_block.pData == NULL) || (bbd_vmmc_block.size == 0))
         {
            if (bbd_vmmc_block.index == 0)
            {
               /* FIXME: improvement: provide a default block here */
               break;
            }
            else
            {
               /* skip further search for this blocktype */
               break;
            }
         }

#if (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO)
         if(pDev->caps.nAudioCnt)
            ret = VMMC_BBD_BlockHandler(pCh, &bbd_vmmc_block);
#else
         if (!bBroadcast)
         {
            /* do the download on a single channel */
            ret = VMMC_BBD_BlockHandler(pCh, &bbd_vmmc_block);
         }
         else
         {
            /* do the downloads on all _analog_ channels */

            /* ATTENTION, if we need to extend the download to other channel
               types, checks must be added per block if the selected channel
               number is smaller or equal the max. number of this type of ch!
            */
            for (ch=0; ch < pDev->caps.nALI; ch++)
            {
               pCh = &pDev->pChannel[ch];
               ret = VMMC_BBD_BlockHandler(pCh, &bbd_vmmc_block);
               if (ret != IFX_SUCCESS)
                  break;
            }
         }
#endif /* (VMMC_CFG_FEATURES & VMMC_FEAT_AUDIO) */
         /* stop every thing if the previous download went wrong */
         if (ret != IFX_SUCCESS)
            break;
      }
      /* stop every thing if the previous download went wrong */
      if (ret != IFX_SUCCESS)
         break;
   }

   if (IFX_TRUE == pDev->bSlicSupportsIdleMode)
   {
      IFX_uint8_t i;

      /* restore IdleExt bit of SDD_BasicConfig */
      if (!bBroadcast)
      {
         /* update SDD_BasicConfig cache*/
         ret = vmmc_BBD_UpdateBasicConfigCache (pCh);
         if (VMMC_SUCCESS(ret))
         {
            ret = vmmc_BBD_RestoreIdleExtBit (pCh);
         }
      }
      else
      {
         for (i=0; i < pDev->caps.nALI; i++)
         {
            pCh = &pDev->pChannel[i];
            /* update SDD_BasicConfig cache*/
            ret = vmmc_BBD_UpdateBasicConfigCache (pCh);
            if (VMMC_SUCCESS(ret))
            {
               ret = vmmc_BBD_RestoreIdleExtBit (pCh);
            }
            if (ret != IFX_SUCCESS)
               break;
         }
      }
      if (ret != IFX_SUCCESS)
      {
         TRACE (VMMC, DBG_LEVEL_HIGH,
                      ("VMMC: BBD download: IdleExt bit restore fail\n"));
         return IFX_ERROR;
      }
   }

   VMMC_OS_UnmapBuffer(bbd.buf);

   return ret;
}

/**
   Download VoFW Coefficients on channel.
\param
   pCh  Handle to the channel
\param
   pBBDblock Handle to bbd VoFW coefficient block
\return
   - VMMC_statusBBDError Writing the command of one parameter of the
     BBD file failed
   - VMMC_statusOk if successful
*/
IFX_int32_t vmmc_BBD_VoFWCoeff (VMMC_CHANNEL *pCh, bbd_block_t *pBBDblock)
{
   VMMC_Msg_t Msg;
   IFX_int32_t  ret = VMMC_statusOk;
   IFX_uint32_t cnt = 0, j;

   /* handle all messages in the buffer */
   do
   {
      Msg.val[0] = ntohl (*((IFX_uint32_t*)/*lint --e(826)*/&pBBDblock->pData[cnt]));
      for (j=4; j < Msg.cmd.LENGTH + CMD_HDR_CNT; j+= 4)
      {
         Msg.val[j/4] = ntohl (*((IFX_uint32_t*)/*lint --e(826)*/&pBBDblock->pData[cnt + j]));
      }
      /* go to the next message in 32 bit steps */
      cnt += Msg.cmd.LENGTH + CMD_HDR_CNT;
      if (cnt > pBBDblock->size)
      {
         return VMMC_statusOk;
      }
      if (Msg.cmd.MOD != MOD_RESOURCE)
      {
         /* errmsg: The BBD content for VoFW coefficients is invalid.
            Only resource coefficients are allowed */
         RETURN_STATUS (VMMC_statusBBDviolation);
      }
      /* Store LEC coefficients */
      if (Msg.cmd.ECMD == RES_LEC_COEF_ECMD)
      {
         /* VMMC_ALM_LecCoeffUpdate (pCh, &Msg.res_LecCoef); */
      }
      else
      {
         ret = CmdWrite (pCh->pParent, Msg.val, Msg.cmd.LENGTH);
      }
   }
   while (VMMC_SUCCESS(ret) && (cnt < pBBDblock->size));

   RETURN_STATUS (ret);
}


/**
   Transparent "white-listed" CmdWrite
\param
   pCh  Handle to the channel
\param
   pBBDblock Handle to bbd VoFW coefficient block
\return
   - DXT_statusBBDError Writing the command of one parameter of the
     BBD file failed
   - DXT_statusOk if successful
\remarks
*/
static IFX_int32_t vmmc_BBD_WhiteListedCmdWr(VMMC_CHANNEL *pCh,
                                             bbd_block_t *pBBDblock)
{
   VMMC_Msg_t   Msg;
   IFX_int32_t  ret = VMMC_statusOk;
   IFX_uint32_t i;

   /* endianess conform header */
   Msg.val[0]   = ntohl (*((IFX_uint32_t*)/*lint --e(826)*/&pBBDblock->pData[0]));
   Msg.cmd.RW   = 0;
   Msg.cmd.CHAN = pCh->nChannel - 1;

   // check pBBDblock->size

   /* endianess conform data */
   for (i=4; i < Msg.cmd.LENGTH + 4; i+= 4)
   {
      Msg.val[i/4] = ntohl (*((IFX_uint32_t*)/*lint --e(826)*/&pBBDblock->pData[i]));
   }

   /* check if the block contains a FW message from the "whitelist",
      which is defined by the msg header fields CMD, MOD and ECMD */
   switch (Msg.val[0] & VMMC_CMD_MOD_ECMD_MASK)
   {
      case VMMC_WL_SDD_BASIC_CFG:
         {
            if (IFX_TRUE == pCh->pParent->bSlicSupportsIdleMode)
            {
               VMMC_SDD_BasicConfig_t *pSDD_BasicConfig =
                                      (VMMC_SDD_BasicConfig_t *)&Msg;
               /* patch IdleExt bit */
               if(1 == pSDD_BasicConfig->IdleExt)
               {
                  pCh->pALM->bIdleExtNeedRestore = IFX_TRUE;
                  pSDD_BasicConfig->IdleExt = 0;
               }
               else
               {
                  pCh->pALM->bIdleExtNeedRestore = IFX_FALSE;
               }
            }
         }
      case VMMC_WL_SDD_RING_CFG:
      case VMMC_WL_SDD_DCDC_CFG:
         ret = CmdWrite (pCh->pParent, Msg.val, Msg.cmd.LENGTH);
         break;

      default:
         /* if command is not whitelisted, ignore it */
         TRACE(VMMC, DBG_LEVEL_HIGH,
              ("VMMC BBD ignoring non whitelisted block:\n"));
         for (i=0; i<= Msg.cmd.LENGTH/4; i++)
         {
            TRACE(VMMC, DBG_LEVEL_HIGH,("VMMC BBD %% %08X\n", Msg.val[i]));
         }
   }

   RETURN_STATUS (ret);
}

/**
   Download CRAM Coefficients on channel.
\param
   pCh        - handle to channel
\param
   bbd_cram   - handle to bbd CRAM block
\return
   IFX_SUCCESS or IFX_ERROR
\remarks
   - It is assumed that the given CRAM bbd block is valid and that the data
     representation in the bbd block is according to BBD specification,
     as follows:

     ofset_16 : 0xXX, 0xXX,
     data_16[]: 0xXX, 0xXX,
                0xXX, 0xXX,
                ...
     crc_16   : 0xXX, 0xXX
*/
static IFX_int32_t vmmc_BBD_DownloadChCram (VMMC_CHANNEL *pCh,
                                            bbd_block_t  *bbd_cram)
{
   IFX_int32_t  ret = IFX_ERROR;
   IFX_uint32_t countWords;
   IFX_uint32_t posBytes = 0;
   IFX_uint8_t  lenBytes, *pByte;
#if defined(SYSTEM_AR9) || defined(SYSTEM_VR9)
   IFX_uint8_t  padBytes = 0;
#endif
   IFX_uint16_t cram_offset, cram_crc,
                pCmd [MAX_CMD_WORD]  = {0};

   /* read offset */
   cpb2w (&cram_offset, &bbd_cram->pData[0], sizeof (IFX_uint16_t));
   /* set CRAM payload pointer */
   pByte = &bbd_cram->pData[2];
   /* set CRAM data size in words, removing offset and crc bytes */
   countWords = (bbd_cram->size - 4) >> 1 ;

#ifdef SYSTEM_DANUBE
   /* CMD1 is a COP command  */
   pCmd[0] = (0x0200) | (pCh->nChannel - 1);
#elif  defined(SYSTEM_AR9) || defined(SYSTEM_VR9)
   /* SDD_Coef command */
   pCmd[0] = (0x0400) | (pCh->nChannel - 1);
   pCmd[1] = (0x0D00);
#else
#error undefined system
#endif


   /* write CRAM data */
   while (countWords > 0)
   {
#if defined SYSTEM_DANUBE
      /* calculate length to download (in words = 16bit),
         maximum allowed length for this message is 56 Bytes = 28 Words */
      if (countWords > ((MAX_CMD_WORD - CMD_HDR_CNT)))
         lenBytes = (MAX_CMD_WORD - CMD_HDR_CNT) << 1;
      else
         lenBytes = countWords << 1;
      /* set CRAM offset in CMD2 */
      pCmd[1] = ((cram_offset + (posBytes >> 1)) << 8);
      /* set CRAM data while taking care of endianess  */
      cpb2w (&pCmd[2], &pByte[posBytes], lenBytes);
#elif defined(SYSTEM_AR9) || defined(SYSTEM_VR9)
      /* calculate length to download (in words = 16bit),
         maximum allowed length for this message is 56 Bytes = 28 Words */
      if (countWords > ((MAX_CMD_WORD - CMD_HDR_CNT - 1)))
         lenBytes = (MAX_CMD_WORD - CMD_HDR_CNT - 1) << 1;
      else
         lenBytes = countWords << 1;
      /* set CRAM offset in CMD2 */
      pCmd[2] = ((cram_offset + (posBytes >> 1)) << 5) | lenBytes>>1;
      /* set CRAM data while taking care of endianess  */
      cpb2w (&pCmd[3], &pByte[posBytes], lenBytes);
      /* the download length is a multiple of 16 bit, nevertheless we
         have to be 32bit aligned */
      if ((lenBytes%4))
      {
         padBytes = 0;
      }
      else
      {
         /* add padding word */
         pCmd[3+(lenBytes>>1)] = 0x0000;
         padBytes = 2;
      }
      /*printk("o=%d l=%d p=%d\n", (cram_offset + (posBytes >> 1)), lenBytes>>1, padBytes>>1);*/
#endif

      /* write Data */
#if defined SYSTEM_DANUBE
      ret = CmdWrite (pCh->pParent, (IFX_uint32_t *) pCmd, lenBytes);
#elif defined(SYSTEM_AR9) || defined(SYSTEM_VR9)
#if 1
      /* lenBytes + 2 bytes for block offset/length which are not calculated
         in the download progress */
      ret = CmdWrite (pCh->pParent, (IFX_uint32_t *) pCmd, lenBytes+padBytes+2);
#else
#warning temp CRAM download just traces
   pCmd[1] &= ~0x00FF;
   pCmd[1] |= lenBytes+2;
   printk(KERN_DEBUG "cram ");
   {int i;
   for (i=0; i<= (4 /*header*/+lenBytes+padBytes)/2; i++) /* i in dwords */
   {
      printk(KERN_DEBUG "%04X", pCmd[i]);
      if (i%2 && i!=0) printk(KERN_DEBUG " ");
   }
   printk(KERN_DEBUG "\n");
   }
   ret = IFX_SUCCESS;
#endif /* 0 */
#endif /* DANUBE/AR9/VR9 */

      if (ret != IFX_SUCCESS)
         break;
      /* \todo eventually call crc calculation routine here */
      /* increment position */
      posBytes += lenBytes;
      /* decrement count */
      countWords -= (lenBytes >> 1);
   }
   /* In case the download went through, read CRC from buffer and do checks */
   if (ret == IFX_SUCCESS)
   {
      cpb2w (&cram_crc, &pByte [posBytes], sizeof (IFX_uint16_t));
      /* \todo Check download CRAM CRC here, either by comparison or by
               reading it back from chip */
   }
   return ret;
}

/**
   Sets Slic value on given channel(s).
\param
   pCh         - handle to channel structure
\param
   bBroadcast  - IFX_TRUE=all channels/IFX_FALSE=given channel.
\param
   slic_val    - selected slic type
\return
   IFX_SUCCESS or IFX_ERROR
\remarks
   In case of broadcast (bBroadcast = IFX_TRUE), slic is set on
   all channels.
*/
static IFX_int32_t vmmc_BBD_SetSlic (VMMC_CHANNEL  *pCh,
                                     IFX_boolean_t  bBroadcast,
                                     IFX_uint16_t   slic_val)
{
   IFX_int32_t     ret = IFX_SUCCESS;
   VMMC_DEVICE     *pDev = pCh->pParent;
   IFX_uint8_t     i, ch_start = 0, ch_stop = pDev->caps.nALI;
   BASIC_CONFIG_CMD_t   basicCfg;

   memset (&basicCfg, 0, sizeof (BASIC_CONFIG_CMD_t));
   basicCfg.CMD    = CMD_ALI;
   basicCfg.MOD    = MOD_DCCTL;
   basicCfg.ECMD   = ECMD_BASIC_CFG_CMD;
   basicCfg.LENGTH = 4;

   /* change start/stop if no broadcast */
   if ((bBroadcast == IFX_FALSE) &&
       ((IFX_uint8_t)(pCh->nChannel - 1) < pDev->caps.nALI))
   {
      ch_start = (pCh->nChannel - 1);
      ch_stop  = ch_start + 1;
   }
   /* set slic value now */
   for (i = ch_start; i < ch_stop; i++)
   {
      basicCfg.CHAN = i;
      /* read previous channel configuration,
         inclusive command header */
      ret = CmdRead(pDev, (IFX_uint32_t *)((IFX_void_t *)&basicCfg),
                          (IFX_uint32_t *)((IFX_void_t *)&basicCfg),
                           basicCfg.LENGTH);
      if (!VMMC_SUCCESS(ret))
         goto error;
      /* map slic type */
      switch (slic_val)
      {
         case BBD_VMMC_SLIC_TYPE_E:
            basicCfg.SLIC_SEL = BASIC_CONFIG_CMD_SLIC_E;
            break;
         case BBD_VMMC_SLIC_TYPE_DC:
         default:
            basicCfg.SLIC_SEL = BASIC_CONFIG_CMD_SLIC_DC;
            break;
      }
      /* write channel configuration with new slic */
      ret = CmdWrite(pDev, (IFX_uint32_t *)((IFX_void_t *)&basicCfg),
                     basicCfg.LENGTH);
      if (ret != IFX_SUCCESS)
         goto error;
   }

error:
   /* no error handling required */
   return ret;
}

/**
   Set Ringing configuration on given channel(s).
\param
   pCh         - handle to channel structure
\param
   bBroadcast  - IFX_TRUE=all channels/IFX_FALSE=given channel.
\param
   p_ringCfg   - handle to ring configuration.
\return
   IFX_SUCCESS or IFX_ERROR
\remarks
   In case of broadcast (bBroadcast = IFX_TRUE), slic is set on
   all channels.
*/
static IFX_int32_t vmmc_BBD_RingCfg (VMMC_CHANNEL   *pCh,
                                    IFX_boolean_t   bBroadcast,
                                    VMMC_RingCfg_t *p_ringCfg)
{
   IFX_int32_t        ret         = IFX_SUCCESS;
   VMMC_DEVICE       *pDev        = pCh->pParent;
   IFX_uint8_t        i, ch_start = 0, ch_stop = pDev->caps.nALI;
   RING_CONFIG_CMD_t  ringCfg;

   memset (&ringCfg, 0, sizeof (RING_CONFIG_CMD_t));
   ringCfg.CMD     = CMD_ALI;
   ringCfg.MOD     = MOD_DCCTL;
   ringCfg.ECMD    = ECMD_RING_CFG_CMD;

   /* change start/stop if no broadcast */
   if ((bBroadcast == IFX_FALSE) &&
       ((IFX_uint8_t)(pCh->nChannel - 1) < pDev->caps.nALI))
   {
      ch_start = (pCh->nChannel - 1);
      ch_stop  = ch_start + 1;
   }

   if (ch_stop > pDev->caps.nALI)
   {
      ch_stop = pDev->caps.nALI;
   }

   for (i = ch_start; i < ch_stop; i++)
   {
#if (VMMC_CFG_FEATURES & VMMC_FEAT_GR909)
      /* reset flag for this channel every time you are here */
      pDev->pChannel[i].pALM->b_ring_cfg       = IFX_FALSE;
#endif /* VMMC_CFG_FEATURES & VMMC_FEAT_GR909 */

      ringCfg.CHAN    = i;
      ringCfg.LENGTH  = 20;

      ret = CmdRead (pDev, (IFX_uint32_t *)((IFX_void_t *)&ringCfg),
                           (IFX_uint32_t *)((IFX_void_t *)&ringCfg),
                            ringCfg.LENGTH);
      if (ret != IFX_SUCCESS)
         goto error;

      /*PRINTF ("rt duptime: old=%d new=%d\n",
                 ringCfg.TRIP_DUP, p_ringCfg->ring_trip_dup_time);*/
      /* set the ring frequency */
      ringCfg.RING_F     = p_ringCfg->ring_freq;
      /* set the ring amplitude */
      ringCfg.RING_A     = p_ringCfg->ring_amp;
      /* set the ring hook level */
      ringCfg.RING_HL    = p_ringCfg->ring_hook_level;
      /* set the ring trip type */
      ringCfg.FAST_RT    = p_ringCfg->ring_trip_type;
      /* set ring trip dup time */
      ringCfg.TRIP_DUP   = p_ringCfg->ring_trip_dup_time;
      /* set the ring dc offset */
      ringCfg.RING_DCO   = p_ringCfg->ring_dco;
      ret = CmdWrite (pDev, (IFX_uint32_t *)((IFX_void_t *)&ringCfg),
                             ringCfg.LENGTH);
      if (ret != IFX_SUCCESS)
         goto error;

#if (VMMC_CFG_FEATURES & VMMC_FEAT_GR909)
      /* Save Actual ring configuration for GR909 */
      pDev->pChannel [i].pALM->ring_cfg   = ringCfg;
      pDev->pChannel [i].pALM->b_ring_cfg = IFX_TRUE;
#endif /* VMMC_CFG_FEATURES & VMMC_FEAT_GR909 */
   }

error:
   return ret;
}

/**
   Set DC Threshold configuration on given channel(s).
\param
   pCh         - handle to channel structure
\param
   bBroadcast  - IFX_TRUE=all channels/IFX_FALSE=given channel.
\param
   p_dcThr     - handle to dc threshold configuration.
\return
   IFX_SUCCESS or IFX_ERROR
\remarks
   In case of broadcast (bBroadcast = IFX_TRUE), slic is set on
   all channels.
*/
static IFX_int32_t vmmc_BBD_SetDcThr (VMMC_CHANNEL *pCh,
                                      IFX_boolean_t bBroadcast,
                                      VMMC_DcThr_t *p_dcThr)
{
   IFX_int32_t         ret         = IFX_SUCCESS;
   VMMC_DEVICE        *pDev = pCh->pParent;
   IFX_uint8_t         i, ch_start = 0, ch_stop = pDev->caps.nALI;
   BASIC_CONFIG_CMD_t  basicCfg;

   memset (&basicCfg, 0, sizeof (BASIC_CONFIG_CMD_t));
   basicCfg.CMD      = CMD_ALI;
   basicCfg.MOD      = MOD_DCCTL;
   basicCfg.ECMD     = ECMD_BASIC_CFG_CMD;
   basicCfg.LENGTH   = 4;

   /* change start/stop if no broadcast */
   if ((bBroadcast == IFX_FALSE) &&
       ((IFX_uint8_t)(pCh->nChannel - 1) < pDev->caps.nALI))
   {
      ch_start = (pCh->nChannel - 1);
      ch_stop  = ch_start + 1;
   }
   /* set slic value now */
   for (i = ch_start; i < ch_stop; i++)
   {
      basicCfg.CHAN     = i;
      ret = CmdRead (pDev, (IFX_uint32_t *)((IFX_void_t *)&basicCfg),
                           (IFX_uint32_t *)((IFX_void_t *)&basicCfg),
                           basicCfg.LENGTH);
      if (ret != IFX_SUCCESS)
         goto error;

      basicCfg.HOOK_SET = p_dcThr->onhook_time;
      basicCfg.HOOK_DUP = p_dcThr->hook_dup_time;
      basicCfg.OVT_DUP  = p_dcThr->ovt_dup_time;

      ret = CmdWrite (pDev, (IFX_uint32_t *)((IFX_void_t *)&basicCfg),
                      basicCfg.LENGTH);
      if (ret != IFX_SUCCESS)
         goto error;
   }

error:
   return ret;
}

/**
   Update IdleExt bit of SDD_BasicConfig on given channel.
\param
   pCh         - handle to channel structure
   val         - value of IdleExtBit of SDD_BasicConfig command
   pbUpdated   - complete indicator
\return
   IFX_SUCCESS or IFX_ERROR
\remarks
*/
static IFX_int32_t vmmc_BBD_UpdateIdleExtBit (VMMC_CHANNEL *pCh,
                                              IFX_uint8_t val,
                                              IFX_boolean_t *pbUpdated)
{
   IFX_int32_t            ret = IFX_SUCCESS;
   VMMC_DEVICE            *pDev;
   VMMC_SDD_BasicConfig_t *pSDD_BasicConfig;

   /* sanity check */
   if (pCh == IFX_NULL || pCh->pALM == IFX_NULL || val > 1)
   {
      return IFX_ERROR;
   }
   pDev = pCh->pParent;
   pSDD_BasicConfig = &pCh->pALM->fw_sdd_basic_config;

   /* default result init */
   if (IFX_NULL != pbUpdated)
   {
      *pbUpdated = IFX_FALSE;
   }

   /* if update is required - write down to FW */
   if (val != pSDD_BasicConfig->IdleExt)
   {
      VMMC_OS_MutexGet (&pCh->chAcc);
      pSDD_BasicConfig->IdleExt = val;
      ret = CmdWrite (pDev, (IFX_uint32_t *)((IFX_void_t *)pSDD_BasicConfig),
                      SDD_BasicConfig_LEN);
      VMMC_OS_MutexRelease (&pCh->chAcc);

      if (ret == IFX_SUCCESS && IFX_NULL != pbUpdated)
      {
         *pbUpdated = IFX_TRUE;
      }
   }

   return ret;
}

/**
   Prepare IdleExt bit of SDD_BasicConfig on given channel.
\param
   pCh         - handle to channel structure
\return
   IFX_SUCCESS or IFX_ERROR
\remarks
   This function resets the IdleExt bit of SDD_BasicConfig
   command. This is required to ensure that SLIC is not in a sleep
   mode.
*/
static IFX_int32_t vmmc_BBD_PrepareIdleExtBit (VMMC_CHANNEL *pCh)
{
   IFX_int32_t ret;
   IFX_boolean_t bUpdated;

   /* reset IdleExt bit of SDD_BasicConfig to ensure that SLIC
      is not in a sleep mode */
   ret = vmmc_BBD_UpdateIdleExtBit (pCh, 0, &bUpdated);
   if (ret == IFX_SUCCESS && bUpdated == IFX_TRUE)
   {
      /* IdleExt bit is now reset to 0.
         It may need or may not need restoration to 1 after BBD
         download finish. This depends on BBD file content, whether
         the BBD file has SDD_BasicConfig command(s) with IdleExt==0.
         The below boolean indicates whether the restoration is needed.
         This indicator can be altered by transparent BBD block
         processing routine. */
      pCh->pALM->bIdleExtNeedRestore = IFX_TRUE;
   }

   return ret;
}

/**
   Restore IdleExt bit of SDD_BasicConfig on given channel.
\param
   pCh         - handle to channel structure
\return
   IFX_SUCCESS or IFX_ERROR
\remarks
   This functions restores IdleExt bit to 1 if bIdleExtNeedRestore is true.
*/
static IFX_int32_t vmmc_BBD_RestoreIdleExtBit (VMMC_CHANNEL *pCh)
{
   IFX_int32_t ret = IFX_SUCCESS;

   if (IFX_TRUE == pCh->pALM->bIdleExtNeedRestore)
   {
      ret = vmmc_BBD_UpdateIdleExtBit (pCh, 1, IFX_NULL);
   }

   return ret;
}

/**
   Update SDD_BasicConfig command cache on given channel.
\param
   pCh         - handle to channel structure
\return
   IFX_SUCCESS or IFX_ERROR
\remarks
*/
static IFX_int32_t vmmc_BBD_UpdateBasicConfigCache (VMMC_CHANNEL *pCh)
{
   IFX_int32_t            ret;
   VMMC_DEVICE            *pDev = pCh->pParent;
   VMMC_SDD_BasicConfig_t *pSDD_BasicConfig = &pCh->pALM->fw_sdd_basic_config;

   /* protect fw msg */
   VMMC_OS_MutexGet (&pCh->chAcc);
   ret = CmdRead (pDev, (IFX_uint32_t *)((IFX_void_t *)pSDD_BasicConfig),
                        (IFX_uint32_t *)((IFX_void_t *)pSDD_BasicConfig),
                        SDD_BasicConfig_LEN);
   VMMC_OS_MutexRelease (&pCh->chAcc);

   return ret;
}
