/******************************************************************************

                              Copyright (c) 2009
                            Lantiq Deutschland GmbH
                     Am Campeon 3; 85579 Neubiberg, Germany

  For licensing information, see the file 'LICENSE' in the root folder of
  this software module.

******************************************************************************/

/**
   \file drv_tapi_pcm.c
   Desription  : Contains PCM Services.
*/

/* ============================= */
/* Includes                      */
/* ============================= */

#include "drv_tapi.h"
#include "drv_tapi_errno.h"
#include "drv_tapi_ll_interface.h"

/* ============================= */
/* Global function definition    */
/* ============================= */


/**
   Sets the configuration of the PCM channel interface

   \param pTAPIDev      Pointer to TAPI_DEV structure.
   \param pPCMif        Contains the configuration for the pcm interface.

   \return
   Returns an error code:
      - \ref IFX_SUCCESS if configuration is set
      - \ref IFX_ERROR   if configuration is NOT set.
*/
IFX_int32_t TAPI_Phone_PCM_IF_Set_Config(TAPI_DEV *pTAPIDev,
                                         IFX_TAPI_PCM_IF_CFG_t const *pPCMif)
{
   IFX_TAPI_DRV_CTX_t *pDrvCtx = pTAPIDev->pDevDrvCtx;
   IFX_int32_t ret = IFX_SUCCESS;

   /* chip specific function */
   if (ptr_chk(pDrvCtx->PCM.ifCfg, "pDrvCtx->PCM.ifCfg"))
      ret = pDrvCtx->PCM.ifCfg (pTAPIDev->pLLDev, pPCMif);

   return ret;
}

/**
   Sets the configuration of one PCM channel

   \param pChannel      Pointer to TAPI_CHANNEL structure.
   \param pPCMConfig    Contains the configuration for the pcm channel.

   \return
   Returns an error code:
      - \ref IFX_SUCCESS if configuration is set
      - \ref IFX_ERROR   if configuration is NOT set.
*/
IFX_int32_t TAPI_Phone_PCM_Set_Config(TAPI_CHANNEL *pChannel,
                                      IFX_TAPI_PCM_CFG_t const *pPCMConfig)
{
   IFX_TAPI_DRV_CTX_t *pDrvCtx = pChannel->pTapiDevice->pDevDrvCtx;
   IFX_int32_t ret = TAPI_statusOk;

   /* chip specific function */
   if (ptr_chk(pDrvCtx->PCM.Cfg, "pDrvCtx->PCM.Cfg"))
   {
      ret = pDrvCtx->PCM.Cfg (pChannel->pLLChannel, pPCMConfig);
      if (!TAPI_SUCCESS(ret))
      {
         RETURN_STATUS (TAPI_statusLLFailed, ret);
      }
   }
   else
   {
      ret = TAPI_statusLLNotSupp;
   }

   /* save configuration */
   if (ret == IFX_SUCCESS)
   {
      memcpy(&pChannel->TapiPCMData.PCMConfig, pPCMConfig,
             sizeof(IFX_TAPI_PCM_CFG_t));

      pChannel->TapiPCMData.bCfgSuccess = IFX_TRUE;
   }
   else
   {
      /* configuration is not saved */
      pChannel->TapiPCMData.bCfgSuccess = IFX_FALSE;
   }

   return ret;
}


/**
   Gets the configuration of one PCM channel

   \param pChannel      Pointer to TAPI_CHANNEL structure.
   \param pPCMConfig    Contains the configuration for pcm channel.

   \return
   Returns an error code:
      - \ref IFX_SUCCESS if configuration is set
      - \ref IFX_ERROR   if configuration is NOT set.
*/
IFX_int32_t TAPI_Phone_PCM_Get_Config(TAPI_CHANNEL *pChannel,
                                      IFX_TAPI_PCM_CFG_t *pPCMConfig)
{
   /* get configuration */
   memcpy(pPCMConfig, &pChannel->TapiPCMData.PCMConfig,
          sizeof(IFX_TAPI_PCM_CFG_t));

   return TAPI_statusOk;
}


/**
   Activate or deactivate the pcm timeslots configured for this channel

   \param pChannel      Pointer to TAPI_CHANNEL structure.
   \param nMode         - 1: timeslot activated
                        - 0: timeslot deactivated
   \return
   \ref IFX_SUCCESS or \ref IFX_ERROR
*/
IFX_int32_t TAPI_Phone_PCM_Set_Activation(TAPI_CHANNEL *pChannel,
                                          IFX_uint32_t nMode)
{
   IFX_TAPI_DRV_CTX_t *pDrvCtx = pChannel->pTapiDevice->pDevDrvCtx;
   IFX_int32_t ret = TAPI_statusOk;

   if (nMode > 1)
   {
      /* unknown mode */
      RETURN_STATUS (TAPI_statusParam, 0);
   }

   /* check if PCM configuration was successful, if not - block activation */
   if (IFX_FALSE == pChannel->TapiPCMData.bCfgSuccess)
   {
      /* error: PCM channel not configured */
      RETURN_STATUS (TAPI_statusPCMChCfgError, 0);
   }

   /* chip specific function */
   if (ptr_chk(pDrvCtx->PCM.Enable, "pDrvCtx->PCM.Enable"))
   {
      ret = pDrvCtx->PCM.Enable (pChannel->pLLChannel, nMode,
                                 &(pChannel->TapiPCMData.PCMConfig));
      if (!TAPI_SUCCESS(ret))
      {
         RETURN_STATUS (TAPI_statusLLFailed, ret);
      }
   }
   else
   {
      ret = TAPI_statusLLNotSupp;
   }

   /* save activation level */
   if (ret == IFX_SUCCESS)
   {
      pChannel->TapiPCMData.bTimeSlotActive = (IFX_boolean_t)(nMode == 1);
   }

   return ret;
}

#ifdef TAPI_HDLC
/**
   Configure and activate the PCM channel with HDLC support

   \param pLLCh           Pointer to Low-level channel structure
   \param pHdlcCfg        Pointer to the HDLC configuration structure

   \return
      IFX_SUCCESS if successful
      IFX_ERROR if an error occured
*/
IFX_int32_t TAPI_Phone_PCM_HDLC_Set (TAPI_CHANNEL *pChannel,
                                     IFX_TAPI_PCM_HDLC_CFG_t const *pHdlcCfg)
{
   IFX_TAPI_DRV_CTX_t *pDrvCtx = pChannel->pTapiDevice->pDevDrvCtx;
   IFX_int32_t err;

   /* Check for LL driver functionality */
   if (!ptr_chk(pDrvCtx->PCM.HDLC_Cfg, "pDrvCtx->PCM.HDLC_Cfg"))
   {
      RETURN_STATUS (TAPI_statusLLNotSupp, 0);
   }

   err = pDrvCtx->PCM.HDLC_Cfg (pChannel->pLLChannel, pHdlcCfg);
   if (!TAPI_SUCCESS(err))
   {
      RETURN_STATUS (TAPI_statusLLFailed, err);
   }

   /* save activation level */
   pChannel->TapiPCMData.bTimeSlotActive =
                  (pHdlcCfg->nEnable == IFX_ENABLE) ? IFX_TRUE : IFX_FALSE;

   RETURN_STATUS (TAPI_statusOk, 0);
}
#endif /* TAPI_HDLC */

/**
   Configure and activate the Loop for two PCM channels

   \param pChannel      Pointer to TAPI_CHANNEL structure.
   \param pLoopCfg      Pointer to the PCM loop configuration structure.

   \return
      IFX_SUCCESS if successful
      IFX_ERROR if an error occured
*/
IFX_int32_t TAPI_Phone_PCM_Loop_Set (TAPI_CHANNEL *pChannel,
                                    IFX_TAPI_PCM_LOOP_CFG_t const *pLoopCfg)
{
   IFX_TAPI_DRV_CTX_t *pDrvCtx = pChannel->pTapiDevice->pDevDrvCtx;
   IFX_int32_t err;

   /* Check for LL driver functionality */
   if (!ptr_chk(pDrvCtx->PCM.Loop, "pDrvCtx->PCM.Loop"))
   {
      RETURN_STATUS (TAPI_statusLLNotSupp, 0);
   }

   err = pDrvCtx->PCM.Loop (pChannel->pLLChannel, pLoopCfg);
   if (!TAPI_SUCCESS(err))
   {
      RETURN_STATUS (TAPI_statusLLFailed, err);
   }

   RETURN_STATUS (TAPI_statusOk, 0);
}

/**
   Get the activation status from the pcm interface

   \param pChannel      Pointer to TAPI_CHANNEL structure.
   \param pbAct         Pointer to a boolean, returning the activation state.

   \return IFX_SUCCESS
*/
IFX_int32_t TAPI_Phone_PCM_Get_Activation (TAPI_CHANNEL *pChannel,
                                          IFX_boolean_t *pbAct)
{
   *pbAct = pChannel->TapiPCMData.bTimeSlotActive;
   return TAPI_statusOk;
}
