/******************************************************************************

                              Copyright (c) 2009
                            Lantiq Deutschland GmbH
                     Am Campeon 3; 85579 Neubiberg, Germany

  For licensing information, see the file 'LICENSE' in the root folder of
  this software module.

******************************************************************************/

/**
   \file drv_tapi_init.c
   Implements the creation and initialisation of the TAPI driver framework.

   This driver is loaded before all others low-level drivers are loaded.
   The low-level drivers which are loaded register with this driver and pass
   a so called driver context structure. This structure contains function
   pointers and variables for the services that the low-level driver offers.
*/

/* ============================= */
/* Includes                      */
/* ============================= */
#include "drv_api.h"
#include "drv_tapi.h"
#include "drv_tapi_errno.h"
#include "drv_tapi_stream.h"
#include "drv_tapi_version.h"
#ifdef TAPI_CID
#include "drv_tapi_cid.h"
#endif /* TAPI_CID */
#ifdef KPI_SUPPORT
#include "drv_tapi_kpi.h"
#endif /* KPI_SUPPORT */
#ifdef QOS_SUPPORT
#include "drv_tapi_qos.h"
#endif /* QOS_SUPPORT */
#ifdef TAPI_ANNOUNCEMENTS
#include "drv_tapi_announcements.h"
#endif /* TAPI_ANNOUNCEMENTS */


/* ============================= */
/* Local Macros & Definitions    */
/* ============================= */

#ifndef _MKSTR_1
#define _MKSTR_1(x)    #x
#define _MKSTR(x)      _MKSTR_1(x)
#endif

/** driver version as string */
#define DRV_TAPI_VER_STR         _MKSTR(DRV_TAPI_VER_MAJOR) "." \
                                 _MKSTR(DRV_TAPI_VER_MINOR) "." \
                                 _MKSTR(DRV_TAPI_VER_STEP)  "." \
                                 _MKSTR(DRV_TAPI_VER_TYPE)

/** low-level API version string */
#define DRV_TAPI_LL_IF_VER_STR   _MKSTR(LL_IF_MAJORSTEP)    "."   \
                                 _MKSTR(LL_IF_MINORSTEP)    "."   \
                                 _MKSTR(LL_IF_VERSIONSTEP)  "."   \
                                 _MKSTR(LL_IF_VERS_TYPE)

/** driver version, what string */
#define DRV_TAPI_WHAT_STR "@(#)Lantiq TAPI device driver, version " DRV_TAPI_VER_STR


/* ============================= */
/* Global variable definition    */
/* ============================= */
/* Array to register the driver context of the low-level drivers. */
IFX_TAPI_HL_DRV_CTX_t gHLDrvCtx [TAPI_MAX_LL_DRIVERS];

/* Variable that is used to set the initial debug level.
   In some OS implementations it can be overwritten when starting the driver. */
#ifdef DEBUG
IFX_uint32_t debug_level = DBG_LEVEL_LOW;
#else
IFX_uint32_t debug_level = DBG_LEVEL_NORMAL;
#endif /* DEBUG */

/** what string support, driver version string */
const IFX_char_t TAPI_WHATVERSION[] = DRV_TAPI_WHAT_STR;
#ifdef HAVE_CONFIG_H
/** which configure options were set */
const IFX_char_t DRV_TAPI_WHICHCONFIG[] = DRV_TAPI_CONFIGURE_STR;
#endif /* HAVE_CONFIG_H */


/* ============================= */
/* Local variable definition     */
/* ============================= */
static IFX_uint32_t nGlobalTapiID = 0;

/* ============================= */
/* Local function declaration    */
/* ============================= */
static IFX_int32_t ifx_tapi_DeviceCreate (IFX_TAPI_DRV_CTX_t *pDrvCtx);
static IFX_int32_t ifx_tapi_DevicePrepare (TAPI_DEV* pTapiDev,
                                           IFX_uint32_t dev_num,
                                           IFX_TAPI_DRV_CTX_t *pDrvCtx);
static IFX_int32_t ifx_tapi_DeviceInit    (TAPI_DEV* pTapiDev);
static IFX_int32_t ifx_tapi_DeviceExit    (TAPI_DEV* pTapiDev);
static IFX_int32_t ifx_tapi_DeviceDelete  (IFX_TAPI_DRV_CTX_t *pDrvCtx);

static IFX_int32_t ifx_tapi_ChannelCreate (TAPI_DEV *pTapiDev);
static IFX_int32_t ifx_tapi_ChannelPrepare(TAPI_CHANNEL *pTapiCh,
                                           IFX_uint32_t ch_num,
                                           TAPI_DEV* pTapiDev);
static IFX_int32_t ifx_tapi_ChannelInit   (TAPI_CHANNEL *pTapiCh);
static IFX_int32_t ifx_tapi_ChannelExit   (TAPI_CHANNEL *pTapiCh);

static IFX_int32_t ifx_tapi_InitCh        (TAPI_CHANNEL *pChannel);
static IFX_void_t  ifx_tapi_ExitCh        (TAPI_CHANNEL *pChannel);

static IFX_int32_t ifx_tapi_PKT_RTP_PT_Defaults (TAPI_CHANNEL *pChannel);


/* ============================= */
/* Global functions declaration  */
/* ============================= */


/* ============================= */
/* Local function definition     */
/* ============================= */


/* ============================= */
/* Global function definition    */
/* ============================= */

/**
   TAPI initialisation upon starting the driver.

   \return
   Returns IFX_ERROR in case of an error, otherwise returns IFX_SUCCESS.
*/
IFX_int32_t IFX_TAPI_Driver_Start(IFX_void_t)
{
   IFX_uint8_t i;
   IFX_int32_t ret;

   SetTraceLevel (TAPI_DRV, debug_level);

   /* Mark all slots as free in the global driver context array. */
   for (i=0; i < TAPI_MAX_LL_DRIVERS; i++)
   {
      /* Setting the pointer to NULL marks this slot as free. */
      gHLDrvCtx[i].pDrvCtx = IFX_NULL;
   }

   /* The bufferpool for voice packets is only created when a LL-driver
      registers that needs it. So when none of the registered LL drivers
      needs the pool it is not created. */

#ifdef KPI_SUPPORT
   /* Initialise the Kernel Packet Interface */
   IFX_TAPI_KPI_Init();
#endif /* KPI_SUPPORT */

#ifdef QOS_SUPPORT
   /* Initialise the QOS service */
   IFX_TAPI_QOS_Init();
#endif /* QOS_SUPPORT */

#ifdef TAPI_ANNOUNCEMENTS
   /* Initialize the announcement service */
   IFX_TAPI_Ann_Init ();
#endif /* TAPI_ANNOUNCEMENTS */

   /* The power save timer is created when the first LL-driver registers which
      needs it. */

   /* Configure the tonetable with predefined tones. */
   ret = TAPI_Phone_Tone_Predef_Config ();

   if (TAPI_SUCCESS(ret))
   {
      /* Initialise the TAPI event handler */
      ret = IFX_TAPI_Event_On_Driver_Start();
   }

   return ret;
}


/**
   TAPI cleanup upon stopping the driver.
*/
IFX_void_t IFX_TAPI_Driver_Stop(IFX_void_t)
{
   IFX_uint8_t i;

   /* Actually all LL drivers should have unregistered here. Being careful
      we force unregister of any drivers which may still be registered. */
   for (i = 0; i < TAPI_MAX_LL_DRIVERS; i++)
   {
      if (gHLDrvCtx[i].pDrvCtx != IFX_NULL)
      {
         IFX_TAPI_Unregister_LL_Drv (gHLDrvCtx[i].pDrvCtx->majorNumber);
      }
   }

   /* remove the Power Save timer */
   IFX_TAPI_PowerSave_Exit();

#ifdef TAPI_ANNOUNCEMENTS
   IFX_TAPI_Ann_Cleanup();
#endif /* TAPI_ANNOUNCEMENTS */

#ifdef QOS_SUPPORT
   /* Cleanup the QOS service */
   IFX_TAPI_QOS_Cleanup();
#endif /* QOS_SUPPORT */

#ifdef KPI_SUPPORT
   /* Cleanup the Kernel Packet Interface */
   IFX_TAPI_KPI_Cleanup();
#endif /* KPI_SUPPORT */

   IFX_TAPI_Event_On_Driver_Stop();

#ifdef TAPI_PACKET
   /* Destruct bufferpool for voice packets. */
   IFX_TAPI_VoiceBufferPool_Delete();
#endif /* TAPI_PACKET */
}


/**
   Create all TAPI devices and link them with the LL-devices.

   Allocate all TAPI device and channel structs if not already existing and
   prepare them by linking the structs to each other and the LL-driver
   equivalents.
   - Allocate all high-level TAPI device structures and store them in the
     device driver context.
   - Call prepare to link the structures to each other and the LL structs.
   - Allocate the high-level TAPI channel structures and store them in the
     high-level TAPI device structure.

   \param  pDrvCtx      Pointer to device driver context.

   \return
   Returns TAPI_statusOk in case of success, otherwise error code.
*/
static IFX_int32_t ifx_tapi_DeviceCreate (IFX_TAPI_DRV_CTX_t *pDrvCtx)
{
   TAPI_DEV *pTapiDev;
   IFX_uint16_t maxDevices, nDev;

   /* This is the number of devices that the LL driver supports. */
   maxDevices  = pDrvCtx->maxDevs;

   /* Allocate an array of TAPI device structures and store it in the low-level
      driver context. The array is created only the first time and kept until
      the corresponding LL-driver is unregistered. Only the devices that are
      adressed will be initialised. All others will stay uninitialised. */
   if (pDrvCtx->pTapiDev == IFX_NULL)
   {
      pDrvCtx->pTapiDev = (TAPI_DEV *)
                          TAPI_OS_Malloc (sizeof(TAPI_DEV)*maxDevices);
      if (pDrvCtx->pTapiDev == IFX_NULL)
      {
         /* Just a return since we do not have a device or channel yet where
            an error could be stored. */
         /* errmsg: Device structure allocation failed */
         return TAPI_statusDevAlloc;
      }
      /* the device structs are set to zero in the prepare function */

      /* prepare all tapi devices just created */
      for (nDev=0; nDev < maxDevices; nDev++)
      {
         pTapiDev = &(pDrvCtx->pTapiDev[nDev]);

         /* Prepare the TAPI device */
         if (ifx_tapi_DevicePrepare (pTapiDev, nDev, pDrvCtx) != TAPI_statusOk)
         {
            RETURN_DEVSTATUS (TAPI_statusInitFail, 0);
         }

         /* Create and prepare the TAPI channels of this device */
         if (ifx_tapi_ChannelCreate (pTapiDev) != TAPI_statusOk)
         {
            RETURN_DEVSTATUS (TAPI_statusInitFail, 0);
         }
      }
   }

   return TAPI_statusOk;
}


/**
   TAPI device structure preparation

   One time preparation of the TAPI device structure. This sets all members
   needed for file descriptor handling and links the HL- to the LL-struct.

   \param  pTapiDev     Pointer to TAPI device structure.
   \param  dev_num      Number of the device in the driver.
   \param  pDrvCtx      Pointer to device driver context.

   \return
   Returns IFX_SUCCESS or IFX_ERROR.
*/
static IFX_int32_t ifx_tapi_DevicePrepare (TAPI_DEV* pTapiDev,
                                           IFX_uint32_t dev_num,
                                           IFX_TAPI_DRV_CTX_t *pDrvCtx)
{
   /* set the device structure to zero */
   memset (pTapiDev, 0x00, sizeof(TAPI_DEV));

   /* set the channel-number to the magic number that indicates a device */
   pTapiDev->nChannel = IFX_TAPI_DEVICE_CH_NUMBER;

   /* assign a unique TAPI device ID */
   pTapiDev->nDevID = nGlobalTapiID++;

   /* store the device driver context */
   pTapiDev->pDevDrvCtx = pDrvCtx;

   /* Mutex to block concurrent IOCTLs on the device */
   TAPI_OS_MutexInit (&pTapiDev->semTapiDevSingleIoctlAccess);

   /* needed for select on the device fds */
   TAPI_OS_DrvSelectQueueInit (&pTapiDev->wqEvent);
   pTapiDev->bNeedWakeup = IFX_TRUE;

   /* reset the SmartSLIC specific handling of FXO ioctls */
   IFX_TAPI_Update_SlicFxo(pTapiDev, IFX_FALSE);

   /* call the LL driver prepare function */
   if (pDrvCtx->Prepare_Dev != IFX_NULL)
   {
      /* prepare the LL device and store the device pointer */
      pTapiDev->pLLDev = pDrvCtx->Prepare_Dev (pTapiDev, dev_num);
   }
   /* if LL function is not available or LL init failed return an error */
   if (pTapiDev->pLLDev == IFX_NULL)
   {
      RETURN_DEVSTATUS (TAPI_statusInitFail, 0);
   }

   return TAPI_statusOk;
}


/**
   TAPI device structure initialization

   Initializes all fifos, semaphores, memory and all other members.
   The TAPI channels are members of the device and also initialised here.

   \param  pTapiDev     Pointer to TAPI device structure.

   \return
   Returns IFX_SUCCESS or IFX_ERROR.
*/
IFX_int32_t ifx_tapi_DeviceInit (TAPI_DEV* pTapiDev)
{
   IFX_TAPI_DRV_CTX_t *pDrvCtx = pTapiDev->pDevDrvCtx;
   IFX_uint16_t nCh;
   IFX_int32_t ret = TAPI_statusErr;

   /* set the number of TAPI channels in this device */
   pTapiDev->nMaxChannel = pDrvCtx->maxChannels;

   /* set the resource count to maximum in case LL does not report anything */
   pTapiDev->nResource.AlmCount =
   pTapiDev->nResource.SigCount =
   pTapiDev->nResource.CodCount =
   pTapiDev->nResource.PcmCount =
   pTapiDev->nResource.DectCount =
   pTapiDev->nResource.AudioCount =
   pTapiDev->nResource.DTMFGCount =
   pTapiDev->nResource.DTMFRCount =
   pTapiDev->nResource.FSKGCount =
   pTapiDev->nResource.FSKRCount =
   pTapiDev->nResource.ToneCount =  pTapiDev->nMaxChannel;

#ifdef TAPI_PACKET
#ifdef TAPI_POLL
   if (pDrvCtx->Write != IFX_NULL)
   {
      /* Create downstream voice fifo */
      IFX_TAPI_DownStreamFifo_Create(pTapiDev);
   }
#endif /* TAPI_POLL */
#endif /* TAPI_PACKET */

   if (ptr_chk(pDrvCtx->Init_Dev, "pDrvCtx->Init_Dev"))
   {
      ret = pDrvCtx->Init_Dev (pTapiDev->pLLDev);
   }

   /* Initialise all TAPI channels of this device */
   for (nCh=0; TAPI_SUCCESS(ret) && (nCh < pTapiDev->nMaxChannel); nCh++)
   {
      TAPI_CHANNEL *pTapiCh = &(pTapiDev->pTapiChanelArray[nCh]);

      ret = ifx_tapi_ChannelInit (pTapiCh);
   }

   if (TAPI_SUCCESS(ret))
   {
      /* All the device members are now initialised */
      pTapiDev->bInitialized = IFX_TRUE;
   }

   RETURN_DEVSTATUS(ret, 0);
}


/**
   TAPI device structure shutdown

   Shutdown all resources in the TAPI device structure. Frees all needed
   fifos, semaphores, memory.
   The TAPI channel struct array and TAPI device struct array are kept until
   the LL driver context is unregistered.

   \param  pTapiDev     Pointer to TAPI device structure.

   \return
   Returns IFX_SUCCESS or IFX_ERROR.
*/
static IFX_int32_t ifx_tapi_DeviceExit (TAPI_DEV* pTapiDev)
{
   TAPI_CHANNEL *pTapiCh  = IFX_NULL;
   IFX_uint16_t i;

   TAPI_ASSERT(pTapiDev != IFX_NULL);

   /* Do channel specific cleanup */
   for (i = 0; i < pTapiDev->nMaxChannel; i++)
   {
      pTapiCh = &(pTapiDev->pTapiChanelArray[i]);

      ifx_tapi_ExitCh(pTapiCh);
      ifx_tapi_ChannelExit(pTapiCh);
   }

   /* placeholder - call a LL driver device exit function if needed */

#ifdef TAPI_PACKET
#ifdef TAPI_POLL
   if (pTapiDev->pDevDrvCtx->Write != IFX_NULL)
   {
      /* Destruct downstream voice packet fifo */
      IFX_TAPI_DownStreamFifo_Delete(pTapiDev);
   }
#endif /* TAPI_POLL */
#endif /* TAPI_PACKET */

   /* mark the device struct as unused and free */
   pTapiDev->bInitialized = IFX_FALSE;

   return TAPI_statusOk;
}


/**
   Delete all TAPI device with all their channels.

   Frees the TAPI device and channel resources. This function also frees the
   memory of the TAPI channel structs and all TAPI device structs.

   \param  pDrvCtx      Pointer to device driver context.

   \return
   Returns TAPI_statusOk in case of success, otherwise error code.

   \remarks
   CAUTION: this function deletes _all_ TAPI devices not just a single one.
*/
static IFX_int32_t ifx_tapi_DeviceDelete (IFX_TAPI_DRV_CTX_t *pDrvCtx)
{
   IFX_uint16_t nDev;

   if (pDrvCtx->pTapiDev == IFX_NULL)
   {
      return TAPI_statusOk;
   }

   for (nDev=0;  nDev < pDrvCtx->maxDevs; nDev++)
   {
      TAPI_DEV* pTapiDev = &(pDrvCtx->pTapiDev[nDev]);

      if (pTapiDev->bInitialized != IFX_FALSE)
      {
         /* cleanup the resources of the device (including all channels) */
         ifx_tapi_DeviceExit(pTapiDev);
      }

      if (pTapiDev->pTapiChanelArray != IFX_NULL)
      {
         /* this deletes all TAPI channels of the device at once */
         TAPI_OS_Free (pTapiDev->pTapiChanelArray);
         pTapiDev->pTapiChanelArray = IFX_NULL;
      }

      TAPI_OS_DrvSelectQueueDelete (&pTapiDev->wqEvent);
      TAPI_OS_MutexDelete (&pTapiDev->semTapiDevSingleIoctlAccess);
   }

   /* deallocate the entire array of TAPI devices stored in the LL-driver ctx */
   if (pDrvCtx->pTapiDev != IFX_NULL)
   {
      TAPI_OS_Free (pDrvCtx->pTapiDev);
      pDrvCtx->pTapiDev = IFX_NULL;
   }

   return TAPI_statusOk;
}


/**
   Create all TAPI channels in one device.

   Allocates the high-level TAPI channel structures and stores them in the
   high-level TAPI device structure. Then link the channels to the LL driver
   equivalents.

   \param  pTapiDev     Pointer to TAPI device structure.

   \return
   Returns TAPI_statusOk in case of success, otherwise error code.
*/
IFX_int32_t ifx_tapi_ChannelCreate (TAPI_DEV *pTapiDev)
{
   IFX_TAPI_DRV_CTX_t *pDrvCtx = pTapiDev->pDevDrvCtx;
   TAPI_CHANNEL *pTapiCh;
   IFX_uint16_t maxChannels, nCh;

   /* This is the number of channels that the LL driver supports for each dev.*/
   maxChannels = pDrvCtx->maxChannels;

   if (pTapiDev->pTapiChanelArray == IFX_NULL)
   {
      /* Allocate an array of TAPI channel structures and store it in the
         device. */
      pTapiDev->pTapiChanelArray =
         (TAPI_CHANNEL *) TAPI_OS_Malloc (sizeof(TAPI_CHANNEL) * maxChannels);
      if (pTapiDev->pTapiChanelArray == IFX_NULL)
      {
         /* errmsg: Channel structure allocation failed */
         RETURN_DEVSTATUS (TAPI_statusChAlloc, 0);
      }
      /* the channel structs are set to zero in the prepare function */

      /* Prepare all TAPI channels of this device */
      for (nCh=0; nCh < maxChannels ; nCh++)
      {
         pTapiCh  = &(pTapiDev->pTapiChanelArray[nCh]);

         if (ifx_tapi_ChannelPrepare (pTapiCh, nCh, pTapiDev) != TAPI_statusOk)
         {
            /* On error deallocate the TAPI channel structures again */
            TAPI_OS_Free (pTapiDev->pTapiChanelArray);
            pTapiDev->pTapiChanelArray = IFX_NULL;
            RETURN_DEVSTATUS (TAPI_statusInitFail, 0);
         }
      }
   }

   return TAPI_statusOk;
}


/**
   TAPI channel structure preparation

   One time preparation of the TAPI channel structures. This sets all members
   needed for file descriptor handling.

   \param  pTapiCh      Pointer to TAPI channel structure to be initialised.
   \param  ch_num       Number of the channel in the device.
   \param  pTapiDev     Pointer to TAPI device structure.
*/
static IFX_int32_t ifx_tapi_ChannelPrepare (TAPI_CHANNEL *pTapiCh,
                                            IFX_uint32_t ch_num,
                                            TAPI_DEV* pTapiDev)
{
   IFX_TAPI_DRV_CTX_t      *pDrvCtx;

   /* set the channel structure to zero */
   memset (pTapiCh, 0x00, sizeof(TAPI_CHANNEL));

   /* set the channel-number */
   pTapiCh->nChannel = ch_num;

   /* store the pointer to the TAPI device structure */
   pTapiCh->pTapiDevice = pTapiDev;

   /* call the LL driver prepare function */
   pDrvCtx = pTapiDev->pDevDrvCtx;
   if (pDrvCtx->Prepare_Ch != IFX_NULL)
   {
      /* prepare the LL channel and store the channel pointer */
      pTapiCh->pLLChannel = pDrvCtx->Prepare_Ch (pTapiCh,
                                                 pTapiDev->pLLDev,
                                                 pTapiCh->nChannel);
   }
   /* if LL function is not available or LL init failed return an error */
   if (pTapiCh->pLLChannel == IFX_NULL)
   {
      RETURN_DEVSTATUS (TAPI_statusInitFail, 0);
   }

   /* select wait queue and semaphore for reading/writing data */
   TAPI_OS_EventInit (&pTapiCh->semReadBlock);

   TAPI_OS_DrvSelectQueueInit (&pTapiCh->wqRead);
   TAPI_OS_DrvSelectQueueInit (&pTapiCh->wqWrite);

   return TAPI_statusOk;
}


/**
   One time TAPI channel structure initialization

   Initializes all needed fifos, semaphores and sets all needed members.

   \param  pTapiCh      Pointer to TAPI channel structure.

   \return
   Returns IFX_ERROR in case of an error, otherwise returns IFX_SUCCESS.
*/
static IFX_int32_t ifx_tapi_ChannelInit (TAPI_CHANNEL *pTapiCh)
{
   IFX_TAPI_DRV_CTX_t      *pDrvCtx = pTapiCh->pTapiDevice->pDevDrvCtx;
   TAPI_DEV                *pTapiDev = pTapiCh->pTapiDevice;
   IFX_int32_t             retLL = TAPI_statusErr;

   /* Initialise the high level channel access mutex */
   TAPI_OS_MutexInit (&pTapiCh->semTapiChSingleIoctlAccess);
   TAPI_OS_MutexInit (&pTapiCh->semTapiChDataLock);

   /* Initialise the event dispatcher */
   if (IFX_TAPI_EventDispatcher_Init(pTapiCh) != TAPI_statusOk)
   {
      RETURN_DEVSTATUS (TAPI_statusInitFail, 0);
   }

#ifdef TAPI_PACKET
   /* Initialize upstream voice packet fifo */
   IFX_TAPI_UpStreamFifo_Create(pTapiCh);
#endif /* TAPI_PACKET */

   if (pDrvCtx->Init_Ch != IFX_NULL)
   {
      /* intialise the LL channel */
      retLL = pDrvCtx->Init_Ch (pTapiCh->pLLChannel);
      if (!TAPI_SUCCESS(retLL))
      {
         /* LL init failed -> return an error */
         RETURN_DEVSTATUS (TAPI_statusInitFail, retLL);
      }
   }
   else
   {
      /* LL function is not available -> return an error */
      RETURN_DEVSTATUS (TAPI_statusLLNotSupp, 0);
   }

   pTapiCh->TapiOpControlData.nLineMode = IFX_TAPI_LINE_FEED_DISABLED;

   /* At initialisation time there are no requests. */
   pTapiCh->bFaxDataRequest = IFX_FALSE;
   pTapiCh->bLinChannelActive = IFX_FALSE;
   pTapiCh->bLinDataRequest = IFX_FALSE;

   /* this channel is now in use */
   pTapiCh->nInUse++;

   return TAPI_statusOk;
}


/**
   TAPI channel structure shutdown

   Shutdown all resources in the TAPI channel structure.
   Frees all needed fifos, event queues, memory.

   \param  pTapiCh      Pointer to TAPI channel structure.

   \return
   Returns IFX_ERROR in case of an error, otherwise returns IFX_SUCCESS.
*/
static IFX_int32_t ifx_tapi_ChannelExit (TAPI_CHANNEL *pTapiCh)
{
   TAPI_ASSERT(pTapiCh != IFX_NULL);

   /* placeholder - call the LL driver channel exit function */

#ifdef TAPI_PACKET
   /* Delete the upstream voice fifo */
   IFX_TAPI_UpStreamFifo_Delete(pTapiCh);
#endif /* TAPI_PACKET */

   /* Exit the event dispatcher */
   if (IFX_TAPI_EventDispatcher_Exit(pTapiCh) != TAPI_statusOk)
   {
      return TAPI_statusErr;
   }

   /* Delete queues for waiting in select */
   TAPI_OS_DrvSelectQueueDelete (&pTapiCh->wqRead);
   TAPI_OS_DrvSelectQueueDelete (&pTapiCh->wqWrite);

   /* Delete the high level channel access mutex */
   TAPI_OS_MutexDelete (&pTapiCh->semTapiChSingleIoctlAccess);
   TAPI_OS_MutexDelete (&pTapiCh->semTapiChDataLock);

   return TAPI_statusOk;
}


/**
   Init one TAPI channel

   \param  pChannel     Pointer to TAPI_CHANNEL structure.

   \return:
    returncode: IFX_SUCCESS  -> init successful
                IFX_ERROR    -> init failed
*/
static IFX_int32_t ifx_tapi_InitCh (TAPI_CHANNEL *pChannel)
{
   /* ask all TAPI modules to initialise themselves */
   if (IFX_TAPI_Dial_Initialise (pChannel) != TAPI_statusOk)
   {
      RETURN_STATUS (TAPI_statusInitDialFail ,0);
   }
   if (IFX_TAPI_Tone_Initialise(pChannel) != TAPI_statusOk)
   {
      RETURN_STATUS (TAPI_statusInitToneFail, 0);
   }
#ifdef TAPI_CID
   if (IFX_TAPI_CID_Initialise (pChannel) != TAPI_statusOk)
   {
      RETURN_STATUS (TAPI_statusInitCIDFail, 0);
   }
#endif /* TAPI_CID */
   /* Init ring after CID because ring will ask for CID status during init */
   if (IFX_TAPI_Ring_Initialise (pChannel) != TAPI_statusOk)
   {
      RETURN_STATUS (TAPI_statusInitRingFail ,0);
   }
#ifdef TAPI_METERING
   if (IFX_TAPI_Meter_Initialise (pChannel) != TAPI_statusOk)
   {
      RETURN_STATUS (TAPI_statusInitMeterFail ,0);
   }
#endif /* TAPI_METERING */
#ifdef TAPI_STATISTICS
   if (IFX_TAPI_Stat_Initialise (pChannel) != TAPI_statusOk)
   {
      RETURN_STATUS (TAPI_statusInitFail ,0);
   }
#endif /* TAPI_STATISTICS */

   pChannel->bInitialized = IFX_TRUE;

   return TAPI_statusOk;
}


/**
   Cleanup one TAPI channel

   \param  pChannel     Pointer to TAPI_CHANNEL structure.
*/
static IFX_void_t ifx_tapi_ExitCh (TAPI_CHANNEL *pChannel)
{
#ifdef TAPI_STATISTICS
   IFX_TAPI_Stat_Cleanup (pChannel);
#endif /* TAPI_STATISTICS */
#ifdef TAPI_METERING
   IFX_TAPI_Meter_Cleanup(pChannel);
#endif /* TAPI_METERING */
   IFX_TAPI_Ring_Cleanup(pChannel);
#ifdef TAPI_CID
   IFX_TAPI_CID_Cleanup(pChannel);
#endif /* TAPI_CID */
   IFX_TAPI_Tone_Cleanup(pChannel);

   IFX_TAPI_Dial_Cleanup(pChannel);

   pChannel->bInitialized = IFX_FALSE;
}


/**
   Set the resource count struct.

   This function is intended to be called by the LL driver as soon as it knows
   how many resources will be available. The values given will be stored and
   used by TAPI functions to check for needed resources before they try to
   use them.

   \param   pTapiDev    Pointer to a TAPI device context.
   \param   pResources  Pointer to struct with resource count.
*/
IFX_void_t IFX_TAPI_ReportResources (TAPI_DEV* pTapiDev,
                                     IFX_TAPI_RESOURCE *pResources)
{
   pTapiDev->nResource = *pResources;
}


/**
   Register the low-level driver.

   This function is called by the low-level driver when this is started.
   It does the following:
      - Checks that HL- and LL-driver versions are matching.
      - Stores the low-level device driver context in a global array
      - Call the OS specific driver registration function.

   \param  pLLDrvCtx    Pointer to device driver context which is provided by
                        the low-level driver. The driver context may only be
                        used from registration until it is unregistered. The
                        memory for the context is at all times owned by the
                        LL-driver who has to keep it available while it is
                        registered.
   \return
   - TAPI_statusOk - Success
   - TAPI_statusInitLL - Driver context pointer must not be NULL
   - TAPI_StatusInitLLRegVersMismatch - Version missmatch of HL and LL driver
   - TAPI_statusRegLLExists - Major number already registered
   - TAPI_statusInitLLReg - No free slot to register the driver
*/
IFX_int32_t IFX_TAPI_Register_LL_Drv (IFX_TAPI_DRV_CTX_t* pLLDrvCtx)
{
   IFX_TAPI_HL_DRV_CTX_t* pHLDrvCtx = IFX_NULL;
   IFX_uint8_t i;
   IFX_size_t l;
   IFX_int32_t ret = TAPI_statusOk;

   /* Sanity check of the function parameter. */
   if (pLLDrvCtx == IFX_NULL)
   {
      return TAPI_statusInitLL;
   }

   l = strlen(pLLDrvCtx->hlLLInterfaceVersion);

   /* Make sure HL- and LL-driver interface versions are matching. */
   if ((l != strlen(DRV_TAPI_LL_IF_VER_STR)) ||
      strncmp (pLLDrvCtx->hlLLInterfaceVersion, DRV_TAPI_LL_IF_VER_STR, l))
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
         ("TAPI: ATTENTION - mismatch of HL-LL Interface\n"));
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
         ("TAPI: please check that drv_tapi and drv_%s driver match.\n",
            pLLDrvCtx->drvName));
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
         ("Version set in LL Driver = %s\n", pLLDrvCtx->hlLLInterfaceVersion));
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
         ("Version expected by HL   = %s\n", DRV_TAPI_LL_IF_VER_STR));
      return TAPI_StatusInitLLRegVersMismatch;
   }

   /* Don't allow more than one device driver to use the same major number. */
   if (IFX_TAPI_DeviceDriverContextGet (pLLDrvCtx->majorNumber) != IFX_NULL)
   {
      /* Here somebody is already using the given major number. */
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Cannot register LL-driver. "
            "LL-driver with this major number is already registered.\n"));
      return TAPI_statusRegLLExists;
   }

   /* Find a free index in gHLDrvCtx. */
   for (i = 0; i < TAPI_MAX_LL_DRIVERS; i++)
   {
      if (gHLDrvCtx[i].pDrvCtx == IFX_NULL)
      {
         /* use this index */
         pHLDrvCtx = &gHLDrvCtx[i];
         /* store the low-level driver context (now this index is in use) */
         pHLDrvCtx->pDrvCtx = pLLDrvCtx;
         break;
      }
   }

   /* Could not find free index. */
   if (pHLDrvCtx == IFX_NULL)
   {
      return TAPI_statusInitLLReg;
   }

   /* Create TAPI devices for all devices that the LL driver supports. */
   ret = ifx_tapi_DeviceCreate (pLLDrvCtx);

   /* OS specific registration of the driver */
   if (TAPI_SUCCESS(ret))
   {
      ret = TAPI_OS_RegisterLLDrv (pLLDrvCtx, pHLDrvCtx);
   }

#ifdef TAPI_PACKET
   /* The bufferpool for voice packets is only created when a LL-driver
      needs it. It is created at registration so that it is already available
      for the IOCTLs that download and setup the voice firmware. */
   if ((pLLDrvCtx->Write != IFX_NULL) || (pLLDrvCtx->bProvidePktRead))
   {
      /* Create bufferpool for voice packets. */
      IFX_TAPI_VoiceBufferPool_Create();
   }
#endif /* TAPI_PACKET */

   return ret;
}


/**
   UnRegister the low-level driver

   This function is called by the low-level driver when this is stopped.
   It does the following:
      - Call the OS specific driver deregistration function.
      - Free the high-level TAPI device structures
      - Forget the low-device driver context

   \param  majorNumber  Device major number.

   \return \ref IFX_ERROR on error, otherwise \ref TAPI_statusOk
   - TAPI_statusOk - Success
   - TAPI_statusInitLLReg - Unregister failed: major number is not registered
*/
IFX_int32_t IFX_TAPI_Unregister_LL_Drv (IFX_int32_t majorNumber)
{
   IFX_TAPI_HL_DRV_CTX_t *pHLDrvCtx = IFX_NULL;
   IFX_TAPI_DRV_CTX_t    *pLLDrvCtx;
   IFX_uint8_t i;
   IFX_return_t ret = TAPI_statusOk;

   TRACE (TAPI_DRV, DBG_LEVEL_HIGH,
      ("INFO: Low level driver unregistering with major number %d\n",
      majorNumber));

   /* find registered driver and get the high-level driver context */
   for (i = 0; i < TAPI_MAX_LL_DRIVERS; i++)
   {
      if (gHLDrvCtx[i].pDrvCtx != IFX_NULL)
      {
         if (majorNumber == gHLDrvCtx [i].pDrvCtx->majorNumber)
         {
            pHLDrvCtx = &gHLDrvCtx [i];
         }
      }
   }
   /* not registered or no high-level driver context */
   if (pHLDrvCtx == IFX_NULL)
   {
      return TAPI_statusInitLLReg;
   }

   /* this is the corresponding low-level driver context */
   pLLDrvCtx = pHLDrvCtx->pDrvCtx;

   ret = TAPI_OS_UnregisterLLDrv (pLLDrvCtx, pHLDrvCtx);

   /* delete all devices in this driver context */
   ifx_tapi_DeviceDelete (pLLDrvCtx);

   /* forget the low-level driver context (now this index is free again) */
   pHLDrvCtx->pDrvCtx = IFX_NULL;

   return ret;
}


/**
   Returns the pointer to the low-level context

   Compares the major number with all the device driver's major numbers and
   if it matches, returns the low-level driver context.

   \param  Major        Device major number.

   \return
   If found it returns the pointer to the low-level context, otherwise IFX_NULL.
*/
IFX_TAPI_DRV_CTX_t* IFX_TAPI_DeviceDriverContextGet (IFX_int32_t Major)
{
   IFX_uint8_t i;

   for (i = 0; i < TAPI_MAX_LL_DRIVERS; i++)
   {
      if (gHLDrvCtx[i].pDrvCtx != IFX_NULL)
      {
         if (Major == gHLDrvCtx[i].pDrvCtx->majorNumber)
         {
            return gHLDrvCtx[i].pDrvCtx;
         }
      }
   }
   return IFX_NULL;
}


#ifdef TAPI_EXT_KEYPAD
/**
   Returns the context of the first channel of first device of the VMMC driver.

   When a key event is reported from hapi the channel context is not known.
   This is used to get 0th channel since Audio has only one channel for INCA2

   \return
   TAPI_CHANNEL pointer or IFX_NULL if no matching driver could be found.
*/
TAPI_CHANNEL *IFX_TAPI_IncaChannelContextGet(IFX_void_t)
{
   TAPI_CHANNEL *pChannel = IFX_NULL;
   IFX_uint8_t i;

   for (i=0; i < TAPI_MAX_LL_DRIVERS; i++)
   {
      if ((gHLDrvCtx[i].pDrvCtx != IFX_NULL) &&
          (strcmp(gHLDrvCtx[i].pDrvCtx->drvName, "vmmc") == 0))
      {
         /* TRACE( TAPI_DRV, DBG_LEVEL_LOW,("Found vmmc at pos. i= %d\n",i)); */
         pChannel = gHLDrvCtx[i].pDrvCtx->pTapiDev->pTapiChanelArray;
         break;
      }
   }

   return pChannel;
}
#endif /* TAPI_EXT_KEYPAD */


/**
   Set RTP Payloadtype defaults.

   \param  pChannel     Pointer to a TAPI channel structure.

   \return \ref IFX_SUCCESS or \ref IFX_ERROR

   \note This function should be called on devices with encoders only,
      which should be checked like:
         if (pDrvCtx->COD.RTP_PayloadTable_Cfg != IFX_NULL)
         {
            err = ifx_tapi_PKT_RTP_PT_Defaults(pChannel);
         }
*/
IFX_int32_t ifx_tapi_PKT_RTP_PT_Defaults (TAPI_CHANNEL *pChannel)
{
   IFX_TAPI_DRV_CTX_t       *pDrvCtx = pChannel->pTapiDevice->pDevDrvCtx;
   IFX_TAPI_PKT_RTP_PT_CFG_t pt;
   IFX_int32_t               ret;

   /* \todo add check for AAL2 mode - in this case we'd simply return */

   memset(&pt, 0, sizeof(IFX_TAPI_PKT_RTP_PT_CFG_t));
   pt.nPTup[IFX_TAPI_COD_TYPE_G723_63]     = 0x04;
   pt.nPTup[IFX_TAPI_COD_TYPE_G723_53]     = 0x04;
   pt.nPTup[IFX_TAPI_COD_TYPE_G728]        = 0x0F; /* fixed acc. RFC 3551 */
   pt.nPTup[IFX_TAPI_COD_TYPE_G729]        = 0x12; /* fixed acc. RFC 3551 */
   pt.nPTup[IFX_TAPI_COD_TYPE_MLAW]        = 0x00; /* fixed acc. RFC 3551 */
   pt.nPTup[IFX_TAPI_COD_TYPE_ALAW]        = 0x08; /* fixed acc. RFC 3551 */
   pt.nPTup[IFX_TAPI_COD_TYPE_G726_16]     = 0x63;
   pt.nPTup[IFX_TAPI_COD_TYPE_G726_24]     = 0x64;
   pt.nPTup[IFX_TAPI_COD_TYPE_G726_32]     = 0x65;
   pt.nPTup[IFX_TAPI_COD_TYPE_G726_40]     = 0x66;
   pt.nPTup[IFX_TAPI_COD_TYPE_G729_E]      = 0x61;
   pt.nPTup[IFX_TAPI_COD_TYPE_ILBC_152]    = 0x67;
   pt.nPTup[IFX_TAPI_COD_TYPE_ILBC_133]    = 0x67;
   pt.nPTup[IFX_TAPI_COD_TYPE_LIN16_16]    = 0x70;
   pt.nPTup[IFX_TAPI_COD_TYPE_LIN16_8]     = 0x71;
   pt.nPTup[IFX_TAPI_COD_TYPE_AMR_4_75]    = 0x72;
   pt.nPTup[IFX_TAPI_COD_TYPE_AMR_5_9]     = 0x72;
   pt.nPTup[IFX_TAPI_COD_TYPE_AMR_5_15]    = 0x72;
   pt.nPTup[IFX_TAPI_COD_TYPE_AMR_6_7]     = 0x72;
   pt.nPTup[IFX_TAPI_COD_TYPE_AMR_7_4]     = 0x72;
   pt.nPTup[IFX_TAPI_COD_TYPE_AMR_7_95]    = 0x72;
   pt.nPTup[IFX_TAPI_COD_TYPE_AMR_10_2]    = 0x72;
   pt.nPTup[IFX_TAPI_COD_TYPE_AMR_12_2]    = 0x72;
   pt.nPTup[IFX_TAPI_COD_TYPE_G722_64]     = 0x09;
   pt.nPTup[IFX_TAPI_COD_TYPE_G7221_24]    = 0x73;
   pt.nPTup[IFX_TAPI_COD_TYPE_G7221_32]    = 0x74;
   pt.nPTup[IFX_TAPI_COD_TYPE_MLAW_VBD]    = 0x75;
   pt.nPTup[IFX_TAPI_COD_TYPE_ALAW_VBD]    = 0x76;

   pt.nPTdown[IFX_TAPI_COD_TYPE_G723_63]   = 0x04;
   pt.nPTdown[IFX_TAPI_COD_TYPE_G723_53]   = 0x04;
   pt.nPTdown[IFX_TAPI_COD_TYPE_G728]      = 0x0F; /* fixed acc. RFC 3551 */
   pt.nPTdown[IFX_TAPI_COD_TYPE_G729]      = 0x12; /* fixed acc. RFC 3551 */
   pt.nPTdown[IFX_TAPI_COD_TYPE_MLAW]      = 0x00; /* fixed acc. RFC 3551 */
   pt.nPTdown[IFX_TAPI_COD_TYPE_ALAW]      = 0x08; /* fixed acc. RFC 3551 */
   pt.nPTdown[IFX_TAPI_COD_TYPE_G726_16]   = 0x63;
   pt.nPTdown[IFX_TAPI_COD_TYPE_G726_24]   = 0x64;
   pt.nPTdown[IFX_TAPI_COD_TYPE_G726_32]   = 0x65;
   pt.nPTdown[IFX_TAPI_COD_TYPE_G726_40]   = 0x66;
   pt.nPTdown[IFX_TAPI_COD_TYPE_G729_E]    = 0x61;
   pt.nPTdown[IFX_TAPI_COD_TYPE_ILBC_152]  = 0x67;
   pt.nPTdown[IFX_TAPI_COD_TYPE_ILBC_133]  = 0x67;
   pt.nPTdown[IFX_TAPI_COD_TYPE_LIN16_16]  = 0x70;
   pt.nPTdown[IFX_TAPI_COD_TYPE_LIN16_8]   = 0x71;
   pt.nPTdown[IFX_TAPI_COD_TYPE_AMR_4_75]  = 0x72;
   pt.nPTdown[IFX_TAPI_COD_TYPE_AMR_5_9]   = 0x72;
   pt.nPTdown[IFX_TAPI_COD_TYPE_AMR_5_15]  = 0x72;
   pt.nPTdown[IFX_TAPI_COD_TYPE_AMR_6_7]   = 0x72;
   pt.nPTdown[IFX_TAPI_COD_TYPE_AMR_7_4]   = 0x72;
   pt.nPTdown[IFX_TAPI_COD_TYPE_AMR_7_95]  = 0x72;
   pt.nPTdown[IFX_TAPI_COD_TYPE_AMR_10_2]  = 0x72;
   pt.nPTdown[IFX_TAPI_COD_TYPE_AMR_12_2]  = 0x72;
   pt.nPTdown[IFX_TAPI_COD_TYPE_G722_64]   = 0x09;
   pt.nPTdown[IFX_TAPI_COD_TYPE_G7221_24]  = 0x73;
   pt.nPTdown[IFX_TAPI_COD_TYPE_G7221_32]  = 0x74;
   pt.nPTdown[IFX_TAPI_COD_TYPE_MLAW_VBD]  = 0x75;
   pt.nPTdown[IFX_TAPI_COD_TYPE_ALAW_VBD]  = 0x76;

   if (ptr_chk(pDrvCtx->COD.RTP_PayloadTable_Cfg,
              "pDrvCtx->COD.RTP_PayloadTable_Cfg"))
   {
      ret = pDrvCtx->COD.RTP_PayloadTable_Cfg(pChannel->pLLChannel, &pt);
   }
   else
   {
      RETURN_STATUS (TAPI_statusRTPConf, 0);
   }

   return ret;
}


/**
   Start the TAPI for a specified device.

   \param  pTapiDev     Pointer to TAPI device structure.
   \param  pDevStartCfg Pointer to IFX_TAPI_DEV_START_CFG_t or IFX_NULL.
                        If the parameter is not needed IFX_NULL must be passed.

   \return
   Returns TAPI_statusOk in case of success, otherwise error code.
*/
IFX_int32_t IFX_TAPI_DeviceStart (TAPI_DEV *pTapiDev,
                                  IFX_TAPI_DEV_START_CFG_t *pDevStartCfg)
{
   IFX_TAPI_DRV_CTX_t  *pDrvCtx   = pTapiDev->pDevDrvCtx;
   IFX_uint16_t         i;
   IFX_uint8_t          nMode = IFX_TAPI_INIT_MODE_VOICE_CODER;
   IFX_int32_t          ret = TAPI_statusOk,
                        retLL = TAPI_statusOk;

   if (pTapiDev->bInitialized != IFX_TRUE)
   {
      ret = ifx_tapi_DeviceInit (pTapiDev);
   }

   if (ptr_chk(pDrvCtx->FW_Init, "pDrvCtx->FW_Init"))
   {
      if (pDevStartCfg != IFX_NULL)
      {
         /* If parameter is given overwrite the default that is set above. */
         nMode = pDevStartCfg->nMode;
      }
      retLL = pDrvCtx->FW_Init (pTapiDev->pLLDev, nMode);
      if (!TAPI_SUCCESS(retLL))
      {
         ret = TAPI_statusInitFail;
      }
   }
   else
   {
      ret = TAPI_statusInitFail;
   }

   /* Initialize all TAPI channel members, if not already done. */
   for (i = 0; (ret == TAPI_statusOk) && (i < pTapiDev->nMaxChannel); i++)
   {
      if (pTapiDev->pTapiChanelArray[i].bInitialized == IFX_FALSE)
      {
         ret = ifx_tapi_InitCh(&pTapiDev->pTapiChanelArray[i]);
      }
   }

   /* Configure default PTs if LL device supports codecs. */
   for (i = 0; (ret == TAPI_statusOk) && (i < pTapiDev->nMaxChannel); i++)
   {
      if ((pDrvCtx->COD.RTP_PayloadTable_Cfg != IFX_NULL) &&
          (pTapiDev->pTapiChanelArray[i].nChannel < pTapiDev->nResource.CodCount))
      {
         retLL = ifx_tapi_PKT_RTP_PT_Defaults(&pTapiDev->pTapiChanelArray[i]);
         if (!TAPI_SUCCESS(retLL))
         {
            ret = TAPI_statusInitFail;
         }
      }
   }

   /* If the LL-driver supports a power save function start the polling that
      periodically calls this function. */
   if (TAPI_SUCCESS(ret) &&
       ptr_chk(pDrvCtx->Pwr_Save_Dev, "pDrvCtx->Pwr_Save_Dev"))
   {
      ret = IFX_TAPI_PowerSave_Init();
   }

   if (!TAPI_SUCCESS(ret))
   {
      /* In case of an error try to roll-back. */
      IFX_TAPI_DeviceStop(pTapiDev);
   }

   RETURN_DEVSTATUS(ret, retLL);
}


/**
   Stop the TAPI for a specified device.

   \param  pTapiDev     Pointer to TAPI device structure to be stopped.

   \return
   - TAPI_statusOk - Success
*/
IFX_int32_t IFX_TAPI_DeviceStop(TAPI_DEV *pTapiDev)
{
   IFX_TAPI_DRV_CTX_t  *pDrvCtx   = pTapiDev->pDevDrvCtx;
   IFX_int32_t          ret = TAPI_statusOk;

   if (pTapiDev->bInitialized != IFX_FALSE)
   {
      if (ptr_chk(pDrvCtx->Exit_Dev, "pDrvCtx->Exit_Dev"))
      {
         pDrvCtx->Exit_Dev (pTapiDev->pLLDev, IFX_TRUE);

         /* if LL support is missing this is not executed */
         ret = ifx_tapi_DeviceExit (pTapiDev);
      }
      else
      {
         /* errmsg: Service is not supported by the low level driver*/
         ret = TAPI_statusLLNotSupp;
      }
   }

   RETURN_DEVSTATUS(ret, 0);
}


/**
   Reset TAPI states

   This function is intended to be issued after an HW reset. It will reset
   all states that TAPI keeps about the device. During this function only
   SW structures will be reset; no HW access will be done.

   \param  pTapiDev     Pointer to TAPI device structure to be stopped.
*/
IFX_void_t IFX_TAPI_DeviceReset (TAPI_DEV *pTapiDev)
{
   IFX_TAPI_DRV_CTX_t  *pDrvCtx   = pTapiDev->pDevDrvCtx;

   if (pTapiDev->bInitialized != IFX_FALSE)
   {
      if (ptr_chk(pDrvCtx->Exit_Dev, "pDrvCtx->Exit_Dev"))
      {
         pDrvCtx->Exit_Dev (pTapiDev->pLLDev, IFX_FALSE);

         /* if LL support is missing this is not executed */
         ifx_tapi_DeviceExit (pTapiDev);
      }
   }
}


/**
   Legacy function to initialise the TAPI.

   This function is kept for the old IOCTL IFX_TAPI_CH_INIT. The IOCTL is
   channel specific but the init is for the entire device.

   \param  pTapiDev     Pointer to TAPI device structure.
   \param  pInit        Pointer to IFX_TAPI_CH_INIT_t structure.

   \return
   error code: IFX_SUCCESS  -> init successful
               IFX_ERROR    -> init not successful
*/
IFX_int32_t IFX_TAPI_Phone_Init(TAPI_DEV *pTapiDev,
                                IFX_TAPI_CH_INIT_t const *pInit)
{
   IFX_TAPI_DRV_CTX_t  *pDrvCtx   = pTapiDev->pDevDrvCtx;
   IFX_int32_t          ret = TAPI_statusInitFail;

   if (ptr_chk(pDrvCtx->FW_Start, "pDrvCtx->FW_Start"))
   {
      IFX_void_t *pProc = (pInit == NULL) ? IFX_NULL : pInit->pProc;

      ret = pDrvCtx->FW_Start (pTapiDev->pLLDev, pProc);
   }

   /* +++++ from here on the firmware is available +++++ */

   if (pTapiDev->bInitialized != IFX_TRUE)
   {
      if (TAPI_SUCCESS(ret))
      {
         IFX_TAPI_DEV_START_CFG_t nDevStartCfg;

         memset(&nDevStartCfg, 0x00, sizeof(nDevStartCfg));
         nDevStartCfg.nMode =
            (pInit == NULL) ? IFX_TAPI_INIT_MODE_VOICE_CODER : pInit->nMode;

         ret = IFX_TAPI_DeviceStart(pTapiDev, &nDevStartCfg);
      }

      /* Do BBD download */
      if (TAPI_SUCCESS(ret) && ptr_chk(pDrvCtx->BBD_Dnld, "pDrvCtx->BBD_Dnld"))
      {
         IFX_void_t *pProc = (pInit == NULL) ? IFX_NULL : pInit->pProc;

         ret = pDrvCtx->BBD_Dnld (pTapiDev->pLLDev, pProc);
      }
   }

   return ret;
}

/**
   Update the SmartSLIC FXO boolean flag.

   The flag is required to indicate systems which have
   SmartSLIC and fused FXO line(s).
   For such systems a special handling of FXO ioctls is required.

   \param  pTapiDev     Pointer to TAPI device structure.
   \param  bVal         new flag value.

   \return
      none

   \remarks
      This function can be called from LL driver.
*/
IFX_void_t IFX_TAPI_Update_SlicFxo(TAPI_DEV *pTapiDev, IFX_boolean_t bVal)
{
   pTapiDev->bSmartSlicFxo = bVal;
}
