/******************************************************************************

                              Copyright (c) 2009
                            Lantiq Deutschland GmbH
                     Am Campeon 3; 85579 Neubiberg, Germany

  For licensing information, see the file 'LICENSE' in the root folder of
  this software module.

******************************************************************************/

/**
   \file drv_tapi_errno.h
   Private file for error handling.
*/

/* ============================= */
/* Includes                      */
/* ============================= */

/* ============================= */
/* Local Macros  Definitions    */
/* ============================= */

/*Errorblock*/
/** Enumeration for function return status. The upper four bits are reserved for
    error clasification */
typedef enum
{
   /** Success - no error */
   TAPI_statusOk = TAPI_statusClassSuccess,
   /** Setting line mode failed */
   TAPI_statusLineModeFail = TAPI_statusClassErr | 0x1,
   /** Starting CID Info failed */
   TAPI_statusCidInfoStartFail,
   /** Invalid parameter */
   TAPI_statusParam,
   /** Invalid ioctl call */
   TAPI_statusInvalidIoctl,
   /** Unknown or unsupported ioctl call */
   TAPI_statusNoIoctl,
   /** Desired action is not supported */
   TAPI_statusNotSupported,
   /** Service is not supported by the low level driver */
   TAPI_statusLLNotSupp,
   /** LL driver init failed */
   TAPI_statusInitLL,
   /** LL driver already registered */
   TAPI_statusInitLLReg,
   /** LL driver registration - incompatible HL/LL version */
   TAPI_StatusInitLLRegVersMismatch,
   /** LL driver with that major number already exists */
   TAPI_statusRegLLExists,
   /** Device structure allocation failed */
   TAPI_statusDevAlloc,
   /** No tone resource found or error in finding it for playing the tone */
   TAPI_statusToneNoRes,
   /** Failed to configure predefined tone table */
   TAPI_statusTonePredefFail,
   /** Tone is already playing */
   TAPI_statusTonePlayAct,
   /** Invalid resource for tone service specified */
   TAPI_statusInvalidToneRes,
   /** Ringing is already active */
   TAPI_statusRingAct,
   /** Ringing cadence not correct */
   TAPI_statusRingCad,
   /** No ring timer available */
   TAPI_statusRingTimer,
   /** Ringing can not be stopped */
   TAPI_statusRingStop,
   /** Ringing can not be configured */
   TAPI_statusRingCfg,
   /** Metering already active */
   TAPI_statusMeterAct,
   /** Line not in active mode */
   TAPI_statusLineNotAct,
   /** Phone in off hook state */
   TAPI_statusPhoneOffHook,
   /** Service called with wrong context fd */
   TAPI_statusCtxErr,
   /** Service not supported on called channel context */
   TAPI_statusInvalidCh,
   /** Copy to of from user space not successful (Linux only) */
   TAPI_statusErrKernCpy,
   /** General initialization failed */
   TAPI_statusInitFail,
   /** Ringing initialization failed */
   TAPI_statusInitRingFail,
   /** Dialing initialization failed */
   TAPI_statusInitDialFail,
   /** Metering initialization failed */
   TAPI_statusInitMeterFail,
   /** CID initialization failed */
   TAPI_statusInitCIDFail,
   /** Tone initialization failed */
   TAPI_statusInitToneFail,
   /** Reference to unconfigured tone code entry  */
   TAPI_statusToneNotAvail,
   /** Not enough spac in CID data buffer */
   TAPI_statusCIDBuffNoMem,
   /** Wrong CID configuration */
   TAPI_statusCIDWrongConf,
   /** Unsuitable line mode for CID Tx */
   TAPI_statusCIDNoLineModeOk,
   /** CID Tx already active */
   TAPI_statusCIDActive,
   /** CID hook state mismatch */
   TAPI_statusCIDHook,
   /** Can not stop CID Tx */
   TAPI_statusCIDStopTx,
   /** Can not start CID Rx */
   TAPI_statusCIDStartRx,
   /** CID Rx stopping failed */
   TAPI_statusCIDStopRx,
   /** CID data not copied */
   TAPI_statusCIDCopy,
   /** Failed to enable COD AGC */
   TAPI_statusCODAGCEn,
   /** PCM mode unknown */
   TAPI_statusPCMUnknownMode,
   /** PCM channel configuration error */
   TAPI_statusPCMChCfgError,
   /** Phone channel not available */
   TAPI_statusPhoneNo,
   /** Data channel not available */
   TAPI_statusDataNo,
   /** Stopping tone failed */
   TAPI_statusToneStop,
   /** RTP payload table configuration failed */
   TAPI_statusRTPConf,
   /** MFTD enable failed */
   TAPI_statusMFTDEnFail,
   /** MFTD disabled failed */
   TAPI_statusMFTDDisFail,
   /** Playing tone in LL driver failed */
   TAPI_statusTonePlayLLFailed,
  /** LL driver returned an error */
   TAPI_statusLLFailed,
   /** Memory not available */
   TAPI_statusNoMem,
   /** FIFO size or FIFO initialization error */
   TAPI_statusFIFO,
   /** Event not known or not handled */
   TAPI_statusEvtNoHandle,
   /** TAPI_DeferWork schedule_work failed */
   TAPI_statusWorkFail,
   /** Timer creation failed */
   TAPI_statusTimerFail,
   /** Ringing was not initialized */
   TAPI_statusRingInit,
   /** Not enough space in buffer for DATE element */
   TAPI_statusCIDBuffNoSpaceForDATE,
   /** Not enough space in buffer for element */
   TAPI_statusCIDBuffNoSpaceForElement,
   /** Not enough space in buffer for STRING element */
   TAPI_statusCIDBuffNoSpaceForSTRING,
   /** Not enough space in buffer for CLI element */
   TAPI_statusCIDBuffNoSpaceForCLI,
   /** Not enough space in buffer for NAME element */
   TAPI_statusCIDBuffNoSpaceForNAME,
   /** No valid data for NTT CID, mandatory elements missing */
   TAPI_statusCIDBuffMandatoryElementMissing,
   /** Wrong size for TRANSPARENT element or
       TRANSPARENT together with other elements types */
   TAPI_statusCIDBuffIncorrectTRANSPARENT,
   /** Configuration of predefined CID alert tone failed */
   TAPI_statusCIDConfAlertToneFailed,
   /** Hookstate not matching with transmission mode */
   TAPI_statusCIDHookMissmatch,
   /** Line mode not suitable for CID sequence transmission */
   TAPI_statusCIDLineModeNotSuitable,
   /** No phone is connected on the data channel main input */
   TAPI_statusCIDNoPhoneAtDataCh,
   /** No data available in CID receiver buffer */
   TAPI_statusCIDRXNoDataAvailable,
   /** Device index exceeds the range of defined devices */
   TAPI_statusDevIdxRange,
   /** Channel structure allocation failed */
   TAPI_statusChAlloc,
   /** IOCTL is blocked until the device is started */
   TAPI_statusIoctlBlocked,
   /** Metering Burst failure */
   TAPI_statusMeterBurstFail,
   /** Unsuitable line mode for ringing */
   TAPI_statusRingLineModeNotOk,
   /** Tone index out of range */
   TAPI_statusToneIdxOutOfRange,
   /** Simple tone on-time time exceeds the maximum range */
   TAPI_statusToneOnTimeOutOfRange,
   /** Simple tone time of first step may not be zero */
   TAPI_statusToneInitialTimeZero,
   /** Simple tone pause time exceeds the maximum supported pause time */
   TAPI_statusTonePauseTimeOutOfRange,
   /** Simple tone frequency exceeds the maximum supported frequency */
   TAPI_statusToneFrequencyOutOfRange,
   /** Simple tone power level exceeds the supported range */
   TAPI_statusTonePowerLevelOutOfRange,
   /** Composed tone max alternate voice time exceeds the maximum */
   TAPI_statusToneAltVoiceTimeOutOfRange,
   /** Composed tone single repetition not allowed when
       alternate voice time is non-zero */
   TAPI_statusToneLoopCountInvalid,
   /** Composed tone number of simple tone fields is out of range */
   TAPI_statusToneCountOutOfRange,
   /** Composed tone contains a simple tone index that is out of range */
   TAPI_statusToneSimpleIdxOutOfRange,
   /** Composed tone contains a simple tone that is not configured */
   TAPI_statusToneSimpleToneUnconfigured,
   /** Invalid character passed in DTMF CID message element */
   TAPI_statusCIDInvalCharInDtmfMsgElem,
   /** Announcement ID already in use */
   TAPI_statusAnnInUse,
   /** Encoder not supported by the firmware */
   TAPI_statusEncoderNotSupported,
   /** Announcement not configured */
   TAPI_statusAnnNotConfigured,
   /** Announcement playout ongoing */
   TAPI_statusAnnActive,
   /* ++ insert new error codes here ++ */
   /** Generic or unknown error occured */
   TAPI_statusErr = IFX_ERROR
}TAPI_Status_t;

#define ERROR_CLASS_MASK           (TAPI_statusClassErr |                  \
                                    TAPI_statusClassWarn |                 \
                                    TAPI_statusClassCritical)

/* Note on both RETURN_STATUS() and RETURN_DEVSTATUS():
   "code" may contain an already combined HL and LL error code. In this case
   the "llcode" field is ignored. In all other cases the error code is the
   combination of the HL and LL error code. The "code" field may also contain
   an old style error code (IFX_ERRROR) which is also treated as an error.
   The error is put on the stack if either "code" or "llcode" contain an
   error code. */

#define RETURN_STATUS(code, llcode)                                        \
   /*lint -save -e{506, 572, 774, 778} */                                  \
   do{                                                                     \
      IFX_uint32_t   HLcode, LLcode;                                       \
      if (((IFX_uint32_t)(code)) & 0xFFFF0000)                             \
      {                                                                    \
         HLcode = (((IFX_uint32_t)(code)) & 0xFFFF0000) >> 16;             \
         LLcode = (((IFX_uint32_t)(code)) & 0x0000FFFF);                   \
      }                                                                    \
      else                                                                 \
      {                                                                    \
         HLcode = (((IFX_uint32_t)(code)) & 0x0000FFFF);                   \
         LLcode = (((IFX_uint32_t)(llcode)) & 0x0000FFFF);                 \
      }                                                                    \
      if (((IFX_return_t)(HLcode) == IFX_ERROR) ||                         \
          ((IFX_return_t)(LLcode) == IFX_ERROR) ||                         \
          ((TAPI_statusClass_t)(HLcode) & ERROR_CLASS_MASK) ||             \
          ((TAPI_statusClass_t)(LLcode) & ERROR_CLASS_MASK)   )            \
      {                                                                    \
         TAPI_ErrorStatus (pChannel->pTapiDevice,                          \
                           (HLcode), (LLcode), __LINE__, __FILE__);        \
      }                                                                    \
      return   (((IFX_uint32_t)(HLcode) & 0x0000FFFF) << 16) |             \
                ((IFX_uint32_t)(LLcode) & 0x0000FFFF);                     \
   }while(0) /*lint -restore */

#define RETURN_DEVSTATUS(code, llcode)                                     \
   /*lint -save -e{506, 572, 774, 778} */                                       \
   do{                                                                     \
      IFX_uint32_t   HLcode, LLcode;                                       \
      if (((IFX_uint32_t)(code)) & 0xFFFF0000)                             \
      {                                                                    \
         HLcode = (((IFX_uint32_t)(code)) & 0xFFFF0000) >> 16;             \
         LLcode = (((IFX_uint32_t)(code)) & 0x0000FFFF);                   \
      }                                                                    \
      else                                                                 \
      {                                                                    \
         HLcode = (((IFX_uint32_t)(code)) & 0x0000FFFF);                   \
         LLcode = (((IFX_uint32_t)(llcode)) & 0x0000FFFF);                 \
      }                                                                    \
      if (((IFX_return_t)(HLcode) == IFX_ERROR) ||                         \
          ((IFX_return_t)(LLcode) == IFX_ERROR) ||                         \
          ((TAPI_statusClass_t)(HLcode) & ERROR_CLASS_MASK) ||             \
          ((TAPI_statusClass_t)(LLcode) & ERROR_CLASS_MASK)   )            \
      {                                                                    \
         TAPI_ErrorStatus (pTapiDev,                                       \
                           (HLcode), (LLcode), __LINE__, __FILE__);        \
      }                                                                    \
      return   (((IFX_uint32_t)(HLcode) & 0x0000FFFF) << 16) |             \
                ((IFX_uint32_t)(LLcode) & 0x0000FFFF);                     \
   }while(0) /*lint -restore */

#define GET_HL_ERROR(code)                                                 \
   (((IFX_uint32_t)(code) >> 16) & 0x0000FFFF)

#define BUILD_HL_ERROR(code)                                               \
   ((IFX_uint32_t)(code) << 16)


/* ============================= */
/* Global variable definition    */
/* ============================= */

/* ============================= */
/* Global function declaration   */
/* ============================= */

extern void TAPI_ErrorStatus (TAPI_DEV *pTapiDevice,
                              TAPI_Status_t nHlCode, IFX_int32_t nLlCode,
                              IFX_uint32_t nLine, const IFX_char_t* sFile);
