/*   FILE: playlist.c -- Playlist support
 * AUTHOR: W. Michael Petullo <mike@flyn.org>
 *   DATE: 12 February 2022
 *
 * Copyright (c) 2009 W. Michael Petullo <new@flyn.org>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <libdmapsharing/dmap.h>
#include <totem-pl-parser.h>
#include <glib.h>

#include "playlist.h"
#include "dmapd-dmap-container-record.h"

typedef struct pl_data_t {
	DmapDb              *db;
	DmapContainerDb     *container_db;
	DmapContainerRecord *container;
} pl_data_t;

static void
_started_cb(G_GNUC_UNUSED TotemPlParser *parser,
                     G_GNUC_UNUSED const char    *uri,
                                   GHashTable    *metadata,
                                   pl_data_t     *data)
{
	const gchar *name = g_hash_table_lookup(metadata, TOTEM_PL_PARSER_FIELD_TITLE);

	g_assert(data->db != NULL);
	g_assert(data->container_db != NULL);
	g_assert(data->container == NULL); /* Gets set below. */

	data->container = DMAP_CONTAINER_RECORD (g_object_new (
		TYPE_DMAPD_DMAP_CONTAINER_RECORD,
		"name",
		name,
		"full-db",
		data->db,
		NULL
	));
}

static void
_entry_parsed_cb(G_GNUC_UNUSED TotemPlParser *parser,
                 G_GNUC_UNUSED const char    *uri,
                               GHashTable    *metadata,
                               pl_data_t     *data)
{
	guint id;
	gchar *url = NULL;

	g_assert(data->db != NULL);
	g_assert(data->container_db != NULL);
	g_assert(data->container != NULL);

	url = g_hash_table_lookup(metadata, TOTEM_PL_PARSER_FIELD_URI);
	if (NULL == url) {
		g_warning("playlist lacks URL");
		goto done;
	}

	id = dmap_db_lookup_id_by_location(data->db, url);
	if (DMAP_DB_ID_BAD == id) {
		g_warning("playlist URL %s not found in database", url);
		goto done;
	}

	dmap_container_record_add_entry(data->container, NULL, id, NULL);

done:
	return;
}

void
playlist_add_playlists(gchar **playlist_dirs, DmapDb *db, DmapContainerDb *container_db)
{
	for (guint i = 0; i < g_strv_length(playlist_dirs); i++) {
		GError *error = NULL;

		pl_data_t data = {
			.db           = db,
			.container_db = container_db,
			.container    = NULL,
		};

		GDir *dir = g_dir_open(playlist_dirs[i], 0, &error);
		if (NULL == dir) {
			g_error("opening playlist %s failed: %s", playlist_dirs[i], error->message);
		}

		for (const gchar *pl = g_dir_read_name(dir); pl != NULL; pl = g_dir_read_name(dir)) {
			TotemPlParser *parser = totem_pl_parser_new();
			TotemPlParserResult rc;
			gchar *url = NULL;

			g_signal_connect(parser, "playlist-started", G_CALLBACK (_started_cb), &data);
			g_signal_connect(parser, "entry-parsed", G_CALLBACK (_entry_parsed_cb), &data);

			url = g_strdup_printf("file://%s/%s", playlist_dirs[i], pl);

			rc = totem_pl_parser_parse(parser, url, FALSE);
			if (rc != TOTEM_PL_PARSER_RESULT_SUCCESS) {
				g_warning("parsing %s failed", url);
			} else {
				if (dmap_container_record_get_entry_count(data.container)) {
					dmap_container_db_add(container_db, data.container, NULL);
				}
			}

			g_free(url);
			g_object_unref(parser);
		}
	}

	return;
}
