/*   FILE: dmapd.c -- A DMAP server
 * AUTHOR: W. Michael Petullo <mike@flyn.org>
 *   DATE: 01 January 2009 
 *
 * Copyright (c) 2009 W. Michael Petullo <new@flyn.org>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <pwd.h>
#include <grp.h>
#include <glib.h>
#include <sys/stat.h>
#include <libdmapsharing/dmap.h>

#include "dmapd-dmap-container-record.h"
#include "dmapd-dmap-container-db.h"
#include "dmapd-dpap-record.h"
#include "dmapd-dpap-record-factory.h"
#include "dmapd-daap-record.h"
#include "dmapd-daap-record-factory.h"
#include "dmapd-module.h"
#include "db-builder.h"
#include "av-meta-reader.h"
#include "av-render.h"
#include "photo-meta-reader.h"
#include "util.h"
#include "util-gst.h"
#include "playlist.h"

#define DEFAULT_CONFIG_FILE            DEFAULT_SYSCONFDIR "/dmapd.conf"
#define DEFAULT_DB_MOD                "ghashtable"
#define DEFAULT_AV_META_READER_MOD    "gst"
#define DEFAULT_AV_RENDER_MOD         "gst"
#define DEFAULT_PHOTO_META_READER_MOD "vips"

typedef enum {
	DAAP,
	DPAP,
	DACP,
	BAD,
} protocol_id_t;

static char *_protocol_map[] = {
	"DAAP",
	"DPAP",
	"DACP",
	 NULL,
};

/* Consolidates these so they may be passed to callback, etc. */
typedef struct workers_t {
	DmapControlShare *dacp_share;
	AvRender         *av_render;
} workers_t;

typedef struct share_def_t {
	protocol_id_t type;
	char         *name;
	char         *password;
	char        **dirs;
	char        **playlist_dirs;
	char        **formats;
	char         *transcode_mimetype;
	gboolean      enable_rt_transcode;
	DbBuilder    *builder;
	DmapShare    *share;
} share_def_t;

static GMainLoop *_loop = NULL;

static GSList  *_share_defs               = NULL;
static gchar  **_music_dirs               = NULL;
static gchar  **_picture_dirs             = NULL;
static gchar  **_playlist_dirs            = NULL;
static gchar  **_music_formats            = NULL;
static gchar  **_picture_formats          = NULL;
static gchar   *_module_dir               = NULL;
static gchar   *_config_file              = DEFAULT_CONFIG_FILE;
static gchar   *_db_dir                   = NULL;
static gchar   *_run_dir                  = NULL;
static gchar   *_lockpath                 = NULL;
static gchar   *_password                 = NULL;
static gchar   *_pidpath                  = NULL;
static gchar   *_user                     = NULL;
static gchar   *_group                    = NULL;
static gchar   *_share_name               = NULL;
static gchar   *_transcode_mimetype       = NULL;
static gchar   *_db_module                = NULL;
static gchar   *_av_meta_reader_module    = NULL;
static gchar   *_av_render_module         = NULL;
static gchar   *_photo_meta_reader_module = NULL;
static guint    _max_thumbnail_width      = 128;
static gboolean _enable_dir_containers    = FALSE;
static gboolean _enable_foreground        = FALSE;
static gboolean _enable_render            = FALSE;
static gboolean _enable_rt_transcode      = FALSE;
static gboolean _enable_version           = FALSE;
static gboolean _calculate_media_hashes   = FALSE;
static gboolean _exit_after_loading       = FALSE;
static gboolean _enable_debug             = FALSE;

// FIXME: make non-global, support mult. remotes and free when done.
// store persistently or set in config file?
static gchar *_guid = NULL;

static void
_free_globals(void)
{
	g_debug ("Free'ing globals");

	g_slist_free_full(_share_defs, g_free);

	g_strfreev(_music_dirs);
	g_strfreev(_picture_dirs);
	g_strfreev(_playlist_dirs);
	g_strfreev(_music_formats);
	g_strfreev(_picture_formats);

	g_free(_db_dir);
	g_free(_run_dir);
	g_free(_lockpath);
	g_free(_password);
	g_free(_pidpath);
	g_free(_user);
	g_free(_group);
	g_free(_share_name);
	g_free(_transcode_mimetype);

	g_main_loop_unref (_loop);
}

static gchar **
_strvappend(gchar **vector, char *str)
{
	gchar **old = vector;
	guint length = vector ? g_strv_length(vector) : 0;
	vector = g_new0(gchar *, length + 2);
	memcpy(vector, old, length * sizeof(gchar *));
	vector[length] = str;
	g_free(old); /* Do not free contained strings. */
	return vector;
}

static gboolean
_add_to_opt_list (const gchar *option_name,
	         const gchar *value,
	         G_GNUC_UNUSED gpointer data,
	         G_GNUC_UNUSED GError **error)
{
	if (strcmp (option_name, "-m") == 0) {
		_music_dirs = _strvappend(_music_dirs, g_strdup(value));
	} else if (strcmp (option_name, "-p") == 0) {
		_picture_dirs = _strvappend(_picture_dirs, g_strdup(value));
	} else if (strcmp (option_name, "-y") == 0) {
		_playlist_dirs = _strvappend(_playlist_dirs, g_strdup(value));
	} else if (strcmp (option_name, "-M") == 0) {
		_music_formats = _strvappend(_music_formats, g_strdup(value));
	} else if (strcmp (option_name, "-P") == 0) {
		_picture_formats = _strvappend(_picture_formats, g_strdup(value));
	} else {
		g_error ("Unsupported option: %s.", option_name);
	}
	return TRUE;
}

/* FIXME: how to enumerate available transcoding formats? */
static GOptionEntry _entries[] = {
	{ "foreground", 'f', 0, G_OPTION_ARG_NONE, &_enable_foreground, "Do not fork; remain in foreground", NULL },
	{ "name", 'n', 0, G_OPTION_ARG_STRING, &_share_name, "Name of media shares", NULL },
	{ "music-dir", 'm', 0, G_OPTION_ARG_CALLBACK, _add_to_opt_list, "Music directory", NULL },
	{ "picture-dir", 'p', 0, G_OPTION_ARG_CALLBACK, _add_to_opt_list, "Picture directory", NULL },
	{ "playlist-dir", 'y', 0, G_OPTION_ARG_CALLBACK, _add_to_opt_list, "Playlist directory", NULL },
	{ "music-format", 'M', 0, G_OPTION_ARG_CALLBACK, _add_to_opt_list, "Acceptable music format", NULL },
	{ "picture-format", 'P', 0, G_OPTION_ARG_CALLBACK, _add_to_opt_list, "Acceptable picture format", NULL },
	{ "lockpath", 'l', 0, G_OPTION_ARG_STRING, &_lockpath, "Path to lockfile", NULL },
	{ "pidpath", 'i', 0, G_OPTION_ARG_STRING, &_pidpath, "Path to PID file", NULL },
	{ "db-dir", 'd', 0, G_OPTION_ARG_STRING, &_db_dir, "Media database directory", NULL },
	{ "run-dir", 'R', 0, G_OPTION_ARG_STRING, &_run_dir, "Runtime directory", NULL },
	{ "user", 'u', 0, G_OPTION_ARG_STRING, &_user, "User to run as", NULL },
	{ "group", 'g', 0, G_OPTION_ARG_STRING, &_group, "Group to run as", NULL },
	{ "render", 'o', 0, G_OPTION_ARG_NONE, &_enable_render, "Render using AirPlay", NULL },
	{ "transcode-mimetype", 't', 0, G_OPTION_ARG_STRING, &_transcode_mimetype, "Target MIME type for transcoding", NULL },
	{ "rt-transcode", 'r', 0, G_OPTION_ARG_NONE, &_enable_rt_transcode, "Perform transcoding in real-time", NULL },
	{ "max-thumbnail-width", 'w', 0, G_OPTION_ARG_INT, &_max_thumbnail_width, "Maximum thumbnail size (may reduce memory use)", NULL },
	{ "directory-containers", 'c', 0, G_OPTION_ARG_NONE, &_enable_dir_containers, "Serve DMAP containers based on filesystem heirarchy", NULL },
	{ "version", 'v', 0, G_OPTION_ARG_NONE, &_enable_version, "Print version number and exit", NULL },
	{ "calculate-media-hashes", 's', 0, G_OPTION_ARG_NONE, &_calculate_media_hashes, "Calculate/check media hashes (slow)", NULL },
	{ "exit-after-loading", 'x', 0, G_OPTION_ARG_NONE, &_exit_after_loading, "Exit after loading database (do not serve)", NULL },
	{ NULL }
};

static char *
_default_share_name (protocol_id_t type)
{
        const gchar *real_name;
	const gchar *type_string;

        real_name = g_get_real_name ();
        if (strcmp (real_name, "Unknown") == 0) {
                real_name = g_get_user_name ();
        }

	switch(type) {
	case DAAP:
		type_string = "Media";
		break;
	case DPAP:
		type_string = "Pictures";
		break;
	default:
		g_error("Unknown share type");
	}

        return g_strdup_printf ("%s's %s", real_name, type_string);
}

static DmapShare *
_create_share (protocol_id_t protocol,
               DmapDb *db,
               DmapContainerDb *container_db,
               share_def_t *share_def)
{
	DmapShare *share = NULL;

	g_debug("Initializing %s share", _protocol_map[protocol]);

	switch(protocol) {
	case DAAP:
		share = DMAP_SHARE(dmap_av_share_new(share_def->name,
		                                     share_def->password,
		                                     db,
		                                     container_db,
		                                     share_def->transcode_mimetype));
		break;
	case DPAP:
		share = DMAP_SHARE(dmap_image_share_new(share_def->name,
		                                        share_def->password,
		                                        db, container_db,
		                                        share_def->transcode_mimetype));
		break;
	default:
		g_error("Unknown share type");
	}

	return share;
}

static void
_drop_root (const char *user, const char *group) {
	if (group) {
		struct group *gr;
		if (!(gr = getgrnam(group))) {	
			g_error("Failed to find group %s", group);
		}

		if (setgid(gr->gr_gid) < 0) {
			g_error("Failed to set GID: %s", strerror(errno));
		}
	}

	if (user) {
		struct passwd *pw;
		if (!(pw = getpwnam(user))) {
			g_error("Failed to find user %s", user);
		}

		if (setuid(pw->pw_uid) < 0) {
			g_error("Failed to set UID: %s", strerror(errno));
		}
	}
}

static void
_daemonize (void)
{
	int child, fd;
	char *pid;
	ssize_t size;
	char *run_dir, *lockpath, *pidpath;

	child = fork ();

	if (child < 0) {
		g_error ("Error forking");
	}
	
	if (child > 0) {
		exit (EXIT_SUCCESS);
	}
	
	if (-1 == setsid ()) {
		g_error("Error executing setsid: %s", strerror(errno));
	}

	run_dir = _run_dir ? _run_dir : DEFAULT_RUNDIR;
	if (-1 == chdir (run_dir)) {
		g_error("Error chdir to %s: %s", run_dir, strerror(errno));
	}

	lockpath = _lockpath ? _lockpath : DEFAULT_LOCKPATH;
	fd = open (lockpath, O_RDWR | O_CREAT, S_IRUSR | S_IWUSR);
	if (-1 == fd) {
		g_error("Error opening %s: %s", lockpath, strerror(errno));
	}

	if (-1 == lockf (fd, F_TLOCK, 0)) {
		g_error("Error locking %s: %s", lockpath, strerror(errno));
	}
	
	pidpath = _pidpath ? _pidpath : DEFAULT_RUNDIR "/dmapd.pid";
	fd = open (pidpath, O_RDWR | O_CREAT, S_IRUSR | S_IWUSR);
	if (-1 == fd) {
		g_error("Error opening %s: %s", pidpath, strerror(errno));
	}

	pid = g_strdup_printf ("%d", getpid ());

	size = write (fd, pid, strlen (pid));
	if (size < 0 || strlen (pid) != (size_t) size) {
		g_error("Error writing PID file %s: %s", pidpath, strerror(errno));
	}

	g_free (pid);

	if (_user != NULL ||  _group != NULL) {
		_drop_root (_user, _group);
	}
}

static const gchar *
_level_to_str(GLogLevelFlags level)
{
	const gchar *str;

	switch (level) {
	case G_LOG_LEVEL_ERROR:
		str = "ERROR";
		break;
	case G_LOG_LEVEL_CRITICAL:
		str = "CRITICAL";
		break;
	case G_LOG_LEVEL_WARNING:
		str = "WARNING";
		break;
	case G_LOG_LEVEL_MESSAGE:
		str = "MESSAGE";
		break;
	case G_LOG_LEVEL_INFO:
		str = "INFO";
		break;
	case G_LOG_LEVEL_DEBUG:
		str = "DEBUG";
		break;
	default:
		str = "UNKNOWN";
		break;
	}

	return str;
}

static void
_log_printf(const char *log_domain,
            GLogLevelFlags level,
            const gchar *message,
            G_GNUC_UNUSED gpointer user_data)
{
	if (level > G_LOG_LEVEL_INFO && !_enable_debug) {
		return;
	}

	g_printerr("%s: %s: %s\n",
		    log_domain,
		    _level_to_str(level & G_LOG_LEVEL_MASK),
		    message);
}

static void
_error_cb(G_GNUC_UNUSED DmapShare *share, GError *error, G_GNUC_UNUSED gpointer user_data)
{
	g_printerr("%s\n", error->message);
}

static DmapShare *
_serve (protocol_id_t protocol,
        DmapRecordFactory *factory,
        share_def_t *share_def,
        GError **error)
{
	DmapDb *db = NULL;
	int rc;
	gboolean ok;
	DmapShare *share = NULL;
	DmapContainerDb *container_db = NULL;

	g_assert (_db_module);

	gchar *db_protocol_dir = g_build_path (G_DIR_SEPARATOR_S,
	                                       _db_dir ? _db_dir : DEFAULT_DBDIR,
	                                       share_def->name,
	                                       _protocol_map[protocol],
	                                       NULL);

	ok = g_file_test (db_protocol_dir, G_FILE_TEST_EXISTS);
	if (!ok) {
		rc = g_mkdir_with_parents (db_protocol_dir, 0700);
		if (-1 == rc) {
			g_set_error (error,
			             DMAP_ERROR,
			             DMAP_STATUS_FAILED,
			             "Failed to create %s: %s",
			             db_protocol_dir,
			             g_strerror (errno));
			goto done;
		}
	}

	db = DMAP_DB (util_object_from_module (TYPE_DMAPD_DMAP_DB, 
	                                  _module_dir,
					  _db_module,
					  "db-dir",
					  db_protocol_dir,
					  "record-factory",
					  factory,
					  NULL));
	g_assert (db);

	if (share_def->formats) {
		g_object_set (db, "acceptable-formats", share_def->formats, NULL);
	}

	container_db = DMAP_CONTAINER_DB (dmapd_dmap_container_db_new ());
	share_def->builder = DB_BUILDER (util_object_from_module (TYPE_DB_BUILDER,
	                                                          _module_dir,
	                                                         "gdir",
	                                                          NULL));

	g_object_set (share_def->builder, "db", db, NULL);

	for (int i = 0; share_def->dirs && share_def->dirs[i]; i++) {
		if (_enable_dir_containers) {
			g_object_set (share_def->builder, "container-db", container_db, NULL);
		}

		ok = db_builder_build_db_starting_at (share_def->builder, share_def->dirs[i], NULL, error);
		if (!ok) {
			goto done;
		}
	}

	if (protocol == DAAP
	 && share_def->transcode_mimetype
	 && ! share_def->enable_rt_transcode) {
		dmap_db_foreach (db,
		                (DmapIdRecordFunc) util_gst_transcode_cache,
		               &(db_dir_and_target_transcode_mimetype_t) {
					db_protocol_dir,
					share_def->transcode_mimetype
		                });
	}

	if (share_def->playlist_dirs) {
		playlist_add_playlists(share_def->playlist_dirs, db, container_db);
	}

	_loop = g_main_loop_new (NULL, FALSE);
	share = _create_share (protocol, db, container_db, share_def);

	g_signal_connect(share, "error", G_CALLBACK(_error_cb), NULL);

	ok = dmap_share_serve(share, error);
	if (!ok) {
		g_object_unref(share);
		share = NULL;
		goto done;
	}

	ok = dmap_share_publish(share, error);
	if (!ok) {
		g_object_unref(share);
		share = NULL;
		goto done;
	}

done:
	if (NULL != container_db) {
		g_object_unref (container_db);
	}

	if (NULL != db) {
		g_object_unref (db);
	}

	g_free (db_protocol_dir);

	return share;
}

static gchar *
_key_file_s_or_default (GKeyFile *f, char *g, char *k, char *def)
{
	gchar *str = g_key_file_get_string (f, g, k, NULL);
	return str ? str : def;
}

static gboolean
_key_file_b_or_default (GKeyFile *f, char *g, char *k, gboolean def)
{
	return g_key_file_get_boolean (f, g, k, NULL) ? g_key_file_get_boolean (f, g, k, NULL) : def;
}

static protocol_id_t
_read_type(GKeyFile *keyfile, char *group)
{
	char *mnemonic;
	protocol_id_t id = -1;

	mnemonic = _key_file_s_or_default(keyfile, group, "Type", "DAAP");

	for (int i = 0; _protocol_map[i] != NULL; i++) {
		if (!strcmp(mnemonic, _protocol_map[i])) {
			id = i;
			break;
		}
	}

	return id;
}

static void
_add_share(protocol_id_t type, char *name, char *password, char **dirs,
          char **playlist_dirs, char **formats, char *transcode_mimetype,
          gboolean enable_rt_transcode)
{
	share_def_t *share_def = g_new0(share_def_t, 1);

	share_def->type                = type;
	share_def->name                = name;
	share_def->password            = password;
	share_def->dirs                = dirs;
	share_def->playlist_dirs       = playlist_dirs;
	share_def->formats             = formats;
	share_def->transcode_mimetype  = transcode_mimetype;
	share_def->enable_rt_transcode = enable_rt_transcode;

	_share_defs = g_slist_append(_share_defs, share_def);
}

static void
_read_keyfile (void)
{
	GError *error = NULL;
	GKeyFile *keyfile;

	keyfile = g_key_file_new ();

	if (!g_key_file_load_from_file (keyfile, _config_file, G_KEY_FILE_NONE, &error)) {
		g_warning("Could not read config %s: %s", _config_file, error->message);
	} else {
		gchar **shares;
		gsize   num_groups;

		_db_dir                = _key_file_s_or_default (keyfile, "General", "Database-Dir", _db_dir);
		_run_dir               = _key_file_s_or_default (keyfile, "General", "Run-Dir", _run_dir);
		_user                  = _key_file_s_or_default (keyfile, "General", "User", _user);
		_group                 = _key_file_s_or_default (keyfile, "General", "Group", _group);
		_enable_dir_containers = _key_file_b_or_default (keyfile, "General", "Dir-Containers", _enable_dir_containers);

		shares                = g_key_file_get_groups (keyfile, &num_groups);

		for (gsize i = 0; i < num_groups; i++) {
			if (!strcmp("General", shares[i])) {
				/* Every group other than "General" is a share. */
				continue;
			}

			_add_share(_read_type(keyfile, shares[i]),
			           g_strdup(shares[i]),
			           _key_file_s_or_default (keyfile, shares[i], "Password", _password),
			           g_key_file_get_string_list (keyfile, shares[i], "Dirs", NULL, NULL),
			           g_key_file_get_string_list (keyfile, shares[i], "PlaylistDirs", NULL, NULL),
			           g_key_file_get_string_list (keyfile, shares[i], "Acceptable-Formats", NULL, NULL),
			           _key_file_s_or_default (keyfile, shares[i], "Transcode-Mimetype", _transcode_mimetype),
			           _key_file_b_or_default (keyfile, shares[i], "Realtime-Transcode", _enable_rt_transcode));
		}

		g_strfreev(shares);
	}

	g_key_file_free (keyfile);
}

static void
_sigterm_handler (G_GNUC_UNUSED int i)
{
	signal (SIGTERM, _sigterm_handler);

	g_debug("Received TERM signal");

	g_main_loop_quit (_loop);
}

static gboolean
_dacp_add_guid_cb (G_GNUC_UNUSED DmapControlShare *share, gchar *guid, G_GNUC_UNUSED gpointer user_data)
{
	// FIXME: handle multiple remotes? See also defn of _guid.
	_guid = g_strdup (guid);
	return TRUE;
}

static gboolean
_dacp_lookup_guid_cb (G_GNUC_UNUSED DmapControlShare *share, gchar *guid, G_GNUC_UNUSED gpointer user_data)
{
	g_debug("Comparing %s to %s", _guid, guid);
	return _guid && ! strcmp (_guid, guid);
}

static void
_dacp_remote_found_cb(DmapControlShare *share,
                      gchar *service_name,
                      G_GNUC_UNUSED gchar *display_name,
                      G_GNUC_UNUSED gpointer user_data)
{
	// FIXME:
	g_print ("Enter passcode: ");
	gchar passcode[5];

	scanf ("%s", passcode);

	dmap_control_share_pair (share, service_name, passcode);
}

static void
_dacp_player_updated_cb(G_GNUC_UNUSED DmapControlPlayer *player, DmapControlShare *share)
{
	dmap_control_share_player_updated (share);
}

static void
_raop_service_added_cb (G_GNUC_UNUSED DmapMdnsBrowser *browser,
                        DmapMdnsService *service,
                        workers_t *workers)
{
	gchar *host = NULL;
	gchar *service_name = NULL;
	gchar *name = NULL;
	gchar *service_host = NULL;
	guint port = 0;
	DmapMdnsServiceTransportProtocol protocol = DMAP_MDNS_SERVICE_TRANSPORT_PROTOCOL_TCP;

	g_object_get (workers->av_render, "host", &host, NULL);
	g_object_get (service, "service-name", &service_name,
	                       "name", &name,
	                       "host", &service_host,
	                       "port", &port,
	                       "transport-protocol", &protocol,
	                        NULL);

	g_debug("RAOP service added %s:%s:%s:%d", service_name, name, service_host, port);

	if (host == NULL) {
		g_warning("RAOP host not set");
	} else if (workers->dacp_share != NULL) {
		g_debug("DACP share already started, not doing it again");	
	} else {
		if (strcmp (service_host, host)) {
			g_debug("Wrong host, %s does not match %s", service_host, host);
		} else {
			DmapDb *db;
			GError *error;
			DmapContainerDb *container_db;

			g_object_set (workers->av_render, "port", port, NULL);
			g_object_set (workers->av_render, "transport-protocol", protocol, NULL);

			// FIXME: set other properties (protocol, generation) from mDNS!
			
			db = NULL;
			container_db = NULL;
			//g_object_get (workers->daap_share, "db", &db, NULL); // FIXME: decompose share we can use db without DAAP.
			//g_object_get (workers->daap_share, "container-db", &container_db, NULL);
			workers->dacp_share = dmap_control_share_new ("FIXME",
			                                      DMAP_CONTROL_PLAYER (workers->av_render),
							      db,
							      container_db);

			g_signal_connect_object (workers->dacp_share, "add-guid", G_CALLBACK (_dacp_add_guid_cb), NULL, 0);
			g_signal_connect_object (workers->dacp_share, "lookup-guid", G_CALLBACK (_dacp_lookup_guid_cb), NULL, 0);
			g_signal_connect_object (workers->dacp_share, "remote-found", G_CALLBACK (_dacp_remote_found_cb), NULL, 0);

			g_signal_connect_object (workers->av_render, "player-updated", G_CALLBACK (_dacp_player_updated_cb), workers->dacp_share, 0);

			dmap_control_share_start_lookup (workers->dacp_share, &error);

			// FIXME: this is to test, remove
			//GList *list = NULL;
			//list = g_list_prepend (list, dmap_db_lookup_by_id (db, G_MAXINT));
			//list = g_list_prepend (list, dmap_db_lookup_by_id (db, G_MAXINT - 1));
			//dacp_player_cue_play(DMAP_CONTROL_PLAYER (workers->av_render), list, 0);
		}
	}
}

int
main (int argc, char *argv[])
{
	int exitval = EXIT_SUCCESS;
	GError *error = NULL;
	GOptionContext *context;
	AvMetaReader *av_meta_reader = NULL;
	PhotoMetaReader *photo_meta_reader = NULL;

	workers_t workers = { NULL, NULL };

	if (getenv ("DMAPD_DEBUG") != NULL) {
		_enable_debug = TRUE;
	}

	g_log_set_handler ("libdmapsharing",
	                    G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION,
	                    _log_printf,
	                    NULL);

	g_log_set_handler ("dmapd",
	                    G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION,
	                    _log_printf,
	                    NULL);

	util_stringleton_init ();

	_loop = g_main_loop_new (NULL, FALSE);
	if (NULL == _loop) {
		g_error ("Could not allocate event loop");
	}

	// These must appear before parsing command-line options because
	// they contribute to which options are available.
	_module_dir = getenv ("DMAPD_MODULEDIR");
	_module_dir = _module_dir ? _module_dir : DEFAULT_MODULEDIR;
	
	_av_meta_reader_module = getenv ("DMAPD_AV_META_READER_MODULE");
	_av_meta_reader_module = _av_meta_reader_module ? _av_meta_reader_module : DEFAULT_AV_META_READER_MOD;

	_av_render_module = getenv ("DMAPD_AV_RENDER_MODULE");
	_av_render_module = _av_render_module ? _av_render_module : DEFAULT_AV_RENDER_MOD;

	_photo_meta_reader_module = getenv ("DMAPD_PHOTO_META_READER_MODULE");
	_photo_meta_reader_module = _photo_meta_reader_module ? _photo_meta_reader_module : DEFAULT_PHOTO_META_READER_MOD;

	_db_module = getenv ("DMAPD_DB_MODULE");
	_db_module = _db_module ? _db_module : DEFAULT_DB_MOD;

	// This must be before _read_keyfile ().
	_config_file = getenv ("DMAPD_CONFIG_FILE");
	_config_file = _config_file ? _config_file : DEFAULT_CONFIG_FILE;

	context = g_option_context_new ("-m | -p: serve media using DMAP");
	g_option_context_add_main_entries (context, _entries, NULL);

	if (strcmp (_av_meta_reader_module, "null") != 0) {
		GOptionGroup *group;
		av_meta_reader = AV_META_READER (util_object_from_module (TYPE_AV_META_READER,
		                                                     _module_dir,
								     _av_meta_reader_module,
								     NULL));
		if (av_meta_reader) {
			group = av_meta_reader_get_option_group (av_meta_reader);
			if (group) {
				g_option_context_add_group (context, group);
			}
		}
	}

	if (strcmp (_av_render_module, "null") != 0) {
		GOptionGroup *group;
		GHashTable *options = g_hash_table_new (g_str_hash, g_str_equal);
		gchar *mod = util_parse_plugin_option (_av_render_module, options);
		workers.av_render = AV_RENDER (util_object_from_module (TYPE_AV_RENDER,
		                                                   _module_dir,
								   mod,
								   NULL));
		g_object_set (workers.av_render, "host", g_hash_table_lookup (options, "host"), NULL);
		if (workers.av_render) {
			group = av_render_get_option_group (workers.av_render);
			if (group) {
				g_option_context_add_group (context, group);
			}
		}
		g_hash_table_destroy (options);
	}

	if (strcmp (_photo_meta_reader_module, "null") != 0) {
		GOptionGroup *group;
		photo_meta_reader = PHOTO_META_READER (util_object_from_module (TYPE_PHOTO_META_READER,
									   _module_dir,
		                                                           _photo_meta_reader_module,
									   NULL));
		if (photo_meta_reader) {
			group = photo_meta_reader_get_option_group (photo_meta_reader);
			if (group) {
				g_option_context_add_group (context, group);
			}
		}
	}

	_read_keyfile ();

	if (! g_option_context_parse (context, &argc, &argv, &error)) {
		g_error ("Option parsing failed: %s", error->message);
	}

	if (NULL != _music_dirs) {
		if (NULL == _share_name) {
			_share_name = _default_share_name(DAAP);
		}

		_add_share(DAAP, _share_name, _password, _music_dirs,
		           _playlist_dirs, _music_formats, _transcode_mimetype,
		           _enable_rt_transcode);
	}

	if (NULL != _picture_dirs) {
		if (NULL == _share_name) {
			_share_name = _default_share_name(DPAP);
		}

		_add_share(DPAP, _share_name, _password, _picture_dirs,
		           NULL, _picture_formats, NULL, FALSE);
	}

	if (_enable_version) {
		g_print ("dmapd version %s\n", VERSION);
		exit (EXIT_SUCCESS);
	}

	g_object_set (photo_meta_reader, "max-thumbnail-width", _max_thumbnail_width, NULL);

	if (! (av_meta_reader || photo_meta_reader)) {
		g_error ("Neither an AV or photograph metadata reader plugin could be loaded");
	}

	if (! av_meta_reader) {
		if (_music_dirs) {
			g_error("Could not load any AV metadata reader plugin but music directory provided");
		} else {
			g_warning("Could not load any AV metadata reader plugin");
		}
	}

	if (! photo_meta_reader) {
		if (_picture_dirs) {
			g_error("Could not load any photograph metadata reader plugin but photograph directory provided");
		} else {
			g_warning("Could not load any photograph metadata reader plugin");
		}
	}

	g_option_context_free (context);

	if (! _enable_foreground) {
		_daemonize ();
	}

	signal (SIGTERM, _sigterm_handler);

	for (GSList *list = _share_defs; list != NULL; list = list->next) {
		DmapRecordFactory *factory;
		share_def_t *share_def = list->data;

		switch(share_def->type) {
		case DAAP:
#if WITH_DAAP
			if (av_meta_reader == NULL) {
				g_error ("Music directory specified but AV metadata reader module is 'null'");
			}
			factory = DMAP_RECORD_FACTORY (
					g_object_new (
						TYPE_DMAPD_DMAP_RECORD_FACTORY,
						"meta-reader",
						av_meta_reader,
						NULL));
			share_def->share = _serve(DAAP, factory, share_def, &error);
			if (NULL == share_def->share) {
				g_error("Error serving: %s", error->message);
			}
#else
		g_error ("DAAP support not present");
#endif
			break;
		case DPAP:
#ifdef WITH_DPAP
			if (photo_meta_reader == NULL) {
				g_error ("Photo directory specified but photo metadata reader module is 'null'");
			}
			factory = DMAP_RECORD_FACTORY (
					g_object_new (
						TYPE_DMAPD_DPAP_RECORD_FACTORY,
						"meta-reader",
						photo_meta_reader,
						NULL));
			share_def->share = _serve(DPAP, factory, share_def, &error);
			if (NULL == share_def->share) {
				g_error("Error serving: %s", error->message);
			}
#else
			g_error ("DPAP support not present");
#endif
			break;
		default:
			g_error ("Bad type configured");
		}
	}

	if (_enable_render && workers.av_render) {
#ifdef WITH_DACP
		GError *error = NULL;
		DmapMdnsBrowser *browser = dmap_mdns_browser_new (DMAP_MDNS_SERVICE_TYPE_RAOP);
		if (browser == NULL) {
			g_error ("Error creating mDNS browser");
		}
		g_signal_connect (G_OBJECT (browser), "service-added", G_CALLBACK (_raop_service_added_cb), &workers);
		dmap_mdns_browser_start (browser, &error);
		if (error) {
		        g_error ("error starting browser. code: %d message: %s", error->code, error->message);
		}
#else
		g_error ("DACP support not present");
#endif
	}

	if (! _exit_after_loading) {
		g_main_loop_run (_loop);
	}

	if (NULL != workers.dacp_share) {
		g_object_unref (workers.dacp_share);
	}

	if (NULL != av_meta_reader) {
		g_object_unref (av_meta_reader);
	}

	if (NULL != workers.av_render) {
		g_object_unref (workers.av_render);
	}

	if (NULL != photo_meta_reader) {
		g_object_unref (photo_meta_reader);
	}

	_free_globals ();
	util_stringleton_deinit ();
	g_debug("Parent exiting");

	exit(exitval);
}
