/*   FILE: dmapd-stress-test.c -- A stress test client for dmapd
 * AUTHOR: W. Michael Petullo <mike@flyn.org>
 *   DATE: 03 February 2011 
 *
 * Copyright (c) 2011 W. Michael Petullo <new@flyn.org>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <glib.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libdmapsharing/dmap.h>

#include "util.h"
#include "dmapd-dmap-db-ghashtable.h"
#include "dmapd-daap-record-factory.h"
#include "dmapd-dpap-record-factory.h"

enum {
	DAAP,
	DPAP
};

static GMainLoop *_loop;

static gchar *_service_name    = NULL;
static guint  _iteration_count = ~0;

static GOptionEntry _entries[] = {
        { "service-name", 's', 0, G_OPTION_ARG_STRING, &_service_name, "Server name to connect to; default is to query user interactively", NULL },
        { "iteration-count", 'i', 0, G_OPTION_ARG_INT, &_iteration_count, "Number of times to hit server; default is forever", NULL },
	{ NULL }
};

static void
_sigterm_handler (int i)
{
	signal (SIGTERM, _sigterm_handler);
	g_debug("Received TERM signal");
	g_main_loop_quit (_loop);
}

static void
_create_connection (const DmapMdnsServiceType protocol,
                    const char *name,
                    const char *host,
                    const guint port);

static void
_process_record (gpointer id, DmapRecord *record, gpointer user_data)
{
	guint status;
	SoupSession *soup_session;
	SoupMessage *soup_message;
	gchar *location = NULL, *desc1 = NULL, *desc2 = NULL;

	if (DMAP_IS_AV_RECORD (record)) {
		/* FIXME: print all record properties; need DAAP and DPAP version: */
		g_object_get (record,
			     "location",   &location,
			     "songartist", &desc1,
			     "title",      &desc2,
			     NULL);

		g_assert (location);
		g_assert (desc1);
		g_assert (desc2);

		g_print ("%d: %s %s %s\n", GPOINTER_TO_UINT (id), desc1, desc2, location);
	} else if (DMAP_IS_IMAGE_RECORD (record)) {
		/* FIXME: print all record properties; need DAAP and DPAP version: */
		g_object_get (record,
			     "location", &location,
			     "filename", &desc1,
			     "format",   &desc2,
			     NULL);

		g_assert (location);
		g_assert (desc1);
		g_assert (desc2);

		g_print ("%d: %s %s %s\n", GPOINTER_TO_UINT (id), desc1, desc2, location);
	} else {
		g_error ("Bad record");
	}

	g_print ("Fetching...");

	soup_session = soup_session_new ();
	g_assert (soup_session);

	soup_message = soup_message_new ("GET", location);
	g_assert (soup_message);

	status = soup_session_send_message (soup_session, soup_message);
	g_assert (SOUP_STATUS_OK == status);

	g_print ("done; length was %zd bytes.\n", soup_message->response_body->length);

	g_free (location);
	g_free (desc1);
	g_free (desc2);
	g_object_unref (soup_session);
	g_object_unref (soup_message);
}

static DmapMdnsServiceType
_get_service_type (DmapConnection *connection)
{
	DmapMdnsServiceType type;

	if (DMAP_IS_AV_CONNECTION (connection)) {
		type = DMAP_MDNS_SERVICE_TYPE_DAAP;
	} else if (DMAP_IS_IMAGE_CONNECTION (connection)) {
		type = DMAP_MDNS_SERVICE_TYPE_DPAP;
	} else {
		g_error ("Bad connection");
	}

	return type;
}

static void
_connected_cb (DmapConnection *connection,
               gboolean        result,
               const char     *reason,
               DmapDb         *db)
{
	guint port;
	char *name, *host;
	static guint count = 1; // This counts up, _iteration_count goes down.

	g_print ("Connection cb., DB has %lu entries\n", dmap_db_count (db));
	g_print ("Iteration %d...\n", count++);

	dmap_db_foreach (db, (DmapIdRecordFunc) _process_record, NULL);

	g_object_get (connection,
		     "name", &name,
		     "host", &host,
		     "port", &port,
		      NULL);

	/* Tear down connection. */
	g_object_unref (connection);

	if (--_iteration_count > 0) {
		/* Create another connection to same service. */
		_create_connection (_get_service_type (connection),
				   name,
				   host,
				   port);
	} else {
		g_main_loop_quit (_loop);
	}

	g_free (name);
	g_free (host);
}

static void
_create_connection (const DmapMdnsServiceType protocol,
                    const char *name,
                    const char *host,
                    const guint port)
{
	DmapRecordFactory *factory;
	DmapConnection *conn;
	DmapDb *db;

	db = DMAP_DB (g_object_new (TYPE_DMAPD_DMAP_DB_GHASHTABLE, NULL));
	if (db == NULL) {
		g_error ("Error creating DB");
	}

	/* FIXME: switch on DMAP_MDNS_SERVICE_TYPE_DAAP, _DPAP, _DACP or _INVALID */
	if (protocol == DMAP_MDNS_SERVICE_TYPE_DAAP) {
		factory = DMAP_RECORD_FACTORY (g_object_new (TYPE_DMAPD_DMAP_RECORD_FACTORY, NULL));
		if (factory == NULL) {
			g_error ("Error creating record factory");
		}
		conn = DMAP_CONNECTION (dmap_av_connection_new (name, host, port, db, factory));
	} else {
		factory = DMAP_RECORD_FACTORY (g_object_new (TYPE_DMAPD_DPAP_RECORD_FACTORY, NULL));
		if (factory == NULL) {
			g_error ("Error creating record factory");
		}
		conn = DMAP_CONNECTION (dmap_image_connection_new (name, host, port, db, factory));
	}
	dmap_connection_start (DMAP_CONNECTION (conn), (DmapConnectionFunc) _connected_cb, db);
}

static void
_service_added_cb (DmapMdnsBrowser *browser,
                   DmapMdnsService *service,
                   gpointer user_data)
{
	char answer, newline;

	gchar *service_name = NULL, *name = NULL, *host = NULL;
	guint port = 0;
	gboolean password_protected = FALSE;

	g_object_get(service, "service-name", &service_name,
	                      "name", &name,
	                      "host", &host,
	                      "port", &port,
	                      "password-protected", &password_protected,
	                       NULL);

	if (NULL == _service_name) {
		fprintf (stdout,
		    "service added %s:%s:%s:%d (%s)\n",
		     service_name,
		     name,
		     host,
		     port,
		     password_protected ? "protected" : "not protected");
		fprintf (stdout, "Stress test this service [Y|N]? ");
		fscanf (stdin, "%c%c", &answer, &newline);
	} else {
		answer = strcmp (_service_name, service_name) ? 'N' : 'Y';
	}

	if (answer == 'Y') {
		_create_connection (dmap_mdns_browser_get_service_type (browser),
				   name,
				   host,
				   port);
	}
}

static void
_debug_null (const char *log_domain,
             GLogLevelFlags log_level,
             const gchar *message,
             gpointer user_data)
{
}

int
main(int argc, char **argv)
{
	GOptionContext *context;
	DmapMdnsBrowser *browserDAAP, *browserDPAP;
	GError *error = NULL;

	util_stringleton_init ();

	g_log_set_handler ("libdmapsharing", G_LOG_LEVEL_DEBUG, _debug_null, NULL);
	g_log_set_handler (NULL, G_LOG_LEVEL_DEBUG, _debug_null, NULL);

	context = g_option_context_new ("Stress test dmapd");
        g_option_context_add_main_entries (context, _entries, NULL);

	if (! g_option_context_parse (context, &argc, &argv, &error)) {
                g_error ("Option parsing failed: %s", error->message);
        }

	_loop = g_main_loop_new (NULL, FALSE);
	signal (SIGTERM, _sigterm_handler);

	browserDAAP = dmap_mdns_browser_new (DMAP_MDNS_SERVICE_TYPE_DAAP);

	g_signal_connect (G_OBJECT (browserDAAP),
		         "service-added",
		          G_CALLBACK (_service_added_cb),
		          NULL);

	browserDPAP = dmap_mdns_browser_new (DMAP_MDNS_SERVICE_TYPE_DPAP);

	g_signal_connect (G_OBJECT (browserDPAP),
		         "service-added",
		          G_CALLBACK (_service_added_cb),
		          NULL);

	g_debug ("starting mdns browsers");

	dmap_mdns_browser_start (browserDAAP, &error);
	if (error) {
		g_warning ("error starting browser. code: %d message: %s",
			    error->code,
			    error->message);
		exit (EXIT_FAILURE);
	}

	dmap_mdns_browser_start (browserDPAP, &error);
	if (error) {
		g_warning ("error starting browser. code: %d message: %s",
			    error->code,
			    error->message);
		exit (EXIT_FAILURE);
	}

	fprintf (stdout, "Waiting for DMAP shares; please run dmapd.\n");
	g_main_loop_run (_loop);

	fprintf (stdout, "Stress test complete.\n");

	exit (EXIT_SUCCESS);
}
