"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.crawlDirWithChecks = exports.crawlDirFast = exports.removeEmptyDirsUp = exports.readDirectory = exports.forEachDirentAsync = exports.fileExists = void 0;
const fs_1 = require("fs");
const path_1 = __importDefault(require("path"));
function hasErrorCode(error, code) {
    return (error === null || error === void 0 ? void 0 : error.code) === code;
}
async function fileExists(filePath) {
    try {
        await fs_1.promises.stat(filePath);
        return true;
    }
    catch (error) {
        if (hasErrorCode(error, 'ENOENT')) {
            return false;
        }
        throw error;
    }
}
exports.fileExists = fileExists;
async function forEachDirentAsync(dirPath, action) {
    let dirFiles = [];
    try {
        dirFiles = await fs_1.promises.readdir(dirPath, { withFileTypes: true });
    }
    catch (error) {
    }
    await Promise.all(dirFiles.map(action));
}
exports.forEachDirentAsync = forEachDirentAsync;
async function readDirectory(dirPath) {
    try {
        const files = await fs_1.promises.readdir(dirPath);
        return files;
    }
    catch (error) {
        if (hasErrorCode(error, 'ENOENT') || hasErrorCode(error, 'ENOTDIR')) {
            return [];
        }
        throw error;
    }
}
exports.readDirectory = readDirectory;
async function removeEmptyDirsUp(checkedDirs, dirPath, count = 0) {
    if (!checkedDirs.has(dirPath)) {
        const files = await readDirectory(dirPath);
        const emptyDir = files.length === 0;
        checkedDirs.add(dirPath);
        if (emptyDir) {
            try {
                await fs_1.promises.rmdir(dirPath);
                count++;
            }
            catch (error) {
            }
            const parentDir = path_1.default.dirname(dirPath);
            count = await removeEmptyDirsUp(checkedDirs, parentDir, count);
        }
    }
    return count;
}
exports.removeEmptyDirsUp = removeEmptyDirsUp;
async function crawlDirFast(filePaths, dirPath) {
    await forEachDirentAsync(dirPath, async (dirent) => {
        const nextPath = dirPath + path_1.default.sep + dirent.name;
        if (dirent.isDirectory()) {
            await crawlDirFast(filePaths, nextPath);
        }
        else {
            filePaths.push(nextPath);
        }
    });
}
exports.crawlDirFast = crawlDirFast;
async function crawlDirWithChecks(filePaths, dirPath, checkDir, checkFile) {
    await forEachDirentAsync(dirPath, async (nextPathDirent) => {
        const nextPath = dirPath + path_1.default.sep + nextPathDirent.name;
        if (nextPathDirent.isDirectory()) {
            if (checkDir(nextPath)) {
                await crawlDirFast(filePaths, nextPath);
            }
            else {
                await crawlDirWithChecks(filePaths, nextPath, checkDir, checkFile);
            }
        }
        else if (checkFile(nextPath)) {
            filePaths.push(nextPath);
        }
    });
    return filePaths;
}
exports.crawlDirWithChecks = crawlDirWithChecks;
