"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.cleanCommand = void 0;
const pretty_bytes_1 = __importDefault(require("pretty-bytes"));
const pretty_ms_1 = __importDefault(require("pretty-ms"));
const __1 = require("../..");
const glob_1 = require("../../utils/glob");
const terminal_1 = require("../../utils/terminal");
async function cleanCommand({ argGlobs, nodeModulesPath, userGlobsFilePath, useDefaultGlobs, keepEmpty, dryRun, json, silent, yes, }) {
    const logger = (0, terminal_1.makeLogger)({ disabled: json || silent });
    if (!json) {
        logger.log((0, terminal_1.bold)('clean-modules'), dryRun ? (0, terminal_1.yellow)('(dry run)') : '');
    }
    if (!yes && !dryRun) {
        const confirmed = await (0, terminal_1.yesOrNo)((0, terminal_1.yellow)(`\nAre you sure you want to clean ${nodeModulesPath}? Files will be permanently removed. (Y/N)\n`));
        if (!confirmed) {
            process.exit(0);
        }
    }
    logger.log('\nCleaning up node_modules...');
    const cleanupStart = new Date().getTime();
    const globLists = await (0, glob_1.getGlobLists)({
        argGlobs,
        useDefaultGlobs,
        userGlobsFilePath,
    });
    const includedFiles = await (0, __1.findFilesToRemove)(nodeModulesPath, globLists);
    const reducedSize = await (0, __1.removeFiles)(includedFiles, { dryRun });
    let removedEmptyDirs = 0;
    const cleanupDuration = new Date().getTime() - cleanupStart;
    logger.log((0, terminal_1.green)(`Done in ${(0, pretty_ms_1.default)(cleanupDuration)}!`));
    if (!keepEmpty) {
        logger.log('\nCleaning up empty dirs...');
        if (dryRun) {
            logger.log((0, terminal_1.yellow)('Skipped on dry runs!'));
        }
        else {
            const emptyDirStart = new Date().getTime();
            removedEmptyDirs = await (0, __1.removeEmptyDirs)(includedFiles);
            const emptyDirDuration = new Date().getTime() - emptyDirStart;
            logger.log((0, terminal_1.green)(`Done in ${(0, pretty_ms_1.default)(emptyDirDuration)}!`));
        }
    }
    if (json) {
        const output = {
            removedFiles: includedFiles.length,
            reducedSize,
            removedEmptyDirs,
            duration: cleanupDuration,
            dryRun,
        };
        console.log(JSON.stringify(output, null, 2));
    }
    else {
        logger.log((0, terminal_1.bold)('\nResults:'));
        logger.log('- size reduced:', (0, terminal_1.green)((0, pretty_bytes_1.default)(reducedSize || 0)));
        logger.log('- files removed:', (0, terminal_1.green)(includedFiles.length));
        logger.log('- empty dirs removed:', (0, terminal_1.green)(removedEmptyDirs || 0));
    }
}
exports.cleanCommand = cleanCommand;
