#!/bin/sh
# Permission is hereby granted, free of charge, to any person obtaining a copy of this software
# and associated documentation files (the "Software"), to deal in the Software without restriction,
# including without limitation the rights to use, copy, modify, merge, publish, distribute,
# sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all copies or
# substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
# BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
# DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

## Go to https://github.com/cjdelisle/node.sh for updates to this script.

DEFAULT_MINVER="v4.6.2"
DEFAULT_DLVER="v6.9.1"

die () { printf '%s\n' "ERROR: $1" >&2; exit 100; }

[ "$UNAME_PATH" != "" ] || UNAME_PATH=$(command -v uname) || die "missing uname";
[ "$SED_PATH" != "" ] || SED_PATH=$(command -v sed) || die "missing sed";
[ "$TR_PATH" != "" ] || TR_PATH=$(command -v tr) || die "missing tr";
[ "$INSTALL_PATH" != "" ] || INSTALL_PATH=$(command -v install) || die "missing install (utility)";
[ "$RM_PATH" != "" ] || RM_PATH=$(command -v rm) || die "missing rm";
[ "$TAR_PATH" != "" ] || TAR_PATH=$(command -v tar) || die "missing tar";
[ "$GREP_PATH" != "" ] || GREP_PATH=$(command -v grep) || die "missing grep";


[ "$PLATFORM" != "" ] || PLATFORM=$("$UNAME_PATH" | "$TR_PATH" '[:upper:]' '[:lower:]')
[ "$MARCH" != "" ] || MARCH=$("$UNAME_PATH" -m | "$SED_PATH" 's/i./x/g')
DEFAULT_NODEDIR="node_$PLATFORM"

usage () {
    printf "node.sh Run a particular script using nodejs which is downloaded if needed\n"
    printf "Usage: MAINJS='./main.js' ./node.sh\n\n"
    printf "Arguments will be passed on to the nodejs script like this:\n"
    printf "MAINJS='./script_which_takes_one_argument.js' ./node.sh arg1\n\n"
    printf "All parameters are passed to the script via env variables\n"

    printf "\nRequired parameters:\n"
    printf "MAINJS      The path to the javascript file to execute using nodejs\n"

    printf "\nOptional parameters:\n"
    printf "MINVER      Minimum nodejs version to accept if on system (default: $DEFAULT_MINVER)\n"
    printf "DLVER       Version of nodejs to download if no good version is present (default: $DEFAULT_DLVER)\n"
    printf "NODEDIR     The directory to use for installing nodejs if needed (default: $DEFAULT_NODEDIR, created if nonexistant)\n"

    printf "\nOverrides: (for overriding default detection)"
    printf "PLATFORM    The name of the system in lowercase (e.g. linux darwin freebsd)\n"
    printf "MARCH       The architecture of the processor (e.g. x86 x86_64 armv7l)\n"
    printf "SED_PATH    The path to a working sed implementation\n"
    printf "TR_PATH     The path to a working tr implementation\n"
    printf "UNAME_PATH  The path to a working uname implementation\n"
    exit 0;
}

[ "$MAINJS" != "" ] || usage;
[ "$NODEDIR" != "" ] || NODEDIR="$DEFAULT_NODEDIR";
[ "$MINVER" != "" ] || MINVER="$DEFAULT_MINVER";
[ "$DLVER" != "" ] || DLVER="$DEFAULT_DLVER";

version_test='
var currentVersion = process.version;
var verArray = currentVersion.substring(1).split(".");
var minVerArray = process.argv[process.argv.length-1].replace(/[^0-9\.]/g, "").split(".");
for (var i = 0; i < minVerArray.length; i++) {
    if (Number(verArray[i]) < Number(minVerArray[i])) {
        process.exit(1);
    } else if (Number(verArray[i]) > Number(minVerArray[i])) {
        process.exit(0);
    }
}
'

# return true if the input command exists in $PATH
cmdExists() {
    command -v $1 >/dev/null 2>/dev/null;
    return $?;
}

checkNode() {
    for node_tool in "$NODEDIR/nodejs/node/bin/node" 'nodejs' 'node'; do
        cmdExists "$node_tool"
        if [ $? = 0 -o -f "$node_tool" ]; then
            printf "$version_test" | "$node_tool" '' "$MINVER" && {
                node_cmd="$node_tool"
                return 0
            }
            printf '%s\n' "You have a version of node [$node_tool] but it is too old [$($node_tool -v)]"
        fi
    done
    return 1
}

getSha() {
    expected_sum='01ba4719c80b6fe911b091a7c05124b64eeece964e09c058ef8f9805daca546b'
    for shasum_tool in 'sha256sum' 'gsha256sum' 'sha256' 'shasum -a 256' 'openssl sha256'; do
        cmdExists "$shasum_tool" || continue;
        printf '\n' | $shasum_tool | "$GREP_PATH" -q "$expected_sum" && {
            shasum_cmd="$shasum_tool"
            return 0
        }
    done
    return 1
}

getNode() {
    printf '%s %s\n\n' '###' "Installing node.js (you can bypass this step by manually installing node.js $MINVER or newer)"
    local march="$MARCH";
    [ "$march" = "x86_64" ] && march="x64";
    [ "$march" = "armv6l" ] && march="arm-pi";
    [ "$march" = "armv7l" ] && march="arm-pi";
    NODEURL="http://nodejs.org/dist/$DLVER/node-$DLVER-$PLATFORM-$march.tar.gz";
    [ "$NODEHASH" = "" ] && getNodeHash;
    [ "$NODEHASH" = "" ] && printf '%s\n%s\n' \
        "No nodejs executable available for $PLATFORM-$MARCH" \
        "Please install nodejs (>= $MINVER) from your distribution package repository or source" \
        && die "";

    [ -d "$NODEDIR/nodejs" ] && "$RM_PATH" -r "$NODEDIR/nodejs"
    "$INSTALL_PATH" -d "$NODEDIR/nodejs"

    cd "$NODEDIR/nodejs"
    node_dl="node-${DLVER}.tar.gz"
    if cmdExists wget; then
        printf '\n%s %s ' '==>' "Downloading $NODEURL with wget..."
        wget -q "$NODEURL" -O "$node_dl" || die "failed to download $NODEURL";
    elif cmdExists curl; then
        printf '%s %s ' '==>' "Downloading $NODEURL with curl..."
        curl -s "$NODEURL" > "$node_dl" || die "failed to download $NODEURL";
    elif cmdExists fetch; then
        printf '%s %s ' '==>' "Downloading $NODEURL with fetch..."
        fetch "$NODEURL" -o "$node_dl" || die "failed to download $NODEURL";
    else
        die 'wget, curl or fetch is required download node.js but you have none!'
    fi
    [ -f "$node_dl" ] || die 'Failed to download node.js'
    printf '%s\n' 'DONE!'

    printf '%s %s ' '==>' "Verifying the checksum of the downloaded archive..."
    $shasum_cmd "$node_dl" | "$GREP_PATH" -q "$NODEHASH" \
        || die 'The downloaded file is damaged! Aborting'
    printf '%s\n' 'DONE!'

    printf '%s %s ' '==>' "Extracting the downloaded archive..."
    "$INSTALL_PATH" -d node
    "$TAR_PATH" xzf "$node_dl" -C node --strip-components=1
    [ -d 'node' ] || die 'An error prevented the archive from being extracted'
    printf '%s\n\n' 'DONE!'
    cd ../../

    # Return with the success status of the checkNode function
    checkNode
}

main() {
    [ -d "$NODEDIR" ] || "$INSTALL_PATH" -d "$NODEDIR" || die "failed to create node dir $NODEDIR"
    getSha || die "couldn't find working sha256 implementation";
    checkNode || getNode || die "couldn't get working node.js implementation";
    "$node_cmd" $MAINJS "$@"
}

getNodeHash() {
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.100/node-v0.1.100.tar.gz" ] && NODEHASH="623d9157017ca3805cbbca653724f8e25a52be689f821d0c608f94717342e1e2"
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.101/node-v0.1.101.tar.gz" ] && NODEHASH="44b08c5c9bd0c23d79d447bc67e1767ec1350a02cec0da6e5ce4c7f790b4e773"
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.102/node-v0.1.102.tar.gz" ] && NODEHASH="bd9b1d09ad40ceaef4bdd46019960c5c2fe87026c9598a6fb23c66457510a22d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.103/node-v0.1.103.tar.gz" ] && NODEHASH="7482b898a0f9514c74137b490c3ad0810ee5ce1586e8886c5182f6446e56711e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.104/node-v0.1.104.tar.gz" ] && NODEHASH="a1c776f44bc07305dc0e56df17cc3260eaafa0394c3b06c27448ad85bec272df"
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.14/node-v0.1.14.tar.gz" ] && NODEHASH="ff74848a9d444923e378e45ec46d6d0867889e78a045c43fa33080fe47c85004"
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.15/node-v0.1.15.tar.gz" ] && NODEHASH="aaf90fb73ba9246adce7ea066ecbf954cb8fa1868246b73871adbe1dd3fcc8e0"
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.16/node-v0.1.16.tar.gz" ] && NODEHASH="3e2088120bbd45881778d105ec53241bd15b0a90f4061d0e3bec2d7faa2955c2"
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.17/node-v0.1.17.tar.gz" ] && NODEHASH="86765206da369900663976b215ff6aa38a6b7518474f60c382b3926a0186528a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.18/node-v0.1.18.tar.gz" ] && NODEHASH="08b2db2dc4f0bb97fc51d79675803e1e4756209531b3c4a9b0f9850802155587"
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.19/node-v0.1.19.tar.gz" ] && NODEHASH="5eff91296a705f72bed6dd493e20af49fd56987e16928f45005605bc6736480a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.20/node-v0.1.20.tar.gz" ] && NODEHASH="4f6684d831554508b2ec4ab3ebe3c417a5f55174a3a00cb5ad67b90d473942fc"
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.21/node-v0.1.21.tar.gz" ] && NODEHASH="f7a7cc74c7b3eff74056bfa4710ae0bde65bf2d86ed9f9fdbc0406e14c0a5100"
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.22/node-v0.1.22.tar.gz" ] && NODEHASH="7884d315b318cdd4e45abe76d42cc44aabb792cce1d2974ff7b4669561fb28ba"
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.23/node-v0.1.23.tar.gz" ] && NODEHASH="69b416d60f288389d25d04797a030b4db6a1ca0cafbd545ca2b8824afaba3035"
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.24/node-v0.1.24.tar.gz" ] && NODEHASH="a1bb4bda5c43142485ea4b0f725bae6319132daf90a243635f626f2d191374f4"
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.25/node-v0.1.25.tar.gz" ] && NODEHASH="48cb603bd55182c22a0030db78be146e137bcf396ff78dc1b7fe748adc9d59a6"
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.26/node-v0.1.26.tar.gz" ] && NODEHASH="8c6e5858bcf1e401af50a7f71d6cbffb3af2f4e41813c46a138edb489bbd84f0"
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.27/node-v0.1.27.tar.gz" ] && NODEHASH="1f83401b9ede7558350e183fc4386234a803d1253e7fe99bce0aba7138270806"
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.28/node-v0.1.28.tar.gz" ] && NODEHASH="06f8a8d5246d34629b1355497c2f0f9f779c7dd9830e3ec51c67cbdae46c9069"
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.29/node-v0.1.29.tar.gz" ] && NODEHASH="b56ddee6fbd89f674df7d38295563f4124e728dbf2dc4e7fb359fd1f6fe9bb06"
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.30/node-v0.1.30.tar.gz" ] && NODEHASH="37ee475b43609b9052300472ea33cf027f6777975bc90611837e54ecdad3f22c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.31/node-v0.1.31.tar.gz" ] && NODEHASH="260972bedf789580b56433ca27a1e3717b131a531a4b3688980a16e65d79abbd"
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.32/node-v0.1.32.tar.gz" ] && NODEHASH="1fda75df1f2c15df3d24c59b8cb3f0c649b6476a59354b9ddad25287cb57f73e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.33/node-v0.1.33.tar.gz" ] && NODEHASH="4a5db1dbacb22baf3a43a77a9c089bd3eb6b903ccc83e874541c48984310c5a7"
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.90/node-v0.1.90.tar.gz" ] && NODEHASH="0dbd47f6be45049a54de6ff268b25a8ccf8cac38bd75788e713dab35a14695c3"
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.91/node-v0.1.91.tar.gz" ] && NODEHASH="dcdbd47c9bbe508c86f2dad4f751e5caaf1247a013cb55b8d5e75f47e800f38a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.92/node-v0.1.92.tar.gz" ] && NODEHASH="f23ba3e4596fa22dabe2f5d5718a55e3100badeb2de2ef623440d9288ce88e04"
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.93/node-v0.1.93.tar.gz" ] && NODEHASH="7f0574212b2addb53adb47422b7343ac0872e0fd9618b8e8877f61579686a95a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.94/node-v0.1.94.tar.gz" ] && NODEHASH="93d19477e069171307f66dffc665c900d80382bca021536451c785f78dba1b3d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.95/node-v0.1.95.tar.gz" ] && NODEHASH="d215477805fc7f48522f97813f0757dcc85b5b9423db1b3e6a16ea78afa375b8"
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.96/node-v0.1.96.tar.gz" ] && NODEHASH="b58c607a3f2549722acc9c7cf1f3ccd57c3c15ca7a8ca4a9e6fe3567e8a0b500"
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.97/node-v0.1.97.tar.gz" ] && NODEHASH="8ad00afa4d43c479a5025293ef83fd69e94ddef5ee2795263ca46222d1c2a954"
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.98/node-v0.1.98.tar.gz" ] && NODEHASH="6ce54b2e81ba1f4085d165bf3e853111a77e2a20de9493a0d7092665a5c9fda8"
[ "$NODEURL" = "http://nodejs.org/dist/v0.1.99/node-v0.1.99.tar.gz" ] && NODEHASH="fcd771a873609096c33e297869c52799bbcd4038b68112a9185785ca73a235d0"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.0/node-v0.10.0-darwin-x64.tar.gz" ] && NODEHASH="6499e4fc0abe82446ffccc0dac6c29e9744577bb76886d0a92cfc74f799794f4"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.0/node-v0.10.0-darwin-x86.tar.gz" ] && NODEHASH="488067bc7ecc96d6bb3df0130304c1f1cd01b5138b85427032c15f83fbf1dd27"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.0/node-v0.10.0-linux-x64.tar.gz" ] && NODEHASH="a91c84f993c1674be7548deb81486bf34ade4aa9154f7932d294ed945228a5be"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.0/node-v0.10.0-linux-x86.tar.gz" ] && NODEHASH="5e91e786246a82df01762f67a2307443628e432da4936be0ed2d1e224c3fee47"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.0/node-v0.10.0-sunos-x64.tar.gz" ] && NODEHASH="6fe0569bdc90d3d2a4cf49fe03ab4213085cfc60bc1714d3bbe058543e8c95be"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.0/node-v0.10.0-sunos-x86.tar.gz" ] && NODEHASH="bc4dd51c9daba761b2eac5af8ce31184f9ecc2036e8e0a2ca1d03a800c275582"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.0/node-v0.10.0.tar.gz" ] && NODEHASH="1624dc37866ebfb5431e3393e6b049cf238cac8ad4d20c6d567263b1259177ab"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.1/node-v0.10.1-darwin-x64.tar.gz" ] && NODEHASH="eb5cc271f39a680b1280a422e7e9da48c0033edf779616656bdc90d4c372c000"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.1/node-v0.10.1-darwin-x86.tar.gz" ] && NODEHASH="303151e500ddfe94c04bdfaec825d953414f599b470b04a8346c914c23918460"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.1/node-v0.10.1-linux-x64.tar.gz" ] && NODEHASH="90c555a55ab5d343148511623fdb7c37a6888952db5340734d1c2b8f0f01dd11"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.1/node-v0.10.1-linux-x86.tar.gz" ] && NODEHASH="c7c90a51239a7dfc7a811b123682546d6f3688afbf3a3377020095b1c182133d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.1/node-v0.10.1-sunos-x64.tar.gz" ] && NODEHASH="0908f81bb72c986e5b80b6a6f4c7848d37a46b360f83f2bdddc090e810196a5b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.1/node-v0.10.1-sunos-x86.tar.gz" ] && NODEHASH="e90bb06af4262bf770bb0e2cbaa5689e1b6bf0ff7c8cfae069d27739f3ba0680"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.1/node-v0.10.1.tar.gz" ] && NODEHASH="2628dbf42fb3ec3927e595dc66f2f96e3c23455990dea690e300296d92afe4d3"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.10/node-v0.10.10-darwin-x64.tar.gz" ] && NODEHASH="ecec87637784f98318efa5f076a25985609387782b3e371b7596e939775d7cb6"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.10/node-v0.10.10-darwin-x86.tar.gz" ] && NODEHASH="a1555490bae2e7bafb96fe0d12a4fb97949c54e6c083b70d6aef672d77f15b26"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.10/node-v0.10.10-linux-x64.tar.gz" ] && NODEHASH="ab42335b0e6e45bac62823d995d8062e9ba0344bc416c76a263a5e45773b2e7d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.10/node-v0.10.10-linux-x86.tar.gz" ] && NODEHASH="61f755eb02fa9520d7d5116740742721158d123058ca4b56bd6ee068d6b9ad96"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.10/node-v0.10.10-sunos-x64.tar.gz" ] && NODEHASH="7dafccd4548b4829f31c77cd9261f3cd64d7799422d03d9903a6fe7b402f7b2b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.10/node-v0.10.10-sunos-x86.tar.gz" ] && NODEHASH="2f38f3229d1efbd13326e9ed7ec0b00ce82fbe6699c3f76f7d90dbee2a7a9b69"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.10/node-v0.10.10.tar.gz" ] && NODEHASH="a54de71d2c3ac7ae864ab9640b6eecb27d7d49c190ac1ca6526243fd7a8ad15c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.11/node-v0.10.11-darwin-x64.tar.gz" ] && NODEHASH="0c3e0ba53f3878aa5b3dac9b87cc1eff0dc8110b75d7b3b004a61ebdda734ece"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.11/node-v0.10.11-darwin-x86.tar.gz" ] && NODEHASH="6c957697248db3a4ba202dbd3d3c01c3051f9a6c6f13075c4c1e2b2f31b071a0"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.11/node-v0.10.11-linux-x64.tar.gz" ] && NODEHASH="0fa2be9b44d6acd4bd43908bade00053de35e6e27f72a2dc41d072c86263b52a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.11/node-v0.10.11-linux-x86.tar.gz" ] && NODEHASH="2a08c5d1e19591ec8ea0fbd54f7a17ebbce447eb6b98d1a89392969cee24c949"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.11/node-v0.10.11-sunos-x64.tar.gz" ] && NODEHASH="2ef86b94f0abc16ffb49cff6444fc4f6eb32b476cc06938e1aeb64f705001c91"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.11/node-v0.10.11-sunos-x86.tar.gz" ] && NODEHASH="60237169d3b698c09d4a8ad5260065f338296e592bbce9ee768ea455c33f65f8"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.11/node-v0.10.11.tar.gz" ] && NODEHASH="ee4b398efde1fa7a334435910447422dae58e93da8711602c2228485f2b58cb1"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.12/node-v0.10.12-darwin-x64.tar.gz" ] && NODEHASH="e41d873b0444937ac1f2816ea8c6c4abb3f0af260c2fc1419e83fb7ea5a23090"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.12/node-v0.10.12-darwin-x86.tar.gz" ] && NODEHASH="650c0c2e90fa386cbea97a0e0c963fa14c2f2203856c5282722ebbce72717753"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.12/node-v0.10.12-linux-x64.tar.gz" ] && NODEHASH="d35f3ddb0e8f2de42f9da225a56c19a7aa5c62276d4278242f31087c0397adb8"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.12/node-v0.10.12-linux-x86.tar.gz" ] && NODEHASH="20729e52c9086bf5887fb2ac297b8cae09a85dab6e54677e614931a74a7c3c0d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.12/node-v0.10.12-sunos-x64.tar.gz" ] && NODEHASH="eb50e180f02f4fa8157527d7d906b28c3ae9ccc7c081b72d28159c90f2f5a861"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.12/node-v0.10.12-sunos-x86.tar.gz" ] && NODEHASH="8d0c36f5a15464ede5c8fe487c81c245def3a772fa014380d506d6bec7f268c5"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.12/node-v0.10.12.tar.gz" ] && NODEHASH="7339a7c333454a567a41c900b6ef2f6c89e8c778062c173beb029611b29496b6"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.13/node-v0.10.13-darwin-x64.tar.gz" ] && NODEHASH="45f014d08120cb2a3f03616338e2931d673190a52692336cde386e1a91310115"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.13/node-v0.10.13-darwin-x86.tar.gz" ] && NODEHASH="f41b547f80f3aaca7540f9d51ccde5f07f1b3185bd2574f445da1a52eb82a3f6"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.13/node-v0.10.13-linux-x64.tar.gz" ] && NODEHASH="dcbad86b863faf4a1e10fec9ecd7864cebbbb6783805f1808f563797ce5db2b8"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.13/node-v0.10.13-linux-x86.tar.gz" ] && NODEHASH="ea7332fcbbee8e33c2f7d9b0e75c9bb299f276b334b26752725aa8b9b0ee3c99"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.13/node-v0.10.13-sunos-x64.tar.gz" ] && NODEHASH="04ec26f8b20606781a08d6afefc17e83f377e59d188b5e491fdacefd1cfb73fa"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.13/node-v0.10.13-sunos-x86.tar.gz" ] && NODEHASH="8a86ef4770aada5d21dd9848c43c036dfd7e13a8423861844a165df0c2b72870"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.13/node-v0.10.13.tar.gz" ] && NODEHASH="a102fad260d216b95611ddd57aeb6531c92ad1038508390654423feb1b51c059"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.14/node-v0.10.14-darwin-x64.tar.gz" ] && NODEHASH="1a2f35f53be472de27bef88e6e2efc69b4d563b384ab33eba39a6035570bcc67"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.14/node-v0.10.14-darwin-x86.tar.gz" ] && NODEHASH="e867defaa3c7e332f13f92a89263089e6fa70bb1d868a7981d708f227b2575d9"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.14/node-v0.10.14-linux-x64.tar.gz" ] && NODEHASH="fbed54f3d87febc679823d5309aae52f19f104b8ac6927849b9b852a6fb7d060"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.14/node-v0.10.14-linux-x86.tar.gz" ] && NODEHASH="0bfba5ba21b1853eedbf4680fa5abb0c3af91a7c652f0cdfcd8d3ed2797e0647"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.14/node-v0.10.14-sunos-x64.tar.gz" ] && NODEHASH="5e2796a91441eb7a5326eb35b35e1bd404b45fc125f7a3b0f7efa812957c5040"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.14/node-v0.10.14-sunos-x86.tar.gz" ] && NODEHASH="ff49a2b8ab36b041889adbebb9e9b72cf79f73264810f057e00500780bb89497"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.14/node-v0.10.14.tar.gz" ] && NODEHASH="5b022575b9f6e5c3c704dfa9ecaf511e1b14eaca196edb3eaf89d246efeb70d9"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.15/node-v0.10.15-darwin-x64.tar.gz" ] && NODEHASH="f4ea534e6ca6326a8bf823b6c44ae50f5d968c105a18c6372a8bde2d50518489"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.15/node-v0.10.15-darwin-x86.tar.gz" ] && NODEHASH="31f26458689948eaa0a1747cc550a205b8a70b4a8b59737ccfc0fca5b1ce43be"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.15/node-v0.10.15-linux-x64.tar.gz" ] && NODEHASH="0b5191748a91b1c49947fef6b143f3e5e5633c9381a31aaa467e7c80efafb6e9"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.15/node-v0.10.15-linux-x86.tar.gz" ] && NODEHASH="7ff9fb6aa19a5269a5a2f7a770040b8cd3c3b528a9c7c07da5da31c0d6dfde4d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.15/node-v0.10.15-sunos-x64.tar.gz" ] && NODEHASH="bcfff390eb599536114d2697272c6f1b19daa88677428cef5ff9535d6cc24551"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.15/node-v0.10.15-sunos-x86.tar.gz" ] && NODEHASH="766eefedc60f7fbd0b96f71738710021570f52ae2c68e936d165312084da84ed"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.15/node-v0.10.15.tar.gz" ] && NODEHASH="87345ab3b96aa02c5250d7b5ae1d80e620e8ae2a7f509f7fa18c4aaa340953e8"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.16/node-v0.10.16-darwin-x64.tar.gz" ] && NODEHASH="ca2cfb44f4b592840643f50b11b06aef56328a647afc73db7dc37d7cc61bf393"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.16/node-v0.10.16-darwin-x86.tar.gz" ] && NODEHASH="8a7dc94a4661ff0674c0e5000526cf1a1c614ca46f48e5a17917e65867c1641c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.16/node-v0.10.16-linux-arm-pi.tar.gz" ] && NODEHASH="fcf96d7d1ed743adfd84903ad374acb4ff885f51ee895619501449cf227d15d1"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.16/node-v0.10.16-linux-x64.tar.gz" ] && NODEHASH="5a2e29d41c6b9eb79e25df6e10fcea84cf44def31c79dec967e8a108c01b02f6"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.16/node-v0.10.16-linux-x86.tar.gz" ] && NODEHASH="045b646f1b9b07bb89e86a46759de9b8a696dcc75b31f77089887efc6808d1ab"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.16/node-v0.10.16-sunos-x64.tar.gz" ] && NODEHASH="2e00695ee36a90c52aa63233827128faf76f93c2d9128220198a344ab3c79a13"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.16/node-v0.10.16-sunos-x86.tar.gz" ] && NODEHASH="9fe6130a62de0c8d9c58a5e7eaa274a413176fce8c8128a3435f5839d629fcdc"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.16/node-v0.10.16.tar.gz" ] && NODEHASH="62db6ec83f34733bffdae2f8ee50d7273d568bba8814be29e78a042c1d64f31a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.17/node-v0.10.17-darwin-x64.tar.gz" ] && NODEHASH="4c2df96d444704aba89ac69952f9cfade9b7651f736c9ccd09e0b40200b56832"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.17/node-v0.10.17-darwin-x86.tar.gz" ] && NODEHASH="430c50fbcf4253eb703ab8d0a9992fbb1141bf51076487fcbdb0dab1a63c5003"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.17/node-v0.10.17-linux-x64.tar.gz" ] && NODEHASH="a4cf2690394cdb2468482816be22365544475777c2e9cf4058ef3015e33b7993"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.17/node-v0.10.17-linux-x86.tar.gz" ] && NODEHASH="2a2dfc7fd173d5c2239ac015fd88b97b0918b9b1a51ee6c965a085e68b27b51c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.17/node-v0.10.17-sunos-x64.tar.gz" ] && NODEHASH="b7b57bb51f0439c2df72a001e2dc6ed5ce2172c2417bac1df4c87d531508128f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.17/node-v0.10.17-sunos-x86.tar.gz" ] && NODEHASH="8a7a2f43a28260bbcc02b0cb3d8a1689f5fdda847c35cce09fb98a47ae62157a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.17/node-v0.10.17.tar.gz" ] && NODEHASH="1f9de40dd2d98bc984a4970ef399ed3ad67f040feaafc711e549f3265bcce61c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.18/node-v0.10.18-darwin-x64.tar.gz" ] && NODEHASH="9a0fafc8142a47ae1517c8df2b53188d181f13022d9560202fc906f35575cef0"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.18/node-v0.10.18-darwin-x86.tar.gz" ] && NODEHASH="60432fbf590b0e88028fc9f969284d051ae045cf8f40f7f6ddb1bb40705f70c2"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.18/node-v0.10.18-linux-x64.tar.gz" ] && NODEHASH="480aed8ec0a2acf6c7cc168650045cea559c5b69c9b8538a181af830662c1262"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.18/node-v0.10.18-linux-x86.tar.gz" ] && NODEHASH="6db67cb804a07365176fcb6e1b2785480050a0321ad0b3e6e7a2421ed581f59f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.18/node-v0.10.18-sunos-x64.tar.gz" ] && NODEHASH="a70735329c07c3cc462011721fc81302e2a4cd7b3c71b5e8586b82dd05df253e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.18/node-v0.10.18-sunos-x86.tar.gz" ] && NODEHASH="d8b0a69fed8d2181dda3c730e4b00d070b23552da2bc939f3dd0ebbaa4ff418d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.18/node-v0.10.18.tar.gz" ] && NODEHASH="3ee4436473869d4d84bb5cad4352b09ace00656467eca7d6db7cd7da5b8c5495"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.19/node-v0.10.19-darwin-x64.tar.gz" ] && NODEHASH="4a51a73b03184dacd10abb12bfc461db1c35c9167f4348ed851f51ed56806af8"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.19/node-v0.10.19-darwin-x86.tar.gz" ] && NODEHASH="79e225bc267a389558ab4a06b6838c2bc288d2088787afff49b6984e495bf0dc"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.19/node-v0.10.19-linux-x64.tar.gz" ] && NODEHASH="d9ff0bbc075149a91ac97dba8dabdf4473506527bc3c9461fe2cce92d3da1191"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.19/node-v0.10.19-linux-x86.tar.gz" ] && NODEHASH="3d5c7f4e2ea66fcf6329de41d31d0f0e44846502cfe2f46e42ae2161d0aa9216"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.19/node-v0.10.19-sunos-x64.tar.gz" ] && NODEHASH="aed9156f656d692e6cc6a40e600df0021e846c58fa0e0b023debe9fc542c0496"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.19/node-v0.10.19-sunos-x86.tar.gz" ] && NODEHASH="92a7627d6faefa8474610ba53c9065f841cde5620528f1b0b73d0750d08dd0e6"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.19/node-v0.10.19.tar.gz" ] && NODEHASH="e50787672cdf6afa6caeef9345ca40c4a69f96a31829a0884ea6ed63dfdde21e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.2/node-v0.10.2-darwin-x64.tar.gz" ] && NODEHASH="cef798ba85522f1e82076cf60cb27ea7d85406ecea461e9dd1b5503d48a23c3b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.2/node-v0.10.2-darwin-x86.tar.gz" ] && NODEHASH="30ec4d0bf99ea44f760f7bf3f4ddbaea305560635991aa96ea2d8ef7590005b2"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.2/node-v0.10.2-linux-x64.tar.gz" ] && NODEHASH="44ff658b1c3ae027b75310e0173b7d069ae70f6adaed23d22f2e087f5048c428"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.2/node-v0.10.2-linux-x86.tar.gz" ] && NODEHASH="9970b12b1dda8cbb6272d90b073da90336bce8667b2d57936106bd50c4a035dd"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.2/node-v0.10.2-sunos-x64.tar.gz" ] && NODEHASH="d5f211013177b65115364419bbdc95b68aabb293e115ae127a9c3f31404feeaa"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.2/node-v0.10.2-sunos-x86.tar.gz" ] && NODEHASH="8689f820535a84e7b488abb9d83c153f1c591b87a7b89a0e6b88e024a1b3d445"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.2/node-v0.10.2.tar.gz" ] && NODEHASH="4eb642897fdb945b49720f2604afc493587aec7a9ff1537e882df659e4dd8aa2"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.20/node-v0.10.20-darwin-x64.tar.gz" ] && NODEHASH="f059b3d9dfd42fa9d7d8542e51aea6c92d87aff1b9023fc1c7c12acb7f3d28e5"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.20/node-v0.10.20-darwin-x86.tar.gz" ] && NODEHASH="69a921ab23ae6e3aecf03ae8c289d33bf9f24a6b29d36b8493b759856a113e7c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.20/node-v0.10.20-linux-x64.tar.gz" ] && NODEHASH="eaebfc66d031f3b5071b72c84dd74f326a9a3c018e14d5de7d107c4f3a36dc96"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.20/node-v0.10.20-linux-x86.tar.gz" ] && NODEHASH="4dc94e7de766523f6427b9de75dd3e4f1d3d15d01464e03d98f9c96e09769746"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.20/node-v0.10.20-sunos-x64.tar.gz" ] && NODEHASH="0cb7827eb73165a22e9c770d4b27aaa47ec13eeb8e49ddd8b5d7111f099bab21"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.20/node-v0.10.20-sunos-x86.tar.gz" ] && NODEHASH="0bc199ab554e28536eba659b931471d69048f3d5c01f5d14cc9cb88dc1f0e0f7"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.20/node-v0.10.20.tar.gz" ] && NODEHASH="b5af00d4d43644f37caa2dad5fa81e6f898ebb7b238f02a1cddfcff11c81e649"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.21/node-v0.10.21-darwin-x64.tar.gz" ] && NODEHASH="26a2c00b3b61d34a1558b70e093230ead737f43de2be2954deeeef2edd54e7bb"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.21/node-v0.10.21-darwin-x86.tar.gz" ] && NODEHASH="88601fa705729002c828d53ac7e8ed716ec860a13985a7c48fcbb5efb8f7f23a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.21/node-v0.10.21-linux-x64.tar.gz" ] && NODEHASH="2791efef0a1e9a9231b937e55e5b783146e23291bca59a65092f8340eb7c87c8"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.21/node-v0.10.21-linux-x86.tar.gz" ] && NODEHASH="c0ed641961a5c5a4602b1316c3d3ed12b3ac330cc18abf3fb980f0b897b5b96b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.21/node-v0.10.21-sunos-x64.tar.gz" ] && NODEHASH="1f6eb3674fbd370f1d5db2d00af39c6805dfa725a2f727adfcdea757f51155e4"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.21/node-v0.10.21-sunos-x86.tar.gz" ] && NODEHASH="11bfafb20045589dddff2bb12e23f2ae98e877a4a4dcc82a76bf2ad4a28ad93f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.21/node-v0.10.21.tar.gz" ] && NODEHASH="7c125bf22c1756064f2a68310d4822f77c8134ce178b2faa6155671a8124140d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.22/node-v0.10.22-darwin-x64.tar.gz" ] && NODEHASH="2e9313ce780c7eb377105295dd1b0c803fe78171a26756b1914781c09c67f4bd"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.22/node-v0.10.22-darwin-x86.tar.gz" ] && NODEHASH="cb17ea51b6d8ac590396a72d4176903b52d6073a73a2a338dd160f326ea3ef4b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.22/node-v0.10.22-linux-x64.tar.gz" ] && NODEHASH="ca5bebc56830260581849c1099f00d1958b549fc59acfc0d37b1f01690e7ed6d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.22/node-v0.10.22-linux-x86.tar.gz" ] && NODEHASH="3823d08199b2c952cd85d1b89ba03d59f2782985ba8d25e040e4cfecdb679aff"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.22/node-v0.10.22-sunos-x64.tar.gz" ] && NODEHASH="550bf8a08074e0260a8c700b377fc7e14253d3f690d8c7d8dc9becc02d406bc7"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.22/node-v0.10.22-sunos-x86.tar.gz" ] && NODEHASH="bb556888914f3d249176dc3a30bf3dbb9a7ce4073f326549b1379f191c1aa610"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.22/node-v0.10.22.tar.gz" ] && NODEHASH="157fc58b3f1d109baefac4eb1d32ae747de5e6d55d87d0e9bec8f8dd10679e7e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.23/node-v0.10.23-darwin-x64.tar.gz" ] && NODEHASH="295720906b3f84bd083949125c14920801d98fe7e72e234ea03caa9829af3f84"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.23/node-v0.10.23-darwin-x86.tar.gz" ] && NODEHASH="30973b77529cc9422831f73f6121e4e9f7d8d0cb856cf6397b2dcd435307fa8a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.23/node-v0.10.23-linux-x64.tar.gz" ] && NODEHASH="0ebee6f9b937ed00efc777c468593e6a277dae897a3700090229c2e18d4a4304"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.23/node-v0.10.23-linux-x86.tar.gz" ] && NODEHASH="190104bfcaf9f8d5a5e6d0cec3c36b1ab4dcff159bc921d5214b9b38f8d0c33a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.23/node-v0.10.23-sunos-x64.tar.gz" ] && NODEHASH="7f41c56a699d48dab5fc725a39f3b399c3244aadba5ba7ee8f3a0ea0f39387fc"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.23/node-v0.10.23-sunos-x86.tar.gz" ] && NODEHASH="d3a55483d7675a335f9c69db8a21995cab2fa371f5870d774bc3ee3ae7f1b75e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.23/node-v0.10.23.tar.gz" ] && NODEHASH="1a370c86720441d227e7aad3c0223da7166ab2e0900bec7158aac633ffebb4dd"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.24/node-v0.10.24-darwin-x64.tar.gz" ] && NODEHASH="c1c523014124a0327d71ba5d6f737a4c866a170f1749f8895482c5fa8be877b0"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.24/node-v0.10.24-darwin-x86.tar.gz" ] && NODEHASH="8b8d2bf9828804c3f8027d7d442713318814a36df12dea97dceda8f4aff42b3c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.24/node-v0.10.24-linux-x64.tar.gz" ] && NODEHASH="6ef93f4a5b53cdd4471786dfc488ba9977cb3944285ed233f70c508b50f0cb5f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.24/node-v0.10.24-linux-x86.tar.gz" ] && NODEHASH="fb6487e72d953451d55e28319c446151c1812ed21919168b82ab1664088ecf46"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.24/node-v0.10.24-sunos-x64.tar.gz" ] && NODEHASH="7cb714df92055b93a908b3b6587ca388a2884b1a9b5247c708a867516994a373"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.24/node-v0.10.24-sunos-x86.tar.gz" ] && NODEHASH="af69ab26aae42b05841c098f5d11d17e21d22d980cd32666e2db45a53ddffe34"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.24/node-v0.10.24.tar.gz" ] && NODEHASH="610cd733186842cb7f554336d6851a61b2d3d956050d62e49fa359a47640377a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.25/node-v0.10.25-darwin-x64.tar.gz" ] && NODEHASH="5ae186f86564df87d82811614f667b9333a6219b02c6f405fd72807b459b8fc7"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.25/node-v0.10.25-darwin-x86.tar.gz" ] && NODEHASH="22af5001e5b0166fa91934107c17727e173677e5c972bd7122373743309c5372"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.25/node-v0.10.25-linux-x64.tar.gz" ] && NODEHASH="1dac61c21fa21e47fc6e799757569c6c3914897ca46fc8f4dd2c8f13f0400626"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.25/node-v0.10.25-linux-x86.tar.gz" ] && NODEHASH="3f1fa0bb332b1354bca8d52d89e92c8884b6469a0f5fc3826ee72c2639279995"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.25/node-v0.10.25-sunos-x64.tar.gz" ] && NODEHASH="931059671413872c5c5e862df5f7a56066fdb1fe2b678b9ee3c3b242b23a4198"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.25/node-v0.10.25-sunos-x86.tar.gz" ] && NODEHASH="61d187416814dd10074d1db5666fd0ca61be6152fbb7920f0aaa4e285db10717"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.25/node-v0.10.25.tar.gz" ] && NODEHASH="46eef3b9d5475a2081dc2b2f7cf1f4c3a56824d1fc9b04e7ed1d7a88e8f6b36f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.26/node-v0.10.26-darwin-x64.tar.gz" ] && NODEHASH="863ec06ea2f9f613de39b15d081cc2fa422937e3492cc09b12c73ac884c80d8f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.26/node-v0.10.26-darwin-x86.tar.gz" ] && NODEHASH="1d851d281d82d49a6b910541dba9fb02851883db485e636e7e890483229effd4"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.26/node-v0.10.26-linux-x64.tar.gz" ] && NODEHASH="305bf2983c65edea6dd2c9f3669b956251af03523d31cf0a0471504fd5920aac"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.26/node-v0.10.26-linux-x86.tar.gz" ] && NODEHASH="8fa2d952556c8b5aa37c077e2735c972c522510facaa4df76d4244be88f4dc0f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.26/node-v0.10.26-sunos-x64.tar.gz" ] && NODEHASH="9d9c05a76cb3353b459e7204d5283ed3988b7a363105ef9698dc0f94243aabf9"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.26/node-v0.10.26-sunos-x86.tar.gz" ] && NODEHASH="ac7ec92eb6839fa9aa45f0be2c586a0787d26fbdaefdcbaedc912f7c5fd17f3d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.26/node-v0.10.26.tar.gz" ] && NODEHASH="ef5e4ea6f2689ed7f781355012b942a2347e0299da0804a58de8e6281c4b1daa"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.27/node-v0.10.27-darwin-x64.tar.gz" ] && NODEHASH="17d4711d0510da53c604bacce68066891ef0dff85e830fbcd2d1b0410bd1ddde"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.27/node-v0.10.27-darwin-x86.tar.gz" ] && NODEHASH="8d52be48abd16bdbcaf5487367c7964632fcb5cd9b523f77382544e4783fd2ed"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.27/node-v0.10.27-linux-x64.tar.gz" ] && NODEHASH="919ef2245045f78725bec9152c711751a1278a8053b86dd181363c0b32465609"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.27/node-v0.10.27-linux-x86.tar.gz" ] && NODEHASH="c4127f6c1dc412c51aecfae4d2015d7221795e5b253f9007a5da21276f62c3e7"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.27/node-v0.10.27-sunos-x64.tar.gz" ] && NODEHASH="37a3a1834c3b7a776ff166c070fa3b84a53b9951bc5ea3b417f62bf3a0970d9b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.27/node-v0.10.27-sunos-x86.tar.gz" ] && NODEHASH="5ec564db1c643c5207c539593d1c37254d9e99f8efee04e35e110f05d190b921"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.27/node-v0.10.27.tar.gz" ] && NODEHASH="911876c38974a77e1ddf141285b0a994d8c98bddac7229802d7f16600d69d1fe"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.28/node-v0.10.28-darwin-x64.tar.gz" ] && NODEHASH="304122e04df8ba3fadaf1ff442b2eb1181bf3ce0d2539a0a7c70a62157aed1f4"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.28/node-v0.10.28-darwin-x86.tar.gz" ] && NODEHASH="24d9304c1a94c486c68860a78f3f086968443d11d4f4e59949bbc7c458b18e55"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.28/node-v0.10.28-linux-x64.tar.gz" ] && NODEHASH="5f41f4a90861bddaea92addc5dfba5357de40962031c2281b1683277a0f75932"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.28/node-v0.10.28-linux-x86.tar.gz" ] && NODEHASH="81ee7f30c35e1743790fd9ca47235bdec4a6c9d2b89a70f33c69e80008cbf422"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.28/node-v0.10.28-sunos-x64.tar.gz" ] && NODEHASH="dadde2dde10ca1429564ac125cb67fd7f34d1e5332c82ed67abbf5e6224ded15"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.28/node-v0.10.28-sunos-x86.tar.gz" ] && NODEHASH="a1820e23a9f9e908079c3b4379a73b884ab4f56acefe28c7593301f51c794d0a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.28/node-v0.10.28.tar.gz" ] && NODEHASH="abddc6441e0f208f6ed8a045e0293f713ea7f6dfb2d6a9a2024bf8b1b4617710"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.29/node-v0.10.29-darwin-x64.tar.gz" ] && NODEHASH="219299cd959819e215c4fcd8dfd3ba3ebbf634972667e6ef70e0d1ead741bb27"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.29/node-v0.10.29-darwin-x86.tar.gz" ] && NODEHASH="ea936c173c8f14c8493983fd4e5bd9df08ac8dee268640a841ccaedbd621d49d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.29/node-v0.10.29-linux-x64.tar.gz" ] && NODEHASH="ac52da27a4e298a6de610de25b22628bdb97b78cb29d11464ef5cfa2e57847d5"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.29/node-v0.10.29-linux-x86.tar.gz" ] && NODEHASH="6caaea6c350a143a69ab0b89063476292437091ca96f9896413e1972aadae093"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.29/node-v0.10.29-sunos-x64.tar.gz" ] && NODEHASH="988d48b7f65a94c0ee7edbef555156f555d13cf302fff54d27424fd4c0d4392f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.29/node-v0.10.29-sunos-x86.tar.gz" ] && NODEHASH="ef5c2a9237c3a2b7c7ce9d4817fbcf63dd223370d2072da8f7ab2f4b46849e17"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.29/node-v0.10.29.tar.gz" ] && NODEHASH="47379d01f765f87c1a1498b4e65de30e45201de50334954860d7375a8258b15d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.3/node-v0.10.3-darwin-x64.tar.gz" ] && NODEHASH="5656918cc7cc480afb9fa99053623eb9a9b392db2e2a3d5f60333a1b27eeb495"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.3/node-v0.10.3-darwin-x86.tar.gz" ] && NODEHASH="3c16193df9bc44e6721f74341c3b26c1f31b5c28067af5cf8397014c37d4d088"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.3/node-v0.10.3-linux-x64.tar.gz" ] && NODEHASH="62ce0353c80023cd2eb0a60999b10839d732f5ae3eb75c10d7b3924745b32d21"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.3/node-v0.10.3-linux-x86.tar.gz" ] && NODEHASH="7603404f5b86048e804fc5f2bdd500d716e8e945ab3480c44881a3fb0074824f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.3/node-v0.10.3-sunos-x64.tar.gz" ] && NODEHASH="d9bed27c97968dc54047ef6f09161d6a1a7dee702301fa65add97ff1dee6e2af"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.3/node-v0.10.3-sunos-x86.tar.gz" ] && NODEHASH="5a4b8e6460a446481fe3fb8eb681d7408711fb3cfbb8bcc06a436706567bc76c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.3/node-v0.10.3.tar.gz" ] && NODEHASH="bc8796ff6414231fa0603e0383404f14648dfd2fe9fb0fa4d4a6043dfddbb328"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.30/node-v0.10.30-darwin-x64.tar.gz" ] && NODEHASH="d13b5ad8294c39175a434cdcb3233cc1ddecfd92b23eb4dfdec3931d65cb5796"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.30/node-v0.10.30-darwin-x86.tar.gz" ] && NODEHASH="a9fc5bc8e8f916010b15cf71305f6e285e64e5d1fd1be8c3472f95ff88106f67"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.30/node-v0.10.30-linux-x64.tar.gz" ] && NODEHASH="173d2b9ba4cbfb45a2472029f2904f965081498381a34d01b3889a850238de2b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.30/node-v0.10.30-linux-x86.tar.gz" ] && NODEHASH="68f1669bbac18d2ec73377dc7518fdbc3b8dd2a43fd557bf29cc3832c17b1e71"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.30/node-v0.10.30-sunos-x64.tar.gz" ] && NODEHASH="a7fd48d2461ce2c0673ae49140692764dc12a9c80ae5cdf6437a353b18895da4"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.30/node-v0.10.30-sunos-x86.tar.gz" ] && NODEHASH="63599ba80410abb30d27bc1360048607bf43d548eb762ec384f673bff2169a48"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.30/node-v0.10.30.tar.gz" ] && NODEHASH="3dfcbd307f5f5f266ef174e1443107da853cd3d0aa0b2493a44235d5908625d2"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.31/node-v0.10.31-darwin-x64.tar.gz" ] && NODEHASH="4dc3e53bafefd02c3ac009dd53e7acd725f109f344121e54f76c2bea8a6638c8"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.31/node-v0.10.31-darwin-x86.tar.gz" ] && NODEHASH="ff93505fb81d93150f5bff7a9e9784e6b6ae86b7ce6bde8ba4589c80b01a0eaa"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.31/node-v0.10.31-linux-x64.tar.gz" ] && NODEHASH="493aa5d4fac0f34df01b07c7d276f1da8d5139df82374c599ab932e740d52147"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.31/node-v0.10.31-linux-x86.tar.gz" ] && NODEHASH="fd6549cc1df26eb769a6f4a170d750acf6425ef2112f4542b95efcd73d51146b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.31/node-v0.10.31-sunos-x64.tar.gz" ] && NODEHASH="9334ffbad58b01e892831b44f122a61fd5453781299793eeb48e59dffdabd989"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.31/node-v0.10.31-sunos-x86.tar.gz" ] && NODEHASH="0362c780c56eb814bc5ae0d4e393663b9accdb4760e1aacd03ed76604559087b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.31/node-v0.10.31.tar.gz" ] && NODEHASH="06c781718a674dfdfb59d646b2629a46af2644bdbf52534fab8d4a0fe34c21f1"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.32/node-v0.10.32-darwin-x64.tar.gz" ] && NODEHASH="730ee5c3a44790998bbee1471ca903e6802e79080d4eba0fabf5ab92035e9037"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.32/node-v0.10.32-darwin-x86.tar.gz" ] && NODEHASH="637dcbf4ad656ef3ee51a527b8f1ef2c5feeeba4fbc25703eb763ed0519af9d8"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.32/node-v0.10.32-linux-x64.tar.gz" ] && NODEHASH="621777798ed9523a4ad1c4d934f94b7bc765871d769a014a53a4f1f7bcb5d5a7"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.32/node-v0.10.32-linux-x86.tar.gz" ] && NODEHASH="05c2ea9edb7a7bea74ecc1a68925428d3d11cececda9b6916aa788c18ce961f7"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.32/node-v0.10.32-sunos-x64.tar.gz" ] && NODEHASH="3abd0a03e6894a7515a6a13b5ffabe9dc98d5f86feb03b9d7c499e47250e0cfa"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.32/node-v0.10.32-sunos-x86.tar.gz" ] && NODEHASH="03925e461542e2601bbe1efb7434334a817c1a0c6b071fb81ab1d9a1805c7e58"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.32/node-v0.10.32.tar.gz" ] && NODEHASH="c2120d0e3d2d191654cb11dbc0a33a7216d53732173317681da9502be0030f10"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.33/node-v0.10.33-darwin-x64.tar.gz" ] && NODEHASH="a5b6c3daafbdef5bee0376a147b3f74990bc032d9ab0b394600b4336b6e19f8f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.33/node-v0.10.33-darwin-x86.tar.gz" ] && NODEHASH="2eaf06d4a7ddf62d83a053026744eb23b53d086b5029094676d71866e9386265"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.33/node-v0.10.33-linux-x64.tar.gz" ] && NODEHASH="159e5485d0fb5c913201baae49f68fd428a7e3b08262e9bf5003c1b399705ca8"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.33/node-v0.10.33-linux-x86.tar.gz" ] && NODEHASH="b34b8481659388207866a7b556f82dc25a9be649fcb1c2eb6aac78b2fc5d952f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.33/node-v0.10.33-sunos-x64.tar.gz" ] && NODEHASH="2ae96ea055262a265e6b32e4ec611efbabeedb9944fabc727035ec73025546aa"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.33/node-v0.10.33-sunos-x86.tar.gz" ] && NODEHASH="f01ba3a4c81c264a7f0052a4ec031f715b5e35e0337cc3686da874f4ffa91ddb"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.33/node-v0.10.33.tar.gz" ] && NODEHASH="75dc26c33144e6d0dc91cb0d68aaf0570ed0a7e4b0c35f3a7a726b500edd081e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.34/node-v0.10.34-darwin-x64.tar.gz" ] && NODEHASH="a3aa311213c3491b550b9b96835d4b4e18754524a2f82c6a1bb7fcdf3a6773e3"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.34/node-v0.10.34-darwin-x86.tar.gz" ] && NODEHASH="538882716236f49cb3e059edaa53397de68e47942cbcbd506bf91b4ba8f6ffb2"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.34/node-v0.10.34-linux-x64.tar.gz" ] && NODEHASH="d9242c1b04327e8b4069ab1da96794383e562b0610942da501656e53243d04dc"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.34/node-v0.10.34-linux-x86.tar.gz" ] && NODEHASH="3fe743566843fbd164cbb7e2ac6ab198025dd5d3d42821c1b33054b3b327cc32"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.34/node-v0.10.34-sunos-x64.tar.gz" ] && NODEHASH="4e1d09ec38413380b5297edc3dc5a1fe4b28bddda3c189dbaceabadb42cae98f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.34/node-v0.10.34-sunos-x86.tar.gz" ] && NODEHASH="36db9122efa7130fa5bcdb1d9660f145c3880432f86c9ced52bd898c53a448f6"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.34/node-v0.10.34.tar.gz" ] && NODEHASH="d7f8473b5849873039f7e62595e12dcdb78c8dffda317e1253b3123876bf3415"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.35/node-v0.10.35-darwin-x64.tar.gz" ] && NODEHASH="e3ab19876c8cfafbda9587a108c90bb9e975d073af5d545a34f6336afe5c7924"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.35/node-v0.10.35-darwin-x86.tar.gz" ] && NODEHASH="c8faabb6afbd26a73f02e655738e5513a2861a51817cea0ce2e6930dc794352c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.35/node-v0.10.35-linux-x64.tar.gz" ] && NODEHASH="11f1e0ba34fb77d87db6f2c56898de881fdcf5bcde3e727f00456bfd976d2603"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.35/node-v0.10.35-linux-x86.tar.gz" ] && NODEHASH="77b5afee7d5c2e089f432dd3b75e4f5df7dde107193c99a18828590e4f422615"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.35/node-v0.10.35-sunos-x64.tar.gz" ] && NODEHASH="0cb651edf73601d35ba48316a36a34662a6412bc54fd4ea4ce019b50d6374720"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.35/node-v0.10.35-sunos-x86.tar.gz" ] && NODEHASH="909ae0755d9c9bfbc7a2a2feb24b4df23fab5801924d4061768c07375b4e7f73"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.35/node-v0.10.35.tar.gz" ] && NODEHASH="0043656bb1724cb09dbdc960a2fd6ee37d3badb2f9c75562b2d11235daa40a03"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.36/node-v0.10.36-darwin-x64.tar.gz" ] && NODEHASH="f5768eaabc692fd3ed0cb063d17962c933951b78dec09af3c30eb76c3189d9ee"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.36/node-v0.10.36-darwin-x86.tar.gz" ] && NODEHASH="b0fe1dc7ed538b14c253de2d3b0bb49317126d0736855f5eed64a79ade99fe4e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.36/node-v0.10.36-linux-x64.tar.gz" ] && NODEHASH="2bc13477684a9fe534bdc9d8f4a8caf6257a11953b57c42cad9b919ee259a0d5"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.36/node-v0.10.36-linux-x86.tar.gz" ] && NODEHASH="65edbf9d139ba572e1ed9e967b41ffee513a4fce0db600bc06b4184aa2111de2"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.36/node-v0.10.36-sunos-x64.tar.gz" ] && NODEHASH="0b4edef67c9ace5f2846b465a26d225850416db7b5272066c08574f28ac15444"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.36/node-v0.10.36-sunos-x86.tar.gz" ] && NODEHASH="8186cf8261d3135a3431cacd0de086c73774d99b27c0ab00110565653945ff06"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.36/node-v0.10.36.tar.gz" ] && NODEHASH="b9d7d1d0294bce46686b13a05da6fc5b1e7743b597544aa888e8e64a9f178c81"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.37/node-v0.10.37-darwin-x64.tar.gz" ] && NODEHASH="260561bd0f19fbaf2bc371b7dd23133e4cb1f97bc0135d1bde807bdcdcfbd017"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.37/node-v0.10.37-darwin-x86.tar.gz" ] && NODEHASH="d2b1a95abf56b1abc55b005ee3354fecc172f50fd2efa741dd98411280cde32d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.37/node-v0.10.37-linux-x64.tar.gz" ] && NODEHASH="a7d597995b2da1b81c99256bd562dc698e18a12114e67162279dcc23add1f06c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.37/node-v0.10.37-linux-x86.tar.gz" ] && NODEHASH="e80ef2284477fc0b1439fba12eff000c0015ee02fc06cbbbad02c6685ee71da6"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.37/node-v0.10.37-sunos-x64.tar.gz" ] && NODEHASH="464c664215213a88d73cadd4a2822c76dae498069f867283e11f3f7debcb3d10"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.37/node-v0.10.37-sunos-x86.tar.gz" ] && NODEHASH="635e219f453ccddb16d2e53cf049687e1d5ea92b087eb820067a0f009e53751c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.37/node-v0.10.37.tar.gz" ] && NODEHASH="a5afad14117bb194731e73b4b6635f36950e9476d3873638856cba8dbb4783a5"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.38/node-v0.10.38-darwin-x64.tar.gz" ] && NODEHASH="039644613dc9c953559ea93fcea90f278dd97c3154289759033426c7902db69f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.38/node-v0.10.38-darwin-x86.tar.gz" ] && NODEHASH="09772c15d1d19cd18c44c6d3ded1ff42168e49a3eddb589a46e369ddfa550c2b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.38/node-v0.10.38-linux-x64.tar.gz" ] && NODEHASH="d0f5771c3adefa4a3c1718206521c603526a3b67d5b1b66abd2e155d0fb77f5e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.38/node-v0.10.38-linux-x86.tar.gz" ] && NODEHASH="08ec8c049b2ed1990624661707682460332bb988d1b8b9ad394d71138179eafa"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.38/node-v0.10.38-sunos-x64.tar.gz" ] && NODEHASH="d277ae4995bc56f39f09240e102e09257d4e7d00b5e9b60428d28fe6ca057e43"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.38/node-v0.10.38-sunos-x86.tar.gz" ] && NODEHASH="3930ad6e6e005a347fdf96d6721e52ebf291dabf9531348d8bb60403b1475701"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.38/node-v0.10.38.tar.gz" ] && NODEHASH="513da8ed5e48abefdfab664f1cabc160238d314a0481148804aff8fc6552b78b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.39/node-v0.10.39-darwin-x64.tar.gz" ] && NODEHASH="0fde1b6449fb4a39b55239241258fd6a2a176eaf30ba0d3c50124f64ba485002"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.39/node-v0.10.39-darwin-x86.tar.gz" ] && NODEHASH="7400527ab1b2025fa862365489d0226d01da8527583cf9389e8e9c72edabfbc4"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.39/node-v0.10.39-linux-x64.tar.gz" ] && NODEHASH="f7b2f1a7fee90d004fc6eb5629ddde358d96f0ca2fd7ed28f53931127f9875be"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.39/node-v0.10.39-linux-x86.tar.gz" ] && NODEHASH="e37ece6e46013feed1ec328fdb7c96f71aabc2d0065b4b4b80b18881493bcc65"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.39/node-v0.10.39-sunos-x64.tar.gz" ] && NODEHASH="d016ef2ccf10f1be625b7b62898e4c4aa91fe35c03dc3a7cc96a481a07fc5728"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.39/node-v0.10.39-sunos-x86.tar.gz" ] && NODEHASH="e0f2b65ae59341c34c401c41ec3d937c16ff32c6b17e32e5e7b9be3a716bd4c5"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.39/node-v0.10.39.tar.gz" ] && NODEHASH="68f8d8f9515c4e77e2a06034b742e19e9848c1fee5bcadedc1d68f3e4302df37"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.4/node-v0.10.4-darwin-x64.tar.gz" ] && NODEHASH="b7d9a8df207e8cd41798ab507e3e28fe92d1a5667014ecfef8c1ce8b9ea39203"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.4/node-v0.10.4-darwin-x86.tar.gz" ] && NODEHASH="7ebdbb7be03ed07f31bd69f17d39c1e122d4a77c9efd01194fa1834953175fbd"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.4/node-v0.10.4-linux-x64.tar.gz" ] && NODEHASH="6d3eb0cf0438513c2a71a1ff5e9ad140574ffca5a7a100308157aa9005c5d333"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.4/node-v0.10.4-linux-x86.tar.gz" ] && NODEHASH="e9af333d611509d146a91c74acfb24224ced52a6da8a377cb52d1e48a8dd0c73"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.4/node-v0.10.4-sunos-x64.tar.gz" ] && NODEHASH="db639ab5e3e309278e9728d899aab00d9ebaa40458a0964282c5457d09bdce49"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.4/node-v0.10.4-sunos-x86.tar.gz" ] && NODEHASH="ef52596d5c6cc3fa25cb43fd6820437473e1bd036d3f3b8c6d75916a34b4930a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.4/node-v0.10.4.tar.gz" ] && NODEHASH="1c960d2822447a9e4f7c46b832ff05e86743033c6643d644975af1cbf6a44fb8"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.40/node-v0.10.40-darwin-x64.tar.gz" ] && NODEHASH="c6587b133c77c0cc858c10910dea3796a719b190aab0f62ad3e46bcee1fe08d1"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.40/node-v0.10.40-darwin-x86.tar.gz" ] && NODEHASH="39610adf6887001d79ea8721f9643ca653cfed6c300bdd558affd007d01cf40c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.40/node-v0.10.40-linux-x64.tar.gz" ] && NODEHASH="0bb15c00fc4668ce3dc1a70a84b80b1aaaaea61ad7efe05dd4eb91165620a17e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.40/node-v0.10.40-linux-x86.tar.gz" ] && NODEHASH="59c1ea305239d6d42fc4b28e40cf4d2df81ebd9d7ab060c6e340e76518e0107d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.40/node-v0.10.40-sunos-x64.tar.gz" ] && NODEHASH="4f90a86d869957592a73b548e1bfad960513f4928b41549f0e19c1e536e1bf3f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.40/node-v0.10.40-sunos-x86.tar.gz" ] && NODEHASH="fc4ff51c3d1549601d421689ec4c504a62d9fd71945fe588a49ef68e3902eb4c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.40/node-v0.10.40.tar.gz" ] && NODEHASH="bae79c2fd959aebe1629af36077bebbb760128db753da226d2344cd91499149f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.41/node-v0.10.41-darwin-x64.tar.gz" ] && NODEHASH="f55050a8774828846670fea91695a8da754768021cf1121cf91f788bb3e89d20"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.41/node-v0.10.41-darwin-x86.tar.gz" ] && NODEHASH="d0deae5ea05b8fae90ca98851e55dbd0fe8b88dab5ed658ebdb61d3e47bc0a5e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.41/node-v0.10.41-headers.tar.gz" ] && NODEHASH="2d8f14df7ae2fd999d0b0f9c1b2129f749976325b392bec3cb62827ac639fcaf"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.41/node-v0.10.41-linux-x64.tar.gz" ] && NODEHASH="ebda18d4c6545ac42b3404d629504feea0b2b9e7c7fa68de2a5bcc9059a6dc6c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.41/node-v0.10.41-linux-x86.tar.gz" ] && NODEHASH="5ab658a14106c4625cadf6976dde223ef5b332d60b03413515764b783eb452ef"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.41/node-v0.10.41-sunos-x64.tar.gz" ] && NODEHASH="9621df2ffed088f87632c5f4d176e5d49438fce5aeb7b4ce8d2eff0de153a5bf"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.41/node-v0.10.41-sunos-x86.tar.gz" ] && NODEHASH="a5f5ed4d8200e231323db083f3f2735cec13ac6584523b94ac953ad0e4874b66"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.41/node-v0.10.41.tar.gz" ] && NODEHASH="79f694e2a5c42543b75d0c69f6860499d7593136d0f6b59e7163b9e66fb2c995"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.42/node-v0.10.42-darwin-x64.tar.gz" ] && NODEHASH="356b4891c7060b6a68cf126837689807c30d43b709120b7fe6f167404612eb5a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.42/node-v0.10.42-darwin-x86.tar.gz" ] && NODEHASH="ff61d24f80db9c5e6b007103751cdb8ac6cb3ff4180972ce1532bcb934847e0c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.42/node-v0.10.42-headers.tar.gz" ] && NODEHASH="64d3e067cda5a675b44cee9875d1d72f8a29ed8bc19e476a16467336b832bfb3"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.42/node-v0.10.42-linux-x64.tar.gz" ] && NODEHASH="a9b80fb22efc483b6aef282ebb0254b5d9b092ed8091521977af593069a81d53"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.42/node-v0.10.42-linux-x86.tar.gz" ] && NODEHASH="5e528360a59314e2357184a8582d4d9dd1aa1c77687145bcf11b23e28f4ddd00"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.42/node-v0.10.42-sunos-x64.tar.gz" ] && NODEHASH="af15246d6889db4449dc46d5c4a549bb56b19481cbb801d30a09153ba71b88e1"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.42/node-v0.10.42-sunos-x86.tar.gz" ] && NODEHASH="3333f1fd394bdde5aafa424d945b002b0d0876e17396f864af9c230c81aac08b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.42/node-v0.10.42.tar.gz" ] && NODEHASH="ebc1d53698f80c5a7b0b948e1108d7858f93d2d9ebf4541c12688d85704de105"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.43/node-v0.10.43-darwin-x64.tar.gz" ] && NODEHASH="c7fb30129206fa2d74d72d3c1a86fa5cb768a1165c41be9f7c724e35c3b53ca9"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.43/node-v0.10.43-darwin-x86.tar.gz" ] && NODEHASH="a62a819832db4d6a1e9c10d286faa74c2917288a011dc7aca0476725b85cc0ae"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.43/node-v0.10.43-headers.tar.gz" ] && NODEHASH="5c6f56f2230e9ef64a443d45dd66526e2614fce4d11a711a5bc89ecc31be05f2"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.43/node-v0.10.43-linux-x64.tar.gz" ] && NODEHASH="8a439e17af1971432798ec79a70abf8fa21e03e2aa994bb7150bc088bfa482f2"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.43/node-v0.10.43-linux-x86.tar.gz" ] && NODEHASH="142bdf1cb8793bd7c7da7974c73bd2d466933f8d5b496d9a28d3b1e343b62bd0"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.43/node-v0.10.43-sunos-x64.tar.gz" ] && NODEHASH="f6cf66e77e1def7fa854ddb83a176d89f9a5b349fbb1e0bcdb742778c31e1510"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.43/node-v0.10.43-sunos-x86.tar.gz" ] && NODEHASH="b3badbc35d085723a2ad30847c5109398363b13649e199db6d22fe5f56e74c52"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.43/node-v0.10.43.tar.gz" ] && NODEHASH="c672452a61dd37cf2779bc158b65a5a22af343da19fec1cddf9bced382a2595a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.44/node-v0.10.44-darwin-x64.tar.gz" ] && NODEHASH="cc9916991a16395c4a561db6d091a9a1e9af13e66dddbd03d3c687d0ab3e52a9"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.44/node-v0.10.44-darwin-x86.tar.gz" ] && NODEHASH="7c582b190050d268b4fa4d1a4e4b3456be1a9c224407a4ef61258af6e95f01d8"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.44/node-v0.10.44-headers.tar.gz" ] && NODEHASH="1f61b9a97e4a4100ad601661d3e53556fc8287c36cc56a7c9bf1617b30c5eed4"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.44/node-v0.10.44-linux-x64.tar.gz" ] && NODEHASH="b5f4acc54e5527d793463e05b5435f11dd1f0997168aa71d53a1ff1a06c7b144"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.44/node-v0.10.44-linux-x86.tar.gz" ] && NODEHASH="8eb2b47928f56b0a7bbbde72074af84bc8864544b75f991facc3fbf4d7ed0c99"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.44/node-v0.10.44-sunos-x64.tar.gz" ] && NODEHASH="ee2867b193b53ffab308bea6f0e4c197222903ae308b1748eceaa786402d6c15"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.44/node-v0.10.44-sunos-x86.tar.gz" ] && NODEHASH="0b385a3aa1f9122bca205515917a2406ab24782ecbbb886ceb5dcba93f3a9758"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.44/node-v0.10.44.tar.gz" ] && NODEHASH="4155639d71e690cafd885f58a8be3bf97a93c28875212aac991923d3ee589be8"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.45/node-v0.10.45-darwin-x64.tar.gz" ] && NODEHASH="d1ab69f49365a0dc6b10a363965452f37f3589adee19974940cbd2b991965e70"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.45/node-v0.10.45-darwin-x86.tar.gz" ] && NODEHASH="3db0e42736fbdedc8a985269595251d9c4e4860594ce1eee4a692f8621ee5918"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.45/node-v0.10.45-headers.tar.gz" ] && NODEHASH="94028ba548c1faabb6cb92143392784d3ad07829c16852f5404933eed7d73da7"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.45/node-v0.10.45-linux-x64.tar.gz" ] && NODEHASH="54d095d12b6227460f08ec81e50f9db930ec51fa05af1b7722fa85bd2cabb5d7"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.45/node-v0.10.45-linux-x86.tar.gz" ] && NODEHASH="0d21b8e9d4213021cd954908098d07407052da1b7a97d6c585f89752c49505b4"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.45/node-v0.10.45-sunos-x64.tar.gz" ] && NODEHASH="019a1c40daff7b05efb2bd27c586d9ba9c5fe5048550b74e67e171d1495d4e30"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.45/node-v0.10.45-sunos-x86.tar.gz" ] && NODEHASH="39a0790a7e533dd77f473c009c11458a5205cfc438777139a5e9658be63dfee2"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.45/node-v0.10.45.tar.gz" ] && NODEHASH="d184bb74758d4ac69826823934cda1d46e81402fc16ebdb2ecacdc1a8fe0b568"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.46/node-v0.10.46-darwin-x64.tar.gz" ] && NODEHASH="ab9eafa76c4c9a2e0d43e6727d82b331157dc7f43fde9a3f2a4881102df9ff3a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.46/node-v0.10.46-darwin-x86.tar.gz" ] && NODEHASH="32c03a60eac3f1650d4a9219fc8510119bff19262255b868ee23850ef56cd0c9"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.46/node-v0.10.46-headers.tar.gz" ] && NODEHASH="3eb90851301b108912cb00ff3e4f6d38fa4cb4b4c47abe0e9df93d26d1c0ea17"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.46/node-v0.10.46-linux-x64.tar.gz" ] && NODEHASH="58116256f3060703e2e71f2cb5dc265a1d9fab7854a4eee15e78a95a0a87c750"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.46/node-v0.10.46-linux-x86.tar.gz" ] && NODEHASH="a0d9ffa349bd8ce0b5ab783f5a3c14787c22037900761ff296fd30c7a8645d21"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.46/node-v0.10.46-sunos-x64.tar.gz" ] && NODEHASH="cb76c6e27ff0a078fa50f5dcacace83b1dd48201210554deaca2556dcf3b31bc"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.46/node-v0.10.46-sunos-x86.tar.gz" ] && NODEHASH="8bce341355320ac2ea9835db561489e35ae9a7d6b9f20c6504ba654a36232886"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.46/node-v0.10.46.tar.gz" ] && NODEHASH="1164f2a4b663c9c0244033bfa649304a4a4e5a5ed6fab509d63084609d53198f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.47/node-v0.10.47-darwin-x64.tar.gz" ] && NODEHASH="6920608a46761c33056d78e504222a3a42dc8c0cf8ab6ff7497cd4a81b06d090"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.47/node-v0.10.47-darwin-x86.tar.gz" ] && NODEHASH="0907e94e81dc63e284e9dcb18925ceed102ceffb8a4cefab8f729c203f371c93"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.47/node-v0.10.47-headers.tar.gz" ] && NODEHASH="6587d1040697dc7be7168413910a912f33a73ed95e0c19739abad4a63681b74e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.47/node-v0.10.47-linux-x64.tar.gz" ] && NODEHASH="80757ae8f7bc3161fe44615344c784918ebd93a51ca6f789a75e3d472972eb77"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.47/node-v0.10.47-linux-x86.tar.gz" ] && NODEHASH="3ee003748c6ce90918a909ef58e21376db05a2988ba6fad92fded28541ca4006"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.47/node-v0.10.47-sunos-x64.tar.gz" ] && NODEHASH="e4c9b4ff3745477c92ebd467606a5b7af2b95a51484f1491c91e1824c7b2b4ae"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.47/node-v0.10.47-sunos-x86.tar.gz" ] && NODEHASH="f79fcd0700367506b6a8a7bf8ac0253146205ecc9617cb376744b108f126b4e3"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.47/node-v0.10.47.tar.gz" ] && NODEHASH="5281fa7ddff755c34602a09ef8027f0bda0f7851b1e374fd0e0c33da93123056"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.48/node-v0.10.48-darwin-x64.tar.gz" ] && NODEHASH="35d510ca5e8dd0c21cb11c2bf33b90f3715f92281aaaa49645f61c565c8adceb"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.48/node-v0.10.48-darwin-x86.tar.gz" ] && NODEHASH="0054a05d99e36a1e50af321d6c13ef84afde895063abd5c5e808bb84d33c2296"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.48/node-v0.10.48-headers.tar.gz" ] && NODEHASH="fd9b79cfd5814aa7a6bf095e34e890206f2eea9a3e0ab7234a1c6d125b5c389f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.48/node-v0.10.48-linux-x64.tar.gz" ] && NODEHASH="82f5fe186349ca69d8889d1079dbb86ae77ce54fce5282b806c359ce360cec7b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.48/node-v0.10.48-linux-x86.tar.gz" ] && NODEHASH="abb69775712b947ac33f219db9bf462ddfe70e895c42ba415b79056e842f0fba"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.48/node-v0.10.48-sunos-x64.tar.gz" ] && NODEHASH="7dbb5cdecedd9d15119fa556f99ba14e900c325c550f57d4f65869c046e7eba5"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.48/node-v0.10.48-sunos-x86.tar.gz" ] && NODEHASH="ebb1a347363b22d5460d35c1250ffc9dde6216be4d67c3281d2702e1de403b06"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.48/node-v0.10.48.tar.gz" ] && NODEHASH="27a1765b86bf4ec9833e2f89e8421ba4bc01a326b883f125de2f0b3494bd5549"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.5/node-v0.10.5-darwin-x64.tar.gz" ] && NODEHASH="5957284e645dd815a4a7a4898584eebe846e2a09a8d2629ec76e0f69087a400a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.5/node-v0.10.5-darwin-x86.tar.gz" ] && NODEHASH="9c488bcd41ce48481248937afded91eb5d8d4ec068b29abe1054b58cc975e651"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.5/node-v0.10.5-linux-arm-pi.tar.gz" ] && NODEHASH="b9e806dd50f66363fbb1350b54da216a764749acd7d7c810bb28a0244d01fdb5"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.5/node-v0.10.5-linux-x64.tar.gz" ] && NODEHASH="8d9cd65fb6c4ce958649b7da993a9a1f58809a1c7abb408fd85918817d0384c1"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.5/node-v0.10.5-linux-x86.tar.gz" ] && NODEHASH="12758aee9c5c9fea5abdc1e28f9111e8d739398e2c5ade476ad66d6c6818ec18"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.5/node-v0.10.5-sunos-x64.tar.gz" ] && NODEHASH="508d295599796814a9cbe79d1a254f8a2072d702ffcb1c288376cfec1dd317fd"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.5/node-v0.10.5-sunos-x86.tar.gz" ] && NODEHASH="3b7626bcd8550b95a08695bae781debebd06d575ac17c8404b388d949c8f3d55"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.5/node-v0.10.5.tar.gz" ] && NODEHASH="1c22bd15cb13b1109610ee256699300ec6999b335f3bc85dc3c0312ec9312cfd"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.6/node-v0.10.6-darwin-x64.tar.gz" ] && NODEHASH="662cbb2068aeb464aa61346cd4cb838b2c4d629ad3519fdf1915ea1da18dc470"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.6/node-v0.10.6-darwin-x86.tar.gz" ] && NODEHASH="330cb1225748991506a0c98fe703ea97c842746f23796d63d9a877f5e2ec186f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.6/node-v0.10.6-linux-x64.tar.gz" ] && NODEHASH="cc7ccfce24ae0ebb0c50661ef8d98b5db07fc1cd4a222c5d1ae232260d5834ca"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.6/node-v0.10.6-linux-x86.tar.gz" ] && NODEHASH="e8c75ffa508bb86b6858855d33139d63013c55b398f2258b020a84d8071b13f3"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.6/node-v0.10.6-sunos-x64.tar.gz" ] && NODEHASH="554ff5ade28f20b4cb17ed18a7265a7a5486de29b971f1228ed8984b40900e95"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.6/node-v0.10.6-sunos-x86.tar.gz" ] && NODEHASH="a0a7412f7087ebbc726e4fffe8205fbc775ec3b59c890e425ebb00c782eb11e2"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.6/node-v0.10.6.tar.gz" ] && NODEHASH="7e2079394efe82f62798178f617888c9d6a39150c76122c432ae9ea73ce28e79"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.7/node-v0.10.7-darwin-x64.tar.gz" ] && NODEHASH="879550ddc6fa8e56ed8ffed7c0bc27dee1d21867b07c398a44180fba34e6b029"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.7/node-v0.10.7-darwin-x86.tar.gz" ] && NODEHASH="cb7b2c196d60a61116f07c4de09ab9ffade9974d6ba46e9162fa9847c5b59df2"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.7/node-v0.10.7-linux-x64.tar.gz" ] && NODEHASH="9fdc924b9732ddf5fe278b7888a6c2c61074b15c71795f10e908b59387d3acd8"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.7/node-v0.10.7-linux-x86.tar.gz" ] && NODEHASH="075c333977dfd0b2d674489b769f16064ec5e8f49aa57d470ea7c2b542fbeec4"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.7/node-v0.10.7-sunos-x64.tar.gz" ] && NODEHASH="1c858bf96ebbd2c88e0a5e47c49fe0e4e679746e81b11ade9adc155a8d8466ff"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.7/node-v0.10.7-sunos-x86.tar.gz" ] && NODEHASH="a947ac0b8eeae290ab3e06d4fa69d9f6424ac793ea2786b9423f5b199c36c7fa"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.7/node-v0.10.7.tar.gz" ] && NODEHASH="22d1d211f5260dfa5b842cebdb04633f28df180843105ff3eb792ca35ed425e0"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.8/node-v0.10.8-darwin-x64.tar.gz" ] && NODEHASH="2253d67aec2e3525e0a649c7befb30a28d71425c1e902a86a5708be92ac6c28e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.8/node-v0.10.8-darwin-x86.tar.gz" ] && NODEHASH="b18264a1c3b781429cf9abca14e410b73eaeda2778c52cd350b968d56f3b07b8"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.8/node-v0.10.8-linux-x64.tar.gz" ] && NODEHASH="47903aa0bc81df9d4503a1fe55f2b2914dfa74ac0dd5be3d554dc4282695f427"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.8/node-v0.10.8-linux-x86.tar.gz" ] && NODEHASH="2b170232755cefc009011137a1bf3b2111810cb5ab974f62f574fca9489cd977"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.8/node-v0.10.8-sunos-x64.tar.gz" ] && NODEHASH="67ed50f8c1d80fd7601c7c2acd3b7fe9cbe667be3210075851b682f1f4b26d04"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.8/node-v0.10.8-sunos-x86.tar.gz" ] && NODEHASH="9d9b4b003df50e856614f104cf38681bd6c2f6c391e8210649b576ee0dc763cb"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.8/node-v0.10.8.tar.gz" ] && NODEHASH="edf6f766c8ccc7ef5b02a50c94567343eb1ffae479db93684ba89976e3f18354"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.9/node-v0.10.9-darwin-x64.tar.gz" ] && NODEHASH="3b0cbd1ac4b7e514ccd68fcd3089ccb1f13606fb41dbd2a776693bb984623bbd"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.9/node-v0.10.9-darwin-x86.tar.gz" ] && NODEHASH="36b935fc789754d48afbb8007acb54fca023c21d6d710aee74838dfa028ed3f5"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.9/node-v0.10.9-linux-x64.tar.gz" ] && NODEHASH="27159f584e108dbe5a9a884a98200a413203d339bf8596a3dbcaff9577fe1b1c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.9/node-v0.10.9-linux-x86.tar.gz" ] && NODEHASH="e2f27fe62888f0a03daf6168aeda86c45f1deef736e31dd45dcbd0b74b1934ef"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.9/node-v0.10.9-sunos-x64.tar.gz" ] && NODEHASH="35cefbee05e13ce6fc78641c0cf5437eee07df844af0d6c0ae56db3e51f907fe"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.9/node-v0.10.9-sunos-x86.tar.gz" ] && NODEHASH="a802e9614349c2b33c0b713dcace56b6aa747107669a3899f97be9e542ffab1c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.10.9/node-v0.10.9.tar.gz" ] && NODEHASH="25fb276ac6765ebb19f44d3e3775ed1c0275f874c896755d0d619226caee9c30"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.0/node-v0.11.0-darwin-x64.tar.gz" ] && NODEHASH="734693d7a388dcc46d5f6e3c3146fecde9b6de0c4afb9d4389c066be655095fc"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.0/node-v0.11.0-darwin-x86.tar.gz" ] && NODEHASH="497dc2d7d594ceff9c28cef53527b7f6337600ee9f1df2151847b42c95ab48d1"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.0/node-v0.11.0-linux-x64.tar.gz" ] && NODEHASH="ab86c554ae27e3938b588083488ae93531a5fba2428bdbcd0fb07a687f514c94"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.0/node-v0.11.0-linux-x86.tar.gz" ] && NODEHASH="f39996cdda3f2ea5d3f55a4fa518cd7944e6f4bd9904fcc081cddf17bc868efe"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.0/node-v0.11.0-sunos-x64.tar.gz" ] && NODEHASH="e6756250650d286da47e9113b2e8a767dcb5fe5f2544fedda36eeb4fbc60a0e9"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.0/node-v0.11.0-sunos-x86.tar.gz" ] && NODEHASH="eddc6728d6f7739b213e23462c7f62f35059d5145713f81858322c890208457f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.0/node-v0.11.0.tar.gz" ] && NODEHASH="a1887957fd6f0091379d1317f1daeb791bbf4301e58d693d17ad1d0fdbfa7898"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.1/node-v0.11.1-darwin-x64.tar.gz" ] && NODEHASH="bfa6644b345dedc26ed8a72f190ca825ce39f4332a423435a5eef67e078232be"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.1/node-v0.11.1-darwin-x86.tar.gz" ] && NODEHASH="96216adbeb90950b945fde0793b1b8cc80d35ae2a283a9ea8b93bce33e356c5c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.1/node-v0.11.1-linux-x64.tar.gz" ] && NODEHASH="c6c977e8b828114002f0f9f3cdc9a37370da41ac856ce107190f00ea0065d0d6"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.1/node-v0.11.1-linux-x86.tar.gz" ] && NODEHASH="bdfd3455de9b4fba4a5540971a4ec25bdfc8815f542f52b9ada72f8beac72c88"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.1/node-v0.11.1-sunos-x64.tar.gz" ] && NODEHASH="22453736b491481eb270ec273e1a938ce73534e5f7a9979c831291865822d726"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.1/node-v0.11.1-sunos-x86.tar.gz" ] && NODEHASH="7d566f108e1b1115aedec5abc988720b75f23f4abee9b638a8c4f8074077bd2e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.1/node-v0.11.1.tar.gz" ] && NODEHASH="1042853f6f288185e9dda60eaf57de50768aec5d32df7c7f462b713c56bd096f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.10/node-v0.11.10-darwin-x64.tar.gz" ] && NODEHASH="3beb0693c395272d07f1fbf7fdcdeeaf018f9571e813ea085422f81a03188eb7"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.10/node-v0.11.10-darwin-x86.tar.gz" ] && NODEHASH="b256f095431a6258505c21cec9bd2f3a3524f6c3881db10ee8e1c1b6d0ff263d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.10/node-v0.11.10-linux-x64.tar.gz" ] && NODEHASH="5397e1e79c3052b7155deb73525761e3a97d5fcb0868d1e269efb25d7ec0c127"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.10/node-v0.11.10-linux-x86.tar.gz" ] && NODEHASH="96fb59f7f206b445dff211d02f2b82e51c77b4eef73c2c0eb1deaa2c286af5b0"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.10/node-v0.11.10-sunos-x64.tar.gz" ] && NODEHASH="eb921e47fcb0deddb9f2c75f8d087ec16de11c3cd3b50ecd6ec68a7ac08c7873"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.10/node-v0.11.10-sunos-x86.tar.gz" ] && NODEHASH="bf4bef098f7fddec5692f9c7aefdab916c2ff8a5e6856423429dfb008b957b4c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.10/node-v0.11.10.tar.gz" ] && NODEHASH="6f7b5971c23049645bb955ad787714bf1e8ead14bab2d4e30da328f13fa86040"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.11/node-v0.11.11-darwin-x64.tar.gz" ] && NODEHASH="f10deff61ab2659fbcb92ebd20d90689008a4152fe1b989e2b3b49cc468f534b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.11/node-v0.11.11-darwin-x86.tar.gz" ] && NODEHASH="d3e11f71abf5f1aa695f98faf9660a4d482db3ae10b07de16840bd4febff83af"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.11/node-v0.11.11-linux-x64.tar.gz" ] && NODEHASH="1cf91a851ecb3cb5c4dbd9c14ab59eb53b77ab0cda714b564190746fed67534c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.11/node-v0.11.11-linux-x86.tar.gz" ] && NODEHASH="cc65c5e3f666eef2c079680062a7da59fe280a9480402f06dac01a696f064226"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.11/node-v0.11.11-sunos-x64.tar.gz" ] && NODEHASH="cc0e76fe45755885c606939ae63be113ddc698bad65820cd19044e128b6fb270"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.11/node-v0.11.11-sunos-x86.tar.gz" ] && NODEHASH="cd58120e6660931ad24eed0534f499edbf152265c64a83b629bd501768f4f579"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.11/node-v0.11.11.tar.gz" ] && NODEHASH="7098763353011a92bca25192c0ed4a7cae5a115805223bcc6d5a81e4d20dc87a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.12/node-v0.11.12-darwin-x64.tar.gz" ] && NODEHASH="2265e5a755e442fbd923119df673816bdf4f2a2931f93821e6c1075fdc678451"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.12/node-v0.11.12-darwin-x86.tar.gz" ] && NODEHASH="4f17a0c42957ba70700e2e5f9c87454a68a99f8a9ea5d94cbf87752cfa17e4c2"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.12/node-v0.11.12-linux-x64.tar.gz" ] && NODEHASH="d5369c5608482bcbfcb7a8cd18a43b493a878020c6e5dc241cf55473dafa374a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.12/node-v0.11.12-linux-x86.tar.gz" ] && NODEHASH="6e578b95fb5d9a7fdc3fab13cd1fc1222b8e50aa8c861d2603b2a3ab7ca0080f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.12/node-v0.11.12-sunos-x64.tar.gz" ] && NODEHASH="7cc56ba26dd0429deb2e72632629b051162d355cfae0bf6732c620595f684ac8"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.12/node-v0.11.12-sunos-x86.tar.gz" ] && NODEHASH="58cbe6416d4dba4c76d68d088d1741133242fb9825e5d3d7ba39e20d334312ed"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.12/node-v0.11.12.tar.gz" ] && NODEHASH="c40968981d9f5f6fbc4abb836557acda74ecb8f8a1e9a30e84ebd2529a8c1b6a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.13/node-v0.11.13-darwin-x64.tar.gz" ] && NODEHASH="d642d825f8815b87a7edfc7c6057853671bda6a53711f1c6fa532735cabdaf07"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.13/node-v0.11.13-darwin-x86.tar.gz" ] && NODEHASH="cc2d724df3370153266ff42650db2cdc8e47915bb7e0dd72321ec3973b1bcf62"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.13/node-v0.11.13-linux-x64.tar.gz" ] && NODEHASH="4609ed7780cb4aaab6703cdd015f593893e3acc09e432465ec0c9cc178c26655"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.13/node-v0.11.13-linux-x86.tar.gz" ] && NODEHASH="1d958416c2a9d1722823e5bef8e04800a27d1ddb9847d4bb46039903c5250fae"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.13/node-v0.11.13-sunos-x64.tar.gz" ] && NODEHASH="71186ff69b914f005641742676e5f65b2c0853fdb09f2c5f8cd6696e85e76763"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.13/node-v0.11.13-sunos-x86.tar.gz" ] && NODEHASH="2e9c0eacd1cb699360b3abd01272314dfb6d7e4ddfd3897c10822f6ac336220c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.13/node-v0.11.13.tar.gz" ] && NODEHASH="15d6e90c16adf907c0401cd5a77841b5264e90dfdaa1051d75184aa587fc8298"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.14/node-v0.11.14-darwin-x64.tar.gz" ] && NODEHASH="074669d2f3d8419496076c55c2743389538996a90e87277ea5bf032f885877ad"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.14/node-v0.11.14-darwin-x86.tar.gz" ] && NODEHASH="5d9bb59a66fd36dcae9bb44c83165dcbc09e3a5d4398e684526b8eab3659102b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.14/node-v0.11.14-linux-x64.tar.gz" ] && NODEHASH="3ae6cb227815e7c794215244cecd90a2d3fcf97ba7a30f09accba861bb6057f8"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.14/node-v0.11.14-linux-x86.tar.gz" ] && NODEHASH="f855db9b633e7805ebdce5b1c98488d44a0ca1a105d7eb09b72e695a2085e959"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.14/node-v0.11.14-sunos-x64.tar.gz" ] && NODEHASH="96cbb3f5b8a19cde8fc9855baf1e2c5664cfdaf9062675661c5920f51eb85319"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.14/node-v0.11.14-sunos-x86.tar.gz" ] && NODEHASH="eae7646ae5d94f055698a271daa5092dcc02fe284de01cc9e0688190d6e06a83"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.14/node-v0.11.14.tar.gz" ] && NODEHASH="ce08b0a2769bcc135ca25639c9d411a038e93e0f5f5a83000ecde9b763c4dd83"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.15/node-v0.11.15-darwin-x64.tar.gz" ] && NODEHASH="e52e79416922eccdcc2451c9e041bac70c0b8aa9e90dd1ce0659602bc82c8ac1"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.15/node-v0.11.15-darwin-x86.tar.gz" ] && NODEHASH="b745bd2177149deaaf8de834420946a410d414e6b0e59d508bf264fbe8941a33"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.15/node-v0.11.15-linux-x64.tar.gz" ] && NODEHASH="940bb9ab99be8be2d9b954fb152e239f2076d28805378e30e781ddcedad382eb"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.15/node-v0.11.15-linux-x86.tar.gz" ] && NODEHASH="30c0f5ec8cfcc94e4a7cd69b4eb76dd7d97125848fe2fe2c316df690e016c168"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.15/node-v0.11.15-sunos-x64.tar.gz" ] && NODEHASH="08ee809bdb1800a4fd629b795038040222ee2233cdf1adb1297a3abdd18d2584"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.15/node-v0.11.15-sunos-x86.tar.gz" ] && NODEHASH="6b4969200aeb10532d4d23c20af4b10eb20b08ee1aa37cc486172b7237dfa0a6"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.15/node-v0.11.15.tar.gz" ] && NODEHASH="e613d274baa4c99a0518038192491433f7877493a66d8505af263f6310991d01"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.16/node-v0.11.16-darwin-x64.tar.gz" ] && NODEHASH="95a05f0b8d6f950121e983b1c17ef676356dbe6f8ea51453c037df200ed09ba0"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.16/node-v0.11.16-darwin-x86.tar.gz" ] && NODEHASH="9b79b863f26531c3a38da6a5093aa4b32d71838dfe2f9db59180e0d8f049706a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.16/node-v0.11.16-linux-x64.tar.gz" ] && NODEHASH="a1bdc19c779d13b772ac22feead14f592c637ce866d86a59ef225a3273dd7c33"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.16/node-v0.11.16-linux-x86.tar.gz" ] && NODEHASH="7380aabb7e12a82ed24e47ad18afb1f21dfb54cbf25892b570116fe774e3f45f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.16/node-v0.11.16-sunos-x64.tar.gz" ] && NODEHASH="ba806e941aa532ace89a0868374dec5ec97d6abd1d3d9d97fd686b419aeb6d05"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.16/node-v0.11.16-sunos-x86.tar.gz" ] && NODEHASH="7a15668536ba979ebcbda23beee3e7350544d4b5e15a96afdc928dbb6d423d5a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.16/node-v0.11.16.tar.gz" ] && NODEHASH="f0d141faa1f7da3aff53e9615d76040d29c0650542be3b09ee80aca2f2cc61f6"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.2/node-v0.11.2-darwin-x64.tar.gz" ] && NODEHASH="3221e877fa2c30a93df5fd1510913b0f85a18e09cd54fec07b363ecace84a429"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.2/node-v0.11.2-darwin-x86.tar.gz" ] && NODEHASH="1df16df8a12a504d6f88025c78ae16ff29d419a6c33fae386b0b88fd387f0a0f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.2/node-v0.11.2-linux-x64.tar.gz" ] && NODEHASH="44989b65a7f784cee48435234b12a253bf8e602651ffcdf0c500f7912798faa2"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.2/node-v0.11.2-linux-x86.tar.gz" ] && NODEHASH="d6d71c2149c6fe9113179700ab94514a03f6b3a3d77037a10ef7b35bc89e1f76"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.2/node-v0.11.2-sunos-x64.tar.gz" ] && NODEHASH="f315d1c5ffa8152eeaafc602d0265c9fde19c338c159e1166e981a0bf34d39eb"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.2/node-v0.11.2-sunos-x86.tar.gz" ] && NODEHASH="125c52610c41638c740a71d564132e56b986d208dff04a72b7ebb97f3347df04"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.2/node-v0.11.2.tar.gz" ] && NODEHASH="d115f01fea0b2c5a4c4ca489d0cc8cec70300f0212f08905d881ac55f642554a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.3/node-v0.11.3-darwin-x64.tar.gz" ] && NODEHASH="7f441d7abde7656f01ebb268a4d61c07ecd63934f099f30a386ea7718cf1486b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.3/node-v0.11.3-darwin-x86.tar.gz" ] && NODEHASH="d6b1ae947d8b4006c5a2ed7519a4e423a2dfd53b64c4d4f853172ba42bd106f8"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.3/node-v0.11.3-linux-x64.tar.gz" ] && NODEHASH="50535f04bf36da6efa52ae57d29354f6e0ffd8dc773c08ec655f44314e6f47d3"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.3/node-v0.11.3-linux-x86.tar.gz" ] && NODEHASH="ee76d9715e05134edef59de7f12bd478b3524205222cdea5578739dd692f0f35"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.3/node-v0.11.3-sunos-x64.tar.gz" ] && NODEHASH="94e68581c673993901381c212cf255ad3f945a1b5a98d7d122ad40ebd9a9c23d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.3/node-v0.11.3-sunos-x86.tar.gz" ] && NODEHASH="d94c49bc39e0a9c8c184d5326c7e059ead5e6be89555a5cabcdbad43f5e919c0"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.3/node-v0.11.3.tar.gz" ] && NODEHASH="aa3189c4e42d3e5dae86c132b28dbe04163d53a480949ea9f1985ddfeaf39955"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.4/./node-v0.11.4.tar.gz" ] && NODEHASH="81f36aafa4a31fa59e0301358699d82766ea7ba178be810ce00444a7fc10db47"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.4/./node-v0.11.4-linux-x64.tar.gz" ] && NODEHASH="9937c02147c01fa7926ffb1cc231bde536a02b6cbed243f88d76de1d445bf97d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.4/./node-v0.11.4-linux-x86.tar.gz" ] && NODEHASH="5795f2aeb00b1d99e324850fe9ff2f0cea91ca33da4a4a12a0abc8f3c7bd352d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.4/./node-v0.11.4-darwin-x86.tar.gz" ] && NODEHASH="7d143d9db01a34382e02997e168dd7548af810ab34ddae9ed451a482a3ec2263"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.4/./node-v0.11.4-darwin-x64.tar.gz" ] && NODEHASH="89eceb0bd58e82d9f72306346c2fbe2036f653ff01120ea867f90f1f85a6eeff"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.4/./node-v0.11.4-sunos-x64.tar.gz" ] && NODEHASH="b7ace7b597704d3b9cad1f581734c1afe51db2aca0da80739b06c557c26fce79"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.4/./node-v0.11.4-sunos-x86.tar.gz" ] && NODEHASH="9041a18c7eecea05fd8ce331f9f8294af1725a25dd705b0410f81f948d73757c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.5/node-v0.11.5-darwin-x64.tar.gz" ] && NODEHASH="62034296248db29751c5748f0403c07e73a0ae3b0ef2c3384956aac70a12cd21"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.5/node-v0.11.5-darwin-x86.tar.gz" ] && NODEHASH="ee05eb433e3215e8918abaff158a02d52e8bb3a018ddbc6e7b0f4335531f5a22"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.5/node-v0.11.5-linux-x64.tar.gz" ] && NODEHASH="c45cfeedbe7149e315f58243ec05dc6575ca2fdd16d4cf0f76853a178eaebf41"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.5/node-v0.11.5-linux-x86.tar.gz" ] && NODEHASH="4534dd00757b96d3e281aa7487bc7e705583fe1770b9589704a89cf1266251ab"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.5/node-v0.11.5-sunos-x64.tar.gz" ] && NODEHASH="efe9e4d9e84805f2ce485cebcfac4ee683066ae80f53c4a050e41845a3bf7de5"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.5/node-v0.11.5-sunos-x86.tar.gz" ] && NODEHASH="7b1a06208026da14b394dafb607408e116a93a1d2dd0d2923870cdc24007fdef"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.5/node-v0.11.5.tar.gz" ] && NODEHASH="72b89f9146a2dd57e1712f1fb822f62bdf00b2d5482689510dc2e4d19ae6559e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.6/./node-v0.11.6-darwin-x64.tar.gz" ] && NODEHASH="71dfb9ae0cb7b43bf14ede3a78885bdcd7e376786439159e22fbac98ff7eb998"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.6/./node-v0.11.6-darwin-x86.tar.gz" ] && NODEHASH="4c609184a7dcf095a39f50b613edbe58c7f0c4df5bff566437739de8ffe0f051"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.6/./node-v0.11.6-linux-x64.tar.gz" ] && NODEHASH="4d0b09d88466933439cd5b87c4fbf998732cb6705314edbc9ed3901c2cc24669"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.6/./node-v0.11.6-linux-x86.tar.gz" ] && NODEHASH="fe818b6356e43b4d5b83247381dce415541639d33129d0ea4a4397ac43a89b0b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.6/./node-v0.11.6-sunos-x64.tar.gz" ] && NODEHASH="a20a411403d693d174860140add6c20499ccefd3c7b41e78a4e7fbf349f03f61"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.6/./node-v0.11.6-sunos-x86.tar.gz" ] && NODEHASH="21ab274fe11405dd2997763e9b4fd065604931150c47c039f22de73ed59f2902"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.6/./node-v0.11.6.tar.gz" ] && NODEHASH="35552aec60077270306c73507effeb4b7d9ef02f03f45681442c0d4e1951e75d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.7/node-v0.11.7-darwin-x64.tar.gz" ] && NODEHASH="1679e0fc69202638186392be31ffca6f39d666f275875779f1f76f4dc2c10ece"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.7/node-v0.11.7-darwin-x86.tar.gz" ] && NODEHASH="d7c9c668ae17c650c368613f8c6f1b321ac1f3de87cf36342bcd6127cf74088f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.7/node-v0.11.7-linux-x64.tar.gz" ] && NODEHASH="67253735b86fdba070ef8af6b328e13c9ee4de38f269c60696ff498449646929"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.7/node-v0.11.7-linux-x86.tar.gz" ] && NODEHASH="bf819ad5d7bf179634bdc391db064d32aa6759dec76bc14b4db4b501039aa26b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.7/node-v0.11.7-sunos-x64.tar.gz" ] && NODEHASH="4d4f10266e05b3d8081d3674d0d35f75703037bda85d4a10cc721d31b72ed77e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.7/node-v0.11.7-sunos-x86.tar.gz" ] && NODEHASH="c1ecf77d9a9f15cd16667a76cea2e950781853e7d4ea0466481bdec82cdd602a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.7/node-v0.11.7.tar.gz" ] && NODEHASH="d915345639e340405b01f259971f386aafb5a10544b162826514cf56ddd371fe"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.8/node-v0.11.8-darwin-x64.tar.gz" ] && NODEHASH="e8d64d423b97aa50d42c917759c7ccd3cdfb3e6b257cdd08a7030eb2347e5a50"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.8/node-v0.11.8-darwin-x86.tar.gz" ] && NODEHASH="2137c56ae3e665508cf9519fcbf6d2b56b8b4272e0fceb6ec060ca528a997ad0"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.8/node-v0.11.8-linux-x64.tar.gz" ] && NODEHASH="5ddc30cb411201cbfde7df9db8a071bd61fac3e3c7d2156b6bef0b10475c934e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.8/node-v0.11.8-linux-x86.tar.gz" ] && NODEHASH="6f4ade3a429878621a737c736b00df6abae261c0f8fddfa60c62491ff68b1a1f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.8/node-v0.11.8-sunos-x64.tar.gz" ] && NODEHASH="0599412586981586bddc2686f3f1a41c48862ae8dd770f810cec83d7453dc8be"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.8/node-v0.11.8-sunos-x86.tar.gz" ] && NODEHASH="66e20972dcd44efbfe51404b7a9e5b67e03f71b439fc966bd4e6d2524ce9c886"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.8/node-v0.11.8.tar.gz" ] && NODEHASH="87c809dea764d5d66f925626fba403fb2fb0c0ccfad408bf79fdb62dc246d65b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.9/node-v0.11.9-darwin-x64.tar.gz" ] && NODEHASH="cc523a93237645fb44d24cda3bfeb2b2d7ebd4d2265c0a175f264e3333706c45"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.9/node-v0.11.9-darwin-x86.tar.gz" ] && NODEHASH="4bea8fbde1baa8b1ae2db187488df44c3607b5f94c84c2d0c301c3e7666cf1b8"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.9/node-v0.11.9-linux-x64.tar.gz" ] && NODEHASH="89a5013f326e67b73ebef638e765b286831a5d72363ebfdfc75b57b0818c178c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.9/node-v0.11.9-linux-x86.tar.gz" ] && NODEHASH="b440d5d668efd6e4516abb5e43f2d7a12393c35c3d64cec7b24c4fff29dcd811"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.9/node-v0.11.9-sunos-x64.tar.gz" ] && NODEHASH="b37fa67fe332c32759be413dbad9e4f8231267ac3afa7afc1d74388d9859dc73"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.9/node-v0.11.9-sunos-x86.tar.gz" ] && NODEHASH="158bafae549cbaac88bd85bc6e00104f3fdbc84ac2562018050d9131b44d7e4f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.11.9/node-v0.11.9.tar.gz" ] && NODEHASH="cfcab9735a7e04a67671a96a8b0b7e71954c60c586ced5e3fe37d5c1a235b444"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.0/node-v0.12.0-darwin-x64.tar.gz" ] && NODEHASH="880c530d11168c796c3b3e4cec1b2fa37af22e9c559faf395854b94bc7bf0cd2"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.0/node-v0.12.0-darwin-x86.tar.gz" ] && NODEHASH="2d75cb34249884c7e36dce4d53809e020a6a215ceb1b6827d9e18ed1ded9293a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.0/node-v0.12.0-linux-x64.tar.gz" ] && NODEHASH="3bdb7267ca7ee24ac59c54ae146741f70a6ae3a8a8afd42d06204647fe9d4206"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.0/node-v0.12.0-linux-x86.tar.gz" ] && NODEHASH="d4130512228439bf9115b7057fe145b095c1e49fa8e62c8d3e192b3dd3fe821b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.0/node-v0.12.0-sunos-x64.tar.gz" ] && NODEHASH="b775ba850a950ad405bc92f57325aae5b3a692f241bb1f01a94b1291c9b43246"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.0/node-v0.12.0-sunos-x86.tar.gz" ] && NODEHASH="c910021a6b756ffe89f3a91a51f5f6ed6fcebae51fd12a5dc2070b06bb400244"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.0/node-v0.12.0.tar.gz" ] && NODEHASH="9700e23af4e9b3643af48cef5f2ad20a1331ff531a12154eef2bfb0bb1682e32"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.1/node-v0.12.1-darwin-x64.tar.gz" ] && NODEHASH="1724e75c32ac58452e0ca3eae25120ab67e3b1383da7c10278b33c0f2d3385bf"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.1/node-v0.12.1-darwin-x86.tar.gz" ] && NODEHASH="c455d68f64e4495ceb97f57399ee8cb28471d1ab7096fce3be5899754680eb8c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.1/node-v0.12.1-linux-x64.tar.gz" ] && NODEHASH="270d478d0ffb06063f01eab932f672b788f6ecf3c117075ac8b87c0c17e0c9de"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.1/node-v0.12.1-linux-x86.tar.gz" ] && NODEHASH="c7bbc47c3517c162b73437157e7f99cf2ae3189dae264e74f79bfc25143f304f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.1/node-v0.12.1-sunos-x64.tar.gz" ] && NODEHASH="d93ddc3f2d17da8320cfff40ef39edb4a19e0877f31254e1bf66988bae0a0616"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.1/node-v0.12.1-sunos-x86.tar.gz" ] && NODEHASH="b7cd96f4791e95b3698e1a6e0461acd040f14138e9fde1e64a76b279da5ea9bb"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.1/node-v0.12.1.tar.gz" ] && NODEHASH="30693376519c9736bcb22d44513252aee1d9463d78ac6c744ecb6d13fd91d680"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.10/node-v0.12.10-darwin-x64.tar.gz" ] && NODEHASH="c95df35ca1ed7b4b0ded815c1d49f36defcb1fdb882f6a8ef6106a07e3f2ffef"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.10/node-v0.12.10-darwin-x86.tar.gz" ] && NODEHASH="d4abd2b778c9d803676ad6121e6fdbc625b9ea73e845b0ecd761c162e86150ca"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.10/node-v0.12.10-headers.tar.gz" ] && NODEHASH="c8e99589d96f9ad598c2d602e3dcac4bb0147a709da4da89a1f2b7f667f4b415"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.10/node-v0.12.10-linux-x64.tar.gz" ] && NODEHASH="8fb4d6ed8934f0b0c92c26878511e1d340b068ee966c131ba0fccc1199f4349d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.10/node-v0.12.10-linux-x86.tar.gz" ] && NODEHASH="6f3ea401d2f488afb6adc57a3056df8658c1c9a57a368637cbc215ed3133c3b7"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.10/node-v0.12.10-sunos-x64.tar.gz" ] && NODEHASH="d67f17540c711eb150b8a389af1b4e6ecdcab66a1648b7ce925af98ab52b2698"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.10/node-v0.12.10-sunos-x86.tar.gz" ] && NODEHASH="beca24cc3615c5b1858817d121bd91eecdc3af5b98ed0c4c171e1ef60afac049"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.10/node-v0.12.10.tar.gz" ] && NODEHASH="edbd3710512ec7518a3de4cabf9bfee6d12f278eef2e4b53422c7b063f6b976d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.11/node-v0.12.11-darwin-x64.tar.gz" ] && NODEHASH="68b949f6c308eb1ad28e96926fca68e3ac0dfb7ec997945f4e26db99fe3ad711"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.11/node-v0.12.11-darwin-x86.tar.gz" ] && NODEHASH="057f701678c0fe0c596eaa517e8b7ba771bdca89d99a7ace8df8e1064784727b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.11/node-v0.12.11-headers.tar.gz" ] && NODEHASH="4d3985bddfbdc867eb1f226e74c3a507db01410f95aaa566da18ada9bfd51d02"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.11/node-v0.12.11-linux-x64.tar.gz" ] && NODEHASH="d98b76b7721a60471801e07e1f90af4fd479e8e42a632d419ede0a7b3c603cc0"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.11/node-v0.12.11-linux-x86.tar.gz" ] && NODEHASH="ecf627bf67bfd231e549869c6e2963acd5551d15d5cf1116b0649d3d14ddde19"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.11/node-v0.12.11-sunos-x64.tar.gz" ] && NODEHASH="6a68d6ad04c9b73ed72e88e39002bbaa95ffc423c6cf47e3c3da5edd0abbc701"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.11/node-v0.12.11-sunos-x86.tar.gz" ] && NODEHASH="9e61254d7437c2498817225c62dc6cfc065bd3b2404a213d8d305419bca07a6c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.11/node-v0.12.11.tar.gz" ] && NODEHASH="e49049d82f2a11fa164549d907d4739fe1293d53c07f48bd70e1df237b238a68"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.12/node-v0.12.12-darwin-x64.tar.gz" ] && NODEHASH="b9b7b7c8f9c35fd6492bd5e60ec485ca8bf60f678651c7b6c46a3d8cf561e13c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.12/node-v0.12.12-darwin-x86.tar.gz" ] && NODEHASH="a2faf1180ffa50dbce0ca0a63b4285982b9fd87623dcda400c8f7e2be3abaa0a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.12/node-v0.12.12-headers.tar.gz" ] && NODEHASH="47d6bc7eb3e50fb7175f111882dd181088f252e7c6fc3b36c3f70c4d0e91df28"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.12/node-v0.12.12-linux-x64.tar.gz" ] && NODEHASH="a3d51cc26060fe46f9ebe69c750b20fe1f6f27a936db6046a73b8a9715bf3559"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.12/node-v0.12.12-linux-x86.tar.gz" ] && NODEHASH="5c69b9f4d51fe7e54afe3c1a83263a03270ad5b09070c8d0769c4e1942c8fec9"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.12/node-v0.12.12-sunos-x64.tar.gz" ] && NODEHASH="ef361aec4a2c1e3789662e7605928717af76b4d0ed360d4facb75a606ac0a2de"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.12/node-v0.12.12-sunos-x86.tar.gz" ] && NODEHASH="f6d0fa1fe8cfc197fbe86c8f7517b42d610efb341218fad1b24bc7f08cc4433f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.12/node-v0.12.12.tar.gz" ] && NODEHASH="61e4c176fd882498778b1a3907a5fe5c9e95e6cc8438b0d053d953aed3620273"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.13/node-v0.12.13-darwin-x64.tar.gz" ] && NODEHASH="e842a8eb8b7658d177675de9d054de4d7fb7d6c77edfcda7d83adb95b029bf3d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.13/node-v0.12.13-darwin-x86.tar.gz" ] && NODEHASH="8ecdcd3b29ef0d3264bb48c0834bc024016d19a8f5c040fa6b51328191f39a60"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.13/node-v0.12.13-headers.tar.gz" ] && NODEHASH="bf6a8de61b08935e91ac345e2dcd54c95d1def963b5db9a8bb33cf75963e13e6"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.13/node-v0.12.13-linux-x64.tar.gz" ] && NODEHASH="3e8b6ee32fc9a726bfe6f3961bcccf3d2b6d0ddd68326abb4434039f16e10f09"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.13/node-v0.12.13-linux-x86.tar.gz" ] && NODEHASH="8300b9cbe363bf2a5fd987e7d05a88cb60842ad5b0ea6cbf3f4342eef547feed"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.13/node-v0.12.13-sunos-x64.tar.gz" ] && NODEHASH="9b40e2b657e560901c6cccf3c93d01a5055cb4d011ccfefe1b977dae7935ea42"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.13/node-v0.12.13-sunos-x86.tar.gz" ] && NODEHASH="83f862b0383ba6c9a15f32a043de48288b087c0f368117eac36d66779491a910"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.13/node-v0.12.13.tar.gz" ] && NODEHASH="0a972ed6442cb526aa7aa1bcb10aa536b65bd90ab4956b5a1aa51b4b7bb071bd"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.14/node-v0.12.14-darwin-x64.tar.gz" ] && NODEHASH="b4de40b8fe9cf748b36b9d7764c06f1fc955c73e3547c4eaab3d90a818663e1c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.14/node-v0.12.14-darwin-x86.tar.gz" ] && NODEHASH="2ac52efd90931ea04dd01947e19c6320e519967a813f7274eb699bf468fe226d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.14/node-v0.12.14-headers.tar.gz" ] && NODEHASH="75f927b98a955207f261d03de5966b167e893da80e641695e6b9d6da44312452"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.14/node-v0.12.14-linux-x64.tar.gz" ] && NODEHASH="0f1f20f6989d32b4b67835f527ae3bf165c1c4a6a7dc3961d489288817956bae"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.14/node-v0.12.14-linux-x86.tar.gz" ] && NODEHASH="212a3a5d1cdd34b71476dd615d95da9209b8a5e490dbe6141926e15f1ccfe386"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.14/node-v0.12.14-sunos-x64.tar.gz" ] && NODEHASH="906a44e9f6024c3f9af05a8aac5ba10c25d84bf56b9fb08c5fc1c26c5a8b9d27"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.14/node-v0.12.14-sunos-x86.tar.gz" ] && NODEHASH="f042bee409d4da3114571dfff496a954ea6cd614e7fb78fd3d9b50d799396757"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.14/node-v0.12.14.tar.gz" ] && NODEHASH="0a55e57cbd3ffa67525c0d93ac7076d3b2ac70887b11c5c97be3e1953cb50b1d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.15/node-v0.12.15-darwin-x64.tar.gz" ] && NODEHASH="125551969069284099d402d85e36e1637d9dd6f19105261f87b8f8b2020b3433"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.15/node-v0.12.15-darwin-x86.tar.gz" ] && NODEHASH="2026bdd953a783d78e2036f67f95590d3334f06541371bbbc87d3ec4991e60e9"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.15/node-v0.12.15-headers.tar.gz" ] && NODEHASH="bb5d26514a2c217eb5895efe4884d533e10f31d31949947c67857da515e03422"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.15/node-v0.12.15-linux-x64.tar.gz" ] && NODEHASH="ab2dc52174552e3959f15a438918b32b59e49409e5640f2acb1a3b9c85cf2a95"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.15/node-v0.12.15-linux-x86.tar.gz" ] && NODEHASH="0ea7a2860311596a13e202fef507ec2a55db9ddbb1d9cb27fa19b7e61d56ee20"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.15/node-v0.12.15-sunos-x64.tar.gz" ] && NODEHASH="6114f3282e6650210cdfedf7ddf7d5dc62ad80b49475b4e2938afce790338d04"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.15/node-v0.12.15-sunos-x86.tar.gz" ] && NODEHASH="29b2fc6d75847ae879bda550f1308bc34abb3ce094030ace93b44a3caba7b2e1"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.15/node-v0.12.15.tar.gz" ] && NODEHASH="c446d40d9b0eb876dab9a87c3dbd1935dc85155d2ad991b36ddd6c5ca3825a5e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.16/node-v0.12.16-darwin-x64.tar.gz" ] && NODEHASH="c602cbced0d6a2fb9d97f25a72833cf564b35fdf742f8627a93b6cdb5132ea95"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.16/node-v0.12.16-darwin-x86.tar.gz" ] && NODEHASH="2cb0b6d33deab16aa261eeadec9c0da40d23e391b626a096f36b75ded62e471f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.16/node-v0.12.16-headers.tar.gz" ] && NODEHASH="cb30a68fc3a03b8bb98d8b17e389537610c7cf663e7dd113868c13c99fcffee6"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.16/node-v0.12.16-linux-x64.tar.gz" ] && NODEHASH="d9e1cd239844f2a5641e02e48b3c7955e3e73ff3c3d20629c24b561f08ab8219"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.16/node-v0.12.16-linux-x86.tar.gz" ] && NODEHASH="ea95fb22f07469d79fd0347e4f768267be50889fd6de8854bf2f834c346c93a4"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.16/node-v0.12.16-sunos-x64.tar.gz" ] && NODEHASH="817671fe6303a1bf2ea47876e2fbbe271ec3ba721b6e688836117a70cc2631c3"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.16/node-v0.12.16-sunos-x86.tar.gz" ] && NODEHASH="452ade498e7b4d0050cc8c60bb6aa2dcb16562026441ad78063f473435691984"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.16/node-v0.12.16.tar.gz" ] && NODEHASH="312c0b74b0815f0514de9bf00667850d4f6ce184126f02f3d8dbf40fd48235eb"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.17/node-v0.12.17-darwin-x64.tar.gz" ] && NODEHASH="4134958fecbc56e9a25374c80e4770ec51e5acfb5eefb67c89ee1af8d80c83da"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.17/node-v0.12.17-darwin-x86.tar.gz" ] && NODEHASH="5a24a5ca37ee53d45e49dafcd286b6b26e9f5e2805bb768b287bab5f289573ba"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.17/node-v0.12.17-headers.tar.gz" ] && NODEHASH="a93ceb4f049a762975c103fefc152fab486b99419fa60ea865841390d8c326ee"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.17/node-v0.12.17-linux-x64.tar.gz" ] && NODEHASH="afeec47fc543e24a1e596d05e9bc8e019c3bdf51d45f0ddeac6aeb04f15eaece"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.17/node-v0.12.17-linux-x86.tar.gz" ] && NODEHASH="5069c77361a0952d426620d9f35f87f0066ce81df4cb57612a3db145ab10c93e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.17/node-v0.12.17-sunos-x64.tar.gz" ] && NODEHASH="0354c8c1e636886ff1759e3106ac150350c9c2e252ac01dee7801e471ee03a96"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.17/node-v0.12.17-sunos-x86.tar.gz" ] && NODEHASH="f17d21f5b04a1a1b21a6bed7fbcda9bc3f0d580fcc2ca8049bba0139eea2bd97"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.17/node-v0.12.17.tar.gz" ] && NODEHASH="ddea5e253812f167608fbdda6ff29ffc30dc6f669cf2f6e4249938d28b9cd44d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.2/node-v0.12.2-darwin-x64.tar.gz" ] && NODEHASH="7d382f8988ea441e2918d9d935003177affd13d4d160020df6b3d8ee7d65ec2d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.2/node-v0.12.2-darwin-x86.tar.gz" ] && NODEHASH="4d814c4b16ea75f12fa9e6511ea3367c84f99116f8bf25e87a9e839d2cf05cde"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.2/node-v0.12.2-linux-x64.tar.gz" ] && NODEHASH="4e1578efc2a2cc67651413a05ccc4c5d43f6b4329c599901c556f24d93cd0508"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.2/node-v0.12.2-linux-x86.tar.gz" ] && NODEHASH="fdf9b3d93748bc736759c6ce3c414007b33cc720735eff2324bc11e26f2f806a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.2/node-v0.12.2-sunos-x64.tar.gz" ] && NODEHASH="48112116f4d7ba15c47cf269d72df1f402efaccde1b0bce41d6176c6b653f1b3"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.2/node-v0.12.2-sunos-x86.tar.gz" ] && NODEHASH="86b07380c92fa4f2a82db7cd5089c63fe9b6235193c5c670736243aa015b9c8a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.2/node-v0.12.2.tar.gz" ] && NODEHASH="ac7e78ade93e633e7ed628532bb8e650caba0c9c33af33581957f3382e2a772d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.3/node-v0.12.3-darwin-x64.tar.gz" ] && NODEHASH="74ddd964bcc7d73bb7cf893174710bf6d0b9d8950e30cccb9a50827df04f1c59"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.3/node-v0.12.3-darwin-x86.tar.gz" ] && NODEHASH="8bfe24802ae4f56c9275a96da49101b37779fd2f92ed6a9acfb6318ebe80ffa0"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.3/node-v0.12.3-linux-x64.tar.gz" ] && NODEHASH="22478ba86906666a95010e4eb73763535211719a53da9139b95daeb5b6c170b8"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.3/node-v0.12.3-linux-x86.tar.gz" ] && NODEHASH="f3e9eb41edb23a4af2f311b93758a2a00bc54f61812edfc2bafbe80780891d3d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.3/node-v0.12.3-sunos-x64.tar.gz" ] && NODEHASH="376318f3e3c14543f4419d0e8c190cebbcb6052882980beeb9203d13548f6cc1"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.3/node-v0.12.3-sunos-x86.tar.gz" ] && NODEHASH="8d6e3400fdfa9fe0df7f441c5f51d25fb2a8975b9ae6c43560c3bf3a71d1497d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.3/node-v0.12.3.tar.gz" ] && NODEHASH="e65d83c6f2c874e28f65c5e192ac0acd2bbb52bfcf9d77e33442d6765a3eb9da"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.4/node-v0.12.4-darwin-x64.tar.gz" ] && NODEHASH="757d1a2e7d3deb03b40277e9f04ae276c370b53e6e5238d44e08846f1f3c85c2"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.4/node-v0.12.4-darwin-x86.tar.gz" ] && NODEHASH="a1b9c671ef3e16d6fe757ac16b48565f340027e217cf1bb0f461059e49394814"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.4/node-v0.12.4-linux-x64.tar.gz" ] && NODEHASH="9095c664c81d7ec90337efd0877e2af72bd055bf8f4f47056d2ac8ea909f561e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.4/node-v0.12.4-linux-x86.tar.gz" ] && NODEHASH="4c31e3652bbc422ec37daf277bb6f87c19d35d33933f3a259b3a9a87294558ba"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.4/node-v0.12.4-sunos-x64.tar.gz" ] && NODEHASH="308cb46091c613f8068a8aa30980ce4a1ac6e85ac14aa464728e83727a981f13"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.4/node-v0.12.4-sunos-x86.tar.gz" ] && NODEHASH="b9bd9a590f73d8c82704676794eb64a17fba01c35b285bc99c14af547dbc3fab"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.4/node-v0.12.4.tar.gz" ] && NODEHASH="3298d0997613a04ac64343e8316da134d04588132554ae402eb344e3369ec912"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.5/node-v0.12.5-darwin-x64.tar.gz" ] && NODEHASH="23cbcbae16488384580324a059a8716667f2ea7798589caebf038ceda2db5355"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.5/node-v0.12.5-darwin-x86.tar.gz" ] && NODEHASH="638163edf9810f067b94641f51fabff3b721f6ac7d893db424f987c3916fec05"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.5/node-v0.12.5-linux-x64.tar.gz" ] && NODEHASH="d4d7efb9e1370d9563ace338e01f7be31df48cf8e04ad670f54b6eb8a3c54e03"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.5/node-v0.12.5-linux-x86.tar.gz" ] && NODEHASH="c853d63f58be110c1eeb4df7f30b1014ffd56b261e3507eb54330dbe3b176f41"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.5/node-v0.12.5-sunos-x64.tar.gz" ] && NODEHASH="ceb0d587ff4f85e47ed7f85fac96ada51bc42199725a218f72e109adb12766c0"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.5/node-v0.12.5-sunos-x86.tar.gz" ] && NODEHASH="e39121eebb362ea619de354af802bf5428105a96a87dc5f40fc20c2979e5b772"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.5/node-v0.12.5.tar.gz" ] && NODEHASH="4bc1e25f4c62ac65324d3cf4aa9de2d801cd708757c3567b6ad2ced7df30cdd2"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.6/node-v0.12.6-darwin-x64.tar.gz" ] && NODEHASH="843b72103c461c780554c590041bc1df1086e192c07dc8eeddb5fe644f072f19"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.6/node-v0.12.6-darwin-x86.tar.gz" ] && NODEHASH="ac491532dca62a2ae7a3315637fe01e4120ed2720be7a2eedb43d465e8f559de"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.6/node-v0.12.6-linux-x64.tar.gz" ] && NODEHASH="ebbd70ffe0daac6b33df74577c9d25cfb678dbc0016a5ea9eff99d2c5bdb3851"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.6/node-v0.12.6-linux-x86.tar.gz" ] && NODEHASH="59c1c83f3ef209206757ba8a8bb0c5dc818fe433b6fc4d6a8868ccf8f447281c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.6/node-v0.12.6-sunos-x64.tar.gz" ] && NODEHASH="cd0d0dd1ec646ac41c5b2486392662910930779b29a90e9650189b0bc2ca2a7e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.6/node-v0.12.6-sunos-x86.tar.gz" ] && NODEHASH="f168959b26c0b1074d5fe24899f070d352741bfa624a556e263890b624af68c5"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.6/node-v0.12.6.tar.gz" ] && NODEHASH="7a3b5ac351973a9dee8edbf0684bc8d0dea44b231e42274ffb008141ffa19ad2"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.7/node-v0.12.7-darwin-x64.tar.gz" ] && NODEHASH="58f24547ae5be8e0c7183bed65f96a1722af1ce363eccb1523a2321f38d83d57"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.7/node-v0.12.7-darwin-x86.tar.gz" ] && NODEHASH="8db41046537a98b60bec7c918eb25b8828c26d1b85adf3918ad260073cdb4a6d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.7/node-v0.12.7-linux-x64.tar.gz" ] && NODEHASH="6a2b3077f293d17e2a1e6dba0297f761c9e981c255a2c82f329d4173acf9b9d5"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.7/node-v0.12.7-linux-x86.tar.gz" ] && NODEHASH="bccf75736b64bd175b45681ed83a020f0dcc59b3626bbcefd5f7438aed9e9c15"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.7/node-v0.12.7-sunos-x64.tar.gz" ] && NODEHASH="e5cb2f36c1e6899c6e8b268d1019459bc9dd125d8144e47fb0d740e8601e5ddb"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.7/node-v0.12.7-sunos-x86.tar.gz" ] && NODEHASH="c7c7af04ca624a7a08c308f172e26a7fd3182187d3da0025e0108c67c2e5c6c9"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.7/node-v0.12.7.tar.gz" ] && NODEHASH="b23d64df051c9c969b0c583f802d5d71de342e53067127a5061415be7e12f39d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.8/node-v0.12.8-darwin-x64.tar.gz" ] && NODEHASH="857155d09d62b59c675baf4091a4e76af0972f8c99a26259a18e3ac99575697b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.8/node-v0.12.8-darwin-x86.tar.gz" ] && NODEHASH="2ec14c53fa69836caf79822b2de071659fadb2c105b1344371b404df398bad39"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.8/node-v0.12.8-headers.tar.gz" ] && NODEHASH="9ad0c77d0c5c2236ba42519dbd7375e9462412e812b6c60e776fcc129ef97084"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.8/node-v0.12.8-linux-x64.tar.gz" ] && NODEHASH="99f9f8792850867a21caeaf12b1f84da9f64d0cf0ac602920facc0fc4b81e8b4"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.8/node-v0.12.8-linux-x86.tar.gz" ] && NODEHASH="b82e3b4a01f9be1f130d97cf6a8534ae727396448fb1bfeb7eb74ec58592bd88"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.8/node-v0.12.8-sunos-x64.tar.gz" ] && NODEHASH="c8ca60698f99b7dc7722b94c5b4110636d08d3a20cb3df80807bd420e2c34376"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.8/node-v0.12.8-sunos-x86.tar.gz" ] && NODEHASH="8d9553a684b6717f0f0f2f5dcc8ed78139db50129e1402ce6033e2494c06cfd3"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.8/node-v0.12.8.tar.gz" ] && NODEHASH="e0c96a6702978e2ed7f031315bebeb86b042e2c80e66d99af8ad864dc0e56436"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.9/node-v0.12.9-darwin-x64.tar.gz" ] && NODEHASH="8a40582c8f346f4acb08ab29bdc171db5fea55603999e02be1ebfcdd2ed3ca83"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.9/node-v0.12.9-darwin-x86.tar.gz" ] && NODEHASH="a89d21abe0eaae1fd4cd4753a7ccde5bb60188148742281f1b36830bb02d50fd"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.9/node-v0.12.9-headers.tar.gz" ] && NODEHASH="0da8dd9dd5bbfa821d4e957a0687f3cc39bc6cbc45f75d6751107142141426ca"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.9/node-v0.12.9-linux-x64.tar.gz" ] && NODEHASH="3416451924c9c996e1d7224f5e5507df84b90dc730d4760e3f4daac1bd4c44df"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.9/node-v0.12.9-linux-x86.tar.gz" ] && NODEHASH="07b25b4a886b1b04d427b2b6414c9e4a913f53bb9574c26f010b35984b70df10"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.9/node-v0.12.9-sunos-x64.tar.gz" ] && NODEHASH="460a75865d6155dc39794204214c567a239b319122caf116a60f870f0987b720"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.9/node-v0.12.9-sunos-x86.tar.gz" ] && NODEHASH="039c710094ac76bea7027cf37daceb5708e46cac0bd082d7004c9710ad77ad1f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.12.9/node-v0.12.9.tar.gz" ] && NODEHASH="35daad301191e5f8dd7e5d2fbb711d081b82d1837d59837b8ee224c256cfe5e4"
[ "$NODEURL" = "http://nodejs.org/dist/v0.2.0/node-v0.2.0.tar.gz" ] && NODEHASH="3d3eff9287c9917af4044f3cef99ae5b17946710a71e83039de4fcb4b0a26631"
[ "$NODEURL" = "http://nodejs.org/dist/v0.2.1/node-v0.2.1.tar.gz" ] && NODEHASH="5bb7d084b2138ce43fcb34739ed894379c450a1dd569a1c710405bc39d2861c2"
[ "$NODEURL" = "http://nodejs.org/dist/v0.2.2/node-v0.2.2.tar.gz" ] && NODEHASH="21dc8e18cd678f55773a2dbe309a559882bf6f3b969c9e5b39f6977fb85ee480"
[ "$NODEURL" = "http://nodejs.org/dist/v0.2.3/node-v0.2.3.tar.gz" ] && NODEHASH="7358f8969b7bf6da7b066185bfa72d3bbc92e80b174ff5ea0e2b536fd357c8cf"
[ "$NODEURL" = "http://nodejs.org/dist/v0.2.4/node-v0.2.4.tar.gz" ] && NODEHASH="e6952007dacf18d9d85ae8ede8228e25cfe46e00be21b31c4d166239ec1fa533"
[ "$NODEURL" = "http://nodejs.org/dist/v0.2.5/node-v0.2.5.tar.gz" ] && NODEHASH="6c964096e2fb7bfa9108b31bdd2a920465a1b7f7a603e3937128eee9538b44bb"
[ "$NODEURL" = "http://nodejs.org/dist/v0.2.6/node-v0.2.6.tar.gz" ] && NODEHASH="e97fe9c81ff4b569ae9a0d46e64a0572a1f171293573a5b5290bcc3996a19701"
[ "$NODEURL" = "http://nodejs.org/dist/v0.3.0/node-v0.3.0.tar.gz" ] && NODEHASH="aa53c3d136ceaa02108ad013d9d9917e6e2ea22f10e3bd7414f4ba6f6a1427b5"
[ "$NODEURL" = "http://nodejs.org/dist/v0.3.1/node-v0.3.1.tar.gz" ] && NODEHASH="19c4c6144af143fbe37f80ec5d2843c4e19b5b6054fb10225bec314b60d2d012"
[ "$NODEURL" = "http://nodejs.org/dist/v0.3.2/node-v0.3.2.tar.gz" ] && NODEHASH="0cfb16b60c3c32c5fe0644108abb09e9425597a192c23844f29782b3ef7f7de2"
[ "$NODEURL" = "http://nodejs.org/dist/v0.3.3/node-v0.3.3.tar.gz" ] && NODEHASH="7f0dd072fbfa2dca8d873f56a4f57fdecbf4aba794b821654bec86daf3f980bc"
[ "$NODEURL" = "http://nodejs.org/dist/v0.3.4/node-v0.3.4.tar.gz" ] && NODEHASH="6980300d371ea182d719722a92706a495d19bd2efc6e1c3cdfe1c8fff74b5717"
[ "$NODEURL" = "http://nodejs.org/dist/v0.3.5/node-v0.3.5.tar.gz" ] && NODEHASH="affdf4cbe8aaab74f99ea0a534d913a6203de353652c2fe01e1ce75707d730c7"
[ "$NODEURL" = "http://nodejs.org/dist/v0.3.6/node-v0.3.6.tar.gz" ] && NODEHASH="682709b86be119927015a95418a519542caadb95cbadb635ef51f7d0732fe305"
[ "$NODEURL" = "http://nodejs.org/dist/v0.3.7/node-v0.3.7.tar.gz" ] && NODEHASH="21c53e74684a8a3c3d14d14a49a07fa8250f0e41514b448182ef0d1f5bbba52f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.3.8/node-v0.3.8.tar.gz" ] && NODEHASH="fc71861d339ec9ced9c85cb714f14e3e6e51651833ef0fec2abf41ad113172f1"
[ "$NODEURL" = "http://nodejs.org/dist/v0.4.0/node-v0.4.0.tar.gz" ] && NODEHASH="4a30bd9963373cb86a994479bdd451ab3b6f2124f0089493366315da79d3408e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.4.1/node-v0.4.1.tar.gz" ] && NODEHASH="fdd61c479a0c9f30102454ee53d2ba0c5fc9f6d06d1073958ae2fd3fc314de23"
[ "$NODEURL" = "http://nodejs.org/dist/v0.4.10/node-v0.4.10.tar.gz" ] && NODEHASH="57fa7ed5a818308ff485bb1c1a8ec8f1eb6a7800e14201dff65d88ce657da50a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.4.11/node-v0.4.11.tar.gz" ] && NODEHASH="e009522d52c4a844c46e51c63b852899d1b7e6d949d1a139cdc16b4f6c4ab63f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.4.12/node-v0.4.12.tar.gz" ] && NODEHASH="c01af05b933ad4d2ca39f63cac057f54f032a4d83cff8711e42650ccee24fce4"
[ "$NODEURL" = "http://nodejs.org/dist/v0.4.2/node-v0.4.2.tar.gz" ] && NODEHASH="09b1100ca6828eedbe52418fbeb3352d71c0b1ff3344c44a5af3efb80c5b908c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.4.3/node-v0.4.3.tar.gz" ] && NODEHASH="945cd6743336933bdd1843c28e6d4c896483c8ffc899555079c1eca7a69bd81c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.4.4/node-v0.4.4.tar.gz" ] && NODEHASH="ea4430909601340cb3e8adb15569facfeca4e1d59129f1932254535bb4bf3e17"
[ "$NODEURL" = "http://nodejs.org/dist/v0.4.5/node-v0.4.5.tar.gz" ] && NODEHASH="63fa6acd7dbf1ea816dc5fd64ba4d066f85380396571d29934b8b9141dc2a0ee"
[ "$NODEURL" = "http://nodejs.org/dist/v0.4.6/node-v0.4.6.tar.gz" ] && NODEHASH="0f07823f1eb32a51351739763a73a66e2b8c80ed6e3e787a1f68eec255b481f6"
[ "$NODEURL" = "http://nodejs.org/dist/v0.4.7/node-v0.4.7.tar.gz" ] && NODEHASH="5c405ff85549ebff49ee06d4e4391f02ed65b2b1177dc0f617f727ab48593dee"
[ "$NODEURL" = "http://nodejs.org/dist/v0.4.8/node-v0.4.8.tar.gz" ] && NODEHASH="c72a0136a022581e9ca5d26fd4a9af277525204547bcf06276dbe6b66e1fa112"
[ "$NODEURL" = "http://nodejs.org/dist/v0.4.9/node-v0.4.9.tar.gz" ] && NODEHASH="f231ea6d19ea9ea4c7f8e7ff5061e7d301f1635bec7ed0ff1eef2512576ea442"
[ "$NODEURL" = "http://nodejs.org/dist/v0.5.0/node-v0.5.0.tar.gz" ] && NODEHASH="ac7e786f69343654dff091e1d5a85ee001f48b7b0fe145c0a0e14040b82978b9"
[ "$NODEURL" = "http://nodejs.org/dist/v0.5.1/node-v0.5.1.tar.gz" ] && NODEHASH="a788469bbfd52ba56f1fa76ff28f796aff8aec9ce1cd92aca62853e72d187839"
[ "$NODEURL" = "http://nodejs.org/dist/v0.5.10/node-v0.5.10.tar.gz" ] && NODEHASH="56396854f85a0d2fafc038436be3d84041f991f59613761e61295fc02d662a40"
[ "$NODEURL" = "http://nodejs.org/dist/v0.5.2/node-v0.5.2.tar.gz" ] && NODEHASH="198f9109bf22a9f8d88cf0b696bd36fc897dfa6b655f30102d7cc55bf033e19b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.5.3/node-v0.5.3.tar.gz" ] && NODEHASH="27e5a488040e59e192b3db6675c5f0b6b00cccdd53f1a7cdf98b6477220fbb1e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.5.4/node-v0.5.4.tar.gz" ] && NODEHASH="d32d3af4e3286b383640df857d76c2fcca1a2e2cb85abb484483a0a49d09ae71"
[ "$NODEURL" = "http://nodejs.org/dist/v0.5.5/node-v0.5.5.tar.gz" ] && NODEHASH="6f7ef8859e43545ff9a0e178e39a070f22c6a2abcf46b2cae079f446b5750e65"
[ "$NODEURL" = "http://nodejs.org/dist/v0.5.6/node-v0.5.6.tar.gz" ] && NODEHASH="f9745ab3b19be29d3ddf40c40cec6d4c4685ae94d9943389d6b67178f11ecd9b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.5.7/node-v0.5.7.tar.gz" ] && NODEHASH="bd5c1f8029517bd8070001e3099a2330bcea696fcf1855b6c857b6fb58e676c6"
[ "$NODEURL" = "http://nodejs.org/dist/v0.5.8/node-v0.5.8.tar.gz" ] && NODEHASH="e15214605c473d14cf73ebac4df1c8ff54d88f405b20b473bda719728e217fd2"
[ "$NODEURL" = "http://nodejs.org/dist/v0.5.9/node-v0.5.9.tar.gz" ] && NODEHASH="5659cde8b36cf5c29433e73a351b0bacfac16be1b0b47e64ea138fe270b5607f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.6.0/node-v0.6.0.tar.gz" ] && NODEHASH="1b6a34b6f2099145c44a0c20d3a5cab7c9ec063de1a195ddeda61ad55d601d7f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.6.1/node-v0.6.1.tar.gz" ] && NODEHASH="b161050ed8cdb2d45f601181d146821e5535a8fcbf5978b2ff064e5476a8e606"
[ "$NODEURL" = "http://nodejs.org/dist/v0.6.10/node-v0.6.10.tar.gz" ] && NODEHASH="d1d060ab53c6079409403530009a2095036f19920aabd3a1c20542e0db586bd5"
[ "$NODEURL" = "http://nodejs.org/dist/v0.6.11/node-v0.6.11.tar.gz" ] && NODEHASH="94bbdb2d62645fd2ad5b96e41cfec68abf004fd03fabaaf7d71c48b39013cbd1"
[ "$NODEURL" = "http://nodejs.org/dist/v0.6.12/node.tar.gz" ] && NODEHASH="a16392fb83b288bd40cb64593253756a44f8111478edf5e8cc439a64622281c4"
[ "$NODEURL" = "http://nodejs.org/dist/v0.6.12/node-v0.6.12.tar.gz" ] && NODEHASH="a16392fb83b288bd40cb64593253756a44f8111478edf5e8cc439a64622281c4"
[ "$NODEURL" = "http://nodejs.org/dist/v0.6.13/node-v0.6.13.tar.gz" ] && NODEHASH="fc4f3ceacfd2cfc4ec75fc59d97f1f2d04947efd5e191efaddeb552df486245b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.6.14/node-v0.6.14.tar.gz" ] && NODEHASH="e41922308155c5197c2d048948ca9cd76ea5f9a51f977e1591bd93fe17d4cf1f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.6.15/node-v0.6.15.tar.gz" ] && NODEHASH="78859a1a31c8e7a64d0efa040326d93c6624cc344d39b2e47e14e9c4dc3136e0"
[ "$NODEURL" = "http://nodejs.org/dist/v0.6.16/node-v0.6.16.tar.gz" ] && NODEHASH="21d9cccaa642794db69619d813adf00b3533080a8928370c2b1a4b3a6478eaa7"
[ "$NODEURL" = "http://nodejs.org/dist/v0.6.17/node-v0.6.17.tar.gz" ] && NODEHASH="8dfe5948de27e37a14af184f06e7bd89a23c3b248af44c8ef5cffcd0e4c65778"
[ "$NODEURL" = "http://nodejs.org/dist/v0.6.18/node-v0.6.18.tar.gz" ] && NODEHASH="6cf4311ecbc1700e88f4382a31b3a7017c1572cd641fd06e653fc1692c2cffff"
[ "$NODEURL" = "http://nodejs.org/dist/v0.6.19/node-v0.6.19-baddocs.tar.gz" ] && NODEHASH="748306c67ac04b9c28675ae5ed4ae66620d3faa801e5920304f2b2f859fe7249"
[ "$NODEURL" = "http://nodejs.org/dist/v0.6.19/node-v0.6.19.tar.gz" ] && NODEHASH="4e33292477b01dfcf50bc628d580fd5af3e5ff807490ec46472b84100fb52fbb"
[ "$NODEURL" = "http://nodejs.org/dist/v0.6.2/node-v0.6.2.tar.gz" ] && NODEHASH="3a24f6f91bb806a230a7b200ca638459a9680ea2daf9a427098c61f847016139"
[ "$NODEURL" = "http://nodejs.org/dist/v0.6.20/node-v0.6.20.tar.gz" ] && NODEHASH="b7bf4cf143ddf46ba5e975761b98a38dd3d72b176fd5d4bb2f9c9e7bbe6c4b15"
[ "$NODEURL" = "http://nodejs.org/dist/v0.6.21/node-v0.6.21.tar.gz" ] && NODEHASH="22265fd07e09c22f1d058156d548e7398c9740210f534e2f848eeab5b9772117"
[ "$NODEURL" = "http://nodejs.org/dist/v0.6.3/node-v0.6.3.tar.gz" ] && NODEHASH="fe5642d26d04cc7e7d47daa426da2a79e244bdcbae1594a12578f0d6fe03082e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.6.4/node-v0.6.4.tar.gz" ] && NODEHASH="67b029f0da10ffa706cda23d6a3bb7c682ca589cd7f6647a578dcfb74a78f916"
[ "$NODEURL" = "http://nodejs.org/dist/v0.6.5/node-v0.6.5.tar.gz" ] && NODEHASH="72364d240fb61e678897c099df6f2913857c5931aa9b1f44e73e432d4629ca2f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.6.6/node-v0.6.6.tar.gz" ] && NODEHASH="7abea518b1b63fd669c9ca436bf33d0bb0b09b252f06d700ccbd290fe5222102"
[ "$NODEURL" = "http://nodejs.org/dist/v0.6.7/node-v0.6.7.tar.gz" ] && NODEHASH="b34387449723352d2f5b7a51d8c1358c247908a5f7acd7849cac45f980246d54"
[ "$NODEURL" = "http://nodejs.org/dist/v0.6.8/node-v0.6.8.tar.gz" ] && NODEHASH="e6cbfc5ccdbe10128dbbd4dc7a88c154d80f8a39c3a8477092cf7d25eef78c9c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.6.9/node-v0.6.9.tar.gz" ] && NODEHASH="484ab6b3da6195339544c16aff17f747aa85d1dd15d765d6724aa8a4ecda03ca"
[ "$NODEURL" = "http://nodejs.org/dist/v0.7.0/node-v0.7.0.tar.gz" ] && NODEHASH="0c716d7c61f77c53b0383e82da3f3ca8a4187f7d1c9831aec95efe7cfda9cafb"
[ "$NODEURL" = "http://nodejs.org/dist/v0.7.1/node-v0.7.1.tar.gz" ] && NODEHASH="903224a2c9e4510b2281d0fa72cbd930b6dfb97a750e6d02bd12442a5aa30032"
[ "$NODEURL" = "http://nodejs.org/dist/v0.7.10/node-v0.7.10.tar.gz" ] && NODEHASH="9094dcd47dba984b6c2ca89a5361bd6664d1990edf41419f9a8a6d26ae774c6e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.7.11/node-v0.7.11.tar.gz" ] && NODEHASH="add0db06f628ed8330b1a3450efa39f18d08c5a54d5827004c76ae23522ca842"
[ "$NODEURL" = "http://nodejs.org/dist/v0.7.12/node-v0.7.12.tar.gz" ] && NODEHASH="e0be9f001467e2a9e728d53969a3c7f079da4af1ff896155822479ad92b3efbe"
[ "$NODEURL" = "http://nodejs.org/dist/v0.7.2/node-v0.7.2.tar.gz" ] && NODEHASH="de0bff9067f4cd63c4bd5fb847352725b86122eac7483aa88b764eb8272807a3"
[ "$NODEURL" = "http://nodejs.org/dist/v0.7.3/node-v0.7.3.tar.gz" ] && NODEHASH="292204f73b4007fecf7aa9d065a0360abc43a1fc007ea1138cf4fa8d553cca58"
[ "$NODEURL" = "http://nodejs.org/dist/v0.7.4/node-v0.7.4.tar.gz" ] && NODEHASH="14bd37525dab52d28271211002fe102ebbcc9afa01064f93d9945d66e3989660"
[ "$NODEURL" = "http://nodejs.org/dist/v0.7.5/node-v0.7.5.tar.gz" ] && NODEHASH="f855ba273ea8d4b036f811008dc205e0070358584185dbc79abd09ffcb99d12b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.7.6/node-v0.7.6.tar.gz" ] && NODEHASH="ce1772b7b649121a04dbb36c94e4e4d8850a109d07746cff6bab5c4d1ed38e1c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.7.7/node-v0.7.7.tar.gz" ] && NODEHASH="40d67cdaba5fdef522fc49e31220ffc3278c90cb5ca1dc5dbb6e43cbc938f8d1"
[ "$NODEURL" = "http://nodejs.org/dist/v0.7.8/node-v0.7.8.tar.gz" ] && NODEHASH="59722d4a71c26963a8bbd31aadc64aeff2c0a2902fb4d9542d161f458487355e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.7.9/node-v0.7.9.tar.gz" ] && NODEHASH="0a94de8743420c0a664b3bc9bec876098e4035b5079a4b2b4e3e3dc1acc9f926"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.0/node-v0.8.0.tar.gz" ] && NODEHASH="ecafca018b5109a28537633d0433d513f68b1bae7191a1821e8eaa84ccf128ee"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.1/node-v0.8.1.tar.gz" ] && NODEHASH="0cda1325a010ce18f68501ae68e0ce97f0094e7a282c34a451f552621643a884"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.10/node-v0.8.10-darwin-x64.tar.gz" ] && NODEHASH="3b347b17f295ef4683278c589eed695db3544d2fba4d5969d0700173153da336"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.10/node-v0.8.10-darwin-x86.tar.gz" ] && NODEHASH="0059d0ad27725a7c3f9b413fdb5fd09dfc7b7879f42025977504849d3c9e2ab5"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.10/node-v0.8.10-linux-x64.tar.gz" ] && NODEHASH="536b88491d8d5db046897a4024209b076be1c25dbf92474b85f8b854812c2c21"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.10/node-v0.8.10-linux-x86.tar.gz" ] && NODEHASH="263820d42c73c07bc2b0e24a58a33f0979f2655f57d8dc73eb750522d096f7a4"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.10/node-v0.8.10-sunos-x64.tar.gz" ] && NODEHASH="15d6adf4d457a820ef62d07e52a00b42d8ce23a59cc55a54a3cc564e86f5c56a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.10/node-v0.8.10-sunos-x86.tar.gz" ] && NODEHASH="1efe7482197330c59067a9646d67c0af3fa638ccec32cd3d3c71ecc6729511f2"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.10/node-v0.8.10.tar.gz" ] && NODEHASH="ce495ab8fee58b4df49d46be2b08fabdd2fcb3880982aedda8e88751e2a2011b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.11/node-v0.8.11-darwin-x64.tar.gz" ] && NODEHASH="7199b8260f3b02c362b7a347a27d9527c9fac37f4ca5d6c5e1d3c2ebd4d271c7"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.11/node-v0.8.11-darwin-x86.tar.gz" ] && NODEHASH="7c857174ba99de693c8134055cb9d675c6953e31af2cd4380615ada6ebb4519d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.11/node-v0.8.11-linux-x64.tar.gz" ] && NODEHASH="8586d1e89977567c5c3d75a974409a2453a5fc9f26f8ea634016dee255595347"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.11/node-v0.8.11-linux-x86.tar.gz" ] && NODEHASH="3ca70266324ac29d70b853ea1966ed81918996f65f276eb8e7144f5f494b8d1c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.11/node-v0.8.11-sunos-x64.tar.gz" ] && NODEHASH="9633b53df777c6b5a41dc3e6c106c51f20fbe151c793d82c80c93ca17ccc6975"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.11/node-v0.8.11-sunos-x86.tar.gz" ] && NODEHASH="038d1b1774bdd9c260c5967380bd89df3145a7aeb190effd681ae2091f9e1676"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.11/node-v0.8.11.tar.gz" ] && NODEHASH="e9594460f992b5862e21fb4d8ef27907839254c646b4ed5e8ab1ec25b4ccd29d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.12/node-v0.8.12-darwin-x64.tar.gz" ] && NODEHASH="05836e0924b67713e0306f710adca8ce0fbf1941dd959a354f000b8d3bb039f6"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.12/node-v0.8.12-darwin-x86.tar.gz" ] && NODEHASH="63807554c451e07fb07496bd9539bf7c67b7d082e52b12a4209f02b015c786d0"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.12/node-v0.8.12-linux-x64.tar.gz" ] && NODEHASH="7a5437d5c088f1b787444b2ac049add8fd11f21f0486819491ac1b3965d6f288"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.12/node-v0.8.12-linux-x86.tar.gz" ] && NODEHASH="758ff08be620f3426c43b3f250f2dcc007b9bdb8be05d2fd9cccaed9b0e30af6"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.12/node-v0.8.12-sunos-x64.tar.gz" ] && NODEHASH="94f9cc5425094f2103f1324e3dbe6d277bdc0dab1171a4be4f90466958df324e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.12/node-v0.8.12-sunos-x86.tar.gz" ] && NODEHASH="f2b5e658fbcd5df94d36b75b9b8fb8c1929adb4a79015ebcdab7e1e494da7264"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.12/node-v0.8.12.tar.gz" ] && NODEHASH="d64a7f2ab8f4419a11bde5379d6065666fd1cc4593ec828cb5ac57385efafa45"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.13/node-v0.8.13-darwin-x64.tar.gz" ] && NODEHASH="2f2317359a8d10429f6b69f18c659c5cbbee40d948d994fb469237e9bb138b73"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.13/node-v0.8.13-darwin-x86.tar.gz" ] && NODEHASH="e86aa770b74ddf64f4beda3171e65c2adb39a2ff99455ad17af91d92703b2d72"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.13/node-v0.8.13-linux-x64.tar.gz" ] && NODEHASH="963b0187643a2c60a597ffcdb71343765e4a0acb8cb53ff14f8c3a3bd4226b58"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.13/node-v0.8.13-linux-x86.tar.gz" ] && NODEHASH="51c8f86a866cdc5b77d1e8c57aaaa0cef73146063ba1ad6151544bf1bb0147d7"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.13/node-v0.8.13-sunos-x64.tar.gz" ] && NODEHASH="e0563da18e7b127e040e2f356a51db38816e3b3a71d50f6735946d2c4646ad7b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.13/node-v0.8.13-sunos-x86.tar.gz" ] && NODEHASH="b070262e2ce8d0f2eedcae8632e512f2fafca31f5eb51a8994a8b58ef65e8083"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.13/node-v0.8.13.tar.gz" ] && NODEHASH="4d85486079aed1749b8bd77899cb728820325384881c874cf78480696f3d8c8c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.14/node-v0.8.14-darwin-x64.tar.gz" ] && NODEHASH="7713f0f5379a8b4a026ea9ec14f55394f34bb61c91e27dd08eea28a7edc31edd"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.14/node-v0.8.14-darwin-x86.tar.gz" ] && NODEHASH="f062fed150892fe18c2f50d5d05d6757216ffcfca58c035ac67e77e76a3e7c48"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.14/node-v0.8.14-linux-x64.tar.gz" ] && NODEHASH="032aeede8ef2f147c10327aa310430554a67c30c6fe3b041426e7afb39932320"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.14/node-v0.8.14-linux-x86.tar.gz" ] && NODEHASH="4d5a02b4927cda17edae385037c5febd14130317f29f98bc37c40568542929c2"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.14/node-v0.8.14-sunos-x64.tar.gz" ] && NODEHASH="bd24375f0064abb29a48b7f0975bd719c4175cc4ae8fb47cace2d26db352f35d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.14/node-v0.8.14-sunos-x86.tar.gz" ] && NODEHASH="76b110657c0bdc9b75d85efc259f0df390c854e885a093213674fdca8d92edc5"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.14/node-v0.8.14.tar.gz" ] && NODEHASH="e5ce2aadb4df3ea4ca7a021106ffe09d286474476454038e9ed0135eac18e6d0"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.15/node-v0.8.15-darwin-x64.tar.gz" ] && NODEHASH="c1050151d178017a2d9fbf842d63e014beb63104c96031b42b5c6d1476f5a2c4"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.15/node-v0.8.15-darwin-x86.tar.gz" ] && NODEHASH="735d20eb41ba56a97ae3d0331179dc64d9cf1296d8f483b8b568d3764b5089a9"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.15/node-v0.8.15-linux-x64.tar.gz" ] && NODEHASH="97026c2a421737ac54b0faa95ea0d261b9940f7524bf2bcf5ce0aadf05f98377"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.15/node-v0.8.15-linux-x86.tar.gz" ] && NODEHASH="033014041f7857463e266b2bba84411c739461eec8b4c036cdd5158a2285d34a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.15/node-v0.8.15-sunos-x64.tar.gz" ] && NODEHASH="fa0e1bf5cbd46c5ff476e88d1fbee571ab96b038cee9eb38f703da5b45c60a6a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.15/node-v0.8.15-sunos-x86.tar.gz" ] && NODEHASH="bb51410611a8c97998c4e587e75aa6ef6746236ef45454fa3b9c42a05fa8a649"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.15/node-v0.8.15.tar.gz" ] && NODEHASH="1758639c6df3e081fe26585472d0f1961c5703b44ba6c57ecdf66a4c015792b1"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.16/node-v0.8.16-darwin-x64.tar.gz" ] && NODEHASH="348dfbb41a322d4e365667aa01a2c3fb722a9650638efd0d5b280a6ec31e5315"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.16/node-v0.8.16-darwin-x86.tar.gz" ] && NODEHASH="3712a28e479ab0836a7cb9b583d7a233a531724524ef337fe1a06f47e0b3aab1"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.16/node-v0.8.16-linux-x64.tar.gz" ] && NODEHASH="528ca5b68c04bf13dd9eaf784328dcfe6128449b32a409c5ef798200e7f15350"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.16/node-v0.8.16-linux-x86.tar.gz" ] && NODEHASH="c563edc9a27556802d31281026ddd13bb968b04099f6a4742b277a764c80fd75"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.16/node-v0.8.16-sunos-x64.tar.gz" ] && NODEHASH="832a394c1cfbbd67be7f397a229561c17bec28df65d00716d767e835ba9cae28"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.16/node-v0.8.16-sunos-x86.tar.gz" ] && NODEHASH="37a3e1f9b9e2f77eb23681792be1ac65b80455f4a5dbf710a7722ac2efb804d0"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.16/node-v0.8.16.tar.gz" ] && NODEHASH="2cd09d4227c787d6886be45dc54dad5aed779d7bd4b1e15ba930101d9d1ed2a4"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.17/node-v0.8.17-darwin-x64.tar.gz" ] && NODEHASH="0b6a2c9f295c59c7c7364a03e51467b0bf6614cb0d46d590d6ee8b89438a6206"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.17/node-v0.8.17-darwin-x86.tar.gz" ] && NODEHASH="ed93a1ce083dd6d1b616ce97c4535ff752865e8a9a45fe2df86960b4f7ece350"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.17/node-v0.8.17-linux-arm-pi.tar.gz" ] && NODEHASH="7caff731ddf6f95543abbd749b7c85f7a15bb3b396549cd64c48ebf8d51f9e0d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.17/node-v0.8.17-linux-x64.tar.gz" ] && NODEHASH="e55ee735ca063026901c1f876ac1b04d345150e06f54977e693b5ce5d082fd01"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.17/node-v0.8.17-linux-x86.tar.gz" ] && NODEHASH="3a68448f88465986c2b15ee6aa4e6d019070c958c836e7259c910f29213e96ab"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.17/node-v0.8.17-sunos-x64.tar.gz" ] && NODEHASH="9bf6565ffbcf47b1e2f328b73bac0748659e5ed0b94b07dd8462c232af2cc91b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.17/node-v0.8.17-sunos-x86.tar.gz" ] && NODEHASH="ddf05292902728bec9494fe0ce460cbb182e0638aebf5bc9d28d929e7c78d87d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.17/node-v0.8.17.tar.gz" ] && NODEHASH="8f070b42ffb84fde9d3ed2f802b08664b94dda327a36bf08a80c8b7efcf8b29e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.18/node-v0.8.18-darwin-x64.tar.gz" ] && NODEHASH="45f2a4ab27018e84dd83b7172c8f1560fbc0c3d24a6b6d03ca1bfc1b33bdb240"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.18/node-v0.8.18-darwin-x86.tar.gz" ] && NODEHASH="fa5addf17bc82cf9b553b00381e1ca8498a5a89085309dfe07f970384eaf925f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.18/node-v0.8.18-linux-arm-pi.tar.gz" ] && NODEHASH="1237c7bffdcc6a3bc6bdcebadd0a4729c2d180a003871e2955a2644e57ce97e1"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.18/node-v0.8.18-linux-x64.tar.gz" ] && NODEHASH="20c3e06f5f0d31b4e95bbba35eaff1f953a2b471021cc49e53b546c2e2efd97b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.18/node-v0.8.18-linux-x86.tar.gz" ] && NODEHASH="8fcc53f814c988aff5e7502f2d755218d34d641d4670cd621127a8cfe694cc75"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.18/node-v0.8.18-sunos-x64.tar.gz" ] && NODEHASH="7c36db277ce3f7dd59dc9cc792f16f02266d9dfdcd22b3658370e425ebe95080"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.18/node-v0.8.18-sunos-x86.tar.gz" ] && NODEHASH="306263031eeff00ded073b08f5344831b572034ab0d3a98af01a5f46c614411b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.18/node-v0.8.18.tar.gz" ] && NODEHASH="1d63dd42f9bd22f087585ddf80a881c6acbe1664891b1dda3b71306fe9ae00f9"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.19/node-v0.8.19-darwin-x64.tar.gz" ] && NODEHASH="39387dfb03c7c5421bdd117963d520ac1db1b26a74e0931125aa636d182e8aa2"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.19/node-v0.8.19-darwin-x86.tar.gz" ] && NODEHASH="e2272cd0985c29d100a4f4114a1d5957c076ab076705b04855ceb68ebc816dd5"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.19/node-v0.8.19-linux-arm-pi.tar.gz" ] && NODEHASH="35a921869eb6957a612f4437db9cc93bbc989d5db2fcea3626f4757d0ce10d2c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.19/node-v0.8.19-linux-x64.tar.gz" ] && NODEHASH="32ab76970568ea17a7c20bb49d31b82b2534138fe30256086544734e4fd728c7"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.19/node-v0.8.19-linux-x86.tar.gz" ] && NODEHASH="4d7459b4d40f612bae33562a904aad9171ecc3e0a18bec2745c0a967ccd8867d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.19/node-v0.8.19-sunos-x64.tar.gz" ] && NODEHASH="fb32343c8bafbd7698db5db2f2b103769e2ef2eccc20586000d8c3ccb8a9fdbc"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.19/node-v0.8.19-sunos-x86.tar.gz" ] && NODEHASH="d296d478ad60114d6f26848c63881d98e24a5c499695805d17e70368d8678f6e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.19/node-v0.8.19.tar.gz" ] && NODEHASH="703207d7b394bd3d4035dc3c94b417ee441fd3ea66aa90cd3d7c9bb28e5f9df4"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.2/node-v0.8.2.tar.gz" ] && NODEHASH="6830ed4eaf6c191243fb3afbe3ca3283d7e3a537c8f3ce508fa2af1328fe4baf"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.20/node-v0.8.20-darwin-x64.tar.gz" ] && NODEHASH="f81d2dff6097a80606b8807cd4eaa4c77b2c83630394a1bcfba885679b6caa1a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.20/node-v0.8.20-darwin-x86.tar.gz" ] && NODEHASH="7f2863ffe3cabebc33930924bfd1ebf3108c070ed052eeb02ccd292c85f6658a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.20/node-v0.8.20-linux-arm-pi.tar.gz" ] && NODEHASH="a051fb79f7f7b68d38e9933669cddea093bd2a1068558db75b5c95e963199a16"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.20/node-v0.8.20-linux-x64.tar.gz" ] && NODEHASH="e78965ffe7b7b56c15fbde44b8a9da7438a3527567b895b71336015917c7923e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.20/node-v0.8.20-linux-x86.tar.gz" ] && NODEHASH="4efa11253344c0765de7d6c0e7a89622589cf4638388463df15fc9ff24b96204"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.20/node-v0.8.20-sunos-x64.tar.gz" ] && NODEHASH="73e8c9951ab6cc4641eadf2bdfffa5f8c8b2efe21fef2ac915f41e4e4915c8af"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.20/node-v0.8.20-sunos-x86.tar.gz" ] && NODEHASH="3595c38edcd478d9d728fa152022511420d3864a89234b79a5b64d7ec3924598"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.20/node-v0.8.20.tar.gz" ] && NODEHASH="e4461bfded531f4880839829ab3bce5b824905d6e181876e3d0309a366bf57ee"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.21/node-v0.8.21-darwin-x64.tar.gz" ] && NODEHASH="6db45e7118329c94b733d49b0adb27f26769d48915c75cf812cf6d4b4fff49bf"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.21/node-v0.8.21-darwin-x86.tar.gz" ] && NODEHASH="50fc316020d7ed1ad9a52a2ff2cd74046b057f18cdd1abaa01def02c860fc22f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.21/node-v0.8.21-linux-arm-pi.tar.gz" ] && NODEHASH="de560f70e6d2e0588c7a13763f27329b443327afc3880b12f1a3464284efa8cd"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.21/node-v0.8.21-linux-x64.tar.gz" ] && NODEHASH="eaedcf7e3e443cf2fa35f834ed62b334885dc20fcbc7a32ea34e8e85f81b2533"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.21/node-v0.8.21-linux-x86.tar.gz" ] && NODEHASH="ea4508e4df3c74d964a02d5740374b54f8192af19db518163c77ee7ff318daa7"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.21/node-v0.8.21-sunos-x64.tar.gz" ] && NODEHASH="51ca34ac742efb321720144110c9df1672f3aa53521f73591c11cbf0a5a8754c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.21/node-v0.8.21-sunos-x86.tar.gz" ] && NODEHASH="f34358c23e9145db8a28f41a9269f3a75c0066b32ff0f19d05bacb04eb2192c6"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.21/node-v0.8.21.tar.gz" ] && NODEHASH="e526f56d22bb2ebee5a607bd1e7a16dcc8530b916e3a372192e6cd5fa97d08e6"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.22/node-v0.8.22-darwin-x64.tar.gz" ] && NODEHASH="640cb11ecad00b9cddaadb2d95d760123523c26e0dcbf2abe9ef5ab3c2e41206"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.22/node-v0.8.22-darwin-x86.tar.gz" ] && NODEHASH="e725e8a2fcebc6d799e94f40eb757c731764cfb76d12e2332858d242cfcb9eeb"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.22/node-v0.8.22-linux-arm-pi.tar.gz" ] && NODEHASH="0fb6c6c6a0e9e24c9343d2216e059f16dd7d7da2701efe882e2de0dc0e0e408b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.22/node-v0.8.22-linux-x64.tar.gz" ] && NODEHASH="6228478b96ed4765bdf9576abfb19088ba8ef333cd86a9e18c31e26014cbbad5"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.22/node-v0.8.22-linux-x86.tar.gz" ] && NODEHASH="967de70216638b97f551262d19a6a63bd99023638e7cc2bf493fc827a5df9609"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.22/node-v0.8.22-sunos-x64.tar.gz" ] && NODEHASH="8807dcc3192d3a8c0aae407b620e7ccea62ea3c8ba00a9c99118684cc9c43b77"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.22/node-v0.8.22-sunos-x86.tar.gz" ] && NODEHASH="1eb0aa09ba0c5e99e81dd267ee63586f40ea6c3c4ea0b1134a659cb3a86472c4"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.22/node-v0.8.22.tar.gz" ] && NODEHASH="3f61152cf5cd8fc1ab5c6c18101819841b947da79e1e44b51418c0ad2e6db8e8"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.23/node-v0.8.23-darwin-x64.tar.gz" ] && NODEHASH="fa1835e56992b0608ae6d19e031e33545a86becbb0d2695809403d70aff58258"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.23/node-v0.8.23-darwin-x86.tar.gz" ] && NODEHASH="1243625fd442e68d232566aa2ed5919f3e792a9bf6a85bb5da360a53f7035691"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.23/node-v0.8.23-linux-x64.tar.gz" ] && NODEHASH="b88284ec2a5944154962bfe3221238a003aa0a6b5306f777ad1554c22e89698b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.23/node-v0.8.23-linux-x86.tar.gz" ] && NODEHASH="5c7eaf94952a1699f961cad8bcd486290087378a1c49da21214f6cc0eb48d40a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.23/node-v0.8.23-sunos-x64.tar.gz" ] && NODEHASH="8bd2f6ab2ed42416dd93e63dae996c5cf4b6ac1f1f5809b31d2c1d6302f55288"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.23/node-v0.8.23-sunos-x86.tar.gz" ] && NODEHASH="5784b479f6430abe8f8c58f1f6a1f457fe7f841bd390bb7e6c2cdd2abf02025c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.23/node-v0.8.23.tar.gz" ] && NODEHASH="382432638aedc25495e655dda338adcf41c6fa1d35f355936d659784c1deed9d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.24/node-v0.8.24-darwin-x64.tar.gz" ] && NODEHASH="8a15b6b40ba03fc361342faf2ddb3cc0480ebf5e63fb9adbaab6cda757697311"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.24/node-v0.8.24-darwin-x86.tar.gz" ] && NODEHASH="1f06a3e8c887322479ae9f9a63f573e0638ed9cc1b72668d2f917062810b8278"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.24/node-v0.8.24-linux-x64.tar.gz" ] && NODEHASH="1296b795fb1d8406d7548372caef82454c460111b3ed4ef6d4b67dca6cfa4a76"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.24/node-v0.8.24-linux-x86.tar.gz" ] && NODEHASH="467135ebd9bba554d25f3174248ba35754a45bdcb82ef6064c44ee705d0b2d4b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.24/node-v0.8.24-sunos-x64.tar.gz" ] && NODEHASH="71578fd0ae413bdd41e336959889db70f1268084da92c37ba8467701b60ac995"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.24/node-v0.8.24-sunos-x86.tar.gz" ] && NODEHASH="ce340c74abff8c6d462ce373ed8c6882c522a1032d327474ef17541403448b82"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.24/node-v0.8.24.tar.gz" ] && NODEHASH="a0836e210da50106329e573532d5c590594ff1fcb9564fe8b55a04f04ff77705"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.25/node-v0.8.25-darwin-x64.tar.gz" ] && NODEHASH="22aa30210b989c8800d0625c880153594cd1e39015a30a85e0c2085c9497c09d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.25/node-v0.8.25-darwin-x86.tar.gz" ] && NODEHASH="c91abac167f9f8e2b3ff455731c77fb6319b2d11afce80757e65df4aa1beac2d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.25/node-v0.8.25-linux-x64.tar.gz" ] && NODEHASH="7eedbece123b5acacfa5ca9d1e7a1ab6bd9b32bed5b3c92f6853cca57be82d07"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.25/node-v0.8.25-linux-x86.tar.gz" ] && NODEHASH="b8ceddb23c002a172f065f1589ed53a4820c8cde8c62b80ba251e79f2515f610"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.25/node-v0.8.25-sunos-x64.tar.gz" ] && NODEHASH="371ba8792e979ad0b2b1e780436e3ed4201ed2c1ffb916c01bc5de94178cd0d8"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.25/node-v0.8.25-sunos-x86.tar.gz" ] && NODEHASH="0ad7c35c45e929edc13ab1ed699ed02b9bfc4155a643db979baa0ee121c5b8d6"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.25/node-v0.8.25.tar.gz" ] && NODEHASH="3dad0149020102e9b5604edae3cecd5842221c6657f26d35717bffaaf376caec"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.26/node-v0.8.26-darwin-x64.tar.gz" ] && NODEHASH="1c25e4ca125ddda22648957e1a40f4c7d37a0eda52cbe9ae1620b8bce9da317b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.26/node-v0.8.26-darwin-x86.tar.gz" ] && NODEHASH="512cb20374466fd4b26b44f52aea6300d7f0cb529e80311fea85b136de242b24"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.26/node-v0.8.26-linux-x64.tar.gz" ] && NODEHASH="57e70b7571393cc32019f0ff6d086183198b8e7824c3690ebed504d37f52c298"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.26/node-v0.8.26-linux-x86.tar.gz" ] && NODEHASH="61df482f8886c0c74d29387adb9ad9c7f211b1eb95275d201576d04d0fc64bba"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.26/node-v0.8.26-sunos-x64.tar.gz" ] && NODEHASH="c216e547c7f84e397c71241b99a1565de78a22bbcc3f9c39005a4b28dd741896"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.26/node-v0.8.26-sunos-x86.tar.gz" ] && NODEHASH="faaaab76ca39712a17af2928ad71a0ce94bd6af599b7d3472912e5e4724e7aab"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.26/node-v0.8.26.tar.gz" ] && NODEHASH="d873216685774b96139af534ce015077d2c93ddfc4e3596e128853f3c08a5413"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.27/node-v0.8.27-darwin-x64.tar.gz" ] && NODEHASH="593044136647bd73d29c709d16c4c27ff295c41129015d8414f5ea1f935efd2d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.27/node-v0.8.27-darwin-x86.tar.gz" ] && NODEHASH="98d086a42fc54b2c90b5890732497ab957b7ded4bfc57aa7997b4984d4faf0ef"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.27/node-v0.8.27-linux-x64.tar.gz" ] && NODEHASH="d65fafd755ed630abc6df04548dc3075282c06661cfd44f1af42c5e15a8c6826"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.27/node-v0.8.27-linux-x86.tar.gz" ] && NODEHASH="969d25008f524a4b397e9bceda6a094d960342f8923a77a1478f0603da6cea39"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.27/node-v0.8.27-sunos-x64.tar.gz" ] && NODEHASH="6340f05c2b56160fcfb796723d6702f28d0100595b7180acf715d7304a03d1bd"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.27/node-v0.8.27-sunos-x86.tar.gz" ] && NODEHASH="5e41e9e31fdd849483ef2b81c5dae7cf9487e70780bda2d216447b34c2707265"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.27/node-v0.8.27.tar.gz" ] && NODEHASH="30608f9dcd9ad122f7e8e6212f95969979e3dc35309d0c422a56486334a9369e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.28/node-v0.8.28-darwin-x64.tar.gz" ] && NODEHASH="e2ce196394765bee8fbb703651dbc4024cd840f802d2277e7adff91f5a6e1656"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.28/node-v0.8.28-darwin-x86.tar.gz" ] && NODEHASH="2f9081ef6e4ab46eb74c52395e7eb886c647466a9aa57add2945649cd351c87a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.28/node-v0.8.28-linux-x64.tar.gz" ] && NODEHASH="9ea0be4f08ade1645d2acb3fc4e294eb4d1ca595a405168caec2d7a0d41ca84e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.28/node-v0.8.28-linux-x86.tar.gz" ] && NODEHASH="a982c630d2ca11cd8ff747c5b6394db52b05e67b11cedf0ffe267a2ce241d160"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.28/node-v0.8.28-sunos-x64.tar.gz" ] && NODEHASH="5966bb3ca9ed4a30a9391b0329c41a1075e01410b38d5d8245271f7d2bfa0db2"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.28/node-v0.8.28-sunos-x86.tar.gz" ] && NODEHASH="784c45b07e1e9b664cbe1cefa5d0ddaf6de5397a70cd3b529f8953182a2b5b4e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.28/node-v0.8.28.tar.gz" ] && NODEHASH="50e9a4282a741c923bd41c3ebb76698edbd7b1324024fe70cedc1e34b782d44f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.3/node-v0.8.3.tar.gz" ] && NODEHASH="600a1d88744937dbad048f65f60d1259b814c59777d7b9d8665def77b1e1ec35"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.4/node-v0.8.4.tar.gz" ] && NODEHASH="d0a3b0d2028ddd6bbaab5d3fe38dd6f80d7e810fe9e55efab3230a7c90d31174"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.5/node-v0.8.5.tar.gz" ] && NODEHASH="022a924973b15291f98b57437b53c59eaabee02348a2534259836e837f9ea6c0"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.6/node-v0.8.6-darwin-x64.tar.gz" ] && NODEHASH="42c853c0adb8052da0d2976cfc2cbb29152dc9904d69efd7e61c71afbf9492af"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.6/node-v0.8.6-darwin-x86.tar.gz" ] && NODEHASH="36ee1c7fbe2a7e9e3dc8a215e5cb64905774f62f694cc710493f9d618016c631"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.6/node-v0.8.6-linux-x64.tar.gz" ] && NODEHASH="d2c04ae64f9070b6ab5b7da7ccfc6d22f019e165ce6cce06d3ab422dfbeded1f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.6/node-v0.8.6-linux-x86.tar.gz" ] && NODEHASH="d5584968c7fcb74df3760b0838720951049aee64ba48a65385f5aba929bd3697"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.6/node-v0.8.6-sunos-x64.tar.gz" ] && NODEHASH="41b42ceec543c8087a5bc3f66ad7788003409d58daf16b9ee872bea29a9ee98d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.6/node-v0.8.6-sunos-x86.tar.gz" ] && NODEHASH="94fa004e74f2e176022c03c07fd3cabb84de3bfc25c801e741be38a152a46341"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.6/node-v0.8.6.tar.gz" ] && NODEHASH="dbd42800e69644beff5c2cf11a9d4cf6dfbd644a9a36ffdd5e8c6b8db9240854"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.7/node-v0.8.7-darwin-x64.tar.gz" ] && NODEHASH="3966ba6007b13ba8f47af101c2a329b39412a532ba9aa089860d4af54b51efcd"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.7/node-v0.8.7-darwin-x86.tar.gz" ] && NODEHASH="60812a5c2e0a70360d855b7d765a7ab45fc8ddd00da1b26c5e086257bb25ff73"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.7/node-v0.8.7-linux-x64.tar.gz" ] && NODEHASH="a9fa9e6ea54bb8cc940264dab60c867f8231aef00b4d1b07be63f3150e91067e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.7/node-v0.8.7-linux-x86.tar.gz" ] && NODEHASH="600c4712ba764e84d1d59c7d04e105d732c9750e276b4da6f7bed0947ef580c3"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.7/node-v0.8.7-sunos-x64.tar.gz" ] && NODEHASH="de4b5134884eec075c5e00eda81086cd92187e81c2e910462aca36a848e008a7"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.7/node-v0.8.7-sunos-x86.tar.gz" ] && NODEHASH="3219c5efd7bf1f069d424c34fd75332b094f7870e24f33eb46916f4d509a1ecb"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.7/node-v0.8.7.tar.gz" ] && NODEHASH="fa979488347ad08ea6e36d3fe9c543807cd6f84cad31b22bfc6179b54b1e9d04"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.8/node-v0.8.8-darwin-x64.tar.gz" ] && NODEHASH="69e5f514e22c2139454f9469c58c8bb0f4092c29e1115bc446afd12f75a467e7"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.8/node-v0.8.8-darwin-x86.tar.gz" ] && NODEHASH="e8e4572c89b16c4f0f48d5060bbe4dd2f8db299711262417300b82b2df790cbb"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.8/node-v0.8.8-linux-x64.tar.gz" ] && NODEHASH="bc0038a4ec650578b8d9a55aa4f86be22b43c0115a1efe0378d6a177a8ee3bca"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.8/node-v0.8.8-linux-x86.tar.gz" ] && NODEHASH="b2c17574a81b290952a8535bb93ac90335421c9c69c12f13ceddb471fe0243f3"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.8/node-v0.8.8-sunos-x64.tar.gz" ] && NODEHASH="b94b63c97bdaa963ce5945a75d9bc73f694eca29df7be3a0070713e03acea295"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.8/node-v0.8.8-sunos-x86.tar.gz" ] && NODEHASH="e5fe9ecbc4dbbe65558cfb8d01678515ce3bb98573fff7a43810f99e404cd33e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.8/node-v0.8.8.tar.gz" ] && NODEHASH="092b7045b8e956f838a2a3da36cdcf7954e9e0d16fb88b14e2b7d090422e3133"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.9/node-v0.8.9-darwin-x64.tar.gz" ] && NODEHASH="c41002cac64f4198e55ed6a78f2e321e721c6150854d3ca27603cb912782c04e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.9/node-v0.8.9-darwin-x86.tar.gz" ] && NODEHASH="b7c67a983777d86fe21395ed2d0fe619364b27788e507bb8249eff56b57e2352"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.9/node-v0.8.9-linux-x64.tar.gz" ] && NODEHASH="ee785197324d0a0335de12a2be124b2eaf49865fc791d600651edff3d3831aea"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.9/node-v0.8.9-linux-x86.tar.gz" ] && NODEHASH="ed23dd5875c99a687640dfc0826b489c444a4bf494bd6e40788aab1230d0b4a1"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.9/node-v0.8.9-sunos-x64.tar.gz" ] && NODEHASH="dbc20cfc4d665497d772d8c1a7f4140f63617cfeab83ac73649d1914fabf9c90"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.9/node-v0.8.9-sunos-x86.tar.gz" ] && NODEHASH="e93a229d09e5032aa7add4a2c255b9aea53ff634419956d732e0f6e154fcd896"
[ "$NODEURL" = "http://nodejs.org/dist/v0.8.9/node-v0.8.9.tar.gz" ] && NODEHASH="320f06877c5e4b4dcc407c76c4d6dcf24384211c2ee22f8bc794a8ec898136ba"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.0/node-v0.9.0.tar.gz" ] && NODEHASH="4d2e5d7c8b345f6e401eed7d06b4bbc6cb012aefc34b46e7c3aedb4a0fccd258"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.1/node-v0.9.1-darwin-x64.tar.gz" ] && NODEHASH="1758e8bcb881e507506e743e3d23d3f1bc8101330492f7d8f0818f1ac4c6317f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.1/node-v0.9.1-darwin-x86.tar.gz" ] && NODEHASH="33cfbbdb5aa14bd0c9ee8a6b424af3421c1197586c7a6a7fa290c1f3b096c5af"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.1/node-v0.9.1-linux-x64.tar.gz" ] && NODEHASH="094afa13511cac6638f69b100ca4990392ea5430f2277b7acd93a1f2378e3b86"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.1/node-v0.9.1-linux-x86.tar.gz" ] && NODEHASH="3102a440c6f6956216baaf1474ca41616477c2dc95cc2d73fc6ce765b3984cac"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.1/node-v0.9.1-sunos-x64.tar.gz" ] && NODEHASH="971283d3b205dd624d750ecf66f45faf8738cae2e05cab24edcf68188bab5d0b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.1/node-v0.9.1-sunos-x86.tar.gz" ] && NODEHASH="3b7cd36578e3703bd3c4141e3af389598c954e180985aac5d33035c698fd1e53"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.1/node-v0.9.1.tar.gz" ] && NODEHASH="12bc0deb1a0c3fdcd5c54ffd241c1e291d372620944c3f97388d38f460f222b9"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.10/node-v0.9.10-darwin-x64.tar.gz" ] && NODEHASH="71d26152e960d85918001c1c8db7a66bbbc3293edd58f75a63d649fbc3c14d20"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.10/node-v0.9.10-darwin-x86.tar.gz" ] && NODEHASH="1ce7bd920e6a6fdbe24ed9c55a8516c7b813785f08ed5312c2969aa3a7bf56d9"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.10/node-v0.9.10-linux-arm-pi.tar.gz" ] && NODEHASH="438f7df584667d54e4756269d78b413d1188600b1d00c60298cdbeb9406c58ec"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.10/node-v0.9.10-linux-x64.tar.gz" ] && NODEHASH="7983aed8394a7f159492bd715e0ea3ad0d4d22b2be388b7f9c59b4945d282188"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.10/node-v0.9.10-linux-x86.tar.gz" ] && NODEHASH="491668b4427656722405ffc88547c09ca65aca0b3039f0d273a1aa5cc20ac62d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.10/node-v0.9.10-sunos-x64.tar.gz" ] && NODEHASH="99ae3de4362861a0eec4c181fb42d585e72fd4ad30e61ff894702fd2eb0d46d0"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.10/node-v0.9.10-sunos-x86.tar.gz" ] && NODEHASH="3623fbe87783fff3508ddd042b9dd23a2ea76ea0e711463dea0a085cda847a0d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.10/node-v0.9.10.tar.gz" ] && NODEHASH="efdb49fc4ce2b4876c2fac00f5a2d8458f6aba42fa85001b0cb01f7a4b234ec9"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.11/node-v0.9.11-darwin-x64.tar.gz" ] && NODEHASH="20653e45e6e1ab3e4ef639ef7072b51fe858616cff27fca105f61a5bba26d434"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.11/node-v0.9.11-darwin-x86.tar.gz" ] && NODEHASH="3e903a5d410122d7819952d85b5fb88af9d75cb6146c42b08919d9881f202c12"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.11/node-v0.9.11-linux-arm-pi.tar.gz" ] && NODEHASH="2a63fe7655a4ac6a795c976ec8f94580a198273344a2f446c442dea1c64dd448"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.11/node-v0.9.11-linux-x64.tar.gz" ] && NODEHASH="511d69123ca08c76692e6c5baf1d7d7dc702993a59bd37ebfd4976b4d833de98"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.11/node-v0.9.11-linux-x86.tar.gz" ] && NODEHASH="97e56ce5418cf136fdfbf6b3ef456d0b7e7e776919c53b4ff15ce6846d226279"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.11/node-v0.9.11-sunos-x64.tar.gz" ] && NODEHASH="c84a2db0e0839400c4ef4ee7de1b9a3cbc44d7bc66beda324497227019d9f44d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.11/node-v0.9.11-sunos-x86.tar.gz" ] && NODEHASH="eb793e5c212571a434b63afd92618fcd6da1b41e325bf6153550dce5a2871389"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.11/node-v0.9.11.tar.gz" ] && NODEHASH="2e22a9ecd838527f69d31e35fa5781d4b8099e3b33891b5f7d7e46f2291d6988"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.12/node-v0.9.12-darwin-x64.tar.gz" ] && NODEHASH="3e9db5d098688b4caa80c3de29f6623ff7c8420e43cb10d3a2b25d50dcf0e1db"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.12/node-v0.9.12-darwin-x86.tar.gz" ] && NODEHASH="688c75eae59835335f537c4effa96a2ea51b0df2a630ef5f410f5796f71e8acd"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.12/node-v0.9.12-linux-arm-pi.tar.gz" ] && NODEHASH="8b44c62a58e8a8b39328a536759b663e8e0710db32b0b48cc0f6217096af0b50"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.12/node-v0.9.12-linux-x64.tar.gz" ] && NODEHASH="bfd58129371ca6abff9adda83dd3d5dee44add6eb4a4c09f074521b97614a7b7"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.12/node-v0.9.12-linux-x86.tar.gz" ] && NODEHASH="c355f8f04f05f38a47b88cc279f65f592d205b33bf6ae7fa34120733d55cf32d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.12/node-v0.9.12-sunos-x64.tar.gz" ] && NODEHASH="b2dcc56ca896e9f0297ea3e41bb521e81cdea801dd6cb3ea1a4fe6b82be358f4"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.12/node-v0.9.12-sunos-x86.tar.gz" ] && NODEHASH="cf3f8db68ae33e1b07a7a0e8eb84cec116765347814188c04528903803a97bdc"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.12/node-v0.9.12.tar.gz" ] && NODEHASH="d1dbf425eb72b7c6d931f5fabf79b1e67ce1b2808c19c4b06a9aafe313ac2474"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.2/node-v0.9.2-darwin-x64.tar.gz" ] && NODEHASH="19d9a9c60f3a2fee9b910846daad9615ef761bbf6ce8dcbea372b725f5e470e8"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.2/node-v0.9.2-darwin-x86.tar.gz" ] && NODEHASH="b275e42626754b575eaea33a27fd29837db02ed28736495577f159e1c55133d4"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.2/node-v0.9.2-linux-x64.tar.gz" ] && NODEHASH="db2b1740f652ffdfeecb073d1ca91863783675f42c31c2fe6f7643da11268331"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.2/node-v0.9.2-linux-x86.tar.gz" ] && NODEHASH="fad058188d62f7f9605a496682943f6917f6afa25d50ec35c703cd8cd8743cd9"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.2/node-v0.9.2-sunos-x64.tar.gz" ] && NODEHASH="0d87e660f1cc57cfcb1a9c1cc85b4ed305f08f6d3463d6d6cdcfd537ca79138a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.2/node-v0.9.2-sunos-x86.tar.gz" ] && NODEHASH="7d47f6c89e9036247f754a80b47eb36ae1973f1f200b9a70cabc413933e17684"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.2/node-v0.9.2.tar.gz" ] && NODEHASH="84dc31888a5d53a0188ebc04ef4b84488b2f0361dd4696b7ae047af7372102c1"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.3/node-v0.9.3-darwin-x64.tar.gz" ] && NODEHASH="d2ceeba6ea15f02cdefa6a91181bbe931452f5ddd69f9594d4bed57c6eeb95c9"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.3/node-v0.9.3-darwin-x86.tar.gz" ] && NODEHASH="c49d32771ddb2d2f0b03db6ba7e698d3d91f7d5cab512cb1b867cd93cef79ce0"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.3/node-v0.9.3-linux-x64.tar.gz" ] && NODEHASH="fa92deff7cffd10130520245c1bc724a22c61d89e50d0e232af5c51eeb80dcfe"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.3/node-v0.9.3-linux-x86.tar.gz" ] && NODEHASH="8d5efd768da36f5d9252b6b335df73df33920dffdbd0c18a6656ec3e578c895c"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.3/node-v0.9.3-sunos-x64.tar.gz" ] && NODEHASH="fc34d3614624f27b3801f4c2f3019f6cd2c7ad5b05d97d8dd28524d9ff4c4ebe"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.3/node-v0.9.3-sunos-x86.tar.gz" ] && NODEHASH="df4cb58d17cb545dd8a91e49b1f270fa1dd4f1b2cd81d6afd8ec24f0ce1c6049"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.3/node-v0.9.3.tar.gz" ] && NODEHASH="7e1750cd47d7b8c13c7cf12457b6a528fa2abf8a10b7c9a35c13ed47cebaab41"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.4/node-v0.9.4-darwin-x64.tar.gz" ] && NODEHASH="a01c42debabffd9b99ed7eef89c8bb87e08bc3e40e147b3eb809be469ff0c947"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.4/node-v0.9.4-darwin-x86.tar.gz" ] && NODEHASH="19e72f86bbb610ec8dc0182e8276c0efd73cf75dd3097c4276730bf6176f4775"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.4/node-v0.9.4-linux-x64.tar.gz" ] && NODEHASH="10daa5991f2e82874ac5a2ad3c223dea4e7f2f67587825135396906edbfb16d1"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.4/node-v0.9.4-linux-x86.tar.gz" ] && NODEHASH="f7d1919ad8714b2893c207286ae15f6b4ac91e5472b864d42c5203863329212f"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.4/node-v0.9.4-sunos-x64.tar.gz" ] && NODEHASH="fd52e94f51f78b8d55d450cf8c389ee5e2e283d4a0deb602c3bb10964181539d"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.4/node-v0.9.4-sunos-x86.tar.gz" ] && NODEHASH="1cac9ed70d26d3465efb45c4ca2ff4d99af786407fe44dbfef135a0ce80ea64a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.4/node-v0.9.4.tar.gz" ] && NODEHASH="6978981caceff7b83a23a6e3c89dba5734d27a623d2c3d26ecd77e41f2bd39f7"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.5/node-v0.9.5-darwin-x64.tar.gz" ] && NODEHASH="490890fa1259aaa5e0aa66a4aae432831c10ef20729c1dee235e86579af2615e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.5/node-v0.9.5-darwin-x86.tar.gz" ] && NODEHASH="0a7aba386032fae9b47a4cd7ee6196c0febc227aac50e47cb7b71b837e53a1fe"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.5/node-v0.9.5-linux-x64.tar.gz" ] && NODEHASH="e6a28d6946bd15132b6af44e00e91c15b7fabfb66713189449b71b2ccf2a3123"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.5/node-v0.9.5-linux-x86.tar.gz" ] && NODEHASH="6258273d0eeefde82286344fb81da7b6ab457de0b57b71eda2dd8b5b1b06e370"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.5/node-v0.9.5-sunos-x64.tar.gz" ] && NODEHASH="c4cf7cedcf151b263a9adca424deba0d21312dc61ac654ec5b77ff277dca1844"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.5/node-v0.9.5-sunos-x86.tar.gz" ] && NODEHASH="1750c8713ac29814a09d470156029e1841b3bc6d210a5d3bdfebf06b6104ae4b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.5/node-v0.9.5.tar.gz" ] && NODEHASH="0235bffe94b6e1fca312436e6b8cf62e6d35fe02c752d0c57e4af9ae63e021d1"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.6/node-v0.9.6-darwin-x64.tar.gz" ] && NODEHASH="e069593a5a97a390ce32443cdd4a385424281a1bbb1b72382763c5ac9834c70b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.6/node-v0.9.6-darwin-x86.tar.gz" ] && NODEHASH="0d7f55021174c1e82e43ac2267ea2ead7b121d07229a69b11f18f960ccea986e"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.6/node-v0.9.6-linux-x64.tar.gz" ] && NODEHASH="739e20d6bc3fefe50b07dd01fa524389b846b6b6caa31a1b2e1ca0246d8f7c56"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.6/node-v0.9.6-linux-x86.tar.gz" ] && NODEHASH="cc88985e3b9b4a0e9d679ec1a89586666617d1c66c98e5e8bbd7380b4b552d92"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.6/node-v0.9.6-sunos-x64.tar.gz" ] && NODEHASH="d9bf25de31b127535f98b109a0481f1c1fcae97aca783b2f3b457205784adbd3"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.6/node-v0.9.6-sunos-x86.tar.gz" ] && NODEHASH="2d36b537597f0c20ab95a14b244c9093e8949c71c522f361bd48f6e97f2d1a0a"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.6/node-v0.9.6.tar.gz" ] && NODEHASH="e314c69555a53241eb809132b8d76b6843d44b7f8afb68cf2f60fcf86e05c916"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.7/node-v0.9.7-darwin-x64.tar.gz" ] && NODEHASH="c7904122355d9756a72c8a49ecf5c3bc23d76d319b25c18616a9c611179aea17"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.7/node-v0.9.7-darwin-x86.tar.gz" ] && NODEHASH="489691e519c60ca428c7613c242fba8b273d0ccddeb5862c01cd16361956a364"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.7/node-v0.9.7-linux-arm-pi.tar.gz" ] && NODEHASH="344d59cb2a92b2bbf95c727baa1bd4a70f9930785d7be8b1ca3cbd49e44a0cb4"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.7/node-v0.9.7-linux-x64.tar.gz" ] && NODEHASH="9bd95268508a59127d901164f1375d3c8908df858cfc781a312fa95410e59679"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.7/node-v0.9.7-linux-x86.tar.gz" ] && NODEHASH="bfc037be2bd1c62e5167565cfd82459155aa5cac639e1ade06f7e71f0377e6af"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.7/node-v0.9.7-sunos-x64.tar.gz" ] && NODEHASH="a5eb8ccf2dae8cc5d532f244ee28621f52535891e321313a9b0afd8f4d6cc737"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.7/node-v0.9.7-sunos-x86.tar.gz" ] && NODEHASH="7533414da857d30e3cd9747b0dfc769fb34c393af80b0935f0877a4bcc6e29ad"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.7/node-v0.9.7.tar.gz" ] && NODEHASH="c05fe0a304d3a7edc2ace4a9ce8e4c37d29332d908d8b578b075d2ac2c2b71b8"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.8/node-v0.9.8-darwin-x64.tar.gz" ] && NODEHASH="370fbeb63a187b6519464d2f7f2ac0f808ea7fe9810dcc74545365488e5627b4"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.8/node-v0.9.8-darwin-x86.tar.gz" ] && NODEHASH="aacd484f35b828266697824053ad8ec8d6316cf4054ca5a53e26ee9c87581789"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.8/node-v0.9.8-linux-arm-pi.tar.gz" ] && NODEHASH="d163a85689e3e62cfcb5ab5e083b485441178933cecae93fb76229d20b6d84f2"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.8/node-v0.9.8-linux-x64.tar.gz" ] && NODEHASH="673e8d79e25d9312725a12c2a10f1c313beffd558e2a49553ac0837a5ab0430b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.8/node-v0.9.8-linux-x86.tar.gz" ] && NODEHASH="c2f5c0b9ecf1f94ac5ee284372e1a4369951c89f893dfcaacc795a4444baaefb"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.8/node-v0.9.8-sunos-x64.tar.gz" ] && NODEHASH="654dd17df28a6ac7704c7b977cf906580153830553dd8a2ed684b3b70cbcf732"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.8/node-v0.9.8-sunos-x86.tar.gz" ] && NODEHASH="2d767fa5502e4b24ba85feb32f6918833a4654b7b2d0236cbf9e039dc2c5a145"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.8/node-v0.9.8.tar.gz" ] && NODEHASH="b7ad7d9f02d5eaca73a288a261578965c67d3847c2392ee09527acf72549b2e5"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.9/node-v0.9.9-darwin-x64.tar.gz" ] && NODEHASH="e1a886746b4afa453e08505284d0c11c765299651135fa11d01b0ffb3493e2fc"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.9/node-v0.9.9-darwin-x86.tar.gz" ] && NODEHASH="771cd29b2e9c026567bd10196636908404589d95a83565026f1b9871d5719ea1"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.9/node-v0.9.9-linux-arm-pi.tar.gz" ] && NODEHASH="af4250daf3d05a1511c76748c42d40a888ac96bb48e79e219b2e3cbeac17dec5"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.9/node-v0.9.9-linux-x64.tar.gz" ] && NODEHASH="76a1b5d17a0bf4de47982c33f9f7e1f1517c4de3defc042cad6f0ef8eac4dc90"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.9/node-v0.9.9-linux-x86.tar.gz" ] && NODEHASH="aaf48db58ead0831131c0e4a1e126f0f3671da3f1c5601c30e8882f638680990"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.9/node-v0.9.9-sunos-x64.tar.gz" ] && NODEHASH="5c74ea6c3c07cb8e7b447c8a6f03b32dff81139d34610bffe651cbd38f8c60fc"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.9/node-v0.9.9-sunos-x86.tar.gz" ] && NODEHASH="492feecab9ed2062ce645027229a8fb05de29a71382afa6ac4fb792342fb234b"
[ "$NODEURL" = "http://nodejs.org/dist/v0.9.9/node-v0.9.9.tar.gz" ] && NODEHASH="d285559044db3e340ab6403d7508943fbec4cc166f1ad96a00ac441794ce8a88"
[ "$NODEURL" = "http://nodejs.org/dist/v4.0.0/node-v4.0.0-darwin-x64.tar.gz" ] && NODEHASH="4893cadaa55fe88b214371b23257d7fae4b30661ef01d352e3f930ba9d402ee0"
[ "$NODEURL" = "http://nodejs.org/dist/v4.0.0/node-v4.0.0-headers.tar.gz" ] && NODEHASH="8812b88e81ee258f7f9446b548f6b364edcdd455f73f06589f14dd71dca415d0"
[ "$NODEURL" = "http://nodejs.org/dist/v4.0.0/node-v4.0.0-linux-arm64.tar.gz" ] && NODEHASH="0436f107e1d82c61c3ee4f916781466a49bece74bf4d4fb4bf4d53a57b81df85"
[ "$NODEURL" = "http://nodejs.org/dist/v4.0.0/node-v4.0.0-linux-armv6l.tar.gz" ] && NODEHASH="6f88e532bbf55cff36d4940898b80359761de3a274cc214fbb58cc176ec49ba2"
[ "$NODEURL" = "http://nodejs.org/dist/v4.0.0/node-v4.0.0-linux-armv7l.tar.gz" ] && NODEHASH="a788ed1b04de00344f4bbb8a278b317c69f9672f5fd03adb24f53d7c3d6fc8f6"
[ "$NODEURL" = "http://nodejs.org/dist/v4.0.0/node-v4.0.0-linux-x64.tar.gz" ] && NODEHASH="df8ada31840e3dc48c7fe7291c7eba70b2ce5a6b6d959ac01157b04731c8a88f"
[ "$NODEURL" = "http://nodejs.org/dist/v4.0.0/node-v4.0.0-linux-x86.tar.gz" ] && NODEHASH="444b0c2e38602699103e4adb76c9f15b76c405f58b1308751cdc194041a39b93"
[ "$NODEURL" = "http://nodejs.org/dist/v4.0.0/node-v4.0.0-sunos-x64.tar.gz" ] && NODEHASH="f17ea8b1ffa6d63ba4eaa37708a365e5aefdb7de136f8b1a5751f2f258507026"
[ "$NODEURL" = "http://nodejs.org/dist/v4.0.0/node-v4.0.0-sunos-x86.tar.gz" ] && NODEHASH="4537640f912b1db217351271e15d092d4654ecbe7cdb0169313988de5e453cd3"
[ "$NODEURL" = "http://nodejs.org/dist/v4.0.0/node-v4.0.0.tar.gz" ] && NODEHASH="e110e5a066f3a6fe565ede7dd66f3727384b9b5c5fbf46f8db723d726e2f5900"
[ "$NODEURL" = "http://nodejs.org/dist/v4.1.0/node-v4.1.0-darwin-x64.tar.gz" ] && NODEHASH="ff1c4b13a10c0789709cd73980a496b27e434207802989377ec4cb56302a1ebc"
[ "$NODEURL" = "http://nodejs.org/dist/v4.1.0/node-v4.1.0-headers.tar.gz" ] && NODEHASH="1ba579ead4cc6a70b1d74bbc3591ca9c9c46e4254ff6cebc9dedfaf98e1657aa"
[ "$NODEURL" = "http://nodejs.org/dist/v4.1.0/node-v4.1.0-linux-arm64.tar.gz" ] && NODEHASH="d27001f51d75c43cc1f444eab8aef0ced4ac4d162598be7eccf58790127e5368"
[ "$NODEURL" = "http://nodejs.org/dist/v4.1.0/node-v4.1.0-linux-armv6l.tar.gz" ] && NODEHASH="6c5a7b6a37b694fb1b1636ce8e17dfe74f70287666cfee08344d33b52a1eca0e"
[ "$NODEURL" = "http://nodejs.org/dist/v4.1.0/node-v4.1.0-linux-armv7l.tar.gz" ] && NODEHASH="3b728cdb93d155cf532a8a7f57f5a86f6f67a73e57a7abe657ff88166995e99e"
[ "$NODEURL" = "http://nodejs.org/dist/v4.1.0/node-v4.1.0-linux-armv7.tar.gz" ] && NODEHASH="3b728cdb93d155cf532a8a7f57f5a86f6f67a73e57a7abe657ff88166995e99e"
[ "$NODEURL" = "http://nodejs.org/dist/v4.1.0/node-v4.1.0-linux-x64.tar.gz" ] && NODEHASH="7c6055e08127143d9a8f779aa56f3fe42035fff8843c2652b0b2726204556382"
[ "$NODEURL" = "http://nodejs.org/dist/v4.1.0/node-v4.1.0-linux-x86.tar.gz" ] && NODEHASH="2ceaf5e66993da50e4514ddc5c270dc15ee712d00c29a2108655d9d259b3b9bb"
[ "$NODEURL" = "http://nodejs.org/dist/v4.1.0/node-v4.1.0-sunos-x64.tar.gz" ] && NODEHASH="b9b176f4e54422ab73458dcc36eec34149ffb298a9f5a5aa042c4b7e59860113"
[ "$NODEURL" = "http://nodejs.org/dist/v4.1.0/node-v4.1.0-sunos-x86.tar.gz" ] && NODEHASH="ed6c5163b3cb2e452e8e9bf4e3857af9e65eccd7fce0959fb97dcf03547c726a"
[ "$NODEURL" = "http://nodejs.org/dist/v4.1.0/node-v4.1.0.tar.gz" ] && NODEHASH="453005f64ee529f7dcf1237eb27ee2fa2415c49f5c9e7463e8b71fba61c5b408"
[ "$NODEURL" = "http://nodejs.org/dist/v4.1.1/node-v4.1.1-darwin-x64.tar.gz" ] && NODEHASH="b7e72bf8364c35992a8bebc57bf68c596d622c33d409c0943bf7d24ca7205c76"
[ "$NODEURL" = "http://nodejs.org/dist/v4.1.1/node-v4.1.1-headers.tar.gz" ] && NODEHASH="d134614e78cfc406611e366c9618704a47c2ee1bf60f0a11909ba84d8b2a9e28"
[ "$NODEURL" = "http://nodejs.org/dist/v4.1.1/node-v4.1.1-linux-arm64.tar.gz" ] && NODEHASH="b2e1915a0c65dd9faee7f05a56792371958980e02d1f7cde447c8260bb805052"
[ "$NODEURL" = "http://nodejs.org/dist/v4.1.1/node-v4.1.1-linux-armv6l.tar.gz" ] && NODEHASH="ca38cef96180916891a262bbb39f335eaa8de6c0c06933609f4f3d7bebdc94b5"
[ "$NODEURL" = "http://nodejs.org/dist/v4.1.1/node-v4.1.1-linux-armv7l.tar.gz" ] && NODEHASH="2896f0ab7c53bb7b489a09f7344e059f898ae929c2a9bfb7dfce85a5846ab9d2"
[ "$NODEURL" = "http://nodejs.org/dist/v4.1.1/node-v4.1.1-linux-x64.tar.gz" ] && NODEHASH="f5f7e11a503c997486d50d8683741a554bdda1d1181125a05ac5844cb29d1572"
[ "$NODEURL" = "http://nodejs.org/dist/v4.1.1/node-v4.1.1-linux-x86.tar.gz" ] && NODEHASH="3f9836b8a7e6e3d6591af6ef59e6055255439420518c3f77e0e65832a8486be1"
[ "$NODEURL" = "http://nodejs.org/dist/v4.1.1/node-v4.1.1-sunos-x64.tar.gz" ] && NODEHASH="e1e991519f4147ccef0c1816d26905ccf0a0be094af08d302a63e1025a7369df"
[ "$NODEURL" = "http://nodejs.org/dist/v4.1.1/node-v4.1.1-sunos-x86.tar.gz" ] && NODEHASH="915ec11b4a64becd817a810b7d8ecb426da3c52465d3ac3dfae50b53ad1ea28c"
[ "$NODEURL" = "http://nodejs.org/dist/v4.1.1/node-v4.1.1.tar.gz" ] && NODEHASH="6a610935ff52de713cf2af6a26002322e24fd7933a444436f0817a2b84e15a58"
[ "$NODEURL" = "http://nodejs.org/dist/v4.1.2/node-v4.1.2-darwin-x64.tar.gz" ] && NODEHASH="058a5b07c9bda34074f4acad75ce03319402592a0482221f574e2fc4aed60131"
[ "$NODEURL" = "http://nodejs.org/dist/v4.1.2/node-v4.1.2-headers.tar.gz" ] && NODEHASH="e8e17c8dc5afbf42d9533b37d32d7b44f105c425f13be4840c9e20cfcd251e78"
[ "$NODEURL" = "http://nodejs.org/dist/v4.1.2/node-v4.1.2-linux-arm64.tar.gz" ] && NODEHASH="ae74c245b9592d52f8632a249a0bdd2eb664dcf7aaf5089d061f9c5b051f101a"
[ "$NODEURL" = "http://nodejs.org/dist/v4.1.2/node-v4.1.2-linux-armv6l.tar.gz" ] && NODEHASH="834b36fa7e397f27f63f8fd56b77c15de4d9297c23f70679d0ceabd83662d950"
[ "$NODEURL" = "http://nodejs.org/dist/v4.1.2/node-v4.1.2-linux-armv7l.tar.gz" ] && NODEHASH="a8e38b1fae35c9b13f0c8cc57991d20a0267603c820028290236508930a522b0"
[ "$NODEURL" = "http://nodejs.org/dist/v4.1.2/node-v4.1.2-linux-x64.tar.gz" ] && NODEHASH="c39aefac81a2a4b0ae12df495e7dcdf6a8b75cbfe3a6efb649c8a4daa3aebdb6"
[ "$NODEURL" = "http://nodejs.org/dist/v4.1.2/node-v4.1.2-linux-x86.tar.gz" ] && NODEHASH="843599c40d3aabb1f60e33144647e6ead92bcfcefa7efca8396651ab92b5f3b9"
[ "$NODEURL" = "http://nodejs.org/dist/v4.1.2/node-v4.1.2-sunos-x64.tar.gz" ] && NODEHASH="5a783eb6b4feaee29f447fe6ea4bfd1b745bee2f00b48d5bbb78f6a2a6488e59"
[ "$NODEURL" = "http://nodejs.org/dist/v4.1.2/node-v4.1.2-sunos-x86.tar.gz" ] && NODEHASH="33719d820fca0a0b44f55ecb9d9de0ce8359211a98d802a3191dfb4e0e5f2c5c"
[ "$NODEURL" = "http://nodejs.org/dist/v4.1.2/node-v4.1.2.tar.gz" ] && NODEHASH="febec3703a3fea9b64c46100a7b6a4408250027aa8259a2a815fd2275d502c49"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.0/node-v4.2.0-darwin-x64.tar.gz" ] && NODEHASH="078c9a2c2fbfb603700d94b618df3b8b211e10f361bced5e4439142715aebde5"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.0/node-v4.2.0-headers.tar.gz" ] && NODEHASH="b82bd1c46b49599e5ee6565b0b4ecb5ef6f3af573f897f9df2470eafe255feed"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.0/node-v4.2.0-linux-arm64.tar.gz" ] && NODEHASH="ae67d6ddcb72505937982d6a0bcb031b3522f7abc11d9e2d449ea021a5f75faf"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.0/node-v4.2.0-linux-armv6l.tar.gz" ] && NODEHASH="449b0914e195ad39db7fb497b1c6a3f7409c5c437794cabbdb962d8641f1e2f6"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.0/node-v4.2.0-linux-armv7l.tar.gz" ] && NODEHASH="c1844dd131020b2b2982de974d73b051159776fe3ebf6fa91f08a2f5e8c46555"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.0/node-v4.2.0-linux-x64.tar.gz" ] && NODEHASH="98b60c86d541f44c5d07111f9ffc9a81848f976cfbe84f71cb9c7d6bfd34e1a4"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.0/node-v4.2.0-linux-x86.tar.gz" ] && NODEHASH="bc5e0791c23d897ea6c83976dd14eec8ac33f74188806012992aca335c4e9214"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.0/node-v4.2.0-sunos-x64.tar.gz" ] && NODEHASH="d074ae0ab26f2f680adf7b94cfae0a8eac5ae4dcf083594a361b837fea6fdc72"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.0/node-v4.2.0-sunos-x86.tar.gz" ] && NODEHASH="fc753685f937d338f1367614ca49158d446e29b35694fac6b8e6be262db5ec38"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.0/node-v4.2.0.tar.gz" ] && NODEHASH="94ea40efeb598fed6e1a78990b698da73275c49876ec551dc36e11513c99fb6b"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.1/node-v4.2.1-darwin-x64.tar.gz" ] && NODEHASH="658686795fac9669d411ac5c5be2da8159058e386767322d8d8151dcdb4810b9"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.1/node-v4.2.1-headers.tar.gz" ] && NODEHASH="b242fd0fa83748d95cc3e68d14579b629960b0436d65da9d86907eef27b96fc1"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.1/node-v4.2.1-linux-arm64.tar.gz" ] && NODEHASH="05df4aeb8a53798f8b10074600518040fc317f2919f9755aeab57b0aaf7227b0"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.1/node-v4.2.1-linux-armv6l.tar.gz" ] && NODEHASH="6f29286464efda4a362d9464d82733398b98882051c20c0f190004480b6e506a"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.1/node-v4.2.1-linux-armv7l.tar.gz" ] && NODEHASH="fb4fbef9306962e800804ab5ba615c06bba28deb4e7a3e945a291dba986ef816"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.1/node-v4.2.1-linux-x64.tar.gz" ] && NODEHASH="e766e387934e17daaad92d0460ed76f756655da62b627a5c9cc07faea4a0b824"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.1/node-v4.2.1-linux-x86.tar.gz" ] && NODEHASH="97b5ccea7044073c87a21bcc4b0762f4a6bd77db9cc958206f684ecdfeb89b1f"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.1/node-v4.2.1-sunos-x64.tar.gz" ] && NODEHASH="acec44790ff0069620c0fd03945d14b9f97c7ccb0a0450f766a5cbe4a906510b"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.1/node-v4.2.1-sunos-x86.tar.gz" ] && NODEHASH="d7bf8dcf353115b5e55bf64d25c2c34fa2d237e201e46dab97ae3d8e3f051583"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.1/node-v4.2.1.tar.gz" ] && NODEHASH="8861b9f4c3b4db380fcda19a710c0430c3d62d03ee176c64db63eef95a672663"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.2/node-v4.2.2-darwin-x64.tar.gz" ] && NODEHASH="17e60f59cfd40bd0a74f62106dfe2d6cf14aea53490c5e872fa7530138d0dfef"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.2/node-v4.2.2-headers.tar.gz" ] && NODEHASH="787a89963b9b0ea41f7d4964705ed3561d16d023e1b5aae80700f973ba7ec4e7"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.2/node-v4.2.2-linux-arm64.tar.gz" ] && NODEHASH="125b2236da6c0c7fed39b6652b3867646200c3fe7c9284df3f8ac07a84d39010"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.2/node-v4.2.2-linux-armv6l.tar.gz" ] && NODEHASH="dfe30ab4dbfb66c31259c0b176609f2dbace29f03d3cf12c5d8bee3dd25db035"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.2/node-v4.2.2-linux-armv7l.tar.gz" ] && NODEHASH="2170dea2d74b6a1c5c82be6b93a9b84ad5a78cafa47415661095be2b358d964a"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.2/node-v4.2.2-linux-x64.tar.gz" ] && NODEHASH="5c39fac55c945be3b8ac381a12bdbe3a64a9bdc5376d27e2ce0c72160eff5942"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.2/node-v4.2.2-linux-x86.tar.gz" ] && NODEHASH="7fe7d716cf52db39fa0590885e2f39a0bdee9ecbba230621f042c8154e8100fe"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.2/node-v4.2.2-sunos-x64.tar.gz" ] && NODEHASH="23d47dee4057a4105c53158b07c2362ad0e962332e9a939c841de9867e541da3"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.2/node-v4.2.2-sunos-x86.tar.gz" ] && NODEHASH="ed69c2a73a55100c044ae16f244119e889d502722ada7df8640793c1fb1a5195"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.2/node-v4.2.2.tar.gz" ] && NODEHASH="00e709353435b436b5bbf5e62272ebb15fe801065422c4c2afe2b93456210cb1"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.3/node-v4.2.3-darwin-x64.tar.gz" ] && NODEHASH="078b033d98367da2248b443ed74f0b8a5808783d07cf0c24884457fd66b68fc1"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.3/node-v4.2.3-headers.tar.gz" ] && NODEHASH="e8fab75b9183a4f35c358813e3c5d451daf2fef737a21494f7aa64bdd538c87c"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.3/node-v4.2.3-linux-arm64.tar.gz" ] && NODEHASH="9ec1becd52959920a0f06c92f01b3c3e8c09dd35b4b4f591d975f1975a5f1689"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.3/node-v4.2.3-linux-armv6l.tar.gz" ] && NODEHASH="50b158bc4c324e78bc5ceb08e2d5e6aefb34f2570798e7ac4fd12bacd6733478"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.3/node-v4.2.3-linux-armv7l.tar.gz" ] && NODEHASH="bebe529dd9ef576193cd7ef40f3f8a16f51317251b624f5e6a9998861b1778f3"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.3/node-v4.2.3-linux-x64.tar.gz" ] && NODEHASH="644d4c0b206ebcb75383fbe42f6025e7253a61992816289359d0f4dcdb6087d7"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.3/node-v4.2.3-linux-x86.tar.gz" ] && NODEHASH="fd30b15327348520dc840bb95a1441d0495e7a843d5a7ec102f012aedc1b3a75"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.3/node-v4.2.3-sunos-x64.tar.gz" ] && NODEHASH="28096b317320bec8d40263975bdc3fdd1b77c941304abcb3fd6c106970b3a936"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.3/node-v4.2.3-sunos-x86.tar.gz" ] && NODEHASH="b37e7652c5d0e08c6c2087e03c0fa860ab55b9c005db33c50373de891eb48dba"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.3/node-v4.2.3.tar.gz" ] && NODEHASH="5008ade5feb4b089f59163f66bffddc113f27de5d78edf203e39435c2c5d554f"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.4/node-v4.2.4-darwin-x64.tar.gz" ] && NODEHASH="3260412c210099b82a42316d614dec3313f4684a8ec63fca8684423675c37b0b"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.4/node-v4.2.4-headers.tar.gz" ] && NODEHASH="b1375e5296d0df6919fb1c3485d7174ac43a55a9507432d0e182c43c1cb5e3a0"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.4/node-v4.2.4-linux-arm64.tar.gz" ] && NODEHASH="96b5d86ca677ddfa351289ea78a338f3a66ef57ceb485cdddb4798ae33e52a08"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.4/node-v4.2.4-linux-armv6l.tar.gz" ] && NODEHASH="52ba11c9b9165ca9e6703bf60bcd0d35e36e30b474812b7d8ea1da51c9e293a9"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.4/node-v4.2.4-linux-armv7l.tar.gz" ] && NODEHASH="7d3645a032b56aefe1e1a023a6592b4900d4966312d034beac352bb833a74b60"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.4/node-v4.2.4-linux-x64.tar.gz" ] && NODEHASH="dcae0c0faf9841ef38953075e67ca477ef9d2ea7c14ac2221de2429813f83a62"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.4/node-v4.2.4-linux-x86.tar.gz" ] && NODEHASH="39c0d90eb65f38f24a4cd2939212b655ea0e8685b92b7de6f407d0df5d6c261c"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.4/node-v4.2.4-sunos-x64.tar.gz" ] && NODEHASH="0a90e813eb1e8bcbdbb55f9b3594f1a93e64bef710a7276bcf94dbf6de79e8f6"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.4/node-v4.2.4-sunos-x86.tar.gz" ] && NODEHASH="214178bc2dc35b96ff73c49f7908c509213aecf355859eb9a3f26a487959afaa"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.4/node-v4.2.4.tar.gz" ] && NODEHASH="4ee244ffede7328d9fa24c3024787e71225b7abaac49fe2b30e68b27460c10ec"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.5/node-v4.2.5-darwin-x64.tar.gz" ] && NODEHASH="c504aa1a0b7d7a51efe2403d72ef35a4742b5321fd23a2af52e06f9986a5ca70"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.5/node-v4.2.5-headers.tar.gz" ] && NODEHASH="01edb44794cfe365560b644b7c8f8431b3531f2c8ad0a47e91d7d0c8761b900f"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.5/node-v4.2.5-linux-arm64.tar.gz" ] && NODEHASH="b09a144acb67a8de6f873bbf05ccdf2accc1fd005127a5ab385f78a830fd3ddc"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.5/node-v4.2.5-linux-armv6l.tar.gz" ] && NODEHASH="bcc74b00760293b886624c844abf3d4502baab06c454a856b46b165bf5a48b78"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.5/node-v4.2.5-linux-armv7l.tar.gz" ] && NODEHASH="0146ba2fe72e81a4a682ac114cb58c802100717ed900172f3ec3664c8f3ee0f7"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.5/node-v4.2.5-linux-x64.tar.gz" ] && NODEHASH="3ed12d7ee8d911b87aed7f19640ec3d97a038330709ca4e0d137e8882b0f9cc4"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.5/node-v4.2.5-linux-x86.tar.gz" ] && NODEHASH="64e1564c0302503c98d70824b940c6679bfe38da74fa7bb387ee0f4665969031"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.5/node-v4.2.5-sunos-x64.tar.gz" ] && NODEHASH="3ce9a69bed1b15f251b6ca714901e45cb429a8e9e0ff9250a7735e7e58e942c9"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.5/node-v4.2.5-sunos-x86.tar.gz" ] && NODEHASH="8acc5a595e3d9bdcdfdb0cfcd7524266fc091306d570fc84cb8cbad5490ca091"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.5/node-v4.2.5.tar.gz" ] && NODEHASH="00162c5a8fcc5c35b27df26c49a83c7f4d52b1c963339a8a20401b81743f7fad"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.6/node-v4.2.6-darwin-x64.tar.gz" ] && NODEHASH="259ea77784013c1124506e3d90ee6847b2b9d3c066b6626ed62ebb31ed8e6fe3"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.6/node-v4.2.6-headers.tar.gz" ] && NODEHASH="a83ba921b951809aec81259a64c6d071c7ed443e8460898c42a8269c598930f9"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.6/node-v4.2.6-linux-arm64.tar.gz" ] && NODEHASH="f0aadf2941d91bfac449cdc8c904a926f6d384e45a8f6443a2f0153501753427"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.6/node-v4.2.6-linux-armv6l.tar.gz" ] && NODEHASH="61eb9963c188fb63565461032db1ccce1bae741f985b7f237043dea16ca85531"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.6/node-v4.2.6-linux-armv7l.tar.gz" ] && NODEHASH="104d75d428a77a23cc646757cc5624d22fd8705026cf712ef6b826368efeae6f"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.6/node-v4.2.6-linux-x64.tar.gz" ] && NODEHASH="656d8bff06cc5e108b83176f81de7e1eb16392ae0958ec4a7bca2a3a309333a1"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.6/node-v4.2.6-linux-x86.tar.gz" ] && NODEHASH="a96ce235e67e994aa5adec8255e13d4fbccd9c5b09b5e9554347dac2b1c5c8a1"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.6/node-v4.2.6-sunos-x64.tar.gz" ] && NODEHASH="96ccfeb7e15123111215d2093aa6c78abc7a43898c0a6c1838a56896e0b5493e"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.6/node-v4.2.6-sunos-x86.tar.gz" ] && NODEHASH="83505ceb87e584ff15f47a2ab25ee77378791be56b06cfe94fe24155f229e772"
[ "$NODEURL" = "http://nodejs.org/dist/v4.2.6/node-v4.2.6.tar.gz" ] && NODEHASH="ea5e357db8817052b17496d607c719d809ed1383e8fcf7c8ffc5214e705aefdd"
[ "$NODEURL" = "http://nodejs.org/dist/v4.3.0/node-v4.3.0-darwin-x64.tar.gz" ] && NODEHASH="72858adf7bc84b632f5635dd4a8030226048af5ad5cb166c7ee169dfcb4645ef"
[ "$NODEURL" = "http://nodejs.org/dist/v4.3.0/node-v4.3.0-headers.tar.gz" ] && NODEHASH="113a7c5689fd7a1d60160398a9cb0b03b0b88632ba0a2df148ef6e4c96986ec9"
[ "$NODEURL" = "http://nodejs.org/dist/v4.3.0/node-v4.3.0-linux-arm64.tar.gz" ] && NODEHASH="47a52191e264efdbc36f5ec6510abd71fd5d3337d75120c2ddc6a285873763b7"
[ "$NODEURL" = "http://nodejs.org/dist/v4.3.0/node-v4.3.0-linux-armv6l.tar.gz" ] && NODEHASH="2e035c649f72b5fbd712e6cf52e83e9f013b9a266dc907d3595b1c143c9906df"
[ "$NODEURL" = "http://nodejs.org/dist/v4.3.0/node-v4.3.0-linux-armv7l.tar.gz" ] && NODEHASH="49dfc4c4e5d1d07c91503c2a601665b68b6f5fc95d94517628f9a0f43b178158"
[ "$NODEURL" = "http://nodejs.org/dist/v4.3.0/node-v4.3.0-linux-x64.tar.gz" ] && NODEHASH="90ce6e23ad9748813742e1cf09e86fa4c0f3d53972d5dbe920a38bcc842e2d09"
[ "$NODEURL" = "http://nodejs.org/dist/v4.3.0/node-v4.3.0-linux-x86.tar.gz" ] && NODEHASH="6972ed77c36f026498a0fde6b237fbc554325fa3a7426ee17ce563bdc08caa69"
[ "$NODEURL" = "http://nodejs.org/dist/v4.3.0/node-v4.3.0-sunos-x64.tar.gz" ] && NODEHASH="c97723abd27c2b48ec2cbc9cc9b7dd057c96bf05c1895d7740414f84955753db"
[ "$NODEURL" = "http://nodejs.org/dist/v4.3.0/node-v4.3.0-sunos-x86.tar.gz" ] && NODEHASH="8f645328daa96702bf110cfcf2021620dc76fcb16ce80423a5f72dd23893cf71"
[ "$NODEURL" = "http://nodejs.org/dist/v4.3.0/node-v4.3.0.tar.gz" ] && NODEHASH="18504ac6d903cd061f60a29dafcda416a078112f3404d23a7901c41a8e9706b9"
[ "$NODEURL" = "http://nodejs.org/dist/v4.3.1/node-v4.3.1-darwin-x64.tar.gz" ] && NODEHASH="9c0751ee88a47c10269eb930d7ad7b103c2ba875c3a96204ca133dc52fc50826"
[ "$NODEURL" = "http://nodejs.org/dist/v4.3.1/node-v4.3.1-headers.tar.gz" ] && NODEHASH="8ba5c1e5eb5509e0f4f00d56e1916ac703fdd05cf353f119451f2b37c51987a5"
[ "$NODEURL" = "http://nodejs.org/dist/v4.3.1/node-v4.3.1-linux-arm64.tar.gz" ] && NODEHASH="cd55ce4426f9dd9be878fb89d715cbaf589210162e4269ce2ccfd6b9674385e9"
[ "$NODEURL" = "http://nodejs.org/dist/v4.3.1/node-v4.3.1-linux-armv6l.tar.gz" ] && NODEHASH="c42383509b7ad8a400da5b5605e2f8663cbe7e214a7c2cfa94964c57b1842ad8"
[ "$NODEURL" = "http://nodejs.org/dist/v4.3.1/node-v4.3.1-linux-armv7l.tar.gz" ] && NODEHASH="8fcf8016b01137d6ccb6d048227af30240a7a01cc40126b0f88f9e99ee40a552"
[ "$NODEURL" = "http://nodejs.org/dist/v4.3.1/node-v4.3.1-linux-x64.tar.gz" ] && NODEHASH="b3af1ed18a9150af42754e9a0385ecc4b4e9b493fcf32bf6ca0d7239d636254b"
[ "$NODEURL" = "http://nodejs.org/dist/v4.3.1/node-v4.3.1-linux-x86.tar.gz" ] && NODEHASH="b0ac5a9ec4c03b8e119498addd70c3d3ab030abe8d9ca1f120627e6ee2673acc"
[ "$NODEURL" = "http://nodejs.org/dist/v4.3.1/node-v4.3.1-sunos-x64.tar.gz" ] && NODEHASH="f5d51b0d4324e88db468b20ddd3a3790d9dcaf10fe977e328e5d39976028573c"
[ "$NODEURL" = "http://nodejs.org/dist/v4.3.1/node-v4.3.1-sunos-x86.tar.gz" ] && NODEHASH="32d2369d00a50661c0ace7dd4d99105bda36430b27c8be129731bb44ac3064b0"
[ "$NODEURL" = "http://nodejs.org/dist/v4.3.1/node-v4.3.1.tar.gz" ] && NODEHASH="61e2d58e861b6c9dbf0ac7624b718198cbd59b88a9de31a39a05de95a32eee73"
[ "$NODEURL" = "http://nodejs.org/dist/v4.3.2/node-v4.3.2-darwin-x64.tar.gz" ] && NODEHASH="6380a36500de02a48758f204c1c5931af986d93609cfa06db1f91216bec9a5eb"
[ "$NODEURL" = "http://nodejs.org/dist/v4.3.2/node-v4.3.2-headers.tar.gz" ] && NODEHASH="f532e7cd3886ac9fdc774cc2add0db21483052c966a23aace7c84ec69b91981f"
[ "$NODEURL" = "http://nodejs.org/dist/v4.3.2/node-v4.3.2-linux-arm64.tar.gz" ] && NODEHASH="5d86c21d47cad54e3e5d7f36c1323b1e2416efc75e4615dafe35b202b59f26c8"
[ "$NODEURL" = "http://nodejs.org/dist/v4.3.2/node-v4.3.2-linux-armv6l.tar.gz" ] && NODEHASH="f8767c0bacc0ed516bfa22802dda573082b5bf463e5ea79a74087272ccb2d1e2"
[ "$NODEURL" = "http://nodejs.org/dist/v4.3.2/node-v4.3.2-linux-armv7l.tar.gz" ] && NODEHASH="f4d5b9ae277aaec50cdbb735480ec1a04e6e02bd820e2a1fcfa913c0c25f2e3d"
[ "$NODEURL" = "http://nodejs.org/dist/v4.3.2/node-v4.3.2-linux-ppc64le.tar.gz" ] && NODEHASH="568b13ae13588853cbf49b7a0772698bf50b6de1168f76cd7a7cd7d7d7ada7c7"
[ "$NODEURL" = "http://nodejs.org/dist/v4.3.2/node-v4.3.2-linux-x64.tar.gz" ] && NODEHASH="f307f173a96dff6652bc70d835af0c732864bb09875cf32a0b6ce7d70cebf77d"
[ "$NODEURL" = "http://nodejs.org/dist/v4.3.2/node-v4.3.2-linux-x86.tar.gz" ] && NODEHASH="3c668a2128db3982987c596c028f9ec1d08c6abf63502e24826d40e552f80a8c"
[ "$NODEURL" = "http://nodejs.org/dist/v4.3.2/node-v4.3.2-sunos-x64.tar.gz" ] && NODEHASH="80fdaa0c54b187d43933336b334b70bcbb79953350510006b07dbaa019929188"
[ "$NODEURL" = "http://nodejs.org/dist/v4.3.2/node-v4.3.2-sunos-x86.tar.gz" ] && NODEHASH="04705681dde835c383028bc5bab3cab9d14dd7977e062ac824c2cfaf999f7c43"
[ "$NODEURL" = "http://nodejs.org/dist/v4.3.2/node-v4.3.2.tar.gz" ] && NODEHASH="1f92f6d31f7292ce56db57d6703efccf3e6c945948f5901610cefa69e78d3498"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.0/node-v4.4.0-darwin-x64.tar.gz" ] && NODEHASH="650c2f1e186a4974304b652321fbb3c4bd22ee7bc7c2c406c6e244dd6a6d38be"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.0/node-v4.4.0-headers.tar.gz" ] && NODEHASH="60aff0f9ea74bc64a8d0653828f63d6e8c5f39c3f7a16dee4a6ab48707071236"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.0/node-v4.4.0-linux-arm64.tar.gz" ] && NODEHASH="649590a9c16d262ba95b8e22a2a69105f7e881375b879e34e647748f18d2c830"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.0/node-v4.4.0-linux-armv6l.tar.gz" ] && NODEHASH="9ae5e1bb3ddeb4ec399922b0cadff47109b1711e79518be64566653454c19df2"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.0/node-v4.4.0-linux-armv7l.tar.gz" ] && NODEHASH="c610f75a848b3334148c8d78ce125550b296356fc6baf4fdd8167142aad24806"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.0/node-v4.4.0-linux-ppc64le.tar.gz" ] && NODEHASH="1978d58ea0b640e0ce1766f6cc71b163baa7ad3cafbff69051f2b71c9edaf353"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.0/node-v4.4.0-linux-x64.tar.gz" ] && NODEHASH="114a865effcff2783022ef0fcd30d1e51624d6c28140db0bdc662bcd0f850d8b"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.0/node-v4.4.0-linux-x86.tar.gz" ] && NODEHASH="53eb070a459838fb5bff9edc8260b6d0007409d36b8f2917bc1e523e92d6d99f"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.0/node-v4.4.0-sunos-x64.tar.gz" ] && NODEHASH="46338fcf165edb93528196f942dd38d241abf56ca6e940200696f22fb40d7fe4"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.0/node-v4.4.0-sunos-x86.tar.gz" ] && NODEHASH="4131f8f4baeb527a534c3b6a951d98727013e412a8c455c25c77a06a7faa0dfc"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.0/node-v4.4.0.tar.gz" ] && NODEHASH="2cfc76292576d17a8f2434329221675972c96e5fd60cd694610f53134079f92e"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.1/node-v4.4.1-darwin-x64.tar.gz" ] && NODEHASH="628195be45f7248a6defb6c23aaed7d0bca78979934f6e868387ddeb311dcc56"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.1/node-v4.4.1-headers.tar.gz" ] && NODEHASH="e759ee28a27dc47a5c80e48b063c0bee015f3b6d2f8f593ad0eabfab0ebb3922"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.1/node-v4.4.1-linux-arm64.tar.gz" ] && NODEHASH="8ff4cfc1452abaa95f4d40cf13f7ca3c4cf38da23a3ebde9247154dd51fbc42d"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.1/node-v4.4.1-linux-armv6l.tar.gz" ] && NODEHASH="f4996cce41a3d0d6b551e64466f591203ad2a637e4613159b515d6fff576df9a"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.1/node-v4.4.1-linux-armv7l.tar.gz" ] && NODEHASH="d61f87dd5994a3a095950d2477a981e0f250052b96ca4d32b1629bfda93e8d64"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.1/node-v4.4.1-linux-ppc64le.tar.gz" ] && NODEHASH="ee8e2611269eae2f84644c7a36416e9ffecf71f9294bd1e463152cb979a35940"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.1/node-v4.4.1-linux-x64.tar.gz" ] && NODEHASH="f0a53527f52dbcab3b98921a6cfe8613e5fe26fb796624988f6d615c30305a95"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.1/node-v4.4.1-linux-x86.tar.gz" ] && NODEHASH="200a3f800110ee5342dd80498a3365dbb704f5e214d26447dcd9dfcf2a08b6d2"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.1/node-v4.4.1-sunos-x64.tar.gz" ] && NODEHASH="5612216d93a5f9ed7d0c054349ec23f45e261eb3b30b6fd40e0ef3a7b339ee82"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.1/node-v4.4.1-sunos-x86.tar.gz" ] && NODEHASH="41c2559c98cc739bae4dd477d9119a130580c489cfed1222b5b67eab05a63d6d"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.1/node-v4.4.1.tar.gz" ] && NODEHASH="f3e604cc4d05a4810c37cd43a838a2dc4399d517bd1e8c53b7670dcffc4dc481"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.2/node-v4.4.2-darwin-x64.tar.gz" ] && NODEHASH="a143d6c71c9f1d1827d4cde1086611fb41d4909247377b8539a7129e1bf8266b"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.2/node-v4.4.2-headers.tar.gz" ] && NODEHASH="6b2ec4064900a8c39357ca7bdd7863587d749dcad1e4735df0de402400142cbb"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.2/node-v4.4.2-linux-arm64.tar.gz" ] && NODEHASH="be881df65ff29ffbec47a14e082800c150d4a9238d1c137ff18cf7c28fafa987"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.2/node-v4.4.2-linux-armv6l.tar.gz" ] && NODEHASH="c574a381e5955c724bf39ff93814882e8f5218293b8c4210ccfafbda06fc4200"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.2/node-v4.4.2-linux-armv7l.tar.gz" ] && NODEHASH="f243f42bbab4757b3881a26e7ac3af4a1ec6d0f5edc1e4415e69768e94cd1389"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.2/node-v4.4.2-linux-ppc64le.tar.gz" ] && NODEHASH="035d202d02ac965618d99ec5c8680aba1ea41f6114caf720a160fe23405809b5"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.2/node-v4.4.2-linux-x64.tar.gz" ] && NODEHASH="b4a44dbe528520397621aad76168bdfd50cdb96fb1f15e99358263f6400c33d2"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.2/node-v4.4.2-linux-x86.tar.gz" ] && NODEHASH="0c2bdf6fed204c6b9bddd7f839f49010cf9b484986f820e20d4f7abb01d8ee9d"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.2/node-v4.4.2-sunos-x64.tar.gz" ] && NODEHASH="006ef87af910bb5b0d78d65d2b979125a6b1049d56dc26088c9cb9deb6a16986"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.2/node-v4.4.2-sunos-x86.tar.gz" ] && NODEHASH="13da507b8ecd88598080c63a63c77efc83e04c8454f4070389c4472d512ced48"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.2/node-v4.4.2.tar.gz" ] && NODEHASH="ae91cb0a31f87b5b13bb2665178845d99abf1037dc3636fa88f442b4a4c65297"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.3/node-v4.4.3-darwin-x64.tar.gz" ] && NODEHASH="7cbb9819926d90c9c9f31c3db4c83499a3d9518ad3c7d9915fa4c66d6d5875df"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.3/node-v4.4.3-headers.tar.gz" ] && NODEHASH="e24fb3eb7c7fe4c5b1ac64a93b63cfc109d56b59bedf2e88a2001f7ce3b1673c"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.3/node-v4.4.3-linux-arm64.tar.gz" ] && NODEHASH="261646b9d606ab3cc42f870d4bcaab79a40f18f7b13740762127598ef29d4ffc"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.3/node-v4.4.3-linux-armv6l.tar.gz" ] && NODEHASH="04d8abedf4199d1ca9b963d18458eb9e99f80cc5663e6f61041735dc28c9eb5f"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.3/node-v4.4.3-linux-armv7l.tar.gz" ] && NODEHASH="b1cd7144bea5564ae8ca0f653f1e1daf45c63e2a0d0e5c391519aaf7167dbd9e"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.3/node-v4.4.3-linux-ppc64le.tar.gz" ] && NODEHASH="dd9d9f8a25b1ba67c73af4724ca1abe4e0661d3ffac3d48e57ef3480c8f3cb2a"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.3/node-v4.4.3-linux-x64.tar.gz" ] && NODEHASH="28ff2b23a837526ecfea66b0db42d43ec84368949998f2cb26dd742e8988ec1f"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.3/node-v4.4.3-linux-x86.tar.gz" ] && NODEHASH="ffdbb81573d77a2bf14415fc1c0d65b110cfc2ba5a20dcdf8b62015519acd079"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.3/node-v4.4.3-sunos-x64.tar.gz" ] && NODEHASH="72ceeafe74e911a418baa5c73cdb2888105c85d62f15245d55009f0e93da543a"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.3/node-v4.4.3-sunos-x86.tar.gz" ] && NODEHASH="182963950c657aeecb18494c4d3765bcf51970aca938dd4848cf6942382fc730"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.3/node-v4.4.3.tar.gz" ] && NODEHASH="8e67b95721aab7bd721179da2fe5dd97f9acc1306c15c9712ee103bcd6381638"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.4/node-v4.4.4-darwin-x64.tar.gz" ] && NODEHASH="3d240db764391faca06df67db1eefd0b20989153edb0a99cbaf16e441e734a4d"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.4/node-v4.4.4-headers.tar.gz" ] && NODEHASH="a5abc953d1ee4295cd3dcfdc116797e4b807e0b2d5c888aa31631fa1de9a9200"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.4/node-v4.4.4-linux-arm64.tar.gz" ] && NODEHASH="4d7336411a61e92eb4815dc5b9042cae92ed49d3bc472da153aa13fd4e812b99"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.4/node-v4.4.4-linux-armv6l.tar.gz" ] && NODEHASH="618efa1636db2d8acd7f28902200cbb42fd0f1d71ef482b6e99a027372584581"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.4/node-v4.4.4-linux-armv7l.tar.gz" ] && NODEHASH="64af655f048fc1ba70591076bbdeb61195347942ceef1ea18157467e5a01f253"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.4/node-v4.4.4-linux-ppc64le.tar.gz" ] && NODEHASH="939cead87316367dd88430807ad335b9bdab4812fe0cd18adf62887c699dd4e7"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.4/node-v4.4.4-linux-x64.tar.gz" ] && NODEHASH="0881eb010c8a4a0e746a1852fe48416d9c21b5f19a20d418cb02c7197fa55576"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.4/node-v4.4.4-linux-x86.tar.gz" ] && NODEHASH="401fcef7908b3620bd28300bac06ad4fe774c95e10167c8617dde18befd78fcd"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.4/node-v4.4.4-sunos-x64.tar.gz" ] && NODEHASH="5748ececfb65a5dfa5c7eb87c31e890aff7a32aeea6ae440e382eca34ec16eba"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.4/node-v4.4.4-sunos-x86.tar.gz" ] && NODEHASH="1da8b38983ba9fd8878c2127225396991c8189934e578760f0860ed15302cdd6"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.4/node-v4.4.4.tar.gz" ] && NODEHASH="53c694c203ee18e7cd393612be08c61ed6ab8b2a165260984a99c014d1741414"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.5/node-v4.4.5-darwin-x64.tar.gz" ] && NODEHASH="1e729f9836e7b543724ee0b2ac902b4dff4a6f7d91031d20314825b61e86127f"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.5/node-v4.4.5-headers.tar.gz" ] && NODEHASH="afd90e4b0ae408301facce1d4837c5f718a686d144bf624a53c55988a473601c"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.5/node-v4.4.5-linux-arm64.tar.gz" ] && NODEHASH="46c89b2cdb3d7f3e87ef1ed8e4d9ee07dcc603e42f3f86f831a6fd7c34be9404"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.5/node-v4.4.5-linux-armv6l.tar.gz" ] && NODEHASH="8f6a09d7326c4a2570275519ae51bc2a9ba6b2809b5cc02658922368f0dc49b0"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.5/node-v4.4.5-linux-armv7l.tar.gz" ] && NODEHASH="2ce99cbfaef7a8a0e8119a3e1303015f5bed6e745c90a95b3dcd6658ad44035b"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.5/node-v4.4.5-linux-ppc64le.tar.gz" ] && NODEHASH="a2f17bee90c650b30c90c3cd96118e7825eb1b1011c4c7584e60d1b36d9b6983"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.5/node-v4.4.5-linux-ppc64.tar.gz" ] && NODEHASH="577688263d15b2d7f8004f99151c5d47e27b319510d501792cfd74f7c834caa4"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.5/node-v4.4.5-linux-x64.tar.gz" ] && NODEHASH="15d57c4a3696df8d5ef1bba452d38e5d27fc3c963760eeb218533c48381e89d5"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.5/node-v4.4.5-linux-x86.tar.gz" ] && NODEHASH="447b17542981c20f6792a20c31eb946c786e2cbd3bd1459f85c2cd0eb400e009"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.5/node-v4.4.5-sunos-x64.tar.gz" ] && NODEHASH="75ebf346d29d441fbb32b177568f5727c357300406ea90b29c49a9e3bd89c506"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.5/node-v4.4.5-sunos-x86.tar.gz" ] && NODEHASH="181c092d49e2efc3ecd82806c9aa088ed1eda908671a5913cfa4704855ae288b"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.5/node-v4.4.5.tar.gz" ] && NODEHASH="ea9c96ae4768feee4f18a26b819b9b4f6e49105ea0ee8c5c9d188dc8d49d4b77"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.6/node-v4.4.6-darwin-x64.tar.gz" ] && NODEHASH="2afbc5f48ce3b3bb9b9a46e64b8a23c08a448bf8bf89c10a5a053fd0ec024a07"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.6/node-v4.4.6-headers.tar.gz" ] && NODEHASH="75c114e9f4f3a10e5b8116a3ead9123b3982d951c392ed7a6c737638035dca21"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.6/node-v4.4.6-linux-arm64.tar.gz" ] && NODEHASH="de0c093ed38934f25cc5eaa1ecd84878fe123e0632db3bf4c3ffcd8af107a62d"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.6/node-v4.4.6-linux-armv6l.tar.gz" ] && NODEHASH="372087f61b2c53fbeb84b5cd0cdb508ef1c7d1fcecf9a56393d957d61ae4b2c2"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.6/node-v4.4.6-linux-armv7l.tar.gz" ] && NODEHASH="736ca6a35cbae8870cf819362a699a9639c76eed27b6ae4f667713fbc9ec809f"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.6/node-v4.4.6-linux-ppc64le.tar.gz" ] && NODEHASH="ce02faf08867b01ba7bdb1431cd6750a13e8742ad30d6e9bdc2b36a417b578ad"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.6/node-v4.4.6-linux-ppc64.tar.gz" ] && NODEHASH="228685937ffd2aad0999460616bfa25fa17538b52df6b328481142c1bbd565eb"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.6/node-v4.4.6-linux-x64.tar.gz" ] && NODEHASH="bef5cc1db30b56d3f40b123c6a40529b6f69c403fa969ec2654b62d4cac95e26"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.6/node-v4.4.6-linux-x86.tar.gz" ] && NODEHASH="86b7d4374b27ecb8d2f63b371147c0f7f7cf8b80f37ff740f42e2f074184329e"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.6/node-v4.4.6-sunos-x64.tar.gz" ] && NODEHASH="d1fa8b52c126a092f7c6d4ce95e12cf39d014e9593700d792aa38f529deb2f1e"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.6/node-v4.4.6-sunos-x86.tar.gz" ] && NODEHASH="cd0c3d55eb0d80787348cc6841ec9381746ef4992bae629d9b34671ef0270c29"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.6/node-v4.4.6.tar.gz" ] && NODEHASH="45eab3d415616e5831ba5961b67cec5423e1f9c175ca7e331de7f560c2998d9f"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.7/node-v4.4.7-darwin-x64.tar.gz" ] && NODEHASH="1971386a2dcf4406b5bc414d111ea40b227cfebf396badd447c37ace0eef2fa9"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.7/node-v4.4.7-headers.tar.gz" ] && NODEHASH="6c5cff955e7ffccc59386e79a811d868b97829fdf0accb2fc152da875171efde"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.7/node-v4.4.7-linux-arm64.tar.gz" ] && NODEHASH="a1e2faf3859976ac7322b950353044863c2e36ad6e2e09a8fc9f80f72fd01b18"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.7/node-v4.4.7-linux-armv6l.tar.gz" ] && NODEHASH="ba51e7487a15b31823e352fe3e3c949fa754fba7cffc4b82f7421c374f22980f"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.7/node-v4.4.7-linux-armv7l.tar.gz" ] && NODEHASH="68a6b3dd2b6554e89d05ed6df4c9cf33763c3eea63041ee2c8b189400f07567b"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.7/node-v4.4.7-linux-ppc64le.tar.gz" ] && NODEHASH="a2202e69b97259d409b1db653d50305570f78b57a9f48a55dd63e0ceced2dad6"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.7/node-v4.4.7-linux-ppc64.tar.gz" ] && NODEHASH="781cb67ac07312c8e6d1c434965f0dca1cd04a525971e28811287af5025563b0"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.7/node-v4.4.7-linux-x64.tar.gz" ] && NODEHASH="5ad10465cc9d837c1fda8db0fd1bdc1a4ce823dd6afbc533ac2127e6a9a64133"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.7/node-v4.4.7-linux-x86.tar.gz" ] && NODEHASH="604c4f85ef00a13c172465c9672a2a0f13524921e255eeb75242afb42af6d814"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.7/node-v4.4.7-sunos-x64.tar.gz" ] && NODEHASH="d86382a337795d054d07677643cd049df545186d46528b8df7041e29ceb84779"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.7/node-v4.4.7-sunos-x86.tar.gz" ] && NODEHASH="d7bae99460f1fb35456303630c743d9c467449d2f56bd48f1dd96f1ecabb7da6"
[ "$NODEURL" = "http://nodejs.org/dist/v4.4.7/node-v4.4.7.tar.gz" ] && NODEHASH="cbe1c6e421969dd5639d0fbaa6d3c1f56c0463b87efe75be8594638da4d8fc4f"
[ "$NODEURL" = "http://nodejs.org/dist/v4.5.0/node-v4.5.0-darwin-x64.tar.gz" ] && NODEHASH="d171f0c859e3895b2430c317001b817866c4de45211ad540c59658ee6a2f689f"
[ "$NODEURL" = "http://nodejs.org/dist/v4.5.0/node-v4.5.0-headers.tar.gz" ] && NODEHASH="12ee966eef2abc928f6d7fcf9cfcf2913ef0e59ae07e2dcc20726246ab174fd8"
[ "$NODEURL" = "http://nodejs.org/dist/v4.5.0/node-v4.5.0-linux-arm64.tar.gz" ] && NODEHASH="ecdbb3cb55d0a87aeb10334b47310f1823393abe6273f1ce7c97bcb509051e68"
[ "$NODEURL" = "http://nodejs.org/dist/v4.5.0/node-v4.5.0-linux-armv6l.tar.gz" ] && NODEHASH="87c6fb8269475d385a9b923c647890def45079f8e30b55c4737d70558fd36418"
[ "$NODEURL" = "http://nodejs.org/dist/v4.5.0/node-v4.5.0-linux-armv7l.tar.gz" ] && NODEHASH="99d222ede2f38041c55a64da8e6a669e0128cfb40d40f968cae74d80e52ad566"
[ "$NODEURL" = "http://nodejs.org/dist/v4.5.0/node-v4.5.0-linux-ppc64le.tar.gz" ] && NODEHASH="a517a56b5550567225c6820f99e44fc84ac9953e2fe9db67f3bab7a8d7661e9f"
[ "$NODEURL" = "http://nodejs.org/dist/v4.5.0/node-v4.5.0-linux-ppc64.tar.gz" ] && NODEHASH="46bebb806e4e8273e66e02faf5f25145f5351958bda75d9aa49bee9df7ccafec"
[ "$NODEURL" = "http://nodejs.org/dist/v4.5.0/node-v4.5.0-linux-x64.tar.gz" ] && NODEHASH="5678ad94ee35e40fc3a2c545e136a0dc946ac4c039fca5898e1ea51ecf9e7c39"
[ "$NODEURL" = "http://nodejs.org/dist/v4.5.0/node-v4.5.0-linux-x86.tar.gz" ] && NODEHASH="a11ef26fa700eff8ea7acfa491fd362dd362590da2967335d87932e7036a06fd"
[ "$NODEURL" = "http://nodejs.org/dist/v4.5.0/node-v4.5.0-sunos-x64.tar.gz" ] && NODEHASH="47f0f3c9e33d02a49a6c467d8a12539b0bb3c5bf8e1916bd938eaa956afb47bb"
[ "$NODEURL" = "http://nodejs.org/dist/v4.5.0/node-v4.5.0-sunos-x86.tar.gz" ] && NODEHASH="190ff116905ce94291a7122779cbdf493af1ba415432a9a12a32263816b90ccd"
[ "$NODEURL" = "http://nodejs.org/dist/v4.5.0/node-v4.5.0.tar.gz" ] && NODEHASH="74ced83b8d890d90e2a8b0d54b0d0e9b5e01d6fd6148cec6e9911ff6eaf0cf21"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.0/node-v4.6.0-darwin-x64.tar.gz" ] && NODEHASH="525ab42c767525edb7e512c600dedb20d826a6f58e1d6d1b774651a1c782a267"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.0/node-v4.6.0-headers.tar.gz" ] && NODEHASH="5eb4b4324d72297066b4b8c91d0b1e7c82cabde9986c986682be66202f37176b"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.0/node-v4.6.0-linux-arm64.tar.gz" ] && NODEHASH="bf03e7384b727bc80c0c59cf38ba5704d83faa7f455f40fa62a67c8331dde7d6"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.0/node-v4.6.0-linux-armv6l.tar.gz" ] && NODEHASH="e7db1c612eb9dd55e3ff246bfa7c35f0b87664e6e2bc7b32891de8cc1e48f5a7"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.0/node-v4.6.0-linux-armv7l.tar.gz" ] && NODEHASH="9e46082bef5b521afd483532c8d3715f33d1d4302b7980b904bea3182817275f"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.0/node-v4.6.0-linux-ppc64le.tar.gz" ] && NODEHASH="2aa9518ea637cc06877a01c40d4608cf9a7f1588000cf3e550e4ab24c170aee6"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.0/node-v4.6.0-linux-ppc64.tar.gz" ] && NODEHASH="ee77fb6a1dfbe166c9faee25b4f110af25723c64b0abcb9085507b8445fa2e7b"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.0/node-v4.6.0-linux-x64.tar.gz" ] && NODEHASH="acf08148cecf245f28126122ac9128ff9909f00938b18d80fc0b92648d1c98a8"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.0/node-v4.6.0-linux-x86.tar.gz" ] && NODEHASH="9aab75618de0dca640d747aa25073cbb5a01342dd8aa177df8112e26a39541f4"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.0/node-v4.6.0-sunos-x64.tar.gz" ] && NODEHASH="e9a02da71d0cd6a1874f4a7d227dfcbe6ab9492eba419b5c9a83c8c95065195f"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.0/node-v4.6.0-sunos-x86.tar.gz" ] && NODEHASH="f8536a25629ef1ad3228b2d712e2fa43bf66980673d3cdf469da37c0407e9633"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.0/node-v4.6.0.tar.gz" ] && NODEHASH="0838f12e329edb252e6e6baddca85632bf5ff2ec900e737e88f9bf9b38946b1b"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.1/node-v4.6.1-darwin-x64.tar.gz" ] && NODEHASH="e5929de3a963f9a75c5923c9c222cde401e03c4b4991ec9281bf81f878f482ff"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.1/node-v4.6.1-headers.tar.gz" ] && NODEHASH="58ce23125a049f0098693f8c48c431d54906586007c1a8b899add656a4e5a2fe"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.1/node-v4.6.1-linux-arm64.tar.gz" ] && NODEHASH="439a4261c13644dee42d8f5ff6a6fc7974a5c5bb169f6c79cc5852b50298a1c4"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.1/node-v4.6.1-linux-armv6l.tar.gz" ] && NODEHASH="a66310bd045260f3c1d55bbafe7bbd6477004882f8b7ef63da0b79ed3d2b9ec4"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.1/node-v4.6.1-linux-armv7l.tar.gz" ] && NODEHASH="b6883d8d3daf121c6fffcd8546747105ca46ae8b1214d2fb6f3640a8aef0879c"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.1/node-v4.6.1-linux-ppc64le.tar.gz" ] && NODEHASH="1c66dd602d5ac022560eb7f2cb8c1be63d7a0d9e32f8f67c0bd7d84fce7179e5"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.1/node-v4.6.1-linux-ppc64.tar.gz" ] && NODEHASH="297a458edf5f0f274e8bba96dcb8cb975e47a802401087a17ba20f550fa85785"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.1/node-v4.6.1-linux-x64.tar.gz" ] && NODEHASH="033243d4ddffd67856c9ccfb512b0d1980e8d1373554bc328472b21bc5de7675"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.1/node-v4.6.1-linux-x86.tar.gz" ] && NODEHASH="16e0216d35ed1f81ac9a40081fa6f9b28e63dcbb38b316956dc1badd8b4bcfa7"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.1/node-v4.6.1-sunos-x64.tar.gz" ] && NODEHASH="892b86782c240b1886b4b655ca266425164e928b18c75e3d3f6960172a34cedf"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.1/node-v4.6.1-sunos-x86.tar.gz" ] && NODEHASH="f87490f0764b6c0a88e3e90e2ee14eb8b67dad0245a7b20fcc4e5fe1e75d3ec9"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.1/node-v4.6.1.tar.gz" ] && NODEHASH="b8ca4db42c5a1fed95baf6996ff776db3c95ad2bcf73c7aed2f1f921a1225de3"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.2/node-v4.6.2-darwin-x64.tar.gz" ] && NODEHASH="0e19ba9c0e16ea1d96df2a3a5edc07e0950325f465ab1049d6ab7b7d98d384f2"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.2/node-v4.6.2-headers.tar.gz" ] && NODEHASH="3b3b7045cfb372a8d1472edab596213111369d6fca30dad5714c9f22a1da9edf"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.2/node-v4.6.2-linux-arm64.tar.gz" ] && NODEHASH="831eab2a0fed0c3716e2c6bb9d9f2d64b8f365e5501ec373eb207ce5621ba6d2"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.2/node-v4.6.2-linux-armv6l.tar.gz" ] && NODEHASH="a45b09f7b1274a5a2c494aa529fb84fbf8f3040c9b24e5be73a8896b24c803e6"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.2/node-v4.6.2-linux-armv7l.tar.gz" ] && NODEHASH="969fa30c2c12b6165fa9b56b8466e9a6113129bdbfe46861b0990b15ff43c483"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.2/node-v4.6.2-linux-ppc64le.tar.gz" ] && NODEHASH="30818e8ebc302bdd5b60d0d7287d336ce4db51302eacb7ad5684d7a1f013af48"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.2/node-v4.6.2-linux-ppc64.tar.gz" ] && NODEHASH="64c06404a3b0cba67e0b62f3ecc2cb7857fd11a22535f99c97beb231520e1ae6"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.2/node-v4.6.2-linux-x64.tar.gz" ] && NODEHASH="0a2d6417526509bc4c0d953e1563b0d5c972fe270a45da0ca8ed02d41fb1c223"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.2/node-v4.6.2-linux-x86.tar.gz" ] && NODEHASH="6f992d823873eebb6cb99c96f5ac4f0dd8a205824bb4a54beb93eb60f7ca22eb"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.2/node-v4.6.2-sunos-x86.tar.gz" ] && NODEHASH="22c32dcfd70e556a01bb13648f240b4911cf990f1682985d4c311ecd8addcb21"
[ "$NODEURL" = "http://nodejs.org/dist/v4.6.2/node-v4.6.2.tar.gz" ] && NODEHASH="3c8f3af398c348cdf90fef9b983e4b46aae96a56238236542925e0bb4ee27ee8"
[ "$NODEURL" = "http://nodejs.org/dist/v5.0.0/node-v5.0.0-darwin-x64.tar.gz" ] && NODEHASH="26f3e42df814b9b42f0b0045901c84eb79233d8196d0dcdf77a6a3c975c6f25d"
[ "$NODEURL" = "http://nodejs.org/dist/v5.0.0/node-v5.0.0-headers.tar.gz" ] && NODEHASH="5133b6145df7083133e272e1b8c96b59c58a19d004d5aacdec80ea78e58c7e46"
[ "$NODEURL" = "http://nodejs.org/dist/v5.0.0/node-v5.0.0-linux-arm64.tar.gz" ] && NODEHASH="2c4517d3fdefc29b5c61aa6ea3386a0dafca831357d3bcd30fc14e97b49139d1"
[ "$NODEURL" = "http://nodejs.org/dist/v5.0.0/node-v5.0.0-linux-armv6l.tar.gz" ] && NODEHASH="96629e271e34d14bdb7be93067a8770cb5326977b9168cf2344c80a721040784"
[ "$NODEURL" = "http://nodejs.org/dist/v5.0.0/node-v5.0.0-linux-armv7l.tar.gz" ] && NODEHASH="6bf5f19cac9f01cb31b7088685d9d6df6d24c9c2a4531ace839387763b8a28ec"
[ "$NODEURL" = "http://nodejs.org/dist/v5.0.0/node-v5.0.0-linux-x64.tar.gz" ] && NODEHASH="ef73b59048a0ed11d01633f0061627b7a9879257deb9add2255e4d0808f8b671"
[ "$NODEURL" = "http://nodejs.org/dist/v5.0.0/node-v5.0.0-linux-x86.tar.gz" ] && NODEHASH="6484c4cac1724f2818354e3f2a8ede66308617b648cd3db624596b8cd58b5f6b"
[ "$NODEURL" = "http://nodejs.org/dist/v5.0.0/node-v5.0.0-sunos-x64.tar.gz" ] && NODEHASH="7cfbed6e6658d8304f8593eae9dd28f8f83b57e7a6345b32f4d5e9830b80203f"
[ "$NODEURL" = "http://nodejs.org/dist/v5.0.0/node-v5.0.0-sunos-x86.tar.gz" ] && NODEHASH="2e9d65dff65843ebfa5c9d623c5b930bd0af7252101cc9c69c55aa8c880fa9eb"
[ "$NODEURL" = "http://nodejs.org/dist/v5.0.0/node-v5.0.0.tar.gz" ] && NODEHASH="698d9662067ae6a20a2586e5c09659735fc0050769a0d8f76f979189ceaccdf4"
[ "$NODEURL" = "http://nodejs.org/dist/v5.1.0/node-v5.1.0-darwin-x64.tar.gz" ] && NODEHASH="4752961731e579a26dd45d765f76e67f70683b0026c0035fc4c30d70c7baf4f0"
[ "$NODEURL" = "http://nodejs.org/dist/v5.1.0/node-v5.1.0-headers.tar.gz" ] && NODEHASH="2dd57f72e3dc7bcce841b2390edd6c8e6257ec41ed85aa4ad4d380a3685cb768"
[ "$NODEURL" = "http://nodejs.org/dist/v5.1.0/node-v5.1.0-linux-arm64.tar.gz" ] && NODEHASH="8e6eb18b4499c4c509b4132d3393121a7d344d4053400798614c843977696ff3"
[ "$NODEURL" = "http://nodejs.org/dist/v5.1.0/node-v5.1.0-linux-armv6l.tar.gz" ] && NODEHASH="a6a64bf9d2e6b792505841b187eeb1c3ac971551840dab2a9a4a4719d1e7d150"
[ "$NODEURL" = "http://nodejs.org/dist/v5.1.0/node-v5.1.0-linux-armv7l.tar.gz" ] && NODEHASH="c7bbb8c51bdaf9b46daffbebdc889ce561c8f7b6cdd3c804c5e8cb7314a4d40d"
[ "$NODEURL" = "http://nodejs.org/dist/v5.1.0/node-v5.1.0-linux-x64.tar.gz" ] && NODEHASH="510e7a2e8639a3ea036f5f6a9f7a66037e3acf8d0c953aeac8d093dea7e41d4c"
[ "$NODEURL" = "http://nodejs.org/dist/v5.1.0/node-v5.1.0-linux-x86.tar.gz" ] && NODEHASH="f08c2cadb965a4f85700e1b3a8645faf3192e8fefe5c740b4c581c0d6110ced1"
[ "$NODEURL" = "http://nodejs.org/dist/v5.1.0/node-v5.1.0-sunos-x64.tar.gz" ] && NODEHASH="eb4497a705e720e38ee71580c94c9d911922211dd5819eeb55d217f4f4839b69"
[ "$NODEURL" = "http://nodejs.org/dist/v5.1.0/node-v5.1.0-sunos-x86.tar.gz" ] && NODEHASH="d51d05e57e77ca780f43e315a5477eee9fbb954822f473985ff6a8ce5d3b05ca"
[ "$NODEURL" = "http://nodejs.org/dist/v5.1.0/node-v5.1.0.tar.gz" ] && NODEHASH="25b2d3b7dd57fe47a483539fea240a3c6bbbdab4d89a45a812134cf1380ecb94"
[ "$NODEURL" = "http://nodejs.org/dist/v5.1.1/node-v5.1.1-darwin-x64.tar.gz" ] && NODEHASH="cb6c831e7c3a8432a14a0e4ddb2000295c0166abce06b2d50134cc2cccb2dc9c"
[ "$NODEURL" = "http://nodejs.org/dist/v5.1.1/node-v5.1.1-headers.tar.gz" ] && NODEHASH="cc320d0a3df073e9e92816d5698d69ade780ce854f26146539f0b29cb5be616a"
[ "$NODEURL" = "http://nodejs.org/dist/v5.1.1/node-v5.1.1-linux-arm64.tar.gz" ] && NODEHASH="1723abf50ee9b2b2209af06374523ae657c5562166bdc44b7b8d32801484c572"
[ "$NODEURL" = "http://nodejs.org/dist/v5.1.1/node-v5.1.1-linux-armv6l.tar.gz" ] && NODEHASH="e7b154a5f7574155df5e3b5df90af762ab0edde128e36eeac0bd8ba2a8f00697"
[ "$NODEURL" = "http://nodejs.org/dist/v5.1.1/node-v5.1.1-linux-armv7l.tar.gz" ] && NODEHASH="fd96f77310708097cf9e783b9842122a4e2859674965734b7b22a615cb756165"
[ "$NODEURL" = "http://nodejs.org/dist/v5.1.1/node-v5.1.1-linux-x64.tar.gz" ] && NODEHASH="0c1a0788dfc07d1cfac08b9789f0e52950e80e61944e1684b27600463a5d2623"
[ "$NODEURL" = "http://nodejs.org/dist/v5.1.1/node-v5.1.1-linux-x86.tar.gz" ] && NODEHASH="30099a0f305899aacdfd974873807b6bd8b6971c0a26209220a6e4cd88c08d70"
[ "$NODEURL" = "http://nodejs.org/dist/v5.1.1/node-v5.1.1-sunos-x64.tar.gz" ] && NODEHASH="14dfdf63f1ac8c9972199e7a62d8c732cf269e081aec6caa9e0e3f46d116d486"
[ "$NODEURL" = "http://nodejs.org/dist/v5.1.1/node-v5.1.1-sunos-x86.tar.gz" ] && NODEHASH="11ccaadb1e22b3c80548686caf19070c34017667866285fe8b6cb8b9e6afca30"
[ "$NODEURL" = "http://nodejs.org/dist/v5.1.1/node-v5.1.1.tar.gz" ] && NODEHASH="a779e024f800b5ec51f375fa1c14eda7254216daa36a1960cc1e4195b9fc22c3"
[ "$NODEURL" = "http://nodejs.org/dist/v5.10.0/node-v5.10.0-darwin-x64.tar.gz" ] && NODEHASH="00407892416649f7567cc20ae6f0c091650dee6186fe58eb33d2bd886f276799"
[ "$NODEURL" = "http://nodejs.org/dist/v5.10.0/node-v5.10.0-headers.tar.gz" ] && NODEHASH="726c61ca135198ef238078ab4f4f5a7ccd1980d049f08f2e1f2dd767c510f000"
[ "$NODEURL" = "http://nodejs.org/dist/v5.10.0/node-v5.10.0-linux-arm64.tar.gz" ] && NODEHASH="df88803bda234b32240906b620315c8f6d6200332047a88cb0ec83009cf25dd5"
[ "$NODEURL" = "http://nodejs.org/dist/v5.10.0/node-v5.10.0-linux-armv6l.tar.gz" ] && NODEHASH="019a257faa5eebf6304686dfeffdbcb4c22f0547aa366f6e563aad39ab1b1ab1"
[ "$NODEURL" = "http://nodejs.org/dist/v5.10.0/node-v5.10.0-linux-armv7l.tar.gz" ] && NODEHASH="3f7524d3db60175c2323bb2a0a13ad1ca7d47d4ede6f42834b6b8425be70e0a2"
[ "$NODEURL" = "http://nodejs.org/dist/v5.10.0/node-v5.10.0-linux-ppc64le.tar.gz" ] && NODEHASH="07058fb1c82e7cd3d21729ab6a5ed5523fdb9f664d975438974668710b7d8ec6"
[ "$NODEURL" = "http://nodejs.org/dist/v5.10.0/node-v5.10.0-linux-x64.tar.gz" ] && NODEHASH="a458ddab5f8d071c9b4f24ccfa685aedd57ccf7338c3ea0e2b99546cf35a3958"
[ "$NODEURL" = "http://nodejs.org/dist/v5.10.0/node-v5.10.0-linux-x86.tar.gz" ] && NODEHASH="e884f070542f49f577fd9785f09cf7734e6c2107d23b6c1b58453d43183c5ec0"
[ "$NODEURL" = "http://nodejs.org/dist/v5.10.0/node-v5.10.0-sunos-x64.tar.gz" ] && NODEHASH="78017489238f8ac86176079d1aa73cd8c40849f410d705080e00345512de8c93"
[ "$NODEURL" = "http://nodejs.org/dist/v5.10.0/node-v5.10.0-sunos-x86.tar.gz" ] && NODEHASH="f9540debe8a396158f21a47f275d2fff564b42193f6621960d04c959e8ac5cc3"
[ "$NODEURL" = "http://nodejs.org/dist/v5.10.0/node-v5.10.0.tar.gz" ] && NODEHASH="8faeb9af5bde641b6e5dfae32bfa99253df2bccf59cf8a62162760026411762b"
[ "$NODEURL" = "http://nodejs.org/dist/v5.10.1/node-v5.10.1-darwin-x64.tar.gz" ] && NODEHASH="00ffc5c662580e1a5062a8740a9b9a40dbf7dadb5c8aa16bdf0ed33c7c1dfbfb"
[ "$NODEURL" = "http://nodejs.org/dist/v5.10.1/node-v5.10.1-headers.tar.gz" ] && NODEHASH="77e1469d6c2c6f49e68cb3fb8052ee90284700e7da013a7d47ad1cefcd8afc63"
[ "$NODEURL" = "http://nodejs.org/dist/v5.10.1/node-v5.10.1-linux-arm64.tar.gz" ] && NODEHASH="98e4f003818968d5b9bcf17c921d33a5e3d6866be63d80510ae7ff8877e817db"
[ "$NODEURL" = "http://nodejs.org/dist/v5.10.1/node-v5.10.1-linux-armv6l.tar.gz" ] && NODEHASH="5d6f652ce962a0fb59edf5e305af3a7e9147489ebb90a1244f3fa67d86fcf54b"
[ "$NODEURL" = "http://nodejs.org/dist/v5.10.1/node-v5.10.1-linux-armv7l.tar.gz" ] && NODEHASH="f1ccddf5fc894a4a4becdf9b32e579b2aad6c6ace189890dbdcb934afbaab060"
[ "$NODEURL" = "http://nodejs.org/dist/v5.10.1/node-v5.10.1-linux-ppc64le.tar.gz" ] && NODEHASH="b7137dc0bc9a6e3a91be67f05330d2722d58b4e50047d3455a7c035c455701b4"
[ "$NODEURL" = "http://nodejs.org/dist/v5.10.1/node-v5.10.1-linux-x64.tar.gz" ] && NODEHASH="897506e1e83cba9b780b030c9cc7299b0ae8872c0b8b0081a86996079025cea5"
[ "$NODEURL" = "http://nodejs.org/dist/v5.10.1/node-v5.10.1-linux-x86.tar.gz" ] && NODEHASH="a4c17570bd03424cd57affd10a4ef7d3b76167d3646656236bc78b86bc622a88"
[ "$NODEURL" = "http://nodejs.org/dist/v5.10.1/node-v5.10.1-sunos-x64.tar.gz" ] && NODEHASH="0cb823dacc340aa79b70a08d7d81a72260c3414b6f07398ab80e755efcc93f85"
[ "$NODEURL" = "http://nodejs.org/dist/v5.10.1/node-v5.10.1-sunos-x86.tar.gz" ] && NODEHASH="1d2731f7ffb46d9f2153993f24ac36a5e98dd136ad28600e45d918205997403f"
[ "$NODEURL" = "http://nodejs.org/dist/v5.10.1/node-v5.10.1.tar.gz" ] && NODEHASH="c6e278b612b53c240ddf85521403e55abfd8f0201d2f2c7e3d2c21383054aacd"
[ "$NODEURL" = "http://nodejs.org/dist/v5.11.0/node-v5.11.0-darwin-x64.tar.gz" ] && NODEHASH="4992e1ba18cdac51383a089b1494131dbca465a5328938dbd4835cbf780f7ecc"
[ "$NODEURL" = "http://nodejs.org/dist/v5.11.0/node-v5.11.0-headers.tar.gz" ] && NODEHASH="d6850717f78dd94238510ec3cb804181a6bd152397def7de4a90dedf08a0eae1"
[ "$NODEURL" = "http://nodejs.org/dist/v5.11.0/node-v5.11.0-linux-arm64.tar.gz" ] && NODEHASH="b6cc0dd471f07b607367b76a3f2ec1f11d9bc05f2fccbcda7b85ce76d31a3e2a"
[ "$NODEURL" = "http://nodejs.org/dist/v5.11.0/node-v5.11.0-linux-armv6l.tar.gz" ] && NODEHASH="6bf29cbf6d78e95a895bfb77774fde49fc3a565d601320b91b7ed5849f01a08d"
[ "$NODEURL" = "http://nodejs.org/dist/v5.11.0/node-v5.11.0-linux-armv7l.tar.gz" ] && NODEHASH="c07487fab4434f667a4b456fab098deffa76d73853e6636ebe05045e7db875de"
[ "$NODEURL" = "http://nodejs.org/dist/v5.11.0/node-v5.11.0-linux-ppc64le.tar.gz" ] && NODEHASH="6d267c7e1294d15d99747437ee74104b941111e7861382da84466c86d7fca139"
[ "$NODEURL" = "http://nodejs.org/dist/v5.11.0/node-v5.11.0-linux-x64.tar.gz" ] && NODEHASH="92602b815ce1c64c63b94d6f72f9b96b427415d023d5f9466c7ffef334bf0386"
[ "$NODEURL" = "http://nodejs.org/dist/v5.11.0/node-v5.11.0-linux-x86.tar.gz" ] && NODEHASH="f167a644eb6bf94f9a289d28d8140a6a497b02190571b02c7919458e12431ce0"
[ "$NODEURL" = "http://nodejs.org/dist/v5.11.0/node-v5.11.0-sunos-x64.tar.gz" ] && NODEHASH="386fed5acd84eb0d9efe958c951db2e9af2dd469d82e745179285b52a087b553"
[ "$NODEURL" = "http://nodejs.org/dist/v5.11.0/node-v5.11.0-sunos-x86.tar.gz" ] && NODEHASH="44dd2e9bae4513bbf586fde1fea11ea310219feece915c21b3a45d4715e14c5e"
[ "$NODEURL" = "http://nodejs.org/dist/v5.11.0/node-v5.11.0.tar.gz" ] && NODEHASH="ef41b01a9e49de5a5d160071166f911d0bc9ff6a0bd0e88ea4863df572d95e91"
[ "$NODEURL" = "http://nodejs.org/dist/v5.11.1/node-v5.11.1-darwin-x64.tar.gz" ] && NODEHASH="7a992f61dc535c696ba2e236e3664ba669680f7e1a204e42166412cc3476503a"
[ "$NODEURL" = "http://nodejs.org/dist/v5.11.1/node-v5.11.1-headers.tar.gz" ] && NODEHASH="85072d2f231a947862984ec79f44b990287a6ba1589342c0ac4fea54d9ec88e2"
[ "$NODEURL" = "http://nodejs.org/dist/v5.11.1/node-v5.11.1-linux-arm64.tar.gz" ] && NODEHASH="8df5fa56ea1f79efc6f8baa9a6784bb1b0596fb7ef1d631694e35a89b3840de6"
[ "$NODEURL" = "http://nodejs.org/dist/v5.11.1/node-v5.11.1-linux-armv6l.tar.gz" ] && NODEHASH="d6ffd43fd0546a5830117bb76979f074a04f9f2323bfc786ca5bd25f254149bc"
[ "$NODEURL" = "http://nodejs.org/dist/v5.11.1/node-v5.11.1-linux-armv7l.tar.gz" ] && NODEHASH="d69ebad9bb86e2d54b6c8774e5bfd055b507dfd4da4b4a0a9ad8543358fd57de"
[ "$NODEURL" = "http://nodejs.org/dist/v5.11.1/node-v5.11.1-linux-ppc64le.tar.gz" ] && NODEHASH="a349b77703902702ec2ec35de27150d19f73ca44045577c577312a5bde9e0b04"
[ "$NODEURL" = "http://nodejs.org/dist/v5.11.1/node-v5.11.1-linux-x64.tar.gz" ] && NODEHASH="d8e30e79a1e4ad56f55ef59facdf913c950e9664528f59f4388e85fdd899dfde"
[ "$NODEURL" = "http://nodejs.org/dist/v5.11.1/node-v5.11.1-linux-x86.tar.gz" ] && NODEHASH="5c8269b21400327eaae94f52d220127a90eca40ab7574f9da07811c48d25570f"
[ "$NODEURL" = "http://nodejs.org/dist/v5.11.1/node-v5.11.1-sunos-x64.tar.gz" ] && NODEHASH="65e80b25ba648b62faae99f4f4df86eeea6ed448c592cb638349a63eefcb5d0f"
[ "$NODEURL" = "http://nodejs.org/dist/v5.11.1/node-v5.11.1-sunos-x86.tar.gz" ] && NODEHASH="24e9095e7fdb1765d2acf569209a194b16c048374a804a97fc5fec123b1b9217"
[ "$NODEURL" = "http://nodejs.org/dist/v5.11.1/node-v5.11.1.tar.gz" ] && NODEHASH="be1211ce415ecc48412ce3e9d5b48cf9ec8e99b34f25b4e1909a02679cc72fab"
[ "$NODEURL" = "http://nodejs.org/dist/v5.12.0/node-v5.12.0-darwin-x64.tar.gz" ] && NODEHASH="bcbfb16896d4b13e08184343420ab00822e9ef09a72f9dbc41ef0cfcc84b99c2"
[ "$NODEURL" = "http://nodejs.org/dist/v5.12.0/node-v5.12.0-headers.tar.gz" ] && NODEHASH="a36e815ac5a4d7d3f486deda97359ae8e01d42259bf4a2be4fbbf41efd585c70"
[ "$NODEURL" = "http://nodejs.org/dist/v5.12.0/node-v5.12.0-linux-arm64.tar.gz" ] && NODEHASH="db02351d2c205a3c60218f937a41a8b8d665f326e7dfa263954ab39f8a8a2bc3"
[ "$NODEURL" = "http://nodejs.org/dist/v5.12.0/node-v5.12.0-linux-armv6l.tar.gz" ] && NODEHASH="f58b9db77eb82830157f814704e8c3b3ba3420079a8ded3ad39302a33e3a30af"
[ "$NODEURL" = "http://nodejs.org/dist/v5.12.0/node-v5.12.0-linux-armv7l.tar.gz" ] && NODEHASH="da51a7025772766453f330b3274a12381995e07d68c99c55357767171af4c538"
[ "$NODEURL" = "http://nodejs.org/dist/v5.12.0/node-v5.12.0-linux-ppc64le.tar.gz" ] && NODEHASH="5cde9c115aade3d3eac494106ad29a2e0743a19516aa8a4ed531783292c7641f"
[ "$NODEURL" = "http://nodejs.org/dist/v5.12.0/node-v5.12.0-linux-ppc64.tar.gz" ] && NODEHASH="a57642f7aa59f7a5248fd5368217f7c7218ae889664ab8974af468797bbfa7bb"
[ "$NODEURL" = "http://nodejs.org/dist/v5.12.0/node-v5.12.0-linux-x64.tar.gz" ] && NODEHASH="c0f459152aa87aba8a019a95899352170db0d8d52c860715c88356cb253fe2c4"
[ "$NODEURL" = "http://nodejs.org/dist/v5.12.0/node-v5.12.0-linux-x86.tar.gz" ] && NODEHASH="1ee1ef4e9f8bfb2976f35ca10658f3828cae10ef462d3fee7c3f159a3e21365b"
[ "$NODEURL" = "http://nodejs.org/dist/v5.12.0/node-v5.12.0-sunos-x64.tar.gz" ] && NODEHASH="e5738517163e9409457ff2fe434aa410b70064ba8639a38173decc430a87c546"
[ "$NODEURL" = "http://nodejs.org/dist/v5.12.0/node-v5.12.0-sunos-x86.tar.gz" ] && NODEHASH="b3831faef7112f4bd71dd4dff3f2296c8ee1f4f8e7b3adfdd205caa91adde198"
[ "$NODEURL" = "http://nodejs.org/dist/v5.12.0/node-v5.12.0.tar.gz" ] && NODEHASH="250c12a561d7319e71e142ee92ab682494c7823d81ce24703c80eb52bdf9ba42"
[ "$NODEURL" = "http://nodejs.org/dist/v5.2.0/node-v5.2.0-darwin-x64.tar.gz" ] && NODEHASH="e3d690b8a1f3aae3caaa57d931aac5ace7dbdea7237d3e9413041c30feecb4e0"
[ "$NODEURL" = "http://nodejs.org/dist/v5.2.0/node-v5.2.0-headers.tar.gz" ] && NODEHASH="72e3e0dfbbb37e85004c6a37a0233464eb10b910775baf9f69f3253de099319e"
[ "$NODEURL" = "http://nodejs.org/dist/v5.2.0/node-v5.2.0-linux-arm64.tar.gz" ] && NODEHASH="3517d9ba80217985cac970272d387f4a905f17e5b87a7c7243efcc1173751531"
[ "$NODEURL" = "http://nodejs.org/dist/v5.2.0/node-v5.2.0-linux-armv6l.tar.gz" ] && NODEHASH="c3f2c2d3105e66d2d895711defb53becc5f8fddde177cd5cc2d981f6d9a5bd4b"
[ "$NODEURL" = "http://nodejs.org/dist/v5.2.0/node-v5.2.0-linux-armv7l.tar.gz" ] && NODEHASH="392cf0f1e1d3c64dbb940ddb750f436310b85f68261714639712816dd539365d"
[ "$NODEURL" = "http://nodejs.org/dist/v5.2.0/node-v5.2.0-linux-x64.tar.gz" ] && NODEHASH="7f31f5db97e1def61454d268d5206a6826385d157f444c21a36230a6c18f40d2"
[ "$NODEURL" = "http://nodejs.org/dist/v5.2.0/node-v5.2.0-linux-x86.tar.gz" ] && NODEHASH="963e3b4e23245db09581901ead2901bc2d516ddc69b7043fcb0cc76263d50152"
[ "$NODEURL" = "http://nodejs.org/dist/v5.2.0/node-v5.2.0-sunos-x64.tar.gz" ] && NODEHASH="3f9eafb213c293ad09637056646ccc7a7746865d65508b01ad425005397fb25f"
[ "$NODEURL" = "http://nodejs.org/dist/v5.2.0/node-v5.2.0-sunos-x86.tar.gz" ] && NODEHASH="4391329491781c0cb1a49360162198d8fe48190dfa9f1a8676b1e12b18a63afe"
[ "$NODEURL" = "http://nodejs.org/dist/v5.2.0/node-v5.2.0.tar.gz" ] && NODEHASH="5df5682f9fdd8e747f652e0b09fed46478a1e3df159797852787a074b8787664"
[ "$NODEURL" = "http://nodejs.org/dist/v5.3.0/node-v5.3.0-darwin-x64.tar.gz" ] && NODEHASH="bfb28ff6a02a6bcb3a77afcb66054dcf44b50e1ccdbeca807865c6220c380b6b"
[ "$NODEURL" = "http://nodejs.org/dist/v5.3.0/node-v5.3.0-headers.tar.gz" ] && NODEHASH="40d7d7982865719af6723e4796d0d773f2d47a64e43133b7a7f89bbac1a52e97"
[ "$NODEURL" = "http://nodejs.org/dist/v5.3.0/node-v5.3.0-linux-arm64.tar.gz" ] && NODEHASH="0a37c919cb2e2511ee7ff60e4fc80266afa3dad7cffa9204dc73da244c3a308a"
[ "$NODEURL" = "http://nodejs.org/dist/v5.3.0/node-v5.3.0-linux-armv6l.tar.gz" ] && NODEHASH="266bedf9f8777c98abd0f28b41b0bb2b0b3d5be888bcfb899bf46fd986188172"
[ "$NODEURL" = "http://nodejs.org/dist/v5.3.0/node-v5.3.0-linux-armv7l.tar.gz" ] && NODEHASH="67205437fd74f7e63219cbe5932f83b064884204eea4503faa6d93f5db2719e1"
[ "$NODEURL" = "http://nodejs.org/dist/v5.3.0/node-v5.3.0-linux-x64.tar.gz" ] && NODEHASH="75b029b30d4a4147d67cf75bf6e034291fb5919c6935ec23f8365cee2d463f12"
[ "$NODEURL" = "http://nodejs.org/dist/v5.3.0/node-v5.3.0-linux-x86.tar.gz" ] && NODEHASH="5e1b2e0aa5277e6dded7bc3cba452035a9695b696e974b02eadcba76070ca7eb"
[ "$NODEURL" = "http://nodejs.org/dist/v5.3.0/node-v5.3.0-sunos-x64.tar.gz" ] && NODEHASH="96be7d379e284012c2da8e08df304896ba7b533fd84be4585b1405bcafbe8c96"
[ "$NODEURL" = "http://nodejs.org/dist/v5.3.0/node-v5.3.0-sunos-x86.tar.gz" ] && NODEHASH="0c28263d118a1819e48d633d5dc62fbc88035cff007246588505eb836eb041af"
[ "$NODEURL" = "http://nodejs.org/dist/v5.3.0/node-v5.3.0.tar.gz" ] && NODEHASH="cc05ff06149c638345835788f448471d264a7e011bf083394f86d5be51975c7e"
[ "$NODEURL" = "http://nodejs.org/dist/v5.4.0/node-v5.4.0-darwin-x64.tar.gz" ] && NODEHASH="efc7422f46ff2c9961ea984ced72b74f9c6c2e4e73f51ad83ff35c63835323f9"
[ "$NODEURL" = "http://nodejs.org/dist/v5.4.0/node-v5.4.0-headers.tar.gz" ] && NODEHASH="6f3de4df2ac30c1205ea228a1a9de6acd841255ee19229169bbb92318a75db0b"
[ "$NODEURL" = "http://nodejs.org/dist/v5.4.0/node-v5.4.0-linux-arm64.tar.gz" ] && NODEHASH="0cb2c093e75090281423a2b3681629c663c83dac4587a12b77022afccd7aedc0"
[ "$NODEURL" = "http://nodejs.org/dist/v5.4.0/node-v5.4.0-linux-armv6l.tar.gz" ] && NODEHASH="108e6314f0b88ab6fdebe85885797a59dbfd7c4857f2976ce5b98c981162e01b"
[ "$NODEURL" = "http://nodejs.org/dist/v5.4.0/node-v5.4.0-linux-armv7l.tar.gz" ] && NODEHASH="e69b3f90b56f027dd0435b8b7a2f4ffa7ed3d8185e2162a08d6d2cf097370a97"
[ "$NODEURL" = "http://nodejs.org/dist/v5.4.0/node-v5.4.0-linux-x64.tar.gz" ] && NODEHASH="f037e2734f52b9de63e6d4a4e80756477b843e6f106e0be05591a16b71ec2bd0"
[ "$NODEURL" = "http://nodejs.org/dist/v5.4.0/node-v5.4.0-linux-x86.tar.gz" ] && NODEHASH="8c3d8afb79d9ba5f381ed549521ca8183975dfad2d50290686657b8ce4304a4c"
[ "$NODEURL" = "http://nodejs.org/dist/v5.4.0/node-v5.4.0-sunos-x64.tar.gz" ] && NODEHASH="97a90d9ee302949b5b8f0e1d64933cc1d64725b09d2eb034bb24deb4c336a511"
[ "$NODEURL" = "http://nodejs.org/dist/v5.4.0/node-v5.4.0-sunos-x86.tar.gz" ] && NODEHASH="dcfe2748bcbc6bd6250be0d91b0f3a432b4e01a8d809fdd769a9b8e646e009a3"
[ "$NODEURL" = "http://nodejs.org/dist/v5.4.0/node-v5.4.0.tar.gz" ] && NODEHASH="1dfe37a00cf0ed62beb73071f571ac56697f544a98cc2ff3318faec6363d72ab"
[ "$NODEURL" = "http://nodejs.org/dist/v5.4.1/node-v5.4.1-darwin-x64.tar.gz" ] && NODEHASH="c523472a5972823e8b6baf2419f837885321c772612ec508a65614c758e25a46"
[ "$NODEURL" = "http://nodejs.org/dist/v5.4.1/node-v5.4.1-headers.tar.gz" ] && NODEHASH="fd91d8fcbe72c76d1e58ddb3ba8bc08073c8c1e6d595bfe4b8b22e9b92855ed7"
[ "$NODEURL" = "http://nodejs.org/dist/v5.4.1/node-v5.4.1-linux-arm64.tar.gz" ] && NODEHASH="362ae4539b6be075b6757ba689f0ae522cfc9340c81061aca880f92fce9595c7"
[ "$NODEURL" = "http://nodejs.org/dist/v5.4.1/node-v5.4.1-linux-armv6l.tar.gz" ] && NODEHASH="21617e86758f1f95a3b9444be965aa87907410d786529cfd6aa2169ab7b5e15b"
[ "$NODEURL" = "http://nodejs.org/dist/v5.4.1/node-v5.4.1-linux-armv7l.tar.gz" ] && NODEHASH="af36dff32934dea9446673a5453efcda6e4621ce7fa73e0a401f1444c2f929af"
[ "$NODEURL" = "http://nodejs.org/dist/v5.4.1/node-v5.4.1-linux-x64.tar.gz" ] && NODEHASH="1880f3421da5579678803a523c314b345f5db00799b51b7fd9484a3248efc068"
[ "$NODEURL" = "http://nodejs.org/dist/v5.4.1/node-v5.4.1-linux-x86.tar.gz" ] && NODEHASH="184790d2ec8f95a75e7e746b3007e848aeba91be14ad7cec415b425df0df92e5"
[ "$NODEURL" = "http://nodejs.org/dist/v5.4.1/node-v5.4.1-sunos-x64.tar.gz" ] && NODEHASH="956b3b0790b5742765966c6851e43c6fca298ecfb8c7102fd87e5fb6294bebd4"
[ "$NODEURL" = "http://nodejs.org/dist/v5.4.1/node-v5.4.1-sunos-x86.tar.gz" ] && NODEHASH="a3eb11d58760b4b9007ac882c02e7608549fe7f65124ca90a387ba71c524b3e2"
[ "$NODEURL" = "http://nodejs.org/dist/v5.4.1/node-v5.4.1.tar.gz" ] && NODEHASH="78455ef2e3dea06b7d13d393c36711009048a91e5de5892523ec4a9be5a55e0c"
[ "$NODEURL" = "http://nodejs.org/dist/v5.5.0/node-v5.5.0-darwin-x64.tar.gz" ] && NODEHASH="d4fd29e2d501963235104fc715fb0b55b302a40b605f432c456069606b939a46"
[ "$NODEURL" = "http://nodejs.org/dist/v5.5.0/node-v5.5.0-headers.tar.gz" ] && NODEHASH="67ebc3989b98980c13ef17c1336b98ed2975aa093d9b69f399dfb93de5ae74b4"
[ "$NODEURL" = "http://nodejs.org/dist/v5.5.0/node-v5.5.0-linux-arm64.tar.gz" ] && NODEHASH="a9ebfce36675cc8d5e1bea6fa57de7fd80e8016f5957340831fcd03560e59845"
[ "$NODEURL" = "http://nodejs.org/dist/v5.5.0/node-v5.5.0-linux-armv6l.tar.gz" ] && NODEHASH="a156dfda7fa00ac7ea86ac3cff8d445c44bcec3c677db375776a0489ad7155bd"
[ "$NODEURL" = "http://nodejs.org/dist/v5.5.0/node-v5.5.0-linux-armv7l.tar.gz" ] && NODEHASH="cf9832efa0cac6365b4fec83df5dee8ecb67d39931c4e7cdaaa4cb933fb4b78b"
[ "$NODEURL" = "http://nodejs.org/dist/v5.5.0/node-v5.5.0-linux-x64.tar.gz" ] && NODEHASH="3e593d91b6d2ad871efaaf8e9a17b3608ca98904959bcfb7c42e6acce89e80f4"
[ "$NODEURL" = "http://nodejs.org/dist/v5.5.0/node-v5.5.0-linux-x86.tar.gz" ] && NODEHASH="e384f8beb392cd5df882cd3401e449371190bc139010d391888c2f3346fb5107"
[ "$NODEURL" = "http://nodejs.org/dist/v5.5.0/node-v5.5.0-sunos-x64.tar.gz" ] && NODEHASH="e8abb364fc8db51e3a6c8dc14f24ec2c289940386c18da4ecb33462021c1bcef"
[ "$NODEURL" = "http://nodejs.org/dist/v5.5.0/node-v5.5.0-sunos-x86.tar.gz" ] && NODEHASH="c49c8ce8c997da2cf1ee1cccb55ed457d31aa85d4ec6f58edd868daaf29ddc45"
[ "$NODEURL" = "http://nodejs.org/dist/v5.5.0/node-v5.5.0.tar.gz" ] && NODEHASH="d69b18cc20699a35434858fb853997616762280610a510ec4b4ff1a94798b432"
[ "$NODEURL" = "http://nodejs.org/dist/v5.6.0/node-v5.6.0-darwin-x64.tar.gz" ] && NODEHASH="1ef8f5b627cf980b0d242d5b70be3c6fbefc8e61ecfcaf97930965d68c927bd9"
[ "$NODEURL" = "http://nodejs.org/dist/v5.6.0/node-v5.6.0-headers.tar.gz" ] && NODEHASH="3eb317571329d1ff345aba83e94d0fd6bf2043697d032fdcffb92265e11b61be"
[ "$NODEURL" = "http://nodejs.org/dist/v5.6.0/node-v5.6.0-linux-arm64.tar.gz" ] && NODEHASH="3b22bb5e1579d6e45f31da88c17baeb17a12ecb297c1c69447de6030d626b08d"
[ "$NODEURL" = "http://nodejs.org/dist/v5.6.0/node-v5.6.0-linux-armv6l.tar.gz" ] && NODEHASH="5df9773a946ac571dad2bbec0d5a9e05201ea16c6512731b36018ffc9d06f902"
[ "$NODEURL" = "http://nodejs.org/dist/v5.6.0/node-v5.6.0-linux-armv7l.tar.gz" ] && NODEHASH="7e258c59576beddd475fc33a8e57b153f0455cf1a5d801aedc6cee17137e9bae"
[ "$NODEURL" = "http://nodejs.org/dist/v5.6.0/node-v5.6.0-linux-x64.tar.gz" ] && NODEHASH="6b10e446b5a1227673b87d840e9a500f5d2dbd2b806d96e2d81d634c3381a5f1"
[ "$NODEURL" = "http://nodejs.org/dist/v5.6.0/node-v5.6.0-linux-x86.tar.gz" ] && NODEHASH="f6fc3391f48a3fc2d077dc0e1673fc3934eb2b9465cbeab334e3967d1503ba49"
[ "$NODEURL" = "http://nodejs.org/dist/v5.6.0/node-v5.6.0-sunos-x64.tar.gz" ] && NODEHASH="a7010c2f7ddc5f6fff7f4d04e1f0973edd387ebba891c8905323daf2ba499a4c"
[ "$NODEURL" = "http://nodejs.org/dist/v5.6.0/node-v5.6.0-sunos-x86.tar.gz" ] && NODEHASH="30ca440291a06e6f7af77ba072dbe4ce771e80dcd6ce4366ff1c6fa18df45f75"
[ "$NODEURL" = "http://nodejs.org/dist/v5.6.0/node-v5.6.0.tar.gz" ] && NODEHASH="3af2cc5e5970afc83e59f2065fea2e2df846a544a100cd3c0527f0db05bec27f"
[ "$NODEURL" = "http://nodejs.org/dist/v5.7.0/node-v5.7.0-darwin-x64.tar.gz" ] && NODEHASH="a68a9d45527077e1e044a4036a9b0fc803faa46c97e30fed71b77a759b4fa2fe"
[ "$NODEURL" = "http://nodejs.org/dist/v5.7.0/node-v5.7.0-headers.tar.gz" ] && NODEHASH="1ac837924a7fe48f56b04b0952fa15ae6cde6ef968fd10118ca07a09fd182107"
[ "$NODEURL" = "http://nodejs.org/dist/v5.7.0/node-v5.7.0-linux-arm64.tar.gz" ] && NODEHASH="2b4de1cca92da4aaec0aec0bc767d7a60d8378e830987abc668176d9b6603ccd"
[ "$NODEURL" = "http://nodejs.org/dist/v5.7.0/node-v5.7.0-linux-armv6l.tar.gz" ] && NODEHASH="844f6707f105f31096579c74ad9928c5a9cce85b2a245bb1d697c06e6a9b1ee8"
[ "$NODEURL" = "http://nodejs.org/dist/v5.7.0/node-v5.7.0-linux-armv7l.tar.gz" ] && NODEHASH="0712cbe3f05e1828d9cf900ad17007184abc70746bdb2d2064b645f1d3be68c2"
[ "$NODEURL" = "http://nodejs.org/dist/v5.7.0/node-v5.7.0-linux-ppc64le.tar.gz" ] && NODEHASH="ff7d52a87ce152817e3e4e9dba24530b629729539648a0b347bb70d733591a07"
[ "$NODEURL" = "http://nodejs.org/dist/v5.7.0/node-v5.7.0-linux-x64.tar.gz" ] && NODEHASH="ae24ae3076393e7968316098ddbb0221bde0830a0e9d878c6493604e1cc553c1"
[ "$NODEURL" = "http://nodejs.org/dist/v5.7.0/node-v5.7.0-linux-x86.tar.gz" ] && NODEHASH="09776060a502118765144284c7ad32bd6a737e681819e8349489b9a015a45bbc"
[ "$NODEURL" = "http://nodejs.org/dist/v5.7.0/node-v5.7.0-sunos-x64.tar.gz" ] && NODEHASH="d8acb8f8fb779d26cda6d3770d48a1f584a0e35ddf154adb4b883bcc4b48a2f2"
[ "$NODEURL" = "http://nodejs.org/dist/v5.7.0/node-v5.7.0-sunos-x86.tar.gz" ] && NODEHASH="208bba5c8a7cf375fa062d489fc413b96665438fd0c3e4a98ca39397c79691d5"
[ "$NODEURL" = "http://nodejs.org/dist/v5.7.0/node-v5.7.0.tar.gz" ] && NODEHASH="2338b46a2f45fbb747089c66931f62555f25a5928511d3a43bbb3a39dcded2d8"
[ "$NODEURL" = "http://nodejs.org/dist/v5.7.1/node-v5.7.1-darwin-x64.tar.gz" ] && NODEHASH="25627633163e6ad47e62cd9aaab04e47707b51ecc5aaa05f35a2d6419dbe054c"
[ "$NODEURL" = "http://nodejs.org/dist/v5.7.1/node-v5.7.1-headers.tar.gz" ] && NODEHASH="6d2241bc1e5049de3651873d63b84c37f8684027b1a34928ce6bbe745057ca70"
[ "$NODEURL" = "http://nodejs.org/dist/v5.7.1/node-v5.7.1-linux-arm64.tar.gz" ] && NODEHASH="b075aa249eb1e00e1e84e6f5964d4f93c39aa6d817c25280bf885bfcf906c7fc"
[ "$NODEURL" = "http://nodejs.org/dist/v5.7.1/node-v5.7.1-linux-armv6l.tar.gz" ] && NODEHASH="ddde92eb9b80e58efec88138fa7351e21000b22414ebff754c7ab38dedbab89f"
[ "$NODEURL" = "http://nodejs.org/dist/v5.7.1/node-v5.7.1-linux-armv7l.tar.gz" ] && NODEHASH="2370e6ad65dedc1eadcc1b3a481e922a8b5afacee9e90eb6700fee876eb59a5b"
[ "$NODEURL" = "http://nodejs.org/dist/v5.7.1/node-v5.7.1-linux-ppc64le.tar.gz" ] && NODEHASH="f3de80e67a3f055bb3e2ecdb9e4b6cc45183a6cbee2b728f243d46a6fd94f542"
[ "$NODEURL" = "http://nodejs.org/dist/v5.7.1/node-v5.7.1-linux-x64.tar.gz" ] && NODEHASH="fcded78b45549e2195eecb36138ba29b6f353d0d136d4e8b80648770418f1e5b"
[ "$NODEURL" = "http://nodejs.org/dist/v5.7.1/node-v5.7.1-linux-x86.tar.gz" ] && NODEHASH="4382b7366f3448d42f09d63f7dfccf45bf67f8be94d2bfe7203bdc79d0ae64e8"
[ "$NODEURL" = "http://nodejs.org/dist/v5.7.1/node-v5.7.1-sunos-x64.tar.gz" ] && NODEHASH="0ebab64c25a417b741f666db1765c7358e413d895d4dd0ca9e967fdbb1770a72"
[ "$NODEURL" = "http://nodejs.org/dist/v5.7.1/node-v5.7.1-sunos-x86.tar.gz" ] && NODEHASH="1deb8d1eee46dfdb15c01fea1b283cc9115d78d2e4757ab6192b80bf52d61f99"
[ "$NODEURL" = "http://nodejs.org/dist/v5.7.1/node-v5.7.1.tar.gz" ] && NODEHASH="ecd4956335c6b86f1b72fd203abd9a833717e6800edb12208d0e3ce625330198"
[ "$NODEURL" = "http://nodejs.org/dist/v5.8.0/node-v5.8.0-darwin-x64.tar.gz" ] && NODEHASH="8c16f21a1c8882ba5875d0da617c817aa5005e514bd460dbf32aaeb3ffa477fe"
[ "$NODEURL" = "http://nodejs.org/dist/v5.8.0/node-v5.8.0-headers.tar.gz" ] && NODEHASH="1f8501149f7160d30341df82a9634e0e65256d096f092c68ad820bcd435000eb"
[ "$NODEURL" = "http://nodejs.org/dist/v5.8.0/node-v5.8.0-linux-arm64.tar.gz" ] && NODEHASH="0c2c0fa859c5be13cd1404f3fb14d37e38a67fb2fc075c7a37d4ae70374544bf"
[ "$NODEURL" = "http://nodejs.org/dist/v5.8.0/node-v5.8.0-linux-armv6l.tar.gz" ] && NODEHASH="54c362af9bc80b9e283bee7807fb2b1e9207ac77e61207b13fcf9f9acecd293f"
[ "$NODEURL" = "http://nodejs.org/dist/v5.8.0/node-v5.8.0-linux-armv7l.tar.gz" ] && NODEHASH="cdb0355fc41ac7010926f1a35f07f0dfe94020572d70296f91e60d18ab4873f5"
[ "$NODEURL" = "http://nodejs.org/dist/v5.8.0/node-v5.8.0-linux-ppc64le.tar.gz" ] && NODEHASH="59f4745346786fdad8d58022b5eefc7cd5ec143d5cce03c66c1243e531fd10e8"
[ "$NODEURL" = "http://nodejs.org/dist/v5.8.0/node-v5.8.0-linux-x64.tar.gz" ] && NODEHASH="3d33efa2421ed0769cc23ac172203b7db9fe4914a186009ddbac28c9aadb0896"
[ "$NODEURL" = "http://nodejs.org/dist/v5.8.0/node-v5.8.0-linux-x86.tar.gz" ] && NODEHASH="b302ce9d5d156857576d892cfe8108363b30c20bd3133011fb2050d56bb238e8"
[ "$NODEURL" = "http://nodejs.org/dist/v5.8.0/node-v5.8.0-sunos-x64.tar.gz" ] && NODEHASH="235783950678e3f343c5ce8e567a4600141ff9ad87aca0364bea659a82563f6c"
[ "$NODEURL" = "http://nodejs.org/dist/v5.8.0/node-v5.8.0-sunos-x86.tar.gz" ] && NODEHASH="5edb23172d0a2b064dddb090ed5dae06a12db50dcc71f652748076d994f607ba"
[ "$NODEURL" = "http://nodejs.org/dist/v5.8.0/node-v5.8.0.tar.gz" ] && NODEHASH="e8cd580cec1c62053e6357633b9e46d49df1fd5b9fc5c804ca59ba80d722b898"
[ "$NODEURL" = "http://nodejs.org/dist/v5.9.0/node-v5.9.0-darwin-x64.tar.gz" ] && NODEHASH="6417022026c30cfb1b8af92f1434c1b202548076ada636708e1874d8af78197b"
[ "$NODEURL" = "http://nodejs.org/dist/v5.9.0/node-v5.9.0-headers.tar.gz" ] && NODEHASH="546c6b43f193ad57638cadded1eb5ee4630b585f4f966e228cd2ae06f44b6c76"
[ "$NODEURL" = "http://nodejs.org/dist/v5.9.0/node-v5.9.0-linux-arm64.tar.gz" ] && NODEHASH="8ce0653a98a7507dc15bd7425154af1113685d054b6dee2c9701fed401feb12a"
[ "$NODEURL" = "http://nodejs.org/dist/v5.9.0/node-v5.9.0-linux-armv6l.tar.gz" ] && NODEHASH="a69cc76e388b44a4c6de8007e0fad67f1308b32284894b4c483180c4aaf10cb4"
[ "$NODEURL" = "http://nodejs.org/dist/v5.9.0/node-v5.9.0-linux-armv7l.tar.gz" ] && NODEHASH="b7ac2bf673eed236ac636d755874d90aa3563b55caed84730c8342cbdffc4ce3"
[ "$NODEURL" = "http://nodejs.org/dist/v5.9.0/node-v5.9.0-linux-ppc64le.tar.gz" ] && NODEHASH="a2996dc3323579f2d9f306cb91a21a9ed6ec534791003eeed3e9114a0a1b27dd"
[ "$NODEURL" = "http://nodejs.org/dist/v5.9.0/node-v5.9.0-linux-x64.tar.gz" ] && NODEHASH="99c4136cf61761fac5ac57f80544140a3793b63e00a65d4a0e528c9db328bf40"
[ "$NODEURL" = "http://nodejs.org/dist/v5.9.0/node-v5.9.0-linux-x86.tar.gz" ] && NODEHASH="fca2df390ff964951cc2d18150d17dfe474e5b59864415976b8bad9452f725eb"
[ "$NODEURL" = "http://nodejs.org/dist/v5.9.0/node-v5.9.0-sunos-x64.tar.gz" ] && NODEHASH="47a45169838b74f62c0dbb31643af2f2c2e86bae0920d030113b3aeb0f29275b"
[ "$NODEURL" = "http://nodejs.org/dist/v5.9.0/node-v5.9.0-sunos-x86.tar.gz" ] && NODEHASH="835f6d8ec09df20f5280851da0eec44f97d64a2f163a213efe2eac668d0a852a"
[ "$NODEURL" = "http://nodejs.org/dist/v5.9.0/node-v5.9.0.tar.gz" ] && NODEHASH="e5175a66481fd9389ef74ef8e8c164d052f139664deffc9e79241d6234760f3e"
[ "$NODEURL" = "http://nodejs.org/dist/v5.9.1/node-v5.9.1-darwin-x64.tar.gz" ] && NODEHASH="90dbbd2072582f0373a738114131112f3f8a2c7f7f64bbf4991a51d2808d4935"
[ "$NODEURL" = "http://nodejs.org/dist/v5.9.1/node-v5.9.1-headers.tar.gz" ] && NODEHASH="fb16ac5bb0c49f7b4de6a93744fee638c55165f8fac88157a324dc358ae24cb5"
[ "$NODEURL" = "http://nodejs.org/dist/v5.9.1/node-v5.9.1-linux-arm64.tar.gz" ] && NODEHASH="09fd524d987e3c70aed7aa52d21f6448fe06cdd05c627a6de326384b98a3bb0e"
[ "$NODEURL" = "http://nodejs.org/dist/v5.9.1/node-v5.9.1-linux-armv6l.tar.gz" ] && NODEHASH="755965b20e4fa991072e7bb07937bd3c075b689b10d21161bfb34037dd5c52b9"
[ "$NODEURL" = "http://nodejs.org/dist/v5.9.1/node-v5.9.1-linux-armv7l.tar.gz" ] && NODEHASH="b4a7880a9906175121a40ff8862009bc2bf80ac2bfd5e0e4aab6d9088b52fe21"
[ "$NODEURL" = "http://nodejs.org/dist/v5.9.1/node-v5.9.1-linux-ppc64le.tar.gz" ] && NODEHASH="0211aa951355eddadb07ffae0b945b954df7b8ba72231d75d2a8a09335a8f0a6"
[ "$NODEURL" = "http://nodejs.org/dist/v5.9.1/node-v5.9.1-linux-x64.tar.gz" ] && NODEHASH="4b9951e6afd75010f53264fc1a61e2d92ae23a590bbb58fea3e62d6f0104f657"
[ "$NODEURL" = "http://nodejs.org/dist/v5.9.1/node-v5.9.1-linux-x86.tar.gz" ] && NODEHASH="749eb56ae38ee0cb4ac659d6de50954a4e57bd2796fda6bc75725f7b98f21887"
[ "$NODEURL" = "http://nodejs.org/dist/v5.9.1/node-v5.9.1-sunos-x64.tar.gz" ] && NODEHASH="17feb55c0ca64d1fb57e7aa54d3226d1b72adc3484c276f3ac040891b566325c"
[ "$NODEURL" = "http://nodejs.org/dist/v5.9.1/node-v5.9.1-sunos-x86.tar.gz" ] && NODEHASH="8a430122c7ad4271414d3d5b735277d19abe7fc8a3f7d3789630b58908166740"
[ "$NODEURL" = "http://nodejs.org/dist/v5.9.1/node-v5.9.1.tar.gz" ] && NODEHASH="9bfa882ba832c526add1f939121c79d95b9c592785ec8afe043086a0c7a30c18"
[ "$NODEURL" = "http://nodejs.org/dist/v6.0.0/node-v6.0.0-darwin-x64.tar.gz" ] && NODEHASH="5be18c5571ce2b33918a3dd0be2f18d2a6939a60754026fb7e7967bec09d9e5e"
[ "$NODEURL" = "http://nodejs.org/dist/v6.0.0/node-v6.0.0-headers.tar.gz" ] && NODEHASH="f72de0e9403dd4ab7b6eec7266a498ba561afe5d5c7aff77b83894b43c959beb"
[ "$NODEURL" = "http://nodejs.org/dist/v6.0.0/node-v6.0.0-linux-arm64.tar.gz" ] && NODEHASH="c4d7da92f76e77d27ef5650ad01085baad74439fab15e5143a8e9fc6cad13101"
[ "$NODEURL" = "http://nodejs.org/dist/v6.0.0/node-v6.0.0-linux-armv6l.tar.gz" ] && NODEHASH="c4e9ce8d1e65290e5b00ffb4d3c27981b033141eb7eeccb108b47ca7585d81c8"
[ "$NODEURL" = "http://nodejs.org/dist/v6.0.0/node-v6.0.0-linux-armv7l.tar.gz" ] && NODEHASH="7a29af935545c07ea2eb5bbf6b6fb643765d50ee69d1b4193345a50fb8cb1093"
[ "$NODEURL" = "http://nodejs.org/dist/v6.0.0/node-v6.0.0-linux-ppc64le.tar.gz" ] && NODEHASH="7044f1049d6d8673cff7b0d38cd8c606be56ba149d3d32b34ca3f017df731a2a"
[ "$NODEURL" = "http://nodejs.org/dist/v6.0.0/node-v6.0.0-linux-x64.tar.gz" ] && NODEHASH="78fa76c77a1168095cf5b8a5018e00e7212d11e485cf10c77ce1c8af4955cdd3"
[ "$NODEURL" = "http://nodejs.org/dist/v6.0.0/node-v6.0.0-linux-x86.tar.gz" ] && NODEHASH="51321999e9706d9c24ea3689a03d049ad96657933228d3ed25d7710bc5d9e9bc"
[ "$NODEURL" = "http://nodejs.org/dist/v6.0.0/node-v6.0.0-sunos-x64.tar.gz" ] && NODEHASH="dc1345210205380af8fa16a63dca996803af105d7b0e5aa4160df7ed88523f08"
[ "$NODEURL" = "http://nodejs.org/dist/v6.0.0/node-v6.0.0-sunos-x86.tar.gz" ] && NODEHASH="61259a9794d2ed7b0e5331d5d2158da06f6ed5d31cbeae6d71da4d5e941a7754"
[ "$NODEURL" = "http://nodejs.org/dist/v6.0.0/node-v6.0.0.tar.gz" ] && NODEHASH="6db56d81a4e07ff4017809ad4ee49a5256fad804cd8527f8724f8e149e3dfc32"
[ "$NODEURL" = "http://nodejs.org/dist/v6.1.0/node-v6.1.0-darwin-x64.tar.gz" ] && NODEHASH="481aeab001b6c79589daf3e1f8f5b9ad7021a3979b49fd842313010482ff7b56"
[ "$NODEURL" = "http://nodejs.org/dist/v6.1.0/node-v6.1.0-headers.tar.gz" ] && NODEHASH="9419fce25791df7afd27d173a1148ef9f53c349cd9a2e8d9bdac5276d80a1bb5"
[ "$NODEURL" = "http://nodejs.org/dist/v6.1.0/node-v6.1.0-linux-arm64.tar.gz" ] && NODEHASH="87670387877d1cbe36642970e3ca84a77121d3200771b80ec2286bc261e060f7"
[ "$NODEURL" = "http://nodejs.org/dist/v6.1.0/node-v6.1.0-linux-armv6l.tar.gz" ] && NODEHASH="aeff5e4b77fba25feb19ec4739116373234e93aff0d2c9662c4ebd87fa4b34ae"
[ "$NODEURL" = "http://nodejs.org/dist/v6.1.0/node-v6.1.0-linux-armv7l.tar.gz" ] && NODEHASH="55d7a2e5071b13db6fe5d952dc5ee5d2223ae4a0de2f2c2d7a68d65e1cfa377b"
[ "$NODEURL" = "http://nodejs.org/dist/v6.1.0/node-v6.1.0-linux-ppc64le.tar.gz" ] && NODEHASH="bca98e8a4f5c11dee9eb4fec2bd0a42c1f8a921863bba417fc88c7274a0c0c04"
[ "$NODEURL" = "http://nodejs.org/dist/v6.1.0/node-v6.1.0-linux-x64.tar.gz" ] && NODEHASH="ce46dd0188181b70661b9162feffdbd8a860cb75cb6661c37d6d61982e3f72c5"
[ "$NODEURL" = "http://nodejs.org/dist/v6.1.0/node-v6.1.0-linux-x86.tar.gz" ] && NODEHASH="07d6f0b11b98c876f2d029722696da6cc9b93a2ce6b34abb7a517e0afb04fb34"
[ "$NODEURL" = "http://nodejs.org/dist/v6.1.0/node-v6.1.0-sunos-x64.tar.gz" ] && NODEHASH="d7cbb0bfe0915ac7bdb37e87ea7145f6e86ef2943e77f306c648e47d3d33c898"
[ "$NODEURL" = "http://nodejs.org/dist/v6.1.0/node-v6.1.0-sunos-x86.tar.gz" ] && NODEHASH="38ec9537ab2fbf6112365b3a291ebe962d14b384b474e54fa77227b2e98624c9"
[ "$NODEURL" = "http://nodejs.org/dist/v6.1.0/node-v6.1.0.tar.gz" ] && NODEHASH="9e67ef0b8611e16e6e311eccf0489a50fe76ceebeea3023ef4f51be647ae4bc3"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.0/node-v6.2.0-darwin-x64.tar.gz" ] && NODEHASH="1c8c16bb3df22c7c98cf0c118bd183838b848e36ef1448992cf511c044339823"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.0/node-v6.2.0-headers.tar.gz" ] && NODEHASH="1127a30aa0e728786e4c1b2a568dd74eab8957a271764cec06d3562febee6baa"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.0/node-v6.2.0-linux-arm64.tar.gz" ] && NODEHASH="bac296f56d071e147e33e7f027efc8b0422a786e3943b18c64c8c8fbf2abae96"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.0/node-v6.2.0-linux-armv6l.tar.gz" ] && NODEHASH="c297855fd11dc9d58c523f3665b18e5ff932145aabb27c389da3b6b27553fe37"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.0/node-v6.2.0-linux-armv7l.tar.gz" ] && NODEHASH="4804a486b0b0db0f52fc273ccf688f83c06e3fc17641824877e2921518d0e50d"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.0/node-v6.2.0-linux-ppc64le.tar.gz" ] && NODEHASH="fbbeaaa5431c6cc5007567c35f13b87c2917d704ae6fd8f6db7792f1ba5349b0"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.0/node-v6.2.0-linux-ppc64.tar.gz" ] && NODEHASH="69f8f3ffec15f1cddca3f8fca53471123e22db40ae58bea9bd65503d900267d8"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.0/node-v6.2.0-linux-x64.tar.gz" ] && NODEHASH="661dba369c277603fa6d0182c4ea7ff074ba6bacd19171826271f872afd6aaa7"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.0/node-v6.2.0-linux-x86.tar.gz" ] && NODEHASH="47c3d613362afd9c64e6468fba0d3f6407ead7d3ea6447deb297c6551dd9d0d4"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.0/node-v6.2.0-sunos-x64.tar.gz" ] && NODEHASH="f1e8447ff644598c2c24eb6effa12a9d32c6fdad7fa832065dd5f381339b723f"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.0/node-v6.2.0-sunos-x86.tar.gz" ] && NODEHASH="bbfce19c47265676fb44a6a1fa10e2e9ec710c8a332a801aa7e3e58648e851a3"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.0/node-v6.2.0.tar.gz" ] && NODEHASH="fcfd81847d78abb690aafaf46223d3b59aedc54fb200a2ab2bfd9a39ad6e94b9"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.1/node-v6.2.1-darwin-x64.tar.gz" ] && NODEHASH="a6e99c2e482133b0f2db6bb8284ef79ea60c554e8dbfc24a1a691671cc5d0517"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.1/node-v6.2.1-headers.tar.gz" ] && NODEHASH="f55aa8ea6de59e074c4f9d5ab65b77e68bcf607971be05ff31b88acb91714f4f"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.1/node-v6.2.1-linux-arm64.tar.gz" ] && NODEHASH="86f7498f0c355e8f8c17cecaf7e0bd6f68e189e00e0adb6700560e291edce08d"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.1/node-v6.2.1-linux-armv6l.tar.gz" ] && NODEHASH="f791c57f336fc67c7231c9c54799b4e39427a2de99decf47898d57353b0fb64b"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.1/node-v6.2.1-linux-armv7l.tar.gz" ] && NODEHASH="01ac0cbae90a88f73b016db7e8052d05f7ab7083d5a14b6b10067a41eea538ff"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.1/node-v6.2.1-linux-ppc64le.tar.gz" ] && NODEHASH="2ca673b76add031e6244b36e2cd3c1da1982e21aec84eeda24bcbc1b77772ab2"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.1/node-v6.2.1-linux-ppc64.tar.gz" ] && NODEHASH="4fa09f89af1b835caf4be73ca4c7a02a1bc640c36c4ec512dbdb0827315003e9"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.1/node-v6.2.1-linux-x64.tar.gz" ] && NODEHASH="c6ae9c90858fb47a2915ad6494e5eb8e6f34f4512de1d5a461e5fb0c003590b1"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.1/node-v6.2.1-linux-x86.tar.gz" ] && NODEHASH="f5656c288c1dfc760a23000d8dc1eb3d0ff484a7798e5f28718a728ae96e9125"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.1/node-v6.2.1-sunos-x64.tar.gz" ] && NODEHASH="c7e931670bca8055ed6d090bab276bee5c2db570e74c0828f64a2b5c6dc9e973"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.1/node-v6.2.1-sunos-x86.tar.gz" ] && NODEHASH="39ac16d0b200a29261767d6aa99d1a9da343e7c4b629ce332f7d6b9fcfb1767a"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.1/node-v6.2.1.tar.gz" ] && NODEHASH="fa26d4380150fbb69a48bddaec6143e176effe043cfe44e512c5ef9d22618b63"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.2/node-v6.2.2-darwin-x64.tar.gz" ] && NODEHASH="03b9eadd71d73daf2a25c8ea833454b326cb702f717a39f1b2a1324179cab5fa"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.2/node-v6.2.2-headers.tar.gz" ] && NODEHASH="c02d78470afb83d4473f196a5ff3725f028cd661c4d50d0337f817c7af02e0e7"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.2/node-v6.2.2-linux-arm64.tar.gz" ] && NODEHASH="1eaac04e632e633197c764a65817909667a700a657b1de463a45efcd40d236c7"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.2/node-v6.2.2-linux-armv6l.tar.gz" ] && NODEHASH="92b41e1c31a64024c5386df3cb6b9c0e00ecb097c2f7d0413364e20b7b836ae9"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.2/node-v6.2.2-linux-armv7l.tar.gz" ] && NODEHASH="ecaa5ccdad2d3e9efc8944e5c57971753bacbd7c171b6a4445e76e5fb9ebc69b"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.2/node-v6.2.2-linux-ppc64le.tar.gz" ] && NODEHASH="b3e2e27ad4e52610fc971ef7c32cfb7c3f9db03a0b49e8bd422383ff30197263"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.2/node-v6.2.2-linux-ppc64.tar.gz" ] && NODEHASH="ff307b8602808e799cef488c55f1fe646cd169c44464c4c53b56e61d181d06a6"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.2/node-v6.2.2-linux-x64.tar.gz" ] && NODEHASH="7a6df881183e70839857b51653811aaabc49a2ffb93416a1c9bd333dcef84ea3"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.2/node-v6.2.2-linux-x86.tar.gz" ] && NODEHASH="71a3ec010acaa3c5d26429b6670f4895cd494199ffa07b1773e63619559c32c1"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.2/node-v6.2.2-sunos-x64.tar.gz" ] && NODEHASH="d4a4c601ee6ce311193c1445ca2efbad75a6adbe407ce787381c4b620ac62eeb"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.2/node-v6.2.2-sunos-x86.tar.gz" ] && NODEHASH="94fecfccdc903d0df685850fa8a34c28528e79634bea7569ca7b570a39d2a7ce"
[ "$NODEURL" = "http://nodejs.org/dist/v6.2.2/node-v6.2.2.tar.gz" ] && NODEHASH="b6baee57a0ede496c7c7765001f7495ad74c8dfe8c34f1a6fb2cd5d8d526ffce"
[ "$NODEURL" = "http://nodejs.org/dist/v6.3.0/node-v6.3.0-darwin-x64.tar.gz" ] && NODEHASH="5c711a62e464f1d455c31afe2a62e9866eeaf1c23d977b57a60285d0bd040ba5"
[ "$NODEURL" = "http://nodejs.org/dist/v6.3.0/node-v6.3.0-headers.tar.gz" ] && NODEHASH="9275894c3ed6373068cddeeb968e4d2ceba76368b6cd3b01aca79f0b592badd6"
[ "$NODEURL" = "http://nodejs.org/dist/v6.3.0/node-v6.3.0-linux-arm64.tar.gz" ] && NODEHASH="58995c3f91962fc4383696f9c64763b3cd27d9b5903b4cf2a5ccfe86c8258e9f"
[ "$NODEURL" = "http://nodejs.org/dist/v6.3.0/node-v6.3.0-linux-armv6l.tar.gz" ] && NODEHASH="d8a440f56cca21f0b4879802c798f74e3f2dcc5ba16a1c56c0f536bb501b56f5"
[ "$NODEURL" = "http://nodejs.org/dist/v6.3.0/node-v6.3.0-linux-armv7l.tar.gz" ] && NODEHASH="de3554545e2d04719ebcd990984ff1eb5d6edbbbb9d24893cb998e2eb15d8bf5"
[ "$NODEURL" = "http://nodejs.org/dist/v6.3.0/node-v6.3.0-linux-ppc64le.tar.gz" ] && NODEHASH="4665fa5c521e7c39bec441c66c41017a0a430944eb49fd0adec6e1c3b72f992b"
[ "$NODEURL" = "http://nodejs.org/dist/v6.3.0/node-v6.3.0-linux-ppc64.tar.gz" ] && NODEHASH="72d6b834e88acc7d805f5bc7b90bce7f418b5b7270efa5126b614ae81a6c8b6c"
[ "$NODEURL" = "http://nodejs.org/dist/v6.3.0/node-v6.3.0-linux-x64.tar.gz" ] && NODEHASH="d26c09fc95ebb457b79fcb0a2890fe8417b2c04f4016dadf2d165c07af762764"
[ "$NODEURL" = "http://nodejs.org/dist/v6.3.0/node-v6.3.0-linux-x86.tar.gz" ] && NODEHASH="9302affb9844ec9a51e774169072bff3421c0232f746b3d504ba6c13c749ff9f"
[ "$NODEURL" = "http://nodejs.org/dist/v6.3.0/node-v6.3.0-sunos-x64.tar.gz" ] && NODEHASH="7d5f8b37ea3809db6bd633cb1702d1f942319de0c606883ffd300b36728283b7"
[ "$NODEURL" = "http://nodejs.org/dist/v6.3.0/node-v6.3.0-sunos-x86.tar.gz" ] && NODEHASH="a494375cce067250bf8729b007a913a9b06b7b3a0ca1629dd263a84da3d172e2"
[ "$NODEURL" = "http://nodejs.org/dist/v6.3.0/node-v6.3.0.tar.gz" ] && NODEHASH="4ed7a99985f8afee337cc22d5fef61b495ab4238dfff3750ac9019e87fc6aae6"
[ "$NODEURL" = "http://nodejs.org/dist/v6.3.1/node-v6.3.1-darwin-x64.tar.gz" ] && NODEHASH="de6d45f63ab281b7454977d8dbf5494015e63a1cd9c9d8fe6f67e2431684f34f"
[ "$NODEURL" = "http://nodejs.org/dist/v6.3.1/node-v6.3.1-headers.tar.gz" ] && NODEHASH="2d7a69f69e9d45453fed9ad68302edf4969e64b870b2e40f9b3d5f98ed44587c"
[ "$NODEURL" = "http://nodejs.org/dist/v6.3.1/node-v6.3.1-linux-arm64.tar.gz" ] && NODEHASH="66ef087709f7709f0bf066904df06815ac7ad213181d6dcc2adb4f9dc831704f"
[ "$NODEURL" = "http://nodejs.org/dist/v6.3.1/node-v6.3.1-linux-armv6l.tar.gz" ] && NODEHASH="fe6722b20b71e117be020792e7e69b124adeeae46ed666a406ac3a132870c418"
[ "$NODEURL" = "http://nodejs.org/dist/v6.3.1/node-v6.3.1-linux-armv7l.tar.gz" ] && NODEHASH="633fc62c5bb2cff7e2746ee5351494b97021468365d5d290f275f56db920337a"
[ "$NODEURL" = "http://nodejs.org/dist/v6.3.1/node-v6.3.1-linux-ppc64le.tar.gz" ] && NODEHASH="b760514341a29a684b86c9d6754a5b8d576a6b8cf1ec00cea012a38ba7ada7dd"
[ "$NODEURL" = "http://nodejs.org/dist/v6.3.1/node-v6.3.1-linux-ppc64.tar.gz" ] && NODEHASH="740824dd86eda983b3c7873e56a952585917f3d8770363f66f8593241009c273"
[ "$NODEURL" = "http://nodejs.org/dist/v6.3.1/node-v6.3.1-linux-x64.tar.gz" ] && NODEHASH="eccc530696d18b07c5785e317b2babbea9c1dd14dbab80be734b820fc241ddea"
[ "$NODEURL" = "http://nodejs.org/dist/v6.3.1/node-v6.3.1-linux-x86.tar.gz" ] && NODEHASH="23f07c6467f69c7e572e71a712bbc9c78be14ccad32f694edd52bc45b272a638"
[ "$NODEURL" = "http://nodejs.org/dist/v6.3.1/node-v6.3.1-sunos-x64.tar.gz" ] && NODEHASH="a1e8f130b2edab12181a254dcd2cf65c72c4d121aacd0e06037b1a23437f3406"
[ "$NODEURL" = "http://nodejs.org/dist/v6.3.1/node-v6.3.1-sunos-x86.tar.gz" ] && NODEHASH="95c0c2d00068976bd96f2fab0dfabd3be0ea7ba7a7b4aaee4d30c7d1849d0e42"
[ "$NODEURL" = "http://nodejs.org/dist/v6.3.1/node-v6.3.1.tar.gz" ] && NODEHASH="3f6144dcf13c210b11c83dc0b0a841219347a769b5a3b883b20f7ab8dc4008f6"
[ "$NODEURL" = "http://nodejs.org/dist/v6.4.0/node-v6.4.0-darwin-x64.tar.gz" ] && NODEHASH="4d04f11e12a360a0d5c6da5cd393fbe75a13f111a492c2a12185992b83542f9c"
[ "$NODEURL" = "http://nodejs.org/dist/v6.4.0/node-v6.4.0-headers.tar.gz" ] && NODEHASH="6e094188a1c6dda967d160a92ec55ff9b46120a2207291eb067404dbf9cbd6c6"
[ "$NODEURL" = "http://nodejs.org/dist/v6.4.0/node-v6.4.0-linux-arm64.tar.gz" ] && NODEHASH="42493fcd0266949b219bb4a0f36e1e765f266aed104c594cff012b906dcc53c1"
[ "$NODEURL" = "http://nodejs.org/dist/v6.4.0/node-v6.4.0-linux-armv6l.tar.gz" ] && NODEHASH="94e97543f4f32f6b8368d37c6ea8eebcbf43dd2fa170e02166d9887646d42ec4"
[ "$NODEURL" = "http://nodejs.org/dist/v6.4.0/node-v6.4.0-linux-armv7l.tar.gz" ] && NODEHASH="51fddefc79a2e8303b975345eea10128a2180a3f40c06238c77daa3c0009b8c4"
[ "$NODEURL" = "http://nodejs.org/dist/v6.4.0/node-v6.4.0-linux-ppc64le.tar.gz" ] && NODEHASH="9e28554ec3f4f18d675d952f00f1f06535254c9513b5cde3d6f9241be4ef063a"
[ "$NODEURL" = "http://nodejs.org/dist/v6.4.0/node-v6.4.0-linux-ppc64.tar.gz" ] && NODEHASH="43b3091418816b3576b48d6a462bc96eec1d2cb107164631d8b5887a597b1992"
[ "$NODEURL" = "http://nodejs.org/dist/v6.4.0/node-v6.4.0-linux-x64.tar.gz" ] && NODEHASH="990636e44b9f7a270cf82f988e5faecb5850fcda9580da65e5721b90ed3dddb2"
[ "$NODEURL" = "http://nodejs.org/dist/v6.4.0/node-v6.4.0-linux-x86.tar.gz" ] && NODEHASH="3630f172fee9c30ef95df3972ae9b2b234273ad504f245e0db363797fa2c08f4"
[ "$NODEURL" = "http://nodejs.org/dist/v6.4.0/node-v6.4.0-sunos-x64.tar.gz" ] && NODEHASH="16f316b4195f480a0b318c27c30d84f60f9a0cf567b4f083aad0db733a64aaec"
[ "$NODEURL" = "http://nodejs.org/dist/v6.4.0/node-v6.4.0-sunos-x86.tar.gz" ] && NODEHASH="96b24de07ad692e74a8543df3e965bb9ee337130a5c2cf2531c6290e934cda54"
[ "$NODEURL" = "http://nodejs.org/dist/v6.4.0/node-v6.4.0.tar.gz" ] && NODEHASH="a58c930618cf5b0862aa329a626e850519036acfcd746983010c9946df1797d0"
[ "$NODEURL" = "http://nodejs.org/dist/v6.5.0/node-v6.5.0-darwin-x64.tar.gz" ] && NODEHASH="352cee438dc55942aebac054e8a2e14b6784056c66465105e2b3bfc55832927f"
[ "$NODEURL" = "http://nodejs.org/dist/v6.5.0/node-v6.5.0-headers.tar.gz" ] && NODEHASH="e69e5cfbd610a624161ed2f7c11ea61c4ea7187dc42bca1dedc434da1901f4c8"
[ "$NODEURL" = "http://nodejs.org/dist/v6.5.0/node-v6.5.0-linux-arm64.tar.gz" ] && NODEHASH="e3d208d3b054301e2bd572d71c7c325ddc0a7e4c2cc4278053e2375e841f6d99"
[ "$NODEURL" = "http://nodejs.org/dist/v6.5.0/node-v6.5.0-linux-armv6l.tar.gz" ] && NODEHASH="fff3d363117b8c9beff3c4a89fd48d51dfd55b9dc2e9c2957be79a3c20bd570f"
[ "$NODEURL" = "http://nodejs.org/dist/v6.5.0/node-v6.5.0-linux-armv7l.tar.gz" ] && NODEHASH="ea03725d8cee9528d75fdb2060f62ce96a719bc0f5f35f5ad2dd982bb09c1429"
[ "$NODEURL" = "http://nodejs.org/dist/v6.5.0/node-v6.5.0-linux-ppc64le.tar.gz" ] && NODEHASH="36c7e7449db335a591fc55b77d7eab13576a296ec3be5ea557c8e7a54c061928"
[ "$NODEURL" = "http://nodejs.org/dist/v6.5.0/node-v6.5.0-linux-ppc64.tar.gz" ] && NODEHASH="0330b57d8161d06d68b5d86b8b8a924f48c7266a70816c2c1d306196246cea21"
[ "$NODEURL" = "http://nodejs.org/dist/v6.5.0/node-v6.5.0-linux-x64.tar.gz" ] && NODEHASH="575638830e4ba11c5afba5c222934bc5e338e74df2f27ca09bad09014b4aa415"
[ "$NODEURL" = "http://nodejs.org/dist/v6.5.0/node-v6.5.0-linux-x86.tar.gz" ] && NODEHASH="12d5b79b8b914c2439b6aa542b47f28c0d6dc9c5438eeda629f26021eb839dac"
[ "$NODEURL" = "http://nodejs.org/dist/v6.5.0/node-v6.5.0-sunos-x64.tar.gz" ] && NODEHASH="71ae256a6bd8bb8c7e1af88ec4702022eecf911cea786e6dac21b25bc5f1ece7"
[ "$NODEURL" = "http://nodejs.org/dist/v6.5.0/node-v6.5.0-sunos-x86.tar.gz" ] && NODEHASH="e1ce32adfd7e205ed949f0a372bd75d1bbb404274de667e30ef2a99386f30a71"
[ "$NODEURL" = "http://nodejs.org/dist/v6.5.0/node-v6.5.0.tar.gz" ] && NODEHASH="d7742558bb3331e41510d6e6f1f7b13c0527aecc00a63c3e05fcfd44427ff778"
[ "$NODEURL" = "http://nodejs.org/dist/v6.6.0/node-v6.6.0-darwin-x64.tar.gz" ] && NODEHASH="c8d1fe38eb794ca46aacf6c8e90676eec7a8aeec83b4b09f57ce503509e7a19f"
[ "$NODEURL" = "http://nodejs.org/dist/v6.6.0/node-v6.6.0-headers.tar.gz" ] && NODEHASH="60b81c7276105a51e71ad8bc7f59163105e7c5dd1d992b173b5b66449b6df3fc"
[ "$NODEURL" = "http://nodejs.org/dist/v6.6.0/node-v6.6.0-linux-arm64.tar.gz" ] && NODEHASH="9abae64e411d8ea1541a4776e78d9cf53ad8e20e8b34cf77d9b3579e8edb6f65"
[ "$NODEURL" = "http://nodejs.org/dist/v6.6.0/node-v6.6.0-linux-armv6l.tar.gz" ] && NODEHASH="d311754cddc9b387a2798226ecb5487e515c555e050fdd08ef3d6665c3c0d336"
[ "$NODEURL" = "http://nodejs.org/dist/v6.6.0/node-v6.6.0-linux-armv7l.tar.gz" ] && NODEHASH="e4dc3295f6602b0f4cd3433a6e520294743e2c342692b4fad388d33910cdd465"
[ "$NODEURL" = "http://nodejs.org/dist/v6.6.0/node-v6.6.0-linux-ppc64le.tar.gz" ] && NODEHASH="b38ff6058f0213567d31a5d194d669ce75894336f6d0324426f01722c989d3c4"
[ "$NODEURL" = "http://nodejs.org/dist/v6.6.0/node-v6.6.0-linux-ppc64.tar.gz" ] && NODEHASH="90d483c63fdbc6594185b3e143bf8d5627812288a029f02f578363d6dd505285"
[ "$NODEURL" = "http://nodejs.org/dist/v6.6.0/node-v6.6.0-linux-s390x.tar.gz" ] && NODEHASH="a2f109eb0fec81d90206e9c0bca05327c706b244b5d7c5bd247ac140f811e54e"
[ "$NODEURL" = "http://nodejs.org/dist/v6.6.0/node-v6.6.0-linux-x64.tar.gz" ] && NODEHASH="c22ab0dfa9d0b8d9de02ef7c0d860298a5d1bf6cae7413fb18b99e8a3d25648a"
[ "$NODEURL" = "http://nodejs.org/dist/v6.6.0/node-v6.6.0-linux-x86.tar.gz" ] && NODEHASH="05f3bfdfe8e1911e66b4bf645a439480212767e71664b8c97f0cba46671e8160"
[ "$NODEURL" = "http://nodejs.org/dist/v6.6.0/node-v6.6.0-sunos-x64.tar.gz" ] && NODEHASH="21b05b3664c338b76ec8c139a7ae80fb9b1c65215ee2da6d899dcbd6c6d67554"
[ "$NODEURL" = "http://nodejs.org/dist/v6.6.0/node-v6.6.0-sunos-x86.tar.gz" ] && NODEHASH="ffa81fc834a7f24958b3c1bf07dc668f6d0e0cda48582a23bc749abca42cae8c"
[ "$NODEURL" = "http://nodejs.org/dist/v6.6.0/node-v6.6.0.tar.gz" ] && NODEHASH="11b957de855a392ceaa8b300ec66236d6f9c6baa184837d00bdaba2da4aefe91"
[ "$NODEURL" = "http://nodejs.org/dist/v6.7.0/node-v6.7.0-aix-ppc64.tar.gz" ] && NODEHASH="132855c123efdfa43dc4a256f15728f1bbd3f9996a4b9e29cd31908248be6bf2"
[ "$NODEURL" = "http://nodejs.org/dist/v6.7.0/node-v6.7.0-darwin-x64.tar.gz" ] && NODEHASH="69fab7a1ebeee54d5e3160eb9366e88a61500731fad86dee98c79c4a14b56bc6"
[ "$NODEURL" = "http://nodejs.org/dist/v6.7.0/node-v6.7.0-headers.tar.gz" ] && NODEHASH="42f0fdcd937409842d76a5852782acfdb0b6fa8e3520df6694f2abaa7c9e942c"
[ "$NODEURL" = "http://nodejs.org/dist/v6.7.0/node-v6.7.0-linux-arm64.tar.gz" ] && NODEHASH="45ffd727bcab41a544ad7862fe985f6beac4fcd96c63e116ca467d1147ba6454"
[ "$NODEURL" = "http://nodejs.org/dist/v6.7.0/node-v6.7.0-linux-armv6l.tar.gz" ] && NODEHASH="0f8e0dbaa6bcccd22db75077fed1afb28c2b225b6cdc185913178ae395a68ef9"
[ "$NODEURL" = "http://nodejs.org/dist/v6.7.0/node-v6.7.0-linux-armv7l.tar.gz" ] && NODEHASH="1e7e138ba8c54d7a0fbf5e3f188442a14a70409dc154b74b17635bcff74e4a81"
[ "$NODEURL" = "http://nodejs.org/dist/v6.7.0/node-v6.7.0-linux-ppc64le.tar.gz" ] && NODEHASH="de8e4ca71caa8be6eaf80e65b89de2a6d152fa4ce08efcbc90ce7e1bfdf130e7"
[ "$NODEURL" = "http://nodejs.org/dist/v6.7.0/node-v6.7.0-linux-ppc64.tar.gz" ] && NODEHASH="e8ce540b592d337304a10f4eb19bb4efee889c6676c5f188d072bfb2a8089927"
[ "$NODEURL" = "http://nodejs.org/dist/v6.7.0/node-v6.7.0-linux-s390x.tar.gz" ] && NODEHASH="e0f2616b4beb4c2505edb19e3cbedbf3d1c958441517cc9a1e918f6feaa4b95b"
[ "$NODEURL" = "http://nodejs.org/dist/v6.7.0/node-v6.7.0-linux-x64.tar.gz" ] && NODEHASH="abe81b4150917cdbbeebc6c6b85003b80c972d32c8f5dfd2970d32e52a6877af"
[ "$NODEURL" = "http://nodejs.org/dist/v6.7.0/node-v6.7.0-linux-x86.tar.gz" ] && NODEHASH="fa94c93a4a3600d68e003a399f5cf5e109c2d10505b4d9456373c25953eb9bf5"
[ "$NODEURL" = "http://nodejs.org/dist/v6.7.0/node-v6.7.0-sunos-x64.tar.gz" ] && NODEHASH="33f240dac58a1293a08b66ecd01a70849c693e3a9e58a94cb7ee92126a894984"
[ "$NODEURL" = "http://nodejs.org/dist/v6.7.0/node-v6.7.0-sunos-x86.tar.gz" ] && NODEHASH="80dacf34a5e17f3eeabe15e212005daeaa189caa424a83a23f302a9fa5996565"
[ "$NODEURL" = "http://nodejs.org/dist/v6.7.0/node-v6.7.0.tar.gz" ] && NODEHASH="02b8ee1719a11b9ab22bef9279519efaaf31dd0d39cba4c3a1176ccda400b8d6"
[ "$NODEURL" = "http://nodejs.org/dist/v6.8.0/node-v6.8.0-aix-ppc64.tar.gz" ] && NODEHASH="6ec44c9dd6e9e7ed5145089aeb4437893873dd4de7cd8dfd7afb875f9b8eeb77"
[ "$NODEURL" = "http://nodejs.org/dist/v6.8.0/node-v6.8.0-darwin-x64.tar.gz" ] && NODEHASH="4ff13860f6bda86ce1053c673548c7f5d301a42ceb2f103c781024c3b88d2083"
[ "$NODEURL" = "http://nodejs.org/dist/v6.8.0/node-v6.8.0-headers.tar.gz" ] && NODEHASH="1cac1f6816e26982cb652314f84fec76c0494c1fbd080fe86a400f841ef215f1"
[ "$NODEURL" = "http://nodejs.org/dist/v6.8.0/node-v6.8.0-linux-arm64.tar.gz" ] && NODEHASH="c7524e35bfc1fd4961dfac6bd6d602f474e909d63e3e48f892906a183bd9aa04"
[ "$NODEURL" = "http://nodejs.org/dist/v6.8.0/node-v6.8.0-linux-armv6l.tar.gz" ] && NODEHASH="979e70d5d727a9412b826a28ef4d7e8a1d23f72679ea6bf94ee116cb2e3c9a5f"
[ "$NODEURL" = "http://nodejs.org/dist/v6.8.0/node-v6.8.0-linux-armv7l.tar.gz" ] && NODEHASH="2ae978777979c4af49945ce46dd014eba18d50a94520a8555e60d10113308b2f"
[ "$NODEURL" = "http://nodejs.org/dist/v6.8.0/node-v6.8.0-linux-ppc64le.tar.gz" ] && NODEHASH="d955c56f404287054bb7433b934636305d8f9788294da1b34e7942288497485c"
[ "$NODEURL" = "http://nodejs.org/dist/v6.8.0/node-v6.8.0-linux-ppc64.tar.gz" ] && NODEHASH="c8bfd55be7f47efa7c6d2d051eef0b99c70d222a2c70c3a013cffb4b171e1808"
[ "$NODEURL" = "http://nodejs.org/dist/v6.8.0/node-v6.8.0-linux-s390x.tar.gz" ] && NODEHASH="76225db134036a6f2426e5c8e854ee62e8e2d4b522d916874e995f88a22ddb14"
[ "$NODEURL" = "http://nodejs.org/dist/v6.8.0/node-v6.8.0-linux-x64.tar.gz" ] && NODEHASH="9ddd118262cbe27fd668bc17c9c786cdd27c6a291cc712b4937013e4665f6e6d"
[ "$NODEURL" = "http://nodejs.org/dist/v6.8.0/node-v6.8.0-linux-x86.tar.gz" ] && NODEHASH="d79eccb95911d3d3cf742d059e2d2f7669db33ef8f7f1f38f023c0e553f26d30"
[ "$NODEURL" = "http://nodejs.org/dist/v6.8.0/node-v6.8.0-sunos-x64.tar.gz" ] && NODEHASH="9346655f3724e81d5fb49140d36097957f5ede725cc5070a22b3be4ce423bf48"
[ "$NODEURL" = "http://nodejs.org/dist/v6.8.0/node-v6.8.0-sunos-x86.tar.gz" ] && NODEHASH="7ae4a80269e254791a57b1759d4e5eeb2127a44a431444a36c02d3ff8727ac88"
[ "$NODEURL" = "http://nodejs.org/dist/v6.8.0/node-v6.8.0.tar.gz" ] && NODEHASH="a8a2d4b74734991dd45e5b2b0b13617864f1283f6d6bc262e1e4d30741114352"
[ "$NODEURL" = "http://nodejs.org/dist/v6.8.1/node-v6.8.1-aix-ppc64.tar.gz" ] && NODEHASH="fc42b85875c2ed593a22c4543f82f5ae4d737969c1915ac3d5f7a1fcf47a05a1"
[ "$NODEURL" = "http://nodejs.org/dist/v6.8.1/node-v6.8.1-darwin-x64.tar.gz" ] && NODEHASH="36058e54c0e502c3e98f37f85533cf6148ea0232cf06cbebaee70a53719e60ca"
[ "$NODEURL" = "http://nodejs.org/dist/v6.8.1/node-v6.8.1-headers.tar.gz" ] && NODEHASH="63091ba93129536252820d0a6dc1e291d6bf602c1051a06979d45ae629265715"
[ "$NODEURL" = "http://nodejs.org/dist/v6.8.1/node-v6.8.1-linux-arm64.tar.gz" ] && NODEHASH="d68acdedba79bb0cca9ccdb985323d18b6edbd85ee8cf896e02433c09cad1da9"
[ "$NODEURL" = "http://nodejs.org/dist/v6.8.1/node-v6.8.1-linux-armv6l.tar.gz" ] && NODEHASH="439c1018d65e224ae9b7a25bf546952c89c0d6faeddd2d932697a1bda6f6935d"
[ "$NODEURL" = "http://nodejs.org/dist/v6.8.1/node-v6.8.1-linux-armv7l.tar.gz" ] && NODEHASH="7afaab589165ba2e59476148010dd23d68cf67cd9f98d2522ce6e5c3430a5cc1"
[ "$NODEURL" = "http://nodejs.org/dist/v6.8.1/node-v6.8.1-linux-ppc64le.tar.gz" ] && NODEHASH="7480f0b371187c3642c39ce058603b662e271823ff8fcb959fd349b580cb694b"
[ "$NODEURL" = "http://nodejs.org/dist/v6.8.1/node-v6.8.1-linux-ppc64.tar.gz" ] && NODEHASH="f478521914ea5d46e0280ff5626ed2c9645acae529c4aa8a557de54353936255"
[ "$NODEURL" = "http://nodejs.org/dist/v6.8.1/node-v6.8.1-linux-s390x.tar.gz" ] && NODEHASH="d6d9cd0a3164f687eaf148f37bd0a443fb5492964e5ef48bc61abafaa3fa8ac5"
[ "$NODEURL" = "http://nodejs.org/dist/v6.8.1/node-v6.8.1-linux-x64.tar.gz" ] && NODEHASH="8d004e6990926508460495450a4083d40836e81710afca303d6a298e032c6b18"
[ "$NODEURL" = "http://nodejs.org/dist/v6.8.1/node-v6.8.1-linux-x86.tar.gz" ] && NODEHASH="1247639f19a3e035bc1ec13bc546b0aea651b94836d0d814a5fe52f379c471e5"
[ "$NODEURL" = "http://nodejs.org/dist/v6.8.1/node-v6.8.1-sunos-x64.tar.gz" ] && NODEHASH="49122d54d53ec38062fb6f66a724e28287b4bf6b67361e4020d936b28f207a59"
[ "$NODEURL" = "http://nodejs.org/dist/v6.8.1/node-v6.8.1-sunos-x86.tar.gz" ] && NODEHASH="4b18bbaeb1f6cfc89545a5791681c40ca25f506d02170cc2d4e0d1fbbfe1da1c"
[ "$NODEURL" = "http://nodejs.org/dist/v6.8.1/node-v6.8.1.tar.gz" ] && NODEHASH="736f7651f59a78d5a0d88ab2430c93fc29a8efd1c06fd3fb4eb915b262bc2769"
[ "$NODEURL" = "http://nodejs.org/dist/v6.9.0/node-v6.9.0-aix-ppc64.tar.gz" ] && NODEHASH="a5738a8061e5964d9851047355bab10c02046997adad0592d3e0c834c19c9599"
[ "$NODEURL" = "http://nodejs.org/dist/v6.9.0/node-v6.9.0-darwin-x64.tar.gz" ] && NODEHASH="fd0f58487cd72d78e857bfc24061cfe77353e1571a17182b4e38273782648edf"
[ "$NODEURL" = "http://nodejs.org/dist/v6.9.0/node-v6.9.0-headers.tar.gz" ] && NODEHASH="c2848e6ef6de3d771d3523ca8da2de6c949729476b3723cabf82836af4b723b8"
[ "$NODEURL" = "http://nodejs.org/dist/v6.9.0/node-v6.9.0-linux-arm64.tar.gz" ] && NODEHASH="e9ff08e622436007594dcbff1b528023aaa2397c38fdc961d130730b90fdc814"
[ "$NODEURL" = "http://nodejs.org/dist/v6.9.0/node-v6.9.0-linux-armv6l.tar.gz" ] && NODEHASH="ce870c8839ce5d46a36d11ae882e59cff6e361c46714428da3464504a3f0769a"
[ "$NODEURL" = "http://nodejs.org/dist/v6.9.0/node-v6.9.0-linux-armv7l.tar.gz" ] && NODEHASH="97aaf1dba42c7544f8a2ad1e34e0f032d7645227e60928b9b522e257bd219256"
[ "$NODEURL" = "http://nodejs.org/dist/v6.9.0/node-v6.9.0-linux-ppc64le.tar.gz" ] && NODEHASH="325b89f57374627d19de246dd1148699ea88b02139cecbdbe218b519fc8a328c"
[ "$NODEURL" = "http://nodejs.org/dist/v6.9.0/node-v6.9.0-linux-ppc64.tar.gz" ] && NODEHASH="bf21e314c2bcf0ea379d2a6b2c017b0446ad8fb822dbc3c2fc04d594b727b23a"
[ "$NODEURL" = "http://nodejs.org/dist/v6.9.0/node-v6.9.0-linux-s390x.tar.gz" ] && NODEHASH="4aa6e76334e3bba1919c65a2a0dcabd28c48a282402b2f523807f31a588896e8"
[ "$NODEURL" = "http://nodejs.org/dist/v6.9.0/node-v6.9.0-linux-x64.tar.gz" ] && NODEHASH="a9aafa2499097b315e1554b882923a6e2f9c446d24eaea53630f0fdbe075b226"
[ "$NODEURL" = "http://nodejs.org/dist/v6.9.0/node-v6.9.0-linux-x86.tar.gz" ] && NODEHASH="5520dc47e11b377064ce99a60e640dfdf9abc29916bcd110307474463af9efb1"
[ "$NODEURL" = "http://nodejs.org/dist/v6.9.0/node-v6.9.0-sunos-x64.tar.gz" ] && NODEHASH="e623e9bd2dd3554fdb9ec53ca3fe6e0d1a3715c05766bfc9a002d3289adbbe39"
[ "$NODEURL" = "http://nodejs.org/dist/v6.9.0/node-v6.9.0-sunos-x86.tar.gz" ] && NODEHASH="2e48eb1506f54549df5e7c5727cd5c171aaf838f22caa848aeb2a3011db7f171"
[ "$NODEURL" = "http://nodejs.org/dist/v6.9.0/node-v6.9.0.tar.gz" ] && NODEHASH="2e2657d2ece89782ca9e2cc0300f9119998e73382caa7ad2995ab81cc26ad923"
[ "$NODEURL" = "http://nodejs.org/dist/v6.9.1/node-v6.9.1-aix-ppc64.tar.gz" ] && NODEHASH="77c14510c4b09188450f8c10b8c163ef4b4f616c8b7f455d538f07a810bc98bf"
[ "$NODEURL" = "http://nodejs.org/dist/v6.9.1/node-v6.9.1-darwin-x64.tar.gz" ] && NODEHASH="392e511ca0d6203c80700ed753187535e04069d0df0074cbfd1e4f1bd571d4c5"
[ "$NODEURL" = "http://nodejs.org/dist/v6.9.1/node-v6.9.1-headers.tar.gz" ] && NODEHASH="bd73a816a1bfde82c5e6d0a4b5e669f7ce08e0149ba7994388a4f0b99216fb41"
[ "$NODEURL" = "http://nodejs.org/dist/v6.9.1/node-v6.9.1-linux-arm64.tar.gz" ] && NODEHASH="8a8da2c3aad9da2d80035eeba0b9aae41230bec394729224fafcfae152fa5f66"
[ "$NODEURL" = "http://nodejs.org/dist/v6.9.1/node-v6.9.1-linux-armv6l.tar.gz" ] && NODEHASH="0b30184fe98bd22b859db7f4cbaa56ecc04f7f526313c8da42315d89fabe23b2"
[ "$NODEURL" = "http://nodejs.org/dist/v6.9.1/node-v6.9.1-linux-armv7l.tar.gz" ] && NODEHASH="c4651804af2422b10cf49e1eeab96feb2664eaeb6c6ae3913933a946134e3375"
[ "$NODEURL" = "http://nodejs.org/dist/v6.9.1/node-v6.9.1-linux-ppc64le.tar.gz" ] && NODEHASH="5714678db7e6ff93ae96417c0b210a1f084ee7a0aabf0f2d33307a1f1d0445ce"
[ "$NODEURL" = "http://nodejs.org/dist/v6.9.1/node-v6.9.1-linux-ppc64.tar.gz" ] && NODEHASH="2accb9e0f8082ea29f224cf4046d4ce318bb68b6cc41415e1291b58929de49a7"
[ "$NODEURL" = "http://nodejs.org/dist/v6.9.1/node-v6.9.1-linux-s390x.tar.gz" ] && NODEHASH="aa4ba50e0af65590903e4627703d90119711aea84c17f3f1b4a9211ed9cc23a8"
[ "$NODEURL" = "http://nodejs.org/dist/v6.9.1/node-v6.9.1-linux-x64.tar.gz" ] && NODEHASH="a9d9e6308931fa2a2b0cada070516d45b76d752430c31c9198933c78f8d54b17"
[ "$NODEURL" = "http://nodejs.org/dist/v6.9.1/node-v6.9.1-linux-x86.tar.gz" ] && NODEHASH="d2f4cb3f7e784a8c1c1b6477db613d67635d69cbf9ca80c17b87c9e85378d3bd"
[ "$NODEURL" = "http://nodejs.org/dist/v6.9.1/node-v6.9.1-sunos-x64.tar.gz" ] && NODEHASH="955d396a888c164d12d134baae098e2e214bfb46898e520f5be2e88a404697e2"
[ "$NODEURL" = "http://nodejs.org/dist/v6.9.1/node-v6.9.1-sunos-x86.tar.gz" ] && NODEHASH="ec758b4a638f4599bb4a782196088f7704f4b08e70509ac235c3dd3f5b62382b"
[ "$NODEURL" = "http://nodejs.org/dist/v6.9.1/node-v6.9.1.tar.gz" ] && NODEHASH="a98997ca3a4d10751f0ebe97839b2308a31ae884b4203cda0c99cf36bc7fe3bf"
[ "$NODEURL" = "http://nodejs.org/dist/v7.0.0/node-v7.0.0-aix-ppc64.tar.gz" ] && NODEHASH="0977be0f261dd0e61433e875fbf0014773d1560b8bb5a882637e6e0289906d84"
[ "$NODEURL" = "http://nodejs.org/dist/v7.0.0/node-v7.0.0-darwin-x64.tar.gz" ] && NODEHASH="2e94b2d3bae3ccac9c376379bc4d2243935c333bc1235b3d948d20d40ef26069"
[ "$NODEURL" = "http://nodejs.org/dist/v7.0.0/node-v7.0.0-headers.tar.gz" ] && NODEHASH="f4792c0f40bd2eee2334424c9d65e4b4b30830507cca1b0cb88507d76044c0f7"
[ "$NODEURL" = "http://nodejs.org/dist/v7.0.0/node-v7.0.0-linux-arm64.tar.gz" ] && NODEHASH="375e897d16956b80e6c7b298f17d5b6b4a3bac1a50b297575bd50f34ce4c612c"
[ "$NODEURL" = "http://nodejs.org/dist/v7.0.0/node-v7.0.0-linux-armv6l.tar.gz" ] && NODEHASH="7378832687d7475074281f06dc98b2ddfc6c07cdc40e7ce4ea08a95347fd2292"
[ "$NODEURL" = "http://nodejs.org/dist/v7.0.0/node-v7.0.0-linux-armv7l.tar.gz" ] && NODEHASH="dbcf05b10f71acf845d4c1357c627ebf2d401a11c80464cb44e475c5f0eca7e4"
[ "$NODEURL" = "http://nodejs.org/dist/v7.0.0/node-v7.0.0-linux-ppc64le.tar.gz" ] && NODEHASH="5d615eefcbf76c25f760de7e94f4d4c3c1a5376978e595800313f524d9634fe5"
[ "$NODEURL" = "http://nodejs.org/dist/v7.0.0/node-v7.0.0-linux-ppc64.tar.gz" ] && NODEHASH="d5297fa59942a0807f785d279f311ebe8346cdedcf49b6aba6e1bf75d84d664c"
[ "$NODEURL" = "http://nodejs.org/dist/v7.0.0/node-v7.0.0-linux-s390x.tar.gz" ] && NODEHASH="e92985271cf5efe6f0ebbf4768bcf7e7a98147da154dd5ca417d180e3c921262"
[ "$NODEURL" = "http://nodejs.org/dist/v7.0.0/node-v7.0.0-linux-x64.tar.gz" ] && NODEHASH="4bc5ce31d2485f0b25e55bf8691d5dabf72c61f0c06b363728b70bfc0292ce7f"
[ "$NODEURL" = "http://nodejs.org/dist/v7.0.0/node-v7.0.0-linux-x86.tar.gz" ] && NODEHASH="96c61f58506725246ba246606c288809a5b7243f20ddfa464136978dd32d6917"
[ "$NODEURL" = "http://nodejs.org/dist/v7.0.0/node-v7.0.0-sunos-x86.tar.gz" ] && NODEHASH="4b6dcc2345b7b5229676c342d7757578973139166129cdb4c7dfddcaab3ef598"
[ "$NODEURL" = "http://nodejs.org/dist/v7.0.0/node-v7.0.0.tar.gz" ] && NODEHASH="95ae4a37b8cf57816f96588ebe9ce5dbbb2c5452d177edd75fcfe6caf3b12b9b"
[ "$NODEURL" = "http://nodejs.org/dist/v7.1.0/node-v7.1.0-aix-ppc64.tar.gz" ] && NODEHASH="5ef780ab1d04674c7ec528eeb112d37a42a58f00c9f15954655929c5adaf1bbe"
[ "$NODEURL" = "http://nodejs.org/dist/v7.1.0/node-v7.1.0-darwin-x64.tar.gz" ] && NODEHASH="b4615c79eac53f1a660b94e047d1dc64d797bf365f305ac9e0c2ac7304340171"
[ "$NODEURL" = "http://nodejs.org/dist/v7.1.0/node-v7.1.0-headers.tar.gz" ] && NODEHASH="b254b8ff711e94ff59895f820ff61d9054a60c2055b5491c042d456af1f70a83"
[ "$NODEURL" = "http://nodejs.org/dist/v7.1.0/node-v7.1.0-linux-arm64.tar.gz" ] && NODEHASH="fcaa2281b2ea4e0a6b598fae4fbee3708aded5b1b0b2397f8fab18c13e3ea3be"
[ "$NODEURL" = "http://nodejs.org/dist/v7.1.0/node-v7.1.0-linux-armv6l.tar.gz" ] && NODEHASH="2e87c4249ebd29e6022aa289ec8c7632e92ba7a5c61a5e035c0acefa8a143772"
[ "$NODEURL" = "http://nodejs.org/dist/v7.1.0/node-v7.1.0-linux-armv7l.tar.gz" ] && NODEHASH="8899d4340c23282ec4dc28b6a04bb20279da46bf92db58abce50a616877cd811"
[ "$NODEURL" = "http://nodejs.org/dist/v7.1.0/node-v7.1.0-linux-ppc64le.tar.gz" ] && NODEHASH="6cc4ccb25c21bd2331b30aa4193c21c4d3ac2d9d1fd7300157285383c288ec29"
[ "$NODEURL" = "http://nodejs.org/dist/v7.1.0/node-v7.1.0-linux-ppc64.tar.gz" ] && NODEHASH="662e3806f9ce46d0f92a347db55b50c3161406439317976c5fdb51e1dd189861"
[ "$NODEURL" = "http://nodejs.org/dist/v7.1.0/node-v7.1.0-linux-s390x.tar.gz" ] && NODEHASH="c642a97c368644e8808cb006c2ad5822c917fa36bce3e8f41a01a4a2bb84740c"
[ "$NODEURL" = "http://nodejs.org/dist/v7.1.0/node-v7.1.0-linux-x64.tar.gz" ] && NODEHASH="0d2f13477ba991950bd9938e38c8d943b9bf2e899adcd4a28e98532f029e9910"
[ "$NODEURL" = "http://nodejs.org/dist/v7.1.0/node-v7.1.0-linux-x86.tar.gz" ] && NODEHASH="f6bcb57e7e693b25ad448faf18300bb9f823bc114aa0b050b7953978511e9a1d"
[ "$NODEURL" = "http://nodejs.org/dist/v7.1.0/node-v7.1.0-sunos-x86.tar.gz" ] && NODEHASH="3dff7316291080f351349977d431f4ee0581c6d8961ec8b4ea747c8a79edd4ab"
[ "$NODEURL" = "http://nodejs.org/dist/v7.1.0/node-v7.1.0.tar.gz" ] && NODEHASH="595e7e2a37d1e0573044a90077bb12c0f750e5d8851899ffa74038238da9a983"
}
main;
