"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    Object.defineProperty(o, k2, { enumerable: true, get: function() { return m[k]; } });
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const mqtt = __importStar(require("mqtt"));
const url = __importStar(require("url"));
const aedes_1 = require("aedes");
const net = __importStar(require("net"));
const tls = __importStar(require("tls"));
const TD = __importStar(require("@node-wot/td-tools"));
const core_1 = require("@node-wot/core");
class MqttBrokerServer {
    constructor(config) {
        this.scheme = "mqtt";
        this.port = -1;
        this.address = undefined;
        this.brokerURI = undefined;
        this.things = new Map();
        this.config = config !== null && config !== void 0 ? config : { uri: "mqtt://localhost:1883" };
        if (config.uri !== undefined) {
            if (config.uri.indexOf("://") === -1) {
                config.uri = this.scheme + "://" + config.uri;
            }
            this.brokerURI = config.uri;
        }
        if (config.selfHost) {
            this.hostedServer = (0, aedes_1.Server)({});
            let server;
            if (config.key) {
                server = tls.createServer({ key: config.key, cert: config.cert }, this.hostedServer.handle);
            }
            else {
                server = net.createServer(this.hostedServer.handle);
            }
            const parsed = new url.URL(this.brokerURI);
            const port = parseInt(parsed.port);
            this.port = port > 0 ? port : 1883;
            this.hostedBroker = server.listen(port);
            this.hostedServer.authenticate = this.selfHostAuthentication.bind(this);
        }
    }
    expose(thing) {
        return __awaiter(this, void 0, void 0, function* () {
            if (this.broker === undefined) {
                return;
            }
            let name = thing.title;
            if (this.things.has(name)) {
                const suffix = name.match(/.+_([0-9]+)$/);
                if (suffix !== null) {
                    name = name.slice(0, -suffix[1].length) + (1 + parseInt(suffix[1]));
                }
                else {
                    name = name + "_2";
                }
            }
            console.debug("[binding-mqtt]", `MqttBrokerServer at ${this.brokerURI} exposes '${thing.title}' as unique '${name}/*'`);
            this.things.set(name, thing);
            for (const propertyName in thing.properties) {
                this.exposeProperty(name, propertyName, thing);
            }
            for (const actionName in thing.actions) {
                this.exposeAction(name, actionName, thing);
            }
            for (const eventName in thing.events) {
                this.exposeEvent(name, eventName, thing);
            }
            this.broker.on("message", this.handleMessage);
            this.broker.publish(name, JSON.stringify(thing.getThingDescription()), { retain: true });
        });
    }
    exposeProperty(name, propertyName, thing) {
        const topic = encodeURIComponent(name) + "/properties/" + encodeURIComponent(propertyName);
        const property = thing.properties[propertyName];
        if (!property.writeOnly) {
            const href = this.brokerURI + "/" + topic;
            const form = new TD.Form(href, core_1.ContentSerdes.DEFAULT);
            form.op = ["readproperty", "observeproperty", "unobserveproperty"];
            property.forms.push(form);
            console.debug("[binding-mqtt]", `MqttBrokerServer at ${this.brokerURI} assigns '${href}' to property '${propertyName}'`);
            const observeListener = (data) => __awaiter(this, void 0, void 0, function* () {
                let content;
                try {
                    content = core_1.ContentSerdes.get().valueToContent(data, property.data);
                }
                catch (err) {
                    console.warn("[binding-mqtt]", `MqttServer cannot process data for Property '${propertyName}': ${err.message}`);
                    thing.handleUnobserveProperty(propertyName, observeListener, {
                        formIndex: property.forms.length - 1,
                    });
                    return;
                }
                console.debug("[binding-mqtt]", `MqttBrokerServer at ${this.brokerURI} publishing to Property topic '${propertyName}' `);
                const buffer = yield core_1.ProtocolHelpers.readStreamFully(content.body);
                this.broker.publish(topic, buffer);
            });
            thing.handleObserveProperty(propertyName, observeListener, { formIndex: property.forms.length - 1 });
        }
        if (!property.readOnly) {
            const href = this.brokerURI + "/" + topic + "/writeproperty";
            this.broker.subscribe(topic + "/writeproperty");
            const form = new TD.Form(href, core_1.ContentSerdes.DEFAULT);
            form.op = ["writeproperty"];
            thing.properties[propertyName].forms.push(form);
            console.debug("[binding-mqtt]", `MqttBrokerServer at ${this.brokerURI} assigns '${href}' to property '${propertyName}'`);
        }
    }
    exposeAction(name, actionName, thing) {
        const topic = encodeURIComponent(name) + "/actions/" + encodeURIComponent(actionName);
        this.broker.subscribe(topic);
        const href = this.brokerURI + "/" + topic;
        const form = new TD.Form(href, core_1.ContentSerdes.DEFAULT);
        form.op = ["invokeaction"];
        thing.actions[actionName].forms.push(form);
        console.debug("[binding-mqtt]", `MqttBrokerServer at ${this.brokerURI} assigns '${href}' to Action '${actionName}'`);
    }
    exposeEvent(name, eventName, thing) {
        const topic = encodeURIComponent(name) + "/events/" + encodeURIComponent(eventName);
        const event = thing.events[eventName];
        const href = this.brokerURI + "/" + topic;
        const form = new TD.Form(href, core_1.ContentSerdes.DEFAULT);
        form.op = ["subscribeevent", "unsubscribeevent"];
        event.forms.push(form);
        console.debug("[binding-mqtt]", `MqttBrokerServer at ${this.brokerURI} assigns '${href}' to Event '${eventName}'`);
        const eventListener = (content) => __awaiter(this, void 0, void 0, function* () {
            if (!content) {
                console.warn("[binding-mqtt]", `MqttBrokerServer on port ${this.getPort()} cannot process data for Event ${eventName}`);
                thing.handleUnsubscribeEvent(eventName, eventListener, { formIndex: event.forms.length - 1 });
                return;
            }
            console.debug("[binding-mqtt]", `MqttBrokerServer at ${this.brokerURI} publishing to Event topic '${eventName}' `);
            const buffer = yield core_1.ProtocolHelpers.readStreamFully(content.body);
            this.broker.publish(topic, buffer);
        });
        thing.handleSubscribeEvent(eventName, eventListener, { formIndex: event.forms.length - 1 });
    }
    handleMessage(receivedTopic, rawPayload, packet) {
        const segments = receivedTopic.split("/");
        let payload;
        if (rawPayload instanceof Buffer) {
            payload = rawPayload;
        }
        else if (typeof rawPayload === "string") {
            payload = Buffer.from(rawPayload);
        }
        if (segments.length === 4) {
            console.debug("[binding-mqtt]", `MqttBrokerServer at ${this.brokerURI} received message for '${receivedTopic}'`);
            const thing = this.things.get(segments[1]);
            if (thing) {
                if (segments[2] === "actions") {
                    const action = thing.actions[segments[3]];
                    if (action) {
                        this.handleAction(action, packet, payload, segments, thing);
                        return;
                    }
                }
            }
        }
        else if (segments.length === 5 && segments[4] === "writeproperty") {
            const thing = this.things.get(segments[1]);
            if (thing) {
                if (segments[2] === "properties") {
                    const property = thing.properties[segments[3]];
                    if (property) {
                        this.handlePropertyWrite(property, packet, payload, segments, thing);
                    }
                }
            }
            return;
        }
        console.warn("[binding-mqtt]", `MqttBrokerServer at ${this.brokerURI} received message for invalid topic '${receivedTopic}'`);
    }
    handleAction(action, packet, payload, segments, thing) {
        let value;
        if ("properties" in packet && "contentType" in packet.properties) {
            try {
                value = core_1.ContentSerdes.get().contentToValue({ type: packet.properties.contentType, body: payload }, action.input);
            }
            catch (err) {
                console.warn(`MqttBrokerServer at ${this.brokerURI} cannot process received message for '${segments[3]}': ${err.message}`);
            }
        }
        else {
            try {
                value = JSON.parse(payload.toString());
            }
            catch (err) {
                console.warn(`MqttBrokerServer at ${this.brokerURI}, packet has no Content Type and does not parse as JSON, relaying raw (string) payload.`);
                value = payload.toString();
            }
        }
        const options = {
            formIndex: core_1.ProtocolHelpers.findRequestMatchingFormIndex(action.forms, this.scheme, this.brokerURI, core_1.ContentSerdes.DEFAULT),
        };
        thing
            .handleInvokeAction(segments[3], value, options)
            .then((output) => {
            if (output) {
                console.warn(`MqttBrokerServer at ${this.brokerURI} cannot return output '${segments[3]}'`);
            }
        })
            .catch((err) => {
            console.error(`MqttBrokerServer at ${this.brokerURI} got error on invoking '${segments[3]}': ${err.message}`);
        });
    }
    handlePropertyWrite(property, packet, payload, segments, thing) {
        if (!property.readOnly) {
            let contentType = core_1.ContentSerdes.DEFAULT;
            if ("contentType" in packet.properties) {
                contentType = packet.properties.contentType;
            }
            const options = {
                formIndex: core_1.ProtocolHelpers.findRequestMatchingFormIndex(property.forms, this.scheme, this.brokerURI, contentType),
            };
            try {
                thing.handleWriteProperty(segments[3], JSON.parse(payload.toString()), options);
            }
            catch (err) {
                console.error("[binding-mqtt]", `MqttBrokerServer at ${this.brokerURI} got error on writing to property '${segments[3]}': ${err.message}`);
            }
        }
        else {
            console.warn("[binding-mqtt]", `MqttBrokerServer at ${this.brokerURI} received message for readOnly property at '${segments.join("/")}'`);
        }
    }
    destroy(thingId) {
        return __awaiter(this, void 0, void 0, function* () {
            console.debug("[binding-mqtt]", `MqttBrokerServer on port ${this.getPort()} destroying thingId '${thingId}'`);
            let removedThing;
            for (const name of Array.from(this.things.keys())) {
                const expThing = this.things.get(name);
                if (expThing != null && expThing.id != null && expThing.id === thingId) {
                    this.things.delete(name);
                    removedThing = expThing;
                }
            }
            if (removedThing) {
                console.info("[binding-mqtt]", `MqttBrokerServer succesfully destroyed '${removedThing.title}'`);
            }
            else {
                console.info("[binding-mqtt]", `MqttBrokerServer failed to destroy thing with thingId '${thingId}'`);
            }
            return removedThing !== undefined;
        });
    }
    start(servient) {
        return new Promise((resolve, reject) => {
            if (this.brokerURI === undefined) {
                console.warn("[binding-mqtt]", `No broker defined for MQTT server binding - skipping`);
                resolve();
            }
            else {
                if (this.config.psw === undefined) {
                    console.debug("[binding-mqtt]", `MqttBrokerServer trying to connect to broker at ${this.brokerURI}`);
                }
                else if (this.config.clientId === undefined) {
                    console.debug("[binding-mqtt]", `MqttBrokerServer trying to connect to secured broker at ${this.brokerURI}`);
                }
                else if (this.config.protocolVersion === undefined) {
                    console.debug("[binding-mqtt]", `MqttBrokerServer trying to connect to secured broker at ${this.brokerURI} with client ID ${this.config.clientId}`);
                }
                else {
                    console.debug("[binding-mqtt]", `MqttBrokerServer trying to connect to secured broker at ${this.brokerURI} with client ID ${this.config.clientId}`);
                }
                this.broker = mqtt.connect(this.brokerURI, this.config);
                this.broker.on("connect", () => {
                    console.info("[binding-mqtt]", `MqttBrokerServer connected to broker at ${this.brokerURI}`);
                    const parsed = new url.URL(this.brokerURI);
                    this.address = parsed.hostname;
                    const port = parseInt(parsed.port);
                    this.port = port > 0 ? port : 1883;
                    resolve();
                });
                this.broker.on("error", (error) => {
                    console.error("[binding-mqtt]", `MqttBrokerServer could not connect to broker at ${this.brokerURI}`);
                    reject(error);
                });
            }
        });
    }
    stop() {
        return __awaiter(this, void 0, void 0, function* () {
            if (this.broker !== undefined) {
                this.broker.unsubscribe("*");
                this.broker.end(true);
            }
            if (this.hostedBroker !== undefined) {
                yield new Promise((resolve) => this.hostedServer.close(() => resolve()));
                yield new Promise((resolve) => this.hostedBroker.close(() => resolve()));
            }
        });
    }
    getPort() {
        return this.port;
    }
    getAddress() {
        return this.address;
    }
    selfHostAuthentication(_client, username, password, done) {
        if (this.config.selfHostAuthentication && username !== undefined) {
            for (let i = 0; i < this.config.selfHostAuthentication.length; i++) {
                if (username === this.config.selfHostAuthentication[i].username &&
                    password.equals(Buffer.from(this.config.selfHostAuthentication[i].password))) {
                    done(undefined, true);
                    return;
                }
            }
            done(undefined, false);
            return;
        }
        done(undefined, true);
    }
}
exports.default = MqttBrokerServer;
//# sourceMappingURL=mqtt-broker-server.js.map