"use strict";
/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsIotMqttConnectionConfigBuilder = void 0;
/**
 * Module for AWS IoT configuration and connection establishment
 *
 * @packageDocumentation
 */
var auth_1 = require("./auth");
var io_1 = require("./io");
var platform = __importStar(require("../common/platform"));
var iot_shared = __importStar(require("../common/aws_iot_shared"));
/**
 * Builder functions to create a {@link MqttConnectionConfig} which can then be used to create
 * a {@link MqttClientConnection}, configured for use with AWS IoT.
 *
 * @category IoT
 */
var AwsIotMqttConnectionConfigBuilder = /** @class */ (function () {
    function AwsIotMqttConnectionConfigBuilder() {
        this.params = {
            client_id: '',
            host_name: '',
            socket_options: new io_1.SocketOptions(),
            port: 8883,
            clean_session: false,
            keep_alive: undefined,
            will: undefined,
            username: '',
            password: undefined,
            websocket: {},
            credentials_provider: undefined
        };
    }
    /**
     * For API compatibility with the native version. Does not set up mTLS.
     *
     * @returns a new websocket connection builder object with default TLS configuration
     */
    AwsIotMqttConnectionConfigBuilder.new_mtls_builder = function () {
        var args = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            args[_i] = arguments[_i];
        }
        return AwsIotMqttConnectionConfigBuilder.new_builder_for_websocket();
    };
    /**
     * For API compatibility with the native version. Alias for {@link new_builder_for_websocket}.
     *
     * @returns a new websocket connection builder object with default TLS configuration
     */
    AwsIotMqttConnectionConfigBuilder.new_with_websockets = function () {
        var args = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            args[_i] = arguments[_i];
        }
        return AwsIotMqttConnectionConfigBuilder.new_builder_for_websocket();
    };
    /**
     * For API compatibility with the native version. Alias for {@link new_builder_for_websocket}.
     *
     * @returns a new websocket connection builder object with default TLS configuration
     */
    AwsIotMqttConnectionConfigBuilder.new_default_builder = function () {
        return AwsIotMqttConnectionConfigBuilder.new_builder_for_websocket();
    };
    /**
     * Creates a new builder using MQTT over websockets (the only option in browser)
     *
     * @returns a new websocket connection builder object with default TLS configuration
     */
    AwsIotMqttConnectionConfigBuilder.new_builder_for_websocket = function () {
        var builder = new AwsIotMqttConnectionConfigBuilder();
        return builder;
    };
    /**
     * Configures the IoT endpoint for this connection
     * @param endpoint The IoT endpoint to connect to
     *
     * @returns this builder object
     */
    AwsIotMqttConnectionConfigBuilder.prototype.with_endpoint = function (endpoint) {
        this.params.host_name = endpoint;
        return this;
    };
    /**
     * The client id to use for this connection
     * @param client_id The client id to use for this connection
     *
     * @returns this builder object
     */
    AwsIotMqttConnectionConfigBuilder.prototype.with_client_id = function (client_id) {
        this.params.client_id = client_id;
        return this;
    };
    /**
     * The port to connect to on the IoT endpoint
     * @param port The port to connect to on the IoT endpoint. Usually 8883 for MQTT, or 443 for websockets
     *
     * @returns this builder object
     */
    AwsIotMqttConnectionConfigBuilder.prototype.with_port = function (port) {
        this.params.port = port;
        return this;
    };
    /**
     * Determines whether or not the service should try to resume prior subscriptions, if it has any
     * @param clean_session true if the session should drop prior subscriptions when this client connects, false to resume the session
     *
     * @returns this builder object
     */
    AwsIotMqttConnectionConfigBuilder.prototype.with_clean_session = function (clean_session) {
        this.params.clean_session = clean_session;
        return this;
    };
    /**
     * Configures the connection to use MQTT over websockets. No-op in the browser.
     *
     * @returns this builder object
     */
    AwsIotMqttConnectionConfigBuilder.prototype.with_use_websockets = function () {
        /* no-op, but valid in the browser */
        return this;
    };
    /**
     * Configures MQTT keep-alive via PING messages. Note that this is not TCP keepalive.
     * @param keep_alive How often in seconds to send an MQTT PING message to the service to keep the connection alive
     *
     * @returns this builder object
     */
    AwsIotMqttConnectionConfigBuilder.prototype.with_keep_alive_seconds = function (keep_alive) {
        this.params.keep_alive = keep_alive;
        return this;
    };
    /**
     * Configures the TCP socket timeout (in milliseconds)
     * @param timeout_ms TCP socket timeout
     * @deprecated in favor of socket options
     *
     * @returns this builder object
     */
    AwsIotMqttConnectionConfigBuilder.prototype.with_timeout_ms = function (timeout_ms) {
        this.with_ping_timeout_ms(timeout_ms);
        return this;
    };
    /**
     * Configures the PINGREQ response timeout (in milliseconds)
     * @param ping_timeout PINGREQ response timeout
     *
     * @returns this builder object
     */
    AwsIotMqttConnectionConfigBuilder.prototype.with_ping_timeout_ms = function (ping_timeout) {
        this.params.ping_timeout = ping_timeout;
        return this;
    };
    /**
     * Configures the will message to be sent when this client disconnects
     * @param will The will topic, qos, and message
     *
     * @returns this builder object
     */
    AwsIotMqttConnectionConfigBuilder.prototype.with_will = function (will) {
        this.params.will = will;
        return this;
    };
    /**
     * Configures the common settings for the socket to use when opening a connection to the server
     * @param socket_options The socket settings
     *
     * @returns this builder object
     */
    AwsIotMqttConnectionConfigBuilder.prototype.with_socket_options = function (socket_options) {
        this.params.socket_options = socket_options;
        return this;
    };
    /**
     * Allows additional headers to be sent when establishing a websocket connection. Useful for custom authentication.
     * @param headers Additional headers to send during websocket connect
     *
     * @returns this builder object
     */
    AwsIotMqttConnectionConfigBuilder.prototype.with_websocket_headers = function (headers) {
        this.params.websocket = {
            headers: headers
        };
        return this;
    };
    /**
     * Configures Static AWS credentials for this connection.
     * Please note that the static credential will fail when the web session expired.
     * @param aws_region The service region to connect to
     * @param aws_access_id IAM Access ID
     * @param aws_secret_key IAM Secret Key
     * @param aws_sts_token session credentials token (optional)
     *
     * @returns this builder object
     */
    AwsIotMqttConnectionConfigBuilder.prototype.with_credentials = function (aws_region, aws_access_id, aws_secret_key, aws_sts_token) {
        var provider = new auth_1.StaticCredentialProvider({ aws_region: aws_region,
            aws_access_id: aws_access_id,
            aws_secret_key: aws_secret_key,
            aws_sts_token: aws_sts_token });
        this.params.credentials_provider = provider;
        return this;
    };
    /**
     * Configures credentials provider (currently support for AWS Cognito Credential Provider) for this connection
     * @param customer_provider credential provider used to update credential when session expired (optional)
     *
     * @returns this builder object
     */
    AwsIotMqttConnectionConfigBuilder.prototype.with_credential_provider = function (customer_provider) {
        this.params.credentials_provider = customer_provider;
        return this;
    };
    /**
     * Sets the custom authorizer settings. This function will modify the username, port, and TLS options.
     *
     * @param username The username to use with the custom authorizer. If an empty string is passed, it will
     *                 check to see if a username has already been set (via WithUsername function). If no
     *                 username is set then no username will be passed with the MQTT connection.
     * @param authorizerName The name of the custom authorizer. If an empty string is passed, then
     *                       'x-amz-customauthorizer-name' will not be added with the MQTT connection.
     * @param authorizerSignature The signature of the custom authorizer. If an empty string is passed, then
     *                            'x-amz-customauthorizer-signature' will not be added with the MQTT connection.
     * @param password The password to use with the custom authorizer. If null is passed, then no password will
     *                 be set.
     */
    AwsIotMqttConnectionConfigBuilder.prototype.with_custom_authorizer = function (username, authorizer_name, authorizer_signature, password) {
        var username_string = iot_shared.populate_username_string_with_custom_authorizer("", username, authorizer_name, authorizer_signature, this.params.username);
        this.params.username = username_string;
        this.params.password = password;
        // Tells the websocket connection we are using a custom authorizer
        if (this.params.websocket) {
            this.params.websocket.protocol = "wss-custom-auth";
        }
        return this;
    };
    /**
     * Sets username for the connection
     *
     * @param username the username that will be passed with the MQTT connection
     */
    AwsIotMqttConnectionConfigBuilder.prototype.with_username = function (username) {
        this.params.username = username;
        return this;
    };
    /**
     * Sets password for the connection
     *
     * @param password the password that will be passed with the MQTT connection
     */
    AwsIotMqttConnectionConfigBuilder.prototype.with_password = function (password) {
        this.params.password = password;
        return this;
    };
    /**
     * Configure the max reconnection period (in second). The reonnection period will
     * be set in range of [reconnect_min_sec,reconnect_max_sec].
     * @param reconnect_max_sec max reconnection period
     */
    AwsIotMqttConnectionConfigBuilder.prototype.with_reconnect_max_sec = function (max_sec) {
        this.params.reconnect_max_sec = max_sec;
        return this;
    };
    /**
     * Configure the min reconnection period (in second). The reonnection period will
     * be set in range of [reconnect_min_sec,reconnect_max_sec].
     * @param reconnect_min_sec min reconnection period
     */
    AwsIotMqttConnectionConfigBuilder.prototype.with_reconnect_min_sec = function (min_sec) {
        this.params.reconnect_min_sec = min_sec;
        return this;
    };
    /**
     * Returns the configured MqttConnectionConfig
     * @returns The configured MqttConnectionConfig
     */
    AwsIotMqttConnectionConfigBuilder.prototype.build = function () {
        if (this.params.client_id === undefined || this.params.host_name === undefined) {
            throw 'client_id and endpoint are required';
        }
        // Add the metrics string
        if (this.params.username == undefined || this.params.username == null || this.params.username == "") {
            this.params.username = "?SDK=NodeJSv2&Version=";
        }
        else {
            if (this.params.username.indexOf("?") != -1) {
                this.params.username += "&SDK=NodeJSv2&Version=";
            }
            else {
                this.params.username += "?SDK=NodeJSv2&Version=";
            }
        }
        this.params.username += platform.crt_version();
        return this.params;
    };
    return AwsIotMqttConnectionConfigBuilder;
}());
exports.AwsIotMqttConnectionConfigBuilder = AwsIotMqttConnectionConfigBuilder;
//# sourceMappingURL=aws_iot.js.map