
/*
--
-- Copyright (C) 2016  <fastrgv@gmail.com>
--
-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You may read the full text of the GNU General Public License
-- at <http://www.gnu.org/licenses/>.
--
*/



/*

BreadthFirstSearch puzzle solver.
=================================================

This code solves block slider puzzles *.blok.

An essential key feature is the encoding each puzzle state
in a way that is unique but does not discriminate between mere
permutations of like pieces.  Exception:  when the puzzle
goal involves swapping positions of two large 2x2 blocks, 
we MUST distinguish those two.

Simpy type the full path to a puzzle file ending with "blok", EG

bfs ./puzzles/l_hard_90mov.blok

and the solution is output to "path.txt".

NOTE...7feb16 update:
In order to handle two puzzles for which this solver previously failed, 
first it was noted that the same logic tolerated 6x4 puzzles, in addition 
to the usual 5x4.  Now, if there is a secondary 2x2 block (eg e_med.blok) 
within the usual 5x4 box, a new tactic is applied whereby its position is
encoded by [falsely] adding 1 to its row.  This is done...
a) because we can, now that 6 rows are tolerated;
b) we must distinguish between the 2 large blocks in order to be
	able to swap their positions;
c) new logic won't disturb other puzzles which have only one 2x2 (#0).

*/


#include "splaytree.h"

#include <cmath>


#include <cassert>
#include <iostream>
#include <fstream>
#include <string>
using std::ifstream;
using std::ofstream;
using std::cout;
using std::cerr;
using std::endl;
using std::string;


#include <chrono>
#include <thread>

#include <unistd.h>







bool Ok(false), playedonce(false), winner(false);

int dblk, nblk, gblk;


int const maxblk(29);

float rowcen[maxblk], colcen[maxblk];
string clrstr[maxblk];
int  bshape[maxblk];
int flen;
static string fname;

static string objectiveText, infilname;

int blank1, blank2, selBlock;

int const nrowmax(9), ncolmax(9); 

int xmax, ymax, nrow, ncol;

float const epsilon(0.01);

float grow[2],gcol[2]; // at most two goal positions

double lastTime;


// 17mar15 : added undo structure to facilitate autosolve
// utility...search "trail" for all undo mechanisms
static int const trailmax(200); // even klotski only needs 116 moves
static int  ntrail(0);

static unsigned long  trailenc22[trailmax];
static unsigned long  trailenc12[trailmax];
static unsigned long  trailenc21[trailmax];
static unsigned long  trailenc11[trailmax];


// This ID scheme assumes that layout is fully defined by the positions of
// [at most 1] 2x2 rectangle
// [at most 4] 2x1 rectangles 
// [at most 4] 1x2 rectangles 
// [at most 4] 1x1 squares

static int  trailsel[trailmax];
static char trailchr[trailmax];


// (I have one 6x4 puzzle!!)
// a 5x4 puzzle has at most 20 upper left corners
// ...and no two pieces can have the same UL corner
int endx( int ulr, int ulc )
// ulr : 0..5   (was ulr : 0..4)
// ulc : 0..3
{
//cout<<ulr<<":"<<ulc<<endl;
	int res=4*ulr+ulc;
	assert( res >= 0 );
	assert( res <= 23 );

	return res;
}


unsigned long bitrep( int e )
{
	// note that sizeof(ulong)==8 => 64 bits
	// thus, 2**23 << 2**63 == MSB(ulong)
	//
	// e in 0..23 (was 0..19)
	unsigned long res= static_cast<unsigned long>( pow(2,e) );
	return res;
}


// example usage:
//
// s22 += bitrep( endx(r-1,c-1) );     UL corner of 2x2
// s12 += bitrep( endx(r-0.5,c-1) );   UL corner of 1x2
// s21 += bitrep( endx(r-1,c-0.5) );   UL corner of 2x1
// s11 += bitrep( endx(r-0.5,c-0.5) ); UL corner of 1x1
//
// trailenc22[ntrail] = s22
// ...
// trailenc11[ntrail] = s11;
//
// if all 4 trailencXX match, puzzle config is equivalent






void dump() // path so far
{
	char letters[13]={'a','b','c','d','e','f','g','h','i','j','k','l','m'};
	cout<<endl;
	cout<<" writing solution to file:  path.txt"<<endl;
	cout<<endl;

	ofstream f("path.txt");

	f<<endl;
	f<<" solution to "<<infilname<<endl;
	f<<"path with #steps="<<ntrail<<endl<<endl;
	f<<"id {a,b,c,...}"<<endl;
	f<<"| dir {u,d,l,r}"<<endl;
	f<<"| |"<<endl;
	
	for(int i=0; i<ntrail; i++) {
		f<< letters[trailsel[i]] <<" "<< trailchr[i]
		<<endl;
	}

	f<<endl;
	f.close();
}










void test4winner()
{

	winner=true;
	for(int g=0; g<gblk; g++) {
		winner=winner&& 
			( fabs(rowcen[g]-grow[g])<epsilon ) && 
			( fabs(colcen[g]-gcol[g])<epsilon );
	}

	if( winner ) { // print out soln

		dump();

		cout<<"Solution found !!!  ... quitting..." <<endl;

		exit(0);

	}
	
}





void Init( string fname )
{
	ifstream gfil( fname );
	assert( gfil);

	getline(gfil, objectiveText);

	// (nrow,ncol) = outer dimension
	// dblk = # non-blank rectangles
	// nblk = # rectangles including two blank 1x1 squares = dblk+2
	// gblk = # goal positions that must be attained
	// (grow,gcol) = goal position[s]
	// bshape = 11 or 12 or 21 or 22 = block shape

	gfil >> nrow >> ncol >> dblk >> gblk;
	nblk=dblk+2;
	for(int g=0; g<gblk; g++)
	{
		gfil >> grow[g] >> gcol[g];
	}

	for(int i=0; i<nblk; i++)
	{
		gfil >> bshape[i] >> rowcen[i] >> colcen[i] >> clrstr[i];
	}
	gfil.close();

	blank1=dblk+1 -1;
	blank2=dblk+2 -1;

	ntrail=0;

} // end Init





int moveleft(bool track) {

	float br1,bc1,br2,bc2, sr, sc;
	int shape;

	br1=rowcen[blank1];
	bc1=colcen[blank1];
	br2=rowcen[blank2];
	bc2=colcen[blank2];

	float obr1,obc1,obr2,obc2;
	obr1=br1; obr2=br2; obc1=bc1; obc2=bc2;

	sr   =rowcen[selBlock];
	sc   =colcen[selBlock];
	shape=bshape[selBlock];

	if( shape==22 ){
		if( (fabs(br1-br2)<1.1) && ((br1+br2)/2 == sr) && (bc1==bc2) && (bc1==sc-1.5) ) 
		//put blanks on rt of big square
		{
			colcen[selBlock] = sc-1.0;
			colcen[blank1]   = bc1+2.0;
			colcen[blank2]   = bc2+2.0;
		}

	}else
	if( shape==21 ){
		if( (fabs(br1-br2)<1.1) && ((br1+br2)/2 == sr) && (bc1==bc2) && (bc1==sc-1.0) ) 
		// put blanks on rt of rect
		{
			colcen[selBlock] = sc-1.0;
			colcen[blank1]   = bc1+1.0;
			colcen[blank2]   = bc2+1.0;
		}


	}else
	if( shape==12 ){
		if( (br1==sr) && (bc1==sc-1.5) ) //swap b1,sel
		{
			colcen[selBlock] = sc-1.0;
			colcen[blank1]   = bc1+2.0;
		}
		else
		if( (br2==sr) && (bc2==sc-1.5) ) //swap b2,sel
		{
			colcen[selBlock] = sc-1.0;
			colcen[blank2]   = bc2+2.0;
		}


	}else
	if( shape==11 ){
		if( (br1==sr) && (bc1==sc-1.0) ) //swap b1,sel
		{
			colcen[selBlock] = bc1;
			colcen[blank1]   = sc;
		}
		else
		if( (br2==sr) && (bc2==sc-1.0) ) //swap b2,sel
		{
			colcen[selBlock] = bc2;
			colcen[blank2]   = sc;
		}
	}


	int ret;
	br1=rowcen[blank1];
	bc1=colcen[blank1];
	br2=rowcen[blank2];
	bc2=colcen[blank2];

	if( (obr1==br1) && (obr2==br2) && (obc1==bc1) && (obc2==bc2) )
	{
		ret=0; // no change
	}
	else	
	{

		if( track ) {

			unsigned long s22(0), s12(0), s21(0), s11(0);

			for(int j=0; j<dblk; j++)
			{

				if( bshape[j] == 22 )
				{
					int r = round(rowcen[j]-1);
					int c = round(colcen[j]-1);
					//s22 += bitrep( endx(r,c) );
					if(j==0) s22 += bitrep( endx(r,c) ); //ignore 2nd 2x2
					else     s22 += bitrep( endx(r+1,c) ); //tolerates 6 rows
				}
				else

				if( bshape[j] == 12 )
				{
					int r = round(rowcen[j]-0.5);
					int c = round(colcen[j]-1);
					s12 += bitrep( endx(r,c) );
				}
				else

				if( bshape[j] == 21 )
				{
					int r = round(rowcen[j]-1);
					int c = round(colcen[j]-0.5);
					s21 += bitrep( endx(r,c) );
				}
				else
				if( bshape[j] == 11 )
				{
					int r = round(rowcen[j]-0.5);
					int c = round(colcen[j]-0.5);
					s11 += bitrep( endx(r,c) );
				}

			}

			trailenc22[ntrail] = s22;
			trailenc12[ntrail] = s12;
			trailenc21[ntrail] = s21;
			trailenc11[ntrail] = s11;

			trailsel[ntrail]=selBlock;
			trailchr[ntrail]='l';
			ntrail++;

		}

		test4winner();
		ret=1;

	}
	return ret;


} // end moveleft









int moveright(bool track) {

	float br1,bc1,br2,bc2, sr, sc;
	int shape;

	br1=rowcen[blank1];
	bc1=colcen[blank1];
	br2=rowcen[blank2];
	bc2=colcen[blank2];

	float obr1,obc1,obr2,obc2;
	obr1=br1; obr2=br2; obc1=bc1; obc2=bc2;

	sr   =rowcen[selBlock];
	sc   =colcen[selBlock];
	shape=bshape[selBlock];

	if( shape==22 ){
		if( (fabs(br1-br2)<1.1) && ((br1+br2)/2 == sr) && (bc1==bc2) && (bc1==sc+1.5) ) 
		//put blanks on lt of big square
		{
			colcen[selBlock] = sc+1.0;
			colcen[blank1]   = bc1-2.0;
			colcen[blank2]   = bc2-2.0;
		}

	}else
	if( shape==21 ){
		if( (fabs(br1-br2)<1.1) && ((br1+br2)/2 == sr) && (bc1==bc2) && (bc1==sc+1.0) )
		{
			colcen[selBlock] = sc+1.0;
			colcen[blank1]   = bc1-1.0;
			colcen[blank2]   = bc2-1.0;
		}

	}else
	if( shape==12 ){
		if( (br1==sr) && (bc1==sc+1.5) ) //swap b1,sel
		{
			colcen[selBlock] = sc+1.0;
			colcen[blank1]   = bc1-2.0;
		}
		else
		if( (br2==sr) && (bc2==sc+1.5) ) //swap b2,sel
		{
			colcen[selBlock] = sc+1.0;
			colcen[blank2]   = bc2-2.0;
		}

	}else
	if( shape==11 ){
		if( (br1==sr) && (bc1==sc+1.0) ) //swap b1,sel
		{
			colcen[selBlock] = bc1;
			colcen[blank1]   = sc;
		}
		else
		if( (br2==sr) && (bc2==sc+1.0) ) //swap b2,sel
		{
			colcen[selBlock] = bc2;
			colcen[blank2]   = sc;
		}
	}

	int ret;
	br1=rowcen[blank1];
	bc1=colcen[blank1];
	br2=rowcen[blank2];
	bc2=colcen[blank2];

	if( (obr1==br1) && (obr2==br2) && (obc1==bc1) && (obc2==bc2) )
	{
		ret=0; // no change
	}
	else	
	{

		if( track ) {

			unsigned long s22(0), s12(0), s21(0), s11(0);

			for(int j=0; j<dblk; j++)
			{

				if( bshape[j] == 22 )
				{
					int r = round(rowcen[j]-1);
					int c = round(colcen[j]-1);
					//s22 += bitrep( endx(r,c) );
					if(j==0) s22 += bitrep( endx(r,c) ); //ignore 2nd 2x2
					else     s22 += bitrep( endx(r+1,c) ); //tolerates 6 rows
				}
				else

				if( bshape[j] == 12 )
				{
					int r = round(rowcen[j]-0.5);
					int c = round(colcen[j]-1);
					s12 += bitrep( endx(r,c) );
				}
				else


				if( bshape[j] == 21 )
				{
					int r = round(rowcen[j]-1);
					int c = round(colcen[j]-0.5);
					s21 += bitrep( endx(r,c) );
				}
				else
				if( bshape[j] == 11 )
				{
					int r = round(rowcen[j]-0.5);
					int c = round(colcen[j]-0.5);
					s11 += bitrep( endx(r,c) );
				}

			}


			trailenc22[ntrail] = s22;
			trailenc12[ntrail] = s12;
			trailenc21[ntrail] = s21;
			trailenc11[ntrail] = s11;


			trailsel[ntrail]=selBlock;
			trailchr[ntrail]='r';
			ntrail++;

		}

		test4winner();
		ret=1;

	}
	return ret;

} // end moveright




int moveup(bool track) {

	float br1,bc1,br2,bc2, sr, sc;
	int shape;

	br1=rowcen[blank1];
	bc1=colcen[blank1];
	br2=rowcen[blank2];
	bc2=colcen[blank2];

	float obr1,obc1,obr2,obc2;
	obr1=br1; obr2=br2; obc1=bc1; obc2=bc2;

	sr   =rowcen[selBlock];
	sc   =colcen[selBlock];
	shape=bshape[selBlock];

	if( shape==22 ){
		if( (fabs(bc1-bc2)<1.1) && ((bc1+bc2)/2 == sc) && (br1==br2) && (br1==sr-1.5) ) 
		//put blanks below big square
		{
			rowcen[selBlock] = sr-1.0;
			rowcen[blank1]   = br1+2.0;
			rowcen[blank2]   = br2+2.0;
		}

	}else
	if( shape==12 ){
		if( (fabs(bc1-bc2)<1.1) && ((bc1+bc2)/2 == sc) && (br1==br2) && (br1==sr-1.0) )
		{
			rowcen[selBlock] = sr-1.0;
			rowcen[blank1]   = br1+1.0;
			rowcen[blank2]   = br2+1.0;
		}

	}else
	if( shape==21 ){
		if( (bc1==sc) && (br1==sr-1.5) ) //swap b1,sel
		{
			rowcen[selBlock] = sr-1.0;
			rowcen[blank1]   = br1+2.0;
		}
		else
		if( (bc2==sc) && (br2==sr-1.5) ) //swap b2,sel
		{
			rowcen[selBlock] = sr-1.0;
			rowcen[blank2]   = br2+2.0;
		}

	}else
	if( shape==11 ){
		if( (bc1==sc) && (br1==sr-1.0) ) //swap b1,sel
		{
			rowcen[selBlock] = br1;
			rowcen[blank1]   = sr;
		}
		else
		if( (bc2==sc) && (br2==sr-1.0) ) //swap b2,sel
		{
			rowcen[selBlock] = br2;
			rowcen[blank2]   = sr;
		}
	}

	
	int ret;
	br1=rowcen[blank1];
	bc1=colcen[blank1];
	br2=rowcen[blank2];
	bc2=colcen[blank2];

	if( (obr1==br1) && (obr2==br2) && (obc1==bc1) && (obc2==bc2) )
	{
		ret=0; // no change
	}
	else	
	{

		if( track ) {

			unsigned long s22(0), s12(0), s21(0), s11(0);

			for(int j=0; j<dblk; j++)
			{

				if( bshape[j] == 22 )
				{
					int r = round(rowcen[j]-1);
					int c = round(colcen[j]-1);
					//s22 += bitrep( endx(r,c) );
					if(j==0) s22 += bitrep( endx(r,c) ); //ignore 2nd 2x2
					else     s22 += bitrep( endx(r+1,c) ); //tolerates 6 rows
				}
				else

				if( bshape[j] == 12 )
				{
					int r = round(rowcen[j]-0.5);
					int c = round(colcen[j]-1);
					s12 += bitrep( endx(r,c) );
				}
				else


				if( bshape[j] == 21 )
				{
					int r = round(rowcen[j]-1);
					int c = round(colcen[j]-0.5);
					s21 += bitrep( endx(r,c) );
				}
				else
				if( bshape[j] == 11 )
				{
					int r = round(rowcen[j]-0.5);
					int c = round(colcen[j]-0.5);
					s11 += bitrep( endx(r,c) );
				}

			}


			trailenc22[ntrail] = s22;
			trailenc12[ntrail] = s12;
			trailenc21[ntrail] = s21;
			trailenc11[ntrail] = s11;


			trailsel[ntrail]=selBlock;
			trailchr[ntrail]='u';
			ntrail++;
		}

		test4winner();
		ret=1;

	}
	return ret;

} // end moveup



int movedown(bool track) {

	float br1,bc1,br2,bc2, sr, sc;
	int shape;

	br1=rowcen[blank1];
	bc1=colcen[blank1];
	br2=rowcen[blank2];
	bc2=colcen[blank2];

	float obr1,obc1,obr2,obc2;
	obr1=br1; obr2=br2; obc1=bc1; obc2=bc2;

	sr   =rowcen[selBlock];
	sc   =colcen[selBlock];
	shape=bshape[selBlock];

	if( shape==22 ){
		if( (fabs(bc1-bc2)<1.1) && ((bc1+bc2)/2 == sc) && (br1==br2) && (br1==sr+1.5) ) 
		//put blanks above big square
		{
			rowcen[selBlock] = sr+1.0;
			rowcen[blank1]   = br1-2.0;
			rowcen[blank2]   = br2-2.0;
		}

	}else
	if( shape==12 ){
		if( (fabs(bc1-bc2)<1.1) && ((bc1+bc2)/2 == sc) && (br1==br2) && (br1==sr+1.0) )
		{
			rowcen[selBlock] = sr+1.0;
			rowcen[blank1]   = br1-1.0;
			rowcen[blank2]   = br2-1.0;
		}

	}else
	if( shape==21 ){
		if( (bc1==sc) && (br1==sr+1.5) ) //swap b1,sel
		{
			rowcen[selBlock] = sr+1.0;
			rowcen[blank1]   = br1-2.0;
		}
		else
		if( (bc2==sc) && (br2==sr+1.5) ) //swap b2,sel
		{
			rowcen[selBlock] = sr+1.0;
			rowcen[blank2]   = br2-2.0;
		}

	}else
	if( shape==11 ){
		if( (bc1==sc) && (br1==sr+1.0) ) //swap b1,sel
		{
			rowcen[selBlock] = br1;
			rowcen[blank1]   = sr;
		}
		else
		if( (bc2==sc) && (br2==sr+1.0) ) //swap b2,sel
		{
			rowcen[selBlock] = br2;
			rowcen[blank2]   = sr;
		}
	}


	int ret;
	br1=rowcen[blank1];
	bc1=colcen[blank1];
	br2=rowcen[blank2];
	bc2=colcen[blank2];

	if( (obr1==br1) && (obr2==br2) && (obc1==bc1) && (obc2==bc2) )
	{
		ret=0; // no change
	}
	else	
	{

		if( track ) {

			unsigned long s22(0), s12(0), s21(0), s11(0);

			for(int j=0; j<dblk; j++)
			{

				if( bshape[j] == 22 )
				{
					int r = round(rowcen[j]-1);
					int c = round(colcen[j]-1);
					//s22 += bitrep( endx(r,c) );
					if(j==0) s22 += bitrep( endx(r,c) ); //ignore 2nd 2x2
					else     s22 += bitrep( endx(r+1,c) ); //tolerates 6 rows
				}
				else

				if( bshape[j] == 12 )
				{
					int r = round(rowcen[j]-0.5);
					int c = round(colcen[j]-1);
					s12 += bitrep( endx(r,c) );
				}
				else


				if( bshape[j] == 21 )
				{
					int r = round(rowcen[j]-1);
					int c = round(colcen[j]-0.5);
					s21 += bitrep( endx(r,c) );
				}
				else
				if( bshape[j] == 11 )
				{
					int r = round(rowcen[j]-0.5);
					int c = round(colcen[j]-0.5);
					s11 += bitrep( endx(r,c) );
				}

			}

			trailenc22[ntrail] = s22;
			trailenc12[ntrail] = s12;
			trailenc21[ntrail] = s21;
			trailenc11[ntrail] = s11;

			trailsel[ntrail]=selBlock;
			trailchr[ntrail]='d';
			ntrail++;

			//if( (selBlock==0) && (rowcen[0]>=2.5) ) dump();

		}

		test4winner();
		ret=1;

	}
	return ret;

} // end movedown




void undo() {

	int res;

	if( ntrail > 0 )
	{

		ntrail--;
		selBlock=trailsel[ntrail];
		switch ( trailchr[ntrail] )
		{
			case 'd':  res=   moveup(false); assert(res>0); break;
			case 'u':  res= movedown(false); assert(res>0); break;
			case 'r':  res= moveleft(false); assert(res>0); break;
			case 'l':  res=moveright(false); assert(res>0); break;
			default:
				cout<<"ERROR in undo:  bad trailchr"<<endl;
				assert(false);
				break;
		} // end inner switch

	} // end if

}



/////////////// hash table begin //////////////////////////////////////////




typedef unsigned long ul;

class Key {

public:

	// maybe should define a copy constructor (although not used here)

	Key():sum11(0),sum12(0),sum21(0),sum22(0) {}; // default constructor
	
	Key(ul a11, ul a12, ul a21, ul a22)
	:sum11(a11),sum12(a12),sum21(a21),sum22(a22){}; 
	// initializing constructor

	~Key(){};

	ul sum11,sum12,sum21,sum22; // these uniquely define the box layout

	bool operator!=(Key other) const
	{
		return 
			(sum11!=other.sum11)||(sum12!=other.sum12)||
			(sum21!=other.sum21)||(sum22!=other.sum22);
	}

	bool operator==(Key other) const
	{
		return 
			(sum11==other.sum11)&&(sum12==other.sum12)&&
			(sum21==other.sum21)&&(sum22==other.sum22);
	}

	bool operator<(Key other) const
	{
		if( sum11 < other.sum11 ) return true;
		else if( sum11 > other.sum11 ) return false;

		else // 1st sum equal

		if( sum12<other.sum12 ) return true;
		else if( sum12>other.sum12 ) return false;

		else // 1st 2 sums equal

		if( sum21<other.sum21 ) return true;
		else if( sum21>other.sum21 ) return false;

		else // 1st 3 sums equal

		if( sum22<other.sum22 ) return true;
		else return false;
	}

}; // end class Key



struct hashnode 
{ 
	char tchr;
	int tsel;
	Key prevKey;
};

typedef struct hashnode hashType;

SplayTree<Key, hashType> hashtable;

static int const mxkey(5000111); 
static Key keylist[mxkey]; 
static int nkey(0);

static hashType rec0; // initial base = beginning config


///////////////// database structs begin /////////////////////////////////////

static int const maxdepth(200); // klotski(118), moms(188)

///////////////// database structs end ... begin methods /////////////////////////




void addifnew(Key okey) { // add to hashtable if new

	int const nt(ntrail-1);
	Key nukey( 
		trailenc11[nt],
		trailenc12[nt],
		trailenc21[nt],
		trailenc22[nt] );

	bool found(false);
	hashType rec;
	hashtable.find( nukey, rec, found );

	if( !found ) {

		rec.prevKey = okey;
		rec.tsel = trailsel[nt];
		rec.tchr = trailchr[nt];

		hashtable.insert( nukey, rec );

		keylist[nkey] = nukey;
		nkey++;

	} // end if not found

} // end addifnew


// recursive:
int getrail(Key pkey) { // load trailsel, trailchr from DB

	int k;
	bool found(true);
	hashType rec;

	hashtable.find( pkey, rec, found );

	bool root = (rec.tchr == rec0.tchr);

//cout<<"Ok @ "<<__LINE__<<endl;

	if( !found || root ) 
	{
		k=0;
	}
	else
	{
		k=getrail( rec.prevKey );

		trailchr[k]=rec.tchr;
		trailsel[k]=rec.tsel;
		k++;
	}

	return k;
} // end getrail


///////////////// end database methods /////////////////////////



void restore(int kkey, Key & okey) {

	Init(infilname); // define puzzle parameters here

	int res(0);

	ntrail=0;

	okey=keylist[kkey];
	ntrail = getrail(okey); // loads trailsel, trailchr


	for(int i=0; i<ntrail; i++) {

		selBlock=trailsel[i];
		switch ( trailchr[i] )
		{
			case 'u':  res=   moveup(false); assert(res>0); break;
			case 'd':  res= movedown(false); assert(res>0); break;
			case 'l':  res= moveleft(false); assert(res>0); break;
			case 'r':  res=moveright(false); assert(res>0); break;
			default:
				cout<<"ERROR in restore:  bad trailchr"<<endl;
				assert(false);
				break;
		} // end switch

	} // end for i


} // end restore



static int start(0);

void trymove(int lev) {

	Key okey;
	int stop=nkey;


for(int config=start; config<stop; config++) {


	restore(config,okey);


	int res(0);

// for each nonblank rectangle, try moving it one step in each direction
	for(int i=0; i<dblk; i++) {

		selBlock=i; res=0;
		res=moveup(true);
		if( res>0 ) {
			addifnew(okey);
			undo();
		}

		selBlock=i;	res=0;
		res=movedown(true);
		if( res>0 ) {
			addifnew(okey);
			undo();
		}

		selBlock=i;	res=0;
		res=moveright(true);
		if( res>0 ) {
			addifnew(okey);
			undo();
		}

		selBlock=i;	res=0;
		res=moveleft(true);
		if( res>0 ) {
			addifnew(okey);
			undo();
		}

	} // end for i loop

} // end config loop

start=stop;
if( nkey<=stop ) cout<<"Sorry, no solution found!"<<endl;
assert( nkey > stop ); // ?any more configs found?

if( lev%10 == 0 )
cout<<"trymove...lev="<<lev<<", #nuCfg="<<nkey-stop<<", tot="<<nkey<<endl;


} // end trymove
















int main(int argc, char *argv[])
{

	if( argc != 2 )
	{
		cout<<"I need the full-path-name of a data file: *.blok"<<endl;
	}
	assert( argc == 2 );
	infilname = ( argv[1] );


	Key nukey(0,0,0,0);
	rec0.prevKey = nukey ;
	rec0.tsel = 0;
	rec0.tchr = 's';

	hashtable.insert( nukey, rec0 );
	keylist[nkey]=nukey;
	nkey++;


	for(int j=0; j<maxdepth; j++) 
	if( !winner ) trymove(j);


	return 0;

} // end main





