/*
 *  Copyright 2010 argius
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */
package net.argius.stew.ui.window;

import static net.argius.stew.Iteration.asIterable;

import java.awt.event.*;

import javax.swing.table.*;
import javax.swing.text.*;
import javax.swing.text.Highlighter.*;
import javax.swing.tree.*;

/**
 * WindowLauncher̕R[h(2)B
 * eLXg@\̐B
 */
final class WindowLaunchHelper2 {

    private WindowLaunchHelper2() {
        // empty
    }

    /**
     * DatabaseInfoTreepTextSearch̐B
     * @param tree
     * @return TextSearch
     */
    static TextSearch createTextSearch(final DatabaseInfoTree tree) {
        return new TextSearch() {

            public boolean search(Matcher matcher) {
                return search(tree.getSelectionPath(), matcher);
            }

            private boolean search(TreePath path, Matcher matcher) {
                if (path == null) {
                    return false;
                }
                TreeNode node = (TreeNode)path.getLastPathComponent();
                if (node == null) {
                    return false;
                }
                boolean found = false;
                found = matcher.find(node.toString());
                if (found) {
                    tree.addSelectionPath(path);
                } else {
                    tree.removeSelectionPath(path);
                }
                if (!node.isLeaf() && node.getChildCount() >= 0) {
                    @SuppressWarnings("unchecked")
                    Iterable<DefaultMutableTreeNode> children = asIterable(node.children());
                    for (DefaultMutableTreeNode child : children) {
                        if (search(path.pathByAddingChild(child), matcher)) {
                            found = true;
                        }
                    }
                }
                return found;
            }

            public void reset() {
                // empty
            }

        };
    }

    /**
     * ResultSetTablepTextSearch̐B
     * @param table
     * @return TextSearch
     */
    static TextSearch createTextSearch(final ResultSetTable table) {
        return new TextSearch() {

            public boolean search(Matcher matcher) {
                final int rowCount = table.getRowCount();
                if (rowCount <= 0) {
                    return false;
                }
                final int columnCount = table.getColumnCount();
                final boolean backward = matcher.isBackward();
                final int amount = backward ? -1 : 1;
                final int rowStart = backward ? rowCount - 1 : 0;
                final int rowEnd = backward ? 0 : rowCount - 1;
                final int columnStart = backward ? columnCount - 1 : 0;
                final int columnEnd = backward ? 0 : columnCount - 1;
                int row = rowStart;
                int column = columnStart;
                if (table.getSelectedColumnCount() > 0) {
                    column = table.getSelectedColumn();
                    row = table.getSelectedRow() + amount;
                    if (backward) {
                        if (row < 0) {
                            --column;
                            if (column < 0) {
                                return false;
                            }
                            row = rowStart;
                        }
                    } else {
                        if (row >= rowCount) {
                            ++column;
                            if (column >= columnCount) {
                                return false;
                            }
                            row = rowStart;
                        }
                    }
                }
                final TableModel m = table.getModel();
                for (; backward ? column >= columnEnd : column <= columnEnd; column += amount) {
                    for (; backward ? row >= rowEnd : row <= rowEnd; row += amount) {
                        if (matcher.find(String.valueOf(m.getValueAt(row, column)))) {
                            table.changeSelection(row, column, false, false);
                            return true;
                        }
                    }
                    row = rowStart;
                }
                return false;
            }

            public void reset() {
                // empty
            }

        };
    }

    /**
     * ConsoleTextAreapTextSearch̐B
     * @param textArea
     * @return TextSearch
     */
    static TextSearch createTextSearch(final ConsoleTextArea textArea) {
        return new TextSearch() {

            public boolean search(Matcher matcher) {
                removeHighlights();
                try {
                    Highlighter highlighter = textArea.getHighlighter();
                    HighlightPainter painter = matcher.getHighlightPainter();
                    String text = textArea.getText();
                    int start = 0;
                    boolean matched = false;
                    while (matcher.find(text, start)) {
                        matched = true;
                        int matchedIndex = matcher.getStart();
                        highlighter.addHighlight(matchedIndex, matcher.getEnd(), painter);
                        start = matchedIndex + 1;
                    }
                    textArea.addKeyListener(new KeyAdapter() {

                        @Override
                        public void keyTyped(KeyEvent e) {
                            textArea.removeKeyListener(this);
                            removeHighlights();
                        }

                    });
                    return matched;
                } catch (BadLocationException ex) {
                    throw new RuntimeException(ex);
                }
            }

            public void reset() {
                removeHighlights();
            }

            void removeHighlights() {
                for (Highlight highlight : textArea.getHighlighter().getHighlights()) {
                    textArea.getHighlighter().removeHighlight(highlight);
                }
            }

        };
    }

}
