/*
 *  Copyright 2010 argius
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */
package net.argius.stew.ui.window;

import static java.awt.event.KeyEvent.VK_ESCAPE;
import static javax.swing.JOptionPane.*;
import static javax.swing.KeyStroke.getKeyStroke;
import static javax.swing.ScrollPaneConstants.*;
import static net.argius.stew.ui.window.Resource.getString;

import java.awt.*;
import java.awt.event.*;
import java.util.*;

import javax.swing.*;
import javax.swing.event.*;

import net.argius.stew.*;

/**
 * ڑݒҏW_CAOB
 */
final class ConnectorMapEditDialog extends JDialog implements ChangeListener {

    private final ConnectorMap connectorMap;
    private final JList idList;
    private final DefaultListModel listModel;

    /**
     * RXgN^B
     * @param owner et[
     * @param env s
     */
    ConnectorMapEditDialog(JFrame owner, Environment env) {
        // [CX^X]
        super(owner);
        final DefaultListModel listModel = new DefaultListModel();
        this.connectorMap = new ConnectorMap(env.getConnectorMap());
        this.idList = new JList(listModel);
        this.listModel = listModel;
        setTitle(getMessage("title"));
        setResizable(false);
        setDefaultCloseOperation(DO_NOTHING_ON_CLOSE);
        FlexiblePanel p = new FlexiblePanel();
        p.c.anchor = GridBagConstraints.CENTER;
        p.c.insets = new Insets(8, 12, 8, 0);
        add(p);
        // [R|[lg̐ݒ]
        // List
        final Component parent = this;
        final ConnectorMap connectorMap = this.connectorMap;
        final JList idList = this.idList;
        idList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        for (ConnectorEntry entry : ConnectorEntry.toList(connectorMap.values())) {
            listModel.addElement(entry);
        }
        JScrollPane pane = new JScrollPane(idList,
                                           VERTICAL_SCROLLBAR_ALWAYS,
                                           HORIZONTAL_SCROLLBAR_NEVER);
        pane.setWheelScrollingEnabled(true);
        idList.addMouseListener(new MouseAdapter() {

            @Override
            public void mouseClicked(MouseEvent e) {
                int clickCount = e.getClickCount();
                if (clickCount % 2 == 0) {
                    ConnectorEntry entry = (ConnectorEntry)idList.getSelectedValue();
                    openConnectorEditDialog(entry.getConnector());
                }
            }

        });
        p.addComponent(pane, false);
        // Button 1
        JPanel p1 = new JPanel(new GridLayout(6, 1, 4, 2));
        p1.add(new JButton(new AbstractAction(getMessage("button.addnew")) {

            public void actionPerformed(ActionEvent e) {
                final String id = showInputDialog(parent, getString("i.input-new-connector-id"));
                if (id == null) {
                    return;
                }
                try {
                    if (connectorMap.containsKey(id)) {
                        final String message = getString("e.id-already-exists", id);
                        showMessageDialog(parent, message, null, ERROR_MESSAGE);
                    } else {
                        openConnectorEditDialog(new Connector(id, new Properties()));
                    }
                } catch (Exception ex) {
                    WindowOutputProcessor.showErrorDialog(parent, ex);
                }
            }

        }));
        p1.add(new JButton(new AbstractAction(getMessage("button.modify")) {

            public void actionPerformed(ActionEvent e) {
                ConnectorEntry entry = (ConnectorEntry)idList.getSelectedValue();
                if (entry != null) {
                    openConnectorEditDialog(entry.getConnector());
                }
            }

        }));
        p1.add(new JButton(new AbstractAction(getMessage("button.rename")) {

            public void actionPerformed(ActionEvent e) {
                Object o = idList.getSelectedValue();
                if (o == null) {
                    return;
                }
                ConnectorEntry entry = (ConnectorEntry)o;
                final String newId = showInputDialog(parent,
                                                     getString("i.input-new-connector-id"),
                                                     entry.getId());
                if (newId == null) {
                    return;
                }
                if (newId.equals(entry.getId())) {
                    return;
                }
                connectorMap.remove(entry);
                connectorMap.put(newId, entry.getConnector());
                DefaultListModel m = (DefaultListModel)idList.getModel();
                Connector newConnector = new Connector(newId, entry.getConnector());
                m.set(m.indexOf(entry), new ConnectorEntry(newId, newConnector));
                idList.repaint();
            }

        }));
        p1.add(new JButton(new AbstractAction(getMessage("button.remove")) {

            public void actionPerformed(ActionEvent e) {
                if (showConfirmDialog(parent, getString("i.confirm-remove"), "", OK_CANCEL_OPTION) != OK_OPTION) {
                    return;
                }
                ConnectorEntry selected = (ConnectorEntry)idList.getSelectedValue();
                connectorMap.remove(selected.getId());
                DefaultListModel m = (DefaultListModel)idList.getModel();
                m.removeElement(selected);
            }

        }));
        p1.add(new JButton(new AbstractAction(getMessage("button.up")) {

            public void actionPerformed(ActionEvent e) {
                shiftSelectedElementUpward();
            }

        }));
        p1.add(new JButton(new AbstractAction(getMessage("button.down")) {

            public void actionPerformed(ActionEvent e) {
                shiftSelectedElementDownward();
            }

        }));
        p.c.gridwidth = GridBagConstraints.REMAINDER;
        p.c.insets = new Insets(8, 32, 8, 32);
        p.addComponent(p1, true);
        // Button 2
        JPanel p2 = new JPanel(new GridLayout(1, 2, 16, 8));
        p2.add(new JButton(new AbstractAction(getMessage("button.submit")) {

            public void actionPerformed(ActionEvent e) {
                requestClose(true);
            }

        }));
        p2.add(new JButton(new AbstractAction(getMessage("button.cancel")) {

            public void actionPerformed(ActionEvent e) {
                requestClose(false);
            }

        }));
        p.c.gridwidth = GridBagConstraints.REMAINDER;
        p.c.fill = GridBagConstraints.NONE;
        p.addComponent(p2, false);
        // [Cxg̐ݒ]
        final String closeKey = "cancel-and-close";
        getRootPane().getActionMap().put(closeKey, new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                requestClose(false);
            }

        });
        getRootPane().getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT)
                     .put(getKeyStroke(VK_ESCAPE, 0), closeKey);
        addWindowListener(new WindowAdapter() {

            @Override
            public void windowClosing(WindowEvent e) {
                requestClose(false);
            }

        });
    }

    /* @see javax.swing.event.ChangeListener#stateChanged(javax.swing.event.ChangeEvent) */
    public void stateChanged(ChangeEvent e) {
        Object source = e.getSource();
        if (source instanceof Connector) {
            Connector connector = (Connector)source;
            final String id = connector.getId();
            connectorMap.setConnector(id, connector);
            ConnectorEntry changed = new ConnectorEntry(id, connector);
            final int index = listModel.indexOf(changed);
            if (index >= 0) {
                listModel.set(index, changed);
                idList.setSelectedIndex(index);
            } else {
                listModel.add(0, changed);
                idList.setSelectedIndex(0);
            }
        }
    }

    /**
     * RlN^ҏW_CAOJB
     * @param connector ҏWΏۃRlN^
     */
    void openConnectorEditDialog(Connector connector) {
        ConnectorEditDialog dialog = new ConnectorEditDialog(this, connector);
        dialog.addChangeListener(this);
        dialog.setModal(true);
        dialog.setLocationRelativeTo(getParent());
        dialog.setSize(dialog.getPreferredSize());
        dialog.setVisible(true);
    }

    /**
     * IvfֈړB
     */
    void shiftSelectedElementUpward() {
        final int index = idList.getSelectedIndex();
        if (index == 0) {
            return;
        }
        final int newIndex = index - 1;
        swap(listModel, index, newIndex);
        idList.setSelectedIndex(newIndex);
        idList.ensureIndexIsVisible(newIndex);
    }

    /**
     * IvfֈړB
     */
    void shiftSelectedElementDownward() {
        final int index = idList.getSelectedIndex();
        final int size = listModel.getSize();
        if (index == size - 1) {
            return;
        }
        final int newIndex = index + 1;
        swap(listModel, index, newIndex);
        idList.setSelectedIndex(newIndex);
        idList.ensureIndexIsVisible(newIndex);
    }

    private static void swap(DefaultListModel listModel, int index1, int index2) {
        Object o = listModel.get(index1);
        listModel.set(index1, listModel.get(index2));
        listModel.set(index2, o);
    }

    /**
     * 鏈vB
     * @param withSaving ۑ𔺂ꍇ <code>true</code>
     */
    void requestClose(boolean withSaving) {
        try {
            if (withSaving) {
                if (showConfirmDialog(this, getString("i.confirm-save"), "", YES_NO_OPTION) != YES_OPTION) {
                    return;
                }
                ConnectorMap m = new ConnectorMap();
                for (Object o : listModel.toArray()) {
                    ConnectorEntry entry = (ConnectorEntry)o;
                    final String id = entry.getId();
                    m.setConnector(id, connectorMap.getConnector(id));
                }
                ConnectorConfiguration.save(m);
                connectorMap.clear();
                connectorMap.putAll(m);
            } else if (!connectorMap.equals(ConnectorConfiguration.load())) {
                if (showConfirmDialog(this,
                                      getString("i.confirm-without-save"),
                                      "",
                                      OK_CANCEL_OPTION) != OK_OPTION) {
                    return;
                }
            }
            dispose();
        } catch (Exception ex) {
            showMessageDialog(this, ex.getMessage());
        }
    }

    /**
     * (O)bZ[W̎擾B
     * @param key L[
     * @return bZ[W
     */
    private static String getMessage(String key) {
        return Resource.getString("ConnectorMapEditDialog." + key);
    }

}
