/*
 *  Copyright 2010 argius
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */
package net.argius.stew.ui.console;

import java.sql.*;
import java.util.*;

import net.argius.stew.*;
import net.argius.stew.text.*;
import net.argius.stew.ui.*;

/**
 * R\[[h̏o͐B
 */
public final class ConsoleOutputProcessor implements OutputProcessor {

    private static final int WIDTH_LIMIT = 30;

    /* @see net.argius.stew.OutputProcessor#output(java.lang.Object) */
    public void output(Object object) {
        if (object instanceof ResultSetReference) {
            outputResult((ResultSetReference)object);
        } else if (object instanceof ResultSet) {
            outputResult(new ResultSetReference((ResultSet)object, ""));
        } else if (object instanceof Prompt) {
            System.err.print(object);
        } else {
            System.out.println(object);
        }
    }

    private void outputResult(ResultSetReference ref) {
        try {
            // ʃZbg
            ResultSet rs = ref.getResultSet();
            ColumnOrder order = ref.getOrder();
            ResultSetMetaData rsmeta = rs.getMetaData();
            final boolean needsOrderChange = order.size() > 0;
            System.err.println();
            // 
            final int columnCount = (needsOrderChange) ? order.size() : rsmeta.getColumnCount();
            int maxWidth = 1;
            StringBuilder borderFormat = new StringBuilder();
            for (int i = 0; i < columnCount; i++) {
                final int index = (needsOrderChange) ? order.getOrder(i) : i + 1;
                int size = rsmeta.getColumnDisplaySize(index);
                if (size > WIDTH_LIMIT) {
                    size = WIDTH_LIMIT;
                } else if (size < 1) {
                    size = 1;
                }
                maxWidth = Math.max(maxWidth, size);
                final int widthExpression;
                switch (rsmeta.getColumnType(index)) {
                    case Types.TINYINT:
                    case Types.SMALLINT:
                    case Types.INTEGER:
                    case Types.BIGINT:
                    case Types.REAL:
                    case Types.DOUBLE:
                    case Types.FLOAT:
                    case Types.DECIMAL:
                    case Types.NUMERIC:
                        widthExpression = size;
                        break;
                    default:
                        widthExpression = -size;
                }
                final String format = "%" + widthExpression + "s";
                borderFormat.append(" " + format);
                if (i != 0) {
                    System.out.print(' ');
                }
                final String name = (needsOrderChange) ? order.getName(i) : rsmeta.getColumnName(index);
                System.out.print(PrintFormat.format(format, name));
            }
            System.out.println();
            // E
            String format = borderFormat.substring(1);
            char[] borderChars = new char[maxWidth];
            Arrays.fill(borderChars, '-');
            Object[] borders = new String[columnCount];
            Arrays.fill(borders, String.valueOf(borderChars));
            System.out.println(PrintFormat.format(format, borders));
            // [vJn
            Object[] a = new Object[columnCount];
            final int limit = LocalSystem.getPropertyAsInt(PropertyKey.ROW_COUNT_LIMIT,
                                                           Integer.MAX_VALUE);
            int count = 0;
            while (rs.next()) {
                if (count >= limit) {
                    System.err.println(Resource.getString("w.exceeded-limit", limit));
                    break;
                }
                ++count;
                for (int i = 0; i < columnCount; i++) {
                    final int index = (needsOrderChange) ? order.getOrder(i) : i + 1;
                    a[i] = rs.getString(index);
                }
                System.out.println(PrintFormat.format(format, a));
            }
            System.out.println();
            // [vI
            ref.setRecordCount(count);
        } catch (SQLException ex) {
            ex.printStackTrace(System.err);
        }
    }

    /* @see net.argius.stew.OutputProcessor#close() */
    public void close() {
        // empty
    }

}
