/*
 *  Copyright 2010 argius
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */
package net.argius.stew;

import java.io.*;
import java.nio.channels.*;
import java.sql.*;
import java.util.*;

import net.argius.logging.*;
import net.argius.stew.ui.*;

/**
 * R}h@\B
 */
final class CommandProcessor {

    private static final Logger log = LoggerFactory.getLogger(CommandProcessor.class);

    private final Environment env;
    private final OutputProcessor op;

    CommandProcessor(Environment env) {
        this.env = env;
        this.op = env.getOutputProcessor();
    }

    /**
     * R}hNB
     * @param parameterString
     * @return 𑱍sꍇ <code>true</code>Afꍇ <code>false</code> 
     * @throws CommandException
     */
    boolean invoke(String parameterString) throws CommandException {
        Parameter p = new Parameter(parameterString);
        if (parameterString.indexOf("-e ") >= 0) {
            List<String> a = new ArrayList<String>();
            StringBuilder buffer = new StringBuilder();
            for (String s : p.asArray()) {
                if (s.equals("-e")) {
                    if (buffer.length() > 0) {
                        a.add(buffer.substring(1));
                        buffer.setLength(0);
                    } else {
                        // ܂̓R}h̖P[X
                    }
                } else {
                    buffer.append(' ');
                    buffer.append(s);
                }
            }
            if (buffer.length() > 0) {
                a.add(buffer.substring(1));
            }
            for (String commandString : a) {
                op.output("> " + commandString);
                if (!invoke(commandString)) {
                    outputMessage("w.exit-not-available-in-sequencial-command");
                }
            }
            return true;
        }
        final String commandName = p.get(1);
        try {
            return invoke(commandName, new Parameter(parameterString));
        } catch (UsageException ex) {
            outputMessage("e.usage", commandName, ex.getMessage());
        } catch (DynamicLoadingException ex) {
            log.error("", ex);
            outputMessage("e.not-found", commandName);
        } catch (CommandException ex) {
            log.error("", ex);
            Throwable cause = ex.getCause();
            String message = (cause == null) ? ex.getMessage() : cause.getMessage();
            outputMessage("e.command", message);
        } catch (IOException ex) {
            log.error("", ex);
            outputMessage("e.command", ex.getMessage());
        } catch (SQLException ex) {
            log.error("", ex);
            SQLException parent = ex;
            while (true) {
                SQLException sqle = parent.getNextException();
                if (sqle == null || sqle == parent) {
                    break;
                }
                log.error("------ SQLException.getNextException ------", sqle);
                parent = sqle;
            }
            outputMessage("e.database", ex.getMessage());
        } catch (UnsupportedOperationException ex) {
            log.warn("", ex);
            outputMessage("e.unsupported", ex.getMessage());
        } catch (RuntimeException ex) {
            log.error("", ex);
            outputMessage("e.runtime", ex.getMessage());
        } catch (Throwable th) {
            log.fatal("", th);
            outputMessage("e.fatal", th.getMessage());
        }
        try {
            Connection conn = env.getCurrentConnection();
            if (conn != null) {
                boolean isClosed = conn.isClosed();
                if (isClosed) {
                    if (log.isInfoEnabled()) {
                        log.info("connection is already closed");
                    }
                    disconnect();
                }
            }
        } catch (SQLException ex) {
            log.warn("", ex);
        }
        return true;
    }

    /**
     * R}hNB
     * @param commandName R}h
     * @param p p[^
     * @return 𑱍sꍇ <code>true</code>Afꍇ <code>false</code>
     * @throws IOException
     * @throws SQLException
     */
    private boolean invoke(String commandName, Parameter p) throws IOException, SQLException {
        assert commandName != null;
        // if blank, skip to invoke
        if (commandName.length() == 0) {
            return true;
        }
        // if exit
        if (commandName.equalsIgnoreCase("exit")) {
            disconnect();
            outputMessage("i.exit");
            return false;
        }
        // if connect 
        if (commandName.equalsIgnoreCase("connect") || commandName.equalsIgnoreCase("-c")) {
            connect(p);
            return true;
        }
        // from file
        if (commandName.equals("-f")) {
            ByteArrayOutputStream bos = new ByteArrayOutputStream();
            File file = new File(p.get(2));
            FileInputStream fis = new FileInputStream(file);
            try {
                fis.getChannel().transferTo(0, file.length(), Channels.newChannel(bos));
            } finally {
                fis.close();
            }
            invoke(new String(bos.toByteArray()));
            return true;
        }
        // cd
        if (commandName.equalsIgnoreCase("cd")) {
            if (p.isEmpty(2)) {
                throw new UsageException(Resource.getString("usage.cd"));
            }
            File olddir = env.getCurrentDirectory();
            File newdir = new File(p.getAll(2));
            if (!newdir.isDirectory()) {
                throw new CommandException("---");
            }
            env.setCurrentDirectory(newdir);
            outputMessage("i.directory-changed", olddir.getAbsolutePath(), newdir.getAbsolutePath());
            return true;
        }
        // at
        if (commandName.equals("@")) {
            op.output(String.format("current dir : %s", env.getCurrentDirectory().getAbsolutePath()));
            op.output(String.format("system  dir : %s", env.getSystemDirectory().getAbsolutePath()));
            return true;
        }
        // report -
        if (commandName.equals("-")) {
            return invoke("report -");
        }
        // connection
        Connection conn = env.getCurrentConnection();
        if (conn == null) {
            outputMessage("e.not-connect");
        } else if (commandName.equalsIgnoreCase("disconnect") || commandName.equalsIgnoreCase("-d")) {
            disconnect();
            outputMessage("i.disconnected");
        } else if (commandName.equalsIgnoreCase("commit")) {
            conn.commit();
            outputMessage("i.commited");
        } else if (commandName.equalsIgnoreCase("rollback")) {
            conn.rollback();
            outputMessage("i.rollbacked");
        } else {
            executeDynamicCommand(commandName, conn, p);
        }
        return true;
    }

    /**
     * ڑB
     * @param p p[^
     * @throws SQLException
     */
    private void connect(Parameter p) throws SQLException {
        if (log.isInfoEnabled()) {
            log.info("connect start");
        }
        disconnect();
        String id = p.get(2);
        Connector connector;
        if (!p.isEmpty(3)) {
            connector = AnonymousConnector.getConnector(id, p.get(3), p.get(4));
        } else if (id.indexOf('@') >= 0) {
            connector = AnonymousConnector.getConnector(id);
        } else {
            connector = env.getConnectorMap().getConnector(id);
        }
        if (connector != null) {
            Connection conn = connector.getConnection();
            try {
                if (connector.isReadOnly()) {
                    conn.setReadOnly(true);
                }
            } catch (RuntimeException ex) {
                log.warn("", ex);
            }
            boolean isAutoCommitAvailable;
            try {
                conn.setAutoCommit(false);
                isAutoCommitAvailable = conn.getAutoCommit();
            } catch (RuntimeException ex) {
                log.warn("", ex);
                isAutoCommitAvailable = false;
            }
            if (isAutoCommitAvailable) {
                outputMessage("w.auto-commit-not-available");
            }
            env.setCurrentConnection(conn);
            env.setCurrentConnector(connector);
            outputMessage("i.connected");
        } else {
            outputMessage("e.no-connector", id);
        }
        if (log.isInfoEnabled()) {
            log.info("connect end");
        }
    }

    /**
     * RlNVؒfB
     */
    private void disconnect() {
        if (log.isDebugEnabled()) {
            log.debug("disconnect start");
        }
        try {
            env.releaseConnection();
        } catch (SQLException ex) {
            outputMessage("w.connection-closed-abnormally");
        }
        if (log.isDebugEnabled()) {
            log.debug("disconnect end");
        }
    }

    /**
     * IR}hsB
     * @param commandName R}h
     * @param conn RlNV
     * @param p p[^
     */
    private void executeDynamicCommand(String commandName, Connection conn, Parameter p) {
        assert commandName != null && !commandName.contains(" ");
        StringBuilder buffer = new StringBuilder("net.argius.stew.command.");
        buffer.append(commandName.substring(0, 1).toUpperCase());
        buffer.append(commandName.substring(1).toLowerCase());
        final String fqcn = buffer.toString();
        Class<? extends Command> c;
        try {
            c = DynamicLoader.loadClass(fqcn);
        } catch (DynamicLoadingException ex) {
            c = Command.isSelect(commandName) ? Select.class : UpdateAndOthers.class;
        }
        Command command = DynamicLoader.newInstance(c);
        try {
            Connector connector = env.getCurrentConnector();
            if (connector.isReadOnly() && !command.isReadOnly()) {
                outputMessage("e.readonly");
                return;
            }
            command.setEnvironment(env);
            if (log.isInfoEnabled()) {
                log.info("command : " + command + " start");
            }
            if (log.isDebugEnabled()) {
                log.debug(p);
            }
            command.initialize();
            command.execute(conn, p);
        } finally {
            command.close();
        }
        if (log.isInfoEnabled()) {
            log.info("command : " + command + " end");
        }
    }

    /**
     * bZ[Wo͂B
     * @param id bZ[WID
     * @param args MessageFormatɓn
     * @throws CommandException
     */
    void outputMessage(String id, Object... args) throws CommandException {
        op.output(Resource.getString(id, args));
    }

    /**
     * SQLR}hB
     */
    abstract static class RawSQL extends Command {

        @Override
        public final void execute(Connection conn, Parameter parameter) throws CommandException {
            final String s = parameter.getAll();
            final int index = s.indexOf(';');
            final String sql;
            final String[] a;
            if (index >= 0) {
                sql = s.substring(0, index - 1);
                a = s.substring(index + 1).split(",");
            } else {
                sql = s;
                a = new String[0];
            }
            try {
                Statement stmt = prepareStatement(conn, sql, a);
                try {
                    setTimeout(stmt);
                    execute(stmt, sql);
                } finally {
                    stmt.close();
                }
            } catch (SQLException ex) {
                throw new CommandException(ex);
            }
        }

        private Statement prepareStatement(Connection conn, String sql, String[] a) throws SQLException {
            if (a.length == 0) {
                return conn.createStatement();
            }
            PreparedStatement pstmt = conn.prepareStatement(sql);
            try {
                int index = 0;
                for (String p : a) {
                    pstmt.setString(++index, p);
                }
            } catch (Throwable th) {
                try {
                    if (th instanceof SQLException) {
                        throw (SQLException)th;
                    }
                    throw new IllegalStateException(th);
                } finally {
                    pstmt.close();
                }
            }
            return pstmt;
        }

        protected abstract void execute(Statement stmt, String sql) throws SQLException;

    }

    /**
     * SELECTR}hB
     */
    static final class Select extends RawSQL {

        public Select() {
            // empty
        }

        @Override
        public boolean isReadOnly() {
            return true;
        }

        @Override
        public void execute(Statement stmt, String sql) throws SQLException {
            long time = System.currentTimeMillis();
            ResultSet rs = (stmt instanceof PreparedStatement)
                    ? ((PreparedStatement)stmt).executeQuery()
                    : stmt.executeQuery(sql);
            try {
                time = System.currentTimeMillis() - time;
                outputMessage("i.response-time", time / 1000f);
                ResultSetReference ref = new ResultSetReference(rs);
                output(ref);
                outputMessage("i.selected", ref.getRecordCount());
            } finally {
                rs.close();
            }
        }

    }

    /**
     * UPDATE(ȂSELECTȊO)R}hB
     */
    static final class UpdateAndOthers extends RawSQL {

        public UpdateAndOthers() {
            // empty
        }

        @Override
        public boolean isReadOnly() {
            return false;
        }

        @Override
        protected void execute(Statement stmt, String sql) throws SQLException {
            final int updatedCount = (stmt instanceof PreparedStatement)
                    ? ((PreparedStatement)stmt).executeUpdate()
                    : stmt.executeUpdate(sql);
            final String msgId;
            if (sql.matches("(?i)\\s*UPDATE.*")) {
                msgId = "i.updated";
            } else if (sql.matches("(?i)\\\\s*INSERT.*")) {
                msgId = "i.inserted";
            } else if (sql.matches("(?i)\\\\s*DELETE.*")) {
                msgId = "i.deleted";
            } else {
                msgId = "i.proceeded";
            }
            outputMessage(msgId, updatedCount);
        }

    }

}
