/*

 SQSToPDFTranslator.java

 Copyright 2004 KUBO Hiroya (hiroya@sfc.keio.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 Created on 2007/09/04

 */
package net.sqs2.translator.impl;

import java.awt.Point;
import java.awt.Rectangle;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.util.Map;

import net.sqs2.translator.TranslatorException;
import net.sqs2.util.FileUtil;
import net.sqs2.xml.XMLUtil;
import net.sqs2.xmlns.SQSNamespaces;

import org.apache.commons.io.output.ByteArrayOutputStream;
import org.apache.fop.apps.FOPException;
import org.apache.fop.apps.FOUserAgent;
import org.apache.fop.apps.Fop;
import org.apache.fop.render.pdf.PageRectangle;
import org.apache.fop.render.pdf.SVGElementIDToPageRectangleMap;
import org.w3c.dom.Element;

import com.lowagie.text.DocumentException;
import com.lowagie.text.pdf.PdfReader;
import com.lowagie.text.pdf.PdfStamper;

public class SQSToPDFTranslator extends FOPTranslator {
	private static final String VERSION = "1.3";
	private static final int WIDTH = 595;
	private static final int HEIGHT = 842;
	private static final Rectangle PDF_PAGE_RECTANGLE = new Rectangle(WIDTH, HEIGHT);

	private static final Point[] MASTER_CORNER_POINT_ARRAY = new Point[] { new Point(99, 29),
			new Point(497, 29), new Point(94, 810), new Point(492, 810) };

	public static final Rectangle UPSIDEDOWN_CHECK_AREA_HEADER = new Rectangle(89, 19, 20, 20);
	public static final Rectangle UPSIDEDOWN_CHECK_AREA_FOOTER = new Rectangle(84, 800, 20, 20);

	public static final Rectangle EVENODD_CHECK_AREA_LEFT = new Rectangle(30, 796, 24, 12);
	public static final Rectangle EVENODD_CHECK_AREA_RIGHT = new Rectangle(531, 796, 24, 12);

	private static final int HORIZONTAL_TRIM = 0;
	//private static final int VERTICAL_TRIM = 14;
	private static final int VERTICAL_TRIM = 0;
	
	// private File targetFile;
	String basename;

	public SQSToPDFTranslator(String groupID, String appID, String fopURL, String xsltURL, String language, String filename)
	throws TranslatorException {
		super(groupID, appID, fopURL, xsltURL, language);
		this.basename = FileUtil.getBasename(filename);
	}

	public SQSToPDFTranslator(String groupID, String appID, String fopURL, String xsltURL, String filename)
	throws TranslatorException {
		super(groupID, appID, fopURL, xsltURL);
		this.basename = FileUtil.getBasename(filename);
	}

	private String getBasename() {
		return basename;
	}

	private byte[] createSVGPrint(FOUserAgent userAgent, int numPages) {
		try {
			ByteArrayOutputStream out = new ByteArrayOutputStream();
			PrintWriter writer = new PrintWriter(new OutputStreamWriter(out, "UTF-8"));
			writer.println("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
			writer.println("<svg:svg ");
			writer.println(" xmlns=\"" + SQSNamespaces.SVG_URI + "\" ");
			writer.println(" xmlns:svg=\"" + SQSNamespaces.SVG_URI + "\" ");
			writer.println(" xmlns:sqs=\"" + SQSNamespaces.SQS2004_URI + "\" ");
			writer.println(" xmlns:xforms=\"" + SQSNamespaces.XFORMS_URI + "\" ");
			writer.println(" xmlns:master=\"" + SQSNamespaces.SQS2007MASTER_URI + "\" ");
			writer.print("svg:width=\"");
			writer.print(PDF_PAGE_RECTANGLE.width);
			writer.print("\" svg:height=\"");
			writer.print(PDF_PAGE_RECTANGLE.height);
			writer.println("\">");
			printPageSet(userAgent, numPages, writer);
			writer.println("</svg:svg>");
			writer.close();
			out.close();
			byte[] svgBytes = out.toByteArray();

			if (false) {
				ByteArrayInputStream svgInputStream = new ByteArrayInputStream(svgBytes);
				OutputStream sqmOutputStream = new BufferedOutputStream(new FileOutputStream("/tmp/sqs.sqm"));
				FileUtil.connect(svgInputStream, sqmOutputStream);
				svgInputStream.close();
				sqmOutputStream.close();
			}

			return svgBytes;
		} catch (IOException ex) {
			ex.printStackTrace();
		}
		return null;
	}

	@SuppressWarnings("unchecked")
	private void printPageSet(FOUserAgent userAgent, int numPages, PrintWriter writer) {
		writer.println("<svg:pageSet>");

		printMasterPage(numPages, writer);

		SVGElementIDToPageRectangleMap svgElementIDToPageRectangleMap = SVGElementIDToPageRectangleMap
				.getInstance();

		for (int pageIndex = 0; pageIndex < numPages; pageIndex++) {
			writer.println("  <svg:page>");
			Map<String, PageRectangle> map = svgElementIDToPageRectangleMap.remove(userAgent, pageIndex);
			if (map != null) {
				for (Map.Entry<String, PageRectangle> entry : map.entrySet()) {
					String id = entry.getKey();
					PageRectangle pageRectangle = entry.getValue();
					pageIndex = pageRectangle.getPageIndex();
					printGElement(id, pageRectangle, writer);
				}
			}
			writer.println("  </svg:page>");
		}

		writer.println("</svg:pageSet>");
	}

	private static final float SCALE = 1.0f;

	private void printGElement(String id, PageRectangle area, PrintWriter writer) {
		writer.print("<svg:g id=\"");
		writer.print(id);
		writer.println("\">");
		writer.print("<svg:rect x=\"");
		writer.print(area.getX() / SCALE + HORIZONTAL_TRIM);
		writer.print("\" y=\"");
		writer.print(HEIGHT - area.getY() / SCALE + VERTICAL_TRIM); //
		writer.print("\" width=\"");
		writer.print(area.getWidth() / SCALE);
		writer.print("\" height=\"");
		writer.print(area.getHeight() / SCALE);
		writer.println("\">");
		writer.println(XMLUtil.createString((Element) area.getSVGMetadataNode()));
		writer.println("</svg:rect>");
		writer.println("</svg:g>");
	}

	private void printMasterPage(int numPages, PrintWriter writer) {
		writer.println(" <svg:masterPage>");
		writer.println("  <svg:metadata>");
		writer.print("      <master:master master:version=\"" + VERSION + "\" master:numPages=\"");
		writer.print(numPages);
		writer.println("\" />");
		printMasterCornerElement(writer);
		printUpsideDownChecker(writer);
		printEvenOddChecker(writer);
		writer.println("  </svg:metadata>");
		writer.println(" </svg:masterPage>");
	}

	private void printEvenOddChecker(PrintWriter writer) {
		writer.println("      <master:evenOddChecker>");
		writer.println("       <master:checkerArea master:side=\"left\">");
		writer.print("         <svg:rect");
		writer.print(" x=\"");
		writer.print(EVENODD_CHECK_AREA_LEFT.x);
		writer.print("\" y=\"");
		writer.print(EVENODD_CHECK_AREA_LEFT.y);
		writer.print("\" width=\"");
		writer.print(EVENODD_CHECK_AREA_LEFT.width);
		writer.print("\" height=\"");
		writer.print(EVENODD_CHECK_AREA_LEFT.height);
		writer.println("\"/>");
		writer.println("     </master:checkerArea>");
		writer.println("     <master:checkerArea master:side=\"right\">");
		writer.print("         <svg:rect");
		writer.print(" x=\"");
		writer.print(EVENODD_CHECK_AREA_RIGHT.x);
		writer.print("\" y=\"");
		writer.print(EVENODD_CHECK_AREA_RIGHT.y);
		writer.print("\" width=\"");
		writer.print(EVENODD_CHECK_AREA_RIGHT.width);
		writer.print("\" height=\"");
		writer.print(EVENODD_CHECK_AREA_RIGHT.height);
		writer.println("\"/>");
		writer.println("     </master:checkerArea>");
		writer.println("     </master:evenOddChecker>");
	}

	private void printUpsideDownChecker(PrintWriter writer) {
		writer.println("      <master:upsideDownChecker>");
		writer.println("       <master:checkerArea master:side=\"header\">");
		writer.print("         <svg:rect");
		writer.print(" x=\"");
		writer.print(UPSIDEDOWN_CHECK_AREA_HEADER.x);
		writer.print("\" y=\"");
		writer.print(UPSIDEDOWN_CHECK_AREA_HEADER.y);
		writer.print("\" width=\"");
		writer.print(UPSIDEDOWN_CHECK_AREA_HEADER.width);
		writer.print("\" height=\"");
		writer.print(UPSIDEDOWN_CHECK_AREA_HEADER.height);
		writer.println("\"/>");
		writer.println("     </master:checkerArea>");
		writer.println("     <master:checkerArea master:side=\"footer\">");
		writer.print("         <svg:rect");
		writer.print(" x=\"");
		writer.print(UPSIDEDOWN_CHECK_AREA_FOOTER.x);
		writer.print("\" y=\"");
		writer.print(UPSIDEDOWN_CHECK_AREA_FOOTER.y);
		writer.print("\" width=\"");
		writer.print(UPSIDEDOWN_CHECK_AREA_FOOTER.width);
		writer.print("\" height=\"");
		writer.print(UPSIDEDOWN_CHECK_AREA_FOOTER.height);
		writer.println("\"/>");
		writer.println("     </master:checkerArea>");
		writer.println("     </master:upsideDownChecker>");
	}

	private void printMasterCornerElement(PrintWriter writer) {
		writer.print("      <master:corner master:x1=\"");
		writer.print(MASTER_CORNER_POINT_ARRAY[0].x);
		writer.print("\" master:y1=\"");
		writer.print(MASTER_CORNER_POINT_ARRAY[0].y);
		writer.print("\" master:x2=\"");
		writer.print(MASTER_CORNER_POINT_ARRAY[1].x);
		writer.print("\" master:y2=\"");
		writer.print(MASTER_CORNER_POINT_ARRAY[1].y);
		writer.print("\" master:x3=\"");
		writer.print(MASTER_CORNER_POINT_ARRAY[2].x);
		writer.print("\" master:y3=\"");
		writer.print(MASTER_CORNER_POINT_ARRAY[2].y);
		writer.print("\" master:x4=\"");
		writer.print(MASTER_CORNER_POINT_ARRAY[3].x);
		writer.print("\" master:y4=\"");
		writer.print(MASTER_CORNER_POINT_ARRAY[3].y);
		writer.println("\" />");
	}
	
	synchronized protected void translate(byte[] sqsSourceBytes, ByteArrayInputStream foInputStream, String systemId, OutputStream pdfOutputStream) throws TranslatorException {
		try {
			ByteArrayOutputStream pdfRawDataOutputStream = new ByteArrayOutputStream(65536);
			if (this.core == null) {
				this.core = createCore(this.language);
			}

			FOUserAgent userAgent = core.getUserAgent();
			userAgent.setBaseURL(systemId);

			synchronized (SQSToPDFTranslator.class) {
				Fop fop = core.createFop(pdfRawDataOutputStream);
				render(fop.getDefaultHandler(), foInputStream, systemId);
			}

			pdfRawDataOutputStream.flush();
			byte[] pdfRawDataBytes = pdfRawDataOutputStream.toByteArray();

			foInputStream.close();
			foInputStream = null;
			combinePDFData(userAgent, sqsSourceBytes, pdfRawDataBytes, pdfOutputStream, getBasename());

			pdfRawDataOutputStream.close();
			pdfRawDataOutputStream = null;
			pdfOutputStream.flush();

		} catch (FOPException ex) {
			ex.printStackTrace();
			throw new TranslatorException(ex);
		} catch (IOException ex) {
			ex.printStackTrace();
			throw new TranslatorException(ex);
		}
	}

	private void combinePDFData(FOUserAgent userAgent, byte[] sqsSourceBytes, byte[] pdfRawDataBytes, OutputStream pdfOutputStream, String basename) throws IOException {
		try {
			PdfReader reader = new PdfReader(pdfRawDataBytes);
			int numPages = reader.getNumberOfPages();
			byte[] svgBytes = createSVGPrint(userAgent, numPages);
			PdfStamper stamp = new PdfStamper(reader, pdfOutputStream);
			stamp.addFileAttachment("SQS Source", sqsSourceBytes, null, basename + ".sqs");
			stamp.addFileAttachment("SQS Master", svgBytes, null, basename + ".sqm");
			stamp.close();
			reader.close();
		} catch (DocumentException ex) {
			ex.printStackTrace();
		}
	}
}
