package net.sqs2.translator.impl;

import java.awt.Container;

import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Date;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.logging.Logger;

import javax.swing.JFrame;
import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.URIResolver;
import javax.xml.transform.sax.SAXResult;
import javax.xml.transform.stream.StreamSource;

import net.sqs2.net.ClassURIResolver;
import net.sqs2.net.ClassURLConnection;
import net.sqs2.net.ClassURLStreamHandlerFactory;
import net.sqs2.translator.AbstractTranslator;
import net.sqs2.translator.TranslatorException;
import net.sqs2.translator.XSLTranslator;
import net.sqs2.util.FileUtil;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.DefaultConfigurationBuilder;
import org.apache.commons.io.output.ByteArrayOutputStream;
import org.apache.fop.apps.FOPException;
import org.apache.fop.apps.FOUserAgent;
import org.apache.fop.apps.Fop;
import org.apache.fop.apps.FopFactory;
import org.apache.fop.apps.MimeConstants;
import org.apache.fop.render.awt.AWTRenderer;
import org.apache.fop.render.awt.viewer.PreviewPanel;
import org.apache.fop.render.awt.viewer.Renderable;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

public abstract class FOPTranslator extends AbstractTranslator {

	static {
		try {
			URL.setURLStreamHandlerFactory(new ClassURLStreamHandlerFactory());
		} catch (Error ignore) {
		}
	}

	private static final String[] SQS2FO = { "cmpl-label.xsl", "cmpl-ref.xsl", "embed-counter.xsl",
			"embed-link.xsl", "convert1.xsl", "convert2.xsl", "convert3.xsl" };

	private final static boolean FOP_OUTPUT_DEBUG = true;

	private final static URIResolver uriResolver = new ClassURIResolver();
	protected SQSToPDFTranslatorCore core = null;

	protected SQSToPDFTranslatorCore createCore(String language) {
		try {
			String userCustomizedXSLTFileBaseURIString = XSLTFileBaseUtil.userCustomizedURI(this.groupID, this.appID);
			return new SQSToPDFTranslatorCore(this.xsltBaseURL, userCustomizedXSLTFileBaseURIString, language);
		} catch (TranslatorException ex) {
			ex.printStackTrace();
		}

		return null;
	}

	private String groupID = null;
	private String appID = null;
	private String fopURL = null;
	private String xsltBaseURL = null;
	protected String language = null;
	
	public FOPTranslator(String groupID, String appID, String fopURL, String xsltBaseURL)
	throws TranslatorException{
		this(groupID, appID, fopURL, xsltBaseURL, Locale.getDefault().getLanguage());
	}
	
	public FOPTranslator(String groupID, String appID, String fopURL, String xsltBaseURL, String language)
	throws TranslatorException {
		super();
		this.groupID = groupID;
		this.appID = appID;
		this.fopURL = fopURL;
		this.xsltBaseURL = xsltBaseURL;
		this.language = language;
	}

	static class PreviewFrame extends JFrame {
		private static final long serialVersionUID = 0L;
		private FOUserAgent userAgent;
		private PreviewPanel previewPanel;

		PreviewFrame(FOUserAgent userAgent) {
			this.setSize(400, 550);
			this.userAgent = userAgent;
		}

		void update(String title, Renderable renderable, AWTRenderer renderer) {
			Container contentPane = this.getContentPane();
			if (0 < contentPane.getComponentCount()) {
				contentPane.remove(0);
			}
			this.previewPanel = new PreviewPanel(this.userAgent, renderable, renderer);
			contentPane.add(this.previewPanel);
			this.setTitle(title);
		}

		void render(Renderable renderable, AWTRenderer renderer) throws FOPException {
			renderer.setRenderable(renderable);
			renderer.clearViewportList();
			renderable.renderTo(this.userAgent, MimeConstants.MIME_FOP_AWT_PREVIEW);
		}

		void show(AWTRenderer renderer) throws FOPException {
			double scale = this.previewPanel.getScaleToFitWindow();
			this.previewPanel.setDisplayMode(PreviewPanel.CONTINUOUS);
			this.previewPanel.setScaleFactor(scale);
			renderer.clearViewportList();
			this.previewPanel.reload();
		}
	}

	class SQSToPDFTranslatorCore {

		// private String baseURI;
		private XSLTranslator xslTranslator;

		private FopFactory fopFactory;
		private FOUserAgent userAgent;
		private PreviewFrame previewFrame;

		private String userCustomizedXSLTFileBaseURI;

		public SQSToPDFTranslatorCore(String baseURI, String userCostomizedXSLTFileBaseURI, String language)
				throws TranslatorException {
			try {
				// this.baseURI = baseURI;
				this.userCustomizedXSLTFileBaseURI = userCostomizedXSLTFileBaseURI;
				this.xslTranslator = new XSLTranslator();
				this.xslTranslator.initialize(new String[] { userCustomizedXSLTFileBaseURI, baseURI },
						SQS2FO, getParameterArrayMap());
				try {
					this.fopFactory = FopFactory.newInstance();

					this.fopFactory.setURIResolver(uriResolver);
					this.fopFactory.setFontBaseURL("class://IPAFont/font/");

					DefaultConfigurationBuilder cfgBuilder = new DefaultConfigurationBuilder();
					String configURL = createConfigURL(language);
					Configuration cfg = cfgBuilder.build(new ClassURLConnection(new URL(configURL))
							.getInputStream());
					this.fopFactory.setUserConfig(cfg);
				} catch (MalformedURLException ex) {
					String configURL = createConfigURL(language);
					Logger.getAnonymousLogger().severe("ERROR not found:"+configURL);
					ex.printStackTrace();
				} catch (SAXException ex) {
					ex.printStackTrace();
				} catch (IOException ex) {
					ex.printStackTrace();
				}

				this.userAgent = createFOUserAgent();

			} catch (Exception ex) {
				ex.printStackTrace();
				throw new TranslatorException(ex);
			}
		}
		
		private String createConfigURL(String language){
			return fopURL + "userconfig_" + language + ".xml";
		}

		FOUserAgent getUserAgent() {
			return this.userAgent;
		}

		AWTRenderer createAWTRenderer() {
			AWTRenderer renderer = new AWTRenderer();
			renderer.setPreviewDialogDisplayed(false);
			renderer.setUserAgent(this.userAgent);
			this.userAgent.setRendererOverride(renderer);
			return renderer;
		}

		private Map<String, Entry[]> getParameterArrayMap() {
			Map<String, Entry[]> ret = new HashMap<String, Entry[]>();
			ret.put("embed-counter.xsl", new Entry[] { new Entry("xhtml.h-attribute..sqs.prefix", "問"),
					new Entry("xhtml.h-attribute..sqs.suffix", "."),
					new Entry("xhtml.h-attribute..sqs.format", "1"),
					new Entry("sqs.counter-attribute..sqs.prefix", "("),
					new Entry("sqs.counter-attribute..sqs.suffix", ")"),
					new Entry("sqs.counter-attribute..sqs.format", "1") });

			ret.put("convert2.xsl", new Entry[] { new Entry("xforms.hint-attribute..sqs.prefix", ""),
					new Entry("xforms.hint-attribute..sqs.suffix", ""),
					new Entry("xforms.hint-attribute..sqs.display", "inline"),
					new Entry("xforms.help-attribute..sqs.prefix", "("),
					new Entry("xforms.help-attribute..sqs.suffix", ")"),
					new Entry("xforms.help-attribute..sqs.display", "inline"),
					new Entry("xforms.alart-attribute..sqs.prefix", "*"),
					new Entry("xforms.alart-attribute..sqs.suffix", ""),
					new Entry("xforms.alart-attribute..sqs.display", "inline") });

			ret.put("xhtmlToBookmark.xsl ", new Entry[] { new Entry("bookmark-root-label", "調査票の構造"),
					new Entry("bookmark-section-label", "問"), new Entry("bookmark-question-label", "設問"),
					new Entry("bookmark-itemset-label", "選択肢") });

			ret.put("convert3.xsl", new Entry[] { new Entry("example-blank-mark-label", " : 空白マーク"),
					new Entry("language", language),
					new Entry("example-filled-mark-label", ": 正しいぬりつぶし"),
					new Entry("example-incomplete-mark-label", ": 不十分なぬりつぶし"),
					new Entry("characters-prohibit-line-break", "。．、，’”）｝」』〕】〉》々〜…ーぁぃぅぇぉっゃゅょゎァィゥェォッャュョヮ)'")
			/*
			 * “‘（｛「『〔【〈《(￥＄
			 */
			});
			return ret;
		}

		private FOUserAgent createFOUserAgent() {
			FOUserAgent userAgent;
			userAgent = this.fopFactory.newFOUserAgent();
			userAgent.setProducer("SQS SourceEditor2.0");
			userAgent.setCreator("");// TODO FOP: CreatorInfo from sqs document
										// xpath /html/head/meta
			userAgent.setAuthor("");// TODO FOP: AuthorInfo from sqs document
									// xpath /html/head/meta
			userAgent.setCreationDate(new Date());
			userAgent.setTitle(""); // TODO FOP: Title from sqs document xpath
									// /html/head/title
			userAgent.setKeywords("SQS XML XSL-FO");
			userAgent.setURIResolver(uriResolver);
			return userAgent;
		}

		PreviewFrame getPreviewFrame(String title, Renderable renderable, AWTRenderer renderer) {
			if (this.previewFrame == null) {
				this.previewFrame = new PreviewFrame(this.userAgent);
			}
			this.previewFrame.update(title, renderable, renderer);
			return previewFrame;
		}

		Fop createFop(FOUserAgent userAgent, String outputFormat) throws FOPException {
			return this.fopFactory.newFop(outputFormat, userAgent);
		}

		Fop createFop(OutputStream pdfOutputStream) throws FOPException {
			return this.fopFactory.newFop(MimeConstants.MIME_PDF, this.userAgent, pdfOutputStream);
		}

		Fop createFopAreaTree(OutputStream areaTreeOutputStream) throws FOPException {
			return this.fopFactory.newFop(MimeConstants.MIME_FOP_AREA_TREE, this.userAgent,
					areaTreeOutputStream);
		}

		void execute(InputStream sqsInputStream, String systemId, OutputStream foOutputStream) throws TranslatorException {
			this.xslTranslator.execute(sqsInputStream, systemId, foOutputStream);
		}
	}
	
	public void setCanceled(boolean isCanceled){
		if (this.core != null) {
			//this.core.getUserAgent().setCanceled(isCanceled);	
			try{
				this.core.getUserAgent().getRendererOverride().stopRenderer();
			}catch(IOException ignore){
				ignore.printStackTrace();
			}
		}
	}

	synchronized private byte[] createFOBytes(byte[] sourceBytes, String systemId, String language) throws TranslatorException, IOException {
		ByteArrayInputStream sqsInputStream = new ByteArrayInputStream(sourceBytes);
		ByteArrayOutputStream foOutputStream = new ByteArrayOutputStream(65536);
		if (this.core == null) {
			this.core = createCore(language);
		}
		this.core.execute(sqsInputStream, systemId, foOutputStream);
		sqsInputStream.close();
		sqsInputStream = null;

		foOutputStream.flush();
		byte[] foBytes = foOutputStream.toByteArray();
		foOutputStream.close();
		foOutputStream = null;
		return foBytes;
	}

	private byte[] createSourceBytes(InputStream sourceInputStream) throws IOException {
		ByteArrayOutputStream sourceOutputStream = new ByteArrayOutputStream(4096);

		FileUtil.pipe(sourceInputStream, sourceOutputStream);
		sourceInputStream.close();
		sourceInputStream = null;

		sourceOutputStream.flush();
		byte[] sourceBytes = sourceOutputStream.toByteArray();
		sourceOutputStream.close();
		sourceOutputStream = null;
		return sourceBytes;
	}

	@Override
	public void execute(InputStream sqsSourceInputStream, String systemId, OutputStream pdfOutputStream) throws TranslatorException {
		try {

			byte[] sqsSourceBytes = createSourceBytes(sqsSourceInputStream);
			byte[] foBytes = createFOBytes(sqsSourceBytes, systemId, this.language);
			ByteArrayInputStream foInputStream = new ByteArrayInputStream(foBytes);

			if (FOP_OUTPUT_DEBUG) {
				File tmpFile = File.createTempFile("sqs-", ".fo");
				tmpFile.deleteOnExit();
				OutputStream foTest = new BufferedOutputStream(new FileOutputStream(tmpFile));
				FileUtil.connect(foInputStream, foTest);
				foTest.close();
				foInputStream.reset();
			}

			translate(sqsSourceBytes, foInputStream, systemId, pdfOutputStream);

		} catch (IOException ex) {
			ex.printStackTrace();
			throw new TranslatorException(ex);
		}
	}

	protected abstract void translate(byte[] sqsSourceBytes, ByteArrayInputStream foInputStream, String systemId, OutputStream pdfOutputStream) throws TranslatorException;

	protected void render(ContentHandler handler, InputStream foInputStream, String systemId) throws TranslatorException, IOException {
		try {
			// System.out.println("pdfRawDataBytes:"+ pdfRawDataBytes.length);

			// Setup JAXP using identity transformer
			TransformerFactory factory = TransformerFactory.newInstance();
			Transformer transformer = factory.newTransformer();

			// Setup input stream
			Source foInputSource = new StreamSource(foInputStream, systemId);

			// Resulting SAX events (the generated FO) must be piped through to
			// FOP
			Result res = new SAXResult(handler);
			// Start XSLT transformation and FO processing
			transformer.transform(foInputSource, res);

		} catch (TransformerException ex) {
			ex.printStackTrace();
			throw new TranslatorException(ex);
		}
	}

}
