package org.sqlite;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.sql.SQLException;
import java.util.Arrays;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.sqlite.swig.SQLite3.SQLite3StmtPtrPtr;
import static org.sqlite.swig.SQLite3Constants.*;
import static org.junit.Assert.*;

/**
 *
 * @author calico
 */
public class StatementTest {

    public StatementTest() {
    }

    @BeforeClass
    public static void setUpClass() throws Exception {
    }

    @AfterClass
    public static void tearDownClass() throws Exception {
    }

    @Before
    public void setUp() {
    }

    @After
    public void tearDown() {
    }

    private static final String DATABASE = System.getProperty("user.dir") + "/test/unittest.db";
    
    private static Database newDatabase() throws ClassNotFoundException, SQLException {
        Class.forName("org.sqlite.Driver");
        return new Database(DATABASE, null);
    }
    
    private static Statement newStatement(Database db, String sql) throws ClassNotFoundException, SQLException {
        return db.prepare(sql);
    }
    
    private static Statement newStatement(Database db, String sql, SQLite3StmtPtrPtr ppStmt) throws ClassNotFoundException, SQLException {
        return db.prepare(sql, ppStmt);
    }
    
    @Test(expected = java.sql.SQLException.class)
    public void getParameterCountThrowSQLException() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT * FROM sqlite_master";
        stmt = newStatement(db, sql);
        stmt.close();
        stmt.getParameterCount();
    
        db.close();
    }
    
    @Test
    public void getParameterCount() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT * FROM sqlite_master";
        stmt = newStatement(db, sql);
        assertEquals(0, stmt.getParameterCount());
        stmt.close();
        
        sql = "SELECT * FROM sqlite_master WHERE ?1 = ?2 AND ?3 = ?4";
        stmt = newStatement(db, sql);
        assertEquals(4, stmt.getParameterCount());
        stmt.close();
        
        sql = "SELECT @col1, @col2, @col3 LIMIT 0";
        stmt = newStatement(db, sql);
        assertEquals(3, stmt.getParameterCount());
        stmt.close();
        
        sql = "SELECT :col1, :col2, :col3, :col4, :col5, :col6 LIMIT 0";
        stmt = newStatement(db, sql);
        assertEquals(6, stmt.getParameterCount());
        stmt.close();

        sql = "SELECT ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ? LIMIT 0";
        stmt = newStatement(db, sql);
        assertEquals(20, stmt.getParameterCount());
        stmt.close();
        
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getParameterIndexThrowSQLException() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT * FROM sqlite_master WHERE type = @type";
        stmt = newStatement(db, sql);
        stmt.close();
        stmt.getParameterIndex("@type");
    
        db.close();
    }
    
    @Test
    public void getParameterIndex() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT @col1, @col2, @col3 LIMIT 0";
        stmt = newStatement(db, sql);
        assertEquals(1, stmt.getParameterIndex("@col1"));
        assertEquals(2, stmt.getParameterIndex("@col2"));
        assertEquals(3, stmt.getParameterIndex("@col3"));
        assertEquals(0, stmt.getParameterIndex("@col4"));
        stmt.close();
        
        sql = "SELECT :col1, :col2, :col3, :col4, :col5, :col6 LIMIT 0";
        stmt = newStatement(db, sql);
        assertEquals(1, stmt.getParameterIndex(":col1"));
        assertEquals(2, stmt.getParameterIndex(":col2"));
        assertEquals(3, stmt.getParameterIndex(":col3"));
        assertEquals(4, stmt.getParameterIndex(":col4"));
        assertEquals(5, stmt.getParameterIndex(":col5"));
        assertEquals(6, stmt.getParameterIndex(":col6"));
        assertEquals(0, stmt.getParameterIndex(":col7"));
        stmt.close();

        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getParameterNameThrowSQLException() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT * FROM sqlite_master WHERE type = @type";
        stmt = newStatement(db, sql);
        stmt.close();
        assertEquals("@type", stmt.getParameterName(1));
    
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getParameterNameOutOfRange() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT * FROM sqlite_master";
        stmt = newStatement(db, sql);
        stmt.close();
        stmt.getParameterName(0);
    
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getParameterNameOutOfRange2() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT * FROM sqlite_master WHERE type = @type AND name = @name";
        stmt = newStatement(db, sql);
        stmt.close();
        assertEquals("@type", stmt.getParameterName(1));
        assertEquals("@name", stmt.getParameterName(2));
        assertEquals("", stmt.getParameterName(3));
    
        db.close();
    }
    
    @Test
    public void getParameterName() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT @col1, @col2, @col3 LIMIT 0";
        stmt = newStatement(db, sql);
        assertEquals("@col1", stmt.getParameterName(1));
        assertEquals("@col2", stmt.getParameterName(2));
        assertEquals("@col3", stmt.getParameterName(3));
        stmt.close();
        
        sql = "SELECT :col1, :col2, :col3, :col4, :col5, :col6 LIMIT 0";
        stmt = newStatement(db, sql);
        assertEquals(":col1", stmt.getParameterName(1));
        assertEquals(":col2", stmt.getParameterName(2));
        assertEquals(":col3", stmt.getParameterName(3));
        assertEquals(":col4", stmt.getParameterName(4));
        assertEquals(":col5", stmt.getParameterName(5));
        assertEquals(":col6", stmt.getParameterName(6));
        stmt.close();

        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getColumnCountThrowSQLException() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT type, name, sql FROM sqlite_master";
        stmt = newStatement(db, sql);
        stmt.close();
        assertEquals(3, stmt.getColumnCount());
    
        db.close();
    }
    
    @Test
    public void getColumnCount() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT type, name, sql FROM sqlite_master";
        stmt = newStatement(db, sql);
        assertEquals(3, stmt.getColumnCount());
        stmt.close();
        
        sql = "SELECT @col1, @col2, @col3 LIMIT 0";
        stmt = newStatement(db, sql);
        assertEquals(3, stmt.getColumnCount());
        stmt.close();
        
        sql = "SELECT :col1, :col2, :col3, :col4, :col5, :col6 LIMIT 0";
        stmt = newStatement(db, sql);
        assertEquals(6, stmt.getColumnCount());
        stmt.close();

        sql = "SELECT ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ? LIMIT 0";
        stmt = newStatement(db, sql);
        assertEquals(20, stmt.getColumnCount());
        stmt.close();
        
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getDataCountThrowSQLException() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT type, name, sql FROM sqlite_master";
        stmt = newStatement(db, sql);
        stmt.close();
        assertEquals(3, stmt.getDataCount());
    
        db.close();
    }
    
    @Test
    public void getDataCount() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;
        sql = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_1("
                + "  ID INTEGER PRIMARY KEY AUTOINCREMENT"
                + ", tVALUE TEXT DEFAULT '_blank'"
            + ")";
        stmt = newStatement(db, sql);
        // SQLITE_DONE
        assertEquals(SQLITE_DONE, stmt.step());
        stmt.close();

        sql = "SELECT type, name, sql FROM sqlite_temp_master";
        stmt = newStatement(db, sql);
        assertEquals(0, stmt.getDataCount());
        stmt.step();
        assertEquals(3, stmt.getDataCount());        
        stmt.close();
        
        sql = "SELECT 1, '2', X'03' LIMIT 0";
        stmt = newStatement(db, sql);
        assertEquals(0, stmt.getDataCount());
        // SQLITE_DONE
        assertEquals(SQLITE_DONE, stmt.step());
        assertEquals(0, stmt.getDataCount());
        stmt.close();
        
        sql = "SELECT 1, '2', X'03', 4.0, '5' LIMIT 1";
        stmt = newStatement(db, sql);
        assertEquals(0, stmt.getDataCount());
        // SQLITE_ROW
        assertEquals(SQLITE_ROW, stmt.step());
        assertEquals(5, stmt.getDataCount());
        // SQLITE_DONE
        assertEquals(SQLITE_DONE, stmt.step());
        assertEquals(0, stmt.getDataCount());
        stmt.close();
        
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getColumnLabelThrowSQLException() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT type, name, sql FROM sqlite_master";
        stmt = newStatement(db, sql);
        stmt.close();
        assertEquals("type", stmt.getColumnLabel(1));
        assertEquals("name", stmt.getColumnLabel(2));
        assertEquals("sql", stmt.getColumnLabel(3));
    
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getColumnLabelOutOfRange() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT type, name, sql FROM sqlite_master";
        stmt = newStatement(db, sql);
        assertEquals("type", stmt.getColumnLabel(0));
        stmt.close();
    
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getColumnLabelOutOfRange2() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT type, name, sql FROM sqlite_master";
        stmt = newStatement(db, sql);
        assertEquals("", stmt.getColumnLabel(4));
        stmt.close();
    
        db.close();
    }
    
    @Test
    public void getColumnLabel() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT type, name, sql FROM sqlite_master";
        stmt = newStatement(db, sql);
        assertEquals("type", stmt.getColumnLabel(1));
        assertEquals("name", stmt.getColumnLabel(2));
        assertEquals("sql", stmt.getColumnLabel(3));
        stmt.close();

        sql = "SELECT type AS TYPE_NAME, name AS TABLE_NAME, sql AS DDL FROM sqlite_master";
        stmt = newStatement(db, sql);
        assertEquals("TYPE_NAME", stmt.getColumnLabel(1));
        assertEquals("TABLE_NAME", stmt.getColumnLabel(2));
        assertEquals("DDL", stmt.getColumnLabel(3));
        stmt.close();
        
        sql = "SELECT 1, '2', X'03' LIMIT 0";
        stmt = newStatement(db, sql);
        assertEquals("1", stmt.getColumnLabel(1));
        assertEquals("'2'", stmt.getColumnLabel(2));
        assertEquals("X'03'", stmt.getColumnLabel(3));
        stmt.close();
        
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getColumnNameThrowSQLException() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT type, name, sql FROM sqlite_master";
        stmt = newStatement(db, sql);
        stmt.close();
        assertEquals("type", stmt.getColumnName(1));
        assertEquals("name", stmt.getColumnName(2));
        assertEquals("sql", stmt.getColumnName(3));
    
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getColumnNameOutOfRange() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT type, name, sql FROM sqlite_master";
        stmt = newStatement(db, sql);
        assertEquals("type", stmt.getColumnName(0));
        stmt.close();
    
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getColumnNameOutOfRange2() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT type, name, sql FROM sqlite_master";
        stmt = newStatement(db, sql);
        assertEquals("", stmt.getColumnName(4));
        stmt.close();
    
        db.close();
    }
    
    @Test
    public void getColumnName() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT type, name, sql FROM sqlite_master";
        stmt = newStatement(db, sql);
        assertEquals("type", stmt.getColumnName(1));
        assertEquals("name", stmt.getColumnName(2));
        assertEquals("sql", stmt.getColumnName(3));
        stmt.close();

        sql = "SELECT type AS TYPE_NAME, name AS TABLE_NAME, sql AS DDL FROM sqlite_master";
        stmt = newStatement(db, sql);
        assertEquals("type", stmt.getColumnName(1));
        assertEquals("name", stmt.getColumnName(2));
        assertEquals("sql", stmt.getColumnName(3));
        stmt.close();
        
        sql = "SELECT 1, '2', X'03' LIMIT 0";
        stmt = newStatement(db, sql);
        assertEquals(SQLITE_DONE, stmt.step());
        assertNull(stmt.getColumnName(1));
        assertNull(stmt.getColumnName(2));
        assertNull(stmt.getColumnName(3));
        stmt.close();
        
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getColumnTypeThrowSQLException() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT type, name, sql FROM sqlite_master";
        stmt = newStatement(db, sql);
        stmt.close();
        assertEquals(SQLITE3_TEXT, stmt.getColumnType(1));
        assertEquals(SQLITE3_TEXT, stmt.getColumnType(2));
        assertEquals(SQLITE3_TEXT, stmt.getColumnType(3));
    
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getColumnTypeOutOfRange() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT type, name, sql FROM sqlite_master";
        stmt = newStatement(db, sql);
        assertEquals(0, stmt.getColumnType(0));
        stmt.close();
    
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getColumnTypeOutOfRange2() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT type, name, sql FROM sqlite_master";
        stmt = newStatement(db, sql);
        assertEquals(0, stmt.getColumnType(4));
        stmt.close();
    
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getColumnTypeOutOfRange3() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT type, name, sql FROM sqlite_master";
        stmt = newStatement(db, sql);
        assertEquals(0, stmt.getColumnType(1));
        stmt.close();
    
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getColumnTypeOutOfRange4() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT type, name, sql FROM sqlite_master";
        stmt = newStatement(db, sql);
        while (stmt.step() == SQLITE_ROW) {
            // skip;
        }
        assertEquals(0, stmt.getColumnType(2));
        stmt.close();
    
        db.close();
    }
    
    @Test
    public void getColumnType() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;
        int ret = 0;
        
        sql = "CREATE TEMPORARY TABLE IF NOT EXISTS tbl_column_type_test("
                + "  ID INTEGER PRIMARY KEY AUTOINCREMENT"
                + ", tVAL TEXT"
                + ", rVAL REAL"
                + ", fVAL FLOAT"
                + ", bVAL BLOB"
                + ", nVAL NULL"
            + ")";
        db.execute(sql);
        sql = "INSERT INTO tbl_column_type_test "
            + "VALUES(1, 'ねこび～ん', 3.141529, 123.456789, X'31323334353637383900', NULL)";
        db.execute(sql);

        sql = "SELECT * FROM tbl_column_type_test";
        stmt = newStatement(db, sql);
        ret = stmt.step();
        assertEquals(SQLITE_ROW, ret);
        assertEquals(SQLITE_INTEGER, stmt.getColumnType(1));
        assertEquals(SQLITE3_TEXT, stmt.getColumnType(2));
        assertEquals(SQLITE_FLOAT, stmt.getColumnType(3));
        assertEquals(SQLITE_FLOAT, stmt.getColumnType(4));
        assertEquals(SQLITE_BLOB, stmt.getColumnType(5));
        assertEquals(SQLITE_NULL, stmt.getColumnType(6));        
        ret = stmt.step();
        assertEquals(SQLITE_DONE, ret);
        stmt.close();
        
        sql = "SELECT * FROM sqlite_master";
        stmt = newStatement(db, sql);
        ret = stmt.step();
        if (ret == SQLITE_ROW) {
            assertEquals(SQLITE3_TEXT, stmt.getColumnType(1));
            assertEquals(SQLITE3_TEXT, stmt.getColumnType(2));
            assertEquals(SQLITE3_TEXT, stmt.getColumnType(3));
        } else {
            assertEquals(SQLITE_DONE, ret);            
        }
        stmt.close();

        sql = "SELECT 1, '2', X'03', 4.0, NULL AS '5' LIMIT 1";
        stmt = newStatement(db, sql);
        assertEquals(SQLITE_ROW, stmt.step());
        assertEquals(SQLITE_INTEGER, stmt.getColumnType(1));
        assertEquals(SQLITE3_TEXT, stmt.getColumnType(2));
        assertEquals(SQLITE_BLOB, stmt.getColumnType(3));
        assertEquals(SQLITE_FLOAT, stmt.getColumnType(4));
        assertEquals(SQLITE_NULL, stmt.getColumnType(5));

        assertEquals(SQLITE_DONE, stmt.step());
        stmt.close();
        
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getColumnTableNameThrowSQLException() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT type, name, sql FROM sqlite_master";
        stmt = newStatement(db, sql);
        stmt.close();
        assertEquals("sqlite_master", stmt.getColumnTableName(1));
        assertEquals("sqlite_master", stmt.getColumnTableName(2));
        assertEquals("sqlite_master", stmt.getColumnTableName(3));
    
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getColumnTableNameOutOfRange() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT type, name, sql FROM sqlite_master";
        stmt = newStatement(db, sql);
        assertEquals("sqlite_master", stmt.getColumnTableName(0));
        stmt.close();
    
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getColumnTableNameOutOfRange2() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT type, name, sql FROM sqlite_master";
        stmt = newStatement(db, sql);
        assertEquals("sqlite_master", stmt.getColumnTableName(4));
        stmt.close();
    
        db.close();
    }
    
    @Test
    public void getColumnTableName() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        db.execute(
                "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_1("
                    + "  ID INTEGER PRIMARY KEY"
                    + ", VALUE TEXT"
                + ")"
            );
        db.execute(
                "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_2("
                    + "  ID INTEGER PRIMARY KEY"
                    + ", TEMP_TBL_1_ID INTEGER"
                    + ", VALUE2 TEXT"
                    + ", FOREIGN KEY(TEMP_TBL_1_ID) REFERENCES temp_tbl_1(ID)"
                + ")"
            );
        
        Statement stmt = null;
        String sql = null;

        sql = "SELECT type, name, sql FROM sqlite_master";
        stmt = newStatement(db, sql);
        assertEquals("sqlite_master", stmt.getColumnTableName(1));
        assertEquals("sqlite_master", stmt.getColumnTableName(2));
        assertEquals("sqlite_master", stmt.getColumnTableName(3));
        stmt.close();

        sql = "SELECT 1, '2', X'03', 4.0, NULL AS '5' LIMIT 1";
        stmt = newStatement(db, sql);
        assertNull(stmt.getColumnTableName(1));
        assertNull(stmt.getColumnTableName(2));
        assertNull(stmt.getColumnTableName(3));
        assertNull(stmt.getColumnTableName(4));
        assertNull(stmt.getColumnTableName(5));
        stmt.close();

        sql = "SELECT T2.ID, T1.ID AS ID1, T2.VALUE2, T1.VALUE"
                + " FROM temp_tbl_2 AS T2"
                + " INNER JOIN temp_tbl_1 AS T1 ON T1.ID = T2.TEMP_TBL_1_ID";
        stmt = newStatement(db, sql);
        assertEquals("temp_tbl_2", stmt.getColumnTableName(1));
        assertEquals("temp_tbl_1", stmt.getColumnTableName(2));
        assertEquals("temp_tbl_2", stmt.getColumnTableName(3));
        assertEquals("temp_tbl_1", stmt.getColumnTableName(4));
        stmt.close();
        
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getColumnTypeNameThrowSQLException() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT type, name, sql FROM sqlite_master";
        stmt = newStatement(db, sql);
        stmt.close();
        assertEquals("text", stmt.getColumnTypeName(1));
        assertEquals("text", stmt.getColumnTypeName(2));
        assertEquals("text", stmt.getColumnTypeName(3));
    
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getColumnTypeNameOutOfRange() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT type, name, sql FROM sqlite_master";
        stmt = newStatement(db, sql);
        assertEquals("", stmt.getColumnTypeName(0));
        stmt.close();
    
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getColumnTypeNameOutOfRange2() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT type, name, sql FROM sqlite_master";
        stmt = newStatement(db, sql);
        assertEquals("", stmt.getColumnTypeName(4));
        stmt.close();
    
        db.close();
    }
    
    @Test
    public void getColumnTypeName() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        db.execute(
                "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_1("
                    + "  ID INTEGER PRIMARY KEY"
                    + ", tVALUE TEXT"
                    + ", rVALUE REAL"
                    + ", bVALUE BLOB"
                    + ", nVALUE NULL"
                    + ", tVALUE2 text"
                    + ", tVALUE3 VarChar"
                + ")"
            );

        Statement stmt = null;
        String sql = null;

        sql = "SELECT type, name, sql FROM sqlite_temp_master";
        stmt = newStatement(db, sql);
        assertEquals("text", stmt.getColumnTypeName(1));
        assertEquals("text", stmt.getColumnTypeName(2));
        assertEquals("text", stmt.getColumnTypeName(3));

        assertEquals(SQLITE_ROW, stmt.step());
        assertEquals("text", stmt.getColumnTypeName(1));
        assertEquals("text", stmt.getColumnTypeName(2));
        assertEquals("text", stmt.getColumnTypeName(3));
        stmt.close();

        sql = "SELECT 1, '2', X'03', 4.0, NULL AS '5' LIMIT 1";
        stmt = newStatement(db, sql);
        assertNull(stmt.getColumnTypeName(1));
        assertNull(stmt.getColumnTypeName(2));
        assertNull(stmt.getColumnTypeName(3));
        assertNull(stmt.getColumnTypeName(4));
        assertNull(stmt.getColumnTypeName(5));

        assertEquals(SQLITE_ROW, stmt.step());
        assertNull(stmt.getColumnTypeName(1));
        assertNull(stmt.getColumnTypeName(2));
        assertNull(stmt.getColumnTypeName(3));
        assertNull(stmt.getColumnTypeName(4));
        assertNull(stmt.getColumnTypeName(5));
        stmt.close();

        sql = "SELECT ID, tVALUE, rVALUE, bVALUE, nVALUE, tVALUE2, tVALUE3 FROM temp_tbl_1";
        stmt = newStatement(db, sql);
        assertEquals("INTEGER", stmt.getColumnTypeName(1));
        assertEquals("TEXT", stmt.getColumnTypeName(2));
        assertEquals("REAL", stmt.getColumnTypeName(3));
        assertEquals("BLOB", stmt.getColumnTypeName(4));
        assertNull(stmt.getColumnTypeName(5));
        assertEquals("text", stmt.getColumnTypeName(6));
        assertEquals("VarChar", stmt.getColumnTypeName(7));
        
        assertEquals(SQLITE_DONE, stmt.step());
        assertEquals("INTEGER", stmt.getColumnTypeName(1));
        assertEquals("TEXT", stmt.getColumnTypeName(2));
        assertEquals("REAL", stmt.getColumnTypeName(3));
        assertEquals("BLOB", stmt.getColumnTypeName(4));
        assertNull(stmt.getColumnTypeName(5));
        assertEquals("text", stmt.getColumnTypeName(6));
        assertEquals("VarChar", stmt.getColumnTypeName(7));
        stmt.close();
        
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getStringThrowSQLException() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT type, name, sql FROM sqlite_master";
        stmt = newStatement(db, sql);
        stmt.close();
        assertEquals("text", stmt.getString(1));
        assertEquals("text", stmt.getString(2));
        assertEquals("text", stmt.getString(3));
    
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getStringOutOfRange() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT type, name, sql FROM sqlite_master";
        stmt = newStatement(db, sql);
        assertEquals("", stmt.getString(0));
        stmt.close();
    
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getStringOutOfRange2() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT type, name, sql FROM sqlite_master";
        stmt = newStatement(db, sql);
        assertEquals("", stmt.getString(4));
        stmt.close();
    
        db.close();
    }
    
    @Test
    public void getString() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        String ddl = null;
        ddl = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_1("
                + "  ID INTEGER PRIMARY KEY"
                + ", tVALUE TEXT"
                + ", rVALUE REAL"
                + ", bVALUE BLOB"
                + ", nVALUE NULL"
                + ", tVALUE2 text"
                + ", tVALUE3 VarChar"
            + ")";
        db.execute(ddl);
        Statement stmt = null;
        String sql = null;

        sql = "SELECT type, name, sql"
                + " FROM sqlite_temp_master"
                + " WHERE name = 'temp_tbl_1' AND type = 'table'";
        stmt = newStatement(db, sql);
        assertEquals(SQLITE_ROW, stmt.step());
        assertEquals("table", stmt.getString(1));
        assertEquals("temp_tbl_1", stmt.getString(2));
//        assertEquals(ddl, stmt.getString(3));
        stmt.close();

//        sql = "SELECT 1, '2', X'E38182E38184E38186E38188E3818A', 4.0, NULL AS '5' LIMIT 1";
        sql = "SELECT 1, '2', X'E38182E38184E38186E38188E3818A00', 4.0, NULL AS '5' LIMIT 1";
        stmt = newStatement(db, sql);
        assertEquals(SQLITE_ROW, stmt.step());
        assertEquals("1", stmt.getString(1));
        assertEquals("2", stmt.getString(2));
        assertEquals("あいうえお", stmt.getString(3));
        assertEquals("4.0", stmt.getString(4));
        assertNull(stmt.getString(5));
        stmt.close();

        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getIntThrowSQLException() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT 1, 2, 3 LIMIT 1";
        stmt = newStatement(db, sql);
        stmt.close();
        assertEquals(1, stmt.getInt(1));
        assertEquals(2, stmt.getInt(2));
        assertEquals(3, stmt.getInt(3));
    
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getIntOutOfRange() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT 1, 2, 3 LIMIT 1";
        stmt = newStatement(db, sql);
        assertEquals(0, stmt.getInt(0));
        stmt.close();
    
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getIntOutOfRange2() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT 1, 2, 3 LIMIT 1";
        stmt = newStatement(db, sql);
        assertEquals(0, stmt.getInt(4));
        stmt.close();
    
        db.close();
    }
    
    @Test
    public void getInt() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT 1, 2, 3 LIMIT 1";
        stmt = newStatement(db, sql);
        assertEquals(SQLITE_ROW, stmt.step());
        assertEquals(1, stmt.getInt(1));
        assertEquals(2, stmt.getInt(2));
        assertEquals(3, stmt.getInt(3));
        stmt.close();

        sql = "SELECT 1, '2', X'3132333435363738393000', 4.4444, NULL AS '5' LIMIT 1";
        stmt = newStatement(db, sql);
        assertEquals(SQLITE_ROW, stmt.step());
        assertEquals(1, stmt.getInt(1));
        assertEquals(2, stmt.getInt(2));
        assertEquals(1234567890, stmt.getInt(3));
        assertEquals(4, stmt.getInt(4));
        assertEquals(0, stmt.getInt(5));
        stmt.close();

        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getLongThrowSQLException() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT 11111111111, 22222222222, 33333333333 LIMIT 1";
        stmt = newStatement(db, sql);
        stmt.close();
        assertEquals(11111111111L, stmt.getLong(1));
        assertEquals(22222222222L, stmt.getLong(2));
        assertEquals(33333333333L, stmt.getLong(3));
    
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getLongOutOfRange() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT 11111111111, 22222222222, 33333333333 LIMIT 1";
        stmt = newStatement(db, sql);
        assertEquals(0L, stmt.getLong(0));
        stmt.close();
    
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getLongOutOfRange2() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT 11111111111, 22222222222, 33333333333 LIMIT 1";
        stmt = newStatement(db, sql);
        assertEquals(0L, stmt.getLong(4));
        stmt.close();
    
        db.close();
    }
    
    @Test
    public void getLong() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT 11111111111, 22222222222, 33333333333 LIMIT 1";
        stmt = newStatement(db, sql);
        assertEquals(SQLITE_ROW, stmt.step());
        assertEquals(11111111111L, stmt.getLong(1));
        assertEquals(22222222222L, stmt.getLong(2));
        assertEquals(33333333333L, stmt.getLong(3));
        stmt.close();

        sql = "SELECT 111111111111, '222222222222', X'31323334353637383930313200', 444444444444.444, NULL AS '5' LIMIT 1";
        stmt = newStatement(db, sql);
        assertEquals(SQLITE_ROW, stmt.step());
        assertEquals(111111111111L, stmt.getLong(1));
        assertEquals(222222222222L, stmt.getLong(2));
        assertEquals(123456789012L, stmt.getLong(3));
        assertEquals(444444444444L, stmt.getLong(4));
        assertEquals(0L, stmt.getLong(5));
        stmt.close();

        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getDoubleThrowSQLException() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT 1.1111111111, 2.2222222222, 3.3333333333 LIMIT 1";
        stmt = newStatement(db, sql);
        stmt.close();
        assertEquals(1.1111111111, stmt.getDouble(1));
        assertEquals(2.2222222222, stmt.getDouble(2));
        assertEquals(3.3333333333, stmt.getDouble(3));
    
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getDoubleOutOfRange() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT 1.1111111111, 2.2222222222, 3.3333333333 LIMIT 1";
        stmt = newStatement(db, sql);
        assertEquals(0.0, stmt.getDouble(0));
        stmt.close();
    
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getDoubleOutOfRange2() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT 1.1111111111, 2.2222222222, 3.3333333333 LIMIT 1";
        stmt = newStatement(db, sql);
        assertEquals(0.0, stmt.getDouble(4));
        stmt.close();
    
        db.close();
    }
    
    @Test
    public void getDouble() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT 1.1111111111, 2.2222222222, 3.3333333333 LIMIT 1";
        stmt = newStatement(db, sql);
        assertEquals(SQLITE_ROW, stmt.step());
        assertEquals(1.1111111111, stmt.getDouble(1));
        assertEquals(2.2222222222, stmt.getDouble(2));
        assertEquals(3.3333333333, stmt.getDouble(3));
        stmt.close();

        sql = "SELECT 111111111111, '222222222222', X'31323334353637383930313200', 444444444444.444, NULL AS '5' LIMIT 1";
        stmt = newStatement(db, sql);
        assertEquals(SQLITE_ROW, stmt.step());
        assertEquals(111111111111.0, stmt.getDouble(1));
        assertEquals(222222222222.0, stmt.getDouble(2));
        assertEquals(123456789012.0, stmt.getDouble(3));
        assertEquals(444444444444.444, stmt.getDouble(4));
        assertEquals(0.0, stmt.getDouble(5));
        stmt.close();

        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getBytesThrowSQLException() throws ClassNotFoundException, SQLException, UnsupportedEncodingException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT 1.1111111111, 2.2222222222, 3.3333333333 LIMIT 1";
        stmt = newStatement(db, sql);
        stmt.close();
        assertTrue(Arrays.equals("1.1111111111".getBytes("UTF-8"), stmt.getBytes(1)));
        assertTrue(Arrays.equals("2.2222222222".getBytes("UTF-8"), stmt.getBytes(2)));
        assertTrue(Arrays.equals("3.3333333333".getBytes("UTF-8"), stmt.getBytes(3)));
    
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getBytesOutOfRange() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT 1.1111111111, 2.2222222222, 3.3333333333 LIMIT 1";
        stmt = newStatement(db, sql);
        assertNull(stmt.getBytes(0));
        stmt.close();
    
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getBytesOutOfRange2() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT 1.1111111111, 2.2222222222, 3.3333333333 LIMIT 1";
        stmt = newStatement(db, sql);
        assertNull(stmt.getBytes(4));
        stmt.close();
    
        db.close();
    }

    @Test
    public void getBytes() throws ClassNotFoundException, SQLException, UnsupportedEncodingException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT 1.1111111111, 2.2222222222, 3.3333333333 LIMIT 1";
        stmt = newStatement(db, sql);
        assertEquals(SQLITE_ROW, stmt.step());
        assertTrue(Arrays.equals("1.1111111111".getBytes("UTF-8"), stmt.getBytes(1)));
        assertTrue(Arrays.equals("2.2222222222".getBytes("UTF-8"), stmt.getBytes(2)));
        assertTrue(Arrays.equals("3.3333333333".getBytes("UTF-8"), stmt.getBytes(3)));
        stmt.close();

        sql = "SELECT 111111111111, '222222222222', X'31323334353637383930313200', 444444444444.444, NULL AS '5' LIMIT 1";
        stmt = newStatement(db, sql);
        assertEquals(SQLITE_ROW, stmt.step());
        assertTrue(Arrays.equals("111111111111".getBytes("UTF-8"), stmt.getBytes(1)));
        assertTrue(Arrays.equals("222222222222".getBytes("UTF-8"), stmt.getBytes(2)));
        assertTrue(Arrays.equals("123456789012\0".getBytes("UTF-8"), stmt.getBytes(3)));
        assertTrue(Arrays.equals("444444444444.444".getBytes("UTF-8"), stmt.getBytes(4)));
        assertNull(stmt.getBytes(5));
        stmt.close();

        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getBlobThrowSQLException() throws ClassNotFoundException, SQLException, UnsupportedEncodingException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT 1.1111111111, 2.2222222222, 3.3333333333 LIMIT 1";
        stmt = newStatement(db, sql);
        stmt.close();
        assertNotNull(stmt.getBlob(1));
        assertNotNull(stmt.getBlob(2));
        assertNotNull(stmt.getBlob(3));
    
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getBlobOutOfRange() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT 1.1111111111, 2.2222222222, 3.3333333333 LIMIT 1";
        stmt = newStatement(db, sql);
        assertNotNull(stmt.getBlob(0));
        stmt.close();
    
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getBlobOutOfRange2() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT 1.1111111111, 2.2222222222, 3.3333333333 LIMIT 1";
        stmt = newStatement(db, sql);
        assertNull(stmt.getBlob(4));
        stmt.close();
    
        db.close();
    }
    
    @Test
    public void getBlob() throws ClassNotFoundException, SQLException, UnsupportedEncodingException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT 1.1111111111, 2.22222222222, 3.333333333333 LIMIT 1";
        stmt = newStatement(db, sql);
        assertEquals(SQLITE_ROW, stmt.step());
        assertNotNull(stmt.getBlob(1));
        assertEquals(12, stmt.getByteLength(1));
        assertNotNull(stmt.getBlob(2));
        assertEquals(13, stmt.getByteLength(2));
        assertNotNull(stmt.getBlob(3));
        assertEquals(14, stmt.getByteLength(3));
        stmt.close();

        sql = "SELECT 1111111111, '22222222222', X'31323334353637383930313200', 444444444444.444, NULL LIMIT 1";
        stmt = newStatement(db, sql);
        assertEquals(SQLITE_ROW, stmt.step());
        assertNotNull(stmt.getBlob(1));
        assertEquals(10, stmt.getByteLength(1));
        assertNotNull(stmt.getBlob(2));
        assertEquals(11, stmt.getByteLength(2));
        assertNotNull(stmt.getBlob(3));
        assertEquals(13, stmt.getByteLength(3));
        assertNotNull(stmt.getBlob(4));
        assertEquals(16, stmt.getByteLength(4));
        assertNotNull(stmt.getBlob(5));
        assertEquals(0, stmt.getByteLength(5));
        
        stmt.close();

        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getByteLengthThrowSQLException() throws ClassNotFoundException, SQLException, UnsupportedEncodingException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT 1.1111111111, 2.2222222222, 3.3333333333 LIMIT 1";
        stmt = newStatement(db, sql);
        stmt.close();
        assertNotNull(stmt.getByteLength(1));
        assertNotNull(stmt.getByteLength(2));
        assertNotNull(stmt.getByteLength(3));
    
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getByteLengthOutOfRange() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT 1.1111111111, 2.2222222222, 3.3333333333 LIMIT 1";
        stmt = newStatement(db, sql);
        assertNotNull(stmt.getByteLength(0));
        stmt.close();
    
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void getByteLengthOutOfRange2() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT 1.1111111111, 2.2222222222, 3.3333333333 LIMIT 1";
        stmt = newStatement(db, sql);
        assertNull(stmt.getByteLength(4));
        stmt.close();
    
        db.close();
    }
    
    @Test
    public void getByteLength() throws ClassNotFoundException, SQLException, UnsupportedEncodingException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT 1.1111, 2.22222, 3.3333333333 LIMIT 1";
        stmt = newStatement(db, sql);
        assertEquals(SQLITE_ROW, stmt.step());
        assertEquals(6, stmt.getByteLength(1));
        assertEquals(7, stmt.getByteLength(2));
        assertEquals(12, stmt.getByteLength(3));
        stmt.close();

        sql = "SELECT 111, '222222', X'31323334353637383930313200', 444444444444.444, NULL AS '5' LIMIT 1";
        stmt = newStatement(db, sql);
        assertEquals(SQLITE_ROW, stmt.step());
        assertEquals(3, stmt.getByteLength(1));
        assertEquals(6, stmt.getByteLength(2));
        assertEquals(13, stmt.getByteLength(3));
        assertEquals(16, stmt.getByteLength(4));
        assertEquals(0, stmt.getByteLength(5));
        stmt.close();

        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void bindNullThrowSQLException() throws ClassNotFoundException, SQLException, UnsupportedEncodingException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT * FROM sqlite_master WHERE type = ?";
        stmt = newStatement(db, sql);
        stmt.close();
        stmt.bindNull(1);
    
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void bindNullOutOfRange() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT * FROM sqlite_master WHERE type = ?";
        stmt = newStatement(db, sql);
        stmt.bindNull(0);
        stmt.close();
    
        db.close();
    }

    @Test(expected = java.sql.SQLException.class)
    public void bindNullOutOfRange2() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT * FROM sqlite_master WHERE type = ?";
        stmt = newStatement(db, sql);
        stmt.bindNull(2);
        stmt.close();
    
        db.close();
    }
    
    @Test
    public void bindNull() throws ClassNotFoundException, SQLException, UnsupportedEncodingException, IOException {
        final Database db = newDatabase();
        Statement stmt = null;
        String sql = null;

        sql = "SELECT * FROM sqlite_master WHERE type = ?";
        stmt = newStatement(db, sql);
        stmt.bindNull(1);
        assertEquals(SQLITE_DONE, stmt.step());
        stmt.reset();
        stmt.bindText(1, "table");
        int ret = stmt.step();
        if (ret != SQLITE_ROW) {
            assertEquals(SQLITE_DONE, ret);
        }
        stmt.reset();
        stmt.bindNull(1);
        assertEquals(SQLITE_DONE, stmt.step());
        stmt.close();
        
        sql = "SELECT * FROM sqlite_master WHERE ? IS NULL";
        stmt = newStatement(db, sql);
        stmt.bindNull(1);
        ret = stmt.step();
        if (ret != SQLITE_ROW) {
            assertEquals(SQLITE_DONE, ret);
        }
        stmt.reset();
        stmt.bindText(1, "table");
        assertEquals(SQLITE_DONE, stmt.step());
        stmt.close();

        db.close();
    }

    @Test
    public void getColumnDatabaseName() throws ClassNotFoundException, SQLException, IOException {
        final Database db = newDatabase();
        db.execute(
                "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_1("
                    + "  ID INTEGER PRIMARY KEY"
                    + ", VALUE TEXT"
                + ")"
            );
        db.execute(
                "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_2("
                    + "  ID INTEGER PRIMARY KEY"
                    + ", TEMP_TBL_1_ID INTEGER"
                    + ", VALUE2 TEXT"
                    + ", FOREIGN KEY(TEMP_TBL_1_ID) REFERENCES temp_tbl_1(ID)"
                + ")"
            );
        
        Statement stmt = null;
        String sql = null;

        sql = "SELECT type, name, sql FROM sqlite_master";
        stmt = newStatement(db, sql);
        assertEquals("main", stmt.getColumnDatabaseName(1));
        stmt.close();
        
        sql = "SELECT * FROM temp_tbl_1";
        stmt = newStatement(db, sql);
        assertEquals("temp", stmt.getColumnDatabaseName(1));
        stmt.close();

        db.close();        
    }

    @Test
    public void close() throws ClassNotFoundException, SQLException {
        final Database db = newDatabase();
        try {
            final String sql = "SELECT * FROM sqlite_master";
            final Statement stmt = db.prepare(sql);
            try {
                int ret = stmt.step();
                if (ret != SQLITE_ROW) {
                    assertEquals(SQLITE_DONE, ret);
                }
            } finally {
                new Thread() {
                    @Override
                    public void run() {
                        try {
                            stmt.close();
                            System.out.println("close statement.");
                        } catch (SQLException ex) {
                            Logger.getLogger(StatementTest.class.getName()).log(Level.SEVERE, null, ex);
                        }
                    }
                }.start();
            }
        } finally {
            new Thread() {
                @Override
                public void run() {
                    try {
                        db.close();
                        System.out.println("close database.");
                    } catch (SQLException ex) {
                        Logger.getLogger(DatabaseTest.class.getName()).log(Level.SEVERE, null, ex);
                    }
                }
            }.start();
        }
    }
    
    @Test
    public void finalizer() throws ClassNotFoundException, SQLException, InterruptedException, IOException {
        final Database db = newDatabase();
        final String sql = "SELECT * FROM sqlite_master";
        final Statement stmt = db.prepare(sql);
        stmt.step();
        final Thread closer
                = new Thread() {
                            @Override
                            public void run() {
                                try {
                                    stmt.close();
                                } catch (SQLException ex) {
                                    Logger.getLogger(StatementTest.class.getName()).log(Level.SEVERE, null, ex);
                                }
                            }
                        };
        closer.start();
        closer.join();
        db.close();
    }
}