/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package org.sqlite;

import org.sqlite.schema.ColumnMetaData;
import java.sql.SQLException;
import java.util.logging.Logger;
import org.sqlite.jdbc.JdbcSQLException;
import org.sqlite.swig.SWIGTYPE_p_sqlite3;
import org.sqlite.swig.SWIGTYPE_p_sqlite3_stmt;
import org.sqlite.swig.SWIGTYPE_p_void;
import org.sqlite.types.SQLite3StmtPtrPtr;
import static org.sqlite.swig.SQLite3.*;

/**
 * sqlite3_stmt wrapper class.<br/>
 * NOTE: SQLite 3.3.5 based.
 * @author calico
 */
public class Statement {
    private final Database db;
    private final SWIGTYPE_p_sqlite3_stmt stmt;
    /** MANAGED Statement is NULL */
    private final SQLite3StmtPtrPtr ppStmt;
    private boolean isClosed;
    
    Statement(Database db, SWIGTYPE_p_sqlite3_stmt stmt) throws SQLException {
        this(db, stmt, null);
    }
    
    Statement(Database db, SQLite3StmtPtrPtr ppStmt) throws SQLException {
        this(db, ppStmt.getSQLite3StmtPtr(), ppStmt);
    }
    
    private Statement(Database db, SWIGTYPE_p_sqlite3_stmt stmt, SQLite3StmtPtrPtr ppStmt) throws SQLException {
        this.db = db;
        this.stmt = stmt;
        this.ppStmt = ppStmt;
        db.addStatement(this);
    }
    
    /**
     * True is returned when generated with PreparedStatement. 
     * @return
     */
    public boolean isManaged() {
        return (ppStmt == null);
    }
    
    /**
     * invoke sqlite3_bind_parameter_count() function.
     * @return parameter count
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/bind_parameter_count.html">Number Of SQL Parameters</a>
     */
    public int getParameterCount() throws SQLException {
        validateStatementOpen();

        return sqlite3_bind_parameter_count(stmt);
    }
    
    /**
     * invoke sqlite3_bind_parameter_index() function.
     * @param parameterName
     * @return parameter index
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/bind_parameter_index.html">Index Of A Parameter With A Given Name</a>
     */
    public int getParameterIndex(String parameterName) throws SQLException {
        validateStatementOpen();

        return sqlite3_bind_parameter_index(stmt, parameterName);
    }
    
    /**
     * invoke sqlite3_bind_parameter_name() function.
     * @param parameterIndex
     * @return parameter name
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/bind_parameter_name.html">Name Of A Host Parameter</a>
     */
    public String getParameterName(int parameterIndex) throws SQLException {
        validateStatementOpen();

        return sqlite3_bind_parameter_name(stmt, parameterIndex);
    }
    
    /**
     * invoke sqlite3_column_count() function.
     * @return column count
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/column_count.html">Number Of Columns In A Result Set</a>
     */
    public int getColumnCount() throws SQLException {
        validateStatementOpen();

        return sqlite3_column_count(stmt);
    }
    
    /**
     * invoke sqlite3_data_count() function.<br/>
     * NOTE: Required to invoke the step() method beforehand.
     * @return data count
     * @throws java.sql.SQLException
     * @see #step()
     * @see <a href="http://sqlite.org/c3ref/data_count.html">Number of columns in a result set</a>
     */
    public int getDataCount() throws SQLException {
        validateStatementOpen();

        return sqlite3_data_count(stmt);
    }
    
    /**
     * invoke sqlite3_column_name() function.
     * @param columnIndex
     * @return column label
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/column_name.html">Column Names In A Result Set</a>
     */
    public String getColumnLabel(int columnIndex) throws SQLException {
        validateColumnIndexRange(columnIndex);
        
        return sqlite3_column_name(stmt, columnIndex - 1);
    }
    
    /**
     * invoke sqlite3_column_origin_name() function.
     * @param columnIndex
     * @return column name
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/column_database_name.html">Source Of Data In A Query Result</a>
     */
    public String getColumnName(int columnIndex) throws SQLException {
        validateColumnIndexRange(columnIndex);
        
        return sqlite3_column_origin_name(stmt, columnIndex - 1);
    }
    
    /**
     * invoke sqlite3_column_type() function.
     * NOTE: Required to invoke the step() method beforehand.
     * @param columnIndex
     * @return column type
     * @throws java.sql.SQLException
     * @see #step()
     * @see <a href="http://sqlite.org/c3ref/column_blob.html">Results Values From A Query</a>
     */
    public int getColumnType(int columnIndex) throws SQLException {
        validateColumnIndexRange(columnIndex);
        
        return sqlite3_column_type(stmt, columnIndex - 1);
    }
    
    /**
     * invoke sqlite3_column_table_name() function.
     * @param columnIndex
     * @return table name
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/column_database_name.html">Source Of Data In A Query Result</a>
     */
    public String getColumnTableName(int columnIndex) throws SQLException {
        validateColumnIndexRange(columnIndex);

        return sqlite3_column_table_name(stmt, columnIndex - 1);
    }
    
    /**
     * invoke sqlite3_column_decltype() function.
     * @param columnIndex
     * @return column declare type name
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/column_decltype.html">Declared Datatype Of A Query Result</a>
     */
    public String getColumnTypeName(int columnIndex) throws SQLException {
        validateColumnIndexRange(columnIndex);
        
        return sqlite3_column_decltype(stmt, columnIndex - 1);
    }
    
    /**
     * invoke org.sqlite.Database#getColumnMetaData() method.
     * @param dbName
     * @param tableName
     * @param columnName
     * @return column meta data
     * @throws java.sql.SQLException
     * @see org.sqlite.Database#getColumnMetaData(String, String, String)
     */
    public ColumnMetaData getColumnMetaData(String dbName, String tableName, String columnName) throws SQLException {
        return db.getColumnMetaData(dbName, tableName, columnName);
    }

    /**
     * invoke sqlite3_column_text() function.
     * @param columnIndex
     * @return
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/column_blob.html">Results Values From A Query</a>
     */
    public String getString(int columnIndex) throws SQLException {
        validateColumnIndexRange(columnIndex);
        
        return sqlite3_column_text(stmt, columnIndex - 1);
    }
    
    /**
     * invoke sqlite3_column_int() function.
     * @param columnIndex
     * @return
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/column_blob.html">Results Values From A Query</a>
     */
    public int getInt(int columnIndex) throws SQLException {
        validateColumnIndexRange(columnIndex);
        
        return sqlite3_column_int(stmt, columnIndex - 1);
    }
    
    /**
     * invoke sqlite3_column_int64() function.
     * @param columnIndex
     * @return
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/column_blob.html">Results Values From A Query</a>
     */
    public long getLong(int columnIndex) throws SQLException {
        validateColumnIndexRange(columnIndex);
        
        return sqlite3_column_int64(stmt, columnIndex - 1);
    }
    
    /**
     * invoke sqlite3_column_double() function.
     * @param columnIndex
     * @return
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/column_blob.html">Results Values From A Query</a>
     */
    public double getDouble(int columnIndex) throws SQLException {
        validateColumnIndexRange(columnIndex);
        
        return sqlite3_column_double(stmt, columnIndex - 1);
    }
    
    /**
     * invoke sqlite3_column_blob() function.
     * @param columnIndex
     * @return
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/column_blob.html">Results Values From A Query</a>
     */
    public byte[] getBytes(int columnIndex) throws SQLException {
        validateColumnIndexRange(columnIndex);
        
        return sqlite3_column_blob_by_bytes(stmt, columnIndex - 1);
    }
    
    /**
     * invoke sqlite3_column_blob() function.
     * @param columnIndex
     * @return
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/column_blob.html">Results Values From A Query</a>
     */
    public SWIGTYPE_p_void getBlob(int columnIndex) throws SQLException {
        validateColumnIndexRange(columnIndex);
        
        return sqlite3_column_blob(stmt, columnIndex - 1);
    }
    
    /**
     * invoke sqlite3_column_bytes() function.
     * @param columnIndex
     * @return
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/column_blob.html">Results Values From A Query</a>
     */
    public int getByteLength(int columnIndex) throws SQLException {
        validateColumnIndexRange(columnIndex);
        
        return sqlite3_column_bytes(stmt, columnIndex - 1);
    }
    
    /**
     * invoke sqlite3_bind_null() function.
     * @param parameterIndex
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/bind_blob.html">Binding Values To Prepared Statements</a>
     */
    public void bindNull(int parameterIndex) throws SQLException {
        validateStatementOpen();
        
        int ret = sqlite3_bind_null(stmt, parameterIndex);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(stmt);
        }
    }
    
    /**
     * invoke sqlite3_bind_int() function.
     * @param parameterIndex
     * @param val
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/bind_blob.html">Binding Values To Prepared Statements</a>
     */
    public void bindInt(int parameterIndex, int val) throws SQLException {
        validateStatementOpen();
        
        int ret = sqlite3_bind_int(stmt, parameterIndex, val);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(stmt);
        }
    }
    
    /**
     * invoke sqlite3_bind_int64() function.
     * @param parameterIndex
     * @param val
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/bind_blob.html">Binding Values To Prepared Statements</a>
     */
    public void bindLong(int parameterIndex, long val) throws SQLException {
        validateStatementOpen();
        
        int ret = sqlite3_bind_int64(stmt, parameterIndex, val);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(stmt);
        }
    }
    
    /**
     * invoke sqlite3_bind_double() function.
     * @param parameterIndex
     * @param val
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/bind_blob.html">Binding Values To Prepared Statements</a>
     */
    public void bindDouble(int parameterIndex, double val) throws SQLException {
        validateStatementOpen();
        
        int ret = sqlite3_bind_double(stmt, parameterIndex, val);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(stmt);
        }
    }
    
    /**
     * invoke sqlite3_bind_text() function.
     * @param parameterIndex
     * @param val
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/bind_blob.html">Binding Values To Prepared Statements</a>
     */
    public void bindText(int parameterIndex, String val) throws SQLException {
        validateStatementOpen();
        
        int ret = sqlite3_bind_text(stmt, parameterIndex, val);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(stmt);
        }
    }
    
    /**
     * invoke sqlite3_bind_blob() function.
     * @param parameterIndex
     * @param val
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/bind_blob.html">Binding Values To Prepared Statements</a>
     */
    public void bindBytes(int parameterIndex, byte[] val) throws SQLException {
        bindBytes(parameterIndex, val, val.length);
    }
    
    /**
     * invoke sqlite3_bind_blob() function.
     * @param parameterIndex
     * @param val
     * @param len
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/bind_blob.html">Binding Values To Prepared Statements</a>
     */
    public void bindBytes(int parameterIndex, byte[] val, int len) throws SQLException {
        validateStatementOpen();
        
        int ret = sqlite3_bind_blob(stmt, parameterIndex, val, len);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(stmt);
        }
    }
    
    /**
     * invoke sqlite3_clear_bindings() function.
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/clear_bindings.html">Reset All Bindings On A Prepared Statement</a>
     */
    public void clearBinding() throws SQLException {
        validateStatementOpen();
        
        int ret = sqlite3_clear_bindings(stmt);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(stmt);
        }
    }

    /**
     * invoke sqlite3_reset() function.
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/reset.html">Reset A Prepared Statement Object</a>
     */
    public void reset() throws SQLException {
        validateStatementOpen();

        int ret = sqlite3_reset(stmt);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(stmt);
        }
    }

    /**
     * invoke sqlite3_step() function.
     * @return
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/step.html">Evaluate An SQL Statement</a>
     */
    public int step() throws SQLException {
        validateStatementOpen();
        
        int ret = 0;
        if (db.getBusyTimeout() == 0) {
            // no limit
            while ((ret = sqlite3_step(stmt)) == SQLITE_BUSY) {
                // waiting...
            }
        } else {
            ret = sqlite3_step(stmt);
            if (ret == SQLITE_BUSY) {
                // timeout
                throw new JdbcSQLException("Timeout expired.", SQLITE_BUSY);
//                throw new JdbcSQLException(stmt);
            }
        }
        if (ret != SQLITE_DONE && ret != SQLITE_ROW) {
            throw new JdbcSQLException(stmt);
        }
        return ret;
    }

    /**
     * invoke step() method.
     * @return
     * @throws java.sql.SQLException
     */
    public int execute() throws SQLException {
        return step();
    }
    
    /**
     * invoke sqlite3_expired() function.
     * @return
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/aggregate_count.html">Obsolete Functions</a>
     */
    public boolean isExpired() throws SQLException {
        validateStatementOpen();

        return (sqlite3_expired(stmt) != 0);
    }
    
    /**
     * invoke sqlite3_db_handle() function.
     * @return
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/db_handle.html">Find The Database Handle Of A Prepared Statement</a>
     */
    public SWIGTYPE_p_sqlite3 getDbHandle() throws SQLException {
        validateStatementOpen();

        return sqlite3_db_handle(stmt);
    }
    
    /**
     * invoke sqlite3_transfer_bindings() function.
     * @throws java.sql.SQLException
     * @see <a href="http://www.3rd-impact.net/Document/SQLite/Translation/Current/Original/capi3ref.html#sqlite3_transfer_bindings">sqlite3_transfer_bindings</a>
     * @deprecated
     */
    public void transferBinding(Statement dest) throws SQLException {
        validateStatementOpen();

        // TODO 3.5.6ではsqlite3_transfer_bindings関数は非推奨になっている
        // http://sqlite.org/c3ref/aggregate_count.html
        int ret = sqlite3_transfer_bindings(stmt, dest.stmt);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(stmt);
        }
    }
    
    /**
     * true if the return value of getColumnCount() is not 0.
     * @return
     * @throws java.sql.SQLException
     */
    public boolean producedResultSet() throws SQLException {
        return (getColumnCount() != 0);
    }
    
    /**
     * invoke org.sqlite.swig.SQLite3#addressOf(SWIGTYPE_p_sqlite3_stmt) method.
     * @return address of SWIGTYPE_p_sqlite3_stmt
     * @throws java.sql.SQLException
     * @see org.sqlite.swig.SQLite3#addressOf(SWIGTYPE_p_sqlite3_stmt)
     */
    long getHandle() throws SQLException {
        return addressOf(stmt);
    }
    
    public boolean isClosed() {
        return (isClosed || db.isClosed());
    }
    
    /**
     * invoke sqlite3_finalize() function.
     * @throws java.sql.SQLException
     * @see <a href="http://sqlite.org/c3ref/finalize.html">Destroy A Prepared Statement Object</a>
     */
    public void close() throws SQLException {
        if (!isClosed) {
            int ret = sqlite3_finalize(stmt);
            if (ret == SQLITE_ABORT) {
                Logger.getLogger(Statement.class.getName()).info("Transaction aborted.");
            } else if (ret != SQLITE_OK) {
                throw new JdbcSQLException(stmt);
            }

            isClosed = true;
            db.removeStatement(this);
            if (ppStmt != null) {
                // UNMANAGED Statement
                ppStmt.delete();
            }
        }
    }
    
    @Override
    protected void finalize() throws Throwable {
        if (!isClosed) {
            Logger.getLogger(Statement.class.getName()).info("Statement is not closed.");
        }
        super.finalize();
    }
    
    /**
     * 
     * @throws java.sql.SQLException
     */
    protected void validateStatementOpen() throws SQLException {
        if (isClosed()) {
            throw new SQLException("Statement is already closed.", "90007");
        }
    }
    
    /**
     * 
     * @param columnIndex
     * @throws java.sql.SQLException
     */
    protected void validateColumnIndexRange(int columnIndex) throws SQLException {
        if (columnIndex < 1 || columnIndex > getColumnCount()) {
            throw new SQLException("Column index out of range.", "90J01");
        }
    }

    /**
     * 
     * @return Database
     * @throws java.sql.SQLException
     */
    public Database getDatabase() throws SQLException {
        validateStatementOpen();

        return db;
    }
}
