/*
 * ファイルのコピーを行う。
 * 一部、copybench-1.0（New BSD License）を参考にした。
*/
#define _GNU_SOURCE

#ifdef _DEBUG_FLAG
#define USE_O_DIRECT_SIZE 1
#endif
#ifndef _DEBUG_FLAG
#define USE_O_DIRECT_SIZE 1024 * 1024 * 1
#endif

#include "ch_time_and_mod.h" // inline
#include "print_error.h" // inline
#include "struct_CPDD.h"
#include "struct_CPInfo.h"
#include "struct_OPArg.h"
#include "verify_hash.h"
#include "xmalloc.h"

#include "md5.h"

#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>

#include <errno.h>
#include <fcntl.h>
#include <linux/falloc.h>
#include <pthread.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <unistd.h>

#define OPEN_CHECK_DST \
{\
	errno = 0;\
	if((o_direct_write_flag == false) || (src_size_local < USE_O_DIRECT_SIZE))\
	{\
		idst = open(cpinfo->dst, O_CREAT | O_WRONLY | O_TRUNC | O_NOFOLLOW, S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH);\
		o_direct_open_dst = false;\
	}\
	else \
	{\
		idst = open(cpinfo->dst, O_CREAT | O_WRONLY | O_TRUNC | O_NOFOLLOW | O_DIRECT, S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH);\
		o_direct_open_dst = true;\
	}\
\
	if(idst == -1)\
	{\
		o_direct_open_dst = false;\
		print_error("open", __FILE__, __LINE__, cpdd);\
	}\
	else \
	{\
		write_now = cpinfo->dst;	/* signalハンドラ用 */\
		open_dst = true;\
		cpdd->mk_file_count++;\
	}\
}

#define MD5_FINISH \
{\
	md5_finish(state, digest);\
	char *hex_output = xmalloc((16 * 2) + 1);\
\
	for(int di = 0; di < 16; ++di)\
	{\
		sprintf(hex_output + di * 2, "%02x", digest[di]);\
	}\
\
	cpinfo->src_md5 = hex_output;\
}

struct verify_data
{
	md5_state_t *state;
	void *buffer;
	int len;
};

static void *buf[2];
static off_t src_size_local;
static off_t written_size[2];
static off_t next_write_size[2];
static off_t total_read_local;
static off_t total_write_local;
static off_t total_read_error; // スレッド用
static off_t total_write_error; // スレッド用
static int isrc;
static int idst;
static int buffer_size;
static int buffer_size_half;
static int buffer_size_thread;
static int ri; // index number
static int wi; // index number
static long psize;
static _Bool verify;

/* ※注意。グローバル変数。 */
char *write_now;

/* 関数プロトタイプ */
void read_write(const OPArg *oparg, const SDir *sdir, CPInfo *cpinfo, CPDD *cpdd);
inline static void mw_small(CPInfo *cpinfo, CPDD *cpdd, md5_state_t *state, md5_byte_t digest[16], struct verify_data *thread_argument, pthread_t *t_verify);
inline static void mw_small_direct(CPInfo *cpinfo, CPDD *cpdd, md5_state_t *state, md5_byte_t digest[16], struct verify_data *thread_argument, pthread_t *t_verify);
inline static void mw_big(CPInfo *cpinfo, CPDD *cpdd, md5_state_t *state, md5_byte_t digest[16], struct verify_data *thread_argument, pthread_t *t_verify);
inline static void mw_big_direct(CPInfo *cpinfo, CPDD *cpdd, md5_state_t *state, md5_byte_t digest[16], struct verify_data *thread_argument, pthread_t *t_verify);
inline static void mw_thread(const OPArg *oparg, CPInfo *cpinfo, CPDD *cpdd, md5_state_t *state, md5_byte_t digest[16], struct verify_data *thread_argument, pthread_t *t_verify);
inline static void mw_thread_direct(const OPArg *oparg, CPInfo *cpinfo, CPDD *cpdd, md5_state_t *state, md5_byte_t digest[16], struct verify_data *thread_argument, pthread_t *t_verify);
static void * read_thread_func(void *unknown);
static void * do_md5_append(void *tmp);

/*******************************************************************************
*******************************************************************************/
void read_write(const OPArg *oparg, const SDir *sdir, CPInfo *cpinfo, CPDD *cpdd)
{
	static _Bool filesystem_check = false;
	static _Bool fallocate_support = false;
	static _Bool o_direct_write_flag = false;
	static _Bool verify_option_check = false;

	static _Bool open_src;
	static _Bool stat_src;
	static _Bool open_dst;
	static _Bool o_direct_open_dst;

	stat_src = false;
	open_src = false;
	open_dst = false;
	o_direct_open_dst = false;

	src_size_local = 0;
	written_size[0] = 0;
	written_size[1] = 0;
	next_write_size[0] = 0;
	next_write_size[1] = 0;
	total_read_local = 0;
	total_write_local = 0;
	total_read_error = 0;
	total_write_error = 0;
	isrc = 0;
	idst = 0;
	ri = 0;
	wi = 0;

	if(buffer_size == 0)
	{
		buffer_size = oparg->buffer_size;
	}

	if(buffer_size_half == 0)
	{
		buffer_size_half = buffer_size / 2;
	}

	if(psize == 0)
	{
		psize = sysconf(_SC_PAGESIZE);
	}

	if(verify_option_check == false)
	{
		if(oparg->CHECK == V_FIVE)
		{
			verify = true;
		}

		verify_option_check = true;
	}

	if(filesystem_check == false)
	{
		char *tmp = cpinfo->dstfs;

		if(tmp != NULL)
		{
			if
			(
				((tmp[0] == 'e') && (tmp[1] == 'x') && (tmp[2] == 't') && (tmp[3] == '4'))
				||
				((tmp[0] == 'b') && (tmp[1] == 't') && (tmp[2] == 'r') && (tmp[2] == 'f') && (tmp[2] == 's'))
				||
				((tmp[0] == 'o') && (tmp[1] == 'c') && (tmp[2] == 'f') && (tmp[2] == 's') && (tmp[2] == '2'))
			)
			{
				#ifdef _DEBUG_FLAG
					puts("fallocate_support");
					fflush(stdout);
				#endif

				fallocate_support = true;
			}
			else if((tmp[0] == 'x') && (tmp[1] == 'f') && (tmp[2] == 's'))
			{
				#ifdef _DEBUG_FLAG
					puts("fallocate_support (xfs)");
					fflush(stdout);
				#endif

				fallocate_support = true;
				o_direct_write_flag = true;
			}
/*
			else if
			(
				((tmp[0] == 'n') && (tmp[1] == 't') && (tmp[2] == 'f') && (tmp[3] == 's'))
				||
				((tmp[0] == 'f') && (tmp[1] == 'u') && (tmp[2] == 's') && (tmp[3] == 'e') && (tmp[4] == 'b') && (tmp[5] == 'l') && (tmp[6] == 'k'))
			)
			{
				if(verify == true)
				{
					fprintf(stderr, "コピー先のファイルシステムがNTFSかつ、ベリファイオプションが有効です。\n");
					fprintf(stderr, "終了します。\n");
					exit(EXIT_FAILURE);
				}
			}
*/
		}

		filesystem_check = true;
	}

	if(cpinfo->copy_mode == SAME)
	{
		if(strcmp(cpinfo->src, cpinfo->dst) == 0)
		{
			fprintf(stderr, "`%s' と `%s' は同じファイルです\n", cpinfo->src, cpinfo->dst);
			return;
		}
	}

	errno = 0;
	if((isrc = open(cpinfo->src, O_RDONLY | O_NOATIME | O_NOFOLLOW)) == -1)
	{
		switch(errno)
		{
		case EPERM:
			errno = 0;

			if((isrc = open(cpinfo->src, O_RDONLY | O_NOFOLLOW)) == -1)
			{
				print_error("open", __FILE__, __LINE__, cpdd);
			}
			else
			{
				open_src = true;
			}
			break;

		default:
			print_error("open", __FILE__, __LINE__, cpdd);
			break;
		}
	}
	else
	{
		open_src = true;
	}

	static struct stat stat_buf;

	if(open_src == true)
	{
		errno = 0;
		if(fstat(isrc, &stat_buf) == -1)
		{
			print_error("fstat", __FILE__, __LINE__, cpdd);
		}
		else
		{
			stat_src = true;
			src_size_local = stat_buf.st_size;
			cpdd->total_source += src_size_local;
		}

		if((cpinfo->file_test == false) || (oparg->WRITE_MODE == OVERWRITE))
		{
			OPEN_CHECK_DST
		}
		else if(stat_src == true)
		{
			struct stat stat_dst;

			errno = 0;
			if(stat(cpinfo->dst, &stat_dst) == -1)
			{
				print_error("stat", __FILE__, __LINE__, cpdd);

				if(errno == ENOENT)
				{
					OPEN_CHECK_DST
				}
			}
			else
			{
				if(oparg->WRITE_MODE == SIZE_OR_TIME)
				{
					if((stat_buf.st_size != stat_dst.st_size) || (stat_buf.st_mtime != stat_dst.st_mtime))
					{
						OPEN_CHECK_DST
					}
					else
					{
						cpinfo->d_para = SKIP;
					}
				}
				else
				{
					if((stat_buf.st_mtime > stat_dst.st_mtime))
					{
						OPEN_CHECK_DST
					}
					else
					{
						cpinfo->d_para = SKIP;
					}
				}
			}
		}
		else
		{
			print_error("NODATA", __FILE__, __LINE__, cpdd);
			fprintf(stderr, "`%s' をコピーできません\n", cpinfo->src);
		}
	}

	#ifdef _DEBUG_FLAG
		if(o_direct_open_dst == true)
		{
			puts("o_direct_open_dst = true");
			fflush(stdout);
		}
	#endif

	/*
	 * ファイルシステムがfallocateをサポートしている場合、それを実行して領域を確保する。
	 * 
	 * posix_fallocateはext4以外の、それをサポートしていないファイルシステムだとかなり遅くなる様子。
	 * ext4だと4秒弱で終わるフォルダのコピーが、
	 * ntfsだと12秒はかかる（何もしないと6秒弱、pwriteだと8秒ほど、ftruncateは何もしない場合とほぼ同じ）。
	*/
	if((src_size_local > 0) && (open_dst == true) && (fallocate_support == true))
	{
		#ifdef _DEBUG_FLAG
			puts("fallocate");
			fflush(stdout);
		#endif

		/* Ubuntu 11.04だとfallocateの第二引数は FALLOC_FL_KEEP_SIZE と FALLOC_FL_PUNCH_HOLE の二つ。linux/falloc.h のincludeが必要。 */
		if(fallocate(idst, FALLOC_FL_KEEP_SIZE, 0, src_size_local) == -1)
		{
			print_error("fallocate", __FILE__, __LINE__, cpdd);
		}
/*
		static int itmp = -1;

		if((itmp = posix_fallocate(idst, 0, src_size_local)) != 0)
		{
			errno = itmp;
			print_error("posix_fallocate", __FILE__, __LINE__, cpdd);
		}
*/
	}

	/* コピー開始 */
	if((src_size_local > 0) && (open_dst == true))
	{
		static md5_state_t state;
		static md5_byte_t digest[16];
		static struct verify_data thread_argument;
		static pthread_t t_verify;

		digest[0] = '\0';

		if(verify == true)
		{
			md5_init(&state);
		}

		if(oparg->V == VERBOS)
		{
			printf("`%s' -> `%s'\n", cpinfo->src, cpinfo->dst);
			fflush(stdout);
		}

		/* 1048576 == 1024 * 1024 == 1MB */
		if(1048576 > src_size_local)
		{
			if(o_direct_open_dst == false)
			{
				mw_small(cpinfo, cpdd, &state, digest, &thread_argument, &t_verify);
			}
			else
			{
				#ifdef _DEBUG_FLAG
					puts("mw_small_direct");
					fflush(stdout);
				#endif

				mw_small_direct(cpinfo, cpdd, &state, digest, &thread_argument, &t_verify);
			}
		}
		else if(cpinfo->copy_mode == DIFFERENT)
		{
			if(o_direct_open_dst == false)
			{
				mw_thread(oparg, cpinfo, cpdd, &state, digest, &thread_argument, &t_verify);
			}
			else
			{
				#ifdef _DEBUG_FLAG
					puts("mw_thread_direct");
					fflush(stdout);
				#endif

				mw_thread_direct(oparg, cpinfo, cpdd, &state, digest, &thread_argument, &t_verify);
			}
		}
		else
		{
			if(o_direct_open_dst == false)
			{
				mw_big(cpinfo, cpdd, &state, digest, &thread_argument, &t_verify);
			}
			else
			{
				#ifdef _DEBUG_FLAG
				puts("mw_big_direct");
				fflush(stdout);
				#endif

				mw_big_direct(cpinfo, cpdd, &state, digest, &thread_argument, &t_verify);
			}
		}

		if((o_direct_open_dst == true) && (src_size_local < total_write_local))
		{
			if(ftruncate(idst, src_size_local) == -1)
			{
				print_error("ftruncate", __FILE__, __LINE__, cpdd);
			}
		}
	}
	else if((src_size_local == 0) && (open_dst == true))
	{
		/* open時にO_TRUNCするようにしたので、ここの処理は必要ない気がする。 */
		if(ftruncate(idst, 0) == -1)
		{
			print_error("ftruncate", __FILE__, __LINE__, cpdd);
		}
	}

	if((stat_src == true) && (open_dst == true))
	{
		ch_time_and_mod(&idst, &stat_buf, cpinfo, cpdd);
	}

/*
 * ext4でfdatasyncするようにしたら、30倍遅くなった。

	if((oparg->CHECK != NOT) && (fallocate_support == true))
	{
		if(fdatasync(idst) == -1)
		{
			print_error("fdatasync", __FILE__, __LINE__, cpdd);
		}
	}
*/

	if(open_src == true)
	{
		if(close(isrc) == -1)
		{
			print_error("close", __FILE__, __LINE__, cpdd);
		}
	}

	if(open_dst == true)
	{
		if(close(idst) == -1)
		{
			print_error("close", __FILE__, __LINE__, cpdd);
		}
	}

	if(total_read_local > 0)
	{
		cpinfo->src_size = src_size_local;
		cpdd->total_read += total_read_local;
	}

	if(total_write_local > 0)
	{
		cpinfo->dst_size = total_write_local;
		cpdd->total_write += total_write_local;
	}

	if(total_read_error > 0)
	{
		cpdd->total_error += total_read_error;
	}

	if(total_write_error > 0)
	{
		cpdd->total_error += total_write_error;
	}

	write_now = NULL;

	if((open_src == true) && (open_dst == true) && (src_size_local <= total_write_local))
	{
		cpinfo->d_para = WRITE;
	}
	else
	{
		cpinfo->d_para = INITIAL;
	}

	if(oparg->CHECK_MODE == BEFORE)
	{
		switch(oparg->CHECK)
		{
		case V_FIVE:
			if((cpinfo->src_md5 != NULL) && (total_write_local > 0) && (cpinfo->d_para == WRITE))
			{
				verify_hash(oparg, sdir, cpinfo, cpdd);
			}
			break;

		default:
			fprintf(stderr, "ハッシュチェックオプションが不正です\n");
			break;
		}

		if(oparg->V == VERBOS)
		{
			puts("");
			fflush(stdout);
		}
	}
}

/*******************************************************************************
*******************************************************************************/
inline static void mw_small(CPInfo *cpinfo, CPDD *cpdd, md5_state_t *state, md5_byte_t digest[16], struct verify_data *thread_argument, pthread_t *t_verify)
{
	errno = 0;
	buf[0] = mmap(NULL, src_size_local, PROT_READ, MAP_PRIVATE, isrc, 0);

	if(buf[0] == MAP_FAILED)
	{
		print_error("mmap", __FILE__, __LINE__, cpdd);
	}
	else
	{
		total_read_local = src_size_local;

		errno = 0;
		if(madvise(buf[0], src_size_local, MADV_WILLNEED) == -1)
		{
			print_error("madvise", __FILE__, __LINE__, cpdd);
		}

		if(verify == true)
		{
			thread_argument->state = state;
			thread_argument->buffer = buf[0];
			thread_argument->len = src_size_local;

			pthread_create(t_verify, NULL, do_md5_append, (void *)thread_argument);
		}

		char *tmp = buf[0];
		const char * const endp = tmp + src_size_local;

		while(tmp < endp)
		{
			errno = 0;
			if((written_size[0] = write(idst, tmp, endp - tmp)) == -1)
			{
				print_error("write", __FILE__, __LINE__, cpdd);
				break;
			}
			tmp += written_size[0];
			total_write_local += written_size[0];
		}

		if(verify == true)
		{
			pthread_join(*t_verify, NULL);

			MD5_FINISH
		}

		errno = 0;
		if(munmap(buf[0], src_size_local) != 0)
		{
			print_error("munmap", __FILE__, __LINE__, cpdd);
		}
	}
}

/*******************************************************************************
*******************************************************************************/
inline static void mw_small_direct(CPInfo *cpinfo, CPDD *cpdd, md5_state_t *state, md5_byte_t digest[16], struct verify_data *thread_argument, pthread_t *t_verify)
{
	off_t read_size;
	off_t write_size;

	for(;;)
	{
		off_t read_remainder = src_size_local - total_read_local;

		if(read_remainder <= 0)
		{
			break;
		}

		off_t surplus = read_remainder % psize;

		if(surplus == 0)
		{
			read_size = read_remainder;
			write_size = read_size;
		}
		else if(surplus == read_remainder)
		{
			read_size = read_remainder;
			write_size = psize;
		}
		else
		{
			read_size = read_remainder - surplus;
			write_size = read_size;
		}

		errno = 0;
		buf[0] = mmap(NULL, read_size, PROT_READ, MAP_PRIVATE, isrc, total_read_local);

		if(buf[0] == MAP_FAILED)
		{
			print_error("mmap", __FILE__, __LINE__, cpdd);
			break;
		}
		else
		{
			total_read_local += read_size;

			errno = 0;
			if(madvise(buf[0], read_size, MADV_WILLNEED) == -1)
			{
				print_error("madvise", __FILE__, __LINE__, cpdd);
			}

			if(verify == true)
			{
				thread_argument->state = state;
				thread_argument->buffer = buf[0];
				thread_argument->len = read_size;

				pthread_create(t_verify, NULL, do_md5_append, (void *)thread_argument);
			}

			for(;;)
			{
				errno = 0;
				if((written_size[0] = pwrite(idst, buf[0], write_size, total_write_local)) == -1)
				{
					print_error("write", __FILE__, __LINE__, cpdd);
					break;
				}

				if(written_size[0] == write_size)
				{
					total_write_local += written_size[0];

					if(lseek(idst, write_size, SEEK_CUR) == -1)
					{
						print_error("lseek", __FILE__, __LINE__, cpdd);
					}

					break;
				}
			}

			if(verify == true)
			{
				pthread_join(*t_verify, NULL);
			}

			errno = 0;

			if(munmap(buf[0], read_size) != 0)
			{
				print_error("munmap", __FILE__, __LINE__, cpdd);
			}
		}
	}

	if(verify == true)
	{
		MD5_FINISH
	}
}

/*******************************************************************************
*******************************************************************************/
inline static void mw_big(CPInfo *cpinfo, CPDD *cpdd, md5_state_t *state, md5_byte_t digest[16], struct verify_data *thread_argument, pthread_t *t_verify)
{
	for(;;)
	{
		off_t read_size = src_size_local - total_read_local;

		if(read_size >= buffer_size)
		{
			read_size = buffer_size;
		}
		else
		{
			off_t tmp = read_size % psize;

			if(tmp != read_size)
			{
				read_size = read_size - tmp;
			}
		}

		errno = 0;
		buf[0] = mmap(NULL, read_size, PROT_READ, MAP_PRIVATE, isrc, total_read_local);

		if(buf[0] == MAP_FAILED)
		{
			print_error("mmap", __FILE__, __LINE__, cpdd);
			break;
		}
		else
		{
			total_read_local += read_size;

			errno = 0;
			if(madvise(buf[0], read_size, MADV_WILLNEED) == -1)
			{
				print_error("madvise", __FILE__, __LINE__, cpdd);
			}

			if(verify == true)
			{
				thread_argument->state = state;
				thread_argument->buffer = buf[0];
				thread_argument->len = read_size;

				pthread_create(t_verify, NULL, do_md5_append, (void *)thread_argument);
			}

			char *tmp = buf[0];
			const char * const endp = tmp + read_size;

			while(tmp < endp)
			{
				errno = 0;
				if((written_size[0] = write(idst, tmp, endp - tmp)) == -1)
				{
					print_error("write", __FILE__, __LINE__, cpdd);

					errno = 0;
					if(munmap(buf[0], read_size) != 0)
					{
						print_error("munmap", __FILE__, __LINE__, cpdd);
					}

					goto LOOP_END;
				}
				tmp += written_size[0];
				total_write_local += written_size[0];
			}

			if(verify == true)
			{
				pthread_join(*t_verify, NULL);
			}

			errno = 0;
			if(munmap(buf[0], read_size) != 0)
			{
				print_error("munmap", __FILE__, __LINE__, cpdd);
			}
		}

		if(total_read_local >= src_size_local)
		{
			break;
		}
	}
LOOP_END:
;
	if(verify == true)
	{
		MD5_FINISH
	}
}

/*******************************************************************************
*******************************************************************************/
inline static void mw_big_direct(CPInfo *cpinfo, CPDD *cpdd, md5_state_t *state, md5_byte_t digest[16], struct verify_data *thread_argument, pthread_t *t_verify)
{
	off_t read_size;
	off_t write_size;

	for(;;)
	{
		off_t read_remainder = src_size_local - total_read_local;

		if(read_remainder <= 0)
		{
			break;
		}

		if(read_remainder >= buffer_size)
		{
			read_size = buffer_size;
			write_size = buffer_size;
		}
		else
		{
			off_t surplus = read_remainder % psize;
			read_size = 0;
			write_size = 0;

			if(surplus == 0)
			{
				read_size = read_remainder;
				write_size = read_size;
			}
			else if(surplus == read_remainder)
			{
				read_size = read_remainder;
				write_size = psize;
			}
			else
			{
				read_size = read_remainder - surplus;
				write_size = read_size;
			}
		}

		errno = 0;
		buf[0] = mmap(NULL, read_size, PROT_READ, MAP_PRIVATE, isrc, total_read_local);

		if(buf[0] == MAP_FAILED)
		{
			print_error("mmap", __FILE__, __LINE__, cpdd);
			break;
		}
		else
		{
			total_read_local += read_size;

			errno = 0;
			if(madvise(buf[0], read_size, MADV_WILLNEED) == -1)
			{
				print_error("madvise", __FILE__, __LINE__, cpdd);
			}

			if(verify == true)
			{
				thread_argument->state = state;
				thread_argument->buffer = buf[0];
				thread_argument->len = read_size;

				pthread_create(t_verify, NULL, do_md5_append, (void *)thread_argument);
			}

			for(;;)
			{
				errno = 0;
				if((written_size[0] = pwrite(idst, buf[0], write_size, total_write_local)) == -1)
				{
					print_error("write", __FILE__, __LINE__, cpdd);
					break;
				}

				if(written_size[0] == write_size)
				{
					total_write_local += written_size[0];

					if(lseek(idst, write_size, SEEK_CUR) == -1)
					{
						print_error("lseek", __FILE__, __LINE__, cpdd);
					}

					break;
				}
			}

			if(verify == true)
			{
				pthread_join(*t_verify, NULL);
			}

			errno = 0;
			if(munmap(buf[0], read_size) != 0)
			{
				print_error("munmap", __FILE__, __LINE__, cpdd);
			}
		}
	}

	if(verify == true)
	{
		MD5_FINISH
	}
}

/*******************************************************************************
*******************************************************************************/
#define GET_BUFFER_SIZE \
cpdd->thread_mode_count++;\
\
{\
	VList *b_size_list = oparg->b_size_list;\
	b_size_list = b_size_list->tail;\
\
	int *i = (int *)b_size_list->data;\
	int low = *i;\
\
	b_size_list = b_size_list->prev;\
	int high;\
\
	for(;;)\
	{\
		i = (int *)b_size_list->data;\
		high = *i;\
\
		if(high > src_size_local)\
		{\
			buffer_size_thread = low;\
			break;\
		}\
		else if(b_size_list->prev == NULL)\
		{\
			buffer_size_thread = buffer_size_half;\
			break;\
		}\
		else \
		{\
			low = high;\
			b_size_list = b_size_list->prev;\
		}\
	}\
}

/*******************************************************************************
*******************************************************************************/
inline static void mw_thread(const OPArg *oparg, CPInfo *cpinfo, CPDD *cpdd, md5_state_t *state, md5_byte_t digest[16], struct verify_data *thread_argument, pthread_t *t_verify)
{
	GET_BUFFER_SIZE

	ri = 0;
	wi = 1;

	static pthread_t t_read;
	pthread_create(&t_read, NULL, read_thread_func, (void *)&ri);
	pthread_join(t_read, NULL);

	int swap;

	for(;;)
	{
		swap = ri;
		ri = wi;
		wi = swap;

		pthread_create(&t_read, NULL, read_thread_func, (void *)&ri);

		if(verify == true)
		{
			thread_argument->state = state;
			thread_argument->buffer = buf[wi];
			thread_argument->len = next_write_size[wi];

			pthread_create(t_verify, NULL, do_md5_append, (void *)thread_argument);
		}

		if(next_write_size[wi] > 0)
		{
			char *tmp = (char *)buf[wi];
			const char * const endp = tmp + next_write_size[wi];

			while(tmp < endp)
			{
				errno = 0;
				if((written_size[wi] = write(idst, tmp, endp - tmp)) == -1)
				{
					print_error("write", __FILE__, __LINE__, cpdd);
					total_write_error++;
					break;
				}

				tmp += written_size[wi];
				total_write_local += written_size[wi];
			}
		}
		else
		{
			written_size[wi] = 0;
		}

		if(verify == true)
		{
			pthread_join(*t_verify, NULL);
		}

		if(next_write_size[wi] > 0)
		{
			errno = 0;
			if(munmap(buf[wi], next_write_size[wi]) == -1)
			{
				print_error("munmap", __FILE__, __LINE__, cpdd);
				total_write_error++;
			}
		}

		pthread_join(t_read, NULL);

		if((src_size_local <= total_write_local) || (written_size[wi] <= 0))
		{
			break;
		}
	}

	if(verify == true)
	{
		MD5_FINISH
	}
}

/*******************************************************************************
*******************************************************************************/
inline static void mw_thread_direct(const OPArg *oparg, CPInfo *cpinfo, CPDD *cpdd, md5_state_t *state, md5_byte_t digest[16], struct verify_data *thread_argument, pthread_t *t_verify)
{
	GET_BUFFER_SIZE

	ri = 0;
	wi = 1;

	static pthread_t t_read;
	pthread_create(&t_read, NULL, read_thread_func, (void *)&ri);
	pthread_join(t_read, NULL);

	int swap;

	for(;;)
	{
		swap = ri;
		ri = wi;
		wi = swap;

		pthread_create(&t_read, NULL, read_thread_func, (void *)&ri);

		if(verify == true)
		{
			thread_argument->state = state;
			thread_argument->buffer = buf[wi];
			thread_argument->len = next_write_size[wi];

			pthread_create(t_verify, NULL, do_md5_append, (void *)thread_argument);
		}

		if(next_write_size[wi] > 0)
		{
			char *tmp = (char *)buf[wi];
			off_t wtmp = next_write_size[wi];
			if(wtmp < psize)
			{
				wtmp = psize;
			}
			const char * const endp = tmp + wtmp;

			while(tmp < endp)
			{
				errno = 0;
				written_size[wi] = pwrite(idst, tmp, endp - tmp, total_write_local);

				if(written_size[wi] == -1)
				{
					print_error("write", __FILE__, __LINE__, cpdd);
					total_write_error++;
					break;
				}
				else if(written_size[wi] == wtmp)
				{
					tmp += written_size[wi];
					total_write_local += written_size[wi];

					if(lseek(idst, total_write_local + written_size[wi], SEEK_SET) == -1)
					{
						print_error("write", __FILE__, __LINE__, cpdd);
						total_write_error++;
						break;
					}
				}
			}
		}
		else
		{
			written_size[wi] = 0;
		}

		if(verify == true)
		{
			pthread_join(*t_verify, NULL);
		}

		if(next_write_size[wi] > 0)
		{
			errno = 0;
			if(munmap(buf[wi], next_write_size[wi]) == -1)
			{
				print_error("munmap", __FILE__, __LINE__, cpdd);
				total_write_error++;
			}
		}

		pthread_join(t_read, NULL);

		if((src_size_local <= total_write_local) || (written_size[wi] <= 0))
		{
			break;
		}
	}

	if(verify == true)
	{
		MD5_FINISH
	}
}

/*******************************************************************************
 * 読み取り関数。引数は使用しない。
*******************************************************************************/
static void * read_thread_func(void *unknown)
{
	int i = ri;
	unknown++;	/* コンパイル時のwarningを出さないようにするため */

	if(src_size_local <= total_read_local)
	{
		next_write_size[i] = 0;
		return NULL;
	}

	off_t read_size = src_size_local - total_read_local;

	if(read_size >= buffer_size_thread)
	{
		read_size = buffer_size_thread;
	}
	else
	{
		off_t tmp = read_size % psize;

		if(tmp != read_size)
		{
			read_size = read_size - tmp;
		}
	}

	errno = 0;
	buf[i] = mmap(NULL, read_size, PROT_READ, MAP_PRIVATE, isrc, total_read_local);

	if(buf[i] == MAP_FAILED)
	{
		/* このcpddに意味はない。print_errorの引数用。 */
		CPDD cpdd;
		cpdd.total_error = 0;
		print_error("mmap", __FILE__, __LINE__, &cpdd);
		total_read_error++;

		next_write_size[i] = 0;
	}
	else
	{
		errno = 0;
		if(madvise(buf[i], read_size, MADV_WILLNEED) == -1)
		{
			/* このcpddに意味はない。print_errorの引数用。 */
			CPDD cpdd;
			cpdd.total_error = 0;
			print_error("madvise", __FILE__, __LINE__, &cpdd);
			total_read_error++;
		}

		total_read_local += read_size;
		next_write_size[i] = read_size;
	}

	return NULL;
}

/*******************************************************************************
 * ベリファイ用
*******************************************************************************/
static void * do_md5_append(void *tmp)
{
	struct verify_data *tmp_vd = tmp;
	md5_state_t *state = tmp_vd->state;
	md5_byte_t *buffer = tmp_vd->buffer;
	int len = tmp_vd->len;

	md5_append(state, buffer, len);

	return NULL;
}
