#ifndef CREATE_SYMBOLICKLINK_H
#define CREATE_SYMBOLICKLINK_H

#define _GNU_SOURCE

#include "ch_time_and_mod_link.h" // inline
#include "struct_CPDD.h"
#include "struct_OPArg.h"

#include <stdbool.h>

#include <glib.h>
#include <linux/limits.h>

#define UNLINK_ERR \
	print_error("unlink", __FILE__, __LINE__, cpdd);\
	fprintf(stderr, "対象を削除出来ませんでした (%s) \n", cpinfo->to);\
	fprintf(stderr, "シンボリックリンクを作成できません\n");\
	cpinfo->write = false;

// 関数プロトタイプ
static inline _Bool create_symboliclink(const OPArg *oparg, CPInfo *cpinfo, CPDD *cpdd) __attribute__((always_inline));
static inline _Bool c_s(const OPArg *oparg, const CPInfo *cpinfo, CPDD *cpdd) __attribute__((always_inline));
/*******************************************************************************
シンボリックリンクを作成
*******************************************************************************/
static inline _Bool create_symboliclink(const OPArg *oparg, CPInfo *cpinfo, CPDD *cpdd)
{
	_Bool check;
	struct stat stat_from;
	struct stat stat_to;

	gboolean link_check = g_file_test(cpinfo->to, G_FILE_TEST_IS_SYMLINK);
	gboolean file_check = g_file_test(cpinfo->to, G_FILE_TEST_EXISTS);

	if((link_check == FALSE) && (file_check == FALSE))
	{
		check = c_s(oparg, cpinfo, cpdd);
	}
	else
	{
		if(oparg->WRITE_MODE == OVERWRITE)
		{
			errno = 0;

			if(unlink(cpinfo->to) == -1)
			{
				check = false;
				UNLINK_ERR
			}
			else
			{
				check = c_s(oparg, cpinfo, cpdd);
			}
		}
		else
		{
			errno = 0;

			if(lstat(cpinfo->from, &stat_from) == -1)
			{
				check = false;
				print_error("lstat", __FILE__, __LINE__, cpdd);
				fprintf(stderr, "ファイル情報取得エラーです (%s) \n", cpinfo->from);
			}
			else if(lstat(cpinfo->to, &stat_to) == -1)
			{
				if(errno == ENOENT)
				{
					check = c_s(oparg, cpinfo, cpdd);
				}
				else
				{
					check = false;
					print_error("lstat", __FILE__, __LINE__, cpdd);
					fprintf(stderr, "ファイル情報取得エラーです (%s) \n", cpinfo->to);
				}
			}
			else
			{
				switch(oparg->WRITE_MODE)
				{
				case SIZE_OR_TIME:
					if(stat_from.st_mtime != stat_to.st_mtime)
					{
						if(unlink(cpinfo->to) == -1)
						{
							check = false;
							UNLINK_ERR
						}
						else
						{
							check = c_s(oparg, cpinfo, cpdd);
						}
					}
					else
					{
						check = false;
					}
					break;

				case NEW_TIME:
					if(stat_from.st_mtime > stat_to.st_mtime)
					{
						if(unlink(cpinfo->to) == -1)
						{
							check = false;
							UNLINK_ERR
						}
						else
						{
							check = c_s(oparg, cpinfo, cpdd);
						}
					}
					else
					{
						check = false;
					}
					break;

				default:
					check = false;
					print_error(__func__, __FILE__, __LINE__, cpdd);
					fprintf(stderr, "上書きモードオプションエラーです (%s) \n", cpinfo->from);
					break;
				}
			}
		}
	}

	return check;
}

/*******************************************************************************
*******************************************************************************/
static inline _Bool c_s(const OPArg *oparg, const CPInfo *cpinfo, CPDD *cpdd)
{
	int readlink_len;
	char link_buf[PATH_MAX];
	_Bool check;

	if((readlink_len = readlink(cpinfo->from, link_buf, PATH_MAX)) == -1)
	{
		check = false;
		print_error("readlink", __FILE__, __LINE__, cpdd);
		fprintf(stderr, "シンボリックリンク読み込みエラーです (%s) \n", cpinfo->from);
	}
	else
	{
		/*
		 * readlinkは終端にヌル文字を付加しない、
		 * という仕様の所為でlink_bufの文字列がおかしくなるバグに悩まされた。
		 * ローカル変数はスコープから抜けると解放されるが、
		 * 即座に解放されるわけではなく（パフォーマンスが落ちるし）、
		 * 領域を使い回すこともしばしばある。
		 * この関数が呼び出されるとlink_bufにシンボリックリンクの内容がコピーされるが、
		 * 関数から抜けてもlink_bufの内容は保持されたままになる。
		 * 次回呼び出されたときにlink_bufの内容が上書きされ、
		 * それが前回よりも短い文字列だった場合、ヌル文字がないので、
		 * 文字列が混ざり合っておかしな事になっていた。
		*/
		link_buf[readlink_len] = '\0';

		if(symlink(link_buf, cpinfo->to) == -1)
		{
			check = false;
			print_error("symlink", __FILE__, __LINE__, cpdd);
			fprintf(stderr, "シンボリックリンク作成エラーです (%s) \n", cpinfo->to);
		}
		else
		{
			check = true;
			cpdd->mk_link_count++;

			if(oparg->V == VERBOS)
			{
				printf("%s -> %s\n", cpinfo->from, cpinfo->to);
			}

			struct stat stat_buf;

			if(lstat(cpinfo->from, &stat_buf) == -1)
			{
				print_error("lstat", __FILE__, __LINE__, cpdd);
				fprintf(stderr, "ファイル情報取得エラーです (%s) \n", cpinfo->from);
			}
			else
			{
				ch_time_and_mod_link(&stat_buf, cpinfo, cpdd);
			}
		}
	}

	return check;
}

#endif
